/*
 * Copyright (c) 2020-2024 Estonian Information System Authority
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#pragma once

#include "pcsc-mock/pcsc-mock.hpp"

const PcscMock::ApduScript ESTEID_IDEMIA_V1_SELECT_AUTH_CERTIFICATE_AND_AUTHENTICATE = {
    // Select main AID.
    {{0x00, 0xA4, 0x04, 0x0C, 0x10, 0xA0, 0x00, 0x00, 0x00, 0x77, 0x01,
      0x08, 0x00, 0x07, 0x00, 0x00, 0xFE, 0x00, 0x00, 0x01, 0x00},
     {0x90, 0x00}},

    // Select authentication certificate file.
    {{0x00, 0xA4, 0x09, 0x04, 0x04, 0xAD, 0xF1, 0x34, 0x01, 0x00},
     {0x62, 0x04, 0x80, 0x02, 0x04, 0x07, 0x90, 0x00}},

    // Read first block.
    {{0x00, 0xb0, 0x00, 0x00, 0xc0},
     {0x30, 0x82, 0x04, 0x03, 0x30, 0x82, 0x03, 0x65, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10,
      0x39, 0x69, 0x01, 0x59, 0x73, 0x43, 0x26, 0x6d, 0x5b, 0xc8, 0x57, 0x77, 0x5e, 0xc5, 0xa4,
      0xbe, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x04, 0x30, 0x60,
      0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x45, 0x45, 0x31, 0x1b,
      0x30, 0x19, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x12, 0x53, 0x4b, 0x20, 0x49, 0x44, 0x20,
      0x53, 0x6f, 0x6c, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x41, 0x53, 0x31, 0x17, 0x30,
      0x15, 0x06, 0x03, 0x55, 0x04, 0x61, 0x0c, 0x0e, 0x4e, 0x54, 0x52, 0x45, 0x45, 0x2d, 0x31,
      0x30, 0x37, 0x34, 0x37, 0x30, 0x31, 0x33, 0x31, 0x1b, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04,
      0x03, 0x0c, 0x12, 0x54, 0x45, 0x53, 0x54, 0x20, 0x6f, 0x66, 0x20, 0x45, 0x53, 0x54, 0x45,
      0x49, 0x44, 0x32, 0x30, 0x31, 0x38, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x38, 0x31, 0x30, 0x31,
      0x38, 0x30, 0x39, 0x35, 0x30, 0x34, 0x37, 0x5a, 0x17, 0x0d, 0x32, 0x33, 0x31, 0x30, 0x31,
      0x37, 0x32, 0x31, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x7f, 0x31, 0x0b, 0x30, 0x09, 0x06,
      0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x45, 0x45, 0x31, 0x2a, 0x30, 0x28, 0x90, 0x00}},

    // Keep reading blocks until done.
    {{0x00, 0xb0, 0x00, 0xc0, 0xc0},
     {0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x21, 0x4a, 0xc3, 0x95, 0x45, 0x4f, 0x52, 0x47, 0x2c,
      0x4a, 0x41, 0x41, 0x4b, 0x2d, 0x4b, 0x52, 0x49, 0x53, 0x54, 0x4a, 0x41, 0x4e, 0x2c, 0x33,
      0x38, 0x30, 0x30, 0x31, 0x30, 0x38, 0x35, 0x37, 0x31, 0x38, 0x31, 0x10, 0x30, 0x0e, 0x06,
      0x03, 0x55, 0x04, 0x04, 0x0c, 0x07, 0x4a, 0xc3, 0x95, 0x45, 0x4f, 0x52, 0x47, 0x31, 0x16,
      0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x2a, 0x0c, 0x0d, 0x4a, 0x41, 0x41, 0x4b, 0x2d, 0x4b,
      0x52, 0x49, 0x53, 0x54, 0x4a, 0x41, 0x4e, 0x31, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55, 0x04,
      0x05, 0x13, 0x11, 0x50, 0x4e, 0x4f, 0x45, 0x45, 0x2d, 0x33, 0x38, 0x30, 0x30, 0x31, 0x30,
      0x38, 0x35, 0x37, 0x31, 0x38, 0x30, 0x76, 0x30, 0x10, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce,
      0x3d, 0x02, 0x01, 0x06, 0x05, 0x2b, 0x81, 0x04, 0x00, 0x22, 0x03, 0x62, 0x00, 0x04, 0x79,
      0x93, 0x59, 0x57, 0xce, 0xf4, 0x9e, 0x23, 0xd3, 0xbf, 0xd6, 0xcd, 0x69, 0x66, 0xf8, 0xe1,
      0x11, 0x6f, 0x27, 0x22, 0xd2, 0x68, 0x1b, 0x41, 0x01, 0x17, 0x19, 0x8b, 0x11, 0x8e, 0x92,
      0xee, 0x48, 0xb5, 0xbc, 0x5d, 0x90, 0xdd, 0x31, 0x03, 0xc7, 0xa7, 0x4d, 0xce, 0xd2, 0x35,
      0x5e, 0xdc, 0x7d, 0xe9, 0xcb, 0x3c, 0xaf, 0x8c, 0xd8, 0x4f, 0x07, 0x62, 0x90, 0x00}},

    {{0x00, 0xb0, 0x01, 0x80, 0xc0},
     {0x42, 0x80, 0xe0, 0xee, 0x4d, 0xdd, 0x89, 0x4b, 0x91, 0xf6, 0x74, 0xd6, 0x1b, 0x14, 0x04,
      0x30, 0xed, 0x45, 0x0e, 0x8e, 0x18, 0xf5, 0xf0, 0x26, 0x88, 0xf1, 0x96, 0x4f, 0xef, 0x86,
      0xb8, 0xe5, 0x68, 0x65, 0x06, 0x37, 0x5f, 0x18, 0xa3, 0x82, 0x01, 0xc3, 0x30, 0x82, 0x01,
      0xbf, 0x30, 0x09, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06,
      0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x03, 0x88, 0x30, 0x47,
      0x06, 0x03, 0x55, 0x1d, 0x20, 0x04, 0x40, 0x30, 0x3e, 0x30, 0x32, 0x06, 0x0b, 0x2b, 0x06,
      0x01, 0x04, 0x01, 0x83, 0x91, 0x21, 0x01, 0x02, 0x01, 0x30, 0x23, 0x30, 0x21, 0x06, 0x08,
      0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x15, 0x68, 0x74, 0x74, 0x70, 0x73,
      0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x73, 0x6b, 0x2e, 0x65, 0x65, 0x2f, 0x43, 0x50,
      0x53, 0x30, 0x08, 0x06, 0x06, 0x04, 0x00, 0x8f, 0x7a, 0x01, 0x02, 0x30, 0x1f, 0x06, 0x03,
      0x55, 0x1d, 0x11, 0x04, 0x18, 0x30, 0x16, 0x81, 0x14, 0x33, 0x38, 0x30, 0x30, 0x31, 0x30,
      0x38, 0x35, 0x37, 0x31, 0x38, 0x40, 0x65, 0x65, 0x73, 0x74, 0x69, 0x2e, 0x65, 0x65, 0x30,
      0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xe4, 0x2c, 0x90, 0x00}},

    {{0x00, 0xb0, 0x02, 0x40, 0xc0},
     {0xbd, 0x34, 0x09, 0x10, 0x15, 0x4c, 0x31, 0x29, 0xa1, 0xc9, 0x95, 0xf9, 0x6e, 0x26, 0xd8,
      0x26, 0xe6, 0xc0, 0x30, 0x61, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x03,
      0x04, 0x55, 0x30, 0x53, 0x30, 0x51, 0x06, 0x06, 0x04, 0x00, 0x8e, 0x46, 0x01, 0x05, 0x30,
      0x47, 0x30, 0x45, 0x16, 0x3f, 0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x73, 0x6b,
      0x2e, 0x65, 0x65, 0x2f, 0x65, 0x6e, 0x2f, 0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x6f,
      0x72, 0x79, 0x2f, 0x63, 0x6f, 0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2d, 0x66,
      0x6f, 0x72, 0x2d, 0x75, 0x73, 0x65, 0x2d, 0x6f, 0x66, 0x2d, 0x63, 0x65, 0x72, 0x74, 0x69,
      0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x73, 0x2f, 0x13, 0x02, 0x45, 0x4e, 0x30, 0x20, 0x06,
      0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06,
      0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03,
      0x04, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xc0,
      0x84, 0x99, 0x29, 0xc4, 0x4e, 0x9f, 0x3b, 0x02, 0x34, 0xf6, 0x99, 0xe1, 0x0a, 0x56, 0x00,
      0x08, 0x29, 0x3e, 0x7b, 0x30, 0x73, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x90, 0x00}},

    {{0x00, 0xb0, 0x03, 0x00, 0xc0},
     {0x05, 0x07, 0x01, 0x01, 0x04, 0x67, 0x30, 0x65, 0x30, 0x2c, 0x06, 0x08, 0x2b, 0x06, 0x01,
      0x05, 0x05, 0x07, 0x30, 0x01, 0x86, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x61,
      0x69, 0x61, 0x2e, 0x64, 0x65, 0x6d, 0x6f, 0x2e, 0x73, 0x6b, 0x2e, 0x65, 0x65, 0x2f, 0x65,
      0x73, 0x74, 0x65, 0x69, 0x64, 0x32, 0x30, 0x31, 0x38, 0x30, 0x35, 0x06, 0x08, 0x2b, 0x06,
      0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x29, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f,
      0x63, 0x2e, 0x73, 0x6b, 0x2e, 0x65, 0x65, 0x2f, 0x54, 0x65, 0x73, 0x74, 0x5f, 0x6f, 0x66,
      0x5f, 0x45, 0x53, 0x54, 0x45, 0x49, 0x44, 0x32, 0x30, 0x31, 0x38, 0x2e, 0x64, 0x65, 0x72,
      0x2e, 0x63, 0x72, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03,
      0x04, 0x03, 0x81, 0x8b, 0x00, 0x30, 0x81, 0x87, 0x02, 0x42, 0x01, 0xf5, 0x52, 0xc9, 0x8c,
      0x76, 0xd2, 0xd9, 0xb6, 0x2e, 0x75, 0x16, 0xad, 0x90, 0x47, 0x8c, 0x14, 0x90, 0x0c, 0x29,
      0xb2, 0x78, 0x6c, 0x05, 0x5d, 0x87, 0x42, 0xa5, 0x17, 0x15, 0x80, 0x49, 0xec, 0x45, 0xe7,
      0x2c, 0x29, 0x1a, 0x5e, 0x37, 0x59, 0x2a, 0x41, 0xe9, 0x00, 0x17, 0xcc, 0xd1, 0x01, 0x31,
      0xed, 0x45, 0x21, 0x68, 0x8e, 0x22, 0xe7, 0x78, 0xe2, 0x27, 0xb1, 0x35, 0x90, 0x00}},

    // Read final block.
    {{0x00, 0xb0, 0x03, 0xc0, 0x47},
     {0x63, 0xb5, 0x8e, 0x00, 0x02, 0x41, 0x0d, 0x7e, 0x33, 0xeb, 0x5a, 0x67, 0x88, 0x72, 0x5a,
      0xa4, 0x8d, 0x5b, 0x90, 0xc2, 0x22, 0x25, 0x0f, 0xed, 0x8b, 0xb8, 0x5a, 0xf1, 0xf7, 0x49,
      0x48, 0xc4, 0xa9, 0x01, 0xdb, 0x39, 0x0b, 0x21, 0xc8, 0xc8, 0x79, 0x10, 0xcd, 0x69, 0x55,
      0xcb, 0xce, 0x9f, 0xf6, 0x60, 0x3b, 0x12, 0xed, 0xe8, 0x44, 0xef, 0xf5, 0xab, 0xe7, 0x85,
      0x53, 0x30, 0xdb, 0x34, 0xdd, 0xd5, 0xbf, 0x2e, 0xe0, 0x41, 0x64, 0x90, 0x00}},

    // 2. PIN Retry count
    // Select main AID.
    {{0x00, 0xA4, 0x04, 0x0C, 0x10, 0xA0, 0x00, 0x00, 0x00, 0x77, 0x01,
      0x08, 0x00, 0x07, 0x00, 0x00, 0xFE, 0x00, 0x00, 0x01, 0x00},
     {0x90, 0x00}},

    // Get retry count
    {{0x00, 0xcb, 0x3f, 0xff, 0x0a, 0x4d, 0x08, 0x70, 0x06, 0xbf, 0x81, 0x01, 0x02, 0xa0, 0x80,
      0x00},
     {0x70, 0x1e, 0xbf, 0x81, 0x01, 0x1a, 0xa0, 0x18, 0x9a, 0x01, 0x03, 0x9b,
      0x01, 0x03, 0xa1, 0x10, 0x8c, 0x06, 0xf3, 0x00, 0x00, 0x73, 0x43, 0x00,
      0x9c, 0x06, 0xf3, 0x00, 0x00, 0x73, 0x43, 0x00, 0x90, 0x00}},

    // 3. Authenticate.
    // Select AWP AID.
    {{0x00, 0xA4, 0x04, 0x0C, 0x0D, 0xe8, 0x28, 0xbd, 0x08, 0x0f, 0xf2, 0x50, 0x4f, 0x54, 0x20,
      0x41, 0x57, 0x50},
     {0x90, 0x00}},

    // Set env
    {{0x00, 0x22, 0x41, 0xa4, 0x06, 0x80, 0x01, 0x04, 0x84, 0x01, 0x81}, {0x90, 0x00}},

    // Verify PIN.
    {{0x00, 0x20, 0x00, 0x01, 0x0c, 0x31, 0x32, 0x33, 0x34, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff},
     {0x90, 0x00}},

// Internal authenticate.
#ifdef NO_SIGNATURE
    {{}, {0x90, 0x00}} // Empty command as hash is changing, bogus response.
#else
    {{0x00, 0x88, 0x00, 0x00, 0x30, 0x86, 0x25, 0x5f, 0xa2, 0xc3, 0x6e, 0x4b, 0x30, 0x96,
      0x9e, 0xae, 0x17, 0xdc, 0x34, 0xc7, 0x72, 0xcb, 0xeb, 0xdf, 0xc5, 0x8b, 0x58, 0x40,
      0x39, 0x00, 0xbe, 0x87, 0x61, 0x4e, 0xb1, 0xa3, 0x4b, 0x87, 0x80, 0x26, 0x3f, 0x25,
      0x5e, 0xb5, 0xe6, 0x5c, 0xa9, 0xbb, 0xb8, 0x64, 0x1c, 0xcc, 0xfe, 0x00},
     {0x6e, 0x52, 0x8b, 0xa6, 0xfb, 0x25, 0x48, 0x24, 0x04, 0xa1, 0xdb, 0x4d, 0x01, 0x38,
      0xa1, 0xc3, 0xa5, 0xef, 0x77, 0x76, 0xae, 0x3c, 0x37, 0x4b, 0x76, 0xbe, 0xbf, 0xa2,
      0x25, 0xc3, 0xb9, 0xd2, 0x16, 0x2e, 0xbc, 0x83, 0xfb, 0xa1, 0x0f, 0x00, 0x3f, 0xda,
      0xff, 0xca, 0x00, 0xbd, 0x54, 0xeb, 0xfa, 0x1c, 0xb5, 0x76, 0x0c, 0x10, 0x5d, 0x86,
      0xeb, 0x61, 0xf0, 0x47, 0x26, 0x2d, 0xf8, 0x9b, 0x43, 0x02, 0x18, 0xf2, 0xce, 0xf1,
      0x80, 0x18, 0xf2, 0xd2, 0x7a, 0x69, 0x22, 0x50, 0xe2, 0x43, 0xb9, 0xfe, 0x5b, 0x74,
      0xa3, 0x87, 0xfa, 0xbd, 0x86, 0x7b, 0x31, 0x75, 0x8a, 0xa6, 0x7f, 0xa0, 0x90, 0x00}}
#endif
};

const PcscMock::ApduScript ESTEID_IDEMIA_V1_SELECT_SIGN_CERTIFICATE_AND_SIGNING = {
    // Select main AID.
    {{0x00, 0xA4, 0x04, 0x0C, 0x10, 0xA0, 0x00, 0x00, 0x00, 0x77, 0x01,
      0x08, 0x00, 0x07, 0x00, 0x00, 0xFE, 0x00, 0x00, 0x01, 0x00},
     {0x90, 0x00}},
    // Select signing certificate file.
    {{0x00, 0xA4, 0x09, 0x04, 0x04, 0xAD, 0xF2, 0x34, 0x1F, 0x00},
     {0x62, 0x04, 0x80, 0x02, 0x03, 0xF0, 0x90, 0x00}},

    // Read first block.
    {{0x00, 0xb0, 0x00, 0x00, 0xc0},
     {0x30, 0x82, 0x03, 0xec, 0x30, 0x82, 0x03, 0x4d, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10,
      0x19, 0x66, 0xaa, 0x25, 0x7f, 0x89, 0x98, 0x71, 0x5b, 0xc8, 0x57, 0x78, 0x6d, 0xad, 0x69,
      0x6a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x04, 0x30, 0x60,
      0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x45, 0x45, 0x31, 0x1b,
      0x30, 0x19, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x0c, 0x12, 0x53, 0x4b, 0x20, 0x49, 0x44, 0x20,
      0x53, 0x6f, 0x6c, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x41, 0x53, 0x31, 0x17, 0x30,
      0x15, 0x06, 0x03, 0x55, 0x04, 0x61, 0x0c, 0x0e, 0x4e, 0x54, 0x52, 0x45, 0x45, 0x2d, 0x31,
      0x30, 0x37, 0x34, 0x37, 0x30, 0x31, 0x33, 0x31, 0x1b, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04,
      0x03, 0x0c, 0x12, 0x54, 0x45, 0x53, 0x54, 0x20, 0x6f, 0x66, 0x20, 0x45, 0x53, 0x54, 0x45,
      0x49, 0x44, 0x32, 0x30, 0x31, 0x38, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x38, 0x31, 0x30, 0x31,
      0x38, 0x30, 0x39, 0x35, 0x30, 0x34, 0x37, 0x5a, 0x17, 0x0d, 0x32, 0x33, 0x31, 0x30, 0x31,
      0x37, 0x32, 0x31, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x7f, 0x31, 0x0b, 0x30, 0x09, 0x06,
      0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x45, 0x45, 0x31, 0x2a, 0x30, 0x28, 0x90, 0x00}},

    // Keep reading blocks until done.
    {{0x00, 0xb0, 0x00, 0xc0, 0xc0},
     {0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x21, 0x4a, 0xc3, 0x95, 0x45, 0x4f, 0x52, 0x47, 0x2c,
      0x4a, 0x41, 0x41, 0x4b, 0x2d, 0x4b, 0x52, 0x49, 0x53, 0x54, 0x4a, 0x41, 0x4e, 0x2c, 0x33,
      0x38, 0x30, 0x30, 0x31, 0x30, 0x38, 0x35, 0x37, 0x31, 0x38, 0x31, 0x10, 0x30, 0x0e, 0x06,
      0x03, 0x55, 0x04, 0x04, 0x0c, 0x07, 0x4a, 0xc3, 0x95, 0x45, 0x4f, 0x52, 0x47, 0x31, 0x16,
      0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x2a, 0x0c, 0x0d, 0x4a, 0x41, 0x41, 0x4b, 0x2d, 0x4b,
      0x52, 0x49, 0x53, 0x54, 0x4a, 0x41, 0x4e, 0x31, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55, 0x04,
      0x05, 0x13, 0x11, 0x50, 0x4e, 0x4f, 0x45, 0x45, 0x2d, 0x33, 0x38, 0x30, 0x30, 0x31, 0x30,
      0x38, 0x35, 0x37, 0x31, 0x38, 0x30, 0x76, 0x30, 0x10, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce,
      0x3d, 0x02, 0x01, 0x06, 0x05, 0x2b, 0x81, 0x04, 0x00, 0x22, 0x03, 0x62, 0x00, 0x04, 0xc5,
      0xd2, 0xd7, 0x3b, 0xe1, 0x98, 0xc4, 0xf5, 0x43, 0x69, 0xe2, 0x25, 0xb0, 0x30, 0x54, 0x3f,
      0xce, 0x89, 0x6b, 0x0c, 0x1f, 0x57, 0x1f, 0xff, 0xc5, 0x25, 0xdc, 0x13, 0xeb, 0x29, 0xd0,
      0x05, 0x3f, 0x2f, 0xe8, 0x73, 0x5d, 0xc5, 0x11, 0x7e, 0xcf, 0xef, 0x0f, 0xf3, 0x57, 0x91,
      0x30, 0x73, 0x5b, 0xf9, 0x57, 0xa1, 0x50, 0xd5, 0x28, 0x6f, 0xfb, 0x39, 0x90, 0x00}},

    {{0x00, 0xb0, 0x01, 0x80, 0xc0},
     {0x7a, 0xde, 0xe2, 0x5b, 0x46, 0xca, 0xdb, 0xcc, 0x90, 0xae, 0x54, 0xdc, 0xc8, 0x77, 0xd0,
      0x37, 0x11, 0xcc, 0x7c, 0x12, 0x45, 0xcc, 0x39, 0x7e, 0xe9, 0x02, 0x26, 0x53, 0x67, 0xd7,
      0x28, 0xa2, 0xb8, 0xa2, 0x8a, 0x55, 0xdb, 0xcb, 0xa3, 0x82, 0x01, 0xab, 0x30, 0x82, 0x01,
      0xa7, 0x30, 0x09, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06,
      0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x06, 0x40, 0x30, 0x48,
      0x06, 0x03, 0x55, 0x1d, 0x20, 0x04, 0x41, 0x30, 0x3f, 0x30, 0x32, 0x06, 0x0b, 0x2b, 0x06,
      0x01, 0x04, 0x01, 0x83, 0x91, 0x21, 0x01, 0x02, 0x01, 0x30, 0x23, 0x30, 0x21, 0x06, 0x08,
      0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x15, 0x68, 0x74, 0x74, 0x70, 0x73,
      0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x73, 0x6b, 0x2e, 0x65, 0x65, 0x2f, 0x43, 0x50,
      0x53, 0x30, 0x09, 0x06, 0x07, 0x04, 0x00, 0x8b, 0xec, 0x40, 0x01, 0x02, 0x30, 0x1d, 0x06,
      0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xe2, 0x83, 0x9c, 0x1c, 0x90, 0xc2, 0xbd,
      0x96, 0xc8, 0x92, 0xf9, 0x26, 0xdc, 0x02, 0xe6, 0x54, 0x3c, 0x85, 0xb1, 0xd4, 0x30, 0x81,
      0x8a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x03, 0x04, 0x90, 0x00}},

    {{0x00, 0xb0, 0x02, 0x40, 0xc0},
     {0x7e, 0x30, 0x7c, 0x30, 0x08, 0x06, 0x06, 0x04, 0x00, 0x8e, 0x46, 0x01, 0x01, 0x30, 0x08,
      0x06, 0x06, 0x04, 0x00, 0x8e, 0x46, 0x01, 0x04, 0x30, 0x13, 0x06, 0x06, 0x04, 0x00, 0x8e,
      0x46, 0x01, 0x06, 0x30, 0x09, 0x06, 0x07, 0x04, 0x00, 0x8e, 0x46, 0x01, 0x06, 0x01, 0x30,
      0x51, 0x06, 0x06, 0x04, 0x00, 0x8e, 0x46, 0x01, 0x05, 0x30, 0x47, 0x30, 0x45, 0x16, 0x3f,
      0x68, 0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x73, 0x6b, 0x2e, 0x65, 0x65, 0x2f, 0x65,
      0x6e, 0x2f, 0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x6f, 0x72, 0x79, 0x2f, 0x63, 0x6f,
      0x6e, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2d, 0x66, 0x6f, 0x72, 0x2d, 0x75, 0x73,
      0x65, 0x2d, 0x6f, 0x66, 0x2d, 0x63, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74,
      0x65, 0x73, 0x2f, 0x13, 0x02, 0x45, 0x4e, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
      0x18, 0x30, 0x16, 0x80, 0x14, 0xc0, 0x84, 0x99, 0x29, 0xc4, 0x4e, 0x9f, 0x3b, 0x02, 0x34,
      0xf6, 0x99, 0xe1, 0x0a, 0x56, 0x00, 0x08, 0x29, 0x3e, 0x7b, 0x30, 0x73, 0x06, 0x08, 0x2b,
      0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x67, 0x30, 0x65, 0x30, 0x2c, 0x06, 0x08,
      0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x86, 0x20, 0x68, 0x74, 0x90, 0x00}},

    {{0x00, 0xb0, 0x03, 0x00, 0xc0},
     {0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x61, 0x69, 0x61, 0x2e, 0x64, 0x65, 0x6d, 0x6f, 0x2e, 0x73,
      0x6b, 0x2e, 0x65, 0x65, 0x2f, 0x65, 0x73, 0x74, 0x65, 0x69, 0x64, 0x32, 0x30, 0x31, 0x38,
      0x30, 0x35, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x29, 0x68,
      0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x63, 0x2e, 0x73, 0x6b, 0x2e, 0x65, 0x65, 0x2f, 0x54,
      0x65, 0x73, 0x74, 0x5f, 0x6f, 0x66, 0x5f, 0x45, 0x53, 0x54, 0x45, 0x49, 0x44, 0x32, 0x30,
      0x31, 0x38, 0x2e, 0x64, 0x65, 0x72, 0x2e, 0x63, 0x72, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a,
      0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x04, 0x03, 0x81, 0x8c, 0x00, 0x30, 0x81, 0x88, 0x02,
      0x42, 0x01, 0x60, 0xa0, 0x10, 0x22, 0x7e, 0x3a, 0xb4, 0x3b, 0x9e, 0x8e, 0xf2, 0x2b, 0xf1,
      0x01, 0x62, 0x3a, 0xcf, 0x20, 0x70, 0x43, 0xc5, 0x69, 0x6a, 0x0d, 0x6a, 0x6a, 0x7b, 0xa1,
      0xed, 0x07, 0x31, 0xb2, 0x96, 0x9c, 0xe2, 0x66, 0x6c, 0x85, 0xcf, 0xa6, 0x69, 0x61, 0xb6,
      0x65, 0xcc, 0x4b, 0x48, 0xd4, 0x8e, 0x27, 0x56, 0xbc, 0x93, 0x13, 0x8a, 0x7b, 0xed, 0xcf,
      0x09, 0xb8, 0xad, 0xb5, 0xc7, 0xf0, 0x3e, 0x02, 0x42, 0x00, 0xd7, 0x3d, 0x75, 0x39, 0xe0,
      0x59, 0x37, 0x3f, 0xf0, 0xf2, 0xcb, 0x5c, 0xce, 0xb6, 0x5b, 0x71, 0x5e, 0x90, 0x00}},

    // Read final block.
    {{0x00, 0xb0, 0x03, 0xc0, 0x30},
     {0x46, 0x78, 0x50, 0xa8, 0x5b, 0x70, 0xfb, 0x96, 0xac, 0x90, 0x82, 0x84, 0x07,
      0x0e, 0x5e, 0xea, 0x66, 0x8e, 0xaf, 0x13, 0xf6, 0x99, 0xcc, 0xfd, 0xca, 0xf9,
      0xdf, 0x83, 0x48, 0xd2, 0xe7, 0x68, 0x10, 0xd2, 0xbf, 0x74, 0xe4, 0x07, 0xf4,
      0x6f, 0x0a, 0x7a, 0xbd, 0x5c, 0x4e, 0x6f, 0x15, 0x54, 0x90, 0x00}},

    // 2. PIN Retry count
    // Select QSCD AID.
    {{0x00, 0xA4, 0x04, 0x0C, 0x10, 0x51, 0x53, 0x43, 0x44, 0x20, 0x41,
      0x70, 0x70, 0x6C, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6F, 0x6E},
     {0x90, 0x00}},

    // Get retry count
    {{0x00, 0xcb, 0x3f, 0xff, 0x0a, 0x4d, 0x08, 0x70, 0x06, 0xbf, 0x81, 0x05, 0x02, 0xa0, 0x80,
      0x00},
     {0x70, 0x1e, 0xbf, 0x81, 0x05, 0x1a, 0xa0, 0x18, 0x9a, 0x01, 0x03, 0x9b,
      0x01, 0x03, 0xa1, 0x10, 0x8c, 0x06, 0xf3, 0x00, 0x00, 0x73, 0xff, 0x00,
      0x9c, 0x06, 0xf3, 0x00, 0x00, 0x73, 0xff, 0x00, 0x90, 0x00}},

    // 3. Signing.
    // Select QSCD AID.
    {{0x00, 0xA4, 0x04, 0x0C, 0x10, 0x51, 0x53, 0x43, 0x44, 0x20, 0x41,
      0x70, 0x70, 0x6C, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6F, 0x6E},
     {0x90, 0x00}},

    // Set ENV
    {{0x00, 0x22, 0x41, 0xb6, 0x06, 0x80, 0x01, 0x54, 0x84, 0x01, 0x9f}, {0x90, 0x00}},

    // Verify PIN.
    {{0x00, 0x20, 0x00, 0x85, 0x0c, 0x31, 0x32, 0x33, 0x34, 0x35, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff},
     {0x90, 0x00}},

    // Compute signature
    {{0x00, 0x2a, 0x9e, 0x9a, 0x30, 0x86, 0x25, 0x5f, 0xa2, 0xc3, 0x6e, 0x4b, 0x30, 0x96,
      0x9e, 0xae, 0x17, 0xdc, 0x34, 0xc7, 0x72, 0xcb, 0xeb, 0xdf, 0xc5, 0x8b, 0x58, 0x40,
      0x39, 0x00, 0xbe, 0x87, 0x61, 0x4e, 0xb1, 0xa3, 0x4b, 0x87, 0x80, 0x26, 0x3f, 0x25,
      0x5e, 0xb5, 0xe6, 0x5c, 0xa9, 0xbb, 0xb8, 0x64, 0x1c, 0xcc, 0xfe, 0x00},
     {0x6c, 0x33, 0xf7, 0xc4, 0x11, 0x5d, 0x67, 0x30, 0x7c, 0xeb, 0xbf, 0xd0, 0xe5, 0x74,
      0x21, 0x32, 0x87, 0xdf, 0x2a, 0x28, 0x79, 0x10, 0x9c, 0xcc, 0xbf, 0x11, 0x84, 0xa8,
      0xfd, 0xc6, 0x80, 0x0a, 0xb2, 0x15, 0x2a, 0x40, 0x87, 0x3d, 0xe9, 0x90, 0x77, 0x64,
      0xe6, 0x73, 0x8f, 0xd4, 0x3d, 0xb5, 0x37, 0x66, 0xcf, 0xfe, 0x65, 0x36, 0x3a, 0xae,
      0x32, 0x7a, 0x6c, 0xb3, 0xe0, 0x68, 0x63, 0x14, 0xe1, 0xe1, 0xf9, 0x80, 0x31, 0xd3,
      0x3a, 0xf5, 0x57, 0x1d, 0x83, 0x94, 0xe0, 0xc3, 0x4c, 0xca, 0x2e, 0x55, 0x43, 0xc0,
      0xc1, 0xdb, 0x4b, 0x6c, 0x98, 0x5c, 0x7c, 0xf5, 0x8c, 0x7d, 0x83, 0xa0, 0x90, 0x00}}};
