/* $Id: wmtempmon.c,v 1.8 2025/09/24 07:43:32 hacki Exp $ */

/*
 * Copyright (c) 2025 Marcus Glocker <marcus@nazgul.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <err.h>
#include <math.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/sensors.h>
#include <sys/sysctl.h>

#include <X11/xpm.h>

#include "xutils.h"
#include "bitmaps/wmtempmon_mask.xbm"
#include "bitmaps/wmtempmon_master.xpm"

/*
 * defines
 */
#define DELAY		20000
#define TSIDLENP	52
#define TSIDLENC	8

#define COLOR_BLUE	0
#define COLOR_ORANGE	12
#define COLOR_RED	24

#define MAX_DEVS	32
#define MAX_DEV_SENSORS	32

/*
 * prototypes
 */
int	main(int, char **);
void	usage(int);
void	signal_handler(const int);
void	draw_string(const char *, const int, const int);
void	draw_temp_device_name(char *, int);
void	draw_temp_value(const char *, const int, const int, int);
int	scroll_lcd(const int, const int, const int, const int, const int,
	    const char *);
int	scroll_bounce(const int, const int, const int, const int, const int,
	    const char *);
int	scroll_fade(const int, const int, const int, const int, const int,
	    const char *);

/*
 * global variables for this file
 */
static const char *version = "0.3";
volatile sig_atomic_t quit = 0;
char TimeColor[30] = "#ffff00";
char BackgroundColor[30] = "#181818";

struct coordinates {
	int x;
	int y;
	int w;
	int h;
};

struct coordinates fonts[128] = {
	{  60, 102,   6,   8 },	/* Dec   0 nul NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   1 soh NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   2 stx NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   3 etx NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   4 eot NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   5 enq NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   6 ack NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   7 bel NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   8  bs NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   9  ht NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  10  nl NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  11  vt NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  12  np NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  13  cr NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  14  so NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  15  si NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  16 dle NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  17 dc1 NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  18 dc2 NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  19 dc3 NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  20 dc4 NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  21 nak NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  22 syn NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  23 etb NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  24 can NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  25  em NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  26 sub NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  27 esc NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  28  fs NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  29  gs NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  30  rs NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  31  us NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  32  sp NOTUSED */
	{  66, 102,   6,   8 },	/* Dec  33 '!' */
	{  72, 102,   6,   8 },	/* Dec  34 '"' */
	{  78, 102,   6,   8 },	/* Dec  35 '#' */
	{  60, 102,   6,   8 }, /* Dec  36 '$' NOTUSED */
	{  90, 101,   6,   8 },	/* Dec  37 '%' */
	{  96, 102,   6,   8 },	/* Dec  38 '&' */
	{ 102, 102,   6,   8 },	/* Dec  39 ''' */
	{ 108, 102,   6,   8 },	/* Dec  40 '(' */
	{ 114, 102,   6,   8 },	/* Dec  41 ')' */
	{ 120, 102,   6,   8 },	/* Dec  42 '*' */
	{ 126, 102,   6,   8 },	/* Dec  43 '+' */
	{ 132, 102,   6,   8 },	/* Dec  44 ',' */
	{ 138, 102,   6,   8 }, /* Dec  45 '-' */
	{ 144, 102,   6,   8 }, /* Dec  46 '.' */
	{ 150, 102,   6,   8 }, /* Dec  47 '/' */
	{   0, 102,   6,   6 },	/* Dec  48 '0' */
	{   6, 102,   6,   6 },	/* Dec  49 '1' */
	{  12, 102,   6,   6 }, /* Dec  50 '2' */
	{  18, 102,   6,   6 }, /* Dec  51 '3' */
	{  24, 102,   6,   6 }, /* Dec  52 '4' */
	{  30, 102,   6,   6 },	/* Dec  53 '5' */
	{  36, 102,   6,   6 },	/* Dec  54 '6' */
	{  42, 102,   6,   6 },	/* Dec  55 '7' */
	{  48, 102,   6,   6 },	/* Dec  56 '8' */
	{  54, 102,   6,   6 },	/* Dec  57 '9' */
	{   0, 112,   6,   8 },	/* Dec  58 ':' */
	{   6, 113,   6,   8 },	/* Dec  59 ';' */
	{  12, 112,   6,   8 },	/* Dec  60 '<'*/
	{  18, 112,   6,   8 },	/* Dec  61 '='*/
	{  24, 112,   6,   8 },	/* Dec  62 '>'*/
	{  30, 112,   6,   8 },	/* Dec  63 '?'*/
	{  36, 112,   6,   8 },	/* Dec  64 '@'*/
	{   0,  79,   6,   8 },	/* Dec  65 'A' */
	{   6,  79,   6,   8 },	/* Dec  66 'B' */
	{  12,  79,   6,   8 },	/* Dec  67 'C' */
	{  18,  79,   6,   8 },	/* Dec  68 'D' */
	{  24,  79,   6,   8 },	/* Dec  69 'E' */
	{  30,  79,   6,   8 },	/* Dec  70 'F' */
	{  36,  79,   6,   8 },	/* Dec  71 'G' */
	{  42,  79,   6,   8 },	/* Dec  72 'H' */
	{  48,  79,   6,   8 },	/* Dec  73 'I' */
	{  54,  79,   6,   8 },	/* Dec  74 'J' */
	{  60,  79,   6,   8 },	/* Dec  75 'K' */
	{  66,  79,   6,   8 },	/* Dec  76 'L' */
	{  72,  79,   6,   8 },	/* Dec  77 'M' */
	{  78,  79,   6,   8 },	/* Dec  78 'N' */
	{  84,  79,   6,   8 },	/* Dec  79 'O' */
	{  90,  79,   6,   8 },	/* Dec  80 'P' */
	{  96,  79,   6,   8 },	/* Dec  81 'Q' */
	{ 102,  79,   6,   8 },	/* Dec  82 'R' */
	{ 108,  79,   6,   8 },	/* Dec  83 'S' */
	{ 114,  79,   6,   8 },	/* Dec  84 'T' */
	{ 120,  79,   6,   8 },	/* Dec  85 'U' */
	{ 126,  79,   6,   8 },	/* Dec  86 'V' */
	{ 132,  79,   6,   8 },	/* Dec  87 'W' */
	{ 138,  79,   6,   8 },	/* Dec  88 'X' */
	{ 144,  79,   6,   8 },	/* Dec  89 'Y' */
	{ 150,  79,   6,   8 },	/* Dec  90 'Z' */
	{  42, 112,   6,   8 },	/* Dec  91 '[' */
	{  48, 112,   6,   8 },	/* Dec  92 '\' */
	{  54, 112,   6,   8 },	/* Dec  93 ']' */
	{  60, 112,   6,   8 },	/* Dec  94 '^' */
	{  66, 113,   6,   8 },	/* Dec  95 '_' */
	{  72, 112,   6,   8 },	/* Dec  96 '`' */
	{   0,  90,   6,   8 },	/* Dec  97 'a' */
	{   6,  90,   6,   8 },	/* Dec  98 'b' */
	{  12,  90,   6,   8 },	/* Dec  99 'c' */
	{  18,  90,   6,   8 },	/* Dec 100 'd' */
	{  24,  90,   6,   8 },	/* Dec 101 'e' */
	{  30,  90,   6,   8 },	/* Dec 102 'f' */
	{  36,  90,   6,   8 },	/* Dec 103 'g' */
	{  42,  90,   6,   8 },	/* Dec 104 'h' */
	{  48,  90,   6,   8 },	/* Dec 105 'i' */
	{  54,  90,   6,   8 },	/* Dec 106 'j' */
	{  60,  90,   6,   8 },	/* Dec 107 'k' */
	{  66,  90,   6,   8 },	/* Dec 108 'l' */
	{  72,  90,   6,   8 },	/* Dec 109 'm' */
	{  78,  90,   6,   8 },	/* Dec 110 'n' */
	{  84,  90,   6,   8 },	/* Dec 111 'o' */
	{  90,  90,   6,   8 },	/* Dec 112 'p' */
	{  96,  90,   6,   8 },	/* Dec 113 'q' */
	{ 102,  90,   6,   8 },	/* Dec 114 'r' */
	{ 108,  90,   6,   8 },	/* Dec 115 's' */
	{ 114,  90,   6,   8 },	/* Dec 116 't' */
	{ 120,  90,   6,   8 },	/* Dec 117 'u' */
	{ 126,  90,   6,   8 },	/* Dec 118 'v' */
	{ 132,  90,   6,   8 },	/* Dec 119 'w' */
	{ 138,  90,   6,   8 },	/* Dec 120 'x' */
	{ 144,  90,   6,   8 },	/* Dec 121 'y' */
	{ 150,  90,   6,   8 },	/* Dec 122 'z' */
	{  78, 112,   6,   8 },	/* Dec 123 '{' */
	{  84, 112,   6,   8 },	/* Dec 124 '|' */
	{  90, 112,   6,   8 },	/* Dec 125 '}' */
	{  60, 112,   6,   8 },	/* Dec 126 '~' */
	{  60, 102,   6,   8 }	/* Dec 127 del NOTUSED */
};

struct coordinates fontsl[128] = {
	{  60, 102,   6,   8 },	/* Dec   0 nul NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   1 soh NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   2 stx NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   3 etx NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   4 eot NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   5 enq NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   6 ack NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   7 bel NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   8  bs NOTUSED */
	{  60, 102,   6,   8 },	/* Dec   9  ht NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  10  nl NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  11  vt NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  12  np NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  13  cr NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  14  so NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  15  si NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  16 dle NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  17 dc1 NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  18 dc2 NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  19 dc3 NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  20 dc4 NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  21 nak NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  22 syn NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  23 etb NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  24 can NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  25  em NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  26 sub NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  27 esc NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  28  fs NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  29  gs NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  30  rs NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  31  us NOTUSED */
	{  60, 102,   6,   8 },	/* Dec  32  sp NOTUSED */
	{  66, 102,   6,   8 },	/* Dec  33 '!' */
	{  72, 102,   6,   8 },	/* Dec  34 '"' */
	{  78, 102,   6,   8 },	/* Dec  35 '#' */
	{  60, 102,   6,   8 }, /* Dec  36 '$' NOTUSED */
	{  90, 101,   6,   8 },	/* Dec  37 '%' */
	{  96, 102,   6,   8 },	/* Dec  38 '&' */
	{ 102, 102,   6,   8 },	/* Dec  39 ''' */
	{ 108, 102,   6,   8 },	/* Dec  40 '(' */
	{ 114, 102,   6,   8 },	/* Dec  41 ')' */
	{ 120, 102,   6,   8 },	/* Dec  42 '*' */
	{ 126, 102,   6,   8 },	/* Dec  43 '+' */
	{ 132, 102,   6,   8 },	/* Dec  44 ',' */
	{  80, 126,   8,  12 }, /* Dec  45 '-' */
	{ 152,   0,   4,  12 }, /* Dec  46 '.' */
	{ 150, 102,   6,   8 }, /* Dec  47 '/' */
	{  72,   0,   8,  12 }, /* Dec  48 '0' */
	{  80,   0,   8,  12 }, /* Dec  49 '1' */
	{  88,   0,   8,  12 }, /* Dec  50 '2' */
	{  96,   0,   8,  12 }, /* Dec  51 '3' */
	{ 104,   0,   8,  12 }, /* Dec  52 '4' */
	{ 112,   0,   8,  12 }, /* Dec  53 '5' */
	{ 120,   0,   8,  12 }, /* Dec  54 '6' */
	{ 128,   0,   8,  12 }, /* Dec  55 '7' */
	{ 136,   0,   8,  12 }, /* Dec  56 '8' */
	{ 144,   0,   8,  12 }, /* Dec  57 '9' */
	{   0, 112,   6,   8 },	/* Dec  58 ':' */
	{   6, 113,   6,   8 },	/* Dec  59 ';' */
	{  12, 112,   6,   8 },	/* Dec  60 '<'*/
	{  18, 112,   6,   8 },	/* Dec  61 '='*/
	{  24, 112,   6,   8 },	/* Dec  62 '>'*/
	{  30, 112,   6,   8 },	/* Dec  63 '?'*/
	{  36, 112,   6,   8 },	/* Dec  64 '@'*/
	{   0,  79,   6,   8 },	/* Dec  65 'A' */
	{   6,  79,   6,   8 },	/* Dec  66 'B' */
	{  12,  79,   6,   8 },	/* Dec  67 'C' */
	{  18,  79,   6,   8 },	/* Dec  68 'D' */
	{  24,  79,   6,   8 },	/* Dec  69 'E' */
	{  30,  79,   6,   8 },	/* Dec  70 'F' */
	{  36,  79,   6,   8 },	/* Dec  71 'G' */
	{  42,  79,   6,   8 },	/* Dec  72 'H' */
	{  48,  79,   6,   8 },	/* Dec  73 'I' */
	{  54,  79,   6,   8 },	/* Dec  74 'J' */
	{  60,  79,   6,   8 },	/* Dec  75 'K' */
	{  66,  79,   6,   8 },	/* Dec  76 'L' */
	{  72,  79,   6,   8 },	/* Dec  77 'M' */
	{  78,  79,   6,   8 },	/* Dec  78 'N' */
	{  84,  79,   6,   8 },	/* Dec  79 'O' */
	{  90,  79,   6,   8 },	/* Dec  80 'P' */
	{  96,  79,   6,   8 },	/* Dec  81 'Q' */
	{ 102,  79,   6,   8 },	/* Dec  82 'R' */
	{ 108,  79,   6,   8 },	/* Dec  83 'S' */
	{ 114,  79,   6,   8 },	/* Dec  84 'T' */
	{ 120,  79,   6,   8 },	/* Dec  85 'U' */
	{ 126,  79,   6,   8 },	/* Dec  86 'V' */
	{ 132,  79,   6,   8 },	/* Dec  87 'W' */
	{ 138,  79,   6,   8 },	/* Dec  88 'X' */
	{ 144,  79,   6,   8 },	/* Dec  89 'Y' */
	{ 150,  79,   6,   8 },	/* Dec  90 'Z' */
	{  42, 112,   6,   8 },	/* Dec  91 '[' */
	{  48, 112,   6,   8 },	/* Dec  92 '\' */
	{  54, 112,   6,   8 },	/* Dec  93 ']' */
	{  60, 112,   6,   8 },	/* Dec  94 '^' */
	{  66, 113,   6,   8 },	/* Dec  95 '_' */
	{  72, 112,   6,   8 },	/* Dec  96 '`' */
	{   0,  90,   6,   8 },	/* Dec  97 'a' */
	{   6,  90,   6,   8 },	/* Dec  98 'b' */
	{  12,  90,   6,   8 },	/* Dec  99 'c' */
	{  18,  90,   6,   8 },	/* Dec 100 'd' */
	{  24,  90,   6,   8 },	/* Dec 101 'e' */
	{  30,  90,   6,   8 },	/* Dec 102 'f' */
	{  36,  90,   6,   8 },	/* Dec 103 'g' */
	{  42,  90,   6,   8 },	/* Dec 104 'h' */
	{  48,  90,   6,   8 },	/* Dec 105 'i' */
	{  54,  90,   6,   8 },	/* Dec 106 'j' */
	{  60,  90,   6,   8 },	/* Dec 107 'k' */
	{  66,  90,   6,   8 },	/* Dec 108 'l' */
	{  72,  90,   6,   8 },	/* Dec 109 'm' */
	{  78,  90,   6,   8 },	/* Dec 110 'n' */
	{  84,  90,   6,   8 },	/* Dec 111 'o' */
	{  90,  90,   6,   8 },	/* Dec 112 'p' */
	{  96,  90,   6,   8 },	/* Dec 113 'q' */
	{ 102,  90,   6,   8 },	/* Dec 114 'r' */
	{ 108,  90,   6,   8 },	/* Dec 115 's' */
	{ 114,  90,   6,   8 },	/* Dec 116 't' */
	{ 120,  90,   6,   8 },	/* Dec 117 'u' */
	{ 126,  90,   6,   8 },	/* Dec 118 'v' */
	{ 132,  90,   6,   8 },	/* Dec 119 'w' */
	{ 138,  90,   6,   8 },	/* Dec 120 'x' */
	{ 144,  90,   6,   8 },	/* Dec 121 'y' */
	{ 150,  90,   6,   8 },	/* Dec 122 'z' */
	{  78, 112,   6,   8 },	/* Dec 123 '{' */
	{  84, 112,   6,   8 },	/* Dec 124 '|' */
	{  90, 112,   6,   8 },	/* Dec 125 '}' */
	{  60, 112,   6,   8 },	/* Dec 126 '~' */
	{  60, 102,   6,   8 }	/* Dec 127 del NOTUSED */
};

/*
 * usage
 */
void
usage(int mode)
{
	extern char *__progname;

	if (mode) {
		fprintf(stderr, "%s %s\n", __progname, version);
		exit(1);
	}

	fprintf(stderr, "usage: %s ", __progname);
	fprintf(stderr, "[-hv] [-t temp_device] [-n temp_device_sensor_num] ");
	fprintf(stderr, "[-s scrolling] [-display display]\n\n");
	fprintf(stderr, "options:\n");
	fprintf(stderr, "  -h\t\t: This help.\n");
	fprintf(stderr, "  -v\t\t: Shows version.\n");
	fprintf(stderr, "  -t\t\t: Select temperature device.\n");
	fprintf(stderr, "  -n\t\t: Select temperature device sensor number.\n");
	fprintf(stderr, "  -s\t\t: Select scrolling type:\n");
	fprintf(stderr, "    \t\t  0 = Disable 1 = LCD (default) 2 = Bounce ");
	fprintf(stderr, "3 = Fade\n");
	fprintf(stderr, "  -display\t: Set display name.\n");

	exit(1);
}

/*
 * signal handler
 */
void
signal_handler(const int sig)
{
	switch (sig) {
	case SIGINT:
		/* FALLTHROUGH */
	case SIGTERM:
		quit = 1;
		break;
	case SIGCHLD:
		/* ignore */
		break;
	case SIGHUP:
		/* ignore */
		break;
	case SIGQUIT:
		/* ignore */
		break;
	case SIGALRM:
		/* ignore */
		break;
	case SIGPIPE:
		/* ignore */
		break;
	default:
		/* ignore */
		break;
	}
}

/*
 * main()
 *	temperature sensor monitor wmdockapp
 * Return:
 *	0
 */
int
main(int argc, char *argv[])
{
	int dev, ch, i, off;
	int temp_device_found, opt_temp_device_sensor_num, opt_scroll;
	int mib[5];
	size_t slen, sdlen;
	float tempi;
	char *opt_temp_device, *opt_display;
	char temps[8], temp_name_full[32];
	const char *errstr;
	struct sensordev sensordev;
	struct sensor sensor;
	XEvent event;

	off = 0;
	temp_device_found = 0;
	opt_temp_device_sensor_num = 0;
	opt_scroll = 1;
	opt_temp_device = NULL;
	opt_display = NULL;

	/*
	 * get command line options
	 */

	/* keep the old good -display tradition */
	for (i = 0; i < argc; i++) {
		if (!strcmp(argv[i], "-display"))
			argv[i] = "-d";
	}

	while ((ch = getopt(argc, argv, "d:hn:t:s:v")) != -1) {
		switch (ch) {
		case 'd':
			opt_display = strdup(optarg);
			break;
		case 'n':
			opt_temp_device_sensor_num =
			    strtonum(optarg, 0, MAX_DEV_SENSORS, &errstr);
			if (errstr)
				usage(0);
			break;
		case 't':
			opt_temp_device = strdup(optarg);
			break;
		case 's':
			opt_scroll = strtonum(optarg, 0, 3, &errstr);
			if (errstr)
				usage(0);
			break;
		case 'v':
			usage(1);
			break;
		case 'h':
			/* FALLTHROUGH */
		default:
			usage(0);
			/* NOTREACHED */
		}
	}

	mib[0] = CTL_HW;
	mib[1] = HW_SENSORS;
	for (dev = 0; dev < MAX_DEVS; dev++) {
		mib[2] = dev;
		sdlen = sizeof(struct sensordev);
		if (sysctl(mib, 3, &sensordev, &sdlen, NULL, 0) == -1)
			break;

		mib[3] = SENSOR_TEMP;
		mib[4] = 0;
		slen = sizeof(struct sensor);
		if (sysctl(mib, 5, &sensor, &slen, NULL, 0) == -1)
			continue;

		if (opt_temp_device == NULL) {
			opt_temp_device = strdup(sensordev.xname);
			temp_device_found = 1;
			break;
		} else if (strcmp(opt_temp_device, sensordev.xname) == 0) {
			temp_device_found = 1;
			break;
		}
	}

	if (!temp_device_found) {
		printf("No temperature device found!\n");
		return 1;
	}

	mib[4] = opt_temp_device_sensor_num;
	if (sysctl(mib, 5, &sensor, &slen, NULL, 0) == -1) {
		printf("%s: temperature device sensor number %d not found!\n",
		    opt_temp_device, opt_temp_device_sensor_num);
		return 1;
	}	
	snprintf(temp_name_full, sizeof(temp_name_full), "%s.temp%d",
	    opt_temp_device, opt_temp_device_sensor_num);

	/*
	 * install signal handler
	 */
	signal(SIGINT, signal_handler);
	signal(SIGTERM, signal_handler);
	signal(SIGCHLD, signal_handler);
	signal(SIGHUP, signal_handler);
	signal(SIGQUIT, signal_handler);
	signal(SIGALRM, signal_handler);
	signal(SIGPIPE, signal_handler);

	/*
	 * init X window
	 */
	initXwindow(opt_display);

	/*
	 * open X window
	 */
	openXwindow(argc, argv, wmtemp_master_xpm, wmtemp_mask_bits,
	    wmtemp_mask_width, wmtemp_mask_height);

	/*
	 * main loop
	 */
	for (i = 100; !quit; i++) {
		/*
		 * get temperature sensor state and draw it
		 */
		if (i == 100) {
			/* reset all */
			i = 0;
			copyXPMArea(0, 0, 64, 64, 0, 0);

			/* draw temperature value */
			if (sysctl(mib, 5, &sensor, &slen, NULL, 0) == -1) {
				snprintf(temps, sizeof(temps), "000.00");
			} else {
				tempi = (float)(sensor.value - 273150000) /
				    1000000;
				snprintf(temps, sizeof(temps), "%3.2f",
				    tempi);
			}
			if (tempi > 49) {
				draw_temp_value(temps, 10, 35, COLOR_ORANGE);
			} else if (tempi > 74) {
				draw_temp_value(temps, 10, 35, COLOR_RED);
			} else {
				draw_temp_value(temps, 10, 35, COLOR_BLUE);
			}
			copyXPMArea(122, 37, 14, 9, 23, 48);
		}

		/* draw temperature device name */
		copyXPMArea(5, 7, 54, 11, 5, 7);
		draw_temp_device_name(temp_name_full, opt_scroll);

		/* draw temperature device description */
		copyXPMArea(5, 22, 54, 11, 5, 22);
		off = scroll_lcd(0, 65, 6, 21, off, sensor.desc);

		/*
	 	 * process pending X events
	 	 */
		while (XPending(display)) {
			XNextEvent(display, &event);
			switch(event.type) {
			case Expose:
				RedrawWindow();
				break;
			}
		}

		/*
		 * redraw
		 */
		RedrawWindow();
		usleep(DELAY);
	}

	exit(0);
}

/*
 * draw string
 */
void
draw_string(const char *string, const int x, const int y)
{
	int	c, i, offset;

	for (i = 0, offset = x; string[i] != '\0'; i++) {
		c = string[i];

		copyXPMArea(fonts[c].x, fonts[c].y, fonts[c].w, fonts[c].h,
		    offset, y);
		offset += fonts[c].w;
	}
}

/*
 * draw temperature device name
 */
void
draw_temp_device_name(char *devid, int scroll)
{
	static int	offset = 0;
	static char	*save_devid;
	char		 ts_id_cut[TSIDLENC + 1];

	if (strlen(devid) > TSIDLENC) {
		if (save_devid == NULL || strcmp(devid, save_devid)) {
			free(save_devid);
			if ((save_devid = strdup(devid)) == NULL)
				err(1, NULL);
			offset = 0;
		}
		switch (scroll) {
		case 0:
			/* off */
			strlcpy(ts_id_cut, devid, sizeof(ts_id_cut));
			draw_string(ts_id_cut, 6, 7);
			break;
		case 1:
			/* lcd */
			offset = scroll_lcd(0, 124, 6, 7, offset, devid);
			break;
		case 2:
			/* bounce */
			offset = scroll_bounce(0, 124, 6, 7, offset, devid);
			break;
		case 3:
			/* fade */
			offset = scroll_fade(0, 124, 6, 7, offset, devid);
			break;
		}
	} else
		draw_string(devid, 6, 7);
}

/*
 * draw temperature value
 */
void
draw_temp_value(const char *string, const int x, const int y, int color)
{
	int c, i, offset;

	for (i = 0, offset = x; string[i] != '\0'; i++) {
		c = string[i];

		copyXPMArea(fontsl[c].x, fontsl[c].y + color, fontsl[c].w,
		    fontsl[c].h, offset, y);
		offset += fontsl[c].w;
	}
}

/*
 * scroll_lcd()
 *	scrolls a string from left to right separated by a space
 * Return:
 *	offset = success
 */
int
scroll_lcd(const int src_x, const int src_y, const int dst_x,
    const int dst_y, const int offset, const char *devid)
{
	int len, pos, frame;

	/* copy offset */
	pos = offset;

	/* draw temperature sensor name */
	if (pos == src_x) {
		copyXPMArea(0, 132, 160, 8, src_x, src_y);
		draw_string(devid, src_x, src_y);
	}

	/* calculate image length */
	len = (strlen(devid) + 1) * 6;

	/* reached end of source, reset */
	if (pos == len)
		pos = src_x;

	/* copy scroll frame to destination */
	frame = len - pos;
	if (frame >= TSIDLENP) 
		copyXPMArea(pos, src_y, TSIDLENP, 8, dst_x, dst_y);
	else {
		copyXPMArea(pos, src_y, frame, 8, dst_x, dst_y);
		copyXPMArea(src_x, src_y, TSIDLENP - frame, 8, dst_x + frame,
		    dst_y);
	}

	/* move to next pixel */
	pos++;

	return pos;
}

/*
 * scroll_bounce()
 *	scrolls a string from left to right and bounces back from right to left
 * Return:
 *	offset = success
 */
int
scroll_bounce(const int src_x, const int src_y, const int dst_x,
    const int dst_y, const int offset, const char *devid)
{
	int len, pos, frame;
	static int delay = 0, direction = 0;

	/* copy offset */
	pos = offset;

	/* draw temperature sensor name */
	if (pos == src_x) {
		copyXPMArea(0, 132, 160, 8, src_x, src_y);
		draw_string(devid, src_x, src_y);
	}

	/* calculate image length */
	len = strlen(devid) * 6;

	/* delay */
	if (delay > 0) {
		if (direction == 0)
			copyXPMArea(pos - 1, src_y, TSIDLENP, 8, dst_x, dst_y);
		if (direction == 1)
			copyXPMArea(pos + 1, src_y, TSIDLENP, 8, dst_x, dst_y);
		delay--;

		return pos;
	}

	/* start */
	if (pos == src_x) {
		delay = 10;
		direction = 0;
	}
	/* end */
	frame = len - pos;
	if (frame == TSIDLENP) {
		delay = 10;
		direction = 1;
	}

	/* copy scroll frame to destination */
	copyXPMArea(pos, src_y, TSIDLENP, 8, dst_x, dst_y);

	/* move to next pixel */
	if (direction == 0)
		pos++;
	if (direction == 1)
		pos--;

	return pos;
}

/*
 * scroll_fade()
 *	scrolls a string from left to right, fading it out and begins again
 * Return:
 *	offset = success
 */
int
scroll_fade(const int src_x, const int src_y, const int dst_x,
    const int dst_y, const int offset, const char *devid)
{
	int len, pos, frame;
	static int delay = 0;

	/* copy offset */
	pos = offset;

	/* draw temperature sensor name */
	if (pos == src_x) {
		copyXPMArea(0, 132, 160, 8, src_x, src_y);
		draw_string(devid, src_x, src_y);
	}

	/* calculate image length */
	len = (strlen(devid) + 1) * 6;

	/* delay */
	if (delay > 0) {
		copyXPMArea(pos - 1, src_y, TSIDLENP, 8, dst_x, dst_y);
		delay--;

		return pos;
	}

	/* end */
	if (pos == len)
		pos = src_x;
	/* start */
	if (pos == src_x)
		delay = 10;

	/* copy scroll frame to destination */
	frame = len - pos;
	if (frame >= TSIDLENP)
		copyXPMArea(pos, src_y, TSIDLENP, 8, dst_x, dst_y);
	else
		copyXPMArea(pos, src_y, frame, 8, dst_x, dst_y);

	/* move to next pixel */
	pos++;

	return pos;
}
