/*	$Id: login_fingerprint.c,v 1.3 2009/05/30 15:23:51 robert Exp $ */
/*
 * Copyright (c) 2009 Robert Nagy <robert@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "common.h"

#define AUTH_OK		0
#define AUTH_FAILED	-1

#define MAX_SWIPES	3

/* login.conf strings */
#define CAP_FINGERPRINT		"x-fingerprint"

FILE *back = NULL;

int
verify(struct fp_dev *dev, struct fp_print_data *data, int fnum)
{
	int r, i;

	for (i = 0; i < MAX_SWIPES; i++) {
		printf("Swipe your %s finger: ", fingernames[fnum]);
		fflush(stdout);
		r = fp_verify_finger(dev, data);
		if (r < 0) {  
			printf("verification failed with error %d :(\n", r);
			return r;
		}
                switch (r) {
                case FP_VERIFY_NO_MATCH:
                        printf("Login incorrect\n");
                        return (-1);
                case FP_VERIFY_MATCH:
			printf("Okay\n");
                        return (0);
                case FP_VERIFY_RETRY:
                        printf("Scan didn't quite work. Please try again.\n");
                        break;
                case FP_VERIFY_RETRY_TOO_SHORT:
                        printf("Swipe was too short, please try again.\n");
                        break;
                case FP_VERIFY_RETRY_CENTER_FINGER:
                        printf("Please center your finger on the sensor and try again.\n");
                        break;
                case FP_VERIFY_RETRY_REMOVE_FINGER:
                        printf("Please remove finger from the sensor and try again.\n");
                        break;
                }
        }
	return (-1);
}

int
main(int argc, char **argv)
{
	struct fp_dscv_dev *ddev;
	struct fp_dscv_dev **discovered_devs;
	struct fp_dev *dev;
	struct fp_print_data *data;
	login_cap_t	*lc;
	struct passwd	*pe;
	
	int opt, ret, r = 1, fnum;
	char *username, *finger;
	char invokinguser[MAXLOGNAME];
	char *class = NULL;
	char *service;
	const char *errstr;

	invokinguser[0] = '\0';

	setpriority(PRIO_PROCESS, 0, 0);

	openlog(NULL, LOG_ODELAY, LOG_AUTH);

	while ((opt = getopt(argc, argv, "ds:v:")) != -1) {
		switch (opt) {
		case 'd':
			back = stdout;
			break;
		case 's':	/* service */
			service = optarg;
			if (strcmp(service, "login") != 0 &&
			    strcmp(service, "challenge") != 0 &&
			    strcmp(service, "response") != 0)
				errx(1, "%s not supported", service);
                        break;
		case 'v':
			break;
		default:
			syslog(LOG_ERR, "usage error1");
			exit(1);
		}
	}

	switch (argc - optind) {
	case 2:
		class = argv[optind + 1];
		/*FALLTHROUGH*/
	case 1:
		username = argv[optind];
		break;
	default:
		syslog(LOG_ERR, "usage error2");
		exit(1);
	}

	if (back == NULL && (back = fdopen(3, "r+")) == NULL) {
		syslog(LOG_ERR, "reopening back channel: %m");
		exit(1);
	}

	ret = AUTH_FAILED;

	lc = login_getclass(class);
	if (lc == NULL) {
		syslog(LOG_ERR, "couldn't find %s in login.conf", class);
		exit(1);
	}

	finger = login_getcapstr(lc, CAP_FINGERPRINT, NULL, NULL);
	if (finger == NULL) {
		syslog(LOG_ERR, "you need to set " CAP_FINGERPRINT " for %s", lc->lc_class);
		exit(1);
	}

	pe = getpwnam(username);
	if (pe == NULL) {
		syslog(LOG_ERR, "unknown user: %s", username);
		exit(1);
	}

	if (strcmp(service, "challenge") == 0) {
		printf(back, BI_SILENT "\n");
		exit(0);
	}

	r = fp_init();
	if (r < 0) {
		syslog(LOG_ERR, "failed to initialize\n");
		exit(1);
	}

	discovered_devs = fp_discover_devs();
	if (!discovered_devs) {
		syslog(LOG_ERR, "failed to discover devices\n");
		exit(1);
	}

	ddev = discover_device(discovered_devs);
	if (!ddev) {
		syslog(LOG_ERR, "unable to find a fingerprint reader\n");
		exit(1);
	}

	dev = fp_dev_open(ddev);
	fp_dscv_devs_free(discovered_devs);
	if (!dev) {
		syslog(LOG_ERR, "unable to open device\n");
		exit(1);
	}

	fnum = strtonum(finger, FINGER_MIN, FINGER_MAX, &errstr);
	if (errstr)
		errx(1, "The finger's number is %s: %s", errstr, optarg);

	setenv("HOME", pe->pw_dir, 1);

	r = fp_print_data_load(dev, fnum, &data);
	if (r != 0) {
		syslog(LOG_ERR, "failed to load fingerprint: %s\n", strerror(r));
		exit(1);
	}
	
	unsetenv("HOME");

	if (verify(dev, data, fnum) == 0)
		ret = AUTH_OK;

	fp_print_data_free(data);
	fp_dev_close(dev);
	fp_exit();

	if (ret != AUTH_OK)
		fprintf(back, BI_REJECT "\n");
	else
		fprintf(back, BI_AUTH "\n");

	closelog();

	exit(0);
}
