//
// Copyright 2025 Signal Messenger, LLC.
// SPDX-License-Identifier: AGPL-3.0-only
//

import XCTest

@testable import LibSignalClient

// These testing endpoints aren't generated in device builds, to save on code size.
#if !os(iOS) || targetEnvironment(simulator)

class UnauthUsernamesServiceTests: UnauthChatServiceTestBase<any UnauthUsernamesService> {
    override class var selector: SelectorCheck { .usernames }

    func testUsernameLookup() async throws {
        let api = self.api
        async let responseFuture = api.lookUpUsernameHash(Data([1, 2, 3, 4]))

        let (request, id) = try await fakeRemote.getNextIncomingRequest()
        XCTAssertEqual(request.method, "GET")

        let uuid = UUID(uuidString: "4FCFE887-A600-40CD-9AB7-FD2A695E9981")!

        try fakeRemote.sendResponse(
            requestId: id,
            ChatResponse(
                status: 200,
                headers: ["content-type": "application/json"],
                body: Data(
                    """
                    {
                        "uuid": "\(uuid)"
                    }
                    """.utf8
                )
            )
        )

        let responseFromServer = try await responseFuture
        XCTAssertEqual(responseFromServer, Aci(fromUUID: uuid))
    }

    func testUsernameLookupMissing() async throws {
        let api = self.api
        async let responseFuture = api.lookUpUsernameHash(Data([1, 2, 3, 4]))

        let (request, id) = try await fakeRemote.getNextIncomingRequest()
        XCTAssertEqual(request.method, "GET")

        try fakeRemote.sendResponse(
            requestId: id,
            ChatResponse(status: 404)
        )

        let responseFromServer = try await responseFuture
        XCTAssertNil(responseFromServer)
    }

    func testChallengeError() async throws {
        let api = self.api
        async let responseFuture = api.lookUpUsernameHash(Data([1, 2, 3, 4]))

        let (request, id) = try await fakeRemote.getNextIncomingRequest()
        XCTAssertEqual(request.method, "GET")

        try fakeRemote.sendResponse(
            requestId: id,
            ChatResponse(
                status: 428,
                headers: ["content-type": "application/json"],
                body: Data(
                    """
                    {
                        "token": "not-legal-tender",
                        "options": ["pushChallenge"]
                    }
                    """.utf8
                )
            )
        )

        do {
            _ = try await responseFuture
            XCTFail("should have failed")
        } catch SignalError.rateLimitChallengeError(let token, let options, _) {
            XCTAssertEqual(token, "not-legal-tender")
            XCTAssertEqual(options, [.pushChallenge])
        }
    }

    func testServerSideError() async throws {
        let api = self.api
        async let responseFuture = api.lookUpUsernameHash(Data([1, 2, 3, 4]))

        let (request, id) = try await fakeRemote.getNextIncomingRequest()
        XCTAssertEqual(request.method, "GET")

        try fakeRemote.sendResponse(
            requestId: id,
            ChatResponse(status: 500)
        )

        do {
            _ = try await responseFuture
            XCTFail("should have failed")
        } catch SignalError.networkProtocolError(_) {
        }
    }
}

#endif
