//
// Copyright 2024 Signal Messenger, LLC.
// SPDX-License-Identifier: AGPL-3.0-only
//

import Foundation
import SignalFfi

/// A token representing a particular ``GroupSendEndorsement``, along with the endorsement's
/// expiration.
///
/// Generated by ``GroupSendEndorsement/Token/toFullToken(expiration:)``, and verified by the chat
/// server.
public class GroupSendFullToken: ByteArray, @unchecked Sendable {
    public required init(contents: Data) throws {
        try super.init(contents, checkValid: signal_group_send_full_token_check_valid_contents)
    }

    /// The expiration embedded in the token.
    public var expiration: Date {
        let expirationSinceEpoch: UInt64 = failOnError {
            try withUnsafeBorrowedBuffer { response in
                try invokeFnReturningInteger {
                    signal_group_send_full_token_get_expiration($0, response)
                }
            }
        }
        return Date(timeIntervalSince1970: TimeInterval(expirationSinceEpoch))
    }

    /// Verifies that this token was generated from an endorsement of `userIds` by `keyPair`.
    ///
    /// The correct `keyPair` must be selected based on ``expiration``.
    ///
    /// - Throws: ``SignalError/verificationFailed(_:)`` if the token is invalid
    public func verify(
        userIds: [ServiceId],
        now: Date = Date(),
        keyPair: GroupSendDerivedKeyPair
    ) throws {
        try withUnsafeBorrowedBuffer { token in
            try ServiceId.concatenatedFixedWidthBinary(userIds).withUnsafeBorrowedBuffer { userIds in
                try keyPair.withUnsafeBorrowedBuffer { keyPair in
                    try checkError(
                        signal_group_send_full_token_verify(token, userIds, UInt64(now.timeIntervalSince1970), keyPair)
                    )
                }
            }
        }
    }
}
