//
// Copyright 2025 Signal Messenger, LLC.
// SPDX-License-Identifier: AGPL-3.0-only
//

//! Converts an Ed25519 public key in PEM format (e.g. as produced by `openssl pkey`) to an XEd25519
//! public key in libsignal's type-byte-prefixed format (hex-encoded).
//!
//! > XEd25519 signatures are valid Ed25519 signatures and vice versa, provided the public keys are
//! > converted with the birational map.
//!
//! from <https://signal.org/docs/specifications/xeddsa/#curve25519>. This tool implements that map
//! (in one direction). Note that libsignal's implementation of XEd25519 diverges from the spec on
//! signal.org in one facet: it accomodates Ed25519 signing keys with a sign bit of 1 by expecting
//! the sign bit of the *signature* to be set to 1 as well.

use std::io::Read;

fn main() {
    let mut input = vec![];
    std::io::stdin()
        .read_to_end(&mut input)
        .expect("can read input");
    let key = boring::pkey::PKey::public_key_from_pem(&input).expect("valid PEM (not DER!)");

    const ED25519_PUBLIC_KEY_LENGTH: usize = 32;
    let mut key_bytes = [0u8; ED25519_PUBLIC_KEY_LENGTH];
    let used_key_bytes = key
        .raw_public_key(&mut key_bytes)
        .expect("can extract public key")
        .len();
    assert_eq!(ED25519_PUBLIC_KEY_LENGTH, used_key_bytes);

    let montgomery_form = curve25519_dalek::edwards::CompressedEdwardsY(key_bytes)
        .decompress()
        .expect("valid Edwards point")
        .to_montgomery();
    let xed_key = libsignal_core::curve::PublicKey::from_djb_public_key_bytes(&montgomery_form.0)
        .expect("valid libsignal key");
    println!("     key: {}", hex::encode(xed_key.serialize()));

    let sign_bit = key_bytes[ED25519_PUBLIC_KEY_LENGTH - 1] >> 7;
    print!("sign bit: {sign_bit} - ");
    if sign_bit == 0 {
        println!("signatures generated by this key will be directly validatable by libsignal");
    } else {
        println!(
            "set the sign bit (0x80 in the final byte) of signatures generated by this key for validation by libsignal"
        );
    }
}
