//
// Copyright 2021 Signal Messenger, LLC.
// SPDX-License-Identifier: AGPL-3.0-only
//

use std::cell::RefCell;
use std::sync::Arc;

use async_trait::async_trait;
use signal_neon_futures::*;
use uuid::Uuid;

use super::*;

pub struct NodePreKeyStore {
    js_channel: Channel,
    store_object: Arc<Root<JsObject>>,
}

impl NodePreKeyStore {
    pub(crate) fn new(cx: &mut FunctionContext, store: Handle<JsObject>) -> Self {
        Self {
            js_channel: cx.channel(),
            store_object: Arc::new(store.root(cx)),
        }
    }

    async fn do_get_pre_key(&self, id: u32) -> Result<PreKeyRecord, String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let id = id.convert_into(cx)?;
            let result = call_method(cx, store_object, "_getPreKey", [id.upcast()])?;
            let result = result.downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<DefaultJsBox<PreKeyRecord>, _>(cx) {
                Ok(obj) => Ok((***obj).clone()),
                Err(_) => Err("result must be an object".to_owned()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }

    async fn do_save_pre_key(&self, id: u32, record: PreKeyRecord) -> Result<(), String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let id: Handle<JsNumber> = id.convert_into(cx)?;
            let record: Handle<JsValue> = record.convert_into(cx)?;
            let result = call_method(cx, store_object, "_savePreKey", [id.upcast(), record])?
                .downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<JsUndefined, _>(cx) {
                Ok(_) => Ok(()),
                Err(_) => Err("unexpected result from _savePreKey".into()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }

    async fn do_remove_pre_key(&self, id: u32) -> Result<(), String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let id: Handle<JsNumber> = id.convert_into(cx)?;
            let result = call_method(cx, store_object, "_removePreKey", [id.upcast()])?
                .downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<JsUndefined, _>(cx) {
                Ok(_) => Ok(()),
                Err(_) => Err("unexpected result from _removePreKey".into()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }
}

impl Finalize for NodePreKeyStore {
    fn finalize<'b, C: neon::prelude::Context<'b>>(self, cx: &mut C) {
        self.store_object.finalize(cx)
    }
}

#[async_trait(?Send)]
impl PreKeyStore for NodePreKeyStore {
    async fn get_pre_key(&self, pre_key_id: PreKeyId) -> Result<PreKeyRecord, SignalProtocolError> {
        self.do_get_pre_key(pre_key_id.into())
            .await
            .map_err(|s| js_error_to_rust("getPreKey", s))
    }

    async fn save_pre_key(
        &mut self,
        pre_key_id: PreKeyId,
        record: &PreKeyRecord,
    ) -> Result<(), SignalProtocolError> {
        self.do_save_pre_key(pre_key_id.into(), record.clone())
            .await
            .map_err(|s| js_error_to_rust("savePreKey", s))
    }

    async fn remove_pre_key(&mut self, pre_key_id: PreKeyId) -> Result<(), SignalProtocolError> {
        self.do_remove_pre_key(pre_key_id.into())
            .await
            .map_err(|s| js_error_to_rust("removePreKey", s))
    }
}

pub struct NodeSignedPreKeyStore {
    js_channel: Channel,
    store_object: Arc<Root<JsObject>>,
}

impl NodeSignedPreKeyStore {
    pub(crate) fn new(cx: &mut FunctionContext, store: Handle<JsObject>) -> Self {
        Self {
            js_channel: cx.channel(),
            store_object: Arc::new(store.root(cx)),
        }
    }

    async fn do_get_signed_pre_key(&self, id: u32) -> Result<SignedPreKeyRecord, String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let id = id.convert_into(cx)?;
            let result = call_method(cx, store_object, "_getSignedPreKey", [id.upcast()])?;
            let result = result.downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<DefaultJsBox<SignedPreKeyRecord>, _>(cx) {
                Ok(obj) => Ok((***obj).clone()),
                Err(_) => Err("result must be an object".to_owned()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }

    async fn do_save_signed_pre_key(
        &self,
        id: u32,
        record: SignedPreKeyRecord,
    ) -> Result<(), String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let id: Handle<JsNumber> = id.convert_into(cx)?;
            let record: Handle<JsValue> = record.convert_into(cx)?;
            let result = call_method(cx, store_object, "_saveSignedPreKey", [id.upcast(), record])?
                .downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<JsUndefined, _>(cx) {
                Ok(_) => Ok(()),
                Err(_) => Err("unexpected result from _saveSignedPreKey".into()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }
}

impl Finalize for NodeSignedPreKeyStore {
    fn finalize<'b, C: neon::prelude::Context<'b>>(self, cx: &mut C) {
        self.store_object.finalize(cx)
    }
}

#[async_trait(?Send)]
impl SignedPreKeyStore for NodeSignedPreKeyStore {
    async fn get_signed_pre_key(
        &self,
        signed_pre_key_id: SignedPreKeyId,
    ) -> Result<SignedPreKeyRecord, SignalProtocolError> {
        self.do_get_signed_pre_key(signed_pre_key_id.into())
            .await
            .map_err(|s| js_error_to_rust("getSignedPreKey", s))
    }

    async fn save_signed_pre_key(
        &mut self,
        signed_pre_key_id: SignedPreKeyId,
        record: &SignedPreKeyRecord,
    ) -> Result<(), SignalProtocolError> {
        self.do_save_signed_pre_key(signed_pre_key_id.into(), record.clone())
            .await
            .map_err(|s| js_error_to_rust("saveSignedPreKey", s))
    }
}

pub struct NodeKyberPreKeyStore {
    js_channel: Channel,
    store_object: Arc<Root<JsObject>>,
}

impl NodeKyberPreKeyStore {
    pub(crate) fn new(cx: &mut FunctionContext, store: Handle<JsObject>) -> Self {
        Self {
            js_channel: cx.channel(),
            store_object: Arc::new(store.root(cx)),
        }
    }

    async fn do_get_kyber_pre_key(&self, id: u32) -> Result<KyberPreKeyRecord, String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let id = id.convert_into(cx)?;
            let result = call_method(cx, store_object, "_getKyberPreKey", [id.upcast()])?;
            let result = result.downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<DefaultJsBox<KyberPreKeyRecord>, _>(cx) {
                Ok(obj) => Ok((***obj).clone()),
                Err(_) => Err("result must be an object".to_owned()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }

    async fn do_save_kyber_pre_key(
        &self,
        id: u32,
        record: KyberPreKeyRecord,
    ) -> Result<(), String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let id: Handle<JsNumber> = id.convert_into(cx)?;
            let record: Handle<JsValue> = record.convert_into(cx)?;
            let result = call_method(cx, store_object, "_saveKyberPreKey", [id.upcast(), record])?
                .downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<JsUndefined, _>(cx) {
                Ok(_) => Ok(()),
                Err(_) => Err("unexpected result from _saveKyberPreKey".into()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }

    async fn do_mark_kyber_pre_key_used(
        &self,
        id: u32,
        ec_prekey_id: u32,
        base_key: PublicKey,
    ) -> Result<(), String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let id: Handle<JsNumber> = id.convert_into(cx)?;
            let ec_prekey_id: Handle<JsNumber> = ec_prekey_id.convert_into(cx)?;
            let base_key: Handle<JsValue> = base_key.convert_into(cx)?;
            let result = call_method(
                cx,
                store_object,
                "_markKyberPreKeyUsed",
                [id.upcast(), ec_prekey_id.upcast(), base_key],
            )?
            .downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<JsUndefined, _>(cx) {
                Ok(_) => Ok(()),
                Err(_) => Err("unexpected result from _markKyberPreKeyUsed".into()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }
}

impl Finalize for NodeKyberPreKeyStore {
    fn finalize<'b, C: neon::prelude::Context<'b>>(self, cx: &mut C) {
        self.store_object.finalize(cx)
    }
}

#[async_trait(?Send)]
impl KyberPreKeyStore for NodeKyberPreKeyStore {
    async fn get_kyber_pre_key(
        &self,
        kyber_pre_key_id: KyberPreKeyId,
    ) -> Result<KyberPreKeyRecord, SignalProtocolError> {
        self.do_get_kyber_pre_key(kyber_pre_key_id.into())
            .await
            .map_err(|s| js_error_to_rust("getKyberPreKey", s))
    }

    async fn save_kyber_pre_key(
        &mut self,
        kyber_pre_key_id: KyberPreKeyId,
        record: &KyberPreKeyRecord,
    ) -> Result<(), SignalProtocolError> {
        self.do_save_kyber_pre_key(kyber_pre_key_id.into(), record.clone())
            .await
            .map_err(|s| js_error_to_rust("saveKyberPreKey", s))
    }

    async fn mark_kyber_pre_key_used(
        &mut self,
        kyber_pre_key_id: KyberPreKeyId,
        ec_prekey_id: SignedPreKeyId,
        base_key: &PublicKey,
    ) -> Result<(), SignalProtocolError> {
        self.do_mark_kyber_pre_key_used(kyber_pre_key_id.into(), ec_prekey_id.into(), *base_key)
            .await
            .map_err(|s| js_error_to_rust("markKyberPreKeyUsed", s))
    }
}

pub struct NodeSessionStore {
    js_channel: Channel,
    store_object: Arc<Root<JsObject>>,
}

impl NodeSessionStore {
    pub(crate) fn new(cx: &mut FunctionContext, store: Handle<JsObject>) -> Self {
        Self {
            js_channel: cx.channel(),
            store_object: Arc::new(store.root(cx)),
        }
    }

    async fn do_get_session(&self, name: ProtocolAddress) -> Result<Option<SessionRecord>, String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let name: Handle<JsValue> = name.convert_into(cx)?;
            let result = call_method(cx, store_object, "_getSession", [name])?;
            let result = result.downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<DefaultJsBox<RefCell<SessionRecord>>, _>(cx) {
                Ok(obj) => Ok(Some((***obj).borrow().clone())),
                Err(_) => {
                    if value.is_a::<JsNull, _>(cx) || value.is_a::<JsUndefined, _>(cx) {
                        Ok(None)
                    } else {
                        Err("_getSession returned unexpected type".into())
                    }
                }
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }

    async fn do_save_session(
        &self,
        name: ProtocolAddress,
        record: SessionRecord,
    ) -> Result<(), String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let name = name.convert_into(cx)?;
            let record = record.convert_into(cx)?;
            let result = call_method(cx, store_object, "_saveSession", [name, record.upcast()])?
                .downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<JsUndefined, _>(cx) {
                Ok(_) => Ok(()),
                Err(_) => Err("unexpected result from _saveSession".into()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }
}

impl Finalize for NodeSessionStore {
    fn finalize<'b, C: neon::prelude::Context<'b>>(self, cx: &mut C) {
        self.store_object.finalize(cx)
    }
}

#[async_trait(?Send)]
impl SessionStore for NodeSessionStore {
    async fn load_session(
        &self,
        name: &ProtocolAddress,
    ) -> Result<Option<SessionRecord>, SignalProtocolError> {
        self.do_get_session(name.clone())
            .await
            .map_err(|s| js_error_to_rust("getSession", s))
    }

    async fn store_session(
        &mut self,
        name: &ProtocolAddress,
        record: &SessionRecord,
    ) -> Result<(), SignalProtocolError> {
        self.do_save_session(name.clone(), record.clone())
            .await
            .map_err(|s| js_error_to_rust("saveSession", s))
    }
}

pub struct NodeIdentityKeyStore {
    js_channel: Channel,
    store_object: Arc<Root<JsObject>>,
}

impl NodeIdentityKeyStore {
    pub(crate) fn new(cx: &mut FunctionContext, store: Handle<JsObject>) -> Self {
        Self {
            js_channel: cx.channel(),
            store_object: Arc::new(store.root(cx)),
        }
    }

    async fn do_get_identity_key(&self) -> Result<PrivateKey, String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let result = call_method(cx, store_object, "_getIdentityKey", [])?;
            let result = result.downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<DefaultJsBox<PrivateKey>, _>(cx) {
                Ok(obj) => Ok(***obj),
                Err(_) => Err("result must be an object".to_owned()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }

    async fn do_get_local_registration_id(&self) -> Result<u32, String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let result = call_method(cx, store_object, "_getLocalRegistrationId", [])?
                .downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<JsNumber, _>(cx) {
                // FIXME: Rust doesn't provide a convenient validation for f64->u32.
                // We could do it manually, but really registration IDs have more constraints than
                // just fitting in a u32. For now, we'll ignore those issues; a valid implementation
                // should not have problems here anyway.
                #[expect(clippy::cast_possible_truncation)]
                Ok(b) => Ok(b.value(cx) as u32),
                Err(_) => Err("unexpected result from _getLocalRegistrationId".into()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }

    async fn do_get_identity(&self, name: ProtocolAddress) -> Result<Option<PublicKey>, String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let name: Handle<JsValue> = name.convert_into(cx)?;
            let result = call_method(cx, store_object, "_getIdentity", [name])?;
            let result = result.downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<DefaultJsBox<PublicKey>, _>(cx) {
                Ok(obj) => Ok(Some(***obj)),
                Err(_) => {
                    if value.is_a::<JsNull, _>(cx) {
                        Ok(None)
                    } else {
                        Err("result must be an object".to_owned())
                    }
                }
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }

    async fn do_save_identity(
        &self,
        name: ProtocolAddress,
        key: PublicKey,
    ) -> Result<IdentityChange, String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let name: Handle<JsValue> = name.convert_into(cx)?;
            let key: Handle<JsValue> = key.convert_into(cx)?;
            let result = call_method(cx, store_object, "_saveIdentity", [name, key])?
                .downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<JsNumber, _>(cx) {
                Err(_) => Err("unexpected result from _saveIdentity".into()),
                Ok(n) => {
                    let n: isize = u8::convert_from(cx, n).map_err(|e| e.to_string())?.into();
                    n.try_into()
                        .map_err(|_| format!("{n} invalid for IdentityChange"))
                }
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }

    async fn do_is_trusted(
        &self,
        name: ProtocolAddress,
        key: PublicKey,
        direction: Direction,
    ) -> Result<bool, String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let name: Handle<JsValue> = name.convert_into(cx)?;
            let key: Handle<JsValue> = key.convert_into(cx)?;
            let sending = direction == Direction::Sending;

            let sending = sending.convert_into(cx)?;
            let result = call_method(
                cx,
                store_object,
                "_isTrustedIdentity",
                [name, key, sending.upcast()],
            )?
            .downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<JsBoolean, _>(cx) {
                Ok(b) => Ok(b.value(cx)),
                Err(_) => Err("unexpected result from _isTrustedIdentity".into()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }
}

impl Finalize for NodeIdentityKeyStore {
    fn finalize<'b, C: neon::prelude::Context<'b>>(self, cx: &mut C) {
        self.store_object.finalize(cx)
    }
}

#[async_trait(?Send)]
impl IdentityKeyStore for NodeIdentityKeyStore {
    async fn get_identity_key_pair(&self) -> Result<IdentityKeyPair, SignalProtocolError> {
        let pk = self
            .do_get_identity_key()
            .await
            .map_err(|s| js_error_to_rust("getIdentityPrivateKey", s))?;

        IdentityKeyPair::try_from(pk)
    }

    async fn get_local_registration_id(&self) -> Result<u32, SignalProtocolError> {
        self.do_get_local_registration_id()
            .await
            .map_err(|s| js_error_to_rust("getLocalRegistrationId", s))
    }

    async fn get_identity(
        &self,
        address: &ProtocolAddress,
    ) -> Result<Option<IdentityKey>, SignalProtocolError> {
        Ok(self
            .do_get_identity(address.clone())
            .await
            .map_err(|s| js_error_to_rust("getIdentity", s))?
            .map(IdentityKey::new))
    }

    async fn save_identity(
        &mut self,
        address: &ProtocolAddress,
        identity: &IdentityKey,
    ) -> Result<IdentityChange, SignalProtocolError> {
        self.do_save_identity(address.clone(), *identity.public_key())
            .await
            .map_err(|s| js_error_to_rust("saveIdentity", s))
    }

    async fn is_trusted_identity(
        &self,
        address: &ProtocolAddress,
        identity: &IdentityKey,
        direction: libsignal_protocol::Direction,
    ) -> Result<bool, SignalProtocolError> {
        self.do_is_trusted(address.clone(), *identity.public_key(), direction)
            .await
            .map_err(|s| js_error_to_rust("isTrustedIdentity", s))
    }
}

pub struct NodeSenderKeyStore {
    js_channel: Channel,
    store_object: Arc<Root<JsObject>>,
}

impl NodeSenderKeyStore {
    pub(crate) fn new(cx: &mut FunctionContext, store: Handle<JsObject>) -> Self {
        Self {
            js_channel: cx.channel(),
            store_object: Arc::new(store.root(cx)),
        }
    }

    async fn do_get_sender_key(
        &self,
        sender: ProtocolAddress,
        distribution_id: Uuid,
    ) -> Result<Option<SenderKeyRecord>, String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let sender: Handle<JsValue> = sender.convert_into(cx)?;
            let distribution_id: Handle<JsValue> = distribution_id.convert_into(cx)?.upcast();
            let result = call_method(cx, store_object, "_getSenderKey", [sender, distribution_id])?;
            let result = result.downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<DefaultJsBox<SenderKeyRecord>, _>(cx) {
                Ok(obj) => Ok(Some((***obj).clone())),
                Err(_) => {
                    if value.is_a::<JsNull, _>(cx) {
                        Ok(None)
                    } else {
                        Err("result must be an object".to_owned())
                    }
                }
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }

    async fn do_save_sender_key(
        &self,
        sender: ProtocolAddress,
        distribution_id: Uuid,
        record: SenderKeyRecord,
    ) -> Result<(), String> {
        let store_object_shared = self.store_object.clone();
        JsFuture::get_promise(&self.js_channel, move |cx| {
            let store_object = store_object_shared.to_inner(cx);
            let sender: Handle<JsValue> = sender.convert_into(cx)?;
            let distribution_id: Handle<JsValue> = distribution_id.convert_into(cx)?.upcast();
            let record: Handle<JsValue> = record.convert_into(cx)?;
            let result = call_method(
                cx,
                store_object,
                "_saveSenderKey",
                [sender, distribution_id, record],
            )?
            .downcast_or_throw(cx)?;
            store_object_shared.finalize(cx);
            Ok(result)
        })
        .then(|cx, result| match result {
            Ok(value) => match value.downcast::<JsUndefined, _>(cx) {
                Ok(_) => Ok(()),
                Err(_) => Err("unexpected result from _saveSenderKey".into()),
            },
            Err(error) => Err(error
                .to_string(cx)
                .expect("can convert to string")
                .value(cx)),
        })
        .await
    }
}

impl Finalize for NodeSenderKeyStore {
    fn finalize<'a, C: Context<'a>>(self, cx: &mut C) {
        self.store_object.finalize(cx)
    }
}

#[async_trait(?Send)]
impl SenderKeyStore for NodeSenderKeyStore {
    async fn load_sender_key(
        &mut self,
        sender: &ProtocolAddress,
        distribution_id: Uuid,
    ) -> Result<Option<SenderKeyRecord>, SignalProtocolError> {
        self.do_get_sender_key(sender.clone(), distribution_id)
            .await
            .map_err(|s| js_error_to_rust("getSenderKey", s))
    }

    async fn store_sender_key(
        &mut self,
        sender: &ProtocolAddress,
        distribution_id: Uuid,
        record: &SenderKeyRecord,
    ) -> Result<(), SignalProtocolError> {
        self.do_save_sender_key(sender.clone(), distribution_id, record.clone())
            .await
            .map_err(|s| js_error_to_rust("saveSenderKey", s))
    }
}
