//
// Copyright 2024 Signal Messenger, LLC.
// SPDX-License-Identifier: AGPL-3.0-only
//

import ByteArray from '../internal/ByteArray.js';
import Native from '../../../Native.js';
import { ServiceId } from '../../Address.js';
import GroupSendDerivedKeyPair from './GroupSendDerivedKeyPair.js';

// For docs
import type GroupSendEndorsement from './GroupSendEndorsement.js';
import type { VerificationFailedError } from '../../Errors.js';

/**
 * A token representing a particular {@link GroupSendEndorsement}, along with the endorsement's
 * expiration.
 *
 * Generated by {@link GroupSendToken#toFullToken}, and verified by the chat server.
 */
export default class GroupSendFullToken extends ByteArray {
  constructor(contents: Uint8Array) {
    super(contents, Native.GroupSendFullToken_CheckValidContents);
  }

  /** Gets the expiration embedded in the token. */
  getExpiration(): Date {
    return new Date(
      1000 * Native.GroupSendFullToken_GetExpiration(this.contents)
    );
  }

  /**
   * Verifies that this token was generated from an endorsement of `userIds` by `keyPair`.
   *
   * The correct `keyPair` must be selected based on {@link #getExpiration}.
   *
   * @throws {VerificationFailedError} if the token is invalid.
   */
  verify(
    userIds: ServiceId[],
    keyPair: GroupSendDerivedKeyPair,
    now: Date = new Date()
  ): void {
    Native.GroupSendFullToken_Verify(
      this.contents,
      ServiceId.toConcatenatedFixedWidthBinary(userIds),
      Math.floor(now.getTime() / 1000),
      keyPair.contents
    );
  }
}
