use strict;
use warnings;

use File::Basename;
use File::Temp;
use RT::Test tests => undef;

my $HTML_FH = File::Temp->new(UNLINK => 0, SUFFIX => '.html');

print $HTML_FH <<EOHTML;
<html>
    <body>
        <h1>Test</h1>
    </body>
</html>
EOHTML

my $HTML_Name = "$HTML_FH";
my $HTML_ShortName = basename($HTML_Name);

$HTML_FH = undef;

END {
    unlink $HTML_Name;
}

foreach my $strictness (1, 0) {
    foreach my $trust_level (0, 1) {
        RT->Config->Set('StrictContentTypes', $strictness);
        RT->Config->Set('TrustHTMLAttachments', $trust_level);

        my ($url, $m) = RT::Test->started_ok;
        ok $m->login, 'logged in';

        my $cf = RT::Test->load_or_create_custom_field(
            Name  => 'cve-test',
            Queue => 0,
            Type  => 'Binary',
        );

        my $queue = RT::Test->load_or_create_queue( Name => 'General' );
        ok( $queue && $queue->id, "Loaded General queue" );

        diag("transaction HTML attachment with normal content type");
        {
            $m->goto_create_ticket( $queue );

            $m->form_name('TicketCreate');
            $m->field('Subject', 'Known Content-Type Handling');
            $m->field('Attach',  $HTML_Name);
            $m->field('Content', 'Some content');

            $m->click('SubmitTicket');
            is($m->status, 200, "request successful");

            $m->content_contains('Known Content-Type Handling', 'we have subject on the page');
            $m->content_contains('Some content', 'and content');
            $m->text_contains($HTML_ShortName, 'page has file name');

            $m->follow_link_ok( { url_regex => qr/$HTML_ShortName/ } );
            like( $m->response->header('Content-Type'),
                qr/;\s*charset=/,
                'Content-Type of text has charset',
            );
            is($m->content_type, ($trust_level ? "text/html" : 'text/plain'), 'Content-Type is expected type');
            is($m->response->header('X-Content-Type-Options'), ($strictness ? 'nosniff' : ()), 'Has X-Content-Type-Options header');
            is($m->content, RT::Test->file_content($HTML_Name), "Text content matches");

            my $ticket = RT::Test->last_ticket;

            $m->goto_ticket($ticket->id);
            $m->follow_link(id => 'page-basics');

            $m->form_name('TicketModify');
            $m->field('Object-RT::Ticket-'.$ticket->id.'-CustomField-'.$cf->id.'-Upload', $HTML_Name);

            $m->submit;
            is($m->status, 200, "modification successful");

            $m->goto_ticket($ticket->id);

            $m->follow_link_ok( { url_regex => qr/$HTML_ShortName/ } );
            like( $m->response->header('Content-Type'),
                qr/;\s*charset=/,
                'Content-Type of text has charset',
            );
            is($m->content_type, ($trust_level ? "text/html" : 'text/plain'), 'Content-Type is expected type');
            is($m->response->header('X-Content-Type-Options'), ($strictness ? 'nosniff' : ()), 'Has X-Content-Type-Options header');
            is($m->content, RT::Test->file_content($HTML_Name), "Text content matches");
        }

        diag("transaction HTML attachment with bogus content type");
        {
            $m->goto_create_ticket( $queue );

            $m->form_name('TicketCreate');
            $m->field('Subject', 'Unknown Content-Type Handling');
            $m->field('Attach',  $HTML_Name);
            $m->current_form->find_input('Attach')->headers('Content-Type' => 'bogus');
            $m->field('Content', 'Some content');

            $m->click('SubmitTicket');
            is($m->status, 200, "request successful");

            $m->content_contains('Unknown Content-Type Handling', 'we have subject on the page');
            $m->content_contains('Some content', 'and content');
            $m->text_contains($HTML_ShortName, 'page has file name');

            $m->follow_link_ok( { url_regex => qr/$HTML_ShortName/ } );
            is($m->content_type, "bogus", 'Content-Type is bogus');
            is($m->content, RT::Test->file_content($HTML_Name), "Text content matches");
            is($m->response->header('X-Content-Type-Options'), ($strictness ? 'nosniff' : ()), 'Has X-Content-Type-Options header');

            my $ticket = RT::Test->last_ticket;

            $m->goto_ticket($ticket->id);
            $m->follow_link(id => 'page-jumbo');

            $m->form_name('TicketModifyAll');
            $m->field('Object-RT::Ticket-'.$ticket->id.'-CustomField-'.$cf->id.'-Upload', $HTML_Name);
            $m->current_form->find_input('Object-RT::Ticket-'.$ticket->id.'-CustomField-'.$cf->id.'-Upload')->headers('Content-Type' => 'bogus');

            $m->submit;
            is($m->status, 200, "modification successful");

            $m->goto_ticket($ticket->id);

            $m->follow_link_ok( { url_regex => qr/$HTML_ShortName/ } );
            is($m->content_type, "bogus", 'Content-Type is bogus');
            is($m->content, RT::Test->file_content($HTML_Name), "Text content matches");
            is($m->response->header('X-Content-Type-Options'), ($strictness ? 'nosniff' : ()), 'Has X-Content-Type-Options header');
        }

        $m->no_warnings_ok;
        $m = undef;

        RT::Test->stop_server;
    }
}

done_testing;

