package TestUtils;

use warnings;
use strict;
use POSIX qw(:sys_wait_h getpwuid getgrgid);
use IO::Socket::INET;
require File::Temp;
require Exporter;
our @ISA = qw(Exporter);
our @EXPORT = qw(start_child reap_children wait_for_type wait_for_port make_config find_free_port test_user test_group);
our $VERSION = '0.01';

$SIG{CHLD} = \&REAPER;

my %children;

sub REAPER {
    my $stiff;
    while (($stiff = waitpid(-1, &WNOHANG)) > 0) {
        # do something with $stiff if you want
        $children{$stiff}->{'running'} = undef;
        $children{$stiff}->{'exit_code'} = $? >> 8;
        $children{$stiff}->{'signal'} = $? & 127;
        $children{$stiff}->{'core_dumped'} = $? & 128;
    }
    $SIG{CHLD} = \&REAPER; # install *after* calling waitpid
}

# Make several requests through the proxy specifying the host header
sub start_child {
    my $type = shift;
    my $child = shift;
    my @args = @_;

    my $pid = fork();
    if (not defined $pid) {
        die("fork: $!");
    } elsif ($pid == 0) {
        undef $SIG{CHLD};
        $child->(@args);
        # Should not be reached
        exit(99);
    }

    $children{$pid} = {
        type => $type,
        pid => $pid,
        running => 1,
        core_dumped => undef,
        signal => undef,
        exit_core => undef,
    };

    return $pid;
}

sub reap_children {
    while (my @hit_list = grep($children{$_}->{'running'}, keys %children)) {
        kill 15, @hit_list;
        sleep 1;
    }

    # Check that all our children exited cleanly
    my @failures = grep($_->{'exit_code'} != 0 || $_->{'core_dumped'}, values %children);
    if (@failures) {
        print "Test failed.\n";
        foreach (@failures) {
            if ($_->{'core_dumped'}) {
                printf "%s died with signal %d, %s coredump\n", $_->{'type'}, $_->{'signal'}, $_->{'core_dumped'} ? 'with' : 'without';
            } else {
                print "$_->{'type'} failed with exit code $_->{'exit_code'}\n";
            }
        }
        exit 1;
    } else {
        # print "Test passed.\n";
        return;
    }
}

sub wait_for_type($) {
    my $type = shift;

    while (grep($children{$_}->{'running'} && $children{$_}->{'type'} eq $type, keys %children) > 0) {
        sleep 1;
    }
}

sub wait_for_port {
    my %args = @_;
    my $ip = $args{'ip'} || 'localhost';
    my $port = $args{'port'} or die "port required";

    my $delay = 1;
    while ($delay < 60) {
        my $port_open = undef;
        eval {
            my $socket = IO::Socket::INET->new(PeerAddr => $ip,
                                               PeerPort => $port,
                                               Proto => "tcp",
                                               Type => SOCK_STREAM);
            if ($socket && $socket->connected()) {
                $socket->shutdown(2);
                $port_open = 1;
            }
        };

        return 1 if ($port_open);

        sleep($delay);
        $delay *= 2;
    }

    return undef;
}

my %allocated_ports;

sub find_free_port {
    while (1) {
        my $socket = IO::Socket::INET->new(
            LocalAddr => '127.0.0.1',
            LocalPort => 0,
            Listen    => 1,
            Proto     => 'tcp',
            ReuseAddr => 1,
        ) or die "failed to allocate port: $!";

        my $port = $socket->sockport;
        close $socket;
        next if $allocated_ports{$port};
        $allocated_ports{$port} = 1;
        return $port;
    }
}

sub make_config($$) {
    my $proxy_port = shift;
    my $httpd_port = shift;

    my ($fh, $filename) = File::Temp::tempfile();
    my $user = test_user();
    my $group = test_group();

    # Write out a test config file
    print $fh <<END;
# Minimal test configuration

user $user
group $group

listen 127.0.0.1 $proxy_port {
    proto http
}

table {
    localhost 127.0.0.1 $httpd_port
}
END

    close ($fh);

    return $filename;
}

1;
my $TEST_USER = $ENV{SNI_PROXY_USER};
$TEST_USER = getpwuid($>) unless defined $TEST_USER && $TEST_USER ne '';
my $gid_list_env = $ENV{SNI_PROXY_GROUP};
my $TEST_GROUP;
if (defined $gid_list_env && $gid_list_env ne '') {
    $TEST_GROUP = $gid_list_env;
} else {
    my $gid_list = $);
    my ($primary_gid) = split(/\s+/, $gid_list);
    $TEST_GROUP = getgrgid($primary_gid) || $primary_gid;
}

sub test_user { $TEST_USER }
sub test_group { $TEST_GROUP }
