// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Code generated by "cmd/gentypes/gentypes.go"; DO NOT EDIT.
// DAP spec: https://microsoft.github.io/debug-adapter-protocol/specification
// See cmd/gentypes/README.md for additional details.

package dap

import "encoding/json"

// Message is an interface that all DAP message types implement with pointer
// receivers. It's not part of the protocol but is used to enforce static
// typing in Go code and provide some common accessors.
//
// Note: the DAP type "Message" (which is used in the body of ErrorResponse)
// is renamed to ErrorMessage to avoid collision with this interface.
type Message interface {
	GetSeq() int
}

// RequestMessage is an interface implemented by all Request-types.
type RequestMessage interface {
	Message
	// GetRequest provides access to the embedded Request.
	GetRequest() *Request
}

// ResponseMessage is an interface implemented by all Response-types.
type ResponseMessage interface {
	Message
	// GetResponse provides access to the embedded Response.
	GetResponse() *Response
}

// EventMessage is an interface implemented by all Event-types.
type EventMessage interface {
	Message
	// GetEvent provides access to the embedded Event.
	GetEvent() *Event
}

// LaunchAttachRequest is an interface implemented by
// LaunchRequest and AttachRequest as they contain shared
// implementation specific arguments that are not part of
// the specification.
type LaunchAttachRequest interface {
	RequestMessage
	// GetArguments provides access to the Arguments map.
	GetArguments() json.RawMessage
}

// ProtocolMessage: Base class of requests, responses, and events.
type ProtocolMessage struct {
	Seq  int    `json:"seq"`
	Type string `json:"type"`
}

func (m *ProtocolMessage) GetSeq() int { return m.Seq }

// Request: A client or debug adapter initiated request.
type Request struct {
	ProtocolMessage

	Command string `json:"command"`
}

func (r *Request) GetRequest() *Request { return r }

// Event: A debug adapter initiated event.
type Event struct {
	ProtocolMessage

	Event string `json:"event"`
}

func (e *Event) GetEvent() *Event { return e }

// Response: Response for a request.
type Response struct {
	ProtocolMessage

	RequestSeq int    `json:"request_seq"`
	Success    bool   `json:"success"`
	Command    string `json:"command"`
	Message    string `json:"message,omitempty"`
}

func (r *Response) GetResponse() *Response { return r }

// ErrorResponse: On error (whenever `success` is false), the body can provide more details.
type ErrorResponse struct {
	Response

	Body ErrorResponseBody `json:"body"`
}

type ErrorResponseBody struct {
	Error *ErrorMessage `json:"error,omitempty"`
}

// CancelRequest: The `cancel` request is used by the client in two situations:
// - to indicate that it is no longer interested in the result produced by a specific request issued earlier
// - to cancel a progress sequence.
// Clients should only call this request if the corresponding capability `supportsCancelRequest` is true.
// This request has a hint characteristic: a debug adapter can only be expected to make a 'best effort' in honoring this request but there are no guarantees.
// The `cancel` request may return an error if it could not cancel an operation but a client should refrain from presenting this error to end users.
// The request that got cancelled still needs to send a response back. This can either be a normal result (`success` attribute true) or an error response (`success` attribute false and the `message` set to `cancelled`).
// Returning partial results from a cancelled request is possible but please note that a client has no generic way for detecting that a response is partial or not.
// The progress that got cancelled still needs to send a `progressEnd` event back.
//
//	A client should not assume that progress just got cancelled after sending the `cancel` request.
type CancelRequest struct {
	Request

	Arguments *CancelArguments `json:"arguments,omitempty"`
}

// CancelArguments: Arguments for `cancel` request.
type CancelArguments struct {
	RequestId  int    `json:"requestId,omitempty"`
	ProgressId string `json:"progressId,omitempty"`
}

// CancelResponse: Response to `cancel` request. This is just an acknowledgement, so no body field is required.
type CancelResponse struct {
	Response
}

// InitializedEvent: This event indicates that the debug adapter is ready to accept configuration requests (e.g. `setBreakpoints`, `setExceptionBreakpoints`).
// A debug adapter is expected to send this event when it is ready to accept configuration requests (but not before the `initialize` request has finished).
// The sequence of events/requests is as follows:
// - adapters sends `initialized` event (after the `initialize` request has returned)
// - client sends zero or more `setBreakpoints` requests
// - client sends one `setFunctionBreakpoints` request (if corresponding capability `supportsFunctionBreakpoints` is true)
// - client sends a `setExceptionBreakpoints` request if one or more `exceptionBreakpointFilters` have been defined (or if `supportsConfigurationDoneRequest` is not true)
// - client sends other future configuration requests
// - client sends one `configurationDone` request to indicate the end of the configuration.
type InitializedEvent struct {
	Event
}

// StoppedEvent: The event indicates that the execution of the debuggee has stopped due to some condition.
// This can be caused by a breakpoint previously set, a stepping request has completed, by executing a debugger statement etc.
type StoppedEvent struct {
	Event

	Body StoppedEventBody `json:"body"`
}

type StoppedEventBody struct {
	Reason            string `json:"reason"`
	Description       string `json:"description,omitempty"`
	ThreadId          int    `json:"threadId,omitempty"`
	PreserveFocusHint bool   `json:"preserveFocusHint,omitempty"`
	Text              string `json:"text,omitempty"`
	AllThreadsStopped bool   `json:"allThreadsStopped,omitempty"`
	HitBreakpointIds  []int  `json:"hitBreakpointIds,omitempty"`
}

// ContinuedEvent: The event indicates that the execution of the debuggee has continued.
// Please note: a debug adapter is not expected to send this event in response to a request that implies that execution continues, e.g. `launch` or `continue`.
// It is only necessary to send a `continued` event if there was no previous request that implied this.
type ContinuedEvent struct {
	Event

	Body ContinuedEventBody `json:"body"`
}

type ContinuedEventBody struct {
	ThreadId            int  `json:"threadId"`
	AllThreadsContinued bool `json:"allThreadsContinued,omitempty"`
}

// ExitedEvent: The event indicates that the debuggee has exited and returns its exit code.
type ExitedEvent struct {
	Event

	Body ExitedEventBody `json:"body"`
}

type ExitedEventBody struct {
	ExitCode int `json:"exitCode"`
}

// TerminatedEvent: The event indicates that debugging of the debuggee has terminated. This does **not** mean that the debuggee itself has exited.
type TerminatedEvent struct {
	Event

	Body TerminatedEventBody `json:"body,omitempty"`
}

type TerminatedEventBody struct {
	Restart json.RawMessage `json:"restart,omitempty"`
}

// ThreadEvent: The event indicates that a thread has started or exited.
type ThreadEvent struct {
	Event

	Body ThreadEventBody `json:"body"`
}

type ThreadEventBody struct {
	Reason   string `json:"reason"`
	ThreadId int    `json:"threadId"`
}

// OutputEvent: The event indicates that the target has produced some output.
type OutputEvent struct {
	Event

	Body OutputEventBody `json:"body"`
}

type OutputEventBody struct {
	Category           string          `json:"category,omitempty"`
	Output             string          `json:"output"`
	Group              string          `json:"group,omitempty"`
	VariablesReference int             `json:"variablesReference,omitempty"`
	Source             *Source         `json:"source,omitempty"`
	Line               int             `json:"line,omitempty"`
	Column             int             `json:"column,omitempty"`
	Data               json.RawMessage `json:"data,omitempty"`
	LocationReference  int             `json:"locationReference,omitempty"`
}

// BreakpointEvent: The event indicates that some information about a breakpoint has changed.
type BreakpointEvent struct {
	Event

	Body BreakpointEventBody `json:"body"`
}

type BreakpointEventBody struct {
	Reason     string     `json:"reason"`
	Breakpoint Breakpoint `json:"breakpoint"`
}

// ModuleEvent: The event indicates that some information about a module has changed.
type ModuleEvent struct {
	Event

	Body ModuleEventBody `json:"body"`
}

type ModuleEventBody struct {
	Reason string `json:"reason"`
	Module Module `json:"module"`
}

// LoadedSourceEvent: The event indicates that some source has been added, changed, or removed from the set of all loaded sources.
type LoadedSourceEvent struct {
	Event

	Body LoadedSourceEventBody `json:"body"`
}

type LoadedSourceEventBody struct {
	Reason string `json:"reason"`
	Source Source `json:"source"`
}

// ProcessEvent: The event indicates that the debugger has begun debugging a new process. Either one that it has launched, or one that it has attached to.
type ProcessEvent struct {
	Event

	Body ProcessEventBody `json:"body"`
}

type ProcessEventBody struct {
	Name            string `json:"name"`
	SystemProcessId int    `json:"systemProcessId,omitempty"`
	IsLocalProcess  bool   `json:"isLocalProcess,omitempty"`
	StartMethod     string `json:"startMethod,omitempty"`
	PointerSize     int    `json:"pointerSize,omitempty"`
}

// CapabilitiesEvent: The event indicates that one or more capabilities have changed.
// Since the capabilities are dependent on the client and its UI, it might not be possible to change that at random times (or too late).
// Consequently this event has a hint characteristic: a client can only be expected to make a 'best effort' in honoring individual capabilities but there are no guarantees.
// Only changed capabilities need to be included, all other capabilities keep their values.
type CapabilitiesEvent struct {
	Event

	Body CapabilitiesEventBody `json:"body"`
}

type CapabilitiesEventBody struct {
	Capabilities Capabilities `json:"capabilities"`
}

// ProgressStartEvent: The event signals that a long running operation is about to start and provides additional information for the client to set up a corresponding progress and cancellation UI.
// The client is free to delay the showing of the UI in order to reduce flicker.
// This event should only be sent if the corresponding capability `supportsProgressReporting` is true.
type ProgressStartEvent struct {
	Event

	Body ProgressStartEventBody `json:"body"`
}

type ProgressStartEventBody struct {
	ProgressId  string `json:"progressId"`
	Title       string `json:"title"`
	RequestId   int    `json:"requestId,omitempty"`
	Cancellable bool   `json:"cancellable,omitempty"`
	Message     string `json:"message,omitempty"`
	Percentage  int    `json:"percentage,omitempty"`
}

// ProgressUpdateEvent: The event signals that the progress reporting needs to be updated with a new message and/or percentage.
// The client does not have to update the UI immediately, but the clients needs to keep track of the message and/or percentage values.
// This event should only be sent if the corresponding capability `supportsProgressReporting` is true.
type ProgressUpdateEvent struct {
	Event

	Body ProgressUpdateEventBody `json:"body"`
}

type ProgressUpdateEventBody struct {
	ProgressId string `json:"progressId"`
	Message    string `json:"message,omitempty"`
	Percentage int    `json:"percentage,omitempty"`
}

// ProgressEndEvent: The event signals the end of the progress reporting with a final message.
// This event should only be sent if the corresponding capability `supportsProgressReporting` is true.
type ProgressEndEvent struct {
	Event

	Body ProgressEndEventBody `json:"body"`
}

type ProgressEndEventBody struct {
	ProgressId string `json:"progressId"`
	Message    string `json:"message,omitempty"`
}

// InvalidatedEvent: This event signals that some state in the debug adapter has changed and requires that the client needs to re-render the data snapshot previously requested.
// Debug adapters do not have to emit this event for runtime changes like stopped or thread events because in that case the client refetches the new state anyway. But the event can be used for example to refresh the UI after rendering formatting has changed in the debug adapter.
// This event should only be sent if the corresponding capability `supportsInvalidatedEvent` is true.
type InvalidatedEvent struct {
	Event

	Body InvalidatedEventBody `json:"body"`
}

type InvalidatedEventBody struct {
	Areas        []InvalidatedAreas `json:"areas,omitempty"`
	ThreadId     int                `json:"threadId,omitempty"`
	StackFrameId int                `json:"stackFrameId,omitempty"`
}

// MemoryEvent: This event indicates that some memory range has been updated. It should only be sent if the corresponding capability `supportsMemoryEvent` is true.
// Clients typically react to the event by re-issuing a `readMemory` request if they show the memory identified by the `memoryReference` and if the updated memory range overlaps the displayed range. Clients should not make assumptions how individual memory references relate to each other, so they should not assume that they are part of a single continuous address range and might overlap.
// Debug adapters can use this event to indicate that the contents of a memory range has changed due to some other request like `setVariable` or `setExpression`. Debug adapters are not expected to emit this event for each and every memory change of a running program, because that information is typically not available from debuggers and it would flood clients with too many events.
type MemoryEvent struct {
	Event

	Body MemoryEventBody `json:"body"`
}

type MemoryEventBody struct {
	MemoryReference string `json:"memoryReference"`
	Offset          int    `json:"offset"`
	Count           int    `json:"count"`
}

// RunInTerminalRequest: This request is sent from the debug adapter to the client to run a command in a terminal.
// This is typically used to launch the debuggee in a terminal provided by the client.
// This request should only be called if the corresponding client capability `supportsRunInTerminalRequest` is true.
// Client implementations of `runInTerminal` are free to run the command however they choose including issuing the command to a command line interpreter (aka 'shell'). Argument strings passed to the `runInTerminal` request must arrive verbatim in the command to be run. As a consequence, clients which use a shell are responsible for escaping any special shell characters in the argument strings to prevent them from being interpreted (and modified) by the shell.
// Some users may wish to take advantage of shell processing in the argument strings. For clients which implement `runInTerminal` using an intermediary shell, the `argsCanBeInterpretedByShell` property can be set to true. In this case the client is requested not to escape any special shell characters in the argument strings.
type RunInTerminalRequest struct {
	Request

	Arguments RunInTerminalRequestArguments `json:"arguments"`
}

// RunInTerminalRequestArguments: Arguments for `runInTerminal` request.
type RunInTerminalRequestArguments struct {
	Kind                        string         `json:"kind,omitempty"`
	Title                       string         `json:"title,omitempty"`
	Cwd                         string         `json:"cwd"`
	Args                        []string       `json:"args"`
	Env                         map[string]any `json:"env,omitempty"`
	ArgsCanBeInterpretedByShell bool           `json:"argsCanBeInterpretedByShell,omitempty"`
}

// RunInTerminalResponse: Response to `runInTerminal` request.
type RunInTerminalResponse struct {
	Response

	Body RunInTerminalResponseBody `json:"body"`
}

type RunInTerminalResponseBody struct {
	ProcessId      int `json:"processId,omitempty"`
	ShellProcessId int `json:"shellProcessId,omitempty"`
}

// StartDebuggingRequest: This request is sent from the debug adapter to the client to start a new debug session of the same type as the caller.
// This request should only be sent if the corresponding client capability `supportsStartDebuggingRequest` is true.
// A client implementation of `startDebugging` should start a new debug session (of the same type as the caller) in the same way that the caller's session was started. If the client supports hierarchical debug sessions, the newly created session can be treated as a child of the caller session.
type StartDebuggingRequest struct {
	Request

	Arguments StartDebuggingRequestArguments `json:"arguments"`
}

// StartDebuggingRequestArguments: Arguments for `startDebugging` request.
type StartDebuggingRequestArguments struct {
	Configuration map[string]any `json:"configuration"`
	Request       string         `json:"request"`
}

// StartDebuggingResponse: Response to `startDebugging` request. This is just an acknowledgement, so no body field is required.
type StartDebuggingResponse struct {
	Response
}

// InitializeRequest: The `initialize` request is sent as the first request from the client to the debug adapter in order to configure it with client capabilities and to retrieve capabilities from the debug adapter.
// Until the debug adapter has responded with an `initialize` response, the client must not send any additional requests or events to the debug adapter.
// In addition the debug adapter is not allowed to send any requests or events to the client until it has responded with an `initialize` response.
// The `initialize` request may only be sent once.
type InitializeRequest struct {
	Request

	Arguments InitializeRequestArguments `json:"arguments"`
}

// InitializeRequestArguments: Arguments for `initialize` request.
type InitializeRequestArguments struct {
	ClientID                            string `json:"clientID,omitempty"`
	ClientName                          string `json:"clientName,omitempty"`
	AdapterID                           string `json:"adapterID"`
	Locale                              string `json:"locale,omitempty"`
	LinesStartAt1                       bool   `json:"linesStartAt1"`
	ColumnsStartAt1                     bool   `json:"columnsStartAt1"`
	PathFormat                          string `json:"pathFormat,omitempty"`
	SupportsVariableType                bool   `json:"supportsVariableType,omitempty"`
	SupportsVariablePaging              bool   `json:"supportsVariablePaging,omitempty"`
	SupportsRunInTerminalRequest        bool   `json:"supportsRunInTerminalRequest,omitempty"`
	SupportsMemoryReferences            bool   `json:"supportsMemoryReferences,omitempty"`
	SupportsProgressReporting           bool   `json:"supportsProgressReporting,omitempty"`
	SupportsInvalidatedEvent            bool   `json:"supportsInvalidatedEvent,omitempty"`
	SupportsMemoryEvent                 bool   `json:"supportsMemoryEvent,omitempty"`
	SupportsArgsCanBeInterpretedByShell bool   `json:"supportsArgsCanBeInterpretedByShell,omitempty"`
	SupportsStartDebuggingRequest       bool   `json:"supportsStartDebuggingRequest,omitempty"`
	SupportsANSIStyling                 bool   `json:"supportsANSIStyling,omitempty"`
}

// InitializeResponse: Response to `initialize` request.
type InitializeResponse struct {
	Response

	Body Capabilities `json:"body,omitempty"`
}

// ConfigurationDoneRequest: This request indicates that the client has finished initialization of the debug adapter.
// So it is the last request in the sequence of configuration requests (which was started by the `initialized` event).
// Clients should only call this request if the corresponding capability `supportsConfigurationDoneRequest` is true.
type ConfigurationDoneRequest struct {
	Request

	Arguments *ConfigurationDoneArguments `json:"arguments,omitempty"`
}

// ConfigurationDoneArguments: Arguments for `configurationDone` request.
type ConfigurationDoneArguments struct {
}

// ConfigurationDoneResponse: Response to `configurationDone` request. This is just an acknowledgement, so no body field is required.
type ConfigurationDoneResponse struct {
	Response
}

// LaunchRequest: This launch request is sent from the client to the debug adapter to start the debuggee with or without debugging (if `noDebug` is true).
// Since launching is debugger/runtime specific, the arguments for this request are not part of this specification.
type LaunchRequest struct {
	Request

	Arguments json.RawMessage `json:"arguments"`
}

func (r *LaunchRequest) GetArguments() json.RawMessage { return r.Arguments }

// LaunchResponse: Response to `launch` request. This is just an acknowledgement, so no body field is required.
type LaunchResponse struct {
	Response
}

// AttachRequest: The `attach` request is sent from the client to the debug adapter to attach to a debuggee that is already running.
// Since attaching is debugger/runtime specific, the arguments for this request are not part of this specification.
type AttachRequest struct {
	Request

	Arguments json.RawMessage `json:"arguments"`
}

func (r *AttachRequest) GetArguments() json.RawMessage { return r.Arguments }

// AttachResponse: Response to `attach` request. This is just an acknowledgement, so no body field is required.
type AttachResponse struct {
	Response
}

// RestartRequest: Restarts a debug session. Clients should only call this request if the corresponding capability `supportsRestartRequest` is true.
// If the capability is missing or has the value false, a typical client emulates `restart` by terminating the debug adapter first and then launching it anew.
type RestartRequest struct {
	Request

	Arguments json.RawMessage `json:"arguments"`
}

// RestartResponse: Response to `restart` request. This is just an acknowledgement, so no body field is required.
type RestartResponse struct {
	Response
}

// DisconnectRequest: The `disconnect` request asks the debug adapter to disconnect from the debuggee (thus ending the debug session) and then to shut down itself (the debug adapter).
// In addition, the debug adapter must terminate the debuggee if it was started with the `launch` request. If an `attach` request was used to connect to the debuggee, then the debug adapter must not terminate the debuggee.
// This implicit behavior of when to terminate the debuggee can be overridden with the `terminateDebuggee` argument (which is only supported by a debug adapter if the corresponding capability `supportTerminateDebuggee` is true).
type DisconnectRequest struct {
	Request

	Arguments *DisconnectArguments `json:"arguments,omitempty"`
}

// DisconnectArguments: Arguments for `disconnect` request.
type DisconnectArguments struct {
	Restart           bool `json:"restart,omitempty"`
	TerminateDebuggee bool `json:"terminateDebuggee,omitempty"`
	SuspendDebuggee   bool `json:"suspendDebuggee,omitempty"`
}

// DisconnectResponse: Response to `disconnect` request. This is just an acknowledgement, so no body field is required.
type DisconnectResponse struct {
	Response
}

// TerminateRequest: The `terminate` request is sent from the client to the debug adapter in order to shut down the debuggee gracefully. Clients should only call this request if the capability `supportsTerminateRequest` is true.
// Typically a debug adapter implements `terminate` by sending a software signal which the debuggee intercepts in order to clean things up properly before terminating itself.
// Please note that this request does not directly affect the state of the debug session: if the debuggee decides to veto the graceful shutdown for any reason by not terminating itself, then the debug session just continues.
// Clients can surface the `terminate` request as an explicit command or they can integrate it into a two stage Stop command that first sends `terminate` to request a graceful shutdown, and if that fails uses `disconnect` for a forceful shutdown.
type TerminateRequest struct {
	Request

	Arguments *TerminateArguments `json:"arguments,omitempty"`
}

// TerminateArguments: Arguments for `terminate` request.
type TerminateArguments struct {
	Restart bool `json:"restart,omitempty"`
}

// TerminateResponse: Response to `terminate` request. This is just an acknowledgement, so no body field is required.
type TerminateResponse struct {
	Response
}

// BreakpointLocationsRequest: The `breakpointLocations` request returns all possible locations for source breakpoints in a given range.
// Clients should only call this request if the corresponding capability `supportsBreakpointLocationsRequest` is true.
type BreakpointLocationsRequest struct {
	Request

	Arguments *BreakpointLocationsArguments `json:"arguments,omitempty"`
}

// BreakpointLocationsArguments: Arguments for `breakpointLocations` request.
type BreakpointLocationsArguments struct {
	Source    Source `json:"source"`
	Line      int    `json:"line"`
	Column    int    `json:"column,omitempty"`
	EndLine   int    `json:"endLine,omitempty"`
	EndColumn int    `json:"endColumn,omitempty"`
}

// BreakpointLocationsResponse: Response to `breakpointLocations` request.
// Contains possible locations for source breakpoints.
type BreakpointLocationsResponse struct {
	Response

	Body BreakpointLocationsResponseBody `json:"body"`
}

type BreakpointLocationsResponseBody struct {
	Breakpoints []BreakpointLocation `json:"breakpoints"`
}

// SetBreakpointsRequest: Sets multiple breakpoints for a single source and clears all previous breakpoints in that source.
// To clear all breakpoint for a source, specify an empty array.
// When a breakpoint is hit, a `stopped` event (with reason `breakpoint`) is generated.
type SetBreakpointsRequest struct {
	Request

	Arguments SetBreakpointsArguments `json:"arguments"`
}

// SetBreakpointsArguments: Arguments for `setBreakpoints` request.
type SetBreakpointsArguments struct {
	Source         Source             `json:"source"`
	Breakpoints    []SourceBreakpoint `json:"breakpoints,omitempty"`
	Lines          []int              `json:"lines,omitempty"`
	SourceModified bool               `json:"sourceModified,omitempty"`
}

// SetBreakpointsResponse: Response to `setBreakpoints` request.
// Returned is information about each breakpoint created by this request.
// This includes the actual code location and whether the breakpoint could be verified.
// The breakpoints returned are in the same order as the elements of the `breakpoints`
// (or the deprecated `lines`) array in the arguments.
type SetBreakpointsResponse struct {
	Response

	Body SetBreakpointsResponseBody `json:"body"`
}

type SetBreakpointsResponseBody struct {
	Breakpoints []Breakpoint `json:"breakpoints"`
}

// SetFunctionBreakpointsRequest: Replaces all existing function breakpoints with new function breakpoints.
// To clear all function breakpoints, specify an empty array.
// When a function breakpoint is hit, a `stopped` event (with reason `function breakpoint`) is generated.
// Clients should only call this request if the corresponding capability `supportsFunctionBreakpoints` is true.
type SetFunctionBreakpointsRequest struct {
	Request

	Arguments SetFunctionBreakpointsArguments `json:"arguments"`
}

// SetFunctionBreakpointsArguments: Arguments for `setFunctionBreakpoints` request.
type SetFunctionBreakpointsArguments struct {
	Breakpoints []FunctionBreakpoint `json:"breakpoints"`
}

// SetFunctionBreakpointsResponse: Response to `setFunctionBreakpoints` request.
// Returned is information about each breakpoint created by this request.
type SetFunctionBreakpointsResponse struct {
	Response

	Body SetFunctionBreakpointsResponseBody `json:"body"`
}

type SetFunctionBreakpointsResponseBody struct {
	Breakpoints []Breakpoint `json:"breakpoints"`
}

// SetExceptionBreakpointsRequest: The request configures the debugger's response to thrown exceptions. Each of the `filters`, `filterOptions`, and `exceptionOptions` in the request are independent configurations to a debug adapter indicating a kind of exception to catch. An exception thrown in a program should result in a `stopped` event from the debug adapter (with reason `exception`) if any of the configured filters match.
// Clients should only call this request if the corresponding capability `exceptionBreakpointFilters` returns one or more filters.
type SetExceptionBreakpointsRequest struct {
	Request

	Arguments SetExceptionBreakpointsArguments `json:"arguments"`
}

// SetExceptionBreakpointsArguments: Arguments for `setExceptionBreakpoints` request.
type SetExceptionBreakpointsArguments struct {
	Filters          []string                 `json:"filters"`
	FilterOptions    []ExceptionFilterOptions `json:"filterOptions,omitempty"`
	ExceptionOptions []ExceptionOptions       `json:"exceptionOptions,omitempty"`
}

// SetExceptionBreakpointsResponse: Response to `setExceptionBreakpoints` request.
// The response contains an array of `Breakpoint` objects with information about each exception breakpoint or filter. The `Breakpoint` objects are in the same order as the elements of the `filters`, `filterOptions`, `exceptionOptions` arrays given as arguments. If both `filters` and `filterOptions` are given, the returned array must start with `filters` information first, followed by `filterOptions` information.
// The `verified` property of a `Breakpoint` object signals whether the exception breakpoint or filter could be successfully created and whether the condition is valid. In case of an error the `message` property explains the problem. The `id` property can be used to introduce a unique ID for the exception breakpoint or filter so that it can be updated subsequently by sending breakpoint events.
// For backward compatibility both the `breakpoints` array and the enclosing `body` are optional. If these elements are missing a client is not able to show problems for individual exception breakpoints or filters.
type SetExceptionBreakpointsResponse struct {
	Response

	Body SetExceptionBreakpointsResponseBody `json:"body,omitempty"`
}

type SetExceptionBreakpointsResponseBody struct {
	Breakpoints []Breakpoint `json:"breakpoints,omitempty"`
}

// DataBreakpointInfoRequest: Obtains information on a possible data breakpoint that could be set on an expression or variable.
// Clients should only call this request if the corresponding capability `supportsDataBreakpoints` is true.
type DataBreakpointInfoRequest struct {
	Request

	Arguments DataBreakpointInfoArguments `json:"arguments"`
}

// DataBreakpointInfoArguments: Arguments for `dataBreakpointInfo` request.
type DataBreakpointInfoArguments struct {
	VariablesReference int    `json:"variablesReference,omitempty"`
	Name               string `json:"name"`
	FrameId            int    `json:"frameId,omitempty"`
	Bytes              int    `json:"bytes,omitempty"`
	AsAddress          bool   `json:"asAddress,omitempty"`
	Mode               string `json:"mode,omitempty"`
}

// DataBreakpointInfoResponse: Response to `dataBreakpointInfo` request.
type DataBreakpointInfoResponse struct {
	Response

	Body DataBreakpointInfoResponseBody `json:"body"`
}

type DataBreakpointInfoResponseBody struct {
	DataId      any                        `json:"dataId"`
	Description string                     `json:"description"`
	AccessTypes []DataBreakpointAccessType `json:"accessTypes,omitempty"`
	CanPersist  bool                       `json:"canPersist,omitempty"`
}

// SetDataBreakpointsRequest: Replaces all existing data breakpoints with new data breakpoints.
// To clear all data breakpoints, specify an empty array.
// When a data breakpoint is hit, a `stopped` event (with reason `data breakpoint`) is generated.
// Clients should only call this request if the corresponding capability `supportsDataBreakpoints` is true.
type SetDataBreakpointsRequest struct {
	Request

	Arguments SetDataBreakpointsArguments `json:"arguments"`
}

// SetDataBreakpointsArguments: Arguments for `setDataBreakpoints` request.
type SetDataBreakpointsArguments struct {
	Breakpoints []DataBreakpoint `json:"breakpoints"`
}

// SetDataBreakpointsResponse: Response to `setDataBreakpoints` request.
// Returned is information about each breakpoint created by this request.
type SetDataBreakpointsResponse struct {
	Response

	Body SetDataBreakpointsResponseBody `json:"body"`
}

type SetDataBreakpointsResponseBody struct {
	Breakpoints []Breakpoint `json:"breakpoints"`
}

// SetInstructionBreakpointsRequest: Replaces all existing instruction breakpoints. Typically, instruction breakpoints would be set from a disassembly window.
// To clear all instruction breakpoints, specify an empty array.
// When an instruction breakpoint is hit, a `stopped` event (with reason `instruction breakpoint`) is generated.
// Clients should only call this request if the corresponding capability `supportsInstructionBreakpoints` is true.
type SetInstructionBreakpointsRequest struct {
	Request

	Arguments SetInstructionBreakpointsArguments `json:"arguments"`
}

// SetInstructionBreakpointsArguments: Arguments for `setInstructionBreakpoints` request
type SetInstructionBreakpointsArguments struct {
	Breakpoints []InstructionBreakpoint `json:"breakpoints"`
}

// SetInstructionBreakpointsResponse: Response to `setInstructionBreakpoints` request
type SetInstructionBreakpointsResponse struct {
	Response

	Body SetInstructionBreakpointsResponseBody `json:"body"`
}

type SetInstructionBreakpointsResponseBody struct {
	Breakpoints []Breakpoint `json:"breakpoints"`
}

// ContinueRequest: The request resumes execution of all threads. If the debug adapter supports single thread execution (see capability `supportsSingleThreadExecutionRequests`), setting the `singleThread` argument to true resumes only the specified thread. If not all threads were resumed, the `allThreadsContinued` attribute of the response should be set to false.
type ContinueRequest struct {
	Request

	Arguments ContinueArguments `json:"arguments"`
}

// ContinueArguments: Arguments for `continue` request.
type ContinueArguments struct {
	ThreadId     int  `json:"threadId"`
	SingleThread bool `json:"singleThread,omitempty"`
}

// ContinueResponse: Response to `continue` request.
type ContinueResponse struct {
	Response

	Body ContinueResponseBody `json:"body"`
}

type ContinueResponseBody struct {
	AllThreadsContinued bool `json:"allThreadsContinued"`
}

// NextRequest: The request executes one step (in the given granularity) for the specified thread and allows all other threads to run freely by resuming them.
// If the debug adapter supports single thread execution (see capability `supportsSingleThreadExecutionRequests`), setting the `singleThread` argument to true prevents other suspended threads from resuming.
// The debug adapter first sends the response and then a `stopped` event (with reason `step`) after the step has completed.
type NextRequest struct {
	Request

	Arguments NextArguments `json:"arguments"`
}

// NextArguments: Arguments for `next` request.
type NextArguments struct {
	ThreadId     int                 `json:"threadId"`
	SingleThread bool                `json:"singleThread,omitempty"`
	Granularity  SteppingGranularity `json:"granularity,omitempty"`
}

// NextResponse: Response to `next` request. This is just an acknowledgement, so no body field is required.
type NextResponse struct {
	Response
}

// StepInRequest: The request resumes the given thread to step into a function/method and allows all other threads to run freely by resuming them.
// If the debug adapter supports single thread execution (see capability `supportsSingleThreadExecutionRequests`), setting the `singleThread` argument to true prevents other suspended threads from resuming.
// If the request cannot step into a target, `stepIn` behaves like the `next` request.
// The debug adapter first sends the response and then a `stopped` event (with reason `step`) after the step has completed.
// If there are multiple function/method calls (or other targets) on the source line,
// the argument `targetId` can be used to control into which target the `stepIn` should occur.
// The list of possible targets for a given source line can be retrieved via the `stepInTargets` request.
type StepInRequest struct {
	Request

	Arguments StepInArguments `json:"arguments"`
}

// StepInArguments: Arguments for `stepIn` request.
type StepInArguments struct {
	ThreadId     int                 `json:"threadId"`
	SingleThread bool                `json:"singleThread,omitempty"`
	TargetId     int                 `json:"targetId,omitempty"`
	Granularity  SteppingGranularity `json:"granularity,omitempty"`
}

// StepInResponse: Response to `stepIn` request. This is just an acknowledgement, so no body field is required.
type StepInResponse struct {
	Response
}

// StepOutRequest: The request resumes the given thread to step out (return) from a function/method and allows all other threads to run freely by resuming them.
// If the debug adapter supports single thread execution (see capability `supportsSingleThreadExecutionRequests`), setting the `singleThread` argument to true prevents other suspended threads from resuming.
// The debug adapter first sends the response and then a `stopped` event (with reason `step`) after the step has completed.
type StepOutRequest struct {
	Request

	Arguments StepOutArguments `json:"arguments"`
}

// StepOutArguments: Arguments for `stepOut` request.
type StepOutArguments struct {
	ThreadId     int                 `json:"threadId"`
	SingleThread bool                `json:"singleThread,omitempty"`
	Granularity  SteppingGranularity `json:"granularity,omitempty"`
}

// StepOutResponse: Response to `stepOut` request. This is just an acknowledgement, so no body field is required.
type StepOutResponse struct {
	Response
}

// StepBackRequest: The request executes one backward step (in the given granularity) for the specified thread and allows all other threads to run backward freely by resuming them.
// If the debug adapter supports single thread execution (see capability `supportsSingleThreadExecutionRequests`), setting the `singleThread` argument to true prevents other suspended threads from resuming.
// The debug adapter first sends the response and then a `stopped` event (with reason `step`) after the step has completed.
// Clients should only call this request if the corresponding capability `supportsStepBack` is true.
type StepBackRequest struct {
	Request

	Arguments StepBackArguments `json:"arguments"`
}

// StepBackArguments: Arguments for `stepBack` request.
type StepBackArguments struct {
	ThreadId     int                 `json:"threadId"`
	SingleThread bool                `json:"singleThread,omitempty"`
	Granularity  SteppingGranularity `json:"granularity,omitempty"`
}

// StepBackResponse: Response to `stepBack` request. This is just an acknowledgement, so no body field is required.
type StepBackResponse struct {
	Response
}

// ReverseContinueRequest: The request resumes backward execution of all threads. If the debug adapter supports single thread execution (see capability `supportsSingleThreadExecutionRequests`), setting the `singleThread` argument to true resumes only the specified thread. If not all threads were resumed, the `allThreadsContinued` attribute of the response should be set to false.
// Clients should only call this request if the corresponding capability `supportsStepBack` is true.
type ReverseContinueRequest struct {
	Request

	Arguments ReverseContinueArguments `json:"arguments"`
}

// ReverseContinueArguments: Arguments for `reverseContinue` request.
type ReverseContinueArguments struct {
	ThreadId     int  `json:"threadId"`
	SingleThread bool `json:"singleThread,omitempty"`
}

// ReverseContinueResponse: Response to `reverseContinue` request. This is just an acknowledgement, so no body field is required.
type ReverseContinueResponse struct {
	Response
}

// RestartFrameRequest: The request restarts execution of the specified stack frame.
// The debug adapter first sends the response and then a `stopped` event (with reason `restart`) after the restart has completed.
// Clients should only call this request if the corresponding capability `supportsRestartFrame` is true.
type RestartFrameRequest struct {
	Request

	Arguments RestartFrameArguments `json:"arguments"`
}

// RestartFrameArguments: Arguments for `restartFrame` request.
type RestartFrameArguments struct {
	FrameId int `json:"frameId"`
}

// RestartFrameResponse: Response to `restartFrame` request. This is just an acknowledgement, so no body field is required.
type RestartFrameResponse struct {
	Response
}

// GotoRequest: The request sets the location where the debuggee will continue to run.
// This makes it possible to skip the execution of code or to execute code again.
// The code between the current location and the goto target is not executed but skipped.
// The debug adapter first sends the response and then a `stopped` event with reason `goto`.
// Clients should only call this request if the corresponding capability `supportsGotoTargetsRequest` is true (because only then goto targets exist that can be passed as arguments).
type GotoRequest struct {
	Request

	Arguments GotoArguments `json:"arguments"`
}

// GotoArguments: Arguments for `goto` request.
type GotoArguments struct {
	ThreadId int `json:"threadId"`
	TargetId int `json:"targetId"`
}

// GotoResponse: Response to `goto` request. This is just an acknowledgement, so no body field is required.
type GotoResponse struct {
	Response
}

// PauseRequest: The request suspends the debuggee.
// The debug adapter first sends the response and then a `stopped` event (with reason `pause`) after the thread has been paused successfully.
type PauseRequest struct {
	Request

	Arguments PauseArguments `json:"arguments"`
}

// PauseArguments: Arguments for `pause` request.
type PauseArguments struct {
	ThreadId int `json:"threadId"`
}

// PauseResponse: Response to `pause` request. This is just an acknowledgement, so no body field is required.
type PauseResponse struct {
	Response
}

// StackTraceRequest: The request returns a stacktrace from the current execution state of a given thread.
// A client can request all stack frames by omitting the startFrame and levels arguments. For performance-conscious clients and if the corresponding capability `supportsDelayedStackTraceLoading` is true, stack frames can be retrieved in a piecemeal way with the `startFrame` and `levels` arguments. The response of the `stackTrace` request may contain a `totalFrames` property that hints at the total number of frames in the stack. If a client needs this total number upfront, it can issue a request for a single (first) frame and depending on the value of `totalFrames` decide how to proceed. In any case a client should be prepared to receive fewer frames than requested, which is an indication that the end of the stack has been reached.
type StackTraceRequest struct {
	Request

	Arguments StackTraceArguments `json:"arguments"`
}

// StackTraceArguments: Arguments for `stackTrace` request.
type StackTraceArguments struct {
	ThreadId   int               `json:"threadId"`
	StartFrame int               `json:"startFrame,omitempty"`
	Levels     int               `json:"levels,omitempty"`
	Format     *StackFrameFormat `json:"format,omitempty"`
}

// StackTraceResponse: Response to `stackTrace` request.
type StackTraceResponse struct {
	Response

	Body StackTraceResponseBody `json:"body"`
}

type StackTraceResponseBody struct {
	StackFrames []StackFrame `json:"stackFrames"`
	TotalFrames int          `json:"totalFrames,omitempty"`
}

// ScopesRequest: The request returns the variable scopes for a given stack frame ID.
type ScopesRequest struct {
	Request

	Arguments ScopesArguments `json:"arguments"`
}

// ScopesArguments: Arguments for `scopes` request.
type ScopesArguments struct {
	FrameId int `json:"frameId"`
}

// ScopesResponse: Response to `scopes` request.
type ScopesResponse struct {
	Response

	Body ScopesResponseBody `json:"body"`
}

type ScopesResponseBody struct {
	Scopes []Scope `json:"scopes"`
}

// VariablesRequest: Retrieves all child variables for the given variable reference.
// A filter can be used to limit the fetched children to either named or indexed children.
type VariablesRequest struct {
	Request

	Arguments VariablesArguments `json:"arguments"`
}

// VariablesArguments: Arguments for `variables` request.
type VariablesArguments struct {
	VariablesReference int          `json:"variablesReference"`
	Filter             string       `json:"filter,omitempty"`
	Start              int          `json:"start,omitempty"`
	Count              int          `json:"count,omitempty"`
	Format             *ValueFormat `json:"format,omitempty"`
}

// VariablesResponse: Response to `variables` request.
type VariablesResponse struct {
	Response

	Body VariablesResponseBody `json:"body"`
}

type VariablesResponseBody struct {
	Variables []Variable `json:"variables"`
}

// SetVariableRequest: Set the variable with the given name in the variable container to a new value. Clients should only call this request if the corresponding capability `supportsSetVariable` is true.
// If a debug adapter implements both `setVariable` and `setExpression`, a client will only use `setExpression` if the variable has an `evaluateName` property.
type SetVariableRequest struct {
	Request

	Arguments SetVariableArguments `json:"arguments"`
}

// SetVariableArguments: Arguments for `setVariable` request.
type SetVariableArguments struct {
	VariablesReference int          `json:"variablesReference"`
	Name               string       `json:"name"`
	Value              string       `json:"value"`
	Format             *ValueFormat `json:"format,omitempty"`
}

// SetVariableResponse: Response to `setVariable` request.
type SetVariableResponse struct {
	Response

	Body SetVariableResponseBody `json:"body"`
}

type SetVariableResponseBody struct {
	Value                  string `json:"value"`
	Type                   string `json:"type,omitempty"`
	VariablesReference     int    `json:"variablesReference,omitempty"`
	NamedVariables         int    `json:"namedVariables,omitempty"`
	IndexedVariables       int    `json:"indexedVariables,omitempty"`
	MemoryReference        string `json:"memoryReference,omitempty"`
	ValueLocationReference int    `json:"valueLocationReference,omitempty"`
}

// SourceRequest: The request retrieves the source code for a given source reference.
type SourceRequest struct {
	Request

	Arguments SourceArguments `json:"arguments"`
}

// SourceArguments: Arguments for `source` request.
type SourceArguments struct {
	Source          *Source `json:"source,omitempty"`
	SourceReference int     `json:"sourceReference"`
}

// SourceResponse: Response to `source` request.
type SourceResponse struct {
	Response

	Body SourceResponseBody `json:"body"`
}

type SourceResponseBody struct {
	Content  string `json:"content"`
	MimeType string `json:"mimeType,omitempty"`
}

// ThreadsRequest: The request retrieves a list of all threads.
type ThreadsRequest struct {
	Request
}

// ThreadsResponse: Response to `threads` request.
type ThreadsResponse struct {
	Response

	Body ThreadsResponseBody `json:"body"`
}

type ThreadsResponseBody struct {
	Threads []Thread `json:"threads"`
}

// TerminateThreadsRequest: The request terminates the threads with the given ids.
// Clients should only call this request if the corresponding capability `supportsTerminateThreadsRequest` is true.
type TerminateThreadsRequest struct {
	Request

	Arguments TerminateThreadsArguments `json:"arguments"`
}

// TerminateThreadsArguments: Arguments for `terminateThreads` request.
type TerminateThreadsArguments struct {
	ThreadIds []int `json:"threadIds,omitempty"`
}

// TerminateThreadsResponse: Response to `terminateThreads` request. This is just an acknowledgement, no body field is required.
type TerminateThreadsResponse struct {
	Response
}

// ModulesRequest: Modules can be retrieved from the debug adapter with this request which can either return all modules or a range of modules to support paging.
// Clients should only call this request if the corresponding capability `supportsModulesRequest` is true.
type ModulesRequest struct {
	Request

	Arguments ModulesArguments `json:"arguments"`
}

// ModulesArguments: Arguments for `modules` request.
type ModulesArguments struct {
	StartModule int `json:"startModule,omitempty"`
	ModuleCount int `json:"moduleCount,omitempty"`
}

// ModulesResponse: Response to `modules` request.
type ModulesResponse struct {
	Response

	Body ModulesResponseBody `json:"body"`
}

type ModulesResponseBody struct {
	Modules      []Module `json:"modules"`
	TotalModules int      `json:"totalModules,omitempty"`
}

// LoadedSourcesRequest: Retrieves the set of all sources currently loaded by the debugged process.
// Clients should only call this request if the corresponding capability `supportsLoadedSourcesRequest` is true.
type LoadedSourcesRequest struct {
	Request

	Arguments *LoadedSourcesArguments `json:"arguments,omitempty"`
}

// LoadedSourcesArguments: Arguments for `loadedSources` request.
type LoadedSourcesArguments struct {
}

// LoadedSourcesResponse: Response to `loadedSources` request.
type LoadedSourcesResponse struct {
	Response

	Body LoadedSourcesResponseBody `json:"body"`
}

type LoadedSourcesResponseBody struct {
	Sources []Source `json:"sources"`
}

// EvaluateRequest: Evaluates the given expression in the context of a stack frame.
// The expression has access to any variables and arguments that are in scope.
type EvaluateRequest struct {
	Request

	Arguments EvaluateArguments `json:"arguments"`
}

// EvaluateArguments: Arguments for `evaluate` request.
type EvaluateArguments struct {
	Expression string       `json:"expression"`
	FrameId    int          `json:"frameId,omitempty"`
	Line       int          `json:"line,omitempty"`
	Column     int          `json:"column,omitempty"`
	Source     *Source      `json:"source,omitempty"`
	Context    string       `json:"context,omitempty"`
	Format     *ValueFormat `json:"format,omitempty"`
}

// EvaluateResponse: Response to `evaluate` request.
type EvaluateResponse struct {
	Response

	Body EvaluateResponseBody `json:"body"`
}

type EvaluateResponseBody struct {
	Result                 string                    `json:"result"`
	Type                   string                    `json:"type,omitempty"`
	PresentationHint       *VariablePresentationHint `json:"presentationHint,omitempty"`
	VariablesReference     int                       `json:"variablesReference"`
	NamedVariables         int                       `json:"namedVariables,omitempty"`
	IndexedVariables       int                       `json:"indexedVariables,omitempty"`
	MemoryReference        string                    `json:"memoryReference,omitempty"`
	ValueLocationReference int                       `json:"valueLocationReference,omitempty"`
}

// SetExpressionRequest: Evaluates the given `value` expression and assigns it to the `expression` which must be a modifiable l-value.
// The expressions have access to any variables and arguments that are in scope of the specified frame.
// Clients should only call this request if the corresponding capability `supportsSetExpression` is true.
// If a debug adapter implements both `setExpression` and `setVariable`, a client uses `setExpression` if the variable has an `evaluateName` property.
type SetExpressionRequest struct {
	Request

	Arguments SetExpressionArguments `json:"arguments"`
}

// SetExpressionArguments: Arguments for `setExpression` request.
type SetExpressionArguments struct {
	Expression string       `json:"expression"`
	Value      string       `json:"value"`
	FrameId    int          `json:"frameId,omitempty"`
	Format     *ValueFormat `json:"format,omitempty"`
}

// SetExpressionResponse: Response to `setExpression` request.
type SetExpressionResponse struct {
	Response

	Body SetExpressionResponseBody `json:"body"`
}

type SetExpressionResponseBody struct {
	Value                  string                    `json:"value"`
	Type                   string                    `json:"type,omitempty"`
	PresentationHint       *VariablePresentationHint `json:"presentationHint,omitempty"`
	VariablesReference     int                       `json:"variablesReference,omitempty"`
	NamedVariables         int                       `json:"namedVariables,omitempty"`
	IndexedVariables       int                       `json:"indexedVariables,omitempty"`
	MemoryReference        string                    `json:"memoryReference,omitempty"`
	ValueLocationReference int                       `json:"valueLocationReference,omitempty"`
}

// StepInTargetsRequest: This request retrieves the possible step-in targets for the specified stack frame.
// These targets can be used in the `stepIn` request.
// Clients should only call this request if the corresponding capability `supportsStepInTargetsRequest` is true.
type StepInTargetsRequest struct {
	Request

	Arguments StepInTargetsArguments `json:"arguments"`
}

// StepInTargetsArguments: Arguments for `stepInTargets` request.
type StepInTargetsArguments struct {
	FrameId int `json:"frameId"`
}

// StepInTargetsResponse: Response to `stepInTargets` request.
type StepInTargetsResponse struct {
	Response

	Body StepInTargetsResponseBody `json:"body"`
}

type StepInTargetsResponseBody struct {
	Targets []StepInTarget `json:"targets"`
}

// GotoTargetsRequest: This request retrieves the possible goto targets for the specified source location.
// These targets can be used in the `goto` request.
// Clients should only call this request if the corresponding capability `supportsGotoTargetsRequest` is true.
type GotoTargetsRequest struct {
	Request

	Arguments GotoTargetsArguments `json:"arguments"`
}

// GotoTargetsArguments: Arguments for `gotoTargets` request.
type GotoTargetsArguments struct {
	Source Source `json:"source"`
	Line   int    `json:"line"`
	Column int    `json:"column,omitempty"`
}

// GotoTargetsResponse: Response to `gotoTargets` request.
type GotoTargetsResponse struct {
	Response

	Body GotoTargetsResponseBody `json:"body"`
}

type GotoTargetsResponseBody struct {
	Targets []GotoTarget `json:"targets"`
}

// CompletionsRequest: Returns a list of possible completions for a given caret position and text.
// Clients should only call this request if the corresponding capability `supportsCompletionsRequest` is true.
type CompletionsRequest struct {
	Request

	Arguments CompletionsArguments `json:"arguments"`
}

// CompletionsArguments: Arguments for `completions` request.
type CompletionsArguments struct {
	FrameId int    `json:"frameId,omitempty"`
	Text    string `json:"text"`
	Column  int    `json:"column"`
	Line    int    `json:"line,omitempty"`
}

// CompletionsResponse: Response to `completions` request.
type CompletionsResponse struct {
	Response

	Body CompletionsResponseBody `json:"body"`
}

type CompletionsResponseBody struct {
	Targets []CompletionItem `json:"targets"`
}

// ExceptionInfoRequest: Retrieves the details of the exception that caused this event to be raised.
// Clients should only call this request if the corresponding capability `supportsExceptionInfoRequest` is true.
type ExceptionInfoRequest struct {
	Request

	Arguments ExceptionInfoArguments `json:"arguments"`
}

// ExceptionInfoArguments: Arguments for `exceptionInfo` request.
type ExceptionInfoArguments struct {
	ThreadId int `json:"threadId"`
}

// ExceptionInfoResponse: Response to `exceptionInfo` request.
type ExceptionInfoResponse struct {
	Response

	Body ExceptionInfoResponseBody `json:"body"`
}

type ExceptionInfoResponseBody struct {
	ExceptionId string             `json:"exceptionId"`
	Description string             `json:"description,omitempty"`
	BreakMode   ExceptionBreakMode `json:"breakMode"`
	Details     *ExceptionDetails  `json:"details,omitempty"`
}

// ReadMemoryRequest: Reads bytes from memory at the provided location.
// Clients should only call this request if the corresponding capability `supportsReadMemoryRequest` is true.
type ReadMemoryRequest struct {
	Request

	Arguments ReadMemoryArguments `json:"arguments"`
}

// ReadMemoryArguments: Arguments for `readMemory` request.
type ReadMemoryArguments struct {
	MemoryReference string `json:"memoryReference"`
	Offset          int    `json:"offset,omitempty"`
	Count           int    `json:"count"`
}

// ReadMemoryResponse: Response to `readMemory` request.
type ReadMemoryResponse struct {
	Response

	Body ReadMemoryResponseBody `json:"body,omitempty"`
}

type ReadMemoryResponseBody struct {
	Address         string `json:"address"`
	UnreadableBytes int    `json:"unreadableBytes,omitempty"`
	Data            string `json:"data,omitempty"`
}

// WriteMemoryRequest: Writes bytes to memory at the provided location.
// Clients should only call this request if the corresponding capability `supportsWriteMemoryRequest` is true.
type WriteMemoryRequest struct {
	Request

	Arguments WriteMemoryArguments `json:"arguments"`
}

// WriteMemoryArguments: Arguments for `writeMemory` request.
type WriteMemoryArguments struct {
	MemoryReference string `json:"memoryReference"`
	Offset          int    `json:"offset,omitempty"`
	AllowPartial    bool   `json:"allowPartial,omitempty"`
	Data            string `json:"data"`
}

// WriteMemoryResponse: Response to `writeMemory` request.
type WriteMemoryResponse struct {
	Response

	Body WriteMemoryResponseBody `json:"body,omitempty"`
}

type WriteMemoryResponseBody struct {
	Offset       int `json:"offset,omitempty"`
	BytesWritten int `json:"bytesWritten,omitempty"`
}

// DisassembleRequest: Disassembles code stored at the provided location.
// Clients should only call this request if the corresponding capability `supportsDisassembleRequest` is true.
type DisassembleRequest struct {
	Request

	Arguments DisassembleArguments `json:"arguments"`
}

// DisassembleArguments: Arguments for `disassemble` request.
type DisassembleArguments struct {
	MemoryReference   string `json:"memoryReference"`
	Offset            int    `json:"offset,omitempty"`
	InstructionOffset int    `json:"instructionOffset,omitempty"`
	InstructionCount  int    `json:"instructionCount"`
	ResolveSymbols    bool   `json:"resolveSymbols,omitempty"`
}

// DisassembleResponse: Response to `disassemble` request.
type DisassembleResponse struct {
	Response

	Body DisassembleResponseBody `json:"body,omitempty"`
}

type DisassembleResponseBody struct {
	Instructions []DisassembledInstruction `json:"instructions"`
}

// LocationsRequest: Looks up information about a location reference previously returned by the debug adapter.
type LocationsRequest struct {
	Request

	Arguments LocationsArguments `json:"arguments"`
}

// LocationsArguments: Arguments for `locations` request.
type LocationsArguments struct {
	LocationReference int `json:"locationReference"`
}

// LocationsResponse: Response to `locations` request.
type LocationsResponse struct {
	Response

	Body LocationsResponseBody `json:"body,omitempty"`
}

type LocationsResponseBody struct {
	Source    Source `json:"source"`
	Line      int    `json:"line"`
	Column    int    `json:"column,omitempty"`
	EndLine   int    `json:"endLine,omitempty"`
	EndColumn int    `json:"endColumn,omitempty"`
}

// Capabilities: Information about the capabilities of a debug adapter.
type Capabilities struct {
	SupportsConfigurationDoneRequest      bool                         `json:"supportsConfigurationDoneRequest,omitempty"`
	SupportsFunctionBreakpoints           bool                         `json:"supportsFunctionBreakpoints,omitempty"`
	SupportsConditionalBreakpoints        bool                         `json:"supportsConditionalBreakpoints,omitempty"`
	SupportsHitConditionalBreakpoints     bool                         `json:"supportsHitConditionalBreakpoints,omitempty"`
	SupportsEvaluateForHovers             bool                         `json:"supportsEvaluateForHovers,omitempty"`
	ExceptionBreakpointFilters            []ExceptionBreakpointsFilter `json:"exceptionBreakpointFilters,omitempty"`
	SupportsStepBack                      bool                         `json:"supportsStepBack,omitempty"`
	SupportsSetVariable                   bool                         `json:"supportsSetVariable,omitempty"`
	SupportsRestartFrame                  bool                         `json:"supportsRestartFrame,omitempty"`
	SupportsGotoTargetsRequest            bool                         `json:"supportsGotoTargetsRequest,omitempty"`
	SupportsStepInTargetsRequest          bool                         `json:"supportsStepInTargetsRequest,omitempty"`
	SupportsCompletionsRequest            bool                         `json:"supportsCompletionsRequest,omitempty"`
	CompletionTriggerCharacters           []string                     `json:"completionTriggerCharacters,omitempty"`
	SupportsModulesRequest                bool                         `json:"supportsModulesRequest,omitempty"`
	AdditionalModuleColumns               []ColumnDescriptor           `json:"additionalModuleColumns,omitempty"`
	SupportedChecksumAlgorithms           []ChecksumAlgorithm          `json:"supportedChecksumAlgorithms,omitempty"`
	SupportsRestartRequest                bool                         `json:"supportsRestartRequest,omitempty"`
	SupportsExceptionOptions              bool                         `json:"supportsExceptionOptions,omitempty"`
	SupportsValueFormattingOptions        bool                         `json:"supportsValueFormattingOptions,omitempty"`
	SupportsExceptionInfoRequest          bool                         `json:"supportsExceptionInfoRequest,omitempty"`
	SupportTerminateDebuggee              bool                         `json:"supportTerminateDebuggee,omitempty"`
	SupportSuspendDebuggee                bool                         `json:"supportSuspendDebuggee,omitempty"`
	SupportsDelayedStackTraceLoading      bool                         `json:"supportsDelayedStackTraceLoading,omitempty"`
	SupportsLoadedSourcesRequest          bool                         `json:"supportsLoadedSourcesRequest,omitempty"`
	SupportsLogPoints                     bool                         `json:"supportsLogPoints,omitempty"`
	SupportsTerminateThreadsRequest       bool                         `json:"supportsTerminateThreadsRequest,omitempty"`
	SupportsSetExpression                 bool                         `json:"supportsSetExpression,omitempty"`
	SupportsTerminateRequest              bool                         `json:"supportsTerminateRequest,omitempty"`
	SupportsDataBreakpoints               bool                         `json:"supportsDataBreakpoints,omitempty"`
	SupportsReadMemoryRequest             bool                         `json:"supportsReadMemoryRequest,omitempty"`
	SupportsWriteMemoryRequest            bool                         `json:"supportsWriteMemoryRequest,omitempty"`
	SupportsDisassembleRequest            bool                         `json:"supportsDisassembleRequest,omitempty"`
	SupportsCancelRequest                 bool                         `json:"supportsCancelRequest,omitempty"`
	SupportsBreakpointLocationsRequest    bool                         `json:"supportsBreakpointLocationsRequest,omitempty"`
	SupportsClipboardContext              bool                         `json:"supportsClipboardContext,omitempty"`
	SupportsSteppingGranularity           bool                         `json:"supportsSteppingGranularity,omitempty"`
	SupportsInstructionBreakpoints        bool                         `json:"supportsInstructionBreakpoints,omitempty"`
	SupportsExceptionFilterOptions        bool                         `json:"supportsExceptionFilterOptions,omitempty"`
	SupportsSingleThreadExecutionRequests bool                         `json:"supportsSingleThreadExecutionRequests,omitempty"`
	SupportsDataBreakpointBytes           bool                         `json:"supportsDataBreakpointBytes,omitempty"`
	BreakpointModes                       []BreakpointMode             `json:"breakpointModes,omitempty"`
	SupportsANSIStyling                   bool                         `json:"supportsANSIStyling,omitempty"`
}

// ExceptionBreakpointsFilter: An `ExceptionBreakpointsFilter` is shown in the UI as an filter option for configuring how exceptions are dealt with.
type ExceptionBreakpointsFilter struct {
	Filter               string `json:"filter"`
	Label                string `json:"label"`
	Description          string `json:"description,omitempty"`
	Default              bool   `json:"default,omitempty"`
	SupportsCondition    bool   `json:"supportsCondition,omitempty"`
	ConditionDescription string `json:"conditionDescription,omitempty"`
}

// ErrorMessage: A structured message object. Used to return errors from requests.
type ErrorMessage struct {
	Id            int               `json:"id"`
	Format        string            `json:"format"`
	Variables     map[string]string `json:"variables,omitempty"`
	SendTelemetry bool              `json:"sendTelemetry,omitempty"`
	ShowUser      bool              `json:"showUser"`
	Url           string            `json:"url,omitempty"`
	UrlLabel      string            `json:"urlLabel,omitempty"`
}

// Module: A Module object represents a row in the modules view.
// The `id` attribute identifies a module in the modules view and is used in a `module` event for identifying a module for adding, updating or deleting.
// The `name` attribute is used to minimally render the module in the UI.
//
// Additional attributes can be added to the module. They show up in the module view if they have a corresponding `ColumnDescriptor`.
//
// To avoid an unnecessary proliferation of additional attributes with similar semantics but different names, we recommend to re-use attributes from the 'recommended' list below first, and only introduce new attributes if nothing appropriate could be found.
type Module struct {
	Id             any    `json:"id"`
	Name           string `json:"name"`
	Path           string `json:"path,omitempty"`
	IsOptimized    bool   `json:"isOptimized,omitempty"`
	IsUserCode     bool   `json:"isUserCode,omitempty"`
	Version        string `json:"version,omitempty"`
	SymbolStatus   string `json:"symbolStatus,omitempty"`
	SymbolFilePath string `json:"symbolFilePath,omitempty"`
	DateTimeStamp  string `json:"dateTimeStamp,omitempty"`
	AddressRange   string `json:"addressRange,omitempty"`
}

// ColumnDescriptor: A `ColumnDescriptor` specifies what module attribute to show in a column of the modules view, how to format it,
// and what the column's label should be.
// It is only used if the underlying UI actually supports this level of customization.
type ColumnDescriptor struct {
	AttributeName string `json:"attributeName"`
	Label         string `json:"label"`
	Format        string `json:"format,omitempty"`
	Type          string `json:"type,omitempty"`
	Width         int    `json:"width,omitempty"`
}

// Thread: A Thread
type Thread struct {
	Id   int    `json:"id"`
	Name string `json:"name"`
}

// Source: A `Source` is a descriptor for source code.
// It is returned from the debug adapter as part of a `StackFrame` and it is used by clients when specifying breakpoints.
type Source struct {
	Name             string          `json:"name,omitempty"`
	Path             string          `json:"path,omitempty"`
	SourceReference  int             `json:"sourceReference,omitempty"`
	PresentationHint string          `json:"presentationHint,omitempty"`
	Origin           string          `json:"origin,omitempty"`
	Sources          []Source        `json:"sources,omitempty"`
	AdapterData      json.RawMessage `json:"adapterData,omitempty"`
	Checksums        []Checksum      `json:"checksums,omitempty"`
}

// StackFrame: A Stackframe contains the source location.
type StackFrame struct {
	Id                          int     `json:"id"`
	Name                        string  `json:"name"`
	Source                      *Source `json:"source,omitempty"`
	Line                        int     `json:"line"`
	Column                      int     `json:"column"`
	EndLine                     int     `json:"endLine,omitempty"`
	EndColumn                   int     `json:"endColumn,omitempty"`
	CanRestart                  bool    `json:"canRestart,omitempty"`
	InstructionPointerReference string  `json:"instructionPointerReference,omitempty"`
	ModuleId                    any     `json:"moduleId,omitempty"`
	PresentationHint            string  `json:"presentationHint,omitempty"`
}

// Scope: A `Scope` is a named container for variables. Optionally a scope can map to a source or a range within a source.
type Scope struct {
	Name               string  `json:"name"`
	PresentationHint   string  `json:"presentationHint,omitempty"`
	VariablesReference int     `json:"variablesReference"`
	NamedVariables     int     `json:"namedVariables,omitempty"`
	IndexedVariables   int     `json:"indexedVariables,omitempty"`
	Expensive          bool    `json:"expensive"`
	Source             *Source `json:"source,omitempty"`
	Line               int     `json:"line,omitempty"`
	Column             int     `json:"column,omitempty"`
	EndLine            int     `json:"endLine,omitempty"`
	EndColumn          int     `json:"endColumn,omitempty"`
}

// Variable: A Variable is a name/value pair.
// The `type` attribute is shown if space permits or when hovering over the variable's name.
// The `kind` attribute is used to render additional properties of the variable, e.g. different icons can be used to indicate that a variable is public or private.
// If the value is structured (has children), a handle is provided to retrieve the children with the `variables` request.
// If the number of named or indexed children is large, the numbers should be returned via the `namedVariables` and `indexedVariables` attributes.
// The client can use this information to present the children in a paged UI and fetch them in chunks.
type Variable struct {
	Name                         string                    `json:"name"`
	Value                        string                    `json:"value"`
	Type                         string                    `json:"type,omitempty"`
	PresentationHint             *VariablePresentationHint `json:"presentationHint,omitempty"`
	EvaluateName                 string                    `json:"evaluateName,omitempty"`
	VariablesReference           int                       `json:"variablesReference"`
	NamedVariables               int                       `json:"namedVariables,omitempty"`
	IndexedVariables             int                       `json:"indexedVariables,omitempty"`
	MemoryReference              string                    `json:"memoryReference,omitempty"`
	DeclarationLocationReference int                       `json:"declarationLocationReference,omitempty"`
	ValueLocationReference       int                       `json:"valueLocationReference,omitempty"`
}

// VariablePresentationHint: Properties of a variable that can be used to determine how to render the variable in the UI.
type VariablePresentationHint struct {
	Kind       string   `json:"kind,omitempty"`
	Attributes []string `json:"attributes,omitempty"`
	Visibility string   `json:"visibility,omitempty"`
	Lazy       bool     `json:"lazy,omitempty"`
}

// BreakpointLocation: Properties of a breakpoint location returned from the `breakpointLocations` request.
type BreakpointLocation struct {
	Line      int `json:"line"`
	Column    int `json:"column,omitempty"`
	EndLine   int `json:"endLine,omitempty"`
	EndColumn int `json:"endColumn,omitempty"`
}

// SourceBreakpoint: Properties of a breakpoint or logpoint passed to the `setBreakpoints` request.
type SourceBreakpoint struct {
	Line         int    `json:"line"`
	Column       int    `json:"column,omitempty"`
	Condition    string `json:"condition,omitempty"`
	HitCondition string `json:"hitCondition,omitempty"`
	LogMessage   string `json:"logMessage,omitempty"`
	Mode         string `json:"mode,omitempty"`
}

// FunctionBreakpoint: Properties of a breakpoint passed to the `setFunctionBreakpoints` request.
type FunctionBreakpoint struct {
	Name         string `json:"name"`
	Condition    string `json:"condition,omitempty"`
	HitCondition string `json:"hitCondition,omitempty"`
}

// DataBreakpointAccessType: This enumeration defines all possible access types for data breakpoints.
type DataBreakpointAccessType string

// DataBreakpoint: Properties of a data breakpoint passed to the `setDataBreakpoints` request.
type DataBreakpoint struct {
	DataId       string                   `json:"dataId"`
	AccessType   DataBreakpointAccessType `json:"accessType,omitempty"`
	Condition    string                   `json:"condition,omitempty"`
	HitCondition string                   `json:"hitCondition,omitempty"`
}

// InstructionBreakpoint: Properties of a breakpoint passed to the `setInstructionBreakpoints` request
type InstructionBreakpoint struct {
	InstructionReference string `json:"instructionReference"`
	Offset               int    `json:"offset,omitempty"`
	Condition            string `json:"condition,omitempty"`
	HitCondition         string `json:"hitCondition,omitempty"`
	Mode                 string `json:"mode,omitempty"`
}

// Breakpoint: Information about a breakpoint created in `setBreakpoints`, `setFunctionBreakpoints`, `setInstructionBreakpoints`, or `setDataBreakpoints` requests.
type Breakpoint struct {
	Id                   int     `json:"id,omitempty"`
	Verified             bool    `json:"verified"`
	Message              string  `json:"message,omitempty"`
	Source               *Source `json:"source,omitempty"`
	Line                 int     `json:"line,omitempty"`
	Column               int     `json:"column,omitempty"`
	EndLine              int     `json:"endLine,omitempty"`
	EndColumn            int     `json:"endColumn,omitempty"`
	InstructionReference string  `json:"instructionReference,omitempty"`
	Offset               int     `json:"offset,omitempty"`
	Reason               string  `json:"reason,omitempty"`
}

// SteppingGranularity: The granularity of one 'step' in the stepping requests `next`, `stepIn`, `stepOut`, and `stepBack`.
type SteppingGranularity string

// StepInTarget: A `StepInTarget` can be used in the `stepIn` request and determines into which single target the `stepIn` request should step.
type StepInTarget struct {
	Id        int    `json:"id"`
	Label     string `json:"label"`
	Line      int    `json:"line,omitempty"`
	Column    int    `json:"column,omitempty"`
	EndLine   int    `json:"endLine,omitempty"`
	EndColumn int    `json:"endColumn,omitempty"`
}

// GotoTarget: A `GotoTarget` describes a code location that can be used as a target in the `goto` request.
// The possible goto targets can be determined via the `gotoTargets` request.
type GotoTarget struct {
	Id                          int    `json:"id"`
	Label                       string `json:"label"`
	Line                        int    `json:"line"`
	Column                      int    `json:"column,omitempty"`
	EndLine                     int    `json:"endLine,omitempty"`
	EndColumn                   int    `json:"endColumn,omitempty"`
	InstructionPointerReference string `json:"instructionPointerReference,omitempty"`
}

// CompletionItem: `CompletionItems` are the suggestions returned from the `completions` request.
type CompletionItem struct {
	Label           string             `json:"label"`
	Text            string             `json:"text,omitempty"`
	SortText        string             `json:"sortText,omitempty"`
	Detail          string             `json:"detail,omitempty"`
	Type            CompletionItemType `json:"type,omitempty"`
	Start           int                `json:"start,omitempty"`
	Length          int                `json:"length,omitempty"`
	SelectionStart  int                `json:"selectionStart,omitempty"`
	SelectionLength int                `json:"selectionLength,omitempty"`
}

// CompletionItemType: Some predefined types for the CompletionItem. Please note that not all clients have specific icons for all of them.
type CompletionItemType string

// ChecksumAlgorithm: Names of checksum algorithms that may be supported by a debug adapter.
type ChecksumAlgorithm string

// Checksum: The checksum of an item calculated by the specified algorithm.
type Checksum struct {
	Algorithm ChecksumAlgorithm `json:"algorithm"`
	Checksum  string            `json:"checksum"`
}

// ValueFormat: Provides formatting information for a value.
type ValueFormat struct {
	Hex bool `json:"hex,omitempty"`
}

// StackFrameFormat: Provides formatting information for a stack frame.
type StackFrameFormat struct {
	ValueFormat

	Parameters      bool `json:"parameters,omitempty"`
	ParameterTypes  bool `json:"parameterTypes,omitempty"`
	ParameterNames  bool `json:"parameterNames,omitempty"`
	ParameterValues bool `json:"parameterValues,omitempty"`
	Line            bool `json:"line,omitempty"`
	Module          bool `json:"module,omitempty"`
	IncludeAll      bool `json:"includeAll,omitempty"`
}

// ExceptionFilterOptions: An `ExceptionFilterOptions` is used to specify an exception filter together with a condition for the `setExceptionBreakpoints` request.
type ExceptionFilterOptions struct {
	FilterId  string `json:"filterId"`
	Condition string `json:"condition,omitempty"`
	Mode      string `json:"mode,omitempty"`
}

// ExceptionOptions: An `ExceptionOptions` assigns configuration options to a set of exceptions.
type ExceptionOptions struct {
	Path      []ExceptionPathSegment `json:"path,omitempty"`
	BreakMode ExceptionBreakMode     `json:"breakMode"`
}

// ExceptionBreakMode: This enumeration defines all possible conditions when a thrown exception should result in a break.
// never: never breaks,
// always: always breaks,
// unhandled: breaks when exception unhandled,
// userUnhandled: breaks if the exception is not handled by user code.
type ExceptionBreakMode string

// ExceptionPathSegment: An `ExceptionPathSegment` represents a segment in a path that is used to match leafs or nodes in a tree of exceptions.
// If a segment consists of more than one name, it matches the names provided if `negate` is false or missing, or it matches anything except the names provided if `negate` is true.
type ExceptionPathSegment struct {
	Negate bool     `json:"negate,omitempty"`
	Names  []string `json:"names"`
}

// ExceptionDetails: Detailed information about an exception that has occurred.
type ExceptionDetails struct {
	Message        string             `json:"message,omitempty"`
	TypeName       string             `json:"typeName,omitempty"`
	FullTypeName   string             `json:"fullTypeName,omitempty"`
	EvaluateName   string             `json:"evaluateName,omitempty"`
	StackTrace     string             `json:"stackTrace,omitempty"`
	InnerException []ExceptionDetails `json:"innerException,omitempty"`
}

// DisassembledInstruction: Represents a single disassembled instruction.
type DisassembledInstruction struct {
	Address          string  `json:"address"`
	InstructionBytes string  `json:"instructionBytes,omitempty"`
	Instruction      string  `json:"instruction"`
	Symbol           string  `json:"symbol,omitempty"`
	Location         *Source `json:"location,omitempty"`
	Line             int     `json:"line,omitempty"`
	Column           int     `json:"column,omitempty"`
	EndLine          int     `json:"endLine,omitempty"`
	EndColumn        int     `json:"endColumn,omitempty"`
	PresentationHint string  `json:"presentationHint,omitempty"`
}

// InvalidatedAreas: Logical areas that can be invalidated by the `invalidated` event.
type InvalidatedAreas string

// BreakpointMode: A `BreakpointMode` is provided as a option when setting breakpoints on sources or instructions.
type BreakpointMode struct {
	Mode        string                        `json:"mode"`
	Label       string                        `json:"label"`
	Description string                        `json:"description,omitempty"`
	AppliesTo   []BreakpointModeApplicability `json:"appliesTo"`
}

// BreakpointModeApplicability: Describes one or more type of breakpoint a `BreakpointMode` applies to. This is a non-exhaustive enumeration and may expand as future breakpoint types are added.
type BreakpointModeApplicability string

// Mapping of request commands and corresponding struct constructors that
// can be passed to json.Unmarshal.
var requestCtor = map[string]messageCtor{
	"cancel":         func() Message { return &CancelRequest{} },
	"runInTerminal":  func() Message { return &RunInTerminalRequest{} },
	"startDebugging": func() Message { return &StartDebuggingRequest{} },
	"initialize": func() Message {
		return &InitializeRequest{
			Arguments: InitializeRequestArguments{
				// Set the default values specified here: https://microsoft.github.io/debug-adapter-protocol/specification#Requests_Initialize.
				LinesStartAt1:   true,
				ColumnsStartAt1: true,
				PathFormat:      "path",
			},
		}
	},
	"configurationDone":         func() Message { return &ConfigurationDoneRequest{} },
	"launch":                    func() Message { return &LaunchRequest{} },
	"attach":                    func() Message { return &AttachRequest{} },
	"restart":                   func() Message { return &RestartRequest{} },
	"disconnect":                func() Message { return &DisconnectRequest{} },
	"terminate":                 func() Message { return &TerminateRequest{} },
	"breakpointLocations":       func() Message { return &BreakpointLocationsRequest{} },
	"setBreakpoints":            func() Message { return &SetBreakpointsRequest{} },
	"setFunctionBreakpoints":    func() Message { return &SetFunctionBreakpointsRequest{} },
	"setExceptionBreakpoints":   func() Message { return &SetExceptionBreakpointsRequest{} },
	"dataBreakpointInfo":        func() Message { return &DataBreakpointInfoRequest{} },
	"setDataBreakpoints":        func() Message { return &SetDataBreakpointsRequest{} },
	"setInstructionBreakpoints": func() Message { return &SetInstructionBreakpointsRequest{} },
	"continue":                  func() Message { return &ContinueRequest{} },
	"next":                      func() Message { return &NextRequest{} },
	"stepIn":                    func() Message { return &StepInRequest{} },
	"stepOut":                   func() Message { return &StepOutRequest{} },
	"stepBack":                  func() Message { return &StepBackRequest{} },
	"reverseContinue":           func() Message { return &ReverseContinueRequest{} },
	"restartFrame":              func() Message { return &RestartFrameRequest{} },
	"goto":                      func() Message { return &GotoRequest{} },
	"pause":                     func() Message { return &PauseRequest{} },
	"stackTrace":                func() Message { return &StackTraceRequest{} },
	"scopes":                    func() Message { return &ScopesRequest{} },
	"variables":                 func() Message { return &VariablesRequest{} },
	"setVariable":               func() Message { return &SetVariableRequest{} },
	"source":                    func() Message { return &SourceRequest{} },
	"threads":                   func() Message { return &ThreadsRequest{} },
	"terminateThreads":          func() Message { return &TerminateThreadsRequest{} },
	"modules":                   func() Message { return &ModulesRequest{} },
	"loadedSources":             func() Message { return &LoadedSourcesRequest{} },
	"evaluate":                  func() Message { return &EvaluateRequest{} },
	"setExpression":             func() Message { return &SetExpressionRequest{} },
	"stepInTargets":             func() Message { return &StepInTargetsRequest{} },
	"gotoTargets":               func() Message { return &GotoTargetsRequest{} },
	"completions":               func() Message { return &CompletionsRequest{} },
	"exceptionInfo":             func() Message { return &ExceptionInfoRequest{} },
	"readMemory":                func() Message { return &ReadMemoryRequest{} },
	"writeMemory":               func() Message { return &WriteMemoryRequest{} },
	"disassemble":               func() Message { return &DisassembleRequest{} },
	"locations":                 func() Message { return &LocationsRequest{} },
}

// Mapping of response commands and corresponding struct constructors that
// can be passed to json.Unmarshal.
var responseCtor = map[string]messageCtor{
	"cancel":                    func() Message { return &CancelResponse{} },
	"runInTerminal":             func() Message { return &RunInTerminalResponse{} },
	"startDebugging":            func() Message { return &StartDebuggingResponse{} },
	"initialize":                func() Message { return &InitializeResponse{} },
	"configurationDone":         func() Message { return &ConfigurationDoneResponse{} },
	"launch":                    func() Message { return &LaunchResponse{} },
	"attach":                    func() Message { return &AttachResponse{} },
	"restart":                   func() Message { return &RestartResponse{} },
	"disconnect":                func() Message { return &DisconnectResponse{} },
	"terminate":                 func() Message { return &TerminateResponse{} },
	"breakpointLocations":       func() Message { return &BreakpointLocationsResponse{} },
	"setBreakpoints":            func() Message { return &SetBreakpointsResponse{} },
	"setFunctionBreakpoints":    func() Message { return &SetFunctionBreakpointsResponse{} },
	"setExceptionBreakpoints":   func() Message { return &SetExceptionBreakpointsResponse{} },
	"dataBreakpointInfo":        func() Message { return &DataBreakpointInfoResponse{} },
	"setDataBreakpoints":        func() Message { return &SetDataBreakpointsResponse{} },
	"setInstructionBreakpoints": func() Message { return &SetInstructionBreakpointsResponse{} },
	"continue":                  func() Message { return &ContinueResponse{} },
	"next":                      func() Message { return &NextResponse{} },
	"stepIn":                    func() Message { return &StepInResponse{} },
	"stepOut":                   func() Message { return &StepOutResponse{} },
	"stepBack":                  func() Message { return &StepBackResponse{} },
	"reverseContinue":           func() Message { return &ReverseContinueResponse{} },
	"restartFrame":              func() Message { return &RestartFrameResponse{} },
	"goto":                      func() Message { return &GotoResponse{} },
	"pause":                     func() Message { return &PauseResponse{} },
	"stackTrace":                func() Message { return &StackTraceResponse{} },
	"scopes":                    func() Message { return &ScopesResponse{} },
	"variables":                 func() Message { return &VariablesResponse{} },
	"setVariable":               func() Message { return &SetVariableResponse{} },
	"source":                    func() Message { return &SourceResponse{} },
	"threads":                   func() Message { return &ThreadsResponse{} },
	"terminateThreads":          func() Message { return &TerminateThreadsResponse{} },
	"modules":                   func() Message { return &ModulesResponse{} },
	"loadedSources":             func() Message { return &LoadedSourcesResponse{} },
	"evaluate":                  func() Message { return &EvaluateResponse{} },
	"setExpression":             func() Message { return &SetExpressionResponse{} },
	"stepInTargets":             func() Message { return &StepInTargetsResponse{} },
	"gotoTargets":               func() Message { return &GotoTargetsResponse{} },
	"completions":               func() Message { return &CompletionsResponse{} },
	"exceptionInfo":             func() Message { return &ExceptionInfoResponse{} },
	"readMemory":                func() Message { return &ReadMemoryResponse{} },
	"writeMemory":               func() Message { return &WriteMemoryResponse{} },
	"disassemble":               func() Message { return &DisassembleResponse{} },
	"locations":                 func() Message { return &LocationsResponse{} },
}

// Mapping of event ids and corresponding struct constructors that
// can be passed to json.Unmarshal.
var eventCtor = map[string]messageCtor{
	"initialized":    func() Message { return &InitializedEvent{} },
	"stopped":        func() Message { return &StoppedEvent{} },
	"continued":      func() Message { return &ContinuedEvent{} },
	"exited":         func() Message { return &ExitedEvent{} },
	"terminated":     func() Message { return &TerminatedEvent{} },
	"thread":         func() Message { return &ThreadEvent{} },
	"output":         func() Message { return &OutputEvent{} },
	"breakpoint":     func() Message { return &BreakpointEvent{} },
	"module":         func() Message { return &ModuleEvent{} },
	"loadedSource":   func() Message { return &LoadedSourceEvent{} },
	"process":        func() Message { return &ProcessEvent{} },
	"capabilities":   func() Message { return &CapabilitiesEvent{} },
	"progressStart":  func() Message { return &ProgressStartEvent{} },
	"progressUpdate": func() Message { return &ProgressUpdateEvent{} },
	"progressEnd":    func() Message { return &ProgressEndEvent{} },
	"invalidated":    func() Message { return &InvalidatedEvent{} },
	"memory":         func() Message { return &MemoryEvent{} },
}
