/* This file is part of the Spring engine (GPL v2 or later), see LICENSE.html */

#include <SDL_mouse.h>

#include "StartPosSelecter.h"
#include "MouseHandler.h"
#include "Game/Game.h"
#include "Game/Camera.h"
#include "Game/GameSetup.h"
#include "Game/GlobalUnsynced.h"
#include "Game/InMapDraw.h"
#include "Game/Players/Player.h"
#include "Map/Ground.h"
#include "Map/ReadMap.h"
#include "Rendering/GL/myGL.h"
#include "Rendering/GL/glExtra.h"
#include "Rendering/Fonts/glFont.h"
#include "Net/Protocol/NetProtocol.h"
#include "Sim/Misc/TeamHandler.h"
#include "Sim/Misc/ModInfo.h"

#include "System/Misc/TracyDefs.h"


CStartPosSelecter* CStartPosSelecter::selector = nullptr;


CStartPosSelecter::CStartPosSelecter() : CInputReceiver(BACK)
{
	RECOIL_DETAILED_TRACY_ZONE;
	showReadyBox = true;
	startPosSet = false;

	readyBox.x1 = 0.71f;
	readyBox.y1 = 0.72f;
	readyBox.x2 = 0.81f;
	readyBox.y2 = 0.76f;

	selector = this;
}

CStartPosSelecter::~CStartPosSelecter()
{
	RECOIL_DETAILED_TRACY_ZONE;
	selector = nullptr;
}


bool CStartPosSelecter::Ready(bool luaForcedReady)
{
	RECOIL_DETAILED_TRACY_ZONE;
	if (!gs->PreSimFrame()) {
		delete this;
		return true;
	}

	const CTeam* mt = teamHandler.Team(gu->myTeam);
	const float3* sp = nullptr;

	// player did not set a startpos yet, so do not let
	// him ready up if he clicked on the ready-box first
	//
	// do allow in the special case where player already
	// sent a RDYSTATE_UPDATED and then rejoined the game
	if (!mt->HasValidStartPos()) {
		if (!startPosSet && !luaForcedReady)
			return false;

		sp = &setStartPos;
	} else {
		sp = &mt->GetStartPos();
	}

	if (luaForcedReady) {
		clientNet->Send(CBaseNetProtocol::Get().SendStartPos(gu->myPlayerNum, gu->myTeam, CPlayer::PLAYER_RDYSTATE_FORCED, sp->x, sp->y, sp->z));
	} else {
		clientNet->Send(CBaseNetProtocol::Get().SendStartPos(gu->myPlayerNum, gu->myTeam, CPlayer::PLAYER_RDYSTATE_READIED, sp->x, sp->y, sp->z));
	}

	delete this;
	return true;
}


bool CStartPosSelecter::MousePress(int x, int y, int button)
{
	RECOIL_DETAILED_TRACY_ZONE;

	if (button != SDL_BUTTON_LEFT)
		return false;

	if (modInfo.useStartPositionSelecter == false) {
		return false;
	}

	// Not ready to process mouse clicks yet. Pass.
	if (!game->IsDoneLoading())
		return false;

	const float mx = MouseX(x);
	const float my = MouseY(y);

	if ((showReadyBox && InBox(mx, my, readyBox)) || !gs->PreSimFrame())
		return (!Ready(false));

	const float dist = CGround::LineGroundCol(camera->GetPos(), camera->GetPos() + mouse->dir * camera->GetFarPlaneDist() * 1.4f, false);

	if (dist < 0.0f)
		return true;

	inMapDrawer->SendErase(setStartPos);
	startPosSet = true;
	setStartPos = camera->GetPos() + mouse->dir * dist;
	clientNet->Send(CBaseNetProtocol::Get().SendStartPos(gu->myPlayerNum, gu->myTeam, CPlayer::PLAYER_RDYSTATE_UPDATED, setStartPos.x, setStartPos.y, setStartPos.z));

	return true;
}

void CStartPosSelecter::Draw()
{
	RECOIL_DETAILED_TRACY_ZONE;
	if (gu->spectating) {
		delete this;
		return;
	}


	auto& rb = RenderBuffer::GetTypedRenderBuffer<VA_TYPE_C>();
	auto& shader = rb.GetShader();

	glEnable(GL_BLEND);

	// lua-fied!
	// DrawStartBox(buffer, shader);

	if (!showReadyBox)
		return;

	const float mx =                               float(mouse->lastx)  * globalRendering->pixelX;
	const float my = (globalRendering->viewSizeY - float(mouse->lasty)) * globalRendering->pixelY;

	{
		gleDrawQuadC(readyBox, InBox(mx, my, readyBox)? SColor{0.7f, 0.2f, 0.2f, guiAlpha}: SColor{0.7f, 0.7f, 0.2f, guiAlpha}, rb);

		glBlendFunc(GL_SRC_ALPHA, GL_ONE);
		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
	}

	{
		gleDrawQuadC(readyBox, InBox(mx, my, readyBox)? SColor{0.7f, 0.2f, 0.2f, guiAlpha}: SColor{0.7f, 0.7f, 0.2f, guiAlpha}, rb);

		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}
	{
		shader.Enable();
		rb.DrawElements(GL_TRIANGLES);
		shader.Disable();
	}

	{
		// fit text into box
		const float ySize = readyBox.y2 - readyBox.y1;
		const float xSize = readyBox.x2 - readyBox.x1;

		const float yPos = 0.5f * (readyBox.y1 + readyBox.y2);
		const float xPos = 0.5f * (readyBox.x1 + readyBox.x2);

		const float unitWidth  = font->GetSize() * font->GetTextWidth("Ready") * globalRendering->pixelX;
		const float unitHeight = font->GetSize() * font->GetLineHeight(      ) * globalRendering->pixelY;

		const float fontScale = 0.9f * std::min(xSize / unitWidth, ySize / unitHeight);

		font->SetColors(); // default
		font->glPrint(xPos, yPos, fontScale, FONT_OUTLINE | FONT_CENTER | FONT_VCENTER | FONT_SCALE | FONT_NORM | FONT_BUFFERED, "Ready");
		font->DrawBuffered();
	}
}

