+++
title = "Documenting Lua"
+++

This article covers working with Lua for **engine developers**. For adding the library to your Recoil game, refer to the [Lua Language Server guide](/docs/guides/getting-started/lua-language-server).

The Lua library is a collection of type definitions that can be included in Recoil games to provide intellisense and type checking. It is found in the [recoil-lua-library](https://github.com/beyond-all-reason/recoil-lua-library) repo.

The library is generated from doc comments within the project.

## Generating the library

### Automatic library generation

The library is generated by the {{< repo "tree/master/.github/workflows/generate-lua-library.yml" "Generate Lua library workflow" >}}. This workflow runs automatically every time a PR is merged.

### Manually generating the library

To test library generation, you can run [lua-doc-extractor](https://github.com/rhys-vdw/lua-doc-extractor) locally.

Install lua-doc-extractor:

```bash
npm install -g lua-doc-extractor
```

At root, run:

```bash
# First delete any previously generated output.
rm -rf rts/Lua/library/generated

# Regenerate the library.
lua-doc-extractor rts/Lua/*.cpp --dest rts/Lua/library/generated
```

## Doc comments

### Doc comments within CPP files

Special comments blocks are parsed by [lua-doc-extractor](https://github.com/rhys-vdw/lua-doc-extractor) and converted into [definition files](https://luals.github.io/wiki/definition-files/).

Comment blocks must start with `/***`, and by convention each line starts with `*`.

```cpp
/***
 * @function Spring.GetFeaturesInScreenRectangle
 *
 * Get features inside a rectangle area on the map
 *
 * @param left number
 * @param top number
 * @param right number
 * @param bottom number
 * @return nil|number[] featureIDs
 */
int LuaUnsyncedRead::GetFeaturesInScreenRectangle(lua_State* L)
{
 // CPP code
}
```

These comments are mostly copied directly into the Lua library, but some code generation is required. For these [custom tags](https://github.com/rhys-vdw/lua-doc-extractor?tab=readme-ov-file#custom-tags) must be used.

Markdown is supported in all text.

### Authored library files

Hand-authored library files exist in `/rts/Lua/Library/`. These are appropriate to use when a Lua type is used in multiple CPP files.

All files under `/rts/Lua/Library/` are directly copied into the library when the workflow runs.

## Annotations

[Full list of annotations](https://luals.github.io/wiki/annotations/).

### Common annotations

- [`@function`](https://github.com/rhys-vdw/lua-doc-extractor?tab=readme-ov-file#function-name)
- [`@table`](https://github.com/rhys-vdw/lua-doc-extractor?tab=readme-ov-file#table-name)
- [`@param`](https://luals.github.io/wiki/annotations/#param)
- [`@return`](https://luals.github.io/wiki/annotations/#return)
- [`@field`](https://luals.github.io/wiki/annotations/#field)

### Types

[List of types](https://luals.github.io/wiki/annotations/#documenting-types)

#### Common types

- `integer`
- `nil`
- `any`
- `boolean`
- `string`
- `number` (for floating point numbers)
- `integer`
- `table<,>`

{: .note }

> Literals (e.g. `true`, `false`, `5`) are also available as types. `true` is useful in the case where a table is being used as a set, e.g.
>
> ```
> table<string, true>
> ```

#### Unions

Union types can be specified with `|`.

```
string|boolean|number
```

Suffix with `?` as a shorthand for a union with `nil`, e.g. `string?` is equivalent to `string|nil`.

#### Arrays

An array type is expressed as `type[]`.

- `number[]` — Array of numbers.
- `string?[]` — Array of string or null.
- `number|string[]` — A single number, or an array of strings.
- `(number|string)[]` — An array with a mix of number and string.

## Examples

### Function

- Must start with `@function TableName.FunctionName`.
- Can have any amount of description. This should be added after the first
- Specify parameters with `@param parameterName type Description...`
- Specify return type with `@return type name Description...`
- For multiple returns use one per line.

{: .warning }

> `@return` must specify the type _before_ the name, whereas `@param` takes the name before the type.

````cpp
/***
 * @function Math.Add
 *
 * Add two integers together.
 *
 * This function will add two numbers together and return the result.
 *
 * ```lua
 * local totalHeight = Math.Add(legLength, upperBodyHeight)
 * ```
 *
 * @param a integer The first number.
 * @param b integer The second number.
 * @returns integer result The sum of `a` and `b`.
 */
````

### Class

Structured data is expressed as a class. This represents a table with expected key/value pairs.

- Must start with `@class ClassName`. This name becomes a type that can be used in annotations.
- Fields are specified by `@field fieldName type Description...`.

```cpp
/***
 * @class Color
 *
 * Describes an RGB color value.
 *
 * @field red number The red value.
 * @field green number The green value.
 * @field blue number The blue value.
 */

/***
 * @function ColorUtility.LerpColor
 * @param from Color
 * @param to Color
 * @param value number The mix (in range `[0,1]`) of colors to combine. `1` will return `to` and `0` will return `from`.
 * @return Color color
 */
```

### Table

A global table can be defined like so:

```cpp
/***
 * @table Spring
 */
```

This will define a table. This table can then be referenced in a function name e.g. `@function Spring.MyFunction`.

A table of constants can also be expressed using `@field`:

```cpp
/***
 * @table CoolNumbers
 * @field Pi number
 * @field SixtyNine integer
 * @field FourTwenty integer
 */
```

> [!WARNING]
> A table is a global that can be accessed in Lua and not a type like `@class`.
