/*
 * Copyright (c) 2007-2014 Hypertriton, Inc. <http://hypertriton.com/>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Generate headers for bundling fonts into executables or libraries.
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

static void
printusage(void)
{
	fprintf(stderr, "Usage: bundlefont [-abv] [-n name] [-o outfile] "
	                "[infile]\n");
}

int
main(int argc, char *argv[])
{
	const char *type = "AG_FONT_VECTOR";
	const char *outfile, *fontname = "myFont", *infile;
	extern char *optarg;
	extern int optind;
	int c, cnt;
	FILE *f, *fin;
	long i, size;
	int append = 0;

	while ((c = getopt(argc, argv, "?abvo:n:")) != -1) {
		switch (c) {
		case 'a':
			append = 1;
			break;
		case 'b':
			type = "AG_FONT_BITMAP";
			break;
		case 'v':
			type = "AG_FONT_VECTOR";
			break;
		case 'o':
			outfile = optarg;
			break;
		case 'n':
			fontname = optarg;
			break;
		case '?':
		case 'h':
		default:
			printusage();
			exit(1);
		}
	}
	if (optind == argc) {
		printusage();
		exit(1);
	}
	infile = argv[optind];
	if (outfile == NULL) {
		printusage();
		exit(1);
	}
	if ((fin = fopen(infile, "r")) == NULL) {
		fprintf(stderr, "%s: %s\n", infile, strerror(errno));
		exit(1);
	}
	if ((f = fopen(outfile, append ? "a" : "w")) == NULL) {
		fprintf(stderr, "%s: %s\n", outfile, strerror(errno));
		fclose(fin);
		exit(1);
	}
	
	fseek(fin, 0, SEEK_END);
	size = ftell(fin);
	fseek(fin, 0, SEEK_SET);

	fprintf(f, "/* File generated by bundlefont */\n");
	
	fprintf(f, "const Uint8 %s_Data[%ld] = {\n", fontname, size);
	for (i = 0, cnt = 0; i < size; i++) {
		unsigned char d;

		fread(&d, 1, 1, fin);
		fprintf(f, "%d,", d);
		if (++cnt > 10000) {
			fprintf(f, "\n");
			cnt = 0;
		}
	}
	fprintf(f, "};\n");
	fprintf(f, "AG_StaticFont %s = {\n", fontname);
	fprintf(f, "\"%s\",\n", fontname);
	fprintf(f, "%s,\n", type);
	fprintf(f, "%ld,\n", size);
	fprintf(f, "%s_Data,\n", fontname);
	fprintf(f, "};\n");

	fclose(f);
	fclose(fin);
	return (0);
}

