/* AbiWord
 * Copyright (C) 2004 Tomas Frydrych <tomasfrydrych@yahoo.co.uk>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  
 * 02110-1301 USA.
 */

#include "ut_uuid.h"
#include "ut_std_string.h"
#include "xap_Dlg_History.h"
#include "xad_Document.h"
#include "xap_App.h"
#include "xap_Strings.h"

#include <locale.h>

XAP_Dialog_History::XAP_Dialog_History(XAP_DialogFactory * pDlgFactory, XAP_Dialog_Id id):
	XAP_Dialog_NonPersistent(pDlgFactory,id),
	m_answer(a_CANCEL),
	m_pDoc(nullptr),
	m_pSS(nullptr),
	m_iId(0)
{
	m_pSS = XAP_App::getApp()->getStringSet();
}

const char * XAP_Dialog_History::getWindowLabel() const
{
	UT_return_val_if_fail(m_pSS, nullptr);
	return m_pSS->getValue(XAP_STRING_ID_DLG_History_WindowLabel);
}

const char *  XAP_Dialog_History::getListTitle() const
{
	UT_return_val_if_fail(m_pSS, nullptr);
	return m_pSS->getValue(XAP_STRING_ID_DLG_History_List_Title);
}


const char * XAP_Dialog_History::getHeaderLabel(UT_uint32 indx) const
{
	UT_return_val_if_fail(m_pSS, nullptr);
	switch(indx)
	{
		case 0: return m_pSS->getValue(XAP_STRING_ID_DLG_History_Path);
		case 1: return m_pSS->getValue(XAP_STRING_ID_DLG_History_Version);
		case 2: return m_pSS->getValue(XAP_STRING_ID_DLG_History_Created);
		case 3: return m_pSS->getValue(XAP_STRING_ID_DLG_History_LastSaved);
		case 4: return m_pSS->getValue(XAP_STRING_ID_DLG_History_EditTime);
		case 5: return m_pSS->getValue(XAP_STRING_ID_DLG_History_Id);
		
		default:;
	}

	UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
	return nullptr;
}


std::string XAP_Dialog_History::getHeaderValue(UT_uint32 indx) const
{
	UT_return_val_if_fail(m_pDoc, "");

	time_t tT;
	struct tm * tM;

	switch(indx)
	{
		case 0:
			return UT_ellipsisPath(m_pDoc->getFilename(), 45, 35);

		case 1:
			return UT_std_string_sprintf("%d",m_pDoc->getDocVersion());

		case 2:
			{
				const UT_UUID * pUUID = m_pDoc->getDocUUID();
				UT_return_val_if_fail(pUUID, "");

				tT = pUUID->getTime();
				tM = localtime(&tT);
				char* s = (char*)g_try_malloc(30);
				if(!s)
					return "";

				size_t len = strftime(s,30,"%c",tM);
				if(!len)
				{
					FREEP(s);
					return "";
				}

				std::string value(s);
				FREEP(s);

				return value;
			}

		case 3:
			{
				tT = m_pDoc->getLastSavedTime();
				tM = localtime(&tT);
				char* s = (char*)g_try_malloc(30);
				if(!s)
					return "";

				size_t len = strftime(s,30,"%c",tM);
				if(!len)
				{
					FREEP(s);
					return nullptr;
				}
				std::string value(s);
				FREEP(s);

				return value;
			}
		case 4:
			{
				time_t iEditTime = m_pDoc->getEditTime();
				time_t iHours = iEditTime / 3600;
				time_t iMinutes = (iEditTime % 3600)/60;
				time_t iSeconds = (iEditTime % 3600) % 60;

				return UT_std_string_sprintf("%.2d:%.2d:%.2d", static_cast<UT_sint32>(iHours), static_cast<UT_sint32>(iMinutes), static_cast<UT_sint32>(iSeconds));
			}

		case 5:
			return m_pDoc->getDocUUIDString();

		default:
			break;
	}

	UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
	return "";
}


const char * XAP_Dialog_History::getButtonLabel(UT_uint32 indx) const
{
	UT_return_val_if_fail(m_pSS, nullptr);

	switch(indx)
	{
		case 0: return m_pSS->getValue(XAP_STRING_ID_DLG_Restore);
			//case 1: return m_pSS->getValue(XAP_STRING_ID_DLG_Show);
		case 1: return m_pSS->getValue(XAP_STRING_ID_DLG_Cancel);

		default:;
	}

	UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
	return nullptr;
}


const char * XAP_Dialog_History::getListHeader(UT_uint32 column) const
{
	UT_return_val_if_fail(m_pSS, nullptr);

	switch(column)
	{
		case 0: return m_pSS->getValue(XAP_STRING_ID_DLG_History_Version_Version);
		case 1: return m_pSS->getValue(XAP_STRING_ID_DLG_History_Version_Started);
		case 2: return m_pSS->getValue(XAP_STRING_ID_DLG_History_Version_AutoRevisioned);

		default:;
	}

	UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
	return nullptr;
}

UT_uint32 XAP_Dialog_History::getListItemCount() const
{
	UT_return_val_if_fail(m_pDoc, 0);
	return m_pDoc->getHistoryCount();
}


std::string XAP_Dialog_History::getListValue(UT_uint32 item, UT_uint32 column) const
{
	UT_return_val_if_fail(m_pDoc, "");

	time_t tT;
	struct tm * tM;

	switch(column)
	{
		case 0:
			return UT_std_string_sprintf("%d",m_pDoc->getHistoryNthId(item));
		case 1:
			{
				tT = m_pDoc->getHistoryNthTimeStarted(item);
				tM = localtime(&tT);
				char* s = (char*)g_try_malloc(30);
				if(!s)
					return nullptr;

				size_t len = strftime(s,30,"%c",tM);
				if(!len)
				{
					FREEP(s);
					return "";
				}
				std::string value(s);
				FREEP(s);
				return value;
			}

		case 2:
			{
				UT_return_val_if_fail(m_pSS, "");

				std::string value;
				if(m_pDoc->getHistoryNthAutoRevisioned(item))
					m_pSS->getValueUTF8(XAP_STRING_ID_DLG_MB_Yes, value);
				else
					m_pSS->getValueUTF8(XAP_STRING_ID_DLG_MB_No, value);

				return value;
			}

		default:
			break;
	}

	UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
	return nullptr;
}


UT_uint32 XAP_Dialog_History::getListItemId(UT_uint32 item) const
{
	UT_return_val_if_fail(m_pDoc, 0);
	return m_pDoc->getHistoryNthId(item);
}

