/*
 *  gblc - (G)ude (B)ootloader(L)oader (C)lient
 *
 *  $Id$
 *
 *  Copyright (C) 2006, 2007,
 *    Martin Bachem, Gude Analog- und Digitalsysteme GmbH
 *    info@gudeads.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version
 *  2 of the License, or (at your option) any later version.
 *
 */

#include <pthread.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <ifaddrs.h>
#include <netdb.h>
#include <errno.h>
#include <string.h>
#ifdef __linux__
#include <asm/byteorder.h>
#include <endian.h>
#else
#include <machine/endian.h>
#endif
#include "config.h"
#include "helper.h"
#include "gbl_devs.h"
#include "gblv4.h"
#include "gblc.h"


static int socket_init = 0;
static int sock = -1;
static pthread_mutex_t mux = PTHREAD_MUTEX_INITIALIZER;
#define GBL_LOCK pthread_mutex_lock(&mux);
#define GBL_UNLOCK pthread_mutex_unlock(&mux);


const char * GBL_CMDS[] = {
    "",
    "GBL-Search", // GBL 1
    "GBL-Save NetConf", // GBL 2
    "GBL-Prepare Firmware Update", // GBL 3
    "GBL-Write Flash Page", // GBL 4
    "GBL-Restore FabSettings", // GBL 5
    "GBL6",
    "GBL7",
    "GBL8",
    "GBL-ping", // GBL 6
    "GBL10",
    "GBL11",
    "GBL12",
    "GBL13",
    "GBL-SetHostname", // GBL 14
    "GBL-Go-Firmware", // GBL 15
    "GBL-Go-BootLoader", // GBL 16
    "",
    "",
    "",
    "",
    "GBL-GetConfig", // GBL 21
    "GBL-SetConfig", // GBL 22
    "", // GBL 23
    "GBL-DelConfig", // GBL 24
    ""
};

const unsigned int GBL_CMD_SZ[] = {
    0,
    0, 13, 0, 0, 13, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0
};

const unsigned int GBL_TIMEOUTS[] = {
    0,
    1, 1, 2, 2, 15, 0, 0, 0,
    1, 0, 0, 0, 0, 1, 0, 0,
    0, 2, 0, 0, 5, 5, 0, 1
};

int gbl_sock_init(void) {
    struct ifreq ifr;
    GBL_LOCK;

    if (socket_init) {
        GBL_UNLOCK;
        return 0;
    }

    struct sockaddr_in s;
    s.sin_family = AF_INET;
    s.sin_port = htons(0);

    if (bind_addr.s_addr) {
        s.sin_addr.s_addr = bind_addr.s_addr;
    } else {
        s.sin_addr.s_addr = htonl(INADDR_ANY);
    }

    if (debug) {
        printl(INFO, "socket bound to addr 0x%x\n", s.sin_addr.s_addr);
    }

    sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (sock == -1) {
        printl(ERROR, "error creating socket: %s\n", strerror(errno));
        GBL_UNLOCK;
        return -1;
    }

    /* Bind to interface only */
    if (iface[0]) {
        memset(&ifr, 0, sizeof(ifr));
        snprintf(ifr.ifr_name, sizeof(ifr.ifr_name), iface);
        if (setsockopt(sock, SOL_SOCKET, SO_BINDTODEVICE, (void *)&ifr, sizeof(ifr)) < 0)
        {
            printl(ERROR, "Failed to bind to eth interface: %s\n", strerror(errno));
            close(sock);
            sock = -1;

            GBL_UNLOCK;
            return -1;
        }
    }

    if (bind(sock, (struct sockaddr *) &s, sizeof (s)) == -1) {
        printl(ERROR, "Failed to bind udp port: %s\n", strerror(errno));
        close(sock);
        sock = -1;

        GBL_UNLOCK;
        return -1;
    }

    if (debug) {
    }

    socket_init = 1;
    GBL_UNLOCK;
    return 0;
}

void gbl_sock_close(void) {
    GBL_LOCK;
    if (!socket_init)
        return;
    socket_init = 0;

    if (socket_init == -1) {
        GBL_UNLOCK;
        return;
    }

    close(sock);
    sock = -1;
    GBL_UNLOCK;
}

int
gbl_send_dgram(gbl_cmd_t * gbl_cmd, struct in_addr * addr) {
    struct sockaddr_in d;

    d.sin_family = AF_INET;
    d.sin_port = htons(GBL_PORT);
    d.sin_addr = *addr;

    if (inet_empty(addr))
        d.sin_addr.s_addr = htonl(INADDR_BROADCAST);

    int i = 0;
    if (d.sin_addr.s_addr == htonl(INADDR_BROADCAST))
        i = 1;

    if (debug) {
        printl(INFO, "TX %s len(%i):\n",
            inet_ntoa(d.sin_addr),
            gbl_cmd->len
            );
        hex_dump(1, gbl_cmd->data, gbl_cmd->len);
        printl(INFO, "\n");
    }

    if (setsockopt(sock, SOL_SOCKET, SO_BROADCAST, (const void*) &i, sizeof (i)) == -1) {
        printl(ERROR, "error setting SO_BROADCAST: %s\n", strerror(errno));
        return -1;
    }

    if (sendto(sock, gbl_cmd->data, gbl_cmd->len, 0, (struct sockaddr *) &d, sizeof (d)) == -1) {
        printl(ERROR, "error sending GBL command: %s\n", strerror(errno));
        return -1;
    }

    return 0;
}

/**
 * receives GBL udp dgram
 * @param gbl_cmd
 * @param cmd
 * @param to
 * @return
 */
int
gbl_recv(gbl_cmd_t *gbl_cmd, unsigned char cmd, const int to) {
    fd_set rfds;
    struct timeval tv;
    struct sockaddr_in s;
    int sl = sizeof (s);

    tv.tv_sec = to;
    tv.tv_usec = 0;

    FD_ZERO(&rfds);
    FD_SET(sock, &rfds);

    if (select(sock + 1, &rfds, NULL, NULL, &tv) != 1)
        return -1;

    memset(gbl_cmd->data, 0, GBL_MAX_PAYLOAD);
    gbl_cmd->len = recvfrom(sock, gbl_cmd->data, GBL_MAX_PAYLOAD, 0, (struct sockaddr *) &s, &sl);

    if (debug) {
        printl(INFO, "RX %s len(%i):\n",
            inet_ntoa(s.sin_addr),
            gbl_cmd->len
            );
        hex_dump(1, gbl_cmd->data, gbl_cmd->len);
        printl(INFO, "\n");
    }

    if (gbl_cmd->len < GBL_MIN_PAYLOAD) {
        printl(ERROR, "cmd len error, len is %d \n", gbl_cmd->len);
        return -2;
    }

    if (memcmp(gbl_cmd->data, GBL4_PREFIX, GBL4_PREFIX_LEN)) {
        printl(ERROR, "gbl prefix error\n");
        return -3;
    }

    if (xor_sum(gbl_cmd->data, gbl_cmd->len - 1) != (gbl_cmd->data[gbl_cmd->len - 1])) {
        printl(ERROR, "gbl checksum error\n");
        return -4;
    }

    if (gbl_cmd->data[GBL4_CMD_POS] != cmd) {
        printl(DEBUG, "unexpectd reply cmd(%d), len(%d)\n", gbl_cmd->data[GBL4_CMD_POS], gbl_cmd->len);
        return -1;
    }

    if (GBL_CMD_SZ[cmd]) {
        if (gbl_cmd->len != GBL_CMD_SZ[cmd]) {
            printl(DEBUG, "cmd len error, len is %d instead %d\n", gbl_cmd->len, GBL_CMD_SZ[cmd]);
            return (-6);
        }
    }

    return 0;
}

/**
 * fill gbl_data->data with prefix and cmd num
 * @param gbl_cmd
 * @return
 */
int
gbl_prepare_cmd(gbl_cmd_t *gbl_cmd) {

    if (gbl_cmd->cmd != 2) {
        memset(gbl_cmd->data + GBL4_RETVAL_POS, 0, GBL_MAX_PAYLOAD - GBL4_RETVAL_POS);
    }
    memcpy(gbl_cmd->data, GBL4_PREFIX, GBL4_PREFIX_LEN);
    gbl_cmd->data[GBL4_CMD_POS] = gbl_cmd->cmd;
    gbl_cmd->len = GBL4_PREFIX_LEN + 1;
}

/**
 * print mac addr in 00:19:32:00:00:00 notatation
 * @param mac
 */
void
dump_mac(unsigned char * mac) {
    int i;
    for (i = 0; i < 5; i++)
        printl(INFO, "%02X:", mac[i]);
    printl(INFO, "%02X", mac[5]);
}

unsigned long long
mtoi_mac(unsigned char * mac) {
    int i;
    unsigned long long m;
    m = 0;
    for (i = 0; i < 6; i++) {
        m += (unsigned long long) mac[i] << ((5 - i)*8);
    }
    return m;
}

/**
 * returns TRUE if device is running BootLoader
 * @param gbl_cmd
 * @return
 */
unsigned char
gbl_dev_bootloader(gbl_cmd_t *gbl_cmd) {
    return (*((unsigned char *) (gbl_cmd->data + 17)) & GBL_MODE_BOOTLOADER);
}

/**
 * returns index for gbldevs[] if device is listed
 * @param fw_id
 * @param hw_id
 * @return 
 */
static int
find_gbldev(unsigned int fw_id, unsigned char hw_id) {
    int i = 0;

    while (gbldevs[i].uc_id) {
        if ((hw_id != 0) && (hw_id != 0xFF)) {
            if ((gbldevs[i].hw_id == hw_id) && (gbldevs[i].fw_id == fw_id))
                return (i);
        } else {
            if (gbldevs[i].fw_id == fw_id)
                return (i);
        }
        i++;
    }
    return -1;
}

/**
 * show GBL 1 device on stdout
 * @param gbl_cmd
 * @param verbose
 */
void
dump_gbl_dev(gbl_cmd_t *gbl_cmd, unsigned char verbose) {
    int gbldev = -1;
    unsigned long long mac;
    ent_1_hw_ids *ent_1_hwid = NULL;
    char dev_name[64];
    char hostname[16];

    mac = mtoi_mac(gbl_cmd->data + NET_CONF_MAC_POS);
    if (is_mac_cache(mac)) {
        return;
    }

    strcpy(dev_name, "GBL Device");
    strcpy(hostname, "");

    switch (gbl_cmd->len) {
        case 131:
            ent_1_hwid = (ent_1_hw_ids *) (gbl_cmd->data + NET_CONF_HWID_POS);
            strncpy(dev_name, ent_1_hwid->prod_name, 64);
            strncpy(hostname, gbl_cmd->data + NET_CONF_HWID_POS + sizeof (ent_1_hw_ids), 16);
            gbldev = -2;
            break;

        case 37:
        default:
            gbldev = find_gbldev(GBL_DATA_UINT16(gbl_cmd->data, NET_CONF_HWMAGIC_POS),
                GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_HWID_POS));
            if (gbldev >= 0) {
                strncpy(dev_name, gbldevs[gbldev].name, 64);
            }
            break;
    }

    if (!verbose) {
        dump_mac(gbl_cmd->data + NET_CONF_MAC_POS);
        printl(INFO, " - %-32s", dev_name);

        printl(INFO, " - v%d.%d",
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_FMRMW_MAJOR_POS),
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_FMRMW_MINOR_POS)
            );

        printl(INFO, " (%d.%d%s)",
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_BLDR_MAJOR_POS),
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_BLDR_MINOR_POS),
            gbl_dev_bootloader(gbl_cmd) ? " (*ACTIVE*)" : ""
            );

        if (strnlen(hostname, 16)) {
            printf(", hostname %s", hostname);
        }

        printl(INFO, " - %d.%d.%d.%d ",
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_IP_POS + 0),
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_IP_POS + 1),
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_IP_POS + 2),
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_IP_POS + 3)
            );


        printl(INFO, "\n");

    } else {
        printl(INFO, "'%s'", dev_name);

        printl(INFO, " v%d.%d\n",
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_FMRMW_MAJOR_POS),
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_FMRMW_MINOR_POS));
        if (strnlen(hostname, 16)) {
            printf("\tHostname:   %s\n", hostname);
        }


        if (ent_1_hwid) {
            // Firmware Magic, HW-ID
            printl(INFO, "\tMagic IDs:  0x%04x, 0x%02x.0x%02x\n",
                GBL_DATA_UINT16(gbl_cmd->data, NET_CONF_HWMAGIC_POS),
                ent_1_hwid->hw_id_pcb,
                ent_1_hwid->hw_id_equip);
        } else {
            // Firmware Magic, HW-ID
            printl(INFO, "\tMagic IDs:  0x%04x, 0x%02x\n",
                GBL_DATA_UINT16(gbl_cmd->data, NET_CONF_HWMAGIC_POS),
                GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_HWID_POS));
        }

        // BootLoader Version, actual Mode
        printl(INFO, "\tBootLdr:    v%d.%d%s\n",
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_BLDR_MAJOR_POS),
            GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_BLDR_MINOR_POS),
            gbl_dev_bootloader(gbl_cmd) ? " (*ACTIVE*)" : "");

        // Mac Addr
        printl(INFO, "\tMAC-addr:   ");
        dump_mac(gbl_cmd->data + NET_CONF_MAC_POS);
        printl(INFO, "\n");

        printl(INFO, "\tPhy conf:   ");
        if (gbl_cmd->data[NET_CONF_PHYCONF_POS] == 0xFF) {
            printl(INFO, "Auto-Augotiation\n");
        } else {
            printl(INFO, "manual probe: ");
            if (gbl_cmd->data[NET_CONF_PHYCONF_POS] & 1)
                printl(INFO, "10 ");
            if (gbl_cmd->data[NET_CONF_PHYCONF_POS] & 2)
                printl(INFO, "100 ");
            if (gbl_cmd->data[NET_CONF_PHYCONF_POS] & 4)
                printl(INFO, "Mbps full-duplex\n");
            else
                printl(INFO, "Mbps half-duplex\n");
        }

        printl(INFO, "\tPhy state:  ");
        if (gbl_cmd->data[NET_CONF_PHYSTATE_POS] & 2)
            printl(INFO, "100");
        else
            printl(INFO, "10");

        if (gbl_cmd->data[NET_CONF_PHYSTATE_POS] & 4)
            printl(INFO, " Mbps full-duplex\n");
        else
            printl(INFO, " Mbps half-duplex\n");

        // IP Addr
        printl(INFO, "\tIP addr:    %s\n",
            inet_ntoa(GBL_DATA_INET(gbl_cmd->data, NET_CONF_IP_POS)));

        // Netmask
        printl(INFO, "\tNetmask:    %s\n",
            inet_ntoa(GBL_DATA_INET(gbl_cmd->data, NET_CONF_NM_POS)));

        // Gateway
        printl(INFO, "\tGateway:    %s\n",
            inet_ntoa(GBL_DATA_INET(gbl_cmd->data, NET_CONF_GW_POS)));

        // DHCP option
        printl(INFO, "\tDHCP:       %s\n",
            (GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_GLOPT_POS) & NET_CONF_GLOPT_DHCP ? "enabled" : "disabled"));

        // HTTP authentification option
        printl(INFO, "\tHTTP port:  %d\n",
            GBL_DATA_UINT16(gbl_cmd->data, NET_CONF_HTTP_PORT_POS));

        // HTTP authentification option
        printl(INFO, "\tHTTP auth:  %s\n",
            (GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_GLOPT_POS) & NET_CONF_GLOPT_AUTH ? "enabled" : "disabled"));

        // IP ACL option
        printl(INFO, "\tIP ACL:     %s\n",
            (GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_GLOPT_POS) & NET_CONF_GLOPT_ACL ? "enabled" : "disabled"));

        // reply pings
        printl(INFO, "\treply ping: %s\n",
            (GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_GLOPT_POS) & NET_CONF_GLOPT_NOPING ? "disabled" : "enabled"));

        // require HTTP auth for index.html Startpage
        printl(INFO, "\tindex auth: %s\n",
            (GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_GLOPT_POS) & NET_CONF_GLOPT_INDEXAUTH ? "enabled" : "disabled"));

        // Go Bootloader enabled (ATTENTION: this enabled BACKDOOR)
        printl(INFO, "\tGo BLDR:    %s\n",
            (GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_GLOPT_POS) & NET_CONF_GLOPT_GBLDR ? "enabled" : "disabled"));

        // GBL related Uart debug
        printl(INFO, "\tGBL debug:  %s\n",
            (GBL_DATA_UINT8(gbl_cmd->data, NET_CONF_GLOPT_POS) & NET_CONF_GLOPT_DEBUG ? "enabled" : "disabled"));

        printl(INFO, "\n");
    }
}

void
dump_eprom_entity_error_code(u_int8_t error_code) {
    switch (error_code) {
        case ((u_int8_t) - 1): printf("Entity not found");
            break;
        case ((u_int8_t) - 2): printf("buffer overrrun");
            break;
        case ((u_int8_t) - 3): printf("internal CRC error");
            break;
        case ((u_int8_t) - 4): printf("max entity count exceeded");
            break;
        case ((u_int8_t) - 5): printf("legacy eprom with no entities");
            break;
        case ((u_int8_t) - 6): printf("eeprom memory exceeded");
            break;
        case ((u_int8_t) - 7): printf("error writing to eeprom");
            break;
        case ((u_int8_t) - 8): printf("error reading from eeprom");
            break;
        case ((u_int8_t) - 9): printf("permission denied");
            break;
        case ((u_int8_t) - 10): printf("illegal entity ID");
            break;
        case ((u_int8_t) - 11): printf("illegal entity size");
            break;
    }
}

void
dump_eprom_entity_flags(u_int32_t flags) {
    if (flags & 0x80000000) {
        printf("STICKY ");
    }
    if (flags & 0x40000000) {
        printf("PROTECTED ");
    }
    if (flags & 0x80000000) {
        printf("SINGLE_SHOT ");
    }
}

int
next_eprom_entity(gbl_cmd_t *gbl_cmd, u_int32_t *next_id, u_int8_t *next_sub_idx) {
    if (memcmp(gbl_cmd->data + GBL4_RETVAL_POS + 0, gbl_cmd->data + GBL4_RETVAL_POS + 5, 5) == 0) {
        return 0;
    }
    *next_id = GBL_DATA_UINT32(gbl_cmd->data, GBL4_RETVAL_POS + 5);
    *next_sub_idx = GBL_DATA_UINT8(gbl_cmd->data, GBL4_RETVAL_POS + 9);
    return 1;
}

void
dump_eprom_entity(gbl_cmd_t *gbl_cmd) {
    u_int32_t id;
    u_int8_t sub_idx;
    u_int32_t next_id;
    u_int8_t next_sub_idx;
    u_int32_t flags;
    u_int16_t addr;
    u_int8_t id_count;
    u_int16_t ent_count;
    u_int8_t err_code;
    u_int8_t num_chunks, chunk_idx;
    int i;

    id = GBL_DATA_UINT32(gbl_cmd->data, GBL4_RETVAL_POS + 0);
    sub_idx = GBL_DATA_UINT8(gbl_cmd->data, GBL4_RETVAL_POS + 4);
    next_id = GBL_DATA_UINT32(gbl_cmd->data, GBL4_RETVAL_POS + 5);
    next_sub_idx = GBL_DATA_UINT8(gbl_cmd->data, GBL4_RETVAL_POS + 9);
    id_count = GBL_DATA_UINT8(gbl_cmd->data, GBL4_RETVAL_POS + 10);
    ent_count = GBL_DATA_UINT16(gbl_cmd->data, GBL4_RETVAL_POS + 11);
    flags = GBL_DATA_UINT32(gbl_cmd->data, GBL4_RETVAL_POS + 13);
    addr = GBL_DATA_UINT16(gbl_cmd->data, GBL4_RETVAL_POS + 17);
    num_chunks = GBL_DATA_UINT8(gbl_cmd->data, GBL4_RETVAL_POS + 19);
    chunk_idx = GBL_DATA_UINT8(gbl_cmd->data, GBL4_RETVAL_POS + 20);
    err_code = GBL_DATA_UINT8(gbl_cmd->data, GBL4_RETVAL_POS + 21);

    if (!err_code) {
        if (debug) {
            printf("id(%d.%i) next(%d.%d) flags(0x%x) addr(0x%x) count(%i/%i) chunk(%i/%i) err_code(%i) data(0x",
                id, sub_idx, next_id, next_sub_idx, flags, addr, id_count, ent_count, chunk_idx, num_chunks, err_code);
        } else {
            printf("ent=%d.%i data=", id, sub_idx);
        }
        for (i = (GBL4_RETVAL_POS + 22); i < (gbl_cmd->len - 1); i++) {
            printf("%02x", gbl_cmd->data[i]);
        }
        if (debug) {
            printf(")\n");
        } else {
            printf("\n");
        }
        if (flags && verbose) {
            printf("Flags: ");
            dump_eprom_entity_flags(flags);
            printf("\n");
        }
    } else {
        printf("id(%d.%i), next(%d.%i), flags(0x%x) addr(0x%x) count(%i/%i) chunk(%i/%i) err_code(%i)\n\tERROR: ",
            id, sub_idx, next_id, next_sub_idx, flags, addr, id_count, ent_count, chunk_idx, num_chunks, err_code);
        dump_eprom_entity_error_code(err_code);
        printf("\n");
    }
}

void
gbl_add_xorsum(gbl_cmd_t * gbl_cmd) {
    gbl_cmd->data[gbl_cmd->len] = xor_sum(gbl_cmd->data, gbl_cmd->len);
    gbl_cmd->len++;
}

int
gbl_poll_reply(gbl_cmd_t * gbl_cmd, unsigned short timeout) {
    int now = time(NULL);
    int start = time(NULL);
    int errcode;
    unsigned char done = 0;

    do {
        now = time(NULL);
        errcode = gbl_recv(gbl_cmd, gbl_cmd->cmd, ((now - start) <= timeout) ? (timeout - now + start) : 0);
        if (!errcode) {
            done = 1;
        } else {
            if ((errcode) < -1)
                printl(ERROR, "gbl_recv error %d\n", errcode);
        }
    } while (((now = time(NULL)) < start + timeout) && !done);

    return (done);
}

/**
 * perform GBL1 (search devices)
 * @param targethost
 * @param gbl_cmd
 * @param timeout
 * @param casttype
 * @param show
 * @param bootloader_only
 * @param verbose
 * @return
 */
int
gbl_query_netconf(struct in_addr * targethost, gbl_cmd_t * gbl_cmd, unsigned short timeout,
    unsigned int casttype, unsigned char show, unsigned int bootloader_only,
    unsigned char verbose) {
    int dev_count = 0;
    int now = time(NULL);
    int start = time(NULL);
    unsigned char done = 0;
    int errcode;
    struct in_addr bcast;
    unsigned char show_dev;

    gbl_prepare_cmd(gbl_cmd);
    gbl_add_xorsum(gbl_cmd);

    if (casttype == GBL_UNICAST) {
        gbl_send_dgram(gbl_cmd, targethost);
    } else {
        memset(&bcast, 0, sizeof (bcast));
        gbl_send_dgram(gbl_cmd, &bcast);
    }

    show_dev = show;

    do {
        now = time(NULL);
        errcode = gbl_recv(gbl_cmd, GBL_SEARCH, ((now - start) <= timeout) ? (timeout - now + start) : 0);

        if (!errcode) {
            if (casttype == GBL_UNICAST) {
                show_dev = 0;
                if (!(inet_empty(targethost))) {
                    done = 1;
                    if (bootloader_only) {
                        if (gbl_dev_bootloader(gbl_cmd)) {
                            dev_count++;
                        }
                    } else {
                        dev_count++;
                    }
                    if (show)
                        show_dev = done;
                }
            }

            if (casttype == GBL_BROADCAST) {
                if (!(inet_empty(targethost))) {
                    show_dev = 0;
                    done = (!(memcmp(targethost, gbl_cmd->data + NET_CONF_IP_POS, 4)));
                    if (done) {
                        if (bootloader_only) {
                            if (gbl_dev_bootloader(gbl_cmd)) {
                                dev_count++;
                            }
                        } else {
                            dev_count++;
                        }
                        if (show)
                            show_dev = done;
                    }
                } else {
                    show_dev = show;
                    if (bootloader_only) {
                        if (gbl_dev_bootloader(gbl_cmd)) {
                            dev_count++;
                        }
                    } else {
                        dev_count++;
                    }
                }
            }

            if (show_dev)
                if ((bootloader_only && (gbl_dev_bootloader(gbl_cmd))) || (!bootloader_only))
                    dump_gbl_dev(gbl_cmd, verbose);
        } else {
            if ((errcode) < -1)
                printl(ERROR, "gbl_recv error %d\n", errcode);
        }
    } while (((now = time(NULL)) < start + timeout) && !done);

    return (dev_count);
}


/*****************************************************************************/
/* GBL Commands  */

/**
 * GBL command 1 : GBL search
 * @param gbl_cmd
 * @param targethost
 * @param bootloader_only
 * @param casttype
 * @param verbose
 * @param timeout
 * @return
 */
int
gbl_search(
    gbl_cmd_t * gbl_cmd,
    struct in_addr * targethost,
    unsigned int bootloader_only,
    unsigned char casttype,
    unsigned char verbose,
    unsigned int timeout
    ) {
    int dev_count;

    if (gbl_sock_init() >= 0) {
        if (verbose > 0) {
            printl(INFO, "searching GBL device%s...\n", (inet_empty(targethost)) ? "s" : "");
        }
        dev_count = gbl_query_netconf(targethost, gbl_cmd, timeout, casttype, 1, bootloader_only, verbose);
    }
    return (dev_count);
}

/**
 * GBL command 2 : GBL Save NetConf
 * @param gbl_cmd
 * @param newconf
 * @return
 */
int
gbl_save_netconf(gbl_cmd_t * gbl_cmd, netconf_t * newconf) {
    unsigned char netconf[16];

    // take netconf reply as default
    netconf[12] = gbl_cmd->data[NET_CONF_PHYCONF_POS];
    netconf[13] = gbl_cmd->data[NET_CONF_GLOPT_POS];
    memcpy(netconf + 14, gbl_cmd->data + NET_CONF_HTTP_PORT_POS, 2);

    // set desired config if set
    if ((newconf->useflags & USE_CONF_IP) && (!(inet_empty(&newconf->ip))))
        memcpy(netconf + 0, &newconf->ip.s_addr, 4);
    else
        memcpy(netconf + 0, gbl_cmd->data + NET_CONF_IP_POS, 4);

    if ((newconf->useflags & USE_CONF_NM) && (!(inet_empty(&newconf->nm))))
        memcpy(netconf + 4, &newconf->nm.s_addr, 4);
    else
        memcpy(netconf + 4, gbl_cmd->data + NET_CONF_NM_POS, 4);

    if (newconf->useflags & USE_CONF_GW)
        memcpy(netconf + 8, &newconf->gw.s_addr, 4);
    else
        memcpy(netconf + 8, gbl_cmd->data + NET_CONF_GW_POS, 4);

    if (newconf->useflags & USE_CONF_DHCP) {
        printl(DEBUG, "USE_CONF_DHCP\n");
        netconf[13] &= ~(USE_CONF_DHCP);
        if (newconf->dhcp)
            netconf[13] |= USE_CONF_DHCP;
    }

    if (newconf->useflags & USE_CONF_HTTPAUTH) {
        printl(DEBUG, "USE_CONF_HTTPAUTH\n");
        netconf[13] &= ~(USE_CONF_HTTPAUTH);
        if (newconf->httpauth)
            netconf[13] |= USE_CONF_HTTPAUTH;
    }

    if (newconf->useflags & USE_CONF_IPACL) {
        printl(DEBUG, "USE_CONF_IPACL\n");
        netconf[13] &= ~(USE_CONF_IPACL);
        if (newconf->ipacl)
            netconf[13] |= USE_CONF_IPACL;
    }

    if (newconf->useflags & USE_CONF_HTTPPORT) {
        printl(DEBUG, "USE_CONF_HTTPPORT\n");
        newconf->httpport = cpu_to_be16(newconf->httpport);
        memcpy(netconf + 14, &newconf->httpport, 2);
    }

    if (newconf->useflags & USE_CONF_PHY) {
        printl(DEBUG, "USE_CONF_PHY\n");
        netconf[12] = newconf->phy;
    }

    if (newconf->useflags & USE_CONF_NOPING) {
        printl(DEBUG, "USE_CONF_NOPING\n");
        netconf[13] &= ~(USE_CONF_NOPING);
        if (newconf->noping)
            netconf[13] |= USE_CONF_NOPING;
    }

    if (newconf->useflags & USE_CONF_INDEXAUTH) {
        printl(DEBUG, "USE_CONF_INDEXAUTH\n");
        netconf[13] &= ~(USE_CONF_INDEXAUTH);
        if (newconf->indexauth)
            netconf[13] |= USE_CONF_INDEXAUTH;
    }

    if (newconf->useflags & USE_CONF_GBLDEBUG) {
        printl(DEBUG, "USE_CONF_GBLDEBUG\n");
        netconf[13] &= ~(USE_CONF_GBLDEBUG);
        if (newconf->gbldebug)
            netconf[13] |= USE_CONF_GBLDEBUG;
    }

    if (newconf->useflags & USE_CONF_GBLDR) {
        printl(DEBUG, "USE_CONF_GBLDR\n");
        netconf[13] &= ~(USE_CONF_GBLDR);
        if (newconf->gbldr)
            netconf[13] |= USE_CONF_GBLDR;
    }

    memcpy(gbl_cmd->data + gbl_cmd->len, netconf, 16);
    return (16);
}

/**
 * GBL3 (prepate firmware update)
 * @param gbl_cmd
 * @param fwupdate
 * @return 
 */
int
gbl_prepare_fwupdate(gbl_cmd_t * gbl_cmd, fw_update_t *fwupdate) {
    char * cmd_buffer;
    int c = 0;
    u_int16_t max_page;
    u_int32_t signature = 0;
    int parse_sig_state = 0;
    int next_byte_pos = 0;
    int spi = 0;
    int spi_start_addr = 0;
    int addr = 0;

    fwupdate->bytesize = 0;
    fwupdate->bytesize_spi = 0;
    fwupdate->xor_sum = 0;
    fwupdate->xor_sum_spi = 0;

    while ((fwupdate->bytesize <= MAX_FW_BYTE_SZ) && (fwupdate->bytesize_spi <= MAX_FW_BYTE_SZ)) {
        c = fgetc(fwupdate->fp);

        if (!spi || (addr < spi_start_addr)) {
            signature = (signature << 8) + c;

            if (!spi) {
                if ((parse_sig_state == 0 && signature == 0x16081147)) {
                    printf("\tsig state %i: 0x%x : 0x%x\n", parse_sig_state, fwupdate->bytesize, signature);
                    parse_sig_state++;
                    next_byte_pos = fwupdate->bytesize + 4;
                }
                if (parse_sig_state == 1 && (fwupdate->bytesize == next_byte_pos)) {
                    printf("\tsig state %i: 0x%x : 0x%x\n", parse_sig_state, fwupdate->bytesize, signature);
                    if (signature) {
                        spi = 1;
                        spi_start_addr = signature;
                    }
                    parse_sig_state++;
                }
            }
            // internal flash
            fwupdate->binbuffer[fwupdate->bytesize] = c;
            fwupdate->xor_sum ^= c;
            fwupdate->bytesize++;
        } else {
            // spi flash
            fwupdate->binbuffer_spi[fwupdate->bytesize_spi] = c;
            fwupdate->xor_sum_spi ^= c;
            fwupdate->bytesize_spi++;
        }

        addr++;
        if (feof(fwupdate->fp))
            break;
    }

    // calc max flash page
    fwupdate->max_page = (fwupdate->bytesize / FLASH_PAGE_SZ);
    if (fwupdate->bytesize % FLASH_PAGE_SZ)
        fwupdate->max_page++;
    if (fwupdate->max_page * FLASH_PAGE_SZ > (MAX_FW_BYTE_SZ))
        fwupdate->max_page = 0; // error

    printf("%d bytes, %d pages, xor sum: 0x%x\n",
        fwupdate->bytesize,
        fwupdate->max_page,
        fwupdate->xor_sum);

    // hex_dump(1, fwupdate->binbuffer, fwupdate->bytesize + 32);
    // hex_dump(1, fwupdate->binbuffer_spi, fwupdate->bytesize_spi + 32);

    if (spi) {
        // calc max SPI flash page
        fwupdate->max_page_spi = (fwupdate->bytesize_spi / FLASH_SPI_PAGE_SZ);
        if (fwupdate->bytesize_spi % FLASH_SPI_PAGE_SZ)
            fwupdate->max_page_spi++;

        printf("%d bytes SPI, %d pages SPI, xor sum: 0x%x\n",
            fwupdate->bytesize_spi,
            fwupdate->max_page_spi,
            fwupdate->xor_sum_spi);
    }

    cmd_buffer = gbl_cmd->data + gbl_cmd->len;

    // prepare gbl cmd
    max_page = cpu_to_be16(fwupdate->max_page);
    memcpy(cmd_buffer, &max_page, 2);
    cmd_buffer += 2;

    // firmware xor checksum
    *cmd_buffer++ = fwupdate->xor_sum;

    // fwupdate->max_page = 0;

    return (3);
}

/**
 * GBL4 (write flash page)
 * @param gbl_cmd
 * @param fwupdate
 * @return
 */
int
gbl_write_flashpage(gbl_cmd_t * gbl_cmd, fw_update_t * fwupdate) {
    u_int16_t page;
    char * cmd_buffer;

    cmd_buffer = gbl_cmd->data + gbl_cmd->len;

    page = cpu_to_be16((u_int16_t) fwupdate->cur_page);
    memcpy(cmd_buffer, &page, 2);
    cmd_buffer += 2;

    memcpy(cmd_buffer, fwupdate->binbuffer + (fwupdate->cur_page * 512), 512);
    return (514);
}

int
gbl_write_flashpage_spi(gbl_cmd_t * gbl_cmd, fw_update_t * fwupdate) {
    u_int32_t page;
    char * cmd_buffer;
    char device = 1;

    cmd_buffer = gbl_cmd->data + gbl_cmd->len;

    // spi device
    memcpy(cmd_buffer, &device, 1);
    cmd_buffer += 1;

    page = cpu_to_be32((u_int32_t) fwupdate->cur_page);
    memcpy(cmd_buffer, &page, 4);
    cmd_buffer += 4;

    memcpy(cmd_buffer, fwupdate->binbuffer_spi + (fwupdate->cur_page * 256), 256);

    return (256 + 4 + 1);
}

/**
 * GBL command 9 : GBL ping
 * @param gbl_cmd
 * @param sz
 * @return
 */
int
gbl_ping(gbl_cmd_t * gbl_cmd, unsigned int sz) {
    int i;
    u_int16_t tmp_be16;

    tmp_be16 = cpu_to_be16(sz);
    memcpy(gbl_cmd->data + gbl_cmd->len, &tmp_be16, 2);
    gbl_cmd->len += 2;

    for (i = 0; i < sz; i++)
        gbl_cmd->data[gbl_cmd->len + i] = i;

    return (sz);
}

/**
 * GBL command 14 : GBL set Hostname
 * @param gbl_cmd
 * @param hostname
 * @return
 */
int gbl_set_hostname(gbl_cmd_t * gbl_cmd, char * hostname) {
    u_int16_t tmp_be16;

    tmp_be16 = cpu_to_be16(0x28);
    memcpy(gbl_cmd->data + gbl_cmd->len, &tmp_be16, 2);

    tmp_be16 = cpu_to_be16(0x10);
    memcpy(gbl_cmd->data + gbl_cmd->len + 2, &tmp_be16, 2);

    strncpy(gbl_cmd->data + gbl_cmd->len + 4, hostname, 15);

    return 4 + 16;
}

/**
 * GBL21 read eeprom config entity
 * @param gbl_cmd
 * @param entity_id
 * @param subidx
 * @return payload length
 */
int gbl_get_config_entity(
    gbl_cmd_t * gbl_cmd,
    u_int32_t entity_id,
    u_int8_t subidx) {
    u_int32_t tmp_be32;
    tmp_be32 = cpu_to_be32(entity_id);
    memcpy(gbl_cmd->data + gbl_cmd->len, &tmp_be32, 4);
    memcpy(gbl_cmd->data + gbl_cmd->len + 4, &subidx, 1);
    return 5;
}

/**
 * GBL22 write eeprom config entity
 * @param gbl_cmd
 * @param entity_id
 * @param subidx
 * @param cfg_ent_data_len
 * @param cfg_ent_data
 * @return payload length
 */
int
gbl_set_config_entity(
    gbl_cmd_t * gbl_cmd,
    u_int32_t entity_id,
    u_int8_t subidx,
    int cfg_ent_data_len,
    char * cfg_ent_data) {
    u_int32_t tmp_be32;
    u_int16_t tmp_be16;
    tmp_be32 = cpu_to_be32(entity_id);
    tmp_be16 = cpu_to_be16((u_int16_t) cfg_ent_data_len);

    memcpy(gbl_cmd->data + gbl_cmd->len, &tmp_be32, 4);
    memcpy(gbl_cmd->data + gbl_cmd->len + 4, &subidx, 1);
    memcpy(gbl_cmd->data + gbl_cmd->len + 5, &tmp_be16, 2);
    memcpy(gbl_cmd->data + gbl_cmd->len + 7, cfg_ent_data, cfg_ent_data_len);
    return 7 + cfg_ent_data_len;
}

/**
 * GBL24 delete eeprom config entity
 * @param gbl_cmd
 * @param entity_id
 * @param subidx
 * @param force
 * @return payload length
 */
int gbl_del_config_entity(
    gbl_cmd_t * gbl_cmd,
    u_int32_t entity_id,
    u_int8_t subidx,
    u_int8_t force) {
    u_int32_t tmp_be32;

    tmp_be32 = cpu_to_be32(entity_id);
    memcpy(gbl_cmd->data + gbl_cmd->len, &tmp_be32, 4);
    memcpy(gbl_cmd->data + gbl_cmd->len + 4, &subidx, 1);

    if (force) {
        int i;
        char magic[] = "xyzzyRandomDevil";

        for (i = 0; i < 16; i++) {
            magic[i] = magic[i] ^ 0xff;
        }
        memcpy(gbl_cmd->data + gbl_cmd->len + 5, magic, 16);
    }

    return 5 + (force ? 16 : 0);
}
