/*
 *  gblc - (G)ude (B)ootloader(L)oader (C)lient
 *
 *  Copyright (C) 2006, 2007,
 *    Martin Bachem, Gude Analog- und Digitalsysteme GmbH
 *    Marco Calignano, Gude Analog- und Digitalsysteme GmbH
 *    info@gudeads.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version
 *  2 of the License, or (at your option) any later version.
 *
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>
#include <stdarg.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef __linux__
#include <asm/byteorder.h>
#include <endian.h>
#else
#include <machine/endian.h>
#endif
#include "config.h"
#include "gblv4.h"
#include "gblc.h"
#include "helper.h"

unsigned int debug;
unsigned int verbose = 0;
unsigned int silent;
unsigned int timeout;
unsigned int bootloader_only;
unsigned int unicast; // use unicast for 'search HOST' and 'netconf' commands
unsigned int force;
unsigned int skipxf = 0;

// rountrip time
unsigned long long ticks;

// firmware update data
fw_update_t fwupdate;

struct in_addr bind_addr;

char iface[MAX_IFACE_PARAM_LEN];

static void
usage(void) {
    fprintf(stderr, "Usage: %s command [HOST] [OPTIONS]\n", PACKAGE);
    fprintf(stderr, "  commands\n");
    fprintf(stderr, "    search [HOST]     search GBL devices by broadcast\n");
    fprintf(stderr, "    ping HOST         send GPL-ping\n");
    fprintf(stderr, "    fabdefaults HOST  reset device to fab defaults\n");
    fprintf(stderr, "    gofirmware HOST   leave bootloader start firmware\n");
    fprintf(stderr, "    gobootldr HOST    leave firmware; start bootloader if devices allows\n");
    fprintf(stderr, "    getconf HOST      read config entitiy from eeprom\n");
    fprintf(stderr, "    setconf HOST      write config entitiy to eeprom (requires BootLoader mode)\n");
    fprintf(stderr, "    delconf HOST      erase config entitiy from eeprom (requires BootLoader mode)\n");
    fprintf(stderr, "    dumpconf HOST     get some nuts\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "    update HOST       load firmware update to flash\n");
    fprintf(stderr, "      --firmware=epc.bin\n");
    fprintf(stderr, "      --skipxf        skip external flash\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "    netconf HOST      change devices' network settings\n");
    fprintf(stderr, "      --[ip,nm,gw]=v.w.x.y\n");
    fprintf(stderr, "      --dhcp=[0,1]\n");
    fprintf(stderr, "      --httpport=80\n");
    fprintf(stderr, "      --httpauth=[0,1]\n");
    fprintf(stderr, "      --indexauth=[0,1]\n");
    fprintf(stderr, "      --ipacl=[0,1]\n");
    fprintf(stderr, "      --phy=[auto,10,100,10_100,10fd,100fd,10_100fd]\n");
    fprintf(stderr, "      --replyping=[0,1]\n");
    fprintf(stderr, "      --gbldebug=[0,1]\n");
    fprintf(stderr, "      --gbldr=[0,1]\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "    getconf/setconf/delconf HOST\n");
    fprintf(stderr, "      --id=[x]             entity ID\n");
    fprintf(stderr, "      --idx=[x]            entity sub index (0: first)\n");
    fprintf(stderr, "      --data=[]            entity playload data, hex ascii\n");
    fprintf(stderr, "      --datafile=filename  entity playload data, binary file\n");
    fprintf(stderr, "      --mac=[]             target device's mac, hex ascii\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "    hostname HOST     change devices' hostname\n");
    fprintf(stderr, "      --name=myhost\n");
    fprintf(stderr, "\n");
    /*
    fprintf(stderr, "    spidump HOST       dump spi flash\n");
    fprintf(stderr, "      --spiimage=image.bin\n");
    fprintf(stderr, "    spiflash HOST      write spi flash\n");
    fprintf(stderr, "      --spiimage=image.bin\n");
    fprintf(stderr, "\n");
     */
    fprintf(stderr, "  [OPTIONS]\n");
    fprintf(stderr, "    -d, --debug       enabled debug info\n");
    fprintf(stderr, "        --verbose=[x] verbose=0 reduces console output\n");
    fprintf(stderr, "    --bind=v.w.x.y    bind UDP socket to this addr\n");
    fprintf(stderr, "    --iface=eth0      bind UDP socket to this interace (needs su)\n");
    fprintf(stderr, "    -s, --silent      dont show progress infos\n");
    fprintf(stderr, "    -u, --unicast     send UNICAST dgrams when possible\n");
    fprintf(stderr, "    -b, --bootldr     just search device in bootloader mode\n");
    fprintf(stderr, "        --timeout=[x] adjust timeout for current command\n");
    fprintf(stderr, "        --version     show version info\n");
}

int
main(int argc, char *argv[]) {
    static struct option long_options[] = {
        {"debug", optional_argument, 0, 'd'},
        {"silent", no_argument, &silent, 1},
        {"verbose", optional_argument, 0, 's'},
        {"timeout", optional_argument, 0, 't'},
        {"bootldr", no_argument, &bootloader_only, 1},
        {"unicast", no_argument, &unicast, 1},
        {"version", no_argument, 0, 'v'},
        {"help", no_argument, 0, 'h'},
        {"bind", required_argument, 0, 'y'},
        {"iface", required_argument, 0, 'z'},

        // extra ops for command 'netconf'
        {"ip", required_argument, 0, 'I'},
        {"nm", required_argument, 0, 'N'},
        {"gw", required_argument, 0, 'G'},
        {"httpport", required_argument, 0, 'P'},
        {"httpauth", required_argument, 0, 'A'},
        {"ipacl", required_argument, 0, 'L'},
        {"phy", required_argument, 0, 'Y'},
        {"dhcp", required_argument, 0, 'D'},
        {"replyping", required_argument, 0, 'R'},
        {"indexauth", required_argument, 0, 'X'},
        {"gbldebug", required_argument, 0, 'E'},
        {"gbldr", required_argument, 0, 'B'},
        {"skipxf", no_argument, 0, 4716},

        // extra opts for command 'update'
        {"firmware", required_argument, 0, 'F'},

        // extra opts for command 'hostname'
        {"name", required_argument, 0, 'M'},

        // extra opts for getconf / setconf
        {"id", required_argument, 0, 4711},
        {"idx", required_argument, 0, 4712},
        {"data", required_argument, 0, 4713},
        {"datafile", required_argument, 0, 4714},
        {"mac", required_argument, 0, 4715},
        {"force", no_argument, 0, 'f'},

        {0, 0, 0, 0}
    };

    int c;
    struct in_addr targethost;
    int ret = 0;
    char * ip = 0;
    char * hostname;
    gbl_cmd_t gbl_cmd;
    int dev_count = 0;
    int gbl_chain = 0;
    unsigned char cmd = 0;

    FILE *ent_fp;
    netconf_t newnetconf;
    int i;

    // extra opts for getcfg / setcfg
    u_int32_t cfg_ent_id = 0;
    u_int8_t cfg_ent_idx = 0;
    unsigned int cfg_ent_data_len = 0;
    char * cfg_ent_data = NULL;
    char mac[6] = {0};
    unsigned int useTargetMac = 0;

    iface[0] = 0;

    bind_addr.s_addr = 0;

    memset(&mac_cache, 0, sizeof (mac_cache));
    memset(&newnetconf, 0, sizeof (newnetconf));

    for (;;) {
        int option_index = 0;

        c = getopt_long(argc, argv, "dsbuvh", long_options, &option_index);
        if (c == -1)
            break;

        switch (c) {
            // --debug=X, -d
            case 'd':
                debug = 1;
                if (optarg) {
                    debug = atoi(optarg);
                }
                break;

            // --verbose=x
            case 's':
                verbose = 1;
                if (optarg) {
                    verbose = atoi(optarg);
                }
                break;

            // --timeout=x
            case 't':
                timeout = 0;
                if (optarg) {
                    timeout = atoi(optarg);
                }
                break;

            // --bootldr, -b
            case 'b':
                bootloader_only = 1;
                break;

            // --unicast, -u
            case 'u':
                unicast = 1;
                break;

            // --bind
            case 'y':
                if (!inet_aton(optarg, &bind_addr)) {
                    printl(ERROR, "illegal inet addr '%s'!\n", optarg);
                    return -1;
                }
                break;

            // --iface
            case 'z':
                if (strnlen(optarg, MAX_IFACE_PARAM_LEN) < MAX_IFACE_PARAM_LEN) {
                    snprintf(iface, MAX_IFACE_PARAM_LEN, "%s", optarg);
                }
                break;

            // --version, -v
            case 'v':
                printf(
                    "%s %s, (c) 2007 Martin Bachem and Marco Calignano\n"
                    "\tGude Analog- und Digitalsysteme GmbH\n"
                    "\tinfo[a]gudeads.com\n\n"
                    "info[a]gudeads.com\n\n",
                    PACKAGE, VERSION
                    );
                return 0;

            // --help, -h -?
            case 'h':
            case '?':
                usage();
                return 1;

            // extra ops for command 'netconf'
            case 'I':
                if (!inet_aton(optarg, &newnetconf.ip)) {
                    printl(ERROR, "illegal inet addr '%s'!\n", optarg);
                    return -1;
                }
                newnetconf.useflags |= USE_CONF_IP;
                break;

            case 'N':
                if (!inet_aton(optarg, &newnetconf.nm)) {
                    printl(ERROR, "illegal inet addr '%s'!\n", optarg);
                    return -1;
                }
                newnetconf.useflags |= USE_CONF_NM;
                break;

            case 'G':
                if (!inet_aton(optarg, &newnetconf.gw)) {
                    printl(ERROR, "illegal inet addr '%s'!\n", optarg);
                    return -1;
                }
                newnetconf.useflags |= USE_CONF_GW;
                break;

            case 'P':
                newnetconf.useflags |= USE_CONF_HTTPPORT;
                newnetconf.httpport = atoi(optarg);
                break;

            case 'A':
                newnetconf.useflags |= USE_CONF_HTTPAUTH;
                newnetconf.httpauth = atoi(optarg);
                break;

            case 'L':
                newnetconf.useflags |= USE_CONF_IPACL;
                newnetconf.ipacl = atoi(optarg);
                break;

            case 'D':
                newnetconf.useflags |= USE_CONF_DHCP;
                newnetconf.dhcp = atoi(optarg);
                break;

            case 'R':
                newnetconf.useflags |= USE_CONF_NOPING;
                newnetconf.noping = !(atoi(optarg));
                break;

            case 'X':
                newnetconf.useflags |= USE_CONF_INDEXAUTH;
                newnetconf.indexauth = atoi(optarg);
                break;

            case 'E':
                newnetconf.useflags |= USE_CONF_GBLDEBUG;
                newnetconf.gbldebug = atoi(optarg);
                break;

            case 'B':
                newnetconf.useflags |= USE_CONF_GBLDR;
                newnetconf.gbldr = atoi(optarg);
                break;

            case 'Y':
                if (strcmp(optarg, "auto") == 0) {
                    newnetconf.phy = PHYCONF_AUTO;
                } else
                    if (strcmp(optarg, "10") == 0) {
                    newnetconf.phy |= PHYCONF_PROBE_10;
                    newnetconf.phy &= ~PHYCONF_PROBE_FD;
                } else
                    if (strcmp(optarg, "100") == 0) {
                    newnetconf.phy |= PHYCONF_PROBE_100;
                    newnetconf.phy &= ~PHYCONF_PROBE_FD;
                } else
                    if (strcmp(optarg, "10fd") == 0) {
                    newnetconf.phy |= PHYCONF_PROBE_10;
                    newnetconf.phy |= PHYCONF_PROBE_FD;
                } else
                    if (strcmp(optarg, "100fd") == 0) {
                    newnetconf.phy |= PHYCONF_PROBE_100;
                    newnetconf.phy |= PHYCONF_PROBE_FD;
                } else
                    if (strcmp(optarg, "10_100fd") == 0) {
                    newnetconf.phy |= (PHYCONF_PROBE_10 + PHYCONF_PROBE_100 + PHYCONF_PROBE_FD);
                } else
                    if (strcmp(optarg, "10_100") == 0) {
                    newnetconf.phy |= (PHYCONF_PROBE_10 + PHYCONF_PROBE_100);
                } else {
                    printl(ERROR, "illegal phy conf '%s'!\n", optarg);
                    return -1;
                }
                newnetconf.useflags |= USE_CONF_PHY;
                break;

            case 'F':
                fwupdate.fp = fopen(optarg, "rb");
                if (!(fwupdate.fp)) {
                    printl(ERROR, "cannot read file '%s'!\n\n", optarg);
                    return -1;
                }

            case 'M':
                hostname = optarg;
                break;

            // --id=[x]
            case 4711:
                cfg_ent_id = atoi(optarg);
                break;

            // --idx=[x]
            case 4712:
                cfg_ent_idx = atoi(optarg);
                break;

            // --data=[x]
            case 4713:
                cfg_ent_data_len = strlen(optarg);
                if ((cfg_ent_data_len % 2) == 1) {
                    cfg_ent_data_len = 0;
                } else {
                    cfg_ent_data_len /= 2;
                }

                if (cfg_ent_data_len) {
                    cfg_ent_data = malloc(cfg_ent_data_len);
                    unsigned int tmp_padded;

                    for (i = 0; i < cfg_ent_data_len; i++) {
                        sscanf(
                                (const char *)optarg + (i * 2),
                                "%02x",
                                &tmp_padded
                        );
                        *(cfg_ent_data + i) = (char)tmp_padded;
                    }
                }
                break;

            // --datafile=filename
            case 4714:
                ent_fp = fopen(optarg, "rb");
                if (!ent_fp) {
                    printl(ERROR, "cannot read file '%s'!\n\n", optarg);
                    return -1;
                }
                fseek(ent_fp, 0, SEEK_END); // seek to end of file
                cfg_ent_data_len = ftell(ent_fp); // get current file pointer
                fseek(ent_fp, 0, SEEK_SET);
                if (cfg_ent_data_len) {
                    cfg_ent_data = (char*) malloc(cfg_ent_data_len);
                    fread(cfg_ent_data, cfg_ent_data_len, 1, ent_fp);
                }
                break;
                
            // --mac=[x]
            case 4715:
                if (strlen(optarg) == 12) {
                    unsigned int tmp_padded;
                    useTargetMac = 1;
                    for (i = 0; i < 6; i++) {
                        sscanf(
                                (const char *)optarg + (i * 2),
                                "%02x",
                                &tmp_padded
                        );
                        mac[i] = (char)tmp_padded;
                    }
                }
                break;

            case 4716:
                skipxf = 1;
                break;

            // [-f, --force]
            case 'f':
                force = 1;
                break;
        }
    }
    cmd = GBL_SEARCH;
    memset(&targethost, 0, sizeof (targethost));

    // parse non-option ARGV-elements: [command] [HOST]
    while (optind < argc) {
        if (strcmp(argv[optind], "search") == 0) {
            cmd = GBL_SEARCH;
        } else if (strcmp(argv[optind], "netconf") == 0) {
            cmd = GBL_NETCONF;
        } else if (strcmp(argv[optind], "ping") == 0) {
            cmd = GBL_PING;
        } else if (strcmp(argv[optind], "fabdefaults") == 0) {
            cmd = GBL_FABSETTINGS;
        } else if (strcmp(argv[optind], "update") == 0) {
            cmd = GBL_FWUPDATE;
            if (!fwupdate.fp)
                cmd = 0; // force usage()
        } else if (strcmp(argv[optind], "hostname") == 0) {
            cmd = GBL_SETHOSTNAME;
        } else if (strcmp(argv[optind], "gofirmware") == 0) {
            cmd = GBL_GOFIRMWARE;
        } else if (strcmp(argv[optind], "gobootldr") == 0) {
            cmd = GBL_GOBOOTLDR;
        } else if (strcmp(argv[optind], "getconf") == 0) {
            if (cfg_ent_id) {
                cmd = GBL_GETCFGENT;
            } else {
                cmd = 0;
            }
        } else if (strcmp(argv[optind], "setconf") == 0) {
            if (cfg_ent_id && cfg_ent_data_len) {
                cmd = GBL_SETCFGENT;
            } else {
                cmd = 0;
            }
        } else if (strcmp(argv[optind], "delconf") == 0) {
            if (cfg_ent_id) {
                cmd = GBL_DELCFGENT;
            } else {
                cmd = 0;
            }
        } else if (strcmp(argv[optind], "dumpconf") == 0) {
            cmd = GBL_GETCFGENT;
            gbl_chain = 1;
            cfg_ent_id = 1;
            cfg_ent_idx = 0;
        } else if (inet_addr_valid(argv[optind])) {
            ip = argv[optind];
            if (!inet_aton(argv[optind], &targethost)) {
                printl(ERROR, "illegal inet addr '%s'!\n", argv[optind]);
                return -1;
            }
        } else {
            cmd = 0; // force usage()
        }
        optind++;
    }

    // force targethost to be set as parameter on certain commands
    if (!ip) {
        switch (cmd) {
            case GBL_SETCFGENT:
            case GBL_GETCFGENT:
                cmd = 0;
                break;
        }
    }

    // print version info
    if (debug) {
        printl(INFO, "%s %s, cmd %d, verbose(%d) timeout(%d) bind(0x%x)\n",
            PACKAGE, VERSION, cmd, verbose, timeout,
            bind_addr.s_addr
            );
        fflush(stdout);
    }


    if (cmd) {
        if (gbl_sock_init() >= 0) {
            gbl_cmd.cmd = cmd;
            if (cmd == GBL_SEARCH) {
                if (!timeout) {
                    timeout = GBL_TIMEOUTS[1];
                }
                dev_count = gbl_search(&gbl_cmd, &targethost, bootloader_only,
                    (unicast) ? GBL_UNICAST : GBL_BROADCAST, verbose, timeout);
                if (verbose > 0) {
                    printl(INFO, "%d devices found\n", dev_count);
                }
            } else {
                if (useTargetMac) {
                    // bypass gbl_query_netconf
                    dev_count = 1;
                    memcpy(gbl_cmd.data + 5, mac, 6);
                } else {
                    gbl_cmd.cmd = GBL_SEARCH;
                    if (unicast)
                        dev_count = gbl_query_netconf(&targethost, &gbl_cmd, 3, GBL_UNICAST, 0, 0, 0);
                    else
                        dev_count = gbl_query_netconf(&targethost, &gbl_cmd, 3, GBL_BROADCAST, 0, 0, 0);
                    gbl_cmd.cmd = cmd;
                }

                if (dev_count == 1) {
                    do {
                        // create GBL datageramm
                        gbl_prepare_cmd(&gbl_cmd);
                        gbl_cmd.len += 6; // use MAC addr from gbl_query_netconf

                        switch (gbl_cmd.cmd) {
                            case GBL_PING:
                                printl(INFO, "sending %s\n", GBL_CMDS[gbl_cmd.cmd]);
                                gbl_cmd.len += gbl_ping(&gbl_cmd, 500);
                                break;

                            case GBL_NETCONF:
                                if (!unicast) {
                                    memset(&targethost, 0, sizeof (targethost));
                                }
                                printl(INFO, "saving net conf, wait approx. %d seconds...  ");
                                fflush(stdout);
                                gbl_cmd.len += gbl_save_netconf(&gbl_cmd, &newnetconf);
                                break;

                            case GBL_FABSETTINGS:
                                if (!unicast) {
                                    memset(&targethost, 0, sizeof (targethost));
                                }
                                printl(INFO, "sending %s, please wait approx. 6 seconds...   ", GBL_CMDS[gbl_cmd.cmd]);
                                fflush(stdout);
                                break;

                            case GBL_FWUPDATE:
                                gbl_cmd.len += gbl_prepare_fwupdate(&gbl_cmd, &fwupdate);
                                if (!fwupdate.max_page) {
                                    printf("Error reading binary file!\n");
                                }
                                printl(INFO, "sending %s, please wait approx. 1 second...   ", GBL_CMDS[gbl_cmd.cmd]);
                                break;

                            case GBL_FLASHWRITE:
                                printl(INFO, " %d/%d\r",
                                    fwupdate.cur_page + 1,
                                    fwupdate.max_page);
                                if (fwupdate.cur_page + 1 == fwupdate.max_page)
                                    printl(INFO, "\n");
                                gbl_cmd.len += gbl_write_flashpage(&gbl_cmd, &fwupdate);
                                break;

                            case GBL_FLASHWRITE_SPI:
                                printl(INFO, " %d/%d SPI\r",
                                    fwupdate.cur_page + 1,
                                    fwupdate.max_page_spi);
                                if (fwupdate.cur_page + 1 == fwupdate.max_page_spi)
                                    printl(INFO, "\n");
                                gbl_cmd.len += gbl_write_flashpage_spi(&gbl_cmd, &fwupdate);
                                break;

                            case GBL_GOBOOTLDR:
                            case GBL_GOFIRMWARE:
                                printl(INFO, "sending %s, please wait approx. 1 second...   ", GBL_CMDS[gbl_cmd.cmd]);
                                break;

                            case GBL_SETHOSTNAME:
                                gbl_cmd.len += gbl_set_hostname(&gbl_cmd, hostname);
                                printl(INFO, "sending %s, please wait approx. 1 second...   ", GBL_CMDS[gbl_cmd.cmd]);
                                break;

                            case GBL_GETCFGENT:
                                if (debug) {
                                    printf("reading config entity id(%i) idx(%i)\n", cfg_ent_id, cfg_ent_idx);
                                }
                                gbl_cmd.len += gbl_get_config_entity(&gbl_cmd, cfg_ent_id, cfg_ent_idx);
                                break;

                            case GBL_SETCFGENT:
                                printf("writing config entity id(%i) idx(%i): ", cfg_ent_id, cfg_ent_idx);
                                if (debug) {
                                    printf("\n");
                                    hex_dump(1, cfg_ent_data, cfg_ent_data_len);
                                }
                                gbl_cmd.len += gbl_set_config_entity(&gbl_cmd, cfg_ent_id, cfg_ent_idx, cfg_ent_data_len, cfg_ent_data);
                                break;

                            case GBL_DELCFGENT:
                                if (debug) {
                                    printf("delete config entity id(%i) idx(%i)\n", cfg_ent_id, cfg_ent_idx);
                                }
                                gbl_cmd.len += gbl_del_config_entity(&gbl_cmd, cfg_ent_id, cfg_ent_idx, force);
                                break;
                        }

                        // send data
                        gbl_add_xorsum(&gbl_cmd);
                        ticks = GetTickCount();
                        gbl_send_dgram(&gbl_cmd, &targethost);

                        // poll reply
                        ret = gbl_poll_reply(&gbl_cmd, GBL_TIMEOUTS[gbl_cmd.cmd]);

                        if (ret) {
                            if (verbose) {
                                printl(INFO, "replied in %3.6f seconds\n",
                                    (((float) (GetTickCount() - ticks)) / TICKS_PER_SEC));
                            }

                            switch (gbl_cmd.cmd) {
                                case GBL_NETCONF:
                                case GBL_FABSETTINGS:
                                    ret = (gbl_cmd.data[GBL4_RETVAL_POS] == GBL_OK);
                                    printl(INFO, "command status: %s\n", (ret) ? "SUCCESFUL" : "FAILED");
                                    break;

                                case GBL_FWUPDATE:
                                    // start firmware update 'write flash page' chain
                                    ret = (gbl_cmd.data[GBL4_RETVAL_POS] == GBL_OK);
                                    printl(INFO, "command status: %s %i\n", (ret) ? "SUCCESFUL" : "FAILED", ret);

                                    if ((ret) && (fwupdate.max_page)) {
                                        gbl_cmd.cmd = GBL_FLASHWRITE;
                                        gbl_chain = 1;
                                        fwupdate.cur_page = 0;
                                        fwupdate.retry_cnt = 0;
                                    }
                                    break;

                                case GBL_FLASHWRITE:
                                    fwupdate.cur_page++;
                                    fwupdate.retry_cnt = 0;
                                    ret = (gbl_cmd.data[GBL4_RETVAL_POS + 2] == GBL_OK);
                                    if ((!ret) || (fwupdate.cur_page >= fwupdate.max_page)) {
                                        if (fwupdate.max_page_spi && !skipxf) {
                                            gbl_cmd.cmd = GBL_FLASHWRITE_SPI;
                                            gbl_chain = 1;
                                            fwupdate.cur_page = 0;
                                        } else {
                                            printf("Firmware Update complete\n");
                                            gbl_chain = 0;
                                        }
                                    }
                                    break;

                                case GBL_FLASHWRITE_SPI:
                                    fwupdate.cur_page++;
                                    fwupdate.retry_cnt = 0;
                                    ret = (gbl_cmd.data[GBL4_RETVAL_POS + 2] == GBL_OK);
                                    if ((!ret) || (fwupdate.cur_page >= fwupdate.max_page_spi)) {
                                        printf("Firmware Update complete\n");
                                        gbl_chain = 0;
                                    }
                                    break;

                                case GBL_GETCFGENT:
                                    dump_eprom_entity(&gbl_cmd);
                                    if (gbl_chain) {
                                        // get next?
                                        gbl_chain = next_eprom_entity(&gbl_cmd, &cfg_ent_id, &cfg_ent_idx);
                                    }
                                    break;

                                case GBL_SETCFGENT:
                                    ret = gbl_cmd.data[GBL4_RETVAL_POS + 10];
                                    if (debug) {
                                        printf("errcode(0x%02x)\n", gbl_cmd.data[GBL4_RETVAL_POS + 10]);
                                    }
                                    if (ret == GBL_OK) {
                                        printf("OK\n");
                                    } else {
                                        printf("ERROR: ");
                                        dump_eprom_entity_error_code(gbl_cmd.data[GBL4_RETVAL_POS + 10]);
                                        printf("\n");
                                    }
                                    break;

                                case GBL_DELCFGENT:
                                    ret = gbl_cmd.data[GBL4_RETVAL_POS + 6];
                                    if (debug) {
                                        printf("errcode(0x%02x)\n", gbl_cmd.data[GBL4_RETVAL_POS + 6]);
                                    }
                                    if (ret == GBL_OK) {
                                        printf("OK\n");
                                    } else {
                                        printf("ERROR: ");
                                        dump_eprom_entity_error_code(gbl_cmd.data[GBL4_RETVAL_POS + 6]);
                                        printf("\n");
                                    }
                                    break;
                            }
                        } else {
                            if ((gbl_cmd.cmd != 15) && (gbl_cmd.cmd != 16)) {
                                printl(INFO, "%s timeout\n", GBL_CMDS[gbl_cmd.cmd]);
                                gbl_chain = 0;
                                if (gbl_cmd.cmd == GBL_FLASHWRITE) {
                                    fwupdate.retry_cnt++;
                                    if (fwupdate.retry_cnt >= FLASH_WRITE_MAX_RETRY) {
                                        gbl_chain = 0;
                                    } else {
                                        gbl_chain = 1;
                                        fwupdate.cur_page = (fwupdate.cur_page / 4) * 4;
                                    }
                                }
                            } else {
                                printf("\n");
                            }
                        }
                    } while (gbl_chain);
                } else {
                    printl(WARN, "gbl device at %s not found!\n", ip);
                }
            }
            gbl_sock_close();

            if (cfg_ent_data) {
                free(cfg_ent_data);
            }

            // console return codes
            switch (cmd) {
                case GBL_SEARCH:
                    return (!(dev_count > 0));
            }

            return (!ret);
        } else {
            printl(ERROR, "cannot init gbl socket!\n", ip);
            return (ret);
        }
    }

    usage();
    return 0;
}
