/*
 * Copyright (c) 2018-2021, 2025 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include "arm_compute/core/Types.h"
#include "arm_compute/runtime/CL/CLTensor.h"
#include "arm_compute/runtime/CL/CLTensorAllocator.h"
#include "arm_compute/runtime/CL/functions/CLSelect.h"

#include "tests/CL/CLAccessor.h"
#include "tests/datasets/ShapeDatasets.h"
#include "tests/framework/Asserts.h"
#include "tests/framework/datasets/Datasets.h"
#include "tests/framework/Macros.h"
#include "tests/PaddingCalculator.h"
#include "tests/validation/fixtures/SelectFixture.h"
#include "tests/validation/Validation.h"

namespace arm_compute
{
namespace test
{
namespace validation
{
using framework::dataset::make;
namespace
{
auto run_small_dataset = combine(datasets::SmallShapes(), make("has_same_rank", {false, true}));
auto run_large_dataset = combine(datasets::LargeShapes(), make("has_same_rank", {false, true}));

} // namespace
TEST_SUITE(CL)
TEST_SUITE(Select)

// *INDENT-OFF*
// clang-format off
DATA_TEST_CASE(Validate, framework::DatasetMode::ALL, zip(make("CInfo", { TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::S8), // Invalid condition datatype
                                            TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::U8), // Invalid output datatype
                                            TensorInfo(TensorShape(13U), 1, DataType::U8),          // Invalid c shape
                                            TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::U8), // Mismatching shapes
                                            TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::U8),
                                            TensorInfo(TensorShape(2U), 1, DataType::U8),
        }),
        make("XInfo",{ TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::U8),
                                           TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::U8),
                                           TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::U8),
                                           TensorInfo(TensorShape(32U, 10U, 2U), 1, DataType::F32),
                                           TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::F32),
                                           TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::F32),
        }),
        make("YInfo",{ TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::U8),
                                           TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::U8),
                                           TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::U8),
                                           TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::F32),
                                           TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::F32),
                                           TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::F32),
        }),
        make("OutputInfo",{ TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::U8),
                                                TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::S8),
                                                TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::U8),
                                                TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::F32),
                                                TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::F32),
                                                TensorInfo(TensorShape(32U, 13U, 2U), 1, DataType::F32),
        }),
        make("Expected", { false, false, false, false, true, true})),
        c_info, x_info, y_info, output_info, expected)
{
    Status s = CLSelect::validate(&c_info.clone()->set_is_resizable(false),
                                  &x_info.clone()->set_is_resizable(false),
                                  &y_info.clone()->set_is_resizable(false),
                                  &output_info.clone()->set_is_resizable(false));
    ARM_COMPUTE_EXPECT(bool(s) == expected, framework::LogLevel::ERRORS);
}
// clang-format on
// *INDENT-ON*

template <typename T>
using CLSelectFixture = SelectValidationFixture<CLTensor, CLAccessor, CLSelect, T>;

TEST_SUITE(Float)
TEST_SUITE(F16)
FIXTURE_DATA_TEST_CASE(RunSmall,
                       CLSelectFixture<half>,
                       framework::DatasetMode::PRECOMMIT,
                       combine(run_small_dataset, make("DataType", DataType::F16)))
{
    // Validate output
    validate(CLAccessor(_target), _reference);
}

FIXTURE_DATA_TEST_CASE(RunOneDim,
                       CLSelectFixture<half>,
                       framework::DatasetMode::PRECOMMIT,
                       combine(make("Shape", TensorShape(1U, 16U)),
                               make("has_same_rank", {false, true}),
                               make("DataType", DataType::F16)))
{
    // Validate output
    validate(CLAccessor(_target), _reference);
}

FIXTURE_DATA_TEST_CASE(RunLarge,
                       CLSelectFixture<half>,
                       framework::DatasetMode::NIGHTLY,
                       combine(run_large_dataset, make("DataType", DataType::F16)))
{
    // Validate output
    validate(CLAccessor(_target), _reference);
}
TEST_SUITE_END() // F16

TEST_SUITE(FP32)
FIXTURE_DATA_TEST_CASE(RunSmall,
                       CLSelectFixture<float>,
                       framework::DatasetMode::PRECOMMIT,
                       combine(run_small_dataset, make("DataType", DataType::F32)))
{
    // Validate output
    validate(CLAccessor(_target), _reference);
}

FIXTURE_DATA_TEST_CASE(RunOneDim,
                       CLSelectFixture<float>,
                       framework::DatasetMode::PRECOMMIT,
                       combine(make("Shape", TensorShape(1U, 16U)),
                               make("has_same_rank", {false, true}),
                               make("DataType", DataType::F32)))
{
    // Validate output
    validate(CLAccessor(_target), _reference);
}

FIXTURE_DATA_TEST_CASE(RunLarge,
                       CLSelectFixture<float>,
                       framework::DatasetMode::NIGHTLY,
                       combine(run_large_dataset, make("DataType", DataType::F32)))
{
    // Validate output
    validate(CLAccessor(_target), _reference);
}
TEST_SUITE_END() // F32
TEST_SUITE_END() // Float

TEST_SUITE(Quantized)
TEST_SUITE(QASYMM8)
FIXTURE_DATA_TEST_CASE(RunSmall,
                       CLSelectFixture<uint8_t>,
                       framework::DatasetMode::PRECOMMIT,
                       combine(run_small_dataset, make("DataType", DataType::QASYMM8)))
{
    // Validate output
    validate(CLAccessor(_target), _reference);
}

FIXTURE_DATA_TEST_CASE(RunOneDim,
                       CLSelectFixture<uint8_t>,
                       framework::DatasetMode::PRECOMMIT,
                       combine(make("Shape", TensorShape(1U, 16U)),
                               make("has_same_rank", {false, true}),
                               make("DataType", DataType::QASYMM8)))
{
    // Validate output
    validate(CLAccessor(_target), _reference);
}

FIXTURE_DATA_TEST_CASE(RunLarge,
                       CLSelectFixture<uint8_t>,
                       framework::DatasetMode::NIGHTLY,
                       combine(run_large_dataset, make("DataType", DataType::QASYMM8)))
{
    // Validate output
    validate(CLAccessor(_target), _reference);
}
TEST_SUITE_END() // QASYMM8
TEST_SUITE_END() // Quantized

TEST_SUITE_END() // Select
TEST_SUITE_END() // CL
} // namespace validation
} // namespace test
} // namespace arm_compute
