/*
 * Copyright (c) 2025-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#pragma once

#if (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(ARM_COMPUTE_ENABLE_SVE) && defined(__aarch64__)
template<>
void MergeResults<3, 8, true>(
    __fp16 *out_ptr,
    const __fp16 * in_ptr,
    const int ldout,
    const int y0, const int ymax,
    const int x0, const int xmax,
    const __fp16 *bias,
    Activation act,
    bool accumulate)
{
    __fp16 maxval = static_cast<__fp16>(std::numeric_limits<float>::infinity());
    __fp16 minval = - static_cast<__fp16>(std::numeric_limits<float>::infinity());

    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            maxval = static_cast<__fp16>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            minval = 0;
            break;
    }

    size_t rows = ymax-y0;
    size_t cols = xmax-x0;

    out_ptr += (y0 * ldout) + x0;
    bias = (bias == nullptr) ? nullptr : bias + x0;

    __asm__ __volatile__(
      "ptrue p3.b\n"
      "cbz %x[cols], 52f\n"
      "cbz %x[rows], 52f\n"
      "mov x12, #0x10\n"
      "dup z12.h, %w[maxval]\n"
      "dup z11.h, %w[minval]\n"
      "mul x12, %x[ldout], x12\n"
      "cbnz %x[accumulate], 34f\n"
      "1:"  // Initial: Row loop
      "cmp %x[rows], #0x7\n"
      "bgt 30f\n"
      "beq 26f\n"
      "cmp %x[rows], #0x5\n"
      "bgt 22f\n"
      "beq 18f\n"
      "cmp %x[rows], #0x3\n"
      "bgt 14f\n"
      "beq 10f\n"
      "cmp %x[rows], #0x1\n"
      "bgt 6f\n"
      "mov x11, %x[cols]\n"
      "mov x10, %x[out_ptr]\n"
      "mov x9, %x[bias]\n"
      "3:"  // Initial: Height 1: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "whilelt p1.h, x21, x11\n"
      "inch x21\n"
      "whilelt p0.h, x21, x11\n"
      "inch x21\n"
      "cbnz %x[bias], 4f\n"
      "mov z21.b, #0\n"
      "mov z20.b, #0\n"
      "mov z19.b, #0\n"
      "b 5f\n"
      "4:"  // Initial: Height 1: Width 3: bias
      "ld1h { z21.h }, p2/Z, [x9]\n"
      "ld1h { z20.h }, p1/Z, [x9, #1, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x9, #2, MUL VL]\n"
      "5:"  // Initial: Height 1: Width 3: init done
      "ld1h { z18.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z17.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "dech x11, ALL, MUL #3\n"
      "incb x9, ALL, MUL #3\n"
      "ld1h { z16.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "fadd z18.h, z18.h, z21.h\n"
      "fadd z17.h, z17.h, z20.h\n"
      "cmp x11, XZR\n"
      "fadd z16.h, z16.h, z19.h\n"
      "fmin z18.h, p3/M, z18.h, z12.h\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmin z16.h, p3/M, z16.h, z12.h\n"
      "fmax z18.h, p3/M, z18.h, z11.h\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "fmax z16.h, p3/M, z16.h, z11.h\n"
      "st1h { z18.h }, p2, [x10]\n"
      "st1h { z17.h }, p1, [x10, #1, MUL VL]\n"
      "st1h { z16.h }, p0, [x10, #2, MUL VL]\n"
      "incb x10, ALL, MUL #3\n"
      "bgt 3b\n"
      "b 52f\n"
      "6:"  // Initial: Height 2
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "mov x9, %x[bias]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "7:"  // Initial: Height 2: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "whilelt p1.h, x21, x11\n"
      "inch x21\n"
      "whilelt p0.h, x21, x11\n"
      "inch x21\n"
      "cbnz %x[bias], 8f\n"
      "mov z24.b, #0\n"
      "mov z23.b, #0\n"
      "mov z22.b, #0\n"
      "b 9f\n"
      "8:"  // Initial: Height 2: Width 3: bias
      "ld1h { z24.h }, p2/Z, [x9]\n"
      "ld1h { z23.h }, p1/Z, [x9, #1, MUL VL]\n"
      "ld1h { z22.h }, p0/Z, [x9, #2, MUL VL]\n"
      "9:"  // Initial: Height 2: Width 3: init done
      "ld1h { z21.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z20.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "dech x11, ALL, MUL #3\n"
      "incb x9, ALL, MUL #3\n"
      "ld1h { z19.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z17.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "ld1h { z16.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "fadd z21.h, z21.h, z24.h\n"
      "fadd z20.h, z20.h, z23.h\n"
      "cmp x11, XZR\n"
      "fadd z19.h, z19.h, z22.h\n"
      "fadd z18.h, z18.h, z24.h\n"
      "fadd z17.h, z17.h, z23.h\n"
      "fadd z16.h, z16.h, z22.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "fmin z18.h, p3/M, z18.h, z12.h\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmin z16.h, p3/M, z16.h, z12.h\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "fmax z18.h, p3/M, z18.h, z11.h\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "fmax z16.h, p3/M, z16.h, z11.h\n"
      "st1h { z21.h }, p2, [x10]\n"
      "st1h { z20.h }, p1, [x10, #1, MUL VL]\n"
      "st1h { z19.h }, p0, [x10, #2, MUL VL]\n"
      "incb x10, ALL, MUL #3\n"
      "st1h { z18.h }, p2, [x28]\n"
      "st1h { z17.h }, p1, [x28, #1, MUL VL]\n"
      "st1h { z16.h }, p0, [x28, #2, MUL VL]\n"
      "incb x28, ALL, MUL #3\n"
      "bgt 7b\n"
      "b 52f\n"
      "10:"  // Initial: Height 3
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "mov x9, %x[bias]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "11:"  // Initial: Height 3: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "whilelt p1.h, x21, x11\n"
      "inch x21\n"
      "whilelt p0.h, x21, x11\n"
      "inch x21\n"
      "cbnz %x[bias], 12f\n"
      "mov z27.b, #0\n"
      "mov z26.b, #0\n"
      "mov z25.b, #0\n"
      "b 13f\n"
      "12:"  // Initial: Height 3: Width 3: bias
      "ld1h { z27.h }, p2/Z, [x9]\n"
      "ld1h { z26.h }, p1/Z, [x9, #1, MUL VL]\n"
      "ld1h { z25.h }, p0/Z, [x9, #2, MUL VL]\n"
      "13:"  // Initial: Height 3: Width 3: init done
      "ld1h { z24.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z23.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "dech x11, ALL, MUL #3\n"
      "incb x9, ALL, MUL #3\n"
      "ld1h { z22.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z21.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z20.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z17.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "fadd z24.h, z24.h, z27.h\n"
      "fadd z23.h, z23.h, z26.h\n"
      "ld1h { z16.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "fadd z22.h, z22.h, z25.h\n"
      "fadd z21.h, z21.h, z27.h\n"
      "cmp x11, XZR\n"
      "fadd z20.h, z20.h, z26.h\n"
      "fadd z19.h, z19.h, z25.h\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "fadd z18.h, z18.h, z27.h\n"
      "fadd z17.h, z17.h, z26.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fadd z16.h, z16.h, z25.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "fmin z18.h, p3/M, z18.h, z12.h\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmin z16.h, p3/M, z16.h, z12.h\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "st1h { z24.h }, p2, [x10]\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "fmax z18.h, p3/M, z18.h, z11.h\n"
      "st1h { z23.h }, p1, [x10, #1, MUL VL]\n"
      "st1h { z22.h }, p0, [x10, #2, MUL VL]\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "fmax z16.h, p3/M, z16.h, z11.h\n"
      "st1h { z21.h }, p2, [x28]\n"
      "incb x10, ALL, MUL #3\n"
      "st1h { z20.h }, p1, [x28, #1, MUL VL]\n"
      "st1h { z19.h }, p0, [x28, #2, MUL VL]\n"
      "incb x28, ALL, MUL #3\n"
      "st1h { z18.h }, p2, [x27]\n"
      "st1h { z17.h }, p1, [x27, #1, MUL VL]\n"
      "st1h { z16.h }, p0, [x27, #2, MUL VL]\n"
      "incb x27, ALL, MUL #3\n"
      "bgt 11b\n"
      "b 52f\n"
      "14:"  // Initial: Height 4
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "mov x9, %x[bias]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "15:"  // Initial: Height 4: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "whilelt p1.h, x21, x11\n"
      "inch x21\n"
      "whilelt p0.h, x21, x11\n"
      "inch x21\n"
      "cbnz %x[bias], 16f\n"
      "mov z30.b, #0\n"
      "mov z29.b, #0\n"
      "mov z28.b, #0\n"
      "b 17f\n"
      "16:"  // Initial: Height 4: Width 3: bias
      "ld1h { z30.h }, p2/Z, [x9]\n"
      "ld1h { z29.h }, p1/Z, [x9, #1, MUL VL]\n"
      "ld1h { z28.h }, p0/Z, [x9, #2, MUL VL]\n"
      "17:"  // Initial: Height 4: Width 3: init done
      "ld1h { z27.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z26.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "dech x11, ALL, MUL #3\n"
      "incb x9, ALL, MUL #3\n"
      "ld1h { z25.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z24.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z23.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "ld1h { z22.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "ld1h { z21.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z20.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "fadd z27.h, z27.h, z30.h\n"
      "fadd z26.h, z26.h, z29.h\n"
      "ld1h { z19.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x20, #-7, MUL VL]\n"
      "fadd z25.h, z25.h, z28.h\n"
      "fadd z24.h, z24.h, z30.h\n"
      "ld1h { z17.h }, p1/Z, [x20, #-6, MUL VL]\n"
      "ld1h { z16.h }, p0/Z, [x20, #-5, MUL VL]\n"
      "fadd z23.h, z23.h, z29.h\n"
      "fadd z22.h, z22.h, z28.h\n"
      "fadd z21.h, z21.h, z30.h\n"
      "fadd z20.h, z20.h, z29.h\n"
      "fmin z27.h, p3/M, z27.h, z12.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "fadd z19.h, z19.h, z28.h\n"
      "fadd z18.h, z18.h, z30.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fadd z17.h, z17.h, z29.h\n"
      "fadd z16.h, z16.h, z28.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "fmin z18.h, p3/M, z18.h, z12.h\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmin z16.h, p3/M, z16.h, z12.h\n"
      "cmp x11, XZR\n"
      "fmax z27.h, p3/M, z27.h, z11.h\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "st1h { z27.h }, p2, [x10]\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "st1h { z26.h }, p1, [x10, #1, MUL VL]\n"
      "st1h { z25.h }, p0, [x10, #2, MUL VL]\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "st1h { z24.h }, p2, [x28]\n"
      "fmax z18.h, p3/M, z18.h, z11.h\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "st1h { z23.h }, p1, [x28, #1, MUL VL]\n"
      "fmax z16.h, p3/M, z16.h, z11.h\n"
      "incb x10, ALL, MUL #3\n"
      "st1h { z22.h }, p0, [x28, #2, MUL VL]\n"
      "incb x28, ALL, MUL #3\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "st1h { z21.h }, p2, [x27]\n"
      "st1h { z20.h }, p1, [x27, #1, MUL VL]\n"
      "st1h { z19.h }, p0, [x27, #2, MUL VL]\n"
      "incb x27, ALL, MUL #3\n"
      "st1h { z18.h }, p2, [x26]\n"
      "st1h { z17.h }, p1, [x26, #1, MUL VL]\n"
      "st1h { z16.h }, p0, [x26, #2, MUL VL]\n"
      "incb x26, ALL, MUL #3\n"
      "bgt 15b\n"
      "b 52f\n"
      "18:"  // Initial: Height 5
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "mov x9, %x[bias]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "19:"  // Initial: Height 5: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "whilelt p1.h, x21, x11\n"
      "inch x21\n"
      "whilelt p0.h, x21, x11\n"
      "inch x21\n"
      "cbnz %x[bias], 20f\n"
      "mov z1.b, #0\n"
      "mov z0.b, #0\n"
      "mov z31.b, #0\n"
      "b 21f\n"
      "20:"  // Initial: Height 5: Width 3: bias
      "ld1h { z1.h }, p2/Z, [x9]\n"
      "ld1h { z0.h }, p1/Z, [x9, #1, MUL VL]\n"
      "ld1h { z31.h }, p0/Z, [x9, #2, MUL VL]\n"
      "21:"  // Initial: Height 5: Width 3: init done
      "ld1h { z30.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z29.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "dech x11, ALL, MUL #3\n"
      "incb x9, ALL, MUL #3\n"
      "ld1h { z28.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z27.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z26.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "ld1h { z25.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "ld1h { z24.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z23.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "fadd z30.h, z30.h, z1.h\n"
      "fadd z29.h, z29.h, z0.h\n"
      "ld1h { z22.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "ld1h { z21.h }, p2/Z, [x20, #-7, MUL VL]\n"
      "fadd z28.h, z28.h, z31.h\n"
      "fadd z27.h, z27.h, z1.h\n"
      "ld1h { z20.h }, p1/Z, [x20, #-6, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x20, #-5, MUL VL]\n"
      "fadd z26.h, z26.h, z0.h\n"
      "fadd z25.h, z25.h, z31.h\n"
      "ld1h { z18.h }, p2/Z, [x20, #-4, MUL VL]\n"
      "ld1h { z17.h }, p1/Z, [x20, #-3, MUL VL]\n"
      "fadd z24.h, z24.h, z1.h\n"
      "fadd z23.h, z23.h, z0.h\n"
      "ld1h { z16.h }, p0/Z, [x20, #-2, MUL VL]\n"
      "fadd z22.h, z22.h, z31.h\n"
      "fadd z21.h, z21.h, z1.h\n"
      "fmin z30.h, p3/M, z30.h, z12.h\n"
      "fadd z20.h, z20.h, z0.h\n"
      "fadd z19.h, z19.h, z31.h\n"
      "fmin z29.h, p3/M, z29.h, z12.h\n"
      "fmin z28.h, p3/M, z28.h, z12.h\n"
      "fadd z18.h, z18.h, z1.h\n"
      "fadd z17.h, z17.h, z0.h\n"
      "fmin z27.h, p3/M, z27.h, z12.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "fadd z16.h, z16.h, z31.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "fmin z18.h, p3/M, z18.h, z12.h\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmin z16.h, p3/M, z16.h, z12.h\n"
      "cmp x11, XZR\n"
      "fmax z30.h, p3/M, z30.h, z11.h\n"
      "fmax z29.h, p3/M, z29.h, z11.h\n"
      "fmax z28.h, p3/M, z28.h, z11.h\n"
      "fmax z27.h, p3/M, z27.h, z11.h\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "st1h { z30.h }, p2, [x10]\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "st1h { z29.h }, p1, [x10, #1, MUL VL]\n"
      "st1h { z28.h }, p0, [x10, #2, MUL VL]\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "st1h { z27.h }, p2, [x28]\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "st1h { z26.h }, p1, [x28, #1, MUL VL]\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "fmax z18.h, p3/M, z18.h, z11.h\n"
      "st1h { z25.h }, p0, [x28, #2, MUL VL]\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "fmax z16.h, p3/M, z16.h, z11.h\n"
      "st1h { z24.h }, p2, [x27]\n"
      "incb x10, ALL, MUL #3\n"
      "incb x28, ALL, MUL #3\n"
      "st1h { z23.h }, p1, [x27, #1, MUL VL]\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "st1h { z22.h }, p0, [x27, #2, MUL VL]\n"
      "incb x27, ALL, MUL #3\n"
      "st1h { z21.h }, p2, [x26]\n"
      "st1h { z20.h }, p1, [x26, #1, MUL VL]\n"
      "st1h { z19.h }, p0, [x26, #2, MUL VL]\n"
      "incb x26, ALL, MUL #3\n"
      "st1h { z18.h }, p2, [x25]\n"
      "st1h { z17.h }, p1, [x25, #1, MUL VL]\n"
      "st1h { z16.h }, p0, [x25, #2, MUL VL]\n"
      "incb x25, ALL, MUL #3\n"
      "bgt 19b\n"
      "b 52f\n"
      "22:"  // Initial: Height 6
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "mov x9, %x[bias]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "23:"  // Initial: Height 6: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "whilelt p1.h, x21, x11\n"
      "inch x21\n"
      "whilelt p0.h, x21, x11\n"
      "inch x21\n"
      "cbnz %x[bias], 24f\n"
      "mov z4.b, #0\n"
      "mov z3.b, #0\n"
      "mov z2.b, #0\n"
      "b 25f\n"
      "24:"  // Initial: Height 6: Width 3: bias
      "ld1h { z4.h }, p2/Z, [x9]\n"
      "ld1h { z3.h }, p1/Z, [x9, #1, MUL VL]\n"
      "ld1h { z2.h }, p0/Z, [x9, #2, MUL VL]\n"
      "25:"  // Initial: Height 6: Width 3: init done
      "ld1h { z1.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z0.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "dech x11, ALL, MUL #3\n"
      "incb x9, ALL, MUL #3\n"
      "ld1h { z31.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z30.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z29.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "ld1h { z28.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "ld1h { z27.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z26.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "fadd z1.h, z1.h, z4.h\n"
      "fadd z0.h, z0.h, z3.h\n"
      "ld1h { z25.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "ld1h { z24.h }, p2/Z, [x20, #-7, MUL VL]\n"
      "fadd z31.h, z31.h, z2.h\n"
      "fadd z30.h, z30.h, z4.h\n"
      "ld1h { z23.h }, p1/Z, [x20, #-6, MUL VL]\n"
      "ld1h { z22.h }, p0/Z, [x20, #-5, MUL VL]\n"
      "fadd z29.h, z29.h, z3.h\n"
      "fadd z28.h, z28.h, z2.h\n"
      "ld1h { z21.h }, p2/Z, [x20, #-4, MUL VL]\n"
      "ld1h { z20.h }, p1/Z, [x20, #-3, MUL VL]\n"
      "fadd z27.h, z27.h, z4.h\n"
      "fadd z26.h, z26.h, z3.h\n"
      "ld1h { z19.h }, p0/Z, [x20, #-2, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x20, #-1, MUL VL]\n"
      "fadd z25.h, z25.h, z2.h\n"
      "fadd z24.h, z24.h, z4.h\n"
      "ld1h { z17.h }, p1/Z, [x20]\n"
      "ld1h { z16.h }, p0/Z, [x20, #1, MUL VL]\n"
      "fadd z23.h, z23.h, z3.h\n"
      "fadd z22.h, z22.h, z2.h\n"
      "fadd z21.h, z21.h, z4.h\n"
      "fadd z20.h, z20.h, z3.h\n"
      "fmin z1.h, p3/M, z1.h, z12.h\n"
      "fmin z0.h, p3/M, z0.h, z12.h\n"
      "fadd z19.h, z19.h, z2.h\n"
      "fadd z18.h, z18.h, z4.h\n"
      "fmin z31.h, p3/M, z31.h, z12.h\n"
      "fmin z30.h, p3/M, z30.h, z12.h\n"
      "fadd z17.h, z17.h, z3.h\n"
      "fadd z16.h, z16.h, z2.h\n"
      "fmin z29.h, p3/M, z29.h, z12.h\n"
      "fmin z28.h, p3/M, z28.h, z12.h\n"
      "fmin z27.h, p3/M, z27.h, z12.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "fmin z18.h, p3/M, z18.h, z12.h\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmin z16.h, p3/M, z16.h, z12.h\n"
      "cmp x11, XZR\n"
      "fmax z1.h, p3/M, z1.h, z11.h\n"
      "fmax z0.h, p3/M, z0.h, z11.h\n"
      "fmax z31.h, p3/M, z31.h, z11.h\n"
      "fmax z30.h, p3/M, z30.h, z11.h\n"
      "fmax z29.h, p3/M, z29.h, z11.h\n"
      "st1h { z1.h }, p2, [x10]\n"
      "fmax z28.h, p3/M, z28.h, z11.h\n"
      "fmax z27.h, p3/M, z27.h, z11.h\n"
      "st1h { z0.h }, p1, [x10, #1, MUL VL]\n"
      "st1h { z31.h }, p0, [x10, #2, MUL VL]\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "st1h { z30.h }, p2, [x28]\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "st1h { z29.h }, p1, [x28, #1, MUL VL]\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "st1h { z28.h }, p0, [x28, #2, MUL VL]\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "st1h { z27.h }, p2, [x27]\n"
      "fmax z18.h, p3/M, z18.h, z11.h\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "st1h { z26.h }, p1, [x27, #1, MUL VL]\n"
      "fmax z16.h, p3/M, z16.h, z11.h\n"
      "incb x10, ALL, MUL #3\n"
      "st1h { z25.h }, p0, [x27, #2, MUL VL]\n"
      "incb x28, ALL, MUL #3\n"
      "incb x27, ALL, MUL #3\n"
      "st1h { z24.h }, p2, [x26]\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "st1h { z23.h }, p1, [x26, #1, MUL VL]\n"
      "st1h { z22.h }, p0, [x26, #2, MUL VL]\n"
      "incb x26, ALL, MUL #3\n"
      "st1h { z21.h }, p2, [x25]\n"
      "st1h { z20.h }, p1, [x25, #1, MUL VL]\n"
      "st1h { z19.h }, p0, [x25, #2, MUL VL]\n"
      "incb x25, ALL, MUL #3\n"
      "st1h { z18.h }, p2, [x24]\n"
      "st1h { z17.h }, p1, [x24, #1, MUL VL]\n"
      "st1h { z16.h }, p0, [x24, #2, MUL VL]\n"
      "incb x24, ALL, MUL #3\n"
      "bgt 23b\n"
      "b 52f\n"
      "26:"  // Initial: Height 7
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "mov x9, %x[bias]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "add x23, x24, %x[ldout], LSL #1\n"
      "27:"  // Initial: Height 7: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "whilelt p1.h, x21, x11\n"
      "inch x21\n"
      "whilelt p0.h, x21, x11\n"
      "inch x21\n"
      "cbnz %x[bias], 28f\n"
      "mov z7.b, #0\n"
      "mov z6.b, #0\n"
      "mov z5.b, #0\n"
      "b 29f\n"
      "28:"  // Initial: Height 7: Width 3: bias
      "ld1h { z7.h }, p2/Z, [x9]\n"
      "ld1h { z6.h }, p1/Z, [x9, #1, MUL VL]\n"
      "ld1h { z5.h }, p0/Z, [x9, #2, MUL VL]\n"
      "29:"  // Initial: Height 7: Width 3: init done
      "ld1h { z4.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z3.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "dech x11, ALL, MUL #3\n"
      "incb x9, ALL, MUL #3\n"
      "ld1h { z2.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z1.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z0.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "ld1h { z31.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "ld1h { z30.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z29.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "fadd z4.h, z4.h, z7.h\n"
      "fadd z3.h, z3.h, z6.h\n"
      "ld1h { z28.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "ld1h { z27.h }, p2/Z, [x20, #-7, MUL VL]\n"
      "fadd z2.h, z2.h, z5.h\n"
      "fadd z1.h, z1.h, z7.h\n"
      "ld1h { z26.h }, p1/Z, [x20, #-6, MUL VL]\n"
      "ld1h { z25.h }, p0/Z, [x20, #-5, MUL VL]\n"
      "fadd z0.h, z0.h, z6.h\n"
      "fadd z31.h, z31.h, z5.h\n"
      "ld1h { z24.h }, p2/Z, [x20, #-4, MUL VL]\n"
      "ld1h { z23.h }, p1/Z, [x20, #-3, MUL VL]\n"
      "fadd z30.h, z30.h, z7.h\n"
      "fadd z29.h, z29.h, z6.h\n"
      "ld1h { z22.h }, p0/Z, [x20, #-2, MUL VL]\n"
      "ld1h { z21.h }, p2/Z, [x20, #-1, MUL VL]\n"
      "fadd z28.h, z28.h, z5.h\n"
      "fadd z27.h, z27.h, z7.h\n"
      "ld1h { z20.h }, p1/Z, [x20]\n"
      "ld1h { z19.h }, p0/Z, [x20, #1, MUL VL]\n"
      "fadd z26.h, z26.h, z6.h\n"
      "fadd z25.h, z25.h, z5.h\n"
      "ld1h { z18.h }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1h { z17.h }, p1/Z, [x20, #3, MUL VL]\n"
      "fadd z24.h, z24.h, z7.h\n"
      "fadd z23.h, z23.h, z6.h\n"
      "ld1h { z16.h }, p0/Z, [x20, #4, MUL VL]\n"
      "fadd z22.h, z22.h, z5.h\n"
      "fadd z21.h, z21.h, z7.h\n"
      "fmin z4.h, p3/M, z4.h, z12.h\n"
      "fadd z20.h, z20.h, z6.h\n"
      "fadd z19.h, z19.h, z5.h\n"
      "fmin z3.h, p3/M, z3.h, z12.h\n"
      "fmin z2.h, p3/M, z2.h, z12.h\n"
      "fadd z18.h, z18.h, z7.h\n"
      "fadd z17.h, z17.h, z6.h\n"
      "fmin z1.h, p3/M, z1.h, z12.h\n"
      "fmin z0.h, p3/M, z0.h, z12.h\n"
      "fadd z16.h, z16.h, z5.h\n"
      "fmin z31.h, p3/M, z31.h, z12.h\n"
      "fmin z30.h, p3/M, z30.h, z12.h\n"
      "fmin z29.h, p3/M, z29.h, z12.h\n"
      "fmin z28.h, p3/M, z28.h, z12.h\n"
      "fmin z27.h, p3/M, z27.h, z12.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "fmin z18.h, p3/M, z18.h, z12.h\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmin z16.h, p3/M, z16.h, z12.h\n"
      "cmp x11, XZR\n"
      "fmax z4.h, p3/M, z4.h, z11.h\n"
      "fmax z3.h, p3/M, z3.h, z11.h\n"
      "fmax z2.h, p3/M, z2.h, z11.h\n"
      "fmax z1.h, p3/M, z1.h, z11.h\n"
      "fmax z0.h, p3/M, z0.h, z11.h\n"
      "st1h { z4.h }, p2, [x10]\n"
      "fmax z31.h, p3/M, z31.h, z11.h\n"
      "fmax z30.h, p3/M, z30.h, z11.h\n"
      "st1h { z3.h }, p1, [x10, #1, MUL VL]\n"
      "st1h { z2.h }, p0, [x10, #2, MUL VL]\n"
      "fmax z29.h, p3/M, z29.h, z11.h\n"
      "fmax z28.h, p3/M, z28.h, z11.h\n"
      "st1h { z1.h }, p2, [x28]\n"
      "fmax z27.h, p3/M, z27.h, z11.h\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "st1h { z0.h }, p1, [x28, #1, MUL VL]\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "st1h { z31.h }, p0, [x28, #2, MUL VL]\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "st1h { z30.h }, p2, [x27]\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "st1h { z29.h }, p1, [x27, #1, MUL VL]\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "fmax z18.h, p3/M, z18.h, z11.h\n"
      "st1h { z28.h }, p0, [x27, #2, MUL VL]\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "fmax z16.h, p3/M, z16.h, z11.h\n"
      "st1h { z27.h }, p2, [x26]\n"
      "incb x10, ALL, MUL #3\n"
      "incb x28, ALL, MUL #3\n"
      "st1h { z26.h }, p1, [x26, #1, MUL VL]\n"
      "incb x27, ALL, MUL #3\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "st1h { z25.h }, p0, [x26, #2, MUL VL]\n"
      "incb x26, ALL, MUL #3\n"
      "st1h { z24.h }, p2, [x25]\n"
      "st1h { z23.h }, p1, [x25, #1, MUL VL]\n"
      "st1h { z22.h }, p0, [x25, #2, MUL VL]\n"
      "incb x25, ALL, MUL #3\n"
      "st1h { z21.h }, p2, [x24]\n"
      "st1h { z20.h }, p1, [x24, #1, MUL VL]\n"
      "st1h { z19.h }, p0, [x24, #2, MUL VL]\n"
      "incb x24, ALL, MUL #3\n"
      "st1h { z18.h }, p2, [x23]\n"
      "st1h { z17.h }, p1, [x23, #1, MUL VL]\n"
      "st1h { z16.h }, p0, [x23, #2, MUL VL]\n"
      "incb x23, ALL, MUL #3\n"
      "bgt 27b\n"
      "b 52f\n"
      "30:"  // Initial: Height 8
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "mov x9, %x[bias]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "add x23, x24, %x[ldout], LSL #1\n"
      "add x22, x23, %x[ldout], LSL #1\n"
      "31:"  // Initial: Height 8: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "whilelt p1.h, x21, x11\n"
      "inch x21\n"
      "whilelt p0.h, x21, x11\n"
      "inch x21\n"
      "cbnz %x[bias], 32f\n"
      "mov z10.b, #0\n"
      "mov z9.b, #0\n"
      "mov z8.b, #0\n"
      "b 33f\n"
      "32:"  // Initial: Height 8: Width 3: bias
      "ld1h { z10.h }, p2/Z, [x9]\n"
      "ld1h { z9.h }, p1/Z, [x9, #1, MUL VL]\n"
      "ld1h { z8.h }, p0/Z, [x9, #2, MUL VL]\n"
      "33:"  // Initial: Height 8: Width 3: init done
      "ld1h { z7.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z6.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "dech x11, ALL, MUL #3\n"
      "incb x9, ALL, MUL #3\n"
      "ld1h { z5.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z4.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z3.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "ld1h { z2.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "ld1h { z1.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z0.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "fadd z7.h, z7.h, z10.h\n"
      "fadd z6.h, z6.h, z9.h\n"
      "ld1h { z31.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "ld1h { z30.h }, p2/Z, [x20, #-7, MUL VL]\n"
      "fadd z5.h, z5.h, z8.h\n"
      "fadd z4.h, z4.h, z10.h\n"
      "ld1h { z29.h }, p1/Z, [x20, #-6, MUL VL]\n"
      "ld1h { z28.h }, p0/Z, [x20, #-5, MUL VL]\n"
      "fadd z3.h, z3.h, z9.h\n"
      "fadd z2.h, z2.h, z8.h\n"
      "ld1h { z27.h }, p2/Z, [x20, #-4, MUL VL]\n"
      "ld1h { z26.h }, p1/Z, [x20, #-3, MUL VL]\n"
      "fadd z1.h, z1.h, z10.h\n"
      "fadd z0.h, z0.h, z9.h\n"
      "ld1h { z25.h }, p0/Z, [x20, #-2, MUL VL]\n"
      "ld1h { z24.h }, p2/Z, [x20, #-1, MUL VL]\n"
      "fadd z31.h, z31.h, z8.h\n"
      "fadd z30.h, z30.h, z10.h\n"
      "ld1h { z23.h }, p1/Z, [x20]\n"
      "ld1h { z22.h }, p0/Z, [x20, #1, MUL VL]\n"
      "fadd z29.h, z29.h, z9.h\n"
      "fadd z28.h, z28.h, z8.h\n"
      "ld1h { z21.h }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1h { z20.h }, p1/Z, [x20, #3, MUL VL]\n"
      "fadd z27.h, z27.h, z10.h\n"
      "fadd z26.h, z26.h, z9.h\n"
      "ld1h { z19.h }, p0/Z, [x20, #4, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x20, #5, MUL VL]\n"
      "fadd z25.h, z25.h, z8.h\n"
      "fadd z24.h, z24.h, z10.h\n"
      "ld1h { z17.h }, p1/Z, [x20, #6, MUL VL]\n"
      "ld1h { z16.h }, p0/Z, [x20, #7, MUL VL]\n"
      "fadd z23.h, z23.h, z9.h\n"
      "fadd z22.h, z22.h, z8.h\n"
      "fadd z21.h, z21.h, z10.h\n"
      "fadd z20.h, z20.h, z9.h\n"
      "fmin z7.h, p3/M, z7.h, z12.h\n"
      "fmin z6.h, p3/M, z6.h, z12.h\n"
      "fadd z19.h, z19.h, z8.h\n"
      "fadd z18.h, z18.h, z10.h\n"
      "fmin z5.h, p3/M, z5.h, z12.h\n"
      "fmin z4.h, p3/M, z4.h, z12.h\n"
      "fadd z17.h, z17.h, z9.h\n"
      "fadd z16.h, z16.h, z8.h\n"
      "fmin z3.h, p3/M, z3.h, z12.h\n"
      "fmin z2.h, p3/M, z2.h, z12.h\n"
      "fmin z1.h, p3/M, z1.h, z12.h\n"
      "fmin z0.h, p3/M, z0.h, z12.h\n"
      "fmin z31.h, p3/M, z31.h, z12.h\n"
      "fmin z30.h, p3/M, z30.h, z12.h\n"
      "fmin z29.h, p3/M, z29.h, z12.h\n"
      "fmin z28.h, p3/M, z28.h, z12.h\n"
      "fmin z27.h, p3/M, z27.h, z12.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "fmin z18.h, p3/M, z18.h, z12.h\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmin z16.h, p3/M, z16.h, z12.h\n"
      "cmp x11, XZR\n"
      "fmax z7.h, p3/M, z7.h, z11.h\n"
      "fmax z6.h, p3/M, z6.h, z11.h\n"
      "fmax z5.h, p3/M, z5.h, z11.h\n"
      "fmax z4.h, p3/M, z4.h, z11.h\n"
      "fmax z3.h, p3/M, z3.h, z11.h\n"
      "st1h { z7.h }, p2, [x10]\n"
      "fmax z2.h, p3/M, z2.h, z11.h\n"
      "fmax z1.h, p3/M, z1.h, z11.h\n"
      "st1h { z6.h }, p1, [x10, #1, MUL VL]\n"
      "st1h { z5.h }, p0, [x10, #2, MUL VL]\n"
      "fmax z0.h, p3/M, z0.h, z11.h\n"
      "fmax z31.h, p3/M, z31.h, z11.h\n"
      "st1h { z4.h }, p2, [x28]\n"
      "fmax z30.h, p3/M, z30.h, z11.h\n"
      "fmax z29.h, p3/M, z29.h, z11.h\n"
      "st1h { z3.h }, p1, [x28, #1, MUL VL]\n"
      "fmax z28.h, p3/M, z28.h, z11.h\n"
      "fmax z27.h, p3/M, z27.h, z11.h\n"
      "st1h { z2.h }, p0, [x28, #2, MUL VL]\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "st1h { z1.h }, p2, [x27]\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "st1h { z0.h }, p1, [x27, #1, MUL VL]\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "st1h { z31.h }, p0, [x27, #2, MUL VL]\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "st1h { z30.h }, p2, [x26]\n"
      "fmax z18.h, p3/M, z18.h, z11.h\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "st1h { z29.h }, p1, [x26, #1, MUL VL]\n"
      "fmax z16.h, p3/M, z16.h, z11.h\n"
      "incb x10, ALL, MUL #3\n"
      "st1h { z28.h }, p0, [x26, #2, MUL VL]\n"
      "incb x28, ALL, MUL #3\n"
      "incb x27, ALL, MUL #3\n"
      "st1h { z27.h }, p2, [x25]\n"
      "incb x26, ALL, MUL #3\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "st1h { z26.h }, p1, [x25, #1, MUL VL]\n"
      "st1h { z25.h }, p0, [x25, #2, MUL VL]\n"
      "incb x25, ALL, MUL #3\n"
      "st1h { z24.h }, p2, [x24]\n"
      "st1h { z23.h }, p1, [x24, #1, MUL VL]\n"
      "st1h { z22.h }, p0, [x24, #2, MUL VL]\n"
      "incb x24, ALL, MUL #3\n"
      "st1h { z21.h }, p2, [x23]\n"
      "st1h { z20.h }, p1, [x23, #1, MUL VL]\n"
      "st1h { z19.h }, p0, [x23, #2, MUL VL]\n"
      "incb x23, ALL, MUL #3\n"
      "st1h { z18.h }, p2, [x22]\n"
      "st1h { z17.h }, p1, [x22, #1, MUL VL]\n"
      "st1h { z16.h }, p0, [x22, #2, MUL VL]\n"
      "incb x22, ALL, MUL #3\n"
      "bgt 31b\n"
      "subs %x[rows], %x[rows], #0x8\n"
      "add %x[out_ptr], %x[out_ptr], x12\n"
      "bgt 1b\n"
      "b 52f\n"
      "34:"  // Accumulate
      "35:"  // Accumulate: Row loop
      "cmp %x[rows], #0x7\n"
      "bgt 50f\n"
      "beq 48f\n"
      "cmp %x[rows], #0x5\n"
      "bgt 46f\n"
      "beq 44f\n"
      "cmp %x[rows], #0x3\n"
      "bgt 42f\n"
      "beq 40f\n"
      "cmp %x[rows], #0x1\n"
      "bgt 38f\n"
      "mov x11, %x[cols]\n"
      "mov x10, %x[out_ptr]\n"
      "37:"  // Accumulate: Height 1: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "ld1h { z19.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z16.h }, p2/Z, [x10]\n"
      "fadd z19.h, z19.h, z16.h\n"
      "whilelt p1.h, x21, x11\n"
      "inch x21\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "ld1h { z18.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "whilelt p0.h, x21, x11\n"
      "dech x11, ALL, MUL #3\n"
      "inch x21\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "ld1h { z17.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "cmp x11, XZR\n"
      "st1h { z19.h }, p2, [x10]\n"
      "ld1h { z16.h }, p1/Z, [x10, #1, MUL VL]\n"
      "fadd z18.h, z18.h, z16.h\n"
      "fmin z18.h, p3/M, z18.h, z12.h\n"
      "fmax z18.h, p3/M, z18.h, z11.h\n"
      "st1h { z18.h }, p1, [x10, #1, MUL VL]\n"
      "ld1h { z16.h }, p0/Z, [x10, #2, MUL VL]\n"
      "fadd z17.h, z17.h, z16.h\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "st1h { z17.h }, p0, [x10, #2, MUL VL]\n"
      "incb x10, ALL, MUL #3\n"
      "bgt 37b\n"
      "b 52f\n"
      "38:"  // Accumulate: Height 2
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "39:"  // Accumulate: Height 2: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "ld1h { z19.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z18.h }, p2/Z, [x10]\n"
      "ld1h { z17.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z16.h }, p2/Z, [x28]\n"
      "fadd z19.h, z19.h, z18.h\n"
      "whilelt p1.h, x21, x11\n"
      "inch x21\n"
      "fadd z17.h, z17.h, z16.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "ld1h { z21.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "ld1h { z20.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "whilelt p0.h, x21, x11\n"
      "dech x11, ALL, MUL #3\n"
      "inch x21\n"
      "st1h { z19.h }, p2, [x10]\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "ld1h { z19.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z18.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "cmp x11, XZR\n"
      "st1h { z17.h }, p2, [x28]\n"
      "ld1h { z17.h }, p1/Z, [x10, #1, MUL VL]\n"
      "ld1h { z16.h }, p1/Z, [x28, #1, MUL VL]\n"
      "fadd z21.h, z21.h, z17.h\n"
      "fadd z20.h, z20.h, z16.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "st1h { z21.h }, p1, [x10, #1, MUL VL]\n"
      "st1h { z20.h }, p1, [x28, #1, MUL VL]\n"
      "ld1h { z17.h }, p0/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p0/Z, [x28, #2, MUL VL]\n"
      "fadd z19.h, z19.h, z17.h\n"
      "fadd z18.h, z18.h, z16.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "fmin z18.h, p3/M, z18.h, z12.h\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "fmax z18.h, p3/M, z18.h, z11.h\n"
      "st1h { z19.h }, p0, [x10, #2, MUL VL]\n"
      "incb x10, ALL, MUL #3\n"
      "st1h { z18.h }, p0, [x28, #2, MUL VL]\n"
      "incb x28, ALL, MUL #3\n"
      "bgt 39b\n"
      "b 52f\n"
      "40:"  // Accumulate: Height 3
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "41:"  // Accumulate: Height 3: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "ld1h { z20.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z19.h }, p2/Z, [x10]\n"
      "ld1h { z25.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x28]\n"
      "ld1h { z17.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z16.h }, p2/Z, [x27]\n"
      "fadd z20.h, z20.h, z19.h\n"
      "whilelt p1.h, x21, x11\n"
      "inch x21\n"
      "fadd z25.h, z25.h, z18.h\n"
      "fadd z17.h, z17.h, z16.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "ld1h { z24.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "ld1h { z23.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "ld1h { z22.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "whilelt p0.h, x21, x11\n"
      "dech x11, ALL, MUL #3\n"
      "inch x21\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "st1h { z20.h }, p2, [x10]\n"
      "ld1h { z21.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z20.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "cmp x11, XZR\n"
      "st1h { z25.h }, p2, [x28]\n"
      "st1h { z17.h }, p2, [x27]\n"
      "ld1h { z18.h }, p1/Z, [x10, #1, MUL VL]\n"
      "ld1h { z17.h }, p1/Z, [x28, #1, MUL VL]\n"
      "ld1h { z16.h }, p1/Z, [x27, #1, MUL VL]\n"
      "fadd z24.h, z24.h, z18.h\n"
      "fadd z23.h, z23.h, z17.h\n"
      "fadd z22.h, z22.h, z16.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "st1h { z24.h }, p1, [x10, #1, MUL VL]\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "st1h { z23.h }, p1, [x28, #1, MUL VL]\n"
      "st1h { z22.h }, p1, [x27, #1, MUL VL]\n"
      "ld1h { z18.h }, p0/Z, [x10, #2, MUL VL]\n"
      "ld1h { z17.h }, p0/Z, [x28, #2, MUL VL]\n"
      "ld1h { z16.h }, p0/Z, [x27, #2, MUL VL]\n"
      "fadd z21.h, z21.h, z18.h\n"
      "fadd z20.h, z20.h, z17.h\n"
      "fadd z19.h, z19.h, z16.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "st1h { z21.h }, p0, [x10, #2, MUL VL]\n"
      "incb x10, ALL, MUL #3\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "st1h { z20.h }, p0, [x28, #2, MUL VL]\n"
      "incb x28, ALL, MUL #3\n"
      "st1h { z19.h }, p0, [x27, #2, MUL VL]\n"
      "incb x27, ALL, MUL #3\n"
      "bgt 41b\n"
      "b 52f\n"
      "42:"  // Accumulate: Height 4
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "43:"  // Accumulate: Height 4: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "ld1h { z29.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z21.h }, p2/Z, [x10]\n"
      "ld1h { z28.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z20.h }, p2/Z, [x28]\n"
      "ld1h { z19.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x27]\n"
      "ld1h { z17.h }, p2/Z, [x20, #-7, MUL VL]\n"
      "ld1h { z16.h }, p2/Z, [x26]\n"
      "fadd z29.h, z29.h, z21.h\n"
      "whilelt p1.h, x21, x11\n"
      "fadd z28.h, z28.h, z20.h\n"
      "inch x21\n"
      "fadd z19.h, z19.h, z18.h\n"
      "fadd z17.h, z17.h, z16.h\n"
      "fmin z29.h, p3/M, z29.h, z12.h\n"
      "ld1h { z27.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "ld1h { z26.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "ld1h { z25.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "ld1h { z24.h }, p1/Z, [x20, #-6, MUL VL]\n"
      "fmin z28.h, p3/M, z28.h, z12.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "whilelt p0.h, x21, x11\n"
      "dech x11, ALL, MUL #3\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmax z29.h, p3/M, z29.h, z11.h\n"
      "fmax z28.h, p3/M, z28.h, z11.h\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "ld1h { z23.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z22.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "ld1h { z21.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "ld1h { z20.h }, p0/Z, [x20, #-5, MUL VL]\n"
      "cmp x11, XZR\n"
      "inch x21\n"
      "st1h { z29.h }, p2, [x10]\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "st1h { z28.h }, p2, [x28]\n"
      "st1h { z19.h }, p2, [x27]\n"
      "st1h { z17.h }, p2, [x26]\n"
      "ld1h { z19.h }, p1/Z, [x10, #1, MUL VL]\n"
      "ld1h { z18.h }, p1/Z, [x28, #1, MUL VL]\n"
      "ld1h { z17.h }, p1/Z, [x27, #1, MUL VL]\n"
      "ld1h { z16.h }, p1/Z, [x26, #1, MUL VL]\n"
      "fadd z27.h, z27.h, z19.h\n"
      "fadd z26.h, z26.h, z18.h\n"
      "fadd z25.h, z25.h, z17.h\n"
      "fadd z24.h, z24.h, z16.h\n"
      "fmin z27.h, p3/M, z27.h, z12.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmax z27.h, p3/M, z27.h, z11.h\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "st1h { z27.h }, p1, [x10, #1, MUL VL]\n"
      "st1h { z26.h }, p1, [x28, #1, MUL VL]\n"
      "st1h { z25.h }, p1, [x27, #1, MUL VL]\n"
      "st1h { z24.h }, p1, [x26, #1, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x10, #2, MUL VL]\n"
      "ld1h { z18.h }, p0/Z, [x28, #2, MUL VL]\n"
      "ld1h { z17.h }, p0/Z, [x27, #2, MUL VL]\n"
      "ld1h { z16.h }, p0/Z, [x26, #2, MUL VL]\n"
      "fadd z23.h, z23.h, z19.h\n"
      "fadd z22.h, z22.h, z18.h\n"
      "fadd z21.h, z21.h, z17.h\n"
      "fadd z20.h, z20.h, z16.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "st1h { z23.h }, p0, [x10, #2, MUL VL]\n"
      "incb x10, ALL, MUL #3\n"
      "st1h { z22.h }, p0, [x28, #2, MUL VL]\n"
      "incb x28, ALL, MUL #3\n"
      "st1h { z21.h }, p0, [x27, #2, MUL VL]\n"
      "incb x27, ALL, MUL #3\n"
      "st1h { z20.h }, p0, [x26, #2, MUL VL]\n"
      "incb x26, ALL, MUL #3\n"
      "bgt 43b\n"
      "b 52f\n"
      "44:"  // Accumulate: Height 5
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "45:"  // Accumulate: Height 5: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "ld1h { z0.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z16.h }, p2/Z, [x10]\n"
      "ld1h { z31.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z21.h }, p2/Z, [x28]\n"
      "ld1h { z30.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z20.h }, p2/Z, [x27]\n"
      "ld1h { z19.h }, p2/Z, [x20, #-7, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x26]\n"
      "fadd z0.h, z0.h, z16.h\n"
      "whilelt p1.h, x21, x11\n"
      "ld1h { z17.h }, p2/Z, [x20, #-4, MUL VL]\n"
      "ld1h { z16.h }, p2/Z, [x25]\n"
      "fadd z31.h, z31.h, z21.h\n"
      "inch x21\n"
      "fadd z30.h, z30.h, z20.h\n"
      "fadd z19.h, z19.h, z18.h\n"
      "fmin z0.h, p3/M, z0.h, z12.h\n"
      "ld1h { z29.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "ld1h { z28.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "ld1h { z27.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "fadd z17.h, z17.h, z16.h\n"
      "fmin z31.h, p3/M, z31.h, z12.h\n"
      "ld1h { z26.h }, p1/Z, [x20, #-6, MUL VL]\n"
      "ld1h { z25.h }, p1/Z, [x20, #-3, MUL VL]\n"
      "whilelt p0.h, x21, x11\n"
      "dech x11, ALL, MUL #3\n"
      "fmin z30.h, p3/M, z30.h, z12.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "fmin z17.h, p3/M, z17.h, z12.h\n"
      "fmax z0.h, p3/M, z0.h, z11.h\n"
      "fmax z31.h, p3/M, z31.h, z11.h\n"
      "ld1h { z24.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z23.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "cmp x11, XZR\n"
      "fmax z30.h, p3/M, z30.h, z11.h\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "ld1h { z22.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "ld1h { z21.h }, p0/Z, [x20, #-5, MUL VL]\n"
      "st1h { z0.h }, p2, [x10]\n"
      "fmax z17.h, p3/M, z17.h, z11.h\n"
      "ld1h { z20.h }, p0/Z, [x20, #-2, MUL VL]\n"
      "inch x21\n"
      "st1h { z31.h }, p2, [x28]\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "st1h { z30.h }, p2, [x27]\n"
      "st1h { z19.h }, p2, [x26]\n"
      "ld1h { z16.h }, p1/Z, [x10, #1, MUL VL]\n"
      "st1h { z17.h }, p2, [x25]\n"
      "ld1h { z19.h }, p1/Z, [x28, #1, MUL VL]\n"
      "ld1h { z18.h }, p1/Z, [x27, #1, MUL VL]\n"
      "ld1h { z17.h }, p1/Z, [x26, #1, MUL VL]\n"
      "fadd z29.h, z29.h, z16.h\n"
      "ld1h { z16.h }, p1/Z, [x25, #1, MUL VL]\n"
      "fadd z28.h, z28.h, z19.h\n"
      "fadd z27.h, z27.h, z18.h\n"
      "fadd z26.h, z26.h, z17.h\n"
      "fadd z25.h, z25.h, z16.h\n"
      "fmin z29.h, p3/M, z29.h, z12.h\n"
      "fmin z28.h, p3/M, z28.h, z12.h\n"
      "fmin z27.h, p3/M, z27.h, z12.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "fmax z29.h, p3/M, z29.h, z11.h\n"
      "fmax z28.h, p3/M, z28.h, z11.h\n"
      "fmax z27.h, p3/M, z27.h, z11.h\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "st1h { z29.h }, p1, [x10, #1, MUL VL]\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "st1h { z28.h }, p1, [x28, #1, MUL VL]\n"
      "st1h { z27.h }, p1, [x27, #1, MUL VL]\n"
      "st1h { z26.h }, p1, [x26, #1, MUL VL]\n"
      "ld1h { z16.h }, p0/Z, [x10, #2, MUL VL]\n"
      "st1h { z25.h }, p1, [x25, #1, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x28, #2, MUL VL]\n"
      "ld1h { z18.h }, p0/Z, [x27, #2, MUL VL]\n"
      "ld1h { z17.h }, p0/Z, [x26, #2, MUL VL]\n"
      "fadd z24.h, z24.h, z16.h\n"
      "ld1h { z16.h }, p0/Z, [x25, #2, MUL VL]\n"
      "fadd z23.h, z23.h, z19.h\n"
      "fadd z22.h, z22.h, z18.h\n"
      "fadd z21.h, z21.h, z17.h\n"
      "fadd z20.h, z20.h, z16.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "st1h { z24.h }, p0, [x10, #2, MUL VL]\n"
      "incb x10, ALL, MUL #3\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "st1h { z23.h }, p0, [x28, #2, MUL VL]\n"
      "incb x28, ALL, MUL #3\n"
      "st1h { z22.h }, p0, [x27, #2, MUL VL]\n"
      "incb x27, ALL, MUL #3\n"
      "st1h { z21.h }, p0, [x26, #2, MUL VL]\n"
      "incb x26, ALL, MUL #3\n"
      "st1h { z20.h }, p0, [x25, #2, MUL VL]\n"
      "incb x25, ALL, MUL #3\n"
      "bgt 45b\n"
      "b 52f\n"
      "46:"  // Accumulate: Height 6
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "47:"  // Accumulate: Height 6: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "ld1h { z3.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z17.h }, p2/Z, [x10]\n"
      "ld1h { z2.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z16.h }, p2/Z, [x28]\n"
      "ld1h { z1.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z21.h }, p2/Z, [x27]\n"
      "ld1h { z0.h }, p2/Z, [x20, #-7, MUL VL]\n"
      "ld1h { z20.h }, p2/Z, [x26]\n"
      "fadd z3.h, z3.h, z17.h\n"
      "whilelt p1.h, x21, x11\n"
      "ld1h { z19.h }, p2/Z, [x20, #-4, MUL VL]\n"
      "ld1h { z17.h }, p2/Z, [x25]\n"
      "fadd z2.h, z2.h, z16.h\n"
      "inch x21\n"
      "ld1h { z18.h }, p2/Z, [x20, #-1, MUL VL]\n"
      "ld1h { z16.h }, p2/Z, [x24]\n"
      "fadd z1.h, z1.h, z21.h\n"
      "fadd z0.h, z0.h, z20.h\n"
      "fmin z3.h, p3/M, z3.h, z12.h\n"
      "ld1h { z31.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "ld1h { z30.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "fadd z19.h, z19.h, z17.h\n"
      "fmin z2.h, p3/M, z2.h, z12.h\n"
      "ld1h { z29.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "ld1h { z28.h }, p1/Z, [x20, #-6, MUL VL]\n"
      "whilelt p0.h, x21, x11\n"
      "fadd z18.h, z18.h, z16.h\n"
      "fmin z1.h, p3/M, z1.h, z12.h\n"
      "fmin z0.h, p3/M, z0.h, z12.h\n"
      "ld1h { z27.h }, p1/Z, [x20, #-3, MUL VL]\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "fmax z3.h, p3/M, z3.h, z11.h\n"
      "ld1h { z26.h }, p1/Z, [x20]\n"
      "fmax z2.h, p3/M, z2.h, z11.h\n"
      "ld1h { z25.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "ld1h { z24.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "dech x11, ALL, MUL #3\n"
      "fmin z18.h, p3/M, z18.h, z12.h\n"
      "fmax z1.h, p3/M, z1.h, z11.h\n"
      "ld1h { z23.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "ld1h { z22.h }, p0/Z, [x20, #-5, MUL VL]\n"
      "st1h { z3.h }, p2, [x10]\n"
      "fmax z0.h, p3/M, z0.h, z11.h\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "ld1h { z21.h }, p0/Z, [x20, #-2, MUL VL]\n"
      "st1h { z2.h }, p2, [x28]\n"
      "ld1h { z20.h }, p0/Z, [x20, #1, MUL VL]\n"
      "cmp x11, XZR\n"
      "inch x21\n"
      "st1h { z1.h }, p2, [x27]\n"
      "fmax z18.h, p3/M, z18.h, z11.h\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "st1h { z0.h }, p2, [x26]\n"
      "ld1h { z17.h }, p1/Z, [x10, #1, MUL VL]\n"
      "st1h { z19.h }, p2, [x25]\n"
      "ld1h { z16.h }, p1/Z, [x28, #1, MUL VL]\n"
      "st1h { z18.h }, p2, [x24]\n"
      "ld1h { z19.h }, p1/Z, [x27, #1, MUL VL]\n"
      "ld1h { z18.h }, p1/Z, [x26, #1, MUL VL]\n"
      "fadd z31.h, z31.h, z17.h\n"
      "ld1h { z17.h }, p1/Z, [x25, #1, MUL VL]\n"
      "fadd z30.h, z30.h, z16.h\n"
      "ld1h { z16.h }, p1/Z, [x24, #1, MUL VL]\n"
      "fadd z29.h, z29.h, z19.h\n"
      "fadd z28.h, z28.h, z18.h\n"
      "fmin z31.h, p3/M, z31.h, z12.h\n"
      "fadd z27.h, z27.h, z17.h\n"
      "fadd z26.h, z26.h, z16.h\n"
      "fmin z30.h, p3/M, z30.h, z12.h\n"
      "fmin z29.h, p3/M, z29.h, z12.h\n"
      "fmin z28.h, p3/M, z28.h, z12.h\n"
      "fmin z27.h, p3/M, z27.h, z12.h\n"
      "fmax z31.h, p3/M, z31.h, z11.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "fmax z30.h, p3/M, z30.h, z11.h\n"
      "fmax z29.h, p3/M, z29.h, z11.h\n"
      "fmax z28.h, p3/M, z28.h, z11.h\n"
      "fmax z27.h, p3/M, z27.h, z11.h\n"
      "st1h { z31.h }, p1, [x10, #1, MUL VL]\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "st1h { z30.h }, p1, [x28, #1, MUL VL]\n"
      "st1h { z29.h }, p1, [x27, #1, MUL VL]\n"
      "st1h { z28.h }, p1, [x26, #1, MUL VL]\n"
      "ld1h { z17.h }, p0/Z, [x10, #2, MUL VL]\n"
      "st1h { z27.h }, p1, [x25, #1, MUL VL]\n"
      "ld1h { z16.h }, p0/Z, [x28, #2, MUL VL]\n"
      "st1h { z26.h }, p1, [x24, #1, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x27, #2, MUL VL]\n"
      "ld1h { z18.h }, p0/Z, [x26, #2, MUL VL]\n"
      "fadd z25.h, z25.h, z17.h\n"
      "ld1h { z17.h }, p0/Z, [x25, #2, MUL VL]\n"
      "fadd z24.h, z24.h, z16.h\n"
      "ld1h { z16.h }, p0/Z, [x24, #2, MUL VL]\n"
      "fadd z23.h, z23.h, z19.h\n"
      "fadd z22.h, z22.h, z18.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "fadd z21.h, z21.h, z17.h\n"
      "fadd z20.h, z20.h, z16.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "st1h { z25.h }, p0, [x10, #2, MUL VL]\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "incb x10, ALL, MUL #3\n"
      "st1h { z24.h }, p0, [x28, #2, MUL VL]\n"
      "st1h { z23.h }, p0, [x27, #2, MUL VL]\n"
      "incb x28, ALL, MUL #3\n"
      "incb x27, ALL, MUL #3\n"
      "st1h { z22.h }, p0, [x26, #2, MUL VL]\n"
      "incb x26, ALL, MUL #3\n"
      "st1h { z21.h }, p0, [x25, #2, MUL VL]\n"
      "incb x25, ALL, MUL #3\n"
      "st1h { z20.h }, p0, [x24, #2, MUL VL]\n"
      "incb x24, ALL, MUL #3\n"
      "bgt 47b\n"
      "b 52f\n"
      "48:"  // Accumulate: Height 7
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "add x23, x24, %x[ldout], LSL #1\n"
      "49:"  // Accumulate: Height 7: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "ld1h { z26.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z18.h }, p2/Z, [x10]\n"
      "ld1h { z25.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z17.h }, p2/Z, [x28]\n"
      "ld1h { z24.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z16.h }, p2/Z, [x27]\n"
      "ld1h { z23.h }, p2/Z, [x20, #-7, MUL VL]\n"
      "ld1h { z22.h }, p2/Z, [x26]\n"
      "fadd z26.h, z26.h, z18.h\n"
      "whilelt p1.h, x21, x11\n"
      "ld1h { z21.h }, p2/Z, [x20, #-4, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x25]\n"
      "fadd z25.h, z25.h, z17.h\n"
      "inch x21\n"
      "ld1h { z20.h }, p2/Z, [x20, #-1, MUL VL]\n"
      "ld1h { z17.h }, p2/Z, [x24]\n"
      "fadd z24.h, z24.h, z16.h\n"
      "ld1h { z19.h }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1h { z16.h }, p2/Z, [x23]\n"
      "fadd z23.h, z23.h, z22.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "fadd z21.h, z21.h, z18.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "ld1h { z1.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "ld1h { z0.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "fadd z20.h, z20.h, z17.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "ld1h { z31.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "ld1h { z30.h }, p1/Z, [x20, #-6, MUL VL]\n"
      "fadd z19.h, z19.h, z16.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "ld1h { z29.h }, p1/Z, [x20, #-3, MUL VL]\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "ld1h { z28.h }, p1/Z, [x20]\n"
      "ld1h { z27.h }, p1/Z, [x20, #3, MUL VL]\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "fmin z19.h, p3/M, z19.h, z12.h\n"
      "st1h { z26.h }, p2, [x10]\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "st1h { z25.h }, p2, [x28]\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "whilelt p0.h, x21, x11\n"
      "st1h { z24.h }, p2, [x27]\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "dech x11, ALL, MUL #3\n"
      "st1h { z23.h }, p2, [x26]\n"
      "fmax z19.h, p3/M, z19.h, z11.h\n"
      "ld1h { z18.h }, p1/Z, [x10, #1, MUL VL]\n"
      "inch x21\n"
      "st1h { z21.h }, p2, [x25]\n"
      "ld1h { z17.h }, p1/Z, [x28, #1, MUL VL]\n"
      "ld1h { z26.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "st1h { z20.h }, p2, [x24]\n"
      "ld1h { z16.h }, p1/Z, [x27, #1, MUL VL]\n"
      "ld1h { z25.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "cmp x11, XZR\n"
      "st1h { z19.h }, p2, [x23]\n"
      "ld1h { z19.h }, p1/Z, [x26, #1, MUL VL]\n"
      "fadd z1.h, z1.h, z18.h\n"
      "ld1h { z24.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "ld1h { z18.h }, p1/Z, [x25, #1, MUL VL]\n"
      "fadd z0.h, z0.h, z17.h\n"
      "ld1h { z23.h }, p0/Z, [x20, #-5, MUL VL]\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "ld1h { z17.h }, p1/Z, [x24, #1, MUL VL]\n"
      "fadd z31.h, z31.h, z16.h\n"
      "ld1h { z22.h }, p0/Z, [x20, #-2, MUL VL]\n"
      "ld1h { z16.h }, p1/Z, [x23, #1, MUL VL]\n"
      "fadd z30.h, z30.h, z19.h\n"
      "fmin z1.h, p3/M, z1.h, z12.h\n"
      "ld1h { z21.h }, p0/Z, [x20, #1, MUL VL]\n"
      "fadd z29.h, z29.h, z18.h\n"
      "fmin z0.h, p3/M, z0.h, z12.h\n"
      "ld1h { z20.h }, p0/Z, [x20, #4, MUL VL]\n"
      "fadd z28.h, z28.h, z17.h\n"
      "fmin z31.h, p3/M, z31.h, z12.h\n"
      "fadd z27.h, z27.h, z16.h\n"
      "fmin z30.h, p3/M, z30.h, z12.h\n"
      "fmax z1.h, p3/M, z1.h, z11.h\n"
      "fmin z29.h, p3/M, z29.h, z12.h\n"
      "fmin z28.h, p3/M, z28.h, z12.h\n"
      "fmax z0.h, p3/M, z0.h, z11.h\n"
      "fmin z27.h, p3/M, z27.h, z12.h\n"
      "fmax z31.h, p3/M, z31.h, z11.h\n"
      "st1h { z1.h }, p1, [x10, #1, MUL VL]\n"
      "fmax z30.h, p3/M, z30.h, z11.h\n"
      "fmax z29.h, p3/M, z29.h, z11.h\n"
      "st1h { z0.h }, p1, [x28, #1, MUL VL]\n"
      "fmax z28.h, p3/M, z28.h, z11.h\n"
      "st1h { z31.h }, p1, [x27, #1, MUL VL]\n"
      "fmax z27.h, p3/M, z27.h, z11.h\n"
      "ld1h { z18.h }, p0/Z, [x10, #2, MUL VL]\n"
      "st1h { z30.h }, p1, [x26, #1, MUL VL]\n"
      "st1h { z29.h }, p1, [x25, #1, MUL VL]\n"
      "ld1h { z17.h }, p0/Z, [x28, #2, MUL VL]\n"
      "st1h { z28.h }, p1, [x24, #1, MUL VL]\n"
      "ld1h { z16.h }, p0/Z, [x27, #2, MUL VL]\n"
      "fadd z26.h, z26.h, z18.h\n"
      "st1h { z27.h }, p1, [x23, #1, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x26, #2, MUL VL]\n"
      "ld1h { z18.h }, p0/Z, [x25, #2, MUL VL]\n"
      "fadd z25.h, z25.h, z17.h\n"
      "ld1h { z17.h }, p0/Z, [x24, #2, MUL VL]\n"
      "fadd z24.h, z24.h, z16.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "ld1h { z16.h }, p0/Z, [x23, #2, MUL VL]\n"
      "fadd z23.h, z23.h, z19.h\n"
      "fadd z22.h, z22.h, z18.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "fadd z21.h, z21.h, z17.h\n"
      "fadd z20.h, z20.h, z16.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "st1h { z26.h }, p0, [x10, #2, MUL VL]\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "st1h { z25.h }, p0, [x28, #2, MUL VL]\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "incb x10, ALL, MUL #3\n"
      "st1h { z24.h }, p0, [x27, #2, MUL VL]\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "incb x28, ALL, MUL #3\n"
      "st1h { z23.h }, p0, [x26, #2, MUL VL]\n"
      "st1h { z22.h }, p0, [x25, #2, MUL VL]\n"
      "incb x27, ALL, MUL #3\n"
      "incb x26, ALL, MUL #3\n"
      "st1h { z21.h }, p0, [x24, #2, MUL VL]\n"
      "incb x25, ALL, MUL #3\n"
      "incb x24, ALL, MUL #3\n"
      "st1h { z20.h }, p0, [x23, #2, MUL VL]\n"
      "incb x23, ALL, MUL #3\n"
      "bgt 49b\n"
      "b 52f\n"
      "50:"  // Accumulate: Height 8
      "mov x10, %x[out_ptr]\n"
      "mov x11, %x[cols]\n"
      "add x28, x10, %x[ldout], LSL #1\n"
      "add x27, x28, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "add x23, x24, %x[ldout], LSL #1\n"
      "add x22, x23, %x[ldout], LSL #1\n"
      "51:"  // Accumulate: Height 8: Block loop
      "mov x21, #0\n"
      "addvl x20, %x[in_ptr], #16\n"
      "whilelt p2.h, x21, x11\n"
      "inch x21\n"
      "ld1h { z27.h }, p2/Z, [%x[in_ptr]]\n"
      "ld1h { z19.h }, p2/Z, [x10]\n"
      "ld1h { z26.h }, p2/Z, [%x[in_ptr], #3, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x28]\n"
      "ld1h { z25.h }, p2/Z, [%x[in_ptr], #6, MUL VL]\n"
      "ld1h { z17.h }, p2/Z, [x27]\n"
      "ld1h { z24.h }, p2/Z, [x20, #-7, MUL VL]\n"
      "ld1h { z16.h }, p2/Z, [x26]\n"
      "fadd z27.h, z27.h, z19.h\n"
      "whilelt p1.h, x21, x11\n"
      "ld1h { z23.h }, p2/Z, [x20, #-4, MUL VL]\n"
      "ld1h { z19.h }, p2/Z, [x25]\n"
      "fadd z26.h, z26.h, z18.h\n"
      "inch x21\n"
      "ld1h { z22.h }, p2/Z, [x20, #-1, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x24]\n"
      "fadd z25.h, z25.h, z17.h\n"
      "ld1h { z21.h }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1h { z17.h }, p2/Z, [x23]\n"
      "fadd z24.h, z24.h, z16.h\n"
      "fmin z27.h, p3/M, z27.h, z12.h\n"
      "ld1h { z20.h }, p2/Z, [x20, #5, MUL VL]\n"
      "ld1h { z16.h }, p2/Z, [x22]\n"
      "fadd z23.h, z23.h, z19.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "fadd z22.h, z22.h, z18.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "ld1h { z3.h }, p1/Z, [%x[in_ptr], #1, MUL VL]\n"
      "ld1h { z2.h }, p1/Z, [%x[in_ptr], #4, MUL VL]\n"
      "fadd z21.h, z21.h, z17.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmax z27.h, p3/M, z27.h, z11.h\n"
      "ld1h { z1.h }, p1/Z, [%x[in_ptr], #7, MUL VL]\n"
      "fadd z20.h, z20.h, z16.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "ld1h { z0.h }, p1/Z, [x20, #-6, MUL VL]\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "ld1h { z31.h }, p1/Z, [x20, #-3, MUL VL]\n"
      "ld1h { z30.h }, p1/Z, [x20]\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "st1h { z27.h }, p2, [x10]\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "ld1h { z29.h }, p1/Z, [x20, #3, MUL VL]\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "st1h { z26.h }, p2, [x28]\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "ld1h { z28.h }, p1/Z, [x20, #6, MUL VL]\n"
      "st1h { z25.h }, p2, [x27]\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "whilelt p0.h, x21, x11\n"
      "st1h { z24.h }, p2, [x26]\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "ld1h { z19.h }, p1/Z, [x10, #1, MUL VL]\n"
      "dech x11, ALL, MUL #3\n"
      "st1h { z23.h }, p2, [x25]\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "ld1h { z18.h }, p1/Z, [x28, #1, MUL VL]\n"
      "inch x21\n"
      "st1h { z22.h }, p2, [x24]\n"
      "ld1h { z17.h }, p1/Z, [x27, #1, MUL VL]\n"
      "ld1h { z27.h }, p0/Z, [%x[in_ptr], #2, MUL VL]\n"
      "st1h { z21.h }, p2, [x23]\n"
      "ld1h { z16.h }, p1/Z, [x26, #1, MUL VL]\n"
      "fadd z3.h, z3.h, z19.h\n"
      "ld1h { z26.h }, p0/Z, [%x[in_ptr], #5, MUL VL]\n"
      "st1h { z20.h }, p2, [x22]\n"
      "ld1h { z19.h }, p1/Z, [x25, #1, MUL VL]\n"
      "fadd z2.h, z2.h, z18.h\n"
      "ld1h { z25.h }, p0/Z, [x20, #-8, MUL VL]\n"
      "ld1h { z18.h }, p1/Z, [x24, #1, MUL VL]\n"
      "fadd z1.h, z1.h, z17.h\n"
      "ld1h { z24.h }, p0/Z, [x20, #-5, MUL VL]\n"
      "cmp x11, XZR\n"
      "ld1h { z17.h }, p1/Z, [x23, #1, MUL VL]\n"
      "fadd z0.h, z0.h, z16.h\n"
      "fmin z3.h, p3/M, z3.h, z12.h\n"
      "ld1h { z23.h }, p0/Z, [x20, #-2, MUL VL]\n"
      "ld1h { z16.h }, p1/Z, [x22, #1, MUL VL]\n"
      "fadd z31.h, z31.h, z19.h\n"
      "fmin z2.h, p3/M, z2.h, z12.h\n"
      "ld1h { z22.h }, p0/Z, [x20, #1, MUL VL]\n"
      "fadd z30.h, z30.h, z18.h\n"
      "fmin z1.h, p3/M, z1.h, z12.h\n"
      "ld1h { z21.h }, p0/Z, [x20, #4, MUL VL]\n"
      "ld1h { z20.h }, p0/Z, [x20, #7, MUL VL]\n"
      "fadd z29.h, z29.h, z17.h\n"
      "fmin z0.h, p3/M, z0.h, z12.h\n"
      "fmax z3.h, p3/M, z3.h, z11.h\n"
      "fadd z28.h, z28.h, z16.h\n"
      "fmin z31.h, p3/M, z31.h, z12.h\n"
      "fmax z2.h, p3/M, z2.h, z11.h\n"
      "fmin z30.h, p3/M, z30.h, z12.h\n"
      "fmax z1.h, p3/M, z1.h, z11.h\n"
      "fmin z29.h, p3/M, z29.h, z12.h\n"
      "fmin z28.h, p3/M, z28.h, z12.h\n"
      "st1h { z3.h }, p1, [x10, #1, MUL VL]\n"
      "st1h { z2.h }, p1, [x28, #1, MUL VL]\n"
      "fmax z0.h, p3/M, z0.h, z11.h\n"
      "fmax z31.h, p3/M, z31.h, z11.h\n"
      "st1h { z1.h }, p1, [x27, #1, MUL VL]\n"
      "fmax z30.h, p3/M, z30.h, z11.h\n"
      "addvl %x[in_ptr], %x[in_ptr], #24\n"
      "fmax z29.h, p3/M, z29.h, z11.h\n"
      "fmax z28.h, p3/M, z28.h, z11.h\n"
      "ld1h { z19.h }, p0/Z, [x10, #2, MUL VL]\n"
      "st1h { z0.h }, p1, [x26, #1, MUL VL]\n"
      "st1h { z31.h }, p1, [x25, #1, MUL VL]\n"
      "ld1h { z18.h }, p0/Z, [x28, #2, MUL VL]\n"
      "st1h { z30.h }, p1, [x24, #1, MUL VL]\n"
      "ld1h { z17.h }, p0/Z, [x27, #2, MUL VL]\n"
      "st1h { z29.h }, p1, [x23, #1, MUL VL]\n"
      "ld1h { z16.h }, p0/Z, [x26, #2, MUL VL]\n"
      "fadd z27.h, z27.h, z19.h\n"
      "st1h { z28.h }, p1, [x22, #1, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x25, #2, MUL VL]\n"
      "fadd z26.h, z26.h, z18.h\n"
      "ld1h { z18.h }, p0/Z, [x24, #2, MUL VL]\n"
      "fadd z25.h, z25.h, z17.h\n"
      "ld1h { z17.h }, p0/Z, [x23, #2, MUL VL]\n"
      "fadd z24.h, z24.h, z16.h\n"
      "fmin z27.h, p3/M, z27.h, z12.h\n"
      "ld1h { z16.h }, p0/Z, [x22, #2, MUL VL]\n"
      "fadd z23.h, z23.h, z19.h\n"
      "fadd z22.h, z22.h, z18.h\n"
      "fmin z26.h, p3/M, z26.h, z12.h\n"
      "fadd z21.h, z21.h, z17.h\n"
      "fmin z25.h, p3/M, z25.h, z12.h\n"
      "fmin z24.h, p3/M, z24.h, z12.h\n"
      "fmax z27.h, p3/M, z27.h, z11.h\n"
      "fadd z20.h, z20.h, z16.h\n"
      "fmin z23.h, p3/M, z23.h, z12.h\n"
      "fmin z22.h, p3/M, z22.h, z12.h\n"
      "fmax z26.h, p3/M, z26.h, z11.h\n"
      "fmin z21.h, p3/M, z21.h, z12.h\n"
      "fmax z25.h, p3/M, z25.h, z11.h\n"
      "fmin z20.h, p3/M, z20.h, z12.h\n"
      "st1h { z27.h }, p0, [x10, #2, MUL VL]\n"
      "fmax z24.h, p3/M, z24.h, z11.h\n"
      "st1h { z26.h }, p0, [x28, #2, MUL VL]\n"
      "fmax z23.h, p3/M, z23.h, z11.h\n"
      "fmax z22.h, p3/M, z22.h, z11.h\n"
      "st1h { z25.h }, p0, [x27, #2, MUL VL]\n"
      "fmax z21.h, p3/M, z21.h, z11.h\n"
      "incb x10, ALL, MUL #3\n"
      "st1h { z24.h }, p0, [x26, #2, MUL VL]\n"
      "fmax z20.h, p3/M, z20.h, z11.h\n"
      "incb x28, ALL, MUL #3\n"
      "st1h { z23.h }, p0, [x25, #2, MUL VL]\n"
      "incb x27, ALL, MUL #3\n"
      "incb x26, ALL, MUL #3\n"
      "st1h { z22.h }, p0, [x24, #2, MUL VL]\n"
      "incb x25, ALL, MUL #3\n"
      "incb x24, ALL, MUL #3\n"
      "st1h { z21.h }, p0, [x23, #2, MUL VL]\n"
      "incb x23, ALL, MUL #3\n"
      "st1h { z20.h }, p0, [x22, #2, MUL VL]\n"
      "incb x22, ALL, MUL #3\n"
      "bgt 51b\n"
      "subs %x[rows], %x[rows], #0x8\n"
      "add %x[out_ptr], %x[out_ptr], x12\n"
      "bgt 35b\n"
      "52:"  // Exit
      : [in_ptr] "+&r" (in_ptr), [out_ptr] "+&r" (out_ptr), [rows] "+&r" (rows)
      : [accumulate] "r" (accumulate), [bias] "r" (bias), [cols] "r" (cols), [ldout] "r" (ldout), [maxval] "r" (maxval), [minval] "r" (minval)
      : "cc", "memory", "p0", "p1", "p2", "p3", "x9", "x10", "x11", "x12", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

#endif // (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(ARM_COMPUTE_ENABLE_SVE) && defined(__aarch64__)

