/*
 * Copyright (c) 2025-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#pragma once

#if (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(__aarch64__)
template<>
void MergeResults<24, 8, false>(
    __fp16 *out_ptr,
    const __fp16 * in_ptr,
    const int ldout,
    const int y0, const int ymax,
    const int x0, const int xmax,
    const __fp16 *bias,
    Activation act,
    bool accumulate)
{
    __fp16 maxval = static_cast<__fp16>(std::numeric_limits<float>::infinity());
    __fp16 minval = - static_cast<__fp16>(std::numeric_limits<float>::infinity());

    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            maxval = static_cast<__fp16>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            minval = 0;
            break;
    }

    size_t rows = ymax-y0;
    size_t cols = xmax-x0;

    out_ptr += (y0 * ldout) + x0;
    bias = (bias == nullptr) ? nullptr : bias + x0;

    __asm__ __volatile__(
      "cbz %x[cols], 108f\n"
      "cbz %x[rows], 108f\n"
      "mov x11, #0x10\n"
      "dup v12.8h, %w[maxval]\n"
      "dup v11.8h, %w[minval]\n"
      "mul x11, %x[ldout], x11\n"
      "cbnz %x[accumulate], 66f\n"
      "1:"  // Initial: Row loop
      "cmp %x[rows], #0x7\n"
      "bgt 58f\n"
      "beq 50f\n"
      "cmp %x[rows], #0x5\n"
      "bgt 42f\n"
      "beq 34f\n"
      "cmp %x[rows], #0x3\n"
      "bgt 26f\n"
      "beq 18f\n"
      "cmp %x[rows], #0x1\n"
      "bgt 10f\n"
      "mov x10, %x[cols]\n"
      "mov x9, %x[out_ptr]\n"
      "mov x28, %x[bias]\n"
      "cmp x10, #0x18\n"
      "blt 6f\n"
      "3:"  // Initial: Height 1: Block loop
      "cbnz %x[bias], 4f\n"
      "movi v21.16b, #0\n"
      "movi v20.16b, #0\n"
      "movi v19.16b, #0\n"
      "b 5f\n"
      "4:"  // Initial: Height 1: Width 3: bias
      "ldr q21, [x28, #0]\n"
      "ldr q20, [x28, #0x10]\n"
      "ldr q19, [x28, #0x20]\n"
      "5:"  // Initial: Height 1: Width 3: init done
      "ldr q18, [%x[in_ptr], #0]\n"
      "ldr q17, [%x[in_ptr], #0x10]\n"
      "sub x10, x10, #0x18\n"
      "add x28, x28, #0x30\n"
      "ldr q16, [%x[in_ptr], #0x20]\n"
      "cmp x10, #0x18\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fadd v18.8h, v18.8h, v21.8h\n"
      "fadd v17.8h, v17.8h, v20.8h\n"
      "fadd v16.8h, v16.8h, v19.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str q18, [x9, #0]\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "str q17, [x9, #0x10]\n"
      "str q16, [x9, #0x20]\n"
      "add x9, x9, #0x30\n"
      "bge 3b\n"
      "6:"  // Initial: Height 1: no full blocks
      "cbz x10, 9f\n"
      "mov x20, %x[in_ptr]\n"
      "7:"  // Initial: Height 1: Single loop
      "movi v17.16b, #0\n"
      "cbz %x[bias], 8f\n"
      "ldr h17, [x28, #0]\n"
      "8:"  // Initial: Height 1: Scalar: no bias
      "ldr h16, [%x[in_ptr], #0]\n"
      "subs x10, x10, #0x1\n"
      "add x28, x28, #0x2\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v16.8h, v16.8h, v17.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "str h16, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "bne 7b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "9:"  // Initial: Height 1: no oddments
      "b 108f\n"
      "10:"  // Initial: Height 2
      "mov x10, %x[cols]\n"
      "mov x9, %x[out_ptr]\n"
      "mov x28, %x[bias]\n"
      "cmp x10, #0x18\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "blt 14f\n"
      "11:"  // Initial: Height 2: Block loop
      "cbnz %x[bias], 12f\n"
      "movi v24.16b, #0\n"
      "movi v23.16b, #0\n"
      "movi v22.16b, #0\n"
      "b 13f\n"
      "12:"  // Initial: Height 2: Width 3: bias
      "ldr q24, [x28, #0]\n"
      "ldr q23, [x28, #0x10]\n"
      "ldr q22, [x28, #0x20]\n"
      "13:"  // Initial: Height 2: Width 3: init done
      "ldr q21, [%x[in_ptr], #0]\n"
      "ldr q20, [%x[in_ptr], #0x10]\n"
      "sub x10, x10, #0x18\n"
      "add x28, x28, #0x30\n"
      "ldr q19, [%x[in_ptr], #0x20]\n"
      "ldr q18, [%x[in_ptr], #0x30]\n"
      "cmp x10, #0x18\n"
      "ldr q17, [%x[in_ptr], #0x40]\n"
      "ldr q16, [%x[in_ptr], #0x50]\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fadd v21.8h, v21.8h, v24.8h\n"
      "fadd v20.8h, v20.8h, v23.8h\n"
      "fadd v19.8h, v19.8h, v22.8h\n"
      "fadd v18.8h, v18.8h, v24.8h\n"
      "fadd v17.8h, v17.8h, v23.8h\n"
      "fadd v16.8h, v16.8h, v22.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str q21, [x9, #0]\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "str q20, [x9, #0x10]\n"
      "str q19, [x9, #0x20]\n"
      "add x9, x9, #0x30\n"
      "str q18, [x27, #0]\n"
      "str q17, [x27, #0x10]\n"
      "str q16, [x27, #0x20]\n"
      "add x27, x27, #0x30\n"
      "bge 11b\n"
      "14:"  // Initial: Height 2: no full blocks
      "cbz x10, 17f\n"
      "mov x20, %x[in_ptr]\n"
      "15:"  // Initial: Height 2: Single loop
      "movi v18.16b, #0\n"
      "cbz %x[bias], 16f\n"
      "ldr h18, [x28, #0]\n"
      "16:"  // Initial: Height 2: Scalar: no bias
      "ldr h17, [%x[in_ptr], #0]\n"
      "ldr h16, [%x[in_ptr], #0x30]\n"
      "subs x10, x10, #0x1\n"
      "add x28, x28, #0x2\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v17.8h, v17.8h, v18.8h\n"
      "fadd v16.8h, v16.8h, v18.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "str h17, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "str h16, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "bne 15b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "17:"  // Initial: Height 2: no oddments
      "b 108f\n"
      "18:"  // Initial: Height 3
      "mov x10, %x[cols]\n"
      "mov x9, %x[out_ptr]\n"
      "mov x28, %x[bias]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "blt 22f\n"
      "19:"  // Initial: Height 3: Block loop
      "cbnz %x[bias], 20f\n"
      "movi v27.16b, #0\n"
      "movi v26.16b, #0\n"
      "movi v25.16b, #0\n"
      "b 21f\n"
      "20:"  // Initial: Height 3: Width 3: bias
      "ldr q27, [x28, #0]\n"
      "ldr q26, [x28, #0x10]\n"
      "ldr q25, [x28, #0x20]\n"
      "21:"  // Initial: Height 3: Width 3: init done
      "ldr q24, [%x[in_ptr], #0]\n"
      "ldr q23, [%x[in_ptr], #0x10]\n"
      "sub x10, x10, #0x18\n"
      "add x28, x28, #0x30\n"
      "ldr q22, [%x[in_ptr], #0x20]\n"
      "ldr q21, [%x[in_ptr], #0x30]\n"
      "cmp x10, #0x18\n"
      "ldr q20, [%x[in_ptr], #0x40]\n"
      "ldr q19, [%x[in_ptr], #0x50]\n"
      "ldr q18, [%x[in_ptr], #0x60]\n"
      "ldr q17, [%x[in_ptr], #0x70]\n"
      "fadd v24.8h, v24.8h, v27.8h\n"
      "fadd v23.8h, v23.8h, v26.8h\n"
      "ldr q16, [%x[in_ptr], #0x80]\n"
      "fadd v22.8h, v22.8h, v25.8h\n"
      "fadd v21.8h, v21.8h, v27.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fadd v20.8h, v20.8h, v26.8h\n"
      "fadd v19.8h, v19.8h, v25.8h\n"
      "fadd v18.8h, v18.8h, v27.8h\n"
      "fadd v17.8h, v17.8h, v26.8h\n"
      "fadd v16.8h, v16.8h, v25.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str q24, [x9, #0]\n"
      "str q23, [x9, #0x10]\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str q22, [x9, #0x20]\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "add x9, x9, #0x30\n"
      "str q21, [x27, #0]\n"
      "str q20, [x27, #0x10]\n"
      "str q19, [x27, #0x20]\n"
      "add x27, x27, #0x30\n"
      "str q18, [x26, #0]\n"
      "str q17, [x26, #0x10]\n"
      "str q16, [x26, #0x20]\n"
      "add x26, x26, #0x30\n"
      "bge 19b\n"
      "22:"  // Initial: Height 3: no full blocks
      "cbz x10, 25f\n"
      "mov x20, %x[in_ptr]\n"
      "23:"  // Initial: Height 3: Single loop
      "movi v19.16b, #0\n"
      "cbz %x[bias], 24f\n"
      "ldr h19, [x28, #0]\n"
      "24:"  // Initial: Height 3: Scalar: no bias
      "ldr h18, [%x[in_ptr], #0]\n"
      "ldr h17, [%x[in_ptr], #0x30]\n"
      "subs x10, x10, #0x1\n"
      "add x28, x28, #0x2\n"
      "ldr h16, [%x[in_ptr], #0x60]\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v18.8h, v18.8h, v19.8h\n"
      "fadd v17.8h, v17.8h, v19.8h\n"
      "fadd v16.8h, v16.8h, v19.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "str h18, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "str h17, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "str h16, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "bne 23b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "25:"  // Initial: Height 3: no oddments
      "b 108f\n"
      "26:"  // Initial: Height 4
      "mov x9, %x[out_ptr]\n"
      "mov x10, %x[cols]\n"
      "mov x28, %x[bias]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "blt 30f\n"
      "27:"  // Initial: Height 4: Block loop
      "cbnz %x[bias], 28f\n"
      "movi v30.16b, #0\n"
      "movi v29.16b, #0\n"
      "movi v28.16b, #0\n"
      "b 29f\n"
      "28:"  // Initial: Height 4: Width 3: bias
      "ldr q30, [x28, #0]\n"
      "ldr q29, [x28, #0x10]\n"
      "ldr q28, [x28, #0x20]\n"
      "29:"  // Initial: Height 4: Width 3: init done
      "ldr q27, [%x[in_ptr], #0]\n"
      "ldr q26, [%x[in_ptr], #0x10]\n"
      "sub x10, x10, #0x18\n"
      "add x28, x28, #0x30\n"
      "ldr q25, [%x[in_ptr], #0x20]\n"
      "ldr q24, [%x[in_ptr], #0x30]\n"
      "cmp x10, #0x18\n"
      "ldr q23, [%x[in_ptr], #0x40]\n"
      "ldr q22, [%x[in_ptr], #0x50]\n"
      "ldr q21, [%x[in_ptr], #0x60]\n"
      "ldr q20, [%x[in_ptr], #0x70]\n"
      "fadd v27.8h, v27.8h, v30.8h\n"
      "fadd v26.8h, v26.8h, v29.8h\n"
      "ldr q19, [%x[in_ptr], #0x80]\n"
      "ldr q18, [%x[in_ptr], #0x90]\n"
      "fadd v25.8h, v25.8h, v28.8h\n"
      "fadd v24.8h, v24.8h, v30.8h\n"
      "ldr q17, [%x[in_ptr], #0xa0]\n"
      "ldr q16, [%x[in_ptr], #0xb0]\n"
      "fadd v23.8h, v23.8h, v29.8h\n"
      "fadd v22.8h, v22.8h, v28.8h\n"
      "fadd v21.8h, v21.8h, v30.8h\n"
      "fadd v20.8h, v20.8h, v29.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fadd v19.8h, v19.8h, v28.8h\n"
      "fadd v18.8h, v18.8h, v30.8h\n"
      "fadd v17.8h, v17.8h, v29.8h\n"
      "fadd v16.8h, v16.8h, v28.8h\n"
      "fmin v27.8h, v27.8h, v12.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v27.8h, v27.8h, v11.8h\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "str q27, [x9, #0]\n"
      "str q26, [x9, #0x10]\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "str q25, [x9, #0x20]\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "add x9, x9, #0x30\n"
      "str q24, [x27, #0]\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "str q23, [x27, #0x10]\n"
      "str q22, [x27, #0x20]\n"
      "add x27, x27, #0x30\n"
      "str q21, [x26, #0]\n"
      "str q20, [x26, #0x10]\n"
      "str q19, [x26, #0x20]\n"
      "add x26, x26, #0x30\n"
      "str q18, [x25, #0]\n"
      "str q17, [x25, #0x10]\n"
      "str q16, [x25, #0x20]\n"
      "add x25, x25, #0x30\n"
      "bge 27b\n"
      "30:"  // Initial: Height 4: no full blocks
      "cbz x10, 33f\n"
      "mov x20, %x[in_ptr]\n"
      "31:"  // Initial: Height 4: Single loop
      "movi v20.16b, #0\n"
      "cbz %x[bias], 32f\n"
      "ldr h20, [x28, #0]\n"
      "32:"  // Initial: Height 4: Scalar: no bias
      "ldr h19, [%x[in_ptr], #0]\n"
      "ldr h18, [%x[in_ptr], #0x30]\n"
      "subs x10, x10, #0x1\n"
      "add x28, x28, #0x2\n"
      "ldr h17, [%x[in_ptr], #0x60]\n"
      "ldr h16, [%x[in_ptr], #0x90]\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v19.8h, v19.8h, v20.8h\n"
      "fadd v18.8h, v18.8h, v20.8h\n"
      "fadd v17.8h, v17.8h, v20.8h\n"
      "fadd v16.8h, v16.8h, v20.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str h19, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "str h18, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "str h17, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "str h16, [x25, #0]\n"
      "add x25, x25, #0x2\n"
      "bne 31b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "33:"  // Initial: Height 4: no oddments
      "b 108f\n"
      "34:"  // Initial: Height 5
      "mov x9, %x[out_ptr]\n"
      "mov x10, %x[cols]\n"
      "mov x28, %x[bias]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "blt 38f\n"
      "35:"  // Initial: Height 5: Block loop
      "cbnz %x[bias], 36f\n"
      "movi v1.16b, #0\n"
      "movi v0.16b, #0\n"
      "movi v31.16b, #0\n"
      "b 37f\n"
      "36:"  // Initial: Height 5: Width 3: bias
      "ldr q1, [x28, #0]\n"
      "ldr q0, [x28, #0x10]\n"
      "ldr q31, [x28, #0x20]\n"
      "37:"  // Initial: Height 5: Width 3: init done
      "ldr q30, [%x[in_ptr], #0]\n"
      "ldr q29, [%x[in_ptr], #0x10]\n"
      "sub x10, x10, #0x18\n"
      "add x28, x28, #0x30\n"
      "ldr q28, [%x[in_ptr], #0x20]\n"
      "ldr q27, [%x[in_ptr], #0x30]\n"
      "cmp x10, #0x18\n"
      "ldr q26, [%x[in_ptr], #0x40]\n"
      "ldr q25, [%x[in_ptr], #0x50]\n"
      "ldr q24, [%x[in_ptr], #0x60]\n"
      "ldr q23, [%x[in_ptr], #0x70]\n"
      "fadd v30.8h, v30.8h, v1.8h\n"
      "fadd v29.8h, v29.8h, v0.8h\n"
      "ldr q22, [%x[in_ptr], #0x80]\n"
      "ldr q21, [%x[in_ptr], #0x90]\n"
      "fadd v28.8h, v28.8h, v31.8h\n"
      "fadd v27.8h, v27.8h, v1.8h\n"
      "ldr q20, [%x[in_ptr], #0xa0]\n"
      "ldr q19, [%x[in_ptr], #0xb0]\n"
      "fadd v26.8h, v26.8h, v0.8h\n"
      "fadd v25.8h, v25.8h, v31.8h\n"
      "ldr q18, [%x[in_ptr], #0xc0]\n"
      "ldr q17, [%x[in_ptr], #0xd0]\n"
      "fadd v24.8h, v24.8h, v1.8h\n"
      "fadd v23.8h, v23.8h, v0.8h\n"
      "ldr q16, [%x[in_ptr], #0xe0]\n"
      "fadd v22.8h, v22.8h, v31.8h\n"
      "fadd v21.8h, v21.8h, v1.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fadd v20.8h, v20.8h, v0.8h\n"
      "fadd v19.8h, v19.8h, v31.8h\n"
      "fadd v18.8h, v18.8h, v1.8h\n"
      "fadd v17.8h, v17.8h, v0.8h\n"
      "fadd v16.8h, v16.8h, v31.8h\n"
      "fmin v30.8h, v30.8h, v12.8h\n"
      "fmin v29.8h, v29.8h, v12.8h\n"
      "fmin v28.8h, v28.8h, v12.8h\n"
      "fmin v27.8h, v27.8h, v12.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v30.8h, v30.8h, v11.8h\n"
      "fmax v29.8h, v29.8h, v11.8h\n"
      "fmax v28.8h, v28.8h, v11.8h\n"
      "fmax v27.8h, v27.8h, v11.8h\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "str q30, [x9, #0]\n"
      "str q29, [x9, #0x10]\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "str q28, [x9, #0x20]\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "add x9, x9, #0x30\n"
      "str q27, [x27, #0]\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str q26, [x27, #0x10]\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str q25, [x27, #0x20]\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "add x27, x27, #0x30\n"
      "str q24, [x26, #0]\n"
      "str q23, [x26, #0x10]\n"
      "str q22, [x26, #0x20]\n"
      "add x26, x26, #0x30\n"
      "str q21, [x25, #0]\n"
      "str q20, [x25, #0x10]\n"
      "str q19, [x25, #0x20]\n"
      "add x25, x25, #0x30\n"
      "str q18, [x24, #0]\n"
      "str q17, [x24, #0x10]\n"
      "str q16, [x24, #0x20]\n"
      "add x24, x24, #0x30\n"
      "bge 35b\n"
      "38:"  // Initial: Height 5: no full blocks
      "cbz x10, 41f\n"
      "mov x20, %x[in_ptr]\n"
      "39:"  // Initial: Height 5: Single loop
      "movi v21.16b, #0\n"
      "cbz %x[bias], 40f\n"
      "ldr h21, [x28, #0]\n"
      "40:"  // Initial: Height 5: Scalar: no bias
      "ldr h20, [%x[in_ptr], #0]\n"
      "ldr h19, [%x[in_ptr], #0x30]\n"
      "subs x10, x10, #0x1\n"
      "add x28, x28, #0x2\n"
      "ldr h18, [%x[in_ptr], #0x60]\n"
      "ldr h17, [%x[in_ptr], #0x90]\n"
      "ldr h16, [%x[in_ptr], #0xc0]\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v20.8h, v20.8h, v21.8h\n"
      "fadd v19.8h, v19.8h, v21.8h\n"
      "fadd v18.8h, v18.8h, v21.8h\n"
      "fadd v17.8h, v17.8h, v21.8h\n"
      "fadd v16.8h, v16.8h, v21.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "str h20, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "str h19, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "str h18, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "str h17, [x25, #0]\n"
      "add x25, x25, #0x2\n"
      "str h16, [x24, #0]\n"
      "add x24, x24, #0x2\n"
      "bne 39b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "41:"  // Initial: Height 5: no oddments
      "b 108f\n"
      "42:"  // Initial: Height 6
      "mov x9, %x[out_ptr]\n"
      "mov x10, %x[cols]\n"
      "mov x28, %x[bias]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "add x23, x24, %x[ldout], LSL #1\n"
      "blt 46f\n"
      "43:"  // Initial: Height 6: Block loop
      "cbnz %x[bias], 44f\n"
      "movi v4.16b, #0\n"
      "movi v3.16b, #0\n"
      "movi v2.16b, #0\n"
      "b 45f\n"
      "44:"  // Initial: Height 6: Width 3: bias
      "ldr q4, [x28, #0]\n"
      "ldr q3, [x28, #0x10]\n"
      "ldr q2, [x28, #0x20]\n"
      "45:"  // Initial: Height 6: Width 3: init done
      "ldr q1, [%x[in_ptr], #0]\n"
      "ldr q0, [%x[in_ptr], #0x10]\n"
      "sub x10, x10, #0x18\n"
      "add x28, x28, #0x30\n"
      "ldr q31, [%x[in_ptr], #0x20]\n"
      "ldr q30, [%x[in_ptr], #0x30]\n"
      "cmp x10, #0x18\n"
      "ldr q29, [%x[in_ptr], #0x40]\n"
      "ldr q28, [%x[in_ptr], #0x50]\n"
      "ldr q27, [%x[in_ptr], #0x60]\n"
      "ldr q26, [%x[in_ptr], #0x70]\n"
      "fadd v1.8h, v1.8h, v4.8h\n"
      "fadd v0.8h, v0.8h, v3.8h\n"
      "ldr q25, [%x[in_ptr], #0x80]\n"
      "ldr q24, [%x[in_ptr], #0x90]\n"
      "fadd v31.8h, v31.8h, v2.8h\n"
      "fadd v30.8h, v30.8h, v4.8h\n"
      "ldr q23, [%x[in_ptr], #0xa0]\n"
      "ldr q22, [%x[in_ptr], #0xb0]\n"
      "fadd v29.8h, v29.8h, v3.8h\n"
      "fadd v28.8h, v28.8h, v2.8h\n"
      "ldr q21, [%x[in_ptr], #0xc0]\n"
      "ldr q20, [%x[in_ptr], #0xd0]\n"
      "fadd v27.8h, v27.8h, v4.8h\n"
      "fadd v26.8h, v26.8h, v3.8h\n"
      "ldr q19, [%x[in_ptr], #0xe0]\n"
      "ldr q18, [%x[in_ptr], #0xf0]\n"
      "fadd v25.8h, v25.8h, v2.8h\n"
      "fadd v24.8h, v24.8h, v4.8h\n"
      "ldr q17, [%x[in_ptr], #0x100]\n"
      "ldr q16, [%x[in_ptr], #0x110]\n"
      "fadd v23.8h, v23.8h, v3.8h\n"
      "fadd v22.8h, v22.8h, v2.8h\n"
      "fadd v21.8h, v21.8h, v4.8h\n"
      "fadd v20.8h, v20.8h, v3.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fadd v19.8h, v19.8h, v2.8h\n"
      "fadd v18.8h, v18.8h, v4.8h\n"
      "fadd v17.8h, v17.8h, v3.8h\n"
      "fadd v16.8h, v16.8h, v2.8h\n"
      "fmin v1.8h, v1.8h, v12.8h\n"
      "fmin v0.8h, v0.8h, v12.8h\n"
      "fmin v31.8h, v31.8h, v12.8h\n"
      "fmin v30.8h, v30.8h, v12.8h\n"
      "fmin v29.8h, v29.8h, v12.8h\n"
      "fmin v28.8h, v28.8h, v12.8h\n"
      "fmin v27.8h, v27.8h, v12.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v1.8h, v1.8h, v11.8h\n"
      "fmax v0.8h, v0.8h, v11.8h\n"
      "fmax v31.8h, v31.8h, v11.8h\n"
      "fmax v30.8h, v30.8h, v11.8h\n"
      "fmax v29.8h, v29.8h, v11.8h\n"
      "fmax v28.8h, v28.8h, v11.8h\n"
      "str q1, [x9, #0]\n"
      "str q0, [x9, #0x10]\n"
      "fmax v27.8h, v27.8h, v11.8h\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "str q31, [x9, #0x20]\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "add x9, x9, #0x30\n"
      "str q30, [x27, #0]\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "str q29, [x27, #0x10]\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "str q28, [x27, #0x20]\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "add x27, x27, #0x30\n"
      "str q27, [x26, #0]\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "str q26, [x26, #0x10]\n"
      "str q25, [x26, #0x20]\n"
      "add x26, x26, #0x30\n"
      "str q24, [x25, #0]\n"
      "str q23, [x25, #0x10]\n"
      "str q22, [x25, #0x20]\n"
      "add x25, x25, #0x30\n"
      "str q21, [x24, #0]\n"
      "str q20, [x24, #0x10]\n"
      "str q19, [x24, #0x20]\n"
      "add x24, x24, #0x30\n"
      "str q18, [x23, #0]\n"
      "str q17, [x23, #0x10]\n"
      "str q16, [x23, #0x20]\n"
      "add x23, x23, #0x30\n"
      "bge 43b\n"
      "46:"  // Initial: Height 6: no full blocks
      "cbz x10, 49f\n"
      "mov x20, %x[in_ptr]\n"
      "47:"  // Initial: Height 6: Single loop
      "movi v22.16b, #0\n"
      "cbz %x[bias], 48f\n"
      "ldr h22, [x28, #0]\n"
      "48:"  // Initial: Height 6: Scalar: no bias
      "ldr h21, [%x[in_ptr], #0]\n"
      "ldr h20, [%x[in_ptr], #0x30]\n"
      "subs x10, x10, #0x1\n"
      "add x28, x28, #0x2\n"
      "ldr h19, [%x[in_ptr], #0x60]\n"
      "ldr h18, [%x[in_ptr], #0x90]\n"
      "ldr h17, [%x[in_ptr], #0xc0]\n"
      "ldr h16, [%x[in_ptr], #0xf0]\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v21.8h, v21.8h, v22.8h\n"
      "fadd v20.8h, v20.8h, v22.8h\n"
      "fadd v19.8h, v19.8h, v22.8h\n"
      "fadd v18.8h, v18.8h, v22.8h\n"
      "fadd v17.8h, v17.8h, v22.8h\n"
      "fadd v16.8h, v16.8h, v22.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str h21, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "str h20, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "str h19, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "str h18, [x25, #0]\n"
      "add x25, x25, #0x2\n"
      "str h17, [x24, #0]\n"
      "add x24, x24, #0x2\n"
      "str h16, [x23, #0]\n"
      "add x23, x23, #0x2\n"
      "bne 47b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "49:"  // Initial: Height 6: no oddments
      "b 108f\n"
      "50:"  // Initial: Height 7
      "mov x9, %x[out_ptr]\n"
      "mov x10, %x[cols]\n"
      "mov x28, %x[bias]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "add x23, x24, %x[ldout], LSL #1\n"
      "add x22, x23, %x[ldout], LSL #1\n"
      "blt 54f\n"
      "51:"  // Initial: Height 7: Block loop
      "cbnz %x[bias], 52f\n"
      "movi v7.16b, #0\n"
      "movi v6.16b, #0\n"
      "movi v5.16b, #0\n"
      "b 53f\n"
      "52:"  // Initial: Height 7: Width 3: bias
      "ldr q7, [x28, #0]\n"
      "ldr q6, [x28, #0x10]\n"
      "ldr q5, [x28, #0x20]\n"
      "53:"  // Initial: Height 7: Width 3: init done
      "ldr q4, [%x[in_ptr], #0]\n"
      "ldr q3, [%x[in_ptr], #0x10]\n"
      "sub x10, x10, #0x18\n"
      "add x28, x28, #0x30\n"
      "ldr q2, [%x[in_ptr], #0x20]\n"
      "ldr q1, [%x[in_ptr], #0x30]\n"
      "cmp x10, #0x18\n"
      "ldr q0, [%x[in_ptr], #0x40]\n"
      "ldr q31, [%x[in_ptr], #0x50]\n"
      "ldr q30, [%x[in_ptr], #0x60]\n"
      "ldr q29, [%x[in_ptr], #0x70]\n"
      "fadd v4.8h, v4.8h, v7.8h\n"
      "fadd v3.8h, v3.8h, v6.8h\n"
      "ldr q28, [%x[in_ptr], #0x80]\n"
      "ldr q27, [%x[in_ptr], #0x90]\n"
      "fadd v2.8h, v2.8h, v5.8h\n"
      "fadd v1.8h, v1.8h, v7.8h\n"
      "ldr q26, [%x[in_ptr], #0xa0]\n"
      "ldr q25, [%x[in_ptr], #0xb0]\n"
      "fadd v0.8h, v0.8h, v6.8h\n"
      "fadd v31.8h, v31.8h, v5.8h\n"
      "ldr q24, [%x[in_ptr], #0xc0]\n"
      "ldr q23, [%x[in_ptr], #0xd0]\n"
      "fadd v30.8h, v30.8h, v7.8h\n"
      "fadd v29.8h, v29.8h, v6.8h\n"
      "ldr q22, [%x[in_ptr], #0xe0]\n"
      "ldr q21, [%x[in_ptr], #0xf0]\n"
      "fadd v28.8h, v28.8h, v5.8h\n"
      "fadd v27.8h, v27.8h, v7.8h\n"
      "ldr q20, [%x[in_ptr], #0x100]\n"
      "ldr q19, [%x[in_ptr], #0x110]\n"
      "fadd v26.8h, v26.8h, v6.8h\n"
      "fadd v25.8h, v25.8h, v5.8h\n"
      "ldr q18, [%x[in_ptr], #0x120]\n"
      "ldr q17, [%x[in_ptr], #0x130]\n"
      "fadd v24.8h, v24.8h, v7.8h\n"
      "fadd v23.8h, v23.8h, v6.8h\n"
      "ldr q16, [%x[in_ptr], #0x140]\n"
      "fadd v22.8h, v22.8h, v5.8h\n"
      "fadd v21.8h, v21.8h, v7.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fadd v20.8h, v20.8h, v6.8h\n"
      "fadd v19.8h, v19.8h, v5.8h\n"
      "fadd v18.8h, v18.8h, v7.8h\n"
      "fadd v17.8h, v17.8h, v6.8h\n"
      "fadd v16.8h, v16.8h, v5.8h\n"
      "fmin v4.8h, v4.8h, v12.8h\n"
      "fmin v3.8h, v3.8h, v12.8h\n"
      "fmin v2.8h, v2.8h, v12.8h\n"
      "fmin v1.8h, v1.8h, v12.8h\n"
      "fmin v0.8h, v0.8h, v12.8h\n"
      "fmin v31.8h, v31.8h, v12.8h\n"
      "fmin v30.8h, v30.8h, v12.8h\n"
      "fmin v29.8h, v29.8h, v12.8h\n"
      "fmin v28.8h, v28.8h, v12.8h\n"
      "fmin v27.8h, v27.8h, v12.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v4.8h, v4.8h, v11.8h\n"
      "fmax v3.8h, v3.8h, v11.8h\n"
      "fmax v2.8h, v2.8h, v11.8h\n"
      "fmax v1.8h, v1.8h, v11.8h\n"
      "fmax v0.8h, v0.8h, v11.8h\n"
      "fmax v31.8h, v31.8h, v11.8h\n"
      "str q4, [x9, #0]\n"
      "str q3, [x9, #0x10]\n"
      "fmax v30.8h, v30.8h, v11.8h\n"
      "fmax v29.8h, v29.8h, v11.8h\n"
      "str q2, [x9, #0x20]\n"
      "fmax v28.8h, v28.8h, v11.8h\n"
      "fmax v27.8h, v27.8h, v11.8h\n"
      "add x9, x9, #0x30\n"
      "str q1, [x27, #0]\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "str q0, [x27, #0x10]\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "str q31, [x27, #0x20]\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "add x27, x27, #0x30\n"
      "str q30, [x26, #0]\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str q29, [x26, #0x10]\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str q28, [x26, #0x20]\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "add x26, x26, #0x30\n"
      "str q27, [x25, #0]\n"
      "str q26, [x25, #0x10]\n"
      "str q25, [x25, #0x20]\n"
      "add x25, x25, #0x30\n"
      "str q24, [x24, #0]\n"
      "str q23, [x24, #0x10]\n"
      "str q22, [x24, #0x20]\n"
      "add x24, x24, #0x30\n"
      "str q21, [x23, #0]\n"
      "str q20, [x23, #0x10]\n"
      "str q19, [x23, #0x20]\n"
      "add x23, x23, #0x30\n"
      "str q18, [x22, #0]\n"
      "str q17, [x22, #0x10]\n"
      "str q16, [x22, #0x20]\n"
      "add x22, x22, #0x30\n"
      "bge 51b\n"
      "54:"  // Initial: Height 7: no full blocks
      "cbz x10, 57f\n"
      "mov x20, %x[in_ptr]\n"
      "55:"  // Initial: Height 7: Single loop
      "movi v23.16b, #0\n"
      "cbz %x[bias], 56f\n"
      "ldr h23, [x28, #0]\n"
      "56:"  // Initial: Height 7: Scalar: no bias
      "ldr h22, [%x[in_ptr], #0]\n"
      "ldr h21, [%x[in_ptr], #0x30]\n"
      "subs x10, x10, #0x1\n"
      "add x28, x28, #0x2\n"
      "ldr h20, [%x[in_ptr], #0x60]\n"
      "ldr h19, [%x[in_ptr], #0x90]\n"
      "ldr h18, [%x[in_ptr], #0xc0]\n"
      "ldr h17, [%x[in_ptr], #0xf0]\n"
      "ldr h16, [%x[in_ptr], #0x120]\n"
      "fadd v22.8h, v22.8h, v23.8h\n"
      "fadd v21.8h, v21.8h, v23.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v20.8h, v20.8h, v23.8h\n"
      "fadd v19.8h, v19.8h, v23.8h\n"
      "fadd v18.8h, v18.8h, v23.8h\n"
      "fadd v17.8h, v17.8h, v23.8h\n"
      "fadd v16.8h, v16.8h, v23.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str h22, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "str h21, [x27, #0]\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "add x27, x27, #0x2\n"
      "str h20, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "str h19, [x25, #0]\n"
      "add x25, x25, #0x2\n"
      "str h18, [x24, #0]\n"
      "add x24, x24, #0x2\n"
      "str h17, [x23, #0]\n"
      "add x23, x23, #0x2\n"
      "str h16, [x22, #0]\n"
      "add x22, x22, #0x2\n"
      "bne 55b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "57:"  // Initial: Height 7: no oddments
      "b 108f\n"
      "58:"  // Initial: Height 8
      "mov x9, %x[out_ptr]\n"
      "mov x10, %x[cols]\n"
      "mov x28, %x[bias]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "add x23, x24, %x[ldout], LSL #1\n"
      "add x22, x23, %x[ldout], LSL #1\n"
      "add x21, x22, %x[ldout], LSL #1\n"
      "blt 62f\n"
      "59:"  // Initial: Height 8: Block loop
      "cbnz %x[bias], 60f\n"
      "movi v10.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v8.16b, #0\n"
      "b 61f\n"
      "60:"  // Initial: Height 8: Width 3: bias
      "ldr q10, [x28, #0]\n"
      "ldr q9, [x28, #0x10]\n"
      "ldr q8, [x28, #0x20]\n"
      "61:"  // Initial: Height 8: Width 3: init done
      "ldr q7, [%x[in_ptr], #0]\n"
      "ldr q6, [%x[in_ptr], #0x10]\n"
      "sub x10, x10, #0x18\n"
      "add x28, x28, #0x30\n"
      "ldr q5, [%x[in_ptr], #0x20]\n"
      "ldr q4, [%x[in_ptr], #0x30]\n"
      "cmp x10, #0x18\n"
      "ldr q3, [%x[in_ptr], #0x40]\n"
      "ldr q2, [%x[in_ptr], #0x50]\n"
      "ldr q1, [%x[in_ptr], #0x60]\n"
      "ldr q0, [%x[in_ptr], #0x70]\n"
      "fadd v7.8h, v7.8h, v10.8h\n"
      "fadd v6.8h, v6.8h, v9.8h\n"
      "ldr q31, [%x[in_ptr], #0x80]\n"
      "ldr q30, [%x[in_ptr], #0x90]\n"
      "fadd v5.8h, v5.8h, v8.8h\n"
      "fadd v4.8h, v4.8h, v10.8h\n"
      "ldr q29, [%x[in_ptr], #0xa0]\n"
      "ldr q28, [%x[in_ptr], #0xb0]\n"
      "fadd v3.8h, v3.8h, v9.8h\n"
      "fadd v2.8h, v2.8h, v8.8h\n"
      "ldr q27, [%x[in_ptr], #0xc0]\n"
      "ldr q26, [%x[in_ptr], #0xd0]\n"
      "fadd v1.8h, v1.8h, v10.8h\n"
      "fadd v0.8h, v0.8h, v9.8h\n"
      "ldr q25, [%x[in_ptr], #0xe0]\n"
      "ldr q24, [%x[in_ptr], #0xf0]\n"
      "fadd v31.8h, v31.8h, v8.8h\n"
      "fadd v30.8h, v30.8h, v10.8h\n"
      "ldr q23, [%x[in_ptr], #0x100]\n"
      "ldr q22, [%x[in_ptr], #0x110]\n"
      "fadd v29.8h, v29.8h, v9.8h\n"
      "fadd v28.8h, v28.8h, v8.8h\n"
      "ldr q21, [%x[in_ptr], #0x120]\n"
      "ldr q20, [%x[in_ptr], #0x130]\n"
      "fadd v27.8h, v27.8h, v10.8h\n"
      "fadd v26.8h, v26.8h, v9.8h\n"
      "ldr q19, [%x[in_ptr], #0x140]\n"
      "ldr q18, [%x[in_ptr], #0x150]\n"
      "fadd v25.8h, v25.8h, v8.8h\n"
      "fadd v24.8h, v24.8h, v10.8h\n"
      "ldr q17, [%x[in_ptr], #0x160]\n"
      "ldr q16, [%x[in_ptr], #0x170]\n"
      "fadd v23.8h, v23.8h, v9.8h\n"
      "fadd v22.8h, v22.8h, v8.8h\n"
      "fadd v21.8h, v21.8h, v10.8h\n"
      "fadd v20.8h, v20.8h, v9.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fadd v19.8h, v19.8h, v8.8h\n"
      "fadd v18.8h, v18.8h, v10.8h\n"
      "fadd v17.8h, v17.8h, v9.8h\n"
      "fadd v16.8h, v16.8h, v8.8h\n"
      "fmin v7.8h, v7.8h, v12.8h\n"
      "fmin v6.8h, v6.8h, v12.8h\n"
      "fmin v5.8h, v5.8h, v12.8h\n"
      "fmin v4.8h, v4.8h, v12.8h\n"
      "fmin v3.8h, v3.8h, v12.8h\n"
      "fmin v2.8h, v2.8h, v12.8h\n"
      "fmin v1.8h, v1.8h, v12.8h\n"
      "fmin v0.8h, v0.8h, v12.8h\n"
      "fmin v31.8h, v31.8h, v12.8h\n"
      "fmin v30.8h, v30.8h, v12.8h\n"
      "fmin v29.8h, v29.8h, v12.8h\n"
      "fmin v28.8h, v28.8h, v12.8h\n"
      "fmin v27.8h, v27.8h, v12.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v7.8h, v7.8h, v11.8h\n"
      "fmax v6.8h, v6.8h, v11.8h\n"
      "fmax v5.8h, v5.8h, v11.8h\n"
      "fmax v4.8h, v4.8h, v11.8h\n"
      "fmax v3.8h, v3.8h, v11.8h\n"
      "fmax v2.8h, v2.8h, v11.8h\n"
      "str q7, [x9, #0]\n"
      "str q6, [x9, #0x10]\n"
      "fmax v1.8h, v1.8h, v11.8h\n"
      "fmax v0.8h, v0.8h, v11.8h\n"
      "str q5, [x9, #0x20]\n"
      "fmax v31.8h, v31.8h, v11.8h\n"
      "fmax v30.8h, v30.8h, v11.8h\n"
      "add x9, x9, #0x30\n"
      "str q4, [x27, #0]\n"
      "fmax v29.8h, v29.8h, v11.8h\n"
      "fmax v28.8h, v28.8h, v11.8h\n"
      "str q3, [x27, #0x10]\n"
      "fmax v27.8h, v27.8h, v11.8h\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "str q2, [x27, #0x20]\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "add x27, x27, #0x30\n"
      "str q1, [x26, #0]\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "str q0, [x26, #0x10]\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "str q31, [x26, #0x20]\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "add x26, x26, #0x30\n"
      "str q30, [x25, #0]\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "str q29, [x25, #0x10]\n"
      "str q28, [x25, #0x20]\n"
      "add x25, x25, #0x30\n"
      "str q27, [x24, #0]\n"
      "str q26, [x24, #0x10]\n"
      "str q25, [x24, #0x20]\n"
      "add x24, x24, #0x30\n"
      "str q24, [x23, #0]\n"
      "str q23, [x23, #0x10]\n"
      "str q22, [x23, #0x20]\n"
      "add x23, x23, #0x30\n"
      "str q21, [x22, #0]\n"
      "str q20, [x22, #0x10]\n"
      "str q19, [x22, #0x20]\n"
      "add x22, x22, #0x30\n"
      "str q18, [x21, #0]\n"
      "str q17, [x21, #0x10]\n"
      "str q16, [x21, #0x20]\n"
      "add x21, x21, #0x30\n"
      "bge 59b\n"
      "62:"  // Initial: Height 8: no full blocks
      "cbz x10, 65f\n"
      "mov x20, %x[in_ptr]\n"
      "63:"  // Initial: Height 8: Single loop
      "movi v24.16b, #0\n"
      "cbz %x[bias], 64f\n"
      "ldr h24, [x28, #0]\n"
      "64:"  // Initial: Height 8: Scalar: no bias
      "ldr h23, [%x[in_ptr], #0]\n"
      "ldr h22, [%x[in_ptr], #0x30]\n"
      "subs x10, x10, #0x1\n"
      "add x28, x28, #0x2\n"
      "ldr h21, [%x[in_ptr], #0x60]\n"
      "ldr h20, [%x[in_ptr], #0x90]\n"
      "ldr h19, [%x[in_ptr], #0xc0]\n"
      "ldr h18, [%x[in_ptr], #0xf0]\n"
      "ldr h17, [%x[in_ptr], #0x120]\n"
      "ldr h16, [%x[in_ptr], #0x150]\n"
      "fadd v23.8h, v23.8h, v24.8h\n"
      "fadd v22.8h, v22.8h, v24.8h\n"
      "fadd v21.8h, v21.8h, v24.8h\n"
      "fadd v20.8h, v20.8h, v24.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v19.8h, v19.8h, v24.8h\n"
      "fadd v18.8h, v18.8h, v24.8h\n"
      "fadd v17.8h, v17.8h, v24.8h\n"
      "fadd v16.8h, v16.8h, v24.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmin v16.8h, v16.8h, v12.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "str h23, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "str h22, [x27, #0]\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "fmax v16.8h, v16.8h, v11.8h\n"
      "add x27, x27, #0x2\n"
      "str h21, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "str h20, [x25, #0]\n"
      "add x25, x25, #0x2\n"
      "str h19, [x24, #0]\n"
      "add x24, x24, #0x2\n"
      "str h18, [x23, #0]\n"
      "add x23, x23, #0x2\n"
      "str h17, [x22, #0]\n"
      "add x22, x22, #0x2\n"
      "str h16, [x21, #0]\n"
      "add x21, x21, #0x2\n"
      "bne 63b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "65:"  // Initial: Height 8: no oddments
      "subs %x[rows], %x[rows], #0x8\n"
      "add %x[out_ptr], %x[out_ptr], x11\n"
      "bgt 1b\n"
      "b 108f\n"
      "66:"  // Accumulate
      "67:"  // Accumulate: Row loop
      "cmp %x[rows], #0x7\n"
      "bgt 103f\n"
      "beq 98f\n"
      "cmp %x[rows], #0x5\n"
      "bgt 93f\n"
      "beq 88f\n"
      "cmp %x[rows], #0x3\n"
      "bgt 83f\n"
      "beq 78f\n"
      "cmp %x[rows], #0x1\n"
      "bgt 73f\n"
      "mov x10, %x[cols]\n"
      "mov x9, %x[out_ptr]\n"
      "cmp x10, #0x18\n"
      "blt 70f\n"
      "69:"  // Accumulate: Height 1: Block loop
      "ldr q19, [%x[in_ptr], #0]\n"
      "ldr q16, [x9, #0]\n"
      "sub x10, x10, #0x18\n"
      "ldr q18, [%x[in_ptr], #0x10]\n"
      "ldr q17, [%x[in_ptr], #0x20]\n"
      "cmp x10, #0x18\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fadd v19.8h, v19.8h, v16.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str q19, [x9, #0]\n"
      "ldr q16, [x9, #0x10]\n"
      "fadd v18.8h, v18.8h, v16.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "str q18, [x9, #0x10]\n"
      "ldr q16, [x9, #0x20]\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str q17, [x9, #0x20]\n"
      "add x9, x9, #0x30\n"
      "bge 69b\n"
      "70:"  // Accumulate: Height 1: no full blocks
      "cbz x10, 72f\n"
      "mov x20, %x[in_ptr]\n"
      "71:"  // Accumulate: Height 1: Single loop
      "ldr h17, [%x[in_ptr], #0]\n"
      "ldr h16, [x9, #0]\n"
      "subs x10, x10, #0x1\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str h17, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "bne 71b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "72:"  // Accumulate: Height 1: no oddments
      "b 108f\n"
      "73:"  // Accumulate: Height 2
      "mov x10, %x[cols]\n"
      "mov x9, %x[out_ptr]\n"
      "cmp x10, #0x18\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "blt 75f\n"
      "74:"  // Accumulate: Height 2: Block loop
      "ldr q23, [%x[in_ptr], #0]\n"
      "ldr q22, [x9, #0]\n"
      "sub x10, x10, #0x18\n"
      "ldr q21, [%x[in_ptr], #0x30]\n"
      "ldr q16, [x27, #0]\n"
      "cmp x10, #0x18\n"
      "ldr q20, [%x[in_ptr], #0x10]\n"
      "ldr q19, [%x[in_ptr], #0x40]\n"
      "ldr q18, [%x[in_ptr], #0x20]\n"
      "ldr q17, [%x[in_ptr], #0x50]\n"
      "fadd v23.8h, v23.8h, v22.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fadd v21.8h, v21.8h, v16.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "str q23, [x9, #0]\n"
      "ldr q16, [x9, #0x10]\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fadd v20.8h, v20.8h, v16.8h\n"
      "str q21, [x27, #0]\n"
      "ldr q16, [x27, #0x10]\n"
      "fadd v19.8h, v19.8h, v16.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "str q20, [x9, #0x10]\n"
      "ldr q16, [x9, #0x20]\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "fadd v18.8h, v18.8h, v16.8h\n"
      "str q19, [x27, #0x10]\n"
      "ldr q16, [x27, #0x20]\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "str q18, [x9, #0x20]\n"
      "add x9, x9, #0x30\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str q17, [x27, #0x20]\n"
      "add x27, x27, #0x30\n"
      "bge 74b\n"
      "75:"  // Accumulate: Height 2: no full blocks
      "cbz x10, 77f\n"
      "mov x20, %x[in_ptr]\n"
      "76:"  // Accumulate: Height 2: Single loop
      "ldr h19, [%x[in_ptr], #0]\n"
      "ldr h18, [x9, #0]\n"
      "subs x10, x10, #0x1\n"
      "ldr h17, [%x[in_ptr], #0x30]\n"
      "ldr h16, [x27, #0]\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v19.8h, v19.8h, v18.8h\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str h19, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str h17, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "bne 76b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "77:"  // Accumulate: Height 2: no oddments
      "b 108f\n"
      "78:"  // Accumulate: Height 3
      "mov x10, %x[cols]\n"
      "mov x9, %x[out_ptr]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "blt 80f\n"
      "79:"  // Accumulate: Height 3: Block loop
      "ldr q25, [%x[in_ptr], #0]\n"
      "ldr q19, [x9, #0]\n"
      "sub x10, x10, #0x18\n"
      "ldr q24, [%x[in_ptr], #0x30]\n"
      "ldr q18, [x27, #0]\n"
      "cmp x10, #0x18\n"
      "ldr q17, [%x[in_ptr], #0x60]\n"
      "ldr q16, [x26, #0]\n"
      "ldr q23, [%x[in_ptr], #0x10]\n"
      "ldr q22, [%x[in_ptr], #0x40]\n"
      "fadd v25.8h, v25.8h, v19.8h\n"
      "ldr q21, [%x[in_ptr], #0x70]\n"
      "ldr q20, [%x[in_ptr], #0x20]\n"
      "fadd v24.8h, v24.8h, v18.8h\n"
      "ldr q19, [%x[in_ptr], #0x50]\n"
      "ldr q18, [%x[in_ptr], #0x80]\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "str q25, [x9, #0]\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "ldr q16, [x9, #0x10]\n"
      "str q24, [x27, #0]\n"
      "str q17, [x26, #0]\n"
      "fadd v23.8h, v23.8h, v16.8h\n"
      "ldr q17, [x27, #0x10]\n"
      "ldr q16, [x26, #0x10]\n"
      "fadd v22.8h, v22.8h, v17.8h\n"
      "fadd v21.8h, v21.8h, v16.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "str q23, [x9, #0x10]\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "ldr q16, [x9, #0x20]\n"
      "str q22, [x27, #0x10]\n"
      "str q21, [x26, #0x10]\n"
      "fadd v20.8h, v20.8h, v16.8h\n"
      "ldr q17, [x27, #0x20]\n"
      "ldr q16, [x26, #0x20]\n"
      "fadd v19.8h, v19.8h, v17.8h\n"
      "fadd v18.8h, v18.8h, v16.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str q20, [x9, #0x20]\n"
      "add x9, x9, #0x30\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "str q19, [x27, #0x20]\n"
      "add x27, x27, #0x30\n"
      "str q18, [x26, #0x20]\n"
      "add x26, x26, #0x30\n"
      "bge 79b\n"
      "80:"  // Accumulate: Height 3: no full blocks
      "cbz x10, 82f\n"
      "mov x20, %x[in_ptr]\n"
      "81:"  // Accumulate: Height 3: Single loop
      "ldr h21, [%x[in_ptr], #0]\n"
      "ldr h20, [x9, #0]\n"
      "subs x10, x10, #0x1\n"
      "ldr h19, [%x[in_ptr], #0x30]\n"
      "ldr h18, [x27, #0]\n"
      "ldr h17, [%x[in_ptr], #0x60]\n"
      "ldr h16, [x26, #0]\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v21.8h, v21.8h, v20.8h\n"
      "fadd v19.8h, v19.8h, v18.8h\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str h21, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str h19, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "str h17, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "bne 81b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "82:"  // Accumulate: Height 3: no oddments
      "b 108f\n"
      "83:"  // Accumulate: Height 4
      "mov x9, %x[out_ptr]\n"
      "mov x10, %x[cols]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "blt 85f\n"
      "84:"  // Accumulate: Height 4: Block loop
      "ldr q29, [%x[in_ptr], #0]\n"
      "ldr q20, [x9, #0]\n"
      "sub x10, x10, #0x18\n"
      "ldr q28, [%x[in_ptr], #0x30]\n"
      "ldr q19, [x27, #0]\n"
      "cmp x10, #0x18\n"
      "ldr q27, [%x[in_ptr], #0x60]\n"
      "ldr q18, [x26, #0]\n"
      "ldr q17, [%x[in_ptr], #0x90]\n"
      "ldr q16, [x25, #0]\n"
      "fadd v29.8h, v29.8h, v20.8h\n"
      "ldr q26, [%x[in_ptr], #0x10]\n"
      "ldr q25, [%x[in_ptr], #0x40]\n"
      "fadd v28.8h, v28.8h, v19.8h\n"
      "ldr q24, [%x[in_ptr], #0x70]\n"
      "ldr q23, [%x[in_ptr], #0xa0]\n"
      "fadd v27.8h, v27.8h, v18.8h\n"
      "ldr q22, [%x[in_ptr], #0x20]\n"
      "ldr q21, [%x[in_ptr], #0x50]\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "fmin v29.8h, v29.8h, v12.8h\n"
      "ldr q20, [%x[in_ptr], #0x80]\n"
      "ldr q19, [%x[in_ptr], #0xb0]\n"
      "fmin v28.8h, v28.8h, v12.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fmin v27.8h, v27.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v29.8h, v29.8h, v11.8h\n"
      "fmax v28.8h, v28.8h, v11.8h\n"
      "fmax v27.8h, v27.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str q29, [x9, #0]\n"
      "str q28, [x27, #0]\n"
      "ldr q16, [x9, #0x10]\n"
      "str q27, [x26, #0]\n"
      "str q17, [x25, #0]\n"
      "ldr q18, [x27, #0x10]\n"
      "fadd v26.8h, v26.8h, v16.8h\n"
      "ldr q17, [x26, #0x10]\n"
      "ldr q16, [x25, #0x10]\n"
      "fadd v25.8h, v25.8h, v18.8h\n"
      "fadd v24.8h, v24.8h, v17.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "fadd v23.8h, v23.8h, v16.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "str q26, [x9, #0x10]\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "ldr q17, [x9, #0x20]\n"
      "str q25, [x27, #0x10]\n"
      "str q24, [x26, #0x10]\n"
      "ldr q16, [x27, #0x20]\n"
      "str q23, [x25, #0x10]\n"
      "fadd v22.8h, v22.8h, v17.8h\n"
      "ldr q17, [x26, #0x20]\n"
      "fadd v21.8h, v21.8h, v16.8h\n"
      "ldr q16, [x25, #0x20]\n"
      "fadd v20.8h, v20.8h, v17.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fadd v19.8h, v19.8h, v16.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "str q22, [x9, #0x20]\n"
      "add x9, x9, #0x30\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "str q21, [x27, #0x20]\n"
      "add x27, x27, #0x30\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str q20, [x26, #0x20]\n"
      "add x26, x26, #0x30\n"
      "str q19, [x25, #0x20]\n"
      "add x25, x25, #0x30\n"
      "bge 84b\n"
      "85:"  // Accumulate: Height 4: no full blocks
      "cbz x10, 87f\n"
      "mov x20, %x[in_ptr]\n"
      "86:"  // Accumulate: Height 4: Single loop
      "ldr h23, [%x[in_ptr], #0]\n"
      "ldr h22, [x9, #0]\n"
      "subs x10, x10, #0x1\n"
      "ldr h21, [%x[in_ptr], #0x30]\n"
      "ldr h20, [x27, #0]\n"
      "ldr h19, [%x[in_ptr], #0x60]\n"
      "ldr h18, [x26, #0]\n"
      "ldr h17, [%x[in_ptr], #0x90]\n"
      "ldr h16, [x25, #0]\n"
      "fadd v23.8h, v23.8h, v22.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v21.8h, v21.8h, v20.8h\n"
      "fadd v19.8h, v19.8h, v18.8h\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str h23, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str h21, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "str h19, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "str h17, [x25, #0]\n"
      "add x25, x25, #0x2\n"
      "bne 86b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "87:"  // Accumulate: Height 4: no oddments
      "b 108f\n"
      "88:"  // Accumulate: Height 5
      "mov x9, %x[out_ptr]\n"
      "mov x10, %x[cols]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "blt 90f\n"
      "89:"  // Accumulate: Height 5: Block loop
      "ldr q0, [%x[in_ptr], #0]\n"
      "ldr q16, [x9, #0]\n"
      "sub x10, x10, #0x18\n"
      "ldr q31, [%x[in_ptr], #0x30]\n"
      "ldr q20, [x27, #0]\n"
      "cmp x10, #0x18\n"
      "ldr q30, [%x[in_ptr], #0x60]\n"
      "ldr q19, [x26, #0]\n"
      "ldr q29, [%x[in_ptr], #0x90]\n"
      "ldr q17, [x25, #0]\n"
      "fadd v0.8h, v0.8h, v16.8h\n"
      "ldr q18, [%x[in_ptr], #0xc0]\n"
      "ldr q16, [x24, #0]\n"
      "fadd v31.8h, v31.8h, v20.8h\n"
      "ldr q28, [%x[in_ptr], #0x10]\n"
      "ldr q27, [%x[in_ptr], #0x40]\n"
      "fadd v30.8h, v30.8h, v19.8h\n"
      "ldr q26, [%x[in_ptr], #0x70]\n"
      "ldr q25, [%x[in_ptr], #0xa0]\n"
      "fadd v29.8h, v29.8h, v17.8h\n"
      "fmin v0.8h, v0.8h, v12.8h\n"
      "ldr q24, [%x[in_ptr], #0xd0]\n"
      "ldr q23, [%x[in_ptr], #0x20]\n"
      "fadd v18.8h, v18.8h, v16.8h\n"
      "fmin v31.8h, v31.8h, v12.8h\n"
      "ldr q22, [%x[in_ptr], #0x50]\n"
      "ldr q21, [%x[in_ptr], #0x80]\n"
      "fmin v30.8h, v30.8h, v12.8h\n"
      "ldr q20, [%x[in_ptr], #0xb0]\n"
      "ldr q19, [%x[in_ptr], #0xe0]\n"
      "fmin v29.8h, v29.8h, v12.8h\n"
      "fmax v0.8h, v0.8h, v11.8h\n"
      "fmin v18.8h, v18.8h, v12.8h\n"
      "fmax v31.8h, v31.8h, v11.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "fmax v30.8h, v30.8h, v11.8h\n"
      "str q0, [x9, #0]\n"
      "fmax v29.8h, v29.8h, v11.8h\n"
      "str q31, [x27, #0]\n"
      "fmax v18.8h, v18.8h, v11.8h\n"
      "ldr q17, [x9, #0x10]\n"
      "str q30, [x26, #0]\n"
      "str q29, [x25, #0]\n"
      "ldr q16, [x27, #0x10]\n"
      "str q18, [x24, #0]\n"
      "fadd v28.8h, v28.8h, v17.8h\n"
      "ldr q18, [x26, #0x10]\n"
      "ldr q17, [x25, #0x10]\n"
      "fadd v27.8h, v27.8h, v16.8h\n"
      "ldr q16, [x24, #0x10]\n"
      "fadd v26.8h, v26.8h, v18.8h\n"
      "fadd v25.8h, v25.8h, v17.8h\n"
      "fmin v28.8h, v28.8h, v12.8h\n"
      "fadd v24.8h, v24.8h, v16.8h\n"
      "fmin v27.8h, v27.8h, v12.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "fmax v28.8h, v28.8h, v11.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fmax v27.8h, v27.8h, v11.8h\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "str q28, [x9, #0x10]\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "str q27, [x27, #0x10]\n"
      "str q26, [x26, #0x10]\n"
      "ldr q17, [x9, #0x20]\n"
      "str q25, [x25, #0x10]\n"
      "ldr q16, [x27, #0x20]\n"
      "str q24, [x24, #0x10]\n"
      "ldr q18, [x26, #0x20]\n"
      "fadd v23.8h, v23.8h, v17.8h\n"
      "ldr q17, [x25, #0x20]\n"
      "fadd v22.8h, v22.8h, v16.8h\n"
      "ldr q16, [x24, #0x20]\n"
      "fadd v21.8h, v21.8h, v18.8h\n"
      "fadd v20.8h, v20.8h, v17.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fadd v19.8h, v19.8h, v16.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "str q23, [x9, #0x20]\n"
      "add x9, x9, #0x30\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str q22, [x27, #0x20]\n"
      "add x27, x27, #0x30\n"
      "str q21, [x26, #0x20]\n"
      "add x26, x26, #0x30\n"
      "str q20, [x25, #0x20]\n"
      "add x25, x25, #0x30\n"
      "str q19, [x24, #0x20]\n"
      "add x24, x24, #0x30\n"
      "bge 89b\n"
      "90:"  // Accumulate: Height 5: no full blocks
      "cbz x10, 92f\n"
      "mov x20, %x[in_ptr]\n"
      "91:"  // Accumulate: Height 5: Single loop
      "ldr h24, [%x[in_ptr], #0]\n"
      "ldr h16, [x9, #0]\n"
      "subs x10, x10, #0x1\n"
      "ldr h23, [%x[in_ptr], #0x30]\n"
      "ldr h22, [x27, #0]\n"
      "ldr h21, [%x[in_ptr], #0x60]\n"
      "ldr h20, [x26, #0]\n"
      "ldr h19, [%x[in_ptr], #0x90]\n"
      "ldr h18, [x25, #0]\n"
      "fadd v24.8h, v24.8h, v16.8h\n"
      "ldr h17, [%x[in_ptr], #0xc0]\n"
      "ldr h16, [x24, #0]\n"
      "fadd v23.8h, v23.8h, v22.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v21.8h, v21.8h, v20.8h\n"
      "fadd v19.8h, v19.8h, v18.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str h24, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "str h23, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "str h21, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "str h19, [x25, #0]\n"
      "add x25, x25, #0x2\n"
      "str h17, [x24, #0]\n"
      "add x24, x24, #0x2\n"
      "bne 91b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "92:"  // Accumulate: Height 5: no oddments
      "b 108f\n"
      "93:"  // Accumulate: Height 6
      "mov x9, %x[out_ptr]\n"
      "mov x10, %x[cols]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "add x23, x24, %x[ldout], LSL #1\n"
      "blt 95f\n"
      "94:"  // Accumulate: Height 6: Block loop
      "ldr q4, [%x[in_ptr], #0]\n"
      "ldr q17, [x9, #0]\n"
      "sub x10, x10, #0x18\n"
      "ldr q3, [%x[in_ptr], #0x30]\n"
      "ldr q16, [x27, #0]\n"
      "cmp x10, #0x18\n"
      "ldr q2, [%x[in_ptr], #0x60]\n"
      "ldr q19, [x26, #0]\n"
      "ldr q1, [%x[in_ptr], #0x90]\n"
      "ldr q18, [x25, #0]\n"
      "fadd v4.8h, v4.8h, v17.8h\n"
      "ldr q0, [%x[in_ptr], #0xc0]\n"
      "ldr q17, [x24, #0]\n"
      "fadd v3.8h, v3.8h, v16.8h\n"
      "ldr q31, [%x[in_ptr], #0xf0]\n"
      "ldr q16, [x23, #0]\n"
      "fadd v2.8h, v2.8h, v19.8h\n"
      "ldr q30, [%x[in_ptr], #0x10]\n"
      "ldr q29, [%x[in_ptr], #0x40]\n"
      "fadd v1.8h, v1.8h, v18.8h\n"
      "fmin v4.8h, v4.8h, v12.8h\n"
      "ldr q28, [%x[in_ptr], #0x70]\n"
      "ldr q27, [%x[in_ptr], #0xa0]\n"
      "fadd v0.8h, v0.8h, v17.8h\n"
      "fmin v3.8h, v3.8h, v12.8h\n"
      "ldr q26, [%x[in_ptr], #0xd0]\n"
      "ldr q25, [%x[in_ptr], #0x100]\n"
      "fadd v31.8h, v31.8h, v16.8h\n"
      "fmin v2.8h, v2.8h, v12.8h\n"
      "ldr q24, [%x[in_ptr], #0x20]\n"
      "ldr q23, [%x[in_ptr], #0x50]\n"
      "fmin v1.8h, v1.8h, v12.8h\n"
      "fmax v4.8h, v4.8h, v11.8h\n"
      "ldr q22, [%x[in_ptr], #0x80]\n"
      "ldr q21, [%x[in_ptr], #0xb0]\n"
      "fmin v0.8h, v0.8h, v12.8h\n"
      "fmax v3.8h, v3.8h, v11.8h\n"
      "ldr q20, [%x[in_ptr], #0xe0]\n"
      "ldr q19, [%x[in_ptr], #0x110]\n"
      "fmin v31.8h, v31.8h, v12.8h\n"
      "fmax v2.8h, v2.8h, v11.8h\n"
      "str q4, [x9, #0]\n"
      "fmax v1.8h, v1.8h, v11.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "str q3, [x27, #0]\n"
      "fmax v0.8h, v0.8h, v11.8h\n"
      "str q2, [x26, #0]\n"
      "fmax v31.8h, v31.8h, v11.8h\n"
      "ldr q18, [x9, #0x10]\n"
      "str q1, [x25, #0]\n"
      "ldr q17, [x27, #0x10]\n"
      "str q0, [x24, #0]\n"
      "ldr q16, [x26, #0x10]\n"
      "str q31, [x23, #0]\n"
      "fadd v30.8h, v30.8h, v18.8h\n"
      "ldr q18, [x25, #0x10]\n"
      "fadd v29.8h, v29.8h, v17.8h\n"
      "ldr q17, [x24, #0x10]\n"
      "fadd v28.8h, v28.8h, v16.8h\n"
      "ldr q16, [x23, #0x10]\n"
      "fadd v27.8h, v27.8h, v18.8h\n"
      "fmin v30.8h, v30.8h, v12.8h\n"
      "fmin v29.8h, v29.8h, v12.8h\n"
      "fadd v26.8h, v26.8h, v17.8h\n"
      "fadd v25.8h, v25.8h, v16.8h\n"
      "fmin v28.8h, v28.8h, v12.8h\n"
      "fmin v27.8h, v27.8h, v12.8h\n"
      "fmax v30.8h, v30.8h, v11.8h\n"
      "fmax v29.8h, v29.8h, v11.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "fmax v28.8h, v28.8h, v11.8h\n"
      "fmax v27.8h, v27.8h, v11.8h\n"
      "str q30, [x9, #0x10]\n"
      "str q29, [x27, #0x10]\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "ldr q16, [x9, #0x20]\n"
      "str q28, [x26, #0x10]\n"
      "str q27, [x25, #0x10]\n"
      "ldr q17, [x27, #0x20]\n"
      "str q26, [x24, #0x10]\n"
      "fadd v24.8h, v24.8h, v16.8h\n"
      "ldr q16, [x26, #0x20]\n"
      "str q25, [x23, #0x10]\n"
      "ldr q18, [x25, #0x20]\n"
      "fadd v23.8h, v23.8h, v17.8h\n"
      "ldr q17, [x24, #0x20]\n"
      "fadd v22.8h, v22.8h, v16.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "ldr q16, [x23, #0x20]\n"
      "fadd v21.8h, v21.8h, v18.8h\n"
      "fadd v20.8h, v20.8h, v17.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fadd v19.8h, v19.8h, v16.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "str q24, [x9, #0x20]\n"
      "add x9, x9, #0x30\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "str q23, [x27, #0x20]\n"
      "add x27, x27, #0x30\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str q22, [x26, #0x20]\n"
      "add x26, x26, #0x30\n"
      "str q21, [x25, #0x20]\n"
      "add x25, x25, #0x30\n"
      "str q20, [x24, #0x20]\n"
      "add x24, x24, #0x30\n"
      "str q19, [x23, #0x20]\n"
      "add x23, x23, #0x30\n"
      "bge 94b\n"
      "95:"  // Accumulate: Height 6: no full blocks
      "cbz x10, 97f\n"
      "mov x20, %x[in_ptr]\n"
      "96:"  // Accumulate: Height 6: Single loop
      "ldr h25, [%x[in_ptr], #0]\n"
      "ldr h17, [x9, #0]\n"
      "subs x10, x10, #0x1\n"
      "ldr h24, [%x[in_ptr], #0x30]\n"
      "ldr h16, [x27, #0]\n"
      "ldr h23, [%x[in_ptr], #0x60]\n"
      "ldr h22, [x26, #0]\n"
      "ldr h21, [%x[in_ptr], #0x90]\n"
      "ldr h20, [x25, #0]\n"
      "fadd v25.8h, v25.8h, v17.8h\n"
      "ldr h19, [%x[in_ptr], #0xc0]\n"
      "ldr h18, [x24, #0]\n"
      "fadd v24.8h, v24.8h, v16.8h\n"
      "ldr h17, [%x[in_ptr], #0xf0]\n"
      "ldr h16, [x23, #0]\n"
      "fadd v23.8h, v23.8h, v22.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v21.8h, v21.8h, v20.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "fadd v19.8h, v19.8h, v18.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "str h25, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "str h24, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str h23, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str h21, [x25, #0]\n"
      "add x25, x25, #0x2\n"
      "str h19, [x24, #0]\n"
      "add x24, x24, #0x2\n"
      "str h17, [x23, #0]\n"
      "add x23, x23, #0x2\n"
      "bne 96b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "97:"  // Accumulate: Height 6: no oddments
      "b 108f\n"
      "98:"  // Accumulate: Height 7
      "mov x9, %x[out_ptr]\n"
      "mov x10, %x[cols]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "add x23, x24, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "add x22, x23, %x[ldout], LSL #1\n"
      "blt 100f\n"
      "99:"  // Accumulate: Height 7: Block loop
      "ldr q7, [%x[in_ptr], #0]\n"
      "ldr q18, [x9, #0]\n"
      "sub x10, x10, #0x18\n"
      "ldr q6, [%x[in_ptr], #0x30]\n"
      "ldr q17, [x27, #0]\n"
      "cmp x10, #0x18\n"
      "ldr q5, [%x[in_ptr], #0x60]\n"
      "ldr q16, [x26, #0]\n"
      "ldr q4, [%x[in_ptr], #0x90]\n"
      "ldr q19, [x25, #0]\n"
      "fadd v7.8h, v7.8h, v18.8h\n"
      "ldr q3, [%x[in_ptr], #0xc0]\n"
      "ldr q18, [x24, #0]\n"
      "fadd v6.8h, v6.8h, v17.8h\n"
      "ldr q2, [%x[in_ptr], #0xf0]\n"
      "ldr q17, [x23, #0]\n"
      "fadd v5.8h, v5.8h, v16.8h\n"
      "ldr q1, [%x[in_ptr], #0x120]\n"
      "ldr q16, [x22, #0]\n"
      "fadd v4.8h, v4.8h, v19.8h\n"
      "fmin v7.8h, v7.8h, v12.8h\n"
      "ldr q0, [%x[in_ptr], #0x10]\n"
      "ldr q31, [%x[in_ptr], #0x40]\n"
      "fadd v3.8h, v3.8h, v18.8h\n"
      "fmin v6.8h, v6.8h, v12.8h\n"
      "ldr q30, [%x[in_ptr], #0x70]\n"
      "ldr q29, [%x[in_ptr], #0xa0]\n"
      "fadd v2.8h, v2.8h, v17.8h\n"
      "fmin v5.8h, v5.8h, v12.8h\n"
      "ldr q28, [%x[in_ptr], #0xd0]\n"
      "ldr q27, [%x[in_ptr], #0x100]\n"
      "fadd v1.8h, v1.8h, v16.8h\n"
      "fmin v4.8h, v4.8h, v12.8h\n"
      "ldr q26, [%x[in_ptr], #0x130]\n"
      "ldr q25, [%x[in_ptr], #0x20]\n"
      "fmin v3.8h, v3.8h, v12.8h\n"
      "fmax v7.8h, v7.8h, v11.8h\n"
      "ldr q24, [%x[in_ptr], #0x50]\n"
      "ldr q23, [%x[in_ptr], #0x80]\n"
      "fmin v2.8h, v2.8h, v12.8h\n"
      "fmax v6.8h, v6.8h, v11.8h\n"
      "ldr q22, [%x[in_ptr], #0xb0]\n"
      "ldr q21, [%x[in_ptr], #0xe0]\n"
      "fmin v1.8h, v1.8h, v12.8h\n"
      "fmax v5.8h, v5.8h, v11.8h\n"
      "ldr q20, [%x[in_ptr], #0x110]\n"
      "ldr q19, [%x[in_ptr], #0x140]\n"
      "str q7, [x9, #0]\n"
      "fmax v4.8h, v4.8h, v11.8h\n"
      "str q6, [x27, #0]\n"
      "fmax v3.8h, v3.8h, v11.8h\n"
      "fmax v2.8h, v2.8h, v11.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "str q5, [x26, #0]\n"
      "fmax v1.8h, v1.8h, v11.8h\n"
      "ldr q16, [x9, #0x10]\n"
      "str q4, [x25, #0]\n"
      "ldr q18, [x27, #0x10]\n"
      "str q3, [x24, #0]\n"
      "ldr q17, [x26, #0x10]\n"
      "str q2, [x23, #0]\n"
      "fadd v0.8h, v0.8h, v16.8h\n"
      "ldr q16, [x25, #0x10]\n"
      "str q1, [x22, #0]\n"
      "fadd v31.8h, v31.8h, v18.8h\n"
      "ldr q18, [x24, #0x10]\n"
      "fadd v30.8h, v30.8h, v17.8h\n"
      "ldr q17, [x23, #0x10]\n"
      "fadd v29.8h, v29.8h, v16.8h\n"
      "fmin v0.8h, v0.8h, v12.8h\n"
      "ldr q16, [x22, #0x10]\n"
      "fadd v28.8h, v28.8h, v18.8h\n"
      "fmin v31.8h, v31.8h, v12.8h\n"
      "fmin v30.8h, v30.8h, v12.8h\n"
      "fadd v27.8h, v27.8h, v17.8h\n"
      "fadd v26.8h, v26.8h, v16.8h\n"
      "fmin v29.8h, v29.8h, v12.8h\n"
      "fmin v28.8h, v28.8h, v12.8h\n"
      "fmax v0.8h, v0.8h, v11.8h\n"
      "fmin v27.8h, v27.8h, v12.8h\n"
      "fmax v31.8h, v31.8h, v11.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "fmax v30.8h, v30.8h, v11.8h\n"
      "str q0, [x9, #0x10]\n"
      "fmax v29.8h, v29.8h, v11.8h\n"
      "fmax v28.8h, v28.8h, v11.8h\n"
      "str q31, [x27, #0x10]\n"
      "fmax v27.8h, v27.8h, v11.8h\n"
      "ldr q16, [x9, #0x20]\n"
      "str q30, [x26, #0x10]\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "ldr q18, [x27, #0x20]\n"
      "str q29, [x25, #0x10]\n"
      "str q28, [x24, #0x10]\n"
      "ldr q17, [x26, #0x20]\n"
      "str q27, [x23, #0x10]\n"
      "fadd v25.8h, v25.8h, v16.8h\n"
      "ldr q16, [x25, #0x20]\n"
      "str q26, [x22, #0x10]\n"
      "fadd v24.8h, v24.8h, v18.8h\n"
      "ldr q18, [x24, #0x20]\n"
      "fadd v23.8h, v23.8h, v17.8h\n"
      "ldr q17, [x23, #0x20]\n"
      "fadd v22.8h, v22.8h, v16.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "ldr q16, [x22, #0x20]\n"
      "fadd v21.8h, v21.8h, v18.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fadd v20.8h, v20.8h, v17.8h\n"
      "fadd v19.8h, v19.8h, v16.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "str q25, [x9, #0x20]\n"
      "add x9, x9, #0x30\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "str q24, [x27, #0x20]\n"
      "add x27, x27, #0x30\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "str q23, [x26, #0x20]\n"
      "add x26, x26, #0x30\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str q22, [x25, #0x20]\n"
      "add x25, x25, #0x30\n"
      "str q21, [x24, #0x20]\n"
      "add x24, x24, #0x30\n"
      "str q20, [x23, #0x20]\n"
      "add x23, x23, #0x30\n"
      "str q19, [x22, #0x20]\n"
      "add x22, x22, #0x30\n"
      "bge 99b\n"
      "100:"  // Accumulate: Height 7: no full blocks
      "cbz x10, 102f\n"
      "mov x20, %x[in_ptr]\n"
      "101:"  // Accumulate: Height 7: Single loop
      "ldr h26, [%x[in_ptr], #0]\n"
      "ldr h18, [x9, #0]\n"
      "subs x10, x10, #0x1\n"
      "ldr h25, [%x[in_ptr], #0x30]\n"
      "ldr h17, [x27, #0]\n"
      "ldr h24, [%x[in_ptr], #0x60]\n"
      "ldr h16, [x26, #0]\n"
      "ldr h23, [%x[in_ptr], #0x90]\n"
      "ldr h22, [x25, #0]\n"
      "fadd v26.8h, v26.8h, v18.8h\n"
      "ldr h21, [%x[in_ptr], #0xc0]\n"
      "ldr h20, [x24, #0]\n"
      "fadd v25.8h, v25.8h, v17.8h\n"
      "ldr h19, [%x[in_ptr], #0xf0]\n"
      "ldr h18, [x23, #0]\n"
      "fadd v24.8h, v24.8h, v16.8h\n"
      "ldr h17, [%x[in_ptr], #0x120]\n"
      "ldr h16, [x22, #0]\n"
      "fadd v23.8h, v23.8h, v22.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "fadd v21.8h, v21.8h, v20.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v19.8h, v19.8h, v18.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "str h26, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "str h25, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str h24, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str h23, [x25, #0]\n"
      "add x25, x25, #0x2\n"
      "str h21, [x24, #0]\n"
      "add x24, x24, #0x2\n"
      "str h19, [x23, #0]\n"
      "add x23, x23, #0x2\n"
      "str h17, [x22, #0]\n"
      "add x22, x22, #0x2\n"
      "bne 101b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "102:"  // Accumulate: Height 7: no oddments
      "b 108f\n"
      "103:"  // Accumulate: Height 8
      "mov x9, %x[out_ptr]\n"
      "mov x10, %x[cols]\n"
      "add x27, x9, %x[ldout], LSL #1\n"
      "add x26, x27, %x[ldout], LSL #1\n"
      "add x25, x26, %x[ldout], LSL #1\n"
      "add x24, x25, %x[ldout], LSL #1\n"
      "add x23, x24, %x[ldout], LSL #1\n"
      "cmp x10, #0x18\n"
      "add x22, x23, %x[ldout], LSL #1\n"
      "add x21, x22, %x[ldout], LSL #1\n"
      "blt 105f\n"
      "104:"  // Accumulate: Height 8: Block loop
      "ldr q20, [%x[in_ptr], #0]\n"
      "ldr q19, [x9, #0]\n"
      "sub x10, x10, #0x18\n"
      "ldr q10, [%x[in_ptr], #0x30]\n"
      "ldr q18, [x27, #0]\n"
      "cmp x10, #0x18\n"
      "ldr q9, [%x[in_ptr], #0x60]\n"
      "ldr q17, [x26, #0]\n"
      "ldr q8, [%x[in_ptr], #0x90]\n"
      "ldr q16, [x25, #0]\n"
      "fadd v20.8h, v20.8h, v19.8h\n"
      "ldr q7, [%x[in_ptr], #0xc0]\n"
      "ldr q19, [x24, #0]\n"
      "fadd v10.8h, v10.8h, v18.8h\n"
      "ldr q6, [%x[in_ptr], #0xf0]\n"
      "ldr q18, [x23, #0]\n"
      "fadd v9.8h, v9.8h, v17.8h\n"
      "ldr q5, [%x[in_ptr], #0x120]\n"
      "ldr q17, [x22, #0]\n"
      "fadd v8.8h, v8.8h, v16.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "ldr q4, [%x[in_ptr], #0x150]\n"
      "ldr q16, [x21, #0]\n"
      "fadd v7.8h, v7.8h, v19.8h\n"
      "fmin v10.8h, v10.8h, v12.8h\n"
      "ldr q3, [%x[in_ptr], #0x10]\n"
      "ldr q2, [%x[in_ptr], #0x40]\n"
      "fadd v6.8h, v6.8h, v18.8h\n"
      "fmin v9.8h, v9.8h, v12.8h\n"
      "ldr q1, [%x[in_ptr], #0x70]\n"
      "ldr q0, [%x[in_ptr], #0xa0]\n"
      "fadd v5.8h, v5.8h, v17.8h\n"
      "fmin v8.8h, v8.8h, v12.8h\n"
      "ldr q31, [%x[in_ptr], #0xd0]\n"
      "ldr q30, [%x[in_ptr], #0x100]\n"
      "fadd v4.8h, v4.8h, v16.8h\n"
      "fmin v7.8h, v7.8h, v12.8h\n"
      "ldr q29, [%x[in_ptr], #0x130]\n"
      "ldr q28, [%x[in_ptr], #0x160]\n"
      "fmin v6.8h, v6.8h, v12.8h\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "ldr q27, [%x[in_ptr], #0x20]\n"
      "ldr q26, [%x[in_ptr], #0x50]\n"
      "fmin v5.8h, v5.8h, v12.8h\n"
      "fmax v10.8h, v10.8h, v11.8h\n"
      "ldr q25, [%x[in_ptr], #0x80]\n"
      "ldr q24, [%x[in_ptr], #0xb0]\n"
      "fmin v4.8h, v4.8h, v12.8h\n"
      "fmax v9.8h, v9.8h, v11.8h\n"
      "ldr q23, [%x[in_ptr], #0xe0]\n"
      "ldr q22, [%x[in_ptr], #0x110]\n"
      "str q20, [x9, #0]\n"
      "fmax v8.8h, v8.8h, v11.8h\n"
      "ldr q21, [%x[in_ptr], #0x140]\n"
      "ldr q20, [%x[in_ptr], #0x170]\n"
      "str q10, [x27, #0]\n"
      "fmax v7.8h, v7.8h, v11.8h\n"
      "str q9, [x26, #0]\n"
      "fmax v6.8h, v6.8h, v11.8h\n"
      "fmax v5.8h, v5.8h, v11.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x180\n"
      "ldr q19, [x9, #0x10]\n"
      "str q8, [x25, #0]\n"
      "fmax v4.8h, v4.8h, v11.8h\n"
      "ldr q16, [x27, #0x10]\n"
      "str q7, [x24, #0]\n"
      "ldr q18, [x26, #0x10]\n"
      "str q6, [x23, #0]\n"
      "ldr q17, [x25, #0x10]\n"
      "str q5, [x22, #0]\n"
      "fadd v3.8h, v3.8h, v19.8h\n"
      "fadd v2.8h, v2.8h, v16.8h\n"
      "ldr q16, [x24, #0x10]\n"
      "str q4, [x21, #0]\n"
      "fadd v1.8h, v1.8h, v18.8h\n"
      "ldr q18, [x23, #0x10]\n"
      "fadd v0.8h, v0.8h, v17.8h\n"
      "ldr q17, [x22, #0x10]\n"
      "fadd v31.8h, v31.8h, v16.8h\n"
      "fmin v3.8h, v3.8h, v12.8h\n"
      "fmin v2.8h, v2.8h, v12.8h\n"
      "ldr q16, [x21, #0x10]\n"
      "fadd v30.8h, v30.8h, v18.8h\n"
      "fmin v1.8h, v1.8h, v12.8h\n"
      "fmin v0.8h, v0.8h, v12.8h\n"
      "fadd v29.8h, v29.8h, v17.8h\n"
      "fmin v31.8h, v31.8h, v12.8h\n"
      "fadd v28.8h, v28.8h, v16.8h\n"
      "fmin v30.8h, v30.8h, v12.8h\n"
      "fmax v3.8h, v3.8h, v11.8h\n"
      "fmax v2.8h, v2.8h, v11.8h\n"
      "fmin v29.8h, v29.8h, v12.8h\n"
      "fmax v1.8h, v1.8h, v11.8h\n"
      "fmin v28.8h, v28.8h, v12.8h\n"
      "fmax v0.8h, v0.8h, v11.8h\n"
      "str q3, [x9, #0x10]\n"
      "str q2, [x27, #0x10]\n"
      "fmax v31.8h, v31.8h, v11.8h\n"
      "fmax v30.8h, v30.8h, v11.8h\n"
      "str q1, [x26, #0x10]\n"
      "fmax v29.8h, v29.8h, v11.8h\n"
      "ldr q17, [x9, #0x20]\n"
      "str q0, [x25, #0x10]\n"
      "fmax v28.8h, v28.8h, v11.8h\n"
      "ldr q16, [x27, #0x20]\n"
      "str q31, [x24, #0x10]\n"
      "ldr q18, [x26, #0x20]\n"
      "str q30, [x23, #0x10]\n"
      "fadd v27.8h, v27.8h, v17.8h\n"
      "ldr q17, [x25, #0x20]\n"
      "str q29, [x22, #0x10]\n"
      "fadd v26.8h, v26.8h, v16.8h\n"
      "ldr q16, [x24, #0x20]\n"
      "str q28, [x21, #0x10]\n"
      "fadd v25.8h, v25.8h, v18.8h\n"
      "ldr q18, [x23, #0x20]\n"
      "fadd v24.8h, v24.8h, v17.8h\n"
      "fmin v27.8h, v27.8h, v12.8h\n"
      "ldr q17, [x22, #0x20]\n"
      "fadd v23.8h, v23.8h, v16.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "ldr q16, [x21, #0x20]\n"
      "fadd v22.8h, v22.8h, v18.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fadd v21.8h, v21.8h, v17.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fadd v20.8h, v20.8h, v16.8h\n"
      "fmin v22.8h, v22.8h, v12.8h\n"
      "fmax v27.8h, v27.8h, v11.8h\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "fmin v20.8h, v20.8h, v12.8h\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "str q27, [x9, #0x20]\n"
      "add x9, x9, #0x30\n"
      "str q26, [x27, #0x20]\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "fmax v22.8h, v22.8h, v11.8h\n"
      "add x27, x27, #0x30\n"
      "str q25, [x26, #0x20]\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "add x26, x26, #0x30\n"
      "str q24, [x25, #0x20]\n"
      "fmax v20.8h, v20.8h, v11.8h\n"
      "add x25, x25, #0x30\n"
      "str q23, [x24, #0x20]\n"
      "add x24, x24, #0x30\n"
      "str q22, [x23, #0x20]\n"
      "add x23, x23, #0x30\n"
      "str q21, [x22, #0x20]\n"
      "add x22, x22, #0x30\n"
      "str q20, [x21, #0x20]\n"
      "add x21, x21, #0x30\n"
      "bge 104b\n"
      "105:"  // Accumulate: Height 8: no full blocks
      "cbz x10, 107f\n"
      "mov x20, %x[in_ptr]\n"
      "106:"  // Accumulate: Height 8: Single loop
      "ldr h27, [%x[in_ptr], #0]\n"
      "ldr h19, [x9, #0]\n"
      "subs x10, x10, #0x1\n"
      "ldr h26, [%x[in_ptr], #0x30]\n"
      "ldr h18, [x27, #0]\n"
      "ldr h25, [%x[in_ptr], #0x60]\n"
      "ldr h17, [x26, #0]\n"
      "ldr h24, [%x[in_ptr], #0x90]\n"
      "ldr h16, [x25, #0]\n"
      "fadd v27.8h, v27.8h, v19.8h\n"
      "ldr h23, [%x[in_ptr], #0xc0]\n"
      "ldr h22, [x24, #0]\n"
      "fadd v26.8h, v26.8h, v18.8h\n"
      "ldr h21, [%x[in_ptr], #0xf0]\n"
      "ldr h20, [x23, #0]\n"
      "fadd v25.8h, v25.8h, v17.8h\n"
      "ldr h19, [%x[in_ptr], #0x120]\n"
      "ldr h18, [x22, #0]\n"
      "fadd v24.8h, v24.8h, v16.8h\n"
      "fmin v27.8h, v27.8h, v12.8h\n"
      "ldr h17, [%x[in_ptr], #0x150]\n"
      "ldr h16, [x21, #0]\n"
      "fadd v23.8h, v23.8h, v22.8h\n"
      "fmin v26.8h, v26.8h, v12.8h\n"
      "fadd v21.8h, v21.8h, v20.8h\n"
      "fmin v25.8h, v25.8h, v12.8h\n"
      "add %x[in_ptr], %x[in_ptr], #0x2\n"
      "fadd v19.8h, v19.8h, v18.8h\n"
      "fmin v24.8h, v24.8h, v12.8h\n"
      "fadd v17.8h, v17.8h, v16.8h\n"
      "fmin v23.8h, v23.8h, v12.8h\n"
      "fmin v21.8h, v21.8h, v12.8h\n"
      "fmax v27.8h, v27.8h, v11.8h\n"
      "fmin v19.8h, v19.8h, v12.8h\n"
      "fmax v26.8h, v26.8h, v11.8h\n"
      "fmin v17.8h, v17.8h, v12.8h\n"
      "fmax v25.8h, v25.8h, v11.8h\n"
      "str h27, [x9, #0]\n"
      "add x9, x9, #0x2\n"
      "fmax v24.8h, v24.8h, v11.8h\n"
      "fmax v23.8h, v23.8h, v11.8h\n"
      "str h26, [x27, #0]\n"
      "add x27, x27, #0x2\n"
      "fmax v21.8h, v21.8h, v11.8h\n"
      "fmax v19.8h, v19.8h, v11.8h\n"
      "str h25, [x26, #0]\n"
      "add x26, x26, #0x2\n"
      "fmax v17.8h, v17.8h, v11.8h\n"
      "str h24, [x25, #0]\n"
      "add x25, x25, #0x2\n"
      "str h23, [x24, #0]\n"
      "add x24, x24, #0x2\n"
      "str h21, [x23, #0]\n"
      "add x23, x23, #0x2\n"
      "str h19, [x22, #0]\n"
      "add x22, x22, #0x2\n"
      "str h17, [x21, #0]\n"
      "add x21, x21, #0x2\n"
      "bne 106b\n"
      "add %x[in_ptr], x20, #0x180\n"
      "107:"  // Accumulate: Height 8: no oddments
      "subs %x[rows], %x[rows], #0x8\n"
      "add %x[out_ptr], %x[out_ptr], x11\n"
      "bgt 67b\n"
      "108:"  // Exit
      : [in_ptr] "+&r" (in_ptr), [out_ptr] "+&r" (out_ptr), [rows] "+&r" (rows)
      : [accumulate] "r" (accumulate), [bias] "r" (bias), [cols] "r" (cols), [ldout] "r" (ldout), [maxval] "r" (maxval), [minval] "r" (minval)
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

#endif // (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(__aarch64__)

