/*
 * Copyright (c) 2021, 2023-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SVE) && defined(__aarch64__)

#include "arm_gemm/arm_gemm.hpp"
#include "arm_common/internal/utils.hpp"
#include <cassert>

namespace arm_gemm {

void sve_hybrid_u8u32_dot_6x4VL_a64fx (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<uint8_t> A_arg,
    size_t M, size_t N, const uint8_t *B_ptr, IndirectOutputArg<uint32_t> output_arg,
    const uint32_t *bias, Activation, bool accumulate
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const uint8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const uint32_t *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    __asm__ __volatile__(
      "ptrue p4.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 56f\n"
      "cmp %x[M], #0x4\n"
      "bgt 45f\n"
      "beq 34f\n"
      "cmp %x[M], #0x2\n"
      "bgt 23f\n"
      "beq 12f\n"
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "mov x20, #0\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "incw x20\n"
      "whilelt p0.s, x20, x12\n"
      "cbz x10, 3f\n"
      "ld1w { z8.s }, p4/Z, [x10]\n"
      "ld1w { z9.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p4/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "b 5f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 4f\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "b 5f\n"
      "4:"  // Height 1: no accumulate
      "mov z8.s, #0\n"
      "mov z9.s, #0\n"
      "mov z10.s, #0\n"
      "mov z11.s, #0\n"
      "5:"  // Height 1: setup done
      "mov x28, #0\n"
      "6:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 7f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "cbnz x28, 8f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "b 8f\n"
      "7:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "8:"  // Height 1: input setup done
      "subs x27, x27, #0x4\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "ble 10f\n"
      "9:"  // Height 1: Multiply loop: Main loop
      "udot z8.s, z6.b, z0.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "udot z9.s, z7.b, z0.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "addvl x11, x11, #4\n"
      "add x26, x26, #0x4\n"
      "subs x27, x27, #0x4\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z11.s, z7.b, z0.b\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "bgt 9b\n"
      "10:"  // Height 1: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "udot z8.s, z6.b, z0.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "udot z9.s, z7.b, z0.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "add x28, x28, #0x1\n"
      "addvl x11, x11, #4\n"
      "cmp x28, x20\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z11.s, z7.b, z0.b\n"
      "bne 6b\n"
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 2b\n"
      "b 68f\n"
      "12:"  // Height 2
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "13:"  // Height 2: Column loop
      "mov x20, #0\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "incw x20\n"
      "whilelt p0.s, x20, x12\n"
      "cbz x10, 14f\n"
      "ld1w { z8.s }, p4/Z, [x10]\n"
      "ld1w { z9.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p4/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "b 16f\n"
      "14:"  // Height 2: no bias
      "tbz %x[flags], #0, 15f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "add x20, x9, x20, LSL #2\n"
      "ld1w { z12.s }, p3/Z, [x20]\n"
      "ld1w { z13.s }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1w { z14.s }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1w { z15.s }, p0/Z, [x20, #3, MUL VL]\n"
      "b 16f\n"
      "15:"  // Height 2: no accumulate
      "mov z8.s, #0\n"
      "mov z9.s, #0\n"
      "mov z10.s, #0\n"
      "mov z11.s, #0\n"
      "mov z12.s, #0\n"
      "mov z13.s, #0\n"
      "mov z14.s, #0\n"
      "mov z15.s, #0\n"
      "16:"  // Height 2: setup done
      "mov x28, #0\n"
      "17:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 18f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 19f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "b 19f\n"
      "18:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "19:"  // Height 2: input setup done
      "subs x27, x27, #0x4\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "ble 21f\n"
      "20:"  // Height 2: Multiply loop: Main loop
      "udot z8.s, z6.b, z0.b\n"
      "udot z12.s, z6.b, z1.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "add x26, x26, #0x4\n"
      "udot z9.s, z7.b, z0.b\n"
      "udot z13.s, z7.b, z1.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "addvl x11, x11, #4\n"
      "subs x27, x27, #0x4\n"
      "add x25, x25, #0x4\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z14.s, z6.b, z1.b\n"
      "udot z11.s, z7.b, z0.b\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "udot z15.s, z7.b, z1.b\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "bgt 20b\n"
      "21:"  // Height 2: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "udot z8.s, z6.b, z0.b\n"
      "udot z12.s, z6.b, z1.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "udot z9.s, z7.b, z0.b\n"
      "udot z13.s, z7.b, z1.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "add x28, x28, #0x1\n"
      "addvl x11, x11, #4\n"
      "cmp x28, x20\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z14.s, z6.b, z1.b\n"
      "udot z11.s, z7.b, z0.b\n"
      "udot z15.s, z7.b, z1.b\n"
      "bne 17b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "add x20, x9, x20, LSL #2\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z12.s }, p3, [x20]\n"
      "st1w { z13.s }, p2, [x20, #1, MUL VL]\n"
      "st1w { z14.s }, p1, [x20, #2, MUL VL]\n"
      "st1w { z15.s }, p0, [x20, #3, MUL VL]\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 13b\n"
      "b 68f\n"
      "23:"  // Height 3
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "24:"  // Height 3: Column loop
      "mov x20, #0\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "incw x20\n"
      "whilelt p0.s, x20, x12\n"
      "cbz x10, 25f\n"
      "ld1w { z8.s }, p4/Z, [x10]\n"
      "ld1w { z9.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p4/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "b 27f\n"
      "25:"  // Height 3: no bias
      "tbz %x[flags], #0, 26f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "add x21, x9, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z12.s }, p3/Z, [x21]\n"
      "ld1w { z13.s }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1w { z14.s }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1w { z15.s }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1w { z16.s }, p3/Z, [x20]\n"
      "ld1w { z17.s }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1w { z18.s }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1w { z19.s }, p0/Z, [x20, #3, MUL VL]\n"
      "b 27f\n"
      "26:"  // Height 3: no accumulate
      "mov z8.s, #0\n"
      "mov z9.s, #0\n"
      "mov z10.s, #0\n"
      "mov z11.s, #0\n"
      "mov z12.s, #0\n"
      "mov z13.s, #0\n"
      "mov z14.s, #0\n"
      "mov z15.s, #0\n"
      "mov z16.s, #0\n"
      "mov z17.s, #0\n"
      "mov z18.s, #0\n"
      "mov z19.s, #0\n"
      "27:"  // Height 3: setup done
      "mov x28, #0\n"
      "28:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 29f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 30f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "b 30f\n"
      "29:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "30:"  // Height 3: input setup done
      "subs x27, x27, #0x4\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "ble 32f\n"
      "31:"  // Height 3: Multiply loop: Main loop
      "udot z8.s, z6.b, z0.b\n"
      "udot z12.s, z6.b, z1.b\n"
      "add x26, x26, #0x4\n"
      "subs x27, x27, #0x4\n"
      "udot z16.s, z6.b, z2.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "udot z9.s, z7.b, z0.b\n"
      "add x25, x25, #0x4\n"
      "udot z13.s, z7.b, z1.b\n"
      "udot z17.s, z7.b, z2.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "addvl x11, x11, #4\n"
      "add x24, x24, #0x4\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z14.s, z6.b, z1.b\n"
      "udot z18.s, z6.b, z2.b\n"
      "udot z11.s, z7.b, z0.b\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "udot z15.s, z7.b, z1.b\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "udot z19.s, z7.b, z2.b\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "bgt 31b\n"
      "32:"  // Height 3: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "udot z8.s, z6.b, z0.b\n"
      "udot z12.s, z6.b, z1.b\n"
      "add x28, x28, #0x1\n"
      "udot z16.s, z6.b, z2.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "udot z9.s, z7.b, z0.b\n"
      "udot z13.s, z7.b, z1.b\n"
      "udot z17.s, z7.b, z2.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "addvl x11, x11, #4\n"
      "cmp x28, x20\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z14.s, z6.b, z1.b\n"
      "udot z18.s, z6.b, z2.b\n"
      "udot z11.s, z7.b, z0.b\n"
      "udot z15.s, z7.b, z1.b\n"
      "udot z19.s, z7.b, z2.b\n"
      "bne 28b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "add x21, x9, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z12.s }, p3, [x21]\n"
      "st1w { z13.s }, p2, [x21, #1, MUL VL]\n"
      "st1w { z14.s }, p1, [x21, #2, MUL VL]\n"
      "st1w { z15.s }, p0, [x21, #3, MUL VL]\n"
      "st1w { z16.s }, p3, [x20]\n"
      "st1w { z17.s }, p2, [x20, #1, MUL VL]\n"
      "st1w { z18.s }, p1, [x20, #2, MUL VL]\n"
      "st1w { z19.s }, p0, [x20, #3, MUL VL]\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 24b\n"
      "b 68f\n"
      "34:"  // Height 4
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "35:"  // Height 4: Column loop
      "mov x20, #0\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "incw x20\n"
      "whilelt p0.s, x20, x12\n"
      "cbz x10, 36f\n"
      "ld1w { z8.s }, p4/Z, [x10]\n"
      "ld1w { z9.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p4/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "mov z20.d, z8.d\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "b 38f\n"
      "36:"  // Height 4: no bias
      "tbz %x[flags], #0, 37f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "add x22, x9, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z12.s }, p3/Z, [x22]\n"
      "ld1w { z13.s }, p2/Z, [x22, #1, MUL VL]\n"
      "ld1w { z14.s }, p1/Z, [x22, #2, MUL VL]\n"
      "ld1w { z15.s }, p0/Z, [x22, #3, MUL VL]\n"
      "ld1w { z16.s }, p3/Z, [x21]\n"
      "ld1w { z17.s }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1w { z18.s }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1w { z19.s }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1w { z20.s }, p3/Z, [x20]\n"
      "ld1w { z21.s }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1w { z22.s }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1w { z23.s }, p0/Z, [x20, #3, MUL VL]\n"
      "b 38f\n"
      "37:"  // Height 4: no accumulate
      "mov z8.s, #0\n"
      "mov z9.s, #0\n"
      "mov z10.s, #0\n"
      "mov z11.s, #0\n"
      "mov z12.s, #0\n"
      "mov z13.s, #0\n"
      "mov z14.s, #0\n"
      "mov z15.s, #0\n"
      "mov z16.s, #0\n"
      "mov z17.s, #0\n"
      "mov z18.s, #0\n"
      "mov z19.s, #0\n"
      "mov z20.s, #0\n"
      "mov z21.s, #0\n"
      "mov z22.s, #0\n"
      "mov z23.s, #0\n"
      "38:"  // Height 4: setup done
      "mov x28, #0\n"
      "39:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 40f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 41f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 41f\n"
      "40:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "41:"  // Height 4: input setup done
      "subs x27, x27, #0x4\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "ble 43f\n"
      "42:"  // Height 4: Multiply loop: Main loop
      "udot z8.s, z6.b, z0.b\n"
      "udot z12.s, z6.b, z1.b\n"
      "add x26, x26, #0x4\n"
      "subs x27, x27, #0x4\n"
      "udot z16.s, z6.b, z2.b\n"
      "udot z20.s, z6.b, z3.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "add x25, x25, #0x4\n"
      "udot z9.s, z7.b, z0.b\n"
      "udot z13.s, z7.b, z1.b\n"
      "add x24, x24, #0x4\n"
      "add x23, x23, #0x4\n"
      "udot z17.s, z7.b, z2.b\n"
      "udot z21.s, z7.b, z3.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "addvl x11, x11, #4\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z14.s, z6.b, z1.b\n"
      "udot z18.s, z6.b, z2.b\n"
      "udot z22.s, z6.b, z3.b\n"
      "udot z11.s, z7.b, z0.b\n"
      "udot z15.s, z7.b, z1.b\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "udot z19.s, z7.b, z2.b\n"
      "udot z23.s, z7.b, z3.b\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "bgt 42b\n"
      "43:"  // Height 4: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "udot z8.s, z6.b, z0.b\n"
      "udot z12.s, z6.b, z1.b\n"
      "add x28, x28, #0x1\n"
      "udot z16.s, z6.b, z2.b\n"
      "udot z20.s, z6.b, z3.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "udot z9.s, z7.b, z0.b\n"
      "udot z13.s, z7.b, z1.b\n"
      "udot z17.s, z7.b, z2.b\n"
      "udot z21.s, z7.b, z3.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "cmp x28, x20\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z14.s, z6.b, z1.b\n"
      "addvl x11, x11, #4\n"
      "udot z18.s, z6.b, z2.b\n"
      "udot z22.s, z6.b, z3.b\n"
      "udot z11.s, z7.b, z0.b\n"
      "udot z15.s, z7.b, z1.b\n"
      "udot z19.s, z7.b, z2.b\n"
      "udot z23.s, z7.b, z3.b\n"
      "bne 39b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "add x22, x9, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z12.s }, p3, [x22]\n"
      "st1w { z13.s }, p2, [x22, #1, MUL VL]\n"
      "st1w { z14.s }, p1, [x22, #2, MUL VL]\n"
      "st1w { z15.s }, p0, [x22, #3, MUL VL]\n"
      "st1w { z16.s }, p3, [x21]\n"
      "st1w { z17.s }, p2, [x21, #1, MUL VL]\n"
      "st1w { z18.s }, p1, [x21, #2, MUL VL]\n"
      "st1w { z19.s }, p0, [x21, #3, MUL VL]\n"
      "st1w { z20.s }, p3, [x20]\n"
      "st1w { z21.s }, p2, [x20, #1, MUL VL]\n"
      "st1w { z22.s }, p1, [x20, #2, MUL VL]\n"
      "st1w { z23.s }, p0, [x20, #3, MUL VL]\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 35b\n"
      "b 68f\n"
      "45:"  // Height 5
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "46:"  // Height 5: Column loop
      "mov x20, #0\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "incw x20\n"
      "whilelt p0.s, x20, x12\n"
      "cbz x10, 47f\n"
      "ld1w { z8.s }, p4/Z, [x10]\n"
      "ld1w { z9.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p4/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "mov z20.d, z8.d\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "mov z24.d, z8.d\n"
      "mov z25.d, z9.d\n"
      "mov z26.d, z10.d\n"
      "mov z27.d, z11.d\n"
      "b 49f\n"
      "47:"  // Height 5: no bias
      "tbz %x[flags], #0, 48f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "add x23, x9, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "ld1w { z12.s }, p3/Z, [x23]\n"
      "ld1w { z13.s }, p2/Z, [x23, #1, MUL VL]\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z14.s }, p1/Z, [x23, #2, MUL VL]\n"
      "ld1w { z15.s }, p0/Z, [x23, #3, MUL VL]\n"
      "ld1w { z16.s }, p3/Z, [x22]\n"
      "ld1w { z17.s }, p2/Z, [x22, #1, MUL VL]\n"
      "ld1w { z18.s }, p1/Z, [x22, #2, MUL VL]\n"
      "ld1w { z19.s }, p0/Z, [x22, #3, MUL VL]\n"
      "ld1w { z20.s }, p3/Z, [x21]\n"
      "ld1w { z21.s }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1w { z22.s }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1w { z23.s }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1w { z24.s }, p3/Z, [x20]\n"
      "ld1w { z25.s }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1w { z26.s }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1w { z27.s }, p0/Z, [x20, #3, MUL VL]\n"
      "b 49f\n"
      "48:"  // Height 5: no accumulate
      "mov z8.s, #0\n"
      "mov z9.s, #0\n"
      "mov z10.s, #0\n"
      "mov z11.s, #0\n"
      "mov z12.s, #0\n"
      "mov z13.s, #0\n"
      "mov z14.s, #0\n"
      "mov z15.s, #0\n"
      "mov z16.s, #0\n"
      "mov z17.s, #0\n"
      "mov z18.s, #0\n"
      "mov z19.s, #0\n"
      "mov z20.s, #0\n"
      "mov z21.s, #0\n"
      "mov z22.s, #0\n"
      "mov z23.s, #0\n"
      "mov z24.s, #0\n"
      "mov z25.s, #0\n"
      "mov z26.s, #0\n"
      "mov z27.s, #0\n"
      "49:"  // Height 5: setup done
      "mov x28, #0\n"
      "50:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 51f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 52f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 52f\n"
      "51:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "52:"  // Height 5: input setup done
      "subs x27, x27, #0x4\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "ld1rw { z4.s }, p4/Z, [x22]\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "ble 54f\n"
      "53:"  // Height 5: Multiply loop: Main loop
      "udot z8.s, z6.b, z0.b\n"
      "udot z12.s, z6.b, z1.b\n"
      "add x26, x26, #0x4\n"
      "subs x27, x27, #0x4\n"
      "udot z16.s, z6.b, z2.b\n"
      "udot z20.s, z6.b, z3.b\n"
      "add x25, x25, #0x4\n"
      "add x24, x24, #0x4\n"
      "udot z24.s, z6.b, z4.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "udot z9.s, z7.b, z0.b\n"
      "add x23, x23, #0x4\n"
      "udot z13.s, z7.b, z1.b\n"
      "udot z17.s, z7.b, z2.b\n"
      "add x22, x22, #0x4\n"
      "udot z21.s, z7.b, z3.b\n"
      "udot z25.s, z7.b, z4.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "addvl x11, x11, #4\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z14.s, z6.b, z1.b\n"
      "udot z18.s, z6.b, z2.b\n"
      "udot z22.s, z6.b, z3.b\n"
      "udot z26.s, z6.b, z4.b\n"
      "udot z11.s, z7.b, z0.b\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "udot z15.s, z7.b, z1.b\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "udot z19.s, z7.b, z2.b\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "udot z23.s, z7.b, z3.b\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "udot z27.s, z7.b, z4.b\n"
      "ld1rw { z4.s }, p4/Z, [x22]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "bgt 53b\n"
      "54:"  // Height 5: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "udot z8.s, z6.b, z0.b\n"
      "udot z12.s, z6.b, z1.b\n"
      "add x28, x28, #0x1\n"
      "udot z16.s, z6.b, z2.b\n"
      "udot z20.s, z6.b, z3.b\n"
      "udot z24.s, z6.b, z4.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "udot z9.s, z7.b, z0.b\n"
      "udot z13.s, z7.b, z1.b\n"
      "udot z17.s, z7.b, z2.b\n"
      "cmp x28, x20\n"
      "udot z21.s, z7.b, z3.b\n"
      "udot z25.s, z7.b, z4.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "addvl x11, x11, #4\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z14.s, z6.b, z1.b\n"
      "udot z18.s, z6.b, z2.b\n"
      "udot z22.s, z6.b, z3.b\n"
      "udot z26.s, z6.b, z4.b\n"
      "udot z11.s, z7.b, z0.b\n"
      "udot z15.s, z7.b, z1.b\n"
      "udot z19.s, z7.b, z2.b\n"
      "udot z23.s, z7.b, z3.b\n"
      "udot z27.s, z7.b, z4.b\n"
      "bne 50b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "add x23, x9, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "add x20, x21, x20, LSL #2\n"
      "st1w { z12.s }, p3, [x23]\n"
      "st1w { z13.s }, p2, [x23, #1, MUL VL]\n"
      "st1w { z14.s }, p1, [x23, #2, MUL VL]\n"
      "st1w { z15.s }, p0, [x23, #3, MUL VL]\n"
      "st1w { z16.s }, p3, [x22]\n"
      "st1w { z17.s }, p2, [x22, #1, MUL VL]\n"
      "st1w { z18.s }, p1, [x22, #2, MUL VL]\n"
      "st1w { z19.s }, p0, [x22, #3, MUL VL]\n"
      "st1w { z20.s }, p3, [x21]\n"
      "st1w { z21.s }, p2, [x21, #1, MUL VL]\n"
      "st1w { z22.s }, p1, [x21, #2, MUL VL]\n"
      "st1w { z23.s }, p0, [x21, #3, MUL VL]\n"
      "st1w { z24.s }, p3, [x20]\n"
      "st1w { z25.s }, p2, [x20, #1, MUL VL]\n"
      "st1w { z26.s }, p1, [x20, #2, MUL VL]\n"
      "st1w { z27.s }, p0, [x20, #3, MUL VL]\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 46b\n"
      "b 68f\n"
      "56:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x18\n"
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "madd x20, x21, x20, x9\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "57:"  // Height 6: Column loop
      "mov x20, #0\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "incw x20\n"
      "whilelt p0.s, x20, x12\n"
      "cbz x10, 58f\n"
      "ld1w { z8.s }, p4/Z, [x10]\n"
      "ld1w { z9.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p4/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "mov z20.d, z8.d\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "mov z24.d, z8.d\n"
      "mov z25.d, z9.d\n"
      "mov z26.d, z10.d\n"
      "mov z27.d, z11.d\n"
      "mov z28.d, z8.d\n"
      "mov z29.d, z9.d\n"
      "mov z30.d, z10.d\n"
      "mov z31.d, z11.d\n"
      "b 60f\n"
      "58:"  // Height 6: no bias
      "tbz %x[flags], #0, 59f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "ld1w { z12.s }, p3/Z, [x24]\n"
      "ld1w { z13.s }, p2/Z, [x24, #1, MUL VL]\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z14.s }, p1/Z, [x24, #2, MUL VL]\n"
      "ld1w { z15.s }, p0/Z, [x24, #3, MUL VL]\n"
      "ld1w { z16.s }, p3/Z, [x23]\n"
      "ld1w { z17.s }, p2/Z, [x23, #1, MUL VL]\n"
      "ld1w { z18.s }, p1/Z, [x23, #2, MUL VL]\n"
      "ld1w { z19.s }, p0/Z, [x23, #3, MUL VL]\n"
      "ld1w { z20.s }, p3/Z, [x22]\n"
      "ld1w { z21.s }, p2/Z, [x22, #1, MUL VL]\n"
      "ld1w { z22.s }, p1/Z, [x22, #2, MUL VL]\n"
      "ld1w { z23.s }, p0/Z, [x22, #3, MUL VL]\n"
      "ld1w { z24.s }, p3/Z, [x21]\n"
      "ld1w { z25.s }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1w { z26.s }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1w { z27.s }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1w { z28.s }, p3/Z, [x20]\n"
      "ld1w { z29.s }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1w { z30.s }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1w { z31.s }, p0/Z, [x20, #3, MUL VL]\n"
      "b 60f\n"
      "59:"  // Height 6: no accumulate
      "mov z8.s, #0\n"
      "mov z9.s, #0\n"
      "mov z10.s, #0\n"
      "mov z11.s, #0\n"
      "mov z12.s, #0\n"
      "mov z13.s, #0\n"
      "mov z14.s, #0\n"
      "mov z15.s, #0\n"
      "mov z16.s, #0\n"
      "mov z17.s, #0\n"
      "mov z18.s, #0\n"
      "mov z19.s, #0\n"
      "mov z20.s, #0\n"
      "mov z21.s, #0\n"
      "mov z22.s, #0\n"
      "mov z23.s, #0\n"
      "mov z24.s, #0\n"
      "mov z25.s, #0\n"
      "mov z26.s, #0\n"
      "mov z27.s, #0\n"
      "mov z28.s, #0\n"
      "mov z29.s, #0\n"
      "mov z30.s, #0\n"
      "mov z31.s, #0\n"
      "60:"  // Height 6: setup done
      "mov x28, #0\n"
      "61:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 62f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 63f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 63f\n"
      "62:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "63:"  // Height 6: input setup done
      "subs x27, x27, #0x4\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "ld1rw { z4.s }, p4/Z, [x22]\n"
      "ld1rw { z5.s }, p4/Z, [x21]\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "ble 65f\n"
      "64:"  // Height 6: Multiply loop: Main loop
      "udot z8.s, z6.b, z0.b\n"
      "udot z12.s, z6.b, z1.b\n"
      "add x26, x26, #0x4\n"
      "subs x27, x27, #0x4\n"
      "udot z16.s, z6.b, z2.b\n"
      "udot z20.s, z6.b, z3.b\n"
      "add x25, x25, #0x4\n"
      "add x24, x24, #0x4\n"
      "udot z24.s, z6.b, z4.b\n"
      "udot z28.s, z6.b, z5.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "add x23, x23, #0x4\n"
      "udot z9.s, z7.b, z0.b\n"
      "udot z13.s, z7.b, z1.b\n"
      "add x22, x22, #0x4\n"
      "add x21, x21, #0x4\n"
      "udot z17.s, z7.b, z2.b\n"
      "udot z21.s, z7.b, z3.b\n"
      "udot z25.s, z7.b, z4.b\n"
      "udot z29.s, z7.b, z5.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "addvl x11, x11, #4\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z14.s, z6.b, z1.b\n"
      "udot z18.s, z6.b, z2.b\n"
      "udot z22.s, z6.b, z3.b\n"
      "udot z26.s, z6.b, z4.b\n"
      "udot z30.s, z6.b, z5.b\n"
      "ld1b { z6.b }, p4/Z, [x11]\n"
      "udot z11.s, z7.b, z0.b\n"
      "udot z15.s, z7.b, z1.b\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "udot z19.s, z7.b, z2.b\n"
      "udot z23.s, z7.b, z3.b\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "udot z27.s, z7.b, z4.b\n"
      "udot z31.s, z7.b, z5.b\n"
      "ld1rw { z4.s }, p4/Z, [x22]\n"
      "ld1rw { z5.s }, p4/Z, [x21]\n"
      "ld1b { z7.b }, p4/Z, [x11, #1, MUL VL]\n"
      "bgt 64b\n"
      "65:"  // Height 6: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "udot z8.s, z6.b, z0.b\n"
      "udot z12.s, z6.b, z1.b\n"
      "add x28, x28, #0x1\n"
      "udot z16.s, z6.b, z2.b\n"
      "udot z20.s, z6.b, z3.b\n"
      "udot z24.s, z6.b, z4.b\n"
      "udot z28.s, z6.b, z5.b\n"
      "ld1b { z6.b }, p4/Z, [x11, #2, MUL VL]\n"
      "udot z9.s, z7.b, z0.b\n"
      "udot z13.s, z7.b, z1.b\n"
      "cmp x28, x20\n"
      "udot z17.s, z7.b, z2.b\n"
      "udot z21.s, z7.b, z3.b\n"
      "udot z25.s, z7.b, z4.b\n"
      "udot z29.s, z7.b, z5.b\n"
      "ld1b { z7.b }, p4/Z, [x11, #3, MUL VL]\n"
      "addvl x11, x11, #4\n"
      "udot z10.s, z6.b, z0.b\n"
      "udot z14.s, z6.b, z1.b\n"
      "udot z18.s, z6.b, z2.b\n"
      "udot z22.s, z6.b, z3.b\n"
      "udot z26.s, z6.b, z4.b\n"
      "udot z30.s, z6.b, z5.b\n"
      "udot z11.s, z7.b, z0.b\n"
      "udot z15.s, z7.b, z1.b\n"
      "udot z19.s, z7.b, z2.b\n"
      "udot z23.s, z7.b, z3.b\n"
      "udot z27.s, z7.b, z4.b\n"
      "udot z31.s, z7.b, z5.b\n"
      "bne 61b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "add x21, x22, x20, LSL #2\n"
      "st1w { z12.s }, p3, [x24]\n"
      "add x20, x21, x20, LSL #2\n"
      "st1w { z13.s }, p2, [x24, #1, MUL VL]\n"
      "st1w { z14.s }, p1, [x24, #2, MUL VL]\n"
      "st1w { z15.s }, p0, [x24, #3, MUL VL]\n"
      "st1w { z16.s }, p3, [x23]\n"
      "st1w { z17.s }, p2, [x23, #1, MUL VL]\n"
      "st1w { z18.s }, p1, [x23, #2, MUL VL]\n"
      "st1w { z19.s }, p0, [x23, #3, MUL VL]\n"
      "st1w { z20.s }, p3, [x22]\n"
      "st1w { z21.s }, p2, [x22, #1, MUL VL]\n"
      "st1w { z22.s }, p1, [x22, #2, MUL VL]\n"
      "st1w { z23.s }, p0, [x22, #3, MUL VL]\n"
      "st1w { z24.s }, p3, [x21]\n"
      "st1w { z25.s }, p2, [x21, #1, MUL VL]\n"
      "st1w { z26.s }, p1, [x21, #2, MUL VL]\n"
      "st1w { z27.s }, p0, [x21, #3, MUL VL]\n"
      "st1w { z28.s }, p3, [x20]\n"
      "st1w { z29.s }, p2, [x20, #1, MUL VL]\n"
      "st1w { z30.s }, p1, [x20, #2, MUL VL]\n"
      "st1w { z31.s }, p0, [x20, #3, MUL VL]\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 57b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 68f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 67f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "67:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "68:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "x9", "x10", "x11", "x12", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm

#endif // defined(ARM_COMPUTE_ENABLE_SVE) && defined(__aarch64__)

