/*
 * Copyright (c) 2021, 2023-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SVE) && defined(__aarch64__)

#include "arm_gemm/arm_gemm.hpp"
#include "arm_common/internal/utils.hpp"
#include <cassert>
#include <limits>

namespace arm_gemm {

void sve_hybrid_fp32_mla_8x1VL_a64fx (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<float> A_arg,
    size_t M, size_t N, const float *B_ptr, IndirectOutputArg<float> output_arg,
    const float *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const float *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const float *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "ptrue p1.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x8\n"
      "bge 85f\n"
      "cmp %x[M], #0x6\n"
      "bgt 73f\n"
      "beq 61f\n"
      "cmp %x[M], #0x4\n"
      "bgt 49f\n"
      "beq 37f\n"
      "cmp %x[M], #0x2\n"
      "bgt 25f\n"
      "beq 13f\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "mov x20, #0\n"
      "whilelt p0.s, x20, x14\n"
      "cbz x12, 3f\n"
      "ld1w { z24.s }, p1/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "b 5f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 4f\n"
      "ld1w { z24.s }, p0/Z, [x11]\n"
      "b 5f\n"
      "4:"  // Height 1: no accumulate
      "mov z24.b, #0\n"
      "5:"  // Height 1: setup done
      "mov x10, #0\n"
      "6:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 7f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0]\n"
      "cbnz x10, 8f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "b 8f\n"
      "7:"  // Height 1: setup direct input
      "mov x28, %x[input_ptr]\n"
      "8:"  // Height 1: input setup done
      "subs x9, x9, #0x1\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "ble 10f\n"
      "9:"  // Height 1: Multiply loop: Main loop
      "ld1w { z8.s }, p1/Z, [x13]\n"
      "add x28, x28, #0x4\n"
      "subs x9, x9, #0x1\n"
      "addvl x13, x13, #1\n"
      "fmla z24.s, p1/M, z8.s, z0.s\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "bgt 9b\n"
      "10:"  // Height 1: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "ld1w { z9.s }, p1/Z, [x13]\n"
      "add x10, x10, #0x1\n"
      "addvl x13, x13, #1\n"
      "cmp x10, x20\n"
      "fmla z24.s, p1/M, z9.s, z0.s\n"
      "bne 6b\n"
      "tbz %x[flags], #1, 11f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z17.s }, p1/Z, [x21]\n"
      "ld1rw { z16.s }, p1/Z, [x20]\n"
      "fmin z24.s, p1/M, z24.s, z17.s\n"
      "fmax z24.s, p1/M, z24.s, z16.s\n"
      "11:"  // Height 1: No activation
      "st1w { z24.s }, p0, [x11]\n"
      "addvl x11, x11, #1\n"
      "decw x14\n"
      "cmp x14, XZR\n"
      "bgt 2b\n"
      "b 98f\n"
      "13:"  // Height 2
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "14:"  // Height 2: Column loop
      "mov x20, #0\n"
      "whilelt p0.s, x20, x14\n"
      "cbz x12, 15f\n"
      "ld1w { z24.s }, p1/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "mov z25.d, z24.d\n"
      "b 17f\n"
      "15:"  // Height 2: no bias
      "tbz %x[flags], #0, 16f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p0/Z, [x11]\n"
      "add x20, x11, x20, LSL #2\n"
      "ld1w { z25.s }, p0/Z, [x20]\n"
      "b 17f\n"
      "16:"  // Height 2: no accumulate
      "mov z24.b, #0\n"
      "mov z25.b, #0\n"
      "17:"  // Height 2: setup done
      "mov x10, #0\n"
      "18:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 19f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0]\n"
      "ldr x27, [x20, #0x8]\n"
      "cbnz x10, 20f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "b 20f\n"
      "19:"  // Height 2: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "20:"  // Height 2: input setup done
      "subs x9, x9, #0x1\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "ble 22f\n"
      "21:"  // Height 2: Multiply loop: Main loop
      "ld1w { z8.s }, p1/Z, [x13]\n"
      "add x28, x28, #0x4\n"
      "subs x9, x9, #0x1\n"
      "add x27, x27, #0x4\n"
      "addvl x13, x13, #1\n"
      "fmla z24.s, p1/M, z8.s, z0.s\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "fmla z25.s, p1/M, z8.s, z1.s\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "bgt 21b\n"
      "22:"  // Height 2: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "ld1w { z9.s }, p1/Z, [x13]\n"
      "add x10, x10, #0x1\n"
      "addvl x13, x13, #1\n"
      "cmp x10, x20\n"
      "fmla z24.s, p1/M, z9.s, z0.s\n"
      "fmla z25.s, p1/M, z9.s, z1.s\n"
      "bne 18b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "tbz %x[flags], #1, 23f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z17.s }, p1/Z, [x21]\n"
      "ld1rw { z16.s }, p1/Z, [x20]\n"
      "fmin z24.s, p1/M, z24.s, z17.s\n"
      "fmin z25.s, p1/M, z25.s, z17.s\n"
      "fmax z24.s, p1/M, z24.s, z16.s\n"
      "fmax z25.s, p1/M, z25.s, z16.s\n"
      "23:"  // Height 2: No activation
      "st1w { z24.s }, p0, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p0, [x28]\n"
      "decw x14\n"
      "cmp x14, XZR\n"
      "bgt 14b\n"
      "b 98f\n"
      "25:"  // Height 3
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "26:"  // Height 3: Column loop
      "mov x20, #0\n"
      "whilelt p0.s, x20, x14\n"
      "cbz x12, 27f\n"
      "ld1w { z24.s }, p1/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "b 29f\n"
      "27:"  // Height 3: no bias
      "tbz %x[flags], #0, 28f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p0/Z, [x11]\n"
      "add x20, x11, x21, LSL #2\n"
      "ld1w { z25.s }, p0/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z26.s }, p0/Z, [x20]\n"
      "b 29f\n"
      "28:"  // Height 3: no accumulate
      "mov z24.b, #0\n"
      "mov z25.b, #0\n"
      "mov z26.b, #0\n"
      "29:"  // Height 3: setup done
      "mov x10, #0\n"
      "30:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 31f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "cbnz x10, 32f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "b 32f\n"
      "31:"  // Height 3: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "32:"  // Height 3: input setup done
      "subs x9, x9, #0x1\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "ble 34f\n"
      "33:"  // Height 3: Multiply loop: Main loop
      "ld1w { z8.s }, p1/Z, [x13]\n"
      "add x28, x28, #0x4\n"
      "subs x9, x9, #0x1\n"
      "add x27, x27, #0x4\n"
      "add x26, x26, #0x4\n"
      "addvl x13, x13, #1\n"
      "fmla z24.s, p1/M, z8.s, z0.s\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "fmla z25.s, p1/M, z8.s, z1.s\n"
      "fmla z26.s, p1/M, z8.s, z2.s\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "bgt 33b\n"
      "34:"  // Height 3: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "ld1w { z9.s }, p1/Z, [x13]\n"
      "add x10, x10, #0x1\n"
      "addvl x13, x13, #1\n"
      "cmp x10, x20\n"
      "fmla z24.s, p1/M, z9.s, z0.s\n"
      "fmla z25.s, p1/M, z9.s, z1.s\n"
      "fmla z26.s, p1/M, z9.s, z2.s\n"
      "bne 30b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "tbz %x[flags], #1, 35f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z17.s }, p1/Z, [x21]\n"
      "ld1rw { z16.s }, p1/Z, [x20]\n"
      "fmin z24.s, p1/M, z24.s, z17.s\n"
      "fmin z25.s, p1/M, z25.s, z17.s\n"
      "fmin z26.s, p1/M, z26.s, z17.s\n"
      "fmax z24.s, p1/M, z24.s, z16.s\n"
      "fmax z25.s, p1/M, z25.s, z16.s\n"
      "fmax z26.s, p1/M, z26.s, z16.s\n"
      "35:"  // Height 3: No activation
      "st1w { z24.s }, p0, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p0, [x28]\n"
      "st1w { z26.s }, p0, [x27]\n"
      "decw x14\n"
      "cmp x14, XZR\n"
      "bgt 26b\n"
      "b 98f\n"
      "37:"  // Height 4
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "38:"  // Height 4: Column loop
      "mov x20, #0\n"
      "whilelt p0.s, x20, x14\n"
      "cbz x12, 39f\n"
      "ld1w { z24.s }, p1/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "mov z27.d, z24.d\n"
      "b 41f\n"
      "39:"  // Height 4: no bias
      "tbz %x[flags], #0, 40f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p0/Z, [x11]\n"
      "add x20, x11, x21, LSL #2\n"
      "ld1w { z25.s }, p0/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z26.s }, p0/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z27.s }, p0/Z, [x20]\n"
      "b 41f\n"
      "40:"  // Height 4: no accumulate
      "mov z24.b, #0\n"
      "mov z25.b, #0\n"
      "mov z26.b, #0\n"
      "mov z27.b, #0\n"
      "41:"  // Height 4: setup done
      "mov x10, #0\n"
      "42:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 43f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "cbnz x10, 44f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "b 44f\n"
      "43:"  // Height 4: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "44:"  // Height 4: input setup done
      "subs x9, x9, #0x1\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "ld1rw { z3.s }, p1/Z, [x25]\n"
      "ble 46f\n"
      "45:"  // Height 4: Multiply loop: Main loop
      "ld1w { z8.s }, p1/Z, [x13]\n"
      "add x28, x28, #0x4\n"
      "subs x9, x9, #0x1\n"
      "add x27, x27, #0x4\n"
      "add x26, x26, #0x4\n"
      "add x25, x25, #0x4\n"
      "addvl x13, x13, #1\n"
      "fmla z24.s, p1/M, z8.s, z0.s\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "fmla z25.s, p1/M, z8.s, z1.s\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "fmla z26.s, p1/M, z8.s, z2.s\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "fmla z27.s, p1/M, z8.s, z3.s\n"
      "ld1rw { z3.s }, p1/Z, [x25]\n"
      "bgt 45b\n"
      "46:"  // Height 4: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "ld1w { z9.s }, p1/Z, [x13]\n"
      "add x10, x10, #0x1\n"
      "addvl x13, x13, #1\n"
      "cmp x10, x20\n"
      "fmla z24.s, p1/M, z9.s, z0.s\n"
      "fmla z25.s, p1/M, z9.s, z1.s\n"
      "fmla z26.s, p1/M, z9.s, z2.s\n"
      "fmla z27.s, p1/M, z9.s, z3.s\n"
      "bne 42b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "tbz %x[flags], #1, 47f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z17.s }, p1/Z, [x21]\n"
      "ld1rw { z16.s }, p1/Z, [x20]\n"
      "fmin z24.s, p1/M, z24.s, z17.s\n"
      "fmin z25.s, p1/M, z25.s, z17.s\n"
      "fmin z26.s, p1/M, z26.s, z17.s\n"
      "fmin z27.s, p1/M, z27.s, z17.s\n"
      "fmax z24.s, p1/M, z24.s, z16.s\n"
      "fmax z25.s, p1/M, z25.s, z16.s\n"
      "fmax z26.s, p1/M, z26.s, z16.s\n"
      "fmax z27.s, p1/M, z27.s, z16.s\n"
      "47:"  // Height 4: No activation
      "st1w { z24.s }, p0, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p0, [x28]\n"
      "st1w { z26.s }, p0, [x27]\n"
      "st1w { z27.s }, p0, [x26]\n"
      "decw x14\n"
      "cmp x14, XZR\n"
      "bgt 38b\n"
      "b 98f\n"
      "49:"  // Height 5
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "50:"  // Height 5: Column loop
      "mov x20, #0\n"
      "whilelt p0.s, x20, x14\n"
      "cbz x12, 51f\n"
      "ld1w { z24.s }, p1/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "mov z27.d, z24.d\n"
      "mov z28.d, z24.d\n"
      "b 53f\n"
      "51:"  // Height 5: no bias
      "tbz %x[flags], #0, 52f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p0/Z, [x11]\n"
      "add x20, x11, x21, LSL #2\n"
      "ld1w { z25.s }, p0/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z26.s }, p0/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z27.s }, p0/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z28.s }, p0/Z, [x20]\n"
      "b 53f\n"
      "52:"  // Height 5: no accumulate
      "mov z24.b, #0\n"
      "mov z25.b, #0\n"
      "mov z26.b, #0\n"
      "mov z27.b, #0\n"
      "mov z28.b, #0\n"
      "53:"  // Height 5: setup done
      "mov x10, #0\n"
      "54:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 55f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "cbnz x10, 56f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "b 56f\n"
      "55:"  // Height 5: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "56:"  // Height 5: input setup done
      "subs x9, x9, #0x1\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "ld1rw { z3.s }, p1/Z, [x25]\n"
      "ld1rw { z4.s }, p1/Z, [x24]\n"
      "ble 58f\n"
      "57:"  // Height 5: Multiply loop: Main loop
      "ld1w { z8.s }, p1/Z, [x13]\n"
      "add x28, x28, #0x4\n"
      "subs x9, x9, #0x1\n"
      "add x27, x27, #0x4\n"
      "add x26, x26, #0x4\n"
      "add x25, x25, #0x4\n"
      "add x24, x24, #0x4\n"
      "addvl x13, x13, #1\n"
      "fmla z24.s, p1/M, z8.s, z0.s\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "fmla z25.s, p1/M, z8.s, z1.s\n"
      "fmla z26.s, p1/M, z8.s, z2.s\n"
      "fmla z27.s, p1/M, z8.s, z3.s\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "fmla z28.s, p1/M, z8.s, z4.s\n"
      "ld1rw { z3.s }, p1/Z, [x25]\n"
      "ld1rw { z4.s }, p1/Z, [x24]\n"
      "bgt 57b\n"
      "58:"  // Height 5: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "ld1w { z9.s }, p1/Z, [x13]\n"
      "add x10, x10, #0x1\n"
      "addvl x13, x13, #1\n"
      "cmp x10, x20\n"
      "fmla z24.s, p1/M, z9.s, z0.s\n"
      "fmla z25.s, p1/M, z9.s, z1.s\n"
      "fmla z26.s, p1/M, z9.s, z2.s\n"
      "fmla z27.s, p1/M, z9.s, z3.s\n"
      "fmla z28.s, p1/M, z9.s, z4.s\n"
      "bne 54b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "tbz %x[flags], #1, 59f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z17.s }, p1/Z, [x21]\n"
      "ld1rw { z16.s }, p1/Z, [x20]\n"
      "fmin z24.s, p1/M, z24.s, z17.s\n"
      "fmin z25.s, p1/M, z25.s, z17.s\n"
      "fmin z26.s, p1/M, z26.s, z17.s\n"
      "fmin z27.s, p1/M, z27.s, z17.s\n"
      "fmin z28.s, p1/M, z28.s, z17.s\n"
      "fmax z24.s, p1/M, z24.s, z16.s\n"
      "fmax z25.s, p1/M, z25.s, z16.s\n"
      "fmax z26.s, p1/M, z26.s, z16.s\n"
      "fmax z27.s, p1/M, z27.s, z16.s\n"
      "fmax z28.s, p1/M, z28.s, z16.s\n"
      "59:"  // Height 5: No activation
      "st1w { z24.s }, p0, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p0, [x28]\n"
      "st1w { z26.s }, p0, [x27]\n"
      "st1w { z27.s }, p0, [x26]\n"
      "st1w { z28.s }, p0, [x25]\n"
      "decw x14\n"
      "cmp x14, XZR\n"
      "bgt 50b\n"
      "b 98f\n"
      "61:"  // Height 6
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "62:"  // Height 6: Column loop
      "mov x20, #0\n"
      "whilelt p0.s, x20, x14\n"
      "cbz x12, 63f\n"
      "ld1w { z24.s }, p1/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "mov z27.d, z24.d\n"
      "mov z28.d, z24.d\n"
      "mov z29.d, z24.d\n"
      "b 65f\n"
      "63:"  // Height 6: no bias
      "tbz %x[flags], #0, 64f\n"
      "ldr x22, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p0/Z, [x11]\n"
      "add x20, x11, x22, LSL #2\n"
      "ld1w { z25.s }, p0/Z, [x20]\n"
      "add x20, x20, x22, LSL #2\n"
      "ld1w { z26.s }, p0/Z, [x20]\n"
      "add x20, x20, x22, LSL #2\n"
      "ld1w { z27.s }, p0/Z, [x20]\n"
      "add x21, x20, x22, LSL #2\n"
      "add x20, x21, x22, LSL #2\n"
      "ld1w { z28.s }, p0/Z, [x21]\n"
      "ld1w { z29.s }, p0/Z, [x20]\n"
      "b 65f\n"
      "64:"  // Height 6: no accumulate
      "mov z24.b, #0\n"
      "mov z25.b, #0\n"
      "mov z26.b, #0\n"
      "mov z27.b, #0\n"
      "mov z28.b, #0\n"
      "mov z29.b, #0\n"
      "65:"  // Height 6: setup done
      "mov x10, #0\n"
      "66:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 67f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "ldr x23, [x20, #0x28]\n"
      "cbnz x10, 68f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "b 68f\n"
      "67:"  // Height 6: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "68:"  // Height 6: input setup done
      "subs x9, x9, #0x1\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "ld1rw { z3.s }, p1/Z, [x25]\n"
      "ld1rw { z4.s }, p1/Z, [x24]\n"
      "ld1rw { z5.s }, p1/Z, [x23]\n"
      "ble 70f\n"
      "69:"  // Height 6: Multiply loop: Main loop
      "ld1w { z8.s }, p1/Z, [x13]\n"
      "add x28, x28, #0x4\n"
      "subs x9, x9, #0x1\n"
      "add x27, x27, #0x4\n"
      "add x26, x26, #0x4\n"
      "add x25, x25, #0x4\n"
      "add x24, x24, #0x4\n"
      "add x23, x23, #0x4\n"
      "addvl x13, x13, #1\n"
      "fmla z24.s, p1/M, z8.s, z0.s\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "fmla z25.s, p1/M, z8.s, z1.s\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "fmla z26.s, p1/M, z8.s, z2.s\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "fmla z27.s, p1/M, z8.s, z3.s\n"
      "fmla z28.s, p1/M, z8.s, z4.s\n"
      "ld1rw { z3.s }, p1/Z, [x25]\n"
      "ld1rw { z4.s }, p1/Z, [x24]\n"
      "fmla z29.s, p1/M, z8.s, z5.s\n"
      "ld1rw { z5.s }, p1/Z, [x23]\n"
      "bgt 69b\n"
      "70:"  // Height 6: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "ld1w { z9.s }, p1/Z, [x13]\n"
      "add x10, x10, #0x1\n"
      "addvl x13, x13, #1\n"
      "cmp x10, x20\n"
      "fmla z24.s, p1/M, z9.s, z0.s\n"
      "fmla z25.s, p1/M, z9.s, z1.s\n"
      "fmla z26.s, p1/M, z9.s, z2.s\n"
      "fmla z27.s, p1/M, z9.s, z3.s\n"
      "fmla z28.s, p1/M, z9.s, z4.s\n"
      "fmla z29.s, p1/M, z9.s, z5.s\n"
      "bne 66b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "tbz %x[flags], #1, 71f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z17.s }, p1/Z, [x21]\n"
      "ld1rw { z16.s }, p1/Z, [x20]\n"
      "fmin z24.s, p1/M, z24.s, z17.s\n"
      "fmin z25.s, p1/M, z25.s, z17.s\n"
      "fmin z26.s, p1/M, z26.s, z17.s\n"
      "fmin z27.s, p1/M, z27.s, z17.s\n"
      "fmin z28.s, p1/M, z28.s, z17.s\n"
      "fmin z29.s, p1/M, z29.s, z17.s\n"
      "fmax z24.s, p1/M, z24.s, z16.s\n"
      "fmax z25.s, p1/M, z25.s, z16.s\n"
      "fmax z26.s, p1/M, z26.s, z16.s\n"
      "fmax z27.s, p1/M, z27.s, z16.s\n"
      "fmax z28.s, p1/M, z28.s, z16.s\n"
      "fmax z29.s, p1/M, z29.s, z16.s\n"
      "71:"  // Height 6: No activation
      "st1w { z24.s }, p0, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p0, [x28]\n"
      "st1w { z26.s }, p0, [x27]\n"
      "st1w { z27.s }, p0, [x26]\n"
      "st1w { z28.s }, p0, [x25]\n"
      "st1w { z29.s }, p0, [x24]\n"
      "decw x14\n"
      "cmp x14, XZR\n"
      "bgt 62b\n"
      "b 98f\n"
      "73:"  // Height 7
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "74:"  // Height 7: Column loop
      "mov x20, #0\n"
      "whilelt p0.s, x20, x14\n"
      "cbz x12, 75f\n"
      "ld1w { z24.s }, p1/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "mov z27.d, z24.d\n"
      "mov z28.d, z24.d\n"
      "mov z29.d, z24.d\n"
      "mov z30.d, z24.d\n"
      "b 77f\n"
      "75:"  // Height 7: no bias
      "tbz %x[flags], #0, 76f\n"
      "ldr x23, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p0/Z, [x11]\n"
      "add x20, x11, x23, LSL #2\n"
      "ld1w { z25.s }, p0/Z, [x20]\n"
      "add x20, x20, x23, LSL #2\n"
      "ld1w { z26.s }, p0/Z, [x20]\n"
      "add x20, x20, x23, LSL #2\n"
      "ld1w { z27.s }, p0/Z, [x20]\n"
      "add x22, x20, x23, LSL #2\n"
      "add x21, x22, x23, LSL #2\n"
      "add x20, x21, x23, LSL #2\n"
      "ld1w { z28.s }, p0/Z, [x22]\n"
      "ld1w { z29.s }, p0/Z, [x21]\n"
      "ld1w { z30.s }, p0/Z, [x20]\n"
      "b 77f\n"
      "76:"  // Height 7: no accumulate
      "mov z24.b, #0\n"
      "mov z25.b, #0\n"
      "mov z26.b, #0\n"
      "mov z27.b, #0\n"
      "mov z28.b, #0\n"
      "mov z29.b, #0\n"
      "mov z30.b, #0\n"
      "77:"  // Height 7: setup done
      "mov x10, #0\n"
      "78:"  // Height 7: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 79f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "ldr x23, [x20, #0x28]\n"
      "ldr x22, [x20, #0x30]\n"
      "cbnz x10, 80f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "b 80f\n"
      "79:"  // Height 7: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "80:"  // Height 7: input setup done
      "subs x9, x9, #0x1\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "ld1rw { z3.s }, p1/Z, [x25]\n"
      "ld1rw { z4.s }, p1/Z, [x24]\n"
      "ld1rw { z5.s }, p1/Z, [x23]\n"
      "ld1rw { z6.s }, p1/Z, [x22]\n"
      "ble 82f\n"
      "81:"  // Height 7: Multiply loop: Main loop
      "ld1w { z8.s }, p1/Z, [x13]\n"
      "add x28, x28, #0x4\n"
      "subs x9, x9, #0x1\n"
      "add x27, x27, #0x4\n"
      "add x26, x26, #0x4\n"
      "add x25, x25, #0x4\n"
      "add x24, x24, #0x4\n"
      "add x23, x23, #0x4\n"
      "add x22, x22, #0x4\n"
      "fmla z24.s, p1/M, z8.s, z0.s\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "addvl x13, x13, #1\n"
      "fmla z25.s, p1/M, z8.s, z1.s\n"
      "fmla z26.s, p1/M, z8.s, z2.s\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "fmla z27.s, p1/M, z8.s, z3.s\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "fmla z28.s, p1/M, z8.s, z4.s\n"
      "ld1rw { z3.s }, p1/Z, [x25]\n"
      "fmla z29.s, p1/M, z8.s, z5.s\n"
      "ld1rw { z4.s }, p1/Z, [x24]\n"
      "fmla z30.s, p1/M, z8.s, z6.s\n"
      "ld1rw { z5.s }, p1/Z, [x23]\n"
      "ld1rw { z6.s }, p1/Z, [x22]\n"
      "bgt 81b\n"
      "82:"  // Height 7: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "ld1w { z9.s }, p1/Z, [x13]\n"
      "add x10, x10, #0x1\n"
      "addvl x13, x13, #1\n"
      "cmp x10, x20\n"
      "fmla z24.s, p1/M, z9.s, z0.s\n"
      "fmla z25.s, p1/M, z9.s, z1.s\n"
      "fmla z26.s, p1/M, z9.s, z2.s\n"
      "fmla z27.s, p1/M, z9.s, z3.s\n"
      "fmla z28.s, p1/M, z9.s, z4.s\n"
      "fmla z29.s, p1/M, z9.s, z5.s\n"
      "fmla z30.s, p1/M, z9.s, z6.s\n"
      "bne 78b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "tbz %x[flags], #1, 83f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z17.s }, p1/Z, [x21]\n"
      "ld1rw { z16.s }, p1/Z, [x20]\n"
      "fmin z24.s, p1/M, z24.s, z17.s\n"
      "fmin z25.s, p1/M, z25.s, z17.s\n"
      "fmin z26.s, p1/M, z26.s, z17.s\n"
      "fmin z27.s, p1/M, z27.s, z17.s\n"
      "fmin z28.s, p1/M, z28.s, z17.s\n"
      "fmin z29.s, p1/M, z29.s, z17.s\n"
      "fmin z30.s, p1/M, z30.s, z17.s\n"
      "fmax z24.s, p1/M, z24.s, z16.s\n"
      "fmax z25.s, p1/M, z25.s, z16.s\n"
      "fmax z26.s, p1/M, z26.s, z16.s\n"
      "fmax z27.s, p1/M, z27.s, z16.s\n"
      "fmax z28.s, p1/M, z28.s, z16.s\n"
      "fmax z29.s, p1/M, z29.s, z16.s\n"
      "fmax z30.s, p1/M, z30.s, z16.s\n"
      "83:"  // Height 7: No activation
      "st1w { z24.s }, p0, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p0, [x28]\n"
      "st1w { z26.s }, p0, [x27]\n"
      "st1w { z27.s }, p0, [x26]\n"
      "st1w { z28.s }, p0, [x25]\n"
      "st1w { z29.s }, p0, [x24]\n"
      "st1w { z30.s }, p0, [x23]\n"
      "decw x14\n"
      "cmp x14, XZR\n"
      "bgt 74b\n"
      "b 98f\n"
      "85:"  // Height 8
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x20\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "madd x20, x21, x20, x11\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "86:"  // Height 8: Column loop
      "mov x20, #0\n"
      "whilelt p0.s, x20, x14\n"
      "cbz x12, 87f\n"
      "ld1w { z24.s }, p1/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "mov z27.d, z24.d\n"
      "mov z28.d, z24.d\n"
      "mov z29.d, z24.d\n"
      "mov z30.d, z24.d\n"
      "mov z31.d, z24.d\n"
      "b 89f\n"
      "87:"  // Height 8: no bias
      "tbz %x[flags], #0, 88f\n"
      "ldr x23, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p0/Z, [x11]\n"
      "add x20, x11, x23, LSL #2\n"
      "ld1w { z25.s }, p0/Z, [x20]\n"
      "add x20, x20, x23, LSL #2\n"
      "ld1w { z26.s }, p0/Z, [x20]\n"
      "add x20, x20, x23, LSL #2\n"
      "ld1w { z27.s }, p0/Z, [x20]\n"
      "add x22, x20, x23, LSL #2\n"
      "add x20, x22, x23, LSL #2\n"
      "add x21, x20, x23, LSL #2\n"
      "ld1w { z28.s }, p0/Z, [x22]\n"
      "ld1w { z29.s }, p0/Z, [x20]\n"
      "add x20, x21, x23, LSL #2\n"
      "ld1w { z30.s }, p0/Z, [x21]\n"
      "ld1w { z31.s }, p0/Z, [x20]\n"
      "b 89f\n"
      "88:"  // Height 8: no accumulate
      "mov z24.b, #0\n"
      "mov z25.b, #0\n"
      "mov z26.b, #0\n"
      "mov z27.b, #0\n"
      "mov z28.b, #0\n"
      "mov z29.b, #0\n"
      "mov z30.b, #0\n"
      "mov z31.b, #0\n"
      "89:"  // Height 8: setup done
      "mov x10, #0\n"
      "90:"  // Height 8: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 91f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "ldr x23, [x20, #0x28]\n"
      "ldr x22, [x20, #0x30]\n"
      "ldr x21, [x20, #0x38]\n"
      "cbnz x10, 92f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "add x21, x21, x20, LSL #2\n"
      "b 92f\n"
      "91:"  // Height 8: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "add x21, x22, x21, LSL #2\n"
      "92:"  // Height 8: input setup done
      "subs x9, x9, #0x1\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "ld1rw { z3.s }, p1/Z, [x25]\n"
      "ld1rw { z4.s }, p1/Z, [x24]\n"
      "ld1rw { z5.s }, p1/Z, [x23]\n"
      "ld1rw { z6.s }, p1/Z, [x22]\n"
      "ld1rw { z7.s }, p1/Z, [x21]\n"
      "ble 94f\n"
      "93:"  // Height 8: Multiply loop: Main loop
      "ld1w { z8.s }, p1/Z, [x13]\n"
      "add x28, x28, #0x4\n"
      "subs x9, x9, #0x1\n"
      "add x27, x27, #0x4\n"
      "add x26, x26, #0x4\n"
      "add x25, x25, #0x4\n"
      "add x24, x24, #0x4\n"
      "add x23, x23, #0x4\n"
      "add x22, x22, #0x4\n"
      "fmla z24.s, p1/M, z8.s, z0.s\n"
      "ld1rw { z0.s }, p1/Z, [x28]\n"
      "add x21, x21, #0x4\n"
      "addvl x13, x13, #1\n"
      "fmla z25.s, p1/M, z8.s, z1.s\n"
      "ld1rw { z1.s }, p1/Z, [x27]\n"
      "fmla z26.s, p1/M, z8.s, z2.s\n"
      "ld1rw { z2.s }, p1/Z, [x26]\n"
      "fmla z27.s, p1/M, z8.s, z3.s\n"
      "ld1rw { z3.s }, p1/Z, [x25]\n"
      "fmla z28.s, p1/M, z8.s, z4.s\n"
      "ld1rw { z4.s }, p1/Z, [x24]\n"
      "fmla z29.s, p1/M, z8.s, z5.s\n"
      "ld1rw { z5.s }, p1/Z, [x23]\n"
      "fmla z30.s, p1/M, z8.s, z6.s\n"
      "fmla z31.s, p1/M, z8.s, z7.s\n"
      "ld1rw { z6.s }, p1/Z, [x22]\n"
      "ld1rw { z7.s }, p1/Z, [x21]\n"
      "bgt 93b\n"
      "94:"  // Height 8: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "ld1w { z9.s }, p1/Z, [x13]\n"
      "add x10, x10, #0x1\n"
      "addvl x13, x13, #1\n"
      "cmp x10, x20\n"
      "fmla z24.s, p1/M, z9.s, z0.s\n"
      "fmla z25.s, p1/M, z9.s, z1.s\n"
      "fmla z26.s, p1/M, z9.s, z2.s\n"
      "fmla z27.s, p1/M, z9.s, z3.s\n"
      "fmla z28.s, p1/M, z9.s, z4.s\n"
      "fmla z29.s, p1/M, z9.s, z5.s\n"
      "fmla z30.s, p1/M, z9.s, z6.s\n"
      "fmla z31.s, p1/M, z9.s, z7.s\n"
      "bne 90b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "tbz %x[flags], #1, 95f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z17.s }, p1/Z, [x21]\n"
      "ld1rw { z16.s }, p1/Z, [x20]\n"
      "fmin z24.s, p1/M, z24.s, z17.s\n"
      "fmin z25.s, p1/M, z25.s, z17.s\n"
      "fmin z26.s, p1/M, z26.s, z17.s\n"
      "fmin z27.s, p1/M, z27.s, z17.s\n"
      "fmin z28.s, p1/M, z28.s, z17.s\n"
      "fmin z29.s, p1/M, z29.s, z17.s\n"
      "fmin z30.s, p1/M, z30.s, z17.s\n"
      "fmin z31.s, p1/M, z31.s, z17.s\n"
      "fmax z24.s, p1/M, z24.s, z16.s\n"
      "fmax z25.s, p1/M, z25.s, z16.s\n"
      "fmax z26.s, p1/M, z26.s, z16.s\n"
      "fmax z27.s, p1/M, z27.s, z16.s\n"
      "fmax z28.s, p1/M, z28.s, z16.s\n"
      "fmax z29.s, p1/M, z29.s, z16.s\n"
      "fmax z30.s, p1/M, z30.s, z16.s\n"
      "fmax z31.s, p1/M, z31.s, z16.s\n"
      "95:"  // Height 8: No activation
      "st1w { z24.s }, p0, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p0, [x28]\n"
      "st1w { z26.s }, p0, [x27]\n"
      "st1w { z27.s }, p0, [x26]\n"
      "st1w { z28.s }, p0, [x25]\n"
      "st1w { z29.s }, p0, [x24]\n"
      "st1w { z30.s }, p0, [x23]\n"
      "st1w { z31.s }, p0, [x22]\n"
      "decw x14\n"
      "cmp x14, XZR\n"
      "bgt 86b\n"
      "subs %x[M], %x[M], #0x8\n"
      "beq 98f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 97f\n"
      "add x21, x21, #0x8\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "97:"  // Update direct input
      "mov x20, #0x20\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "98:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_maxval] "I" (offsetof(KernelArgs, maxval)), [offsetof_minval] "I" (offsetof(KernelArgs, minval)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "p0", "p1", "x9", "x10", "x11", "x12", "x13", "x14", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z16", "z17", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm

#endif // defined(ARM_COMPUTE_ENABLE_SVE) && defined(__aarch64__)

