/*
 * Copyright (c) 2021, 2023-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_BF16) && defined(ARM_COMPUTE_ENABLE_SVE) && defined(__aarch64__)

#include "arm_gemm/arm_gemm.hpp"
#include "arm_common/internal/utils.hpp"
#include "arm_common/bfloat.hpp"
#include <cassert>
#include <limits>

namespace arm_gemm {

void sve_hybrid_bf16fp32_mmla_6x4VL (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<bfloat16> A_arg,
    size_t M, size_t N, const bfloat16 *B_ptr, IndirectOutputArg<float> output_arg,
    const float *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const bfloat16 *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const float *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "ptrue p5.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 66f\n"
      "cmp %x[M], #0x4\n"
      "bgt 53f\n"
      "beq 40f\n"
      "cmp %x[M], #0x2\n"
      "bgt 27f\n"
      "beq 14f\n"
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "mov x20, #0\n"
      "whilelt p4.s, x20, x12\n"
      "incw x20\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "cbz x10, 3f\n"
      "ld1w { z8.s }, p5/Z, [x10]\n"
      "ld1w { z9.s }, p5/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "zip2 z12.d, z8.d, z8.d\n"
      "zip1 z8.d, z8.d, z8.d\n"
      "zip2 z13.d, z9.d, z9.d\n"
      "zip1 z9.d, z9.d, z9.d\n"
      "zip2 z14.d, z10.d, z10.d\n"
      "zip1 z10.d, z10.d, z10.d\n"
      "zip2 z15.d, z11.d, z11.d\n"
      "zip1 z11.d, z11.d, z11.d\n"
      "b 5f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 4f\n"
      "ld1w { z9.s }, p4/Z, [x9]\n"
      "ld1w { z10.s }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1w { z11.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "zip1 z8.d, z9.d, z12.d\n"
      "zip2 z12.d, z9.d, z12.d\n"
      "zip1 z9.d, z10.d, z13.d\n"
      "zip2 z13.d, z10.d, z13.d\n"
      "zip1 z10.d, z11.d, z14.d\n"
      "zip2 z14.d, z11.d, z14.d\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "b 5f\n"
      "4:"  // Height 1: no accumulate
      "mov z8.b, #0\n"
      "mov z9.b, #0\n"
      "mov z10.b, #0\n"
      "mov z11.b, #0\n"
      "mov z12.b, #0\n"
      "mov z13.b, #0\n"
      "mov z14.b, #0\n"
      "mov z15.b, #0\n"
      "5:"  // Height 1: setup done
      "mov x28, #0\n"
      "6:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 7f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "cbnz x28, 8f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "b 8f\n"
      "7:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "8:"  // Height 1: input setup done
      "cmp x27, #0x8\n"
      "ble 10f\n"
      "9:"  // Height 1: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      "sub x27, x27, #0x8\n"
      "cmp x27, #0x8\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "trn1 z0.d, z1.d, z2.d\n"
      "trn2 z1.d, z1.d, z2.d\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #16\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-8, MUL VL]\n"
      "ld1h { z6.h }, p5/Z, [x11, #-7, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-6, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-5, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-3, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-2, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-1, MUL VL]\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      "bgt 9b\n"
      "10:"  // Height 1: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "trn1 z0.d, z1.d, z2.d\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      "trn2 z1.d, z1.d, z2.d\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      "ble 11f\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      "11:"  // Height 1: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 6b\n"
      "uzp1 z8.d, z8.d, z12.d\n"
      "uzp1 z9.d, z9.d, z13.d\n"
      "uzp1 z10.d, z10.d, z14.d\n"
      "uzp1 z11.d, z11.d, z15.d\n"
      "tbz %x[flags], #1, 12f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z17.s }, p5/Z, [x21]\n"
      "ld1rw { z16.s }, p5/Z, [x20]\n"
      "fmin z8.s, p5/M, z8.s, z17.s\n"
      "fmin z9.s, p5/M, z9.s, z17.s\n"
      "fmin z10.s, p5/M, z10.s, z17.s\n"
      "fmin z11.s, p5/M, z11.s, z17.s\n"
      "fmax z8.s, p5/M, z8.s, z16.s\n"
      "fmax z9.s, p5/M, z9.s, z16.s\n"
      "fmax z10.s, p5/M, z10.s, z16.s\n"
      "fmax z11.s, p5/M, z11.s, z16.s\n"
      "12:"  // Height 1: No activation
      "st1w { z8.s }, p4, [x9]\n"
      "st1w { z9.s }, p3, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p2, [x9, #2, MUL VL]\n"
      "st1w { z11.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 2b\n"
      "b 80f\n"
      "14:"  // Height 2
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "15:"  // Height 2: Column loop
      "mov x20, #0\n"
      "whilelt p4.s, x20, x12\n"
      "incw x20\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "cbz x10, 16f\n"
      "ld1w { z8.s }, p5/Z, [x10]\n"
      "ld1w { z9.s }, p5/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "zip2 z12.d, z8.d, z8.d\n"
      "zip1 z8.d, z8.d, z8.d\n"
      "zip2 z13.d, z9.d, z9.d\n"
      "zip1 z9.d, z9.d, z9.d\n"
      "zip2 z14.d, z10.d, z10.d\n"
      "zip1 z10.d, z10.d, z10.d\n"
      "zip2 z15.d, z11.d, z11.d\n"
      "zip1 z11.d, z11.d, z11.d\n"
      "b 18f\n"
      "16:"  // Height 2: no bias
      "tbz %x[flags], #0, 17f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z9.s }, p4/Z, [x9]\n"
      "ld1w { z10.s }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1w { z11.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "add x20, x9, x20, LSL #2\n"
      "ld1w { z12.s }, p4/Z, [x20]\n"
      "ld1w { z13.s }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1w { z14.s }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1w { z15.s }, p1/Z, [x20, #3, MUL VL]\n"
      "zip1 z8.d, z9.d, z12.d\n"
      "zip2 z12.d, z9.d, z12.d\n"
      "zip1 z9.d, z10.d, z13.d\n"
      "zip2 z13.d, z10.d, z13.d\n"
      "zip1 z10.d, z11.d, z14.d\n"
      "zip2 z14.d, z11.d, z14.d\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "b 18f\n"
      "17:"  // Height 2: no accumulate
      "mov z8.b, #0\n"
      "mov z9.b, #0\n"
      "mov z10.b, #0\n"
      "mov z11.b, #0\n"
      "mov z12.b, #0\n"
      "mov z13.b, #0\n"
      "mov z14.b, #0\n"
      "mov z15.b, #0\n"
      "18:"  // Height 2: setup done
      "mov x28, #0\n"
      "19:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 20f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 21f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "b 21f\n"
      "20:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "21:"  // Height 2: input setup done
      "cmp x27, #0x8\n"
      "ble 23f\n"
      "22:"  // Height 2: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      "sub x27, x27, #0x8\n"
      "cmp x27, #0x8\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqh { z2.h }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "trn1 z0.d, z1.d, z2.d\n"
      "trn2 z1.d, z1.d, z2.d\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #16\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-8, MUL VL]\n"
      "ld1h { z6.h }, p5/Z, [x11, #-7, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-6, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-5, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-3, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-2, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-1, MUL VL]\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      "bgt 22b\n"
      "23:"  // Height 2: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "ld1rqh { z2.h }, p0/Z, [x25]\n"
      "trn1 z0.d, z1.d, z2.d\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      "trn2 z1.d, z1.d, z2.d\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      "ble 24f\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      "24:"  // Height 2: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 19b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 z7.d, z8.d, z12.d\n"
      "uzp2 z8.d, z8.d, z12.d\n"
      "uzp1 z12.d, z9.d, z13.d\n"
      "uzp2 z9.d, z9.d, z13.d\n"
      "uzp1 z13.d, z10.d, z14.d\n"
      "uzp2 z10.d, z10.d, z14.d\n"
      "uzp1 z14.d, z11.d, z15.d\n"
      "uzp2 z11.d, z11.d, z15.d\n"
      "add x26, x9, x20, LSL #2\n"
      "tbz %x[flags], #1, 25f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z17.s }, p5/Z, [x21]\n"
      "ld1rw { z16.s }, p5/Z, [x20]\n"
      "fmin z7.s, p5/M, z7.s, z17.s\n"
      "fmin z12.s, p5/M, z12.s, z17.s\n"
      "fmin z13.s, p5/M, z13.s, z17.s\n"
      "fmin z14.s, p5/M, z14.s, z17.s\n"
      "fmin z8.s, p5/M, z8.s, z17.s\n"
      "fmin z9.s, p5/M, z9.s, z17.s\n"
      "fmin z10.s, p5/M, z10.s, z17.s\n"
      "fmin z11.s, p5/M, z11.s, z17.s\n"
      "fmax z7.s, p5/M, z7.s, z16.s\n"
      "fmax z12.s, p5/M, z12.s, z16.s\n"
      "fmax z13.s, p5/M, z13.s, z16.s\n"
      "fmax z14.s, p5/M, z14.s, z16.s\n"
      "fmax z8.s, p5/M, z8.s, z16.s\n"
      "fmax z9.s, p5/M, z9.s, z16.s\n"
      "fmax z10.s, p5/M, z10.s, z16.s\n"
      "fmax z11.s, p5/M, z11.s, z16.s\n"
      "25:"  // Height 2: No activation
      "st1w { z7.s }, p4, [x9]\n"
      "st1w { z12.s }, p3, [x9, #1, MUL VL]\n"
      "st1w { z13.s }, p2, [x9, #2, MUL VL]\n"
      "st1w { z14.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z8.s }, p4, [x26]\n"
      "st1w { z9.s }, p3, [x26, #1, MUL VL]\n"
      "st1w { z10.s }, p2, [x26, #2, MUL VL]\n"
      "st1w { z11.s }, p1, [x26, #3, MUL VL]\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 15b\n"
      "b 80f\n"
      "27:"  // Height 3
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "28:"  // Height 3: Column loop
      "mov x20, #0\n"
      "whilelt p4.s, x20, x12\n"
      "incw x20\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "cbz x10, 29f\n"
      "ld1w { z8.s }, p5/Z, [x10]\n"
      "ld1w { z9.s }, p5/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "zip2 z12.d, z8.d, z8.d\n"
      "zip1 z8.d, z8.d, z8.d\n"
      "zip2 z13.d, z9.d, z9.d\n"
      "zip1 z9.d, z9.d, z9.d\n"
      "zip2 z14.d, z10.d, z10.d\n"
      "zip1 z10.d, z10.d, z10.d\n"
      "zip2 z15.d, z11.d, z11.d\n"
      "zip1 z11.d, z11.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z20.d, z12.d\n"
      "mov z17.d, z9.d\n"
      "mov z21.d, z13.d\n"
      "mov z18.d, z10.d\n"
      "mov z22.d, z14.d\n"
      "mov z19.d, z11.d\n"
      "mov z23.d, z15.d\n"
      "b 31f\n"
      "29:"  // Height 3: no bias
      "tbz %x[flags], #0, 30f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z9.s }, p4/Z, [x9]\n"
      "ld1w { z10.s }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1w { z11.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "add x21, x9, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z12.s }, p4/Z, [x21]\n"
      "ld1w { z13.s }, p3/Z, [x21, #1, MUL VL]\n"
      "ld1w { z14.s }, p2/Z, [x21, #2, MUL VL]\n"
      "ld1w { z15.s }, p1/Z, [x21, #3, MUL VL]\n"
      "ld1w { z17.s }, p4/Z, [x20]\n"
      "ld1w { z18.s }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1w { z19.s }, p2/Z, [x20, #2, MUL VL]\n"
      "zip1 z8.d, z9.d, z12.d\n"
      "zip2 z12.d, z9.d, z12.d\n"
      "ld1w { z24.s }, p1/Z, [x20, #3, MUL VL]\n"
      "zip1 z9.d, z10.d, z13.d\n"
      "zip2 z13.d, z10.d, z13.d\n"
      "zip1 z10.d, z11.d, z14.d\n"
      "zip2 z14.d, z11.d, z14.d\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "zip1 z16.d, z17.d, z20.d\n"
      "zip2 z20.d, z17.d, z20.d\n"
      "zip1 z17.d, z18.d, z21.d\n"
      "zip2 z21.d, z18.d, z21.d\n"
      "zip1 z18.d, z19.d, z22.d\n"
      "zip2 z22.d, z19.d, z22.d\n"
      "zip1 z19.d, z24.d, z23.d\n"
      "zip2 z23.d, z24.d, z23.d\n"
      "b 31f\n"
      "30:"  // Height 3: no accumulate
      "mov z8.b, #0\n"
      "mov z9.b, #0\n"
      "mov z10.b, #0\n"
      "mov z11.b, #0\n"
      "mov z12.b, #0\n"
      "mov z13.b, #0\n"
      "mov z14.b, #0\n"
      "mov z15.b, #0\n"
      "mov z16.b, #0\n"
      "mov z17.b, #0\n"
      "mov z18.b, #0\n"
      "mov z19.b, #0\n"
      "mov z20.b, #0\n"
      "mov z21.b, #0\n"
      "mov z22.b, #0\n"
      "mov z23.b, #0\n"
      "31:"  // Height 3: setup done
      "mov x28, #0\n"
      "32:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 33f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 34f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "b 34f\n"
      "33:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "34:"  // Height 3: input setup done
      "cmp x27, #0x8\n"
      "ble 36f\n"
      "35:"  // Height 3: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      "sub x27, x27, #0x8\n"
      "cmp x27, #0x8\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqh { z2.h }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqh { z3.h }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "trn1 z0.d, z1.d, z2.d\n"
      "trn2 z1.d, z1.d, z2.d\n"
      "trn1 z2.d, z3.d, z4.d\n"
      "trn2 z3.d, z3.d, z4.d\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      ".inst 0x6467e450  // bfmmla z16.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e454  // bfmmla z20.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      ".inst 0x6467e451  // bfmmla z17.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      ".inst 0x6466e455  // bfmmla z21.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      ".inst 0x6467e452  // bfmmla z18.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      ".inst 0x6466e456  // bfmmla z22.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #16\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6467e453  // bfmmla z19.s, z2.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      ".inst 0x6466e457  // bfmmla z23.s, z2.h, z6.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-8, MUL VL]\n"
      "ld1h { z6.h }, p5/Z, [x11, #-7, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      ".inst 0x6467e470  // bfmmla z16.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-6, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      ".inst 0x6466e474  // bfmmla z20.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-5, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      ".inst 0x6467e471  // bfmmla z17.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      ".inst 0x6466e475  // bfmmla z21.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-3, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      ".inst 0x6467e472  // bfmmla z18.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-2, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      ".inst 0x6466e476  // bfmmla z22.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-1, MUL VL]\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6467e473  // bfmmla z19.s, z3.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      ".inst 0x6466e477  // bfmmla z23.s, z3.h, z6.h\n"
      "bgt 35b\n"
      "36:"  // Height 3: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "ld1rqh { z2.h }, p0/Z, [x25]\n"
      "ld1rqh { z3.h }, p0/Z, [x24]\n"
      "trn1 z0.d, z1.d, z2.d\n"
      "trn2 z1.d, z1.d, z2.d\n"
      "trn1 z2.d, z3.d, z4.d\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      "trn2 z3.d, z3.d, z4.d\n"
      ".inst 0x6467e450  // bfmmla z16.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e454  // bfmmla z20.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      ".inst 0x6467e451  // bfmmla z17.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      ".inst 0x6466e455  // bfmmla z21.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      ".inst 0x6467e452  // bfmmla z18.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      ".inst 0x6466e456  // bfmmla z22.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6467e453  // bfmmla z19.s, z2.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      ".inst 0x6466e457  // bfmmla z23.s, z2.h, z6.h\n"
      "ble 37f\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      ".inst 0x6467e470  // bfmmla z16.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      ".inst 0x6466e474  // bfmmla z20.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      ".inst 0x6467e471  // bfmmla z17.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      ".inst 0x6466e475  // bfmmla z21.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      ".inst 0x6467e472  // bfmmla z18.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      ".inst 0x6466e476  // bfmmla z22.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6467e473  // bfmmla z19.s, z3.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      ".inst 0x6466e477  // bfmmla z23.s, z3.h, z6.h\n"
      "37:"  // Height 3: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 32b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 z7.d, z8.d, z12.d\n"
      "uzp2 z8.d, z8.d, z12.d\n"
      "uzp1 z12.d, z9.d, z13.d\n"
      "uzp2 z9.d, z9.d, z13.d\n"
      "uzp1 z13.d, z10.d, z14.d\n"
      "uzp2 z10.d, z10.d, z14.d\n"
      "add x26, x9, x20, LSL #2\n"
      "uzp1 z14.d, z11.d, z15.d\n"
      "uzp2 z11.d, z11.d, z15.d\n"
      "add x25, x26, x20, LSL #2\n"
      "uzp1 z16.d, z16.d, z20.d\n"
      "uzp1 z17.d, z17.d, z21.d\n"
      "uzp1 z18.d, z18.d, z22.d\n"
      "uzp1 z19.d, z19.d, z23.d\n"
      "tbz %x[flags], #1, 38f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z21.s }, p5/Z, [x21]\n"
      "ld1rw { z20.s }, p5/Z, [x20]\n"
      "fmin z7.s, p5/M, z7.s, z21.s\n"
      "fmin z12.s, p5/M, z12.s, z21.s\n"
      "fmin z13.s, p5/M, z13.s, z21.s\n"
      "fmin z14.s, p5/M, z14.s, z21.s\n"
      "fmin z8.s, p5/M, z8.s, z21.s\n"
      "fmin z9.s, p5/M, z9.s, z21.s\n"
      "fmin z10.s, p5/M, z10.s, z21.s\n"
      "fmin z11.s, p5/M, z11.s, z21.s\n"
      "fmin z16.s, p5/M, z16.s, z21.s\n"
      "fmin z17.s, p5/M, z17.s, z21.s\n"
      "fmin z18.s, p5/M, z18.s, z21.s\n"
      "fmin z19.s, p5/M, z19.s, z21.s\n"
      "fmax z7.s, p5/M, z7.s, z20.s\n"
      "fmax z12.s, p5/M, z12.s, z20.s\n"
      "fmax z13.s, p5/M, z13.s, z20.s\n"
      "fmax z14.s, p5/M, z14.s, z20.s\n"
      "fmax z8.s, p5/M, z8.s, z20.s\n"
      "fmax z9.s, p5/M, z9.s, z20.s\n"
      "fmax z10.s, p5/M, z10.s, z20.s\n"
      "fmax z11.s, p5/M, z11.s, z20.s\n"
      "fmax z16.s, p5/M, z16.s, z20.s\n"
      "fmax z17.s, p5/M, z17.s, z20.s\n"
      "fmax z18.s, p5/M, z18.s, z20.s\n"
      "fmax z19.s, p5/M, z19.s, z20.s\n"
      "38:"  // Height 3: No activation
      "st1w { z7.s }, p4, [x9]\n"
      "st1w { z12.s }, p3, [x9, #1, MUL VL]\n"
      "st1w { z13.s }, p2, [x9, #2, MUL VL]\n"
      "st1w { z14.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z8.s }, p4, [x26]\n"
      "st1w { z9.s }, p3, [x26, #1, MUL VL]\n"
      "st1w { z10.s }, p2, [x26, #2, MUL VL]\n"
      "st1w { z11.s }, p1, [x26, #3, MUL VL]\n"
      "st1w { z16.s }, p4, [x25]\n"
      "st1w { z17.s }, p3, [x25, #1, MUL VL]\n"
      "st1w { z18.s }, p2, [x25, #2, MUL VL]\n"
      "st1w { z19.s }, p1, [x25, #3, MUL VL]\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 28b\n"
      "b 80f\n"
      "40:"  // Height 4
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "41:"  // Height 4: Column loop
      "mov x20, #0\n"
      "whilelt p4.s, x20, x12\n"
      "incw x20\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "cbz x10, 42f\n"
      "ld1w { z8.s }, p5/Z, [x10]\n"
      "ld1w { z9.s }, p5/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "zip2 z12.d, z8.d, z8.d\n"
      "zip1 z8.d, z8.d, z8.d\n"
      "zip2 z13.d, z9.d, z9.d\n"
      "zip1 z9.d, z9.d, z9.d\n"
      "zip2 z14.d, z10.d, z10.d\n"
      "zip1 z10.d, z10.d, z10.d\n"
      "zip2 z15.d, z11.d, z11.d\n"
      "zip1 z11.d, z11.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z20.d, z12.d\n"
      "mov z17.d, z9.d\n"
      "mov z21.d, z13.d\n"
      "mov z18.d, z10.d\n"
      "mov z22.d, z14.d\n"
      "mov z19.d, z11.d\n"
      "mov z23.d, z15.d\n"
      "b 44f\n"
      "42:"  // Height 4: no bias
      "tbz %x[flags], #0, 43f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z9.s }, p4/Z, [x9]\n"
      "ld1w { z10.s }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1w { z11.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "add x22, x9, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z12.s }, p4/Z, [x22]\n"
      "ld1w { z13.s }, p3/Z, [x22, #1, MUL VL]\n"
      "ld1w { z14.s }, p2/Z, [x22, #2, MUL VL]\n"
      "ld1w { z15.s }, p1/Z, [x22, #3, MUL VL]\n"
      "ld1w { z17.s }, p4/Z, [x21]\n"
      "ld1w { z18.s }, p3/Z, [x21, #1, MUL VL]\n"
      "ld1w { z19.s }, p2/Z, [x21, #2, MUL VL]\n"
      "ld1w { z24.s }, p1/Z, [x21, #3, MUL VL]\n"
      "zip1 z8.d, z9.d, z12.d\n"
      "zip2 z12.d, z9.d, z12.d\n"
      "ld1w { z20.s }, p4/Z, [x20]\n"
      "ld1w { z21.s }, p3/Z, [x20, #1, MUL VL]\n"
      "zip1 z9.d, z10.d, z13.d\n"
      "zip2 z13.d, z10.d, z13.d\n"
      "ld1w { z22.s }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1w { z23.s }, p1/Z, [x20, #3, MUL VL]\n"
      "zip1 z10.d, z11.d, z14.d\n"
      "zip2 z14.d, z11.d, z14.d\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "zip1 z16.d, z17.d, z20.d\n"
      "zip2 z20.d, z17.d, z20.d\n"
      "zip1 z17.d, z18.d, z21.d\n"
      "zip2 z21.d, z18.d, z21.d\n"
      "zip1 z18.d, z19.d, z22.d\n"
      "zip2 z22.d, z19.d, z22.d\n"
      "zip1 z19.d, z24.d, z23.d\n"
      "zip2 z23.d, z24.d, z23.d\n"
      "b 44f\n"
      "43:"  // Height 4: no accumulate
      "mov z8.b, #0\n"
      "mov z9.b, #0\n"
      "mov z10.b, #0\n"
      "mov z11.b, #0\n"
      "mov z12.b, #0\n"
      "mov z13.b, #0\n"
      "mov z14.b, #0\n"
      "mov z15.b, #0\n"
      "mov z16.b, #0\n"
      "mov z17.b, #0\n"
      "mov z18.b, #0\n"
      "mov z19.b, #0\n"
      "mov z20.b, #0\n"
      "mov z21.b, #0\n"
      "mov z22.b, #0\n"
      "mov z23.b, #0\n"
      "44:"  // Height 4: setup done
      "mov x28, #0\n"
      "45:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 46f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 47f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "b 47f\n"
      "46:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "47:"  // Height 4: input setup done
      "cmp x27, #0x8\n"
      "ble 49f\n"
      "48:"  // Height 4: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      "sub x27, x27, #0x8\n"
      "cmp x27, #0x8\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqh { z2.h }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqh { z3.h }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "ld1rqh { z4.h }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      "trn1 z0.d, z1.d, z2.d\n"
      "trn2 z1.d, z1.d, z2.d\n"
      "trn1 z2.d, z3.d, z4.d\n"
      "trn2 z3.d, z3.d, z4.d\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      ".inst 0x6467e450  // bfmmla z16.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e454  // bfmmla z20.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      ".inst 0x6467e451  // bfmmla z17.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      ".inst 0x6466e455  // bfmmla z21.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      ".inst 0x6467e452  // bfmmla z18.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      ".inst 0x6466e456  // bfmmla z22.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #16\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6467e453  // bfmmla z19.s, z2.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      ".inst 0x6466e457  // bfmmla z23.s, z2.h, z6.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-8, MUL VL]\n"
      "ld1h { z6.h }, p5/Z, [x11, #-7, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      ".inst 0x6467e470  // bfmmla z16.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-6, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      ".inst 0x6466e474  // bfmmla z20.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-5, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      ".inst 0x6467e471  // bfmmla z17.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      ".inst 0x6466e475  // bfmmla z21.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-3, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      ".inst 0x6467e472  // bfmmla z18.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-2, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      ".inst 0x6466e476  // bfmmla z22.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-1, MUL VL]\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6467e473  // bfmmla z19.s, z3.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      ".inst 0x6466e477  // bfmmla z23.s, z3.h, z6.h\n"
      "bgt 48b\n"
      "49:"  // Height 4: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "ld1rqh { z2.h }, p0/Z, [x25]\n"
      "ld1rqh { z3.h }, p0/Z, [x24]\n"
      "ld1rqh { z4.h }, p0/Z, [x23]\n"
      "trn1 z0.d, z1.d, z2.d\n"
      "trn2 z1.d, z1.d, z2.d\n"
      "trn1 z2.d, z3.d, z4.d\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      "trn2 z3.d, z3.d, z4.d\n"
      ".inst 0x6467e450  // bfmmla z16.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e454  // bfmmla z20.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      ".inst 0x6467e451  // bfmmla z17.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      ".inst 0x6466e455  // bfmmla z21.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      ".inst 0x6467e452  // bfmmla z18.s, z2.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      ".inst 0x6466e456  // bfmmla z22.s, z2.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6467e453  // bfmmla z19.s, z2.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      ".inst 0x6466e457  // bfmmla z23.s, z2.h, z6.h\n"
      "ble 50f\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      ".inst 0x6467e470  // bfmmla z16.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      ".inst 0x6466e474  // bfmmla z20.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      ".inst 0x6467e471  // bfmmla z17.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      ".inst 0x6466e475  // bfmmla z21.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      ".inst 0x6467e472  // bfmmla z18.s, z3.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      ".inst 0x6466e476  // bfmmla z22.s, z3.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6467e473  // bfmmla z19.s, z3.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      ".inst 0x6466e477  // bfmmla z23.s, z3.h, z6.h\n"
      "50:"  // Height 4: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 45b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 z7.d, z8.d, z12.d\n"
      "uzp2 z8.d, z8.d, z12.d\n"
      "uzp1 z12.d, z9.d, z13.d\n"
      "uzp2 z9.d, z9.d, z13.d\n"
      "uzp1 z13.d, z10.d, z14.d\n"
      "uzp2 z10.d, z10.d, z14.d\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "uzp1 z14.d, z11.d, z15.d\n"
      "uzp2 z11.d, z11.d, z15.d\n"
      "add x24, x25, x20, LSL #2\n"
      "uzp1 z15.d, z16.d, z20.d\n"
      "uzp2 z16.d, z16.d, z20.d\n"
      "uzp1 z20.d, z17.d, z21.d\n"
      "uzp2 z17.d, z17.d, z21.d\n"
      "uzp1 z21.d, z18.d, z22.d\n"
      "uzp2 z18.d, z18.d, z22.d\n"
      "uzp1 z22.d, z19.d, z23.d\n"
      "uzp2 z19.d, z19.d, z23.d\n"
      "tbz %x[flags], #1, 51f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z24.s }, p5/Z, [x21]\n"
      "ld1rw { z23.s }, p5/Z, [x20]\n"
      "fmin z7.s, p5/M, z7.s, z24.s\n"
      "fmin z12.s, p5/M, z12.s, z24.s\n"
      "fmin z13.s, p5/M, z13.s, z24.s\n"
      "fmin z14.s, p5/M, z14.s, z24.s\n"
      "fmin z8.s, p5/M, z8.s, z24.s\n"
      "fmin z9.s, p5/M, z9.s, z24.s\n"
      "fmin z10.s, p5/M, z10.s, z24.s\n"
      "fmin z11.s, p5/M, z11.s, z24.s\n"
      "fmin z15.s, p5/M, z15.s, z24.s\n"
      "fmin z20.s, p5/M, z20.s, z24.s\n"
      "fmin z21.s, p5/M, z21.s, z24.s\n"
      "fmin z22.s, p5/M, z22.s, z24.s\n"
      "fmin z16.s, p5/M, z16.s, z24.s\n"
      "fmin z17.s, p5/M, z17.s, z24.s\n"
      "fmin z18.s, p5/M, z18.s, z24.s\n"
      "fmin z19.s, p5/M, z19.s, z24.s\n"
      "fmax z7.s, p5/M, z7.s, z23.s\n"
      "fmax z12.s, p5/M, z12.s, z23.s\n"
      "fmax z13.s, p5/M, z13.s, z23.s\n"
      "fmax z14.s, p5/M, z14.s, z23.s\n"
      "fmax z8.s, p5/M, z8.s, z23.s\n"
      "fmax z9.s, p5/M, z9.s, z23.s\n"
      "fmax z10.s, p5/M, z10.s, z23.s\n"
      "fmax z11.s, p5/M, z11.s, z23.s\n"
      "fmax z15.s, p5/M, z15.s, z23.s\n"
      "fmax z20.s, p5/M, z20.s, z23.s\n"
      "fmax z21.s, p5/M, z21.s, z23.s\n"
      "fmax z22.s, p5/M, z22.s, z23.s\n"
      "fmax z16.s, p5/M, z16.s, z23.s\n"
      "fmax z17.s, p5/M, z17.s, z23.s\n"
      "fmax z18.s, p5/M, z18.s, z23.s\n"
      "fmax z19.s, p5/M, z19.s, z23.s\n"
      "51:"  // Height 4: No activation
      "st1w { z7.s }, p4, [x9]\n"
      "st1w { z12.s }, p3, [x9, #1, MUL VL]\n"
      "st1w { z13.s }, p2, [x9, #2, MUL VL]\n"
      "st1w { z14.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z8.s }, p4, [x26]\n"
      "st1w { z9.s }, p3, [x26, #1, MUL VL]\n"
      "st1w { z10.s }, p2, [x26, #2, MUL VL]\n"
      "st1w { z11.s }, p1, [x26, #3, MUL VL]\n"
      "st1w { z15.s }, p4, [x25]\n"
      "st1w { z20.s }, p3, [x25, #1, MUL VL]\n"
      "st1w { z21.s }, p2, [x25, #2, MUL VL]\n"
      "st1w { z22.s }, p1, [x25, #3, MUL VL]\n"
      "st1w { z16.s }, p4, [x24]\n"
      "st1w { z17.s }, p3, [x24, #1, MUL VL]\n"
      "st1w { z18.s }, p2, [x24, #2, MUL VL]\n"
      "st1w { z19.s }, p1, [x24, #3, MUL VL]\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 41b\n"
      "b 80f\n"
      "53:"  // Height 5
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "54:"  // Height 5: Column loop
      "mov x20, #0\n"
      "whilelt p4.s, x20, x12\n"
      "incw x20\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "cbz x10, 55f\n"
      "ld1w { z8.s }, p5/Z, [x10]\n"
      "ld1w { z9.s }, p5/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "zip2 z12.d, z8.d, z8.d\n"
      "zip1 z8.d, z8.d, z8.d\n"
      "zip2 z13.d, z9.d, z9.d\n"
      "zip1 z9.d, z9.d, z9.d\n"
      "zip2 z14.d, z10.d, z10.d\n"
      "zip1 z10.d, z10.d, z10.d\n"
      "zip2 z15.d, z11.d, z11.d\n"
      "zip1 z11.d, z11.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z20.d, z12.d\n"
      "mov z17.d, z9.d\n"
      "mov z21.d, z13.d\n"
      "mov z18.d, z10.d\n"
      "mov z22.d, z14.d\n"
      "mov z19.d, z11.d\n"
      "mov z23.d, z15.d\n"
      "mov z24.d, z8.d\n"
      "mov z28.d, z12.d\n"
      "mov z25.d, z9.d\n"
      "mov z29.d, z13.d\n"
      "mov z26.d, z10.d\n"
      "mov z30.d, z14.d\n"
      "mov z27.d, z11.d\n"
      "mov z31.d, z15.d\n"
      "b 57f\n"
      "55:"  // Height 5: no bias
      "tbz %x[flags], #0, 56f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z9.s }, p4/Z, [x9]\n"
      "ld1w { z10.s }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1w { z11.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "add x23, x9, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "ld1w { z12.s }, p4/Z, [x23]\n"
      "ld1w { z13.s }, p3/Z, [x23, #1, MUL VL]\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z14.s }, p2/Z, [x23, #2, MUL VL]\n"
      "ld1w { z15.s }, p1/Z, [x23, #3, MUL VL]\n"
      "ld1w { z17.s }, p4/Z, [x22]\n"
      "ld1w { z18.s }, p3/Z, [x22, #1, MUL VL]\n"
      "ld1w { z19.s }, p2/Z, [x22, #2, MUL VL]\n"
      "ld1w { z24.s }, p1/Z, [x22, #3, MUL VL]\n"
      "zip1 z8.d, z9.d, z12.d\n"
      "zip2 z12.d, z9.d, z12.d\n"
      "ld1w { z20.s }, p4/Z, [x21]\n"
      "ld1w { z21.s }, p3/Z, [x21, #1, MUL VL]\n"
      "zip1 z9.d, z10.d, z13.d\n"
      "zip2 z13.d, z10.d, z13.d\n"
      "ld1w { z22.s }, p2/Z, [x21, #2, MUL VL]\n"
      "ld1w { z23.s }, p1/Z, [x21, #3, MUL VL]\n"
      "zip1 z10.d, z11.d, z14.d\n"
      "zip2 z14.d, z11.d, z14.d\n"
      "ld1w { z25.s }, p4/Z, [x20]\n"
      "ld1w { z26.s }, p3/Z, [x20, #1, MUL VL]\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "ld1w { z27.s }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1w { z6.s }, p1/Z, [x20, #3, MUL VL]\n"
      "zip1 z16.d, z17.d, z20.d\n"
      "zip2 z20.d, z17.d, z20.d\n"
      "zip1 z17.d, z18.d, z21.d\n"
      "zip2 z21.d, z18.d, z21.d\n"
      "zip1 z18.d, z19.d, z22.d\n"
      "zip2 z22.d, z19.d, z22.d\n"
      "zip1 z19.d, z24.d, z23.d\n"
      "zip2 z23.d, z24.d, z23.d\n"
      "zip1 z24.d, z25.d, z28.d\n"
      "zip2 z28.d, z25.d, z28.d\n"
      "zip1 z25.d, z26.d, z29.d\n"
      "zip2 z29.d, z26.d, z29.d\n"
      "zip1 z26.d, z27.d, z30.d\n"
      "zip2 z30.d, z27.d, z30.d\n"
      "zip1 z27.d, z6.d, z31.d\n"
      "zip2 z31.d, z6.d, z31.d\n"
      "b 57f\n"
      "56:"  // Height 5: no accumulate
      "mov z8.b, #0\n"
      "mov z9.b, #0\n"
      "mov z10.b, #0\n"
      "mov z11.b, #0\n"
      "mov z12.b, #0\n"
      "mov z13.b, #0\n"
      "mov z14.b, #0\n"
      "mov z15.b, #0\n"
      "mov z16.b, #0\n"
      "mov z17.b, #0\n"
      "mov z18.b, #0\n"
      "mov z19.b, #0\n"
      "mov z20.b, #0\n"
      "mov z21.b, #0\n"
      "mov z22.b, #0\n"
      "mov z23.b, #0\n"
      "mov z24.b, #0\n"
      "mov z25.b, #0\n"
      "mov z26.b, #0\n"
      "mov z27.b, #0\n"
      "mov z28.b, #0\n"
      "mov z29.b, #0\n"
      "mov z30.b, #0\n"
      "mov z31.b, #0\n"
      "57:"  // Height 5: setup done
      "mov x28, #0\n"
      "58:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 59f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 60f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "b 60f\n"
      "59:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "60:"  // Height 5: input setup done
      "cmp x27, #0x8\n"
      "ble 62f\n"
      "61:"  // Height 5: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "sub x27, x27, #0x8\n"
      "cmp x27, #0x8\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqh { z2.h }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqh { z3.h }, p0/Z, [x24]\n"
      "ld1rqh { z4.h }, p0/Z, [x23]\n"
      "ld1rqh { z5.h }, p0/Z, [x22]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "trn1 z0.d, z1.d, z2.d\n"
      "trn2 z1.d, z1.d, z2.d\n"
      "trn1 z2.d, z3.d, z4.d\n"
      "trn2 z3.d, z3.d, z4.d\n"
      "trn1 z4.d, z5.d, z6.d\n"
      "trn2 z5.d, z5.d, z6.d\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      ".inst 0x6467e450  // bfmmla z16.s, z2.h, z7.h\n"
      ".inst 0x6467e498  // bfmmla z24.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      ".inst 0x6466e454  // bfmmla z20.s, z2.h, z6.h\n"
      ".inst 0x6466e49c  // bfmmla z28.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      ".inst 0x6467e451  // bfmmla z17.s, z2.h, z7.h\n"
      ".inst 0x6467e499  // bfmmla z25.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      ".inst 0x6466e455  // bfmmla z21.s, z2.h, z6.h\n"
      ".inst 0x6466e49d  // bfmmla z29.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      ".inst 0x6467e452  // bfmmla z18.s, z2.h, z7.h\n"
      ".inst 0x6467e49a  // bfmmla z26.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      ".inst 0x6466e456  // bfmmla z22.s, z2.h, z6.h\n"
      ".inst 0x6466e49e  // bfmmla z30.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #16\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6467e453  // bfmmla z19.s, z2.h, z7.h\n"
      ".inst 0x6467e49b  // bfmmla z27.s, z4.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      ".inst 0x6466e457  // bfmmla z23.s, z2.h, z6.h\n"
      ".inst 0x6466e49f  // bfmmla z31.s, z4.h, z6.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-8, MUL VL]\n"
      "ld1h { z6.h }, p5/Z, [x11, #-7, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      ".inst 0x6467e470  // bfmmla z16.s, z3.h, z7.h\n"
      ".inst 0x6467e4b8  // bfmmla z24.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-6, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      ".inst 0x6466e474  // bfmmla z20.s, z3.h, z6.h\n"
      ".inst 0x6466e4bc  // bfmmla z28.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-5, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      ".inst 0x6467e471  // bfmmla z17.s, z3.h, z7.h\n"
      ".inst 0x6467e4b9  // bfmmla z25.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      ".inst 0x6466e475  // bfmmla z21.s, z3.h, z6.h\n"
      ".inst 0x6466e4bd  // bfmmla z29.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-3, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      ".inst 0x6467e472  // bfmmla z18.s, z3.h, z7.h\n"
      ".inst 0x6467e4ba  // bfmmla z26.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-2, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      ".inst 0x6466e476  // bfmmla z22.s, z3.h, z6.h\n"
      ".inst 0x6466e4be  // bfmmla z30.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-1, MUL VL]\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6467e473  // bfmmla z19.s, z3.h, z7.h\n"
      ".inst 0x6467e4bb  // bfmmla z27.s, z5.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      ".inst 0x6466e477  // bfmmla z23.s, z3.h, z6.h\n"
      ".inst 0x6466e4bf  // bfmmla z31.s, z5.h, z6.h\n"
      "bgt 61b\n"
      "62:"  // Height 5: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "ld1rqh { z2.h }, p0/Z, [x25]\n"
      "ld1rqh { z3.h }, p0/Z, [x24]\n"
      "ld1rqh { z4.h }, p0/Z, [x23]\n"
      "ld1rqh { z5.h }, p0/Z, [x22]\n"
      "trn1 z0.d, z1.d, z2.d\n"
      "trn2 z1.d, z1.d, z2.d\n"
      "trn1 z2.d, z3.d, z4.d\n"
      "trn2 z3.d, z3.d, z4.d\n"
      "trn1 z4.d, z5.d, z6.d\n"
      "trn2 z5.d, z5.d, z6.d\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      ".inst 0x6467e450  // bfmmla z16.s, z2.h, z7.h\n"
      ".inst 0x6467e498  // bfmmla z24.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      ".inst 0x6466e454  // bfmmla z20.s, z2.h, z6.h\n"
      ".inst 0x6466e49c  // bfmmla z28.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      ".inst 0x6467e451  // bfmmla z17.s, z2.h, z7.h\n"
      ".inst 0x6467e499  // bfmmla z25.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      ".inst 0x6466e455  // bfmmla z21.s, z2.h, z6.h\n"
      ".inst 0x6466e49d  // bfmmla z29.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      ".inst 0x6467e452  // bfmmla z18.s, z2.h, z7.h\n"
      ".inst 0x6467e49a  // bfmmla z26.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      ".inst 0x6466e456  // bfmmla z22.s, z2.h, z6.h\n"
      ".inst 0x6466e49e  // bfmmla z30.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6467e453  // bfmmla z19.s, z2.h, z7.h\n"
      ".inst 0x6467e49b  // bfmmla z27.s, z4.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      ".inst 0x6466e457  // bfmmla z23.s, z2.h, z6.h\n"
      ".inst 0x6466e49f  // bfmmla z31.s, z4.h, z6.h\n"
      "ble 63f\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      ".inst 0x6467e470  // bfmmla z16.s, z3.h, z7.h\n"
      ".inst 0x6467e4b8  // bfmmla z24.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      ".inst 0x6466e474  // bfmmla z20.s, z3.h, z6.h\n"
      ".inst 0x6466e4bc  // bfmmla z28.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      ".inst 0x6467e471  // bfmmla z17.s, z3.h, z7.h\n"
      ".inst 0x6467e4b9  // bfmmla z25.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      ".inst 0x6466e475  // bfmmla z21.s, z3.h, z6.h\n"
      ".inst 0x6466e4bd  // bfmmla z29.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      ".inst 0x6467e472  // bfmmla z18.s, z3.h, z7.h\n"
      ".inst 0x6467e4ba  // bfmmla z26.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      ".inst 0x6466e476  // bfmmla z22.s, z3.h, z6.h\n"
      ".inst 0x6466e4be  // bfmmla z30.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6467e473  // bfmmla z19.s, z3.h, z7.h\n"
      ".inst 0x6467e4bb  // bfmmla z27.s, z5.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      ".inst 0x6466e477  // bfmmla z23.s, z3.h, z6.h\n"
      ".inst 0x6466e4bf  // bfmmla z31.s, z5.h, z6.h\n"
      "63:"  // Height 5: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 58b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 z7.d, z8.d, z12.d\n"
      "uzp2 z8.d, z8.d, z12.d\n"
      "uzp1 z12.d, z9.d, z13.d\n"
      "uzp2 z9.d, z9.d, z13.d\n"
      "uzp1 z13.d, z10.d, z14.d\n"
      "uzp2 z10.d, z10.d, z14.d\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "uzp1 z14.d, z11.d, z15.d\n"
      "uzp2 z11.d, z11.d, z15.d\n"
      "uzp1 z15.d, z16.d, z20.d\n"
      "uzp2 z16.d, z16.d, z20.d\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 z20.d, z17.d, z21.d\n"
      "uzp2 z17.d, z17.d, z21.d\n"
      "uzp1 z21.d, z18.d, z22.d\n"
      "uzp2 z18.d, z18.d, z22.d\n"
      "uzp1 z22.d, z19.d, z23.d\n"
      "uzp2 z19.d, z19.d, z23.d\n"
      "uzp1 z24.d, z24.d, z28.d\n"
      "uzp1 z25.d, z25.d, z29.d\n"
      "uzp1 z26.d, z26.d, z30.d\n"
      "uzp1 z27.d, z27.d, z31.d\n"
      "tbz %x[flags], #1, 64f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z28.s }, p5/Z, [x21]\n"
      "ld1rw { z23.s }, p5/Z, [x20]\n"
      "fmin z7.s, p5/M, z7.s, z28.s\n"
      "fmin z12.s, p5/M, z12.s, z28.s\n"
      "fmin z13.s, p5/M, z13.s, z28.s\n"
      "fmin z14.s, p5/M, z14.s, z28.s\n"
      "fmin z8.s, p5/M, z8.s, z28.s\n"
      "fmin z9.s, p5/M, z9.s, z28.s\n"
      "fmin z10.s, p5/M, z10.s, z28.s\n"
      "fmin z11.s, p5/M, z11.s, z28.s\n"
      "fmin z15.s, p5/M, z15.s, z28.s\n"
      "fmin z20.s, p5/M, z20.s, z28.s\n"
      "fmin z21.s, p5/M, z21.s, z28.s\n"
      "fmin z22.s, p5/M, z22.s, z28.s\n"
      "fmin z16.s, p5/M, z16.s, z28.s\n"
      "fmin z17.s, p5/M, z17.s, z28.s\n"
      "fmin z18.s, p5/M, z18.s, z28.s\n"
      "fmin z19.s, p5/M, z19.s, z28.s\n"
      "fmin z24.s, p5/M, z24.s, z28.s\n"
      "fmin z25.s, p5/M, z25.s, z28.s\n"
      "fmin z26.s, p5/M, z26.s, z28.s\n"
      "fmin z27.s, p5/M, z27.s, z28.s\n"
      "fmax z7.s, p5/M, z7.s, z23.s\n"
      "fmax z12.s, p5/M, z12.s, z23.s\n"
      "fmax z13.s, p5/M, z13.s, z23.s\n"
      "fmax z14.s, p5/M, z14.s, z23.s\n"
      "fmax z8.s, p5/M, z8.s, z23.s\n"
      "fmax z9.s, p5/M, z9.s, z23.s\n"
      "fmax z10.s, p5/M, z10.s, z23.s\n"
      "fmax z11.s, p5/M, z11.s, z23.s\n"
      "fmax z15.s, p5/M, z15.s, z23.s\n"
      "fmax z20.s, p5/M, z20.s, z23.s\n"
      "fmax z21.s, p5/M, z21.s, z23.s\n"
      "fmax z22.s, p5/M, z22.s, z23.s\n"
      "fmax z16.s, p5/M, z16.s, z23.s\n"
      "fmax z17.s, p5/M, z17.s, z23.s\n"
      "fmax z18.s, p5/M, z18.s, z23.s\n"
      "fmax z19.s, p5/M, z19.s, z23.s\n"
      "fmax z24.s, p5/M, z24.s, z23.s\n"
      "fmax z25.s, p5/M, z25.s, z23.s\n"
      "fmax z26.s, p5/M, z26.s, z23.s\n"
      "fmax z27.s, p5/M, z27.s, z23.s\n"
      "64:"  // Height 5: No activation
      "st1w { z7.s }, p4, [x9]\n"
      "st1w { z12.s }, p3, [x9, #1, MUL VL]\n"
      "st1w { z13.s }, p2, [x9, #2, MUL VL]\n"
      "st1w { z14.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z8.s }, p4, [x26]\n"
      "st1w { z9.s }, p3, [x26, #1, MUL VL]\n"
      "st1w { z10.s }, p2, [x26, #2, MUL VL]\n"
      "st1w { z11.s }, p1, [x26, #3, MUL VL]\n"
      "st1w { z15.s }, p4, [x25]\n"
      "st1w { z20.s }, p3, [x25, #1, MUL VL]\n"
      "st1w { z21.s }, p2, [x25, #2, MUL VL]\n"
      "st1w { z22.s }, p1, [x25, #3, MUL VL]\n"
      "st1w { z16.s }, p4, [x24]\n"
      "st1w { z17.s }, p3, [x24, #1, MUL VL]\n"
      "st1w { z18.s }, p2, [x24, #2, MUL VL]\n"
      "st1w { z19.s }, p1, [x24, #3, MUL VL]\n"
      "st1w { z24.s }, p4, [x23]\n"
      "st1w { z25.s }, p3, [x23, #1, MUL VL]\n"
      "st1w { z26.s }, p2, [x23, #2, MUL VL]\n"
      "st1w { z27.s }, p1, [x23, #3, MUL VL]\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 54b\n"
      "b 80f\n"
      "66:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x18\n"
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "madd x20, x21, x20, x9\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "67:"  // Height 6: Column loop
      "mov x20, #0\n"
      "whilelt p4.s, x20, x12\n"
      "incw x20\n"
      "whilelt p3.s, x20, x12\n"
      "incw x20\n"
      "whilelt p2.s, x20, x12\n"
      "incw x20\n"
      "whilelt p1.s, x20, x12\n"
      "cbz x10, 68f\n"
      "ld1w { z8.s }, p5/Z, [x10]\n"
      "ld1w { z9.s }, p5/Z, [x10, #1, MUL VL]\n"
      "ld1w { z10.s }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1w { z11.s }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "zip2 z12.d, z8.d, z8.d\n"
      "zip1 z8.d, z8.d, z8.d\n"
      "zip2 z13.d, z9.d, z9.d\n"
      "zip1 z9.d, z9.d, z9.d\n"
      "zip2 z14.d, z10.d, z10.d\n"
      "zip1 z10.d, z10.d, z10.d\n"
      "zip2 z15.d, z11.d, z11.d\n"
      "zip1 z11.d, z11.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z20.d, z12.d\n"
      "mov z17.d, z9.d\n"
      "mov z21.d, z13.d\n"
      "mov z18.d, z10.d\n"
      "mov z22.d, z14.d\n"
      "mov z19.d, z11.d\n"
      "mov z23.d, z15.d\n"
      "mov z24.d, z8.d\n"
      "mov z28.d, z12.d\n"
      "mov z25.d, z9.d\n"
      "mov z29.d, z13.d\n"
      "mov z26.d, z10.d\n"
      "mov z30.d, z14.d\n"
      "mov z27.d, z11.d\n"
      "mov z31.d, z15.d\n"
      "b 70f\n"
      "68:"  // Height 6: no bias
      "tbz %x[flags], #0, 69f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z9.s }, p4/Z, [x9]\n"
      "ld1w { z10.s }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1w { z11.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1w { z16.s }, p1/Z, [x9, #3, MUL VL]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "ld1w { z12.s }, p4/Z, [x24]\n"
      "ld1w { z13.s }, p3/Z, [x24, #1, MUL VL]\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z14.s }, p2/Z, [x24, #2, MUL VL]\n"
      "ld1w { z15.s }, p1/Z, [x24, #3, MUL VL]\n"
      "ld1w { z17.s }, p4/Z, [x23]\n"
      "ld1w { z18.s }, p3/Z, [x23, #1, MUL VL]\n"
      "ld1w { z19.s }, p2/Z, [x23, #2, MUL VL]\n"
      "ld1w { z24.s }, p1/Z, [x23, #3, MUL VL]\n"
      "zip1 z8.d, z9.d, z12.d\n"
      "zip2 z12.d, z9.d, z12.d\n"
      "ld1w { z20.s }, p4/Z, [x22]\n"
      "ld1w { z21.s }, p3/Z, [x22, #1, MUL VL]\n"
      "zip1 z9.d, z10.d, z13.d\n"
      "zip2 z13.d, z10.d, z13.d\n"
      "ld1w { z22.s }, p2/Z, [x22, #2, MUL VL]\n"
      "ld1w { z23.s }, p1/Z, [x22, #3, MUL VL]\n"
      "zip1 z10.d, z11.d, z14.d\n"
      "zip2 z14.d, z11.d, z14.d\n"
      "ld1w { z25.s }, p4/Z, [x21]\n"
      "ld1w { z26.s }, p3/Z, [x21, #1, MUL VL]\n"
      "zip1 z11.d, z16.d, z15.d\n"
      "zip2 z15.d, z16.d, z15.d\n"
      "ld1w { z27.s }, p2/Z, [x21, #2, MUL VL]\n"
      "ld1w { z6.s }, p1/Z, [x21, #3, MUL VL]\n"
      "zip1 z16.d, z17.d, z20.d\n"
      "zip2 z20.d, z17.d, z20.d\n"
      "ld1w { z28.s }, p4/Z, [x20]\n"
      "ld1w { z29.s }, p3/Z, [x20, #1, MUL VL]\n"
      "zip1 z17.d, z18.d, z21.d\n"
      "zip2 z21.d, z18.d, z21.d\n"
      "ld1w { z30.s }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1w { z31.s }, p1/Z, [x20, #3, MUL VL]\n"
      "zip1 z18.d, z19.d, z22.d\n"
      "zip2 z22.d, z19.d, z22.d\n"
      "zip1 z19.d, z24.d, z23.d\n"
      "zip2 z23.d, z24.d, z23.d\n"
      "zip1 z24.d, z25.d, z28.d\n"
      "zip2 z28.d, z25.d, z28.d\n"
      "zip1 z25.d, z26.d, z29.d\n"
      "zip2 z29.d, z26.d, z29.d\n"
      "zip1 z26.d, z27.d, z30.d\n"
      "zip2 z30.d, z27.d, z30.d\n"
      "zip1 z27.d, z6.d, z31.d\n"
      "zip2 z31.d, z6.d, z31.d\n"
      "b 70f\n"
      "69:"  // Height 6: no accumulate
      "mov z8.b, #0\n"
      "mov z9.b, #0\n"
      "mov z10.b, #0\n"
      "mov z11.b, #0\n"
      "mov z12.b, #0\n"
      "mov z13.b, #0\n"
      "mov z14.b, #0\n"
      "mov z15.b, #0\n"
      "mov z16.b, #0\n"
      "mov z17.b, #0\n"
      "mov z18.b, #0\n"
      "mov z19.b, #0\n"
      "mov z20.b, #0\n"
      "mov z21.b, #0\n"
      "mov z22.b, #0\n"
      "mov z23.b, #0\n"
      "mov z24.b, #0\n"
      "mov z25.b, #0\n"
      "mov z26.b, #0\n"
      "mov z27.b, #0\n"
      "mov z28.b, #0\n"
      "mov z29.b, #0\n"
      "mov z30.b, #0\n"
      "mov z31.b, #0\n"
      "70:"  // Height 6: setup done
      "mov x28, #0\n"
      "71:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 72f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 73f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "add x21, x21, x20, LSL #1\n"
      "b 73f\n"
      "72:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "add x21, x22, x21, LSL #1\n"
      "73:"  // Height 6: input setup done
      "cmp x27, #0x8\n"
      "ble 75f\n"
      "74:"  // Height 6: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "sub x27, x27, #0x8\n"
      "cmp x27, #0x8\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqh { z2.h }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqh { z3.h }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "ld1rqh { z4.h }, p0/Z, [x23]\n"
      "ld1rqh { z5.h }, p0/Z, [x22]\n"
      "ld1rqh { z6.h }, p0/Z, [x21]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "trn1 z0.d, z1.d, z2.d\n"
      "trn2 z1.d, z1.d, z2.d\n"
      "add x21, x21, #0x10\n"
      "trn1 z2.d, z3.d, z4.d\n"
      "trn2 z3.d, z3.d, z4.d\n"
      "trn1 z4.d, z5.d, z6.d\n"
      "trn2 z5.d, z5.d, z6.d\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      ".inst 0x6467e450  // bfmmla z16.s, z2.h, z7.h\n"
      ".inst 0x6467e498  // bfmmla z24.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      ".inst 0x6466e454  // bfmmla z20.s, z2.h, z6.h\n"
      ".inst 0x6466e49c  // bfmmla z28.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      ".inst 0x6467e451  // bfmmla z17.s, z2.h, z7.h\n"
      ".inst 0x6467e499  // bfmmla z25.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      ".inst 0x6466e455  // bfmmla z21.s, z2.h, z6.h\n"
      ".inst 0x6466e49d  // bfmmla z29.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      ".inst 0x6467e452  // bfmmla z18.s, z2.h, z7.h\n"
      ".inst 0x6467e49a  // bfmmla z26.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      ".inst 0x6466e456  // bfmmla z22.s, z2.h, z6.h\n"
      ".inst 0x6466e49e  // bfmmla z30.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #16\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6467e453  // bfmmla z19.s, z2.h, z7.h\n"
      ".inst 0x6467e49b  // bfmmla z27.s, z4.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      ".inst 0x6466e457  // bfmmla z23.s, z2.h, z6.h\n"
      ".inst 0x6466e49f  // bfmmla z31.s, z4.h, z6.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-8, MUL VL]\n"
      "ld1h { z6.h }, p5/Z, [x11, #-7, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      ".inst 0x6467e470  // bfmmla z16.s, z3.h, z7.h\n"
      ".inst 0x6467e4b8  // bfmmla z24.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-6, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      ".inst 0x6466e474  // bfmmla z20.s, z3.h, z6.h\n"
      ".inst 0x6466e4bc  // bfmmla z28.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-5, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      ".inst 0x6467e471  // bfmmla z17.s, z3.h, z7.h\n"
      ".inst 0x6467e4b9  // bfmmla z25.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      ".inst 0x6466e475  // bfmmla z21.s, z3.h, z6.h\n"
      ".inst 0x6466e4bd  // bfmmla z29.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-3, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      ".inst 0x6467e472  // bfmmla z18.s, z3.h, z7.h\n"
      ".inst 0x6467e4ba  // bfmmla z26.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #-2, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      ".inst 0x6466e476  // bfmmla z22.s, z3.h, z6.h\n"
      ".inst 0x6466e4be  // bfmmla z30.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #-1, MUL VL]\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6467e473  // bfmmla z19.s, z3.h, z7.h\n"
      ".inst 0x6467e4bb  // bfmmla z27.s, z5.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      ".inst 0x6466e477  // bfmmla z23.s, z3.h, z6.h\n"
      ".inst 0x6466e4bf  // bfmmla z31.s, z5.h, z6.h\n"
      "bgt 74b\n"
      "75:"  // Height 6: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "ld1rqh { z2.h }, p0/Z, [x25]\n"
      "ld1rqh { z3.h }, p0/Z, [x24]\n"
      "ld1rqh { z4.h }, p0/Z, [x23]\n"
      "ld1rqh { z5.h }, p0/Z, [x22]\n"
      "ld1rqh { z6.h }, p0/Z, [x21]\n"
      "trn1 z0.d, z1.d, z2.d\n"
      "trn2 z1.d, z1.d, z2.d\n"
      "trn1 z2.d, z3.d, z4.d\n"
      "trn2 z3.d, z3.d, z4.d\n"
      "trn1 z4.d, z5.d, z6.d\n"
      "trn2 z5.d, z5.d, z6.d\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      ".inst 0x6467e408  // bfmmla z8.s, z0.h, z7.h\n"
      ".inst 0x6467e450  // bfmmla z16.s, z2.h, z7.h\n"
      ".inst 0x6467e498  // bfmmla z24.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e40c  // bfmmla z12.s, z0.h, z6.h\n"
      ".inst 0x6466e454  // bfmmla z20.s, z2.h, z6.h\n"
      ".inst 0x6466e49c  // bfmmla z28.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e409  // bfmmla z9.s, z0.h, z7.h\n"
      ".inst 0x6467e451  // bfmmla z17.s, z2.h, z7.h\n"
      ".inst 0x6467e499  // bfmmla z25.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e40d  // bfmmla z13.s, z0.h, z6.h\n"
      ".inst 0x6466e455  // bfmmla z21.s, z2.h, z6.h\n"
      ".inst 0x6466e49d  // bfmmla z29.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e40a  // bfmmla z10.s, z0.h, z7.h\n"
      ".inst 0x6467e452  // bfmmla z18.s, z2.h, z7.h\n"
      ".inst 0x6467e49a  // bfmmla z26.s, z4.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e40e  // bfmmla z14.s, z0.h, z6.h\n"
      ".inst 0x6466e456  // bfmmla z22.s, z2.h, z6.h\n"
      ".inst 0x6466e49e  // bfmmla z30.s, z4.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e40b  // bfmmla z11.s, z0.h, z7.h\n"
      ".inst 0x6467e453  // bfmmla z19.s, z2.h, z7.h\n"
      ".inst 0x6467e49b  // bfmmla z27.s, z4.h, z7.h\n"
      ".inst 0x6466e40f  // bfmmla z15.s, z0.h, z6.h\n"
      ".inst 0x6466e457  // bfmmla z23.s, z2.h, z6.h\n"
      ".inst 0x6466e49f  // bfmmla z31.s, z4.h, z6.h\n"
      "ble 76f\n"
      "ld1h { z7.h }, p5/Z, [x11]\n"
      "ld1h { z6.h }, p5/Z, [x11, #1, MUL VL]\n"
      ".inst 0x6467e428  // bfmmla z8.s, z1.h, z7.h\n"
      ".inst 0x6467e470  // bfmmla z16.s, z3.h, z7.h\n"
      ".inst 0x6467e4b8  // bfmmla z24.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #2, MUL VL]\n"
      ".inst 0x6466e42c  // bfmmla z12.s, z1.h, z6.h\n"
      ".inst 0x6466e474  // bfmmla z20.s, z3.h, z6.h\n"
      ".inst 0x6466e4bc  // bfmmla z28.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #3, MUL VL]\n"
      ".inst 0x6467e429  // bfmmla z9.s, z1.h, z7.h\n"
      ".inst 0x6467e471  // bfmmla z17.s, z3.h, z7.h\n"
      ".inst 0x6467e4b9  // bfmmla z25.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #4, MUL VL]\n"
      ".inst 0x6466e42d  // bfmmla z13.s, z1.h, z6.h\n"
      ".inst 0x6466e475  // bfmmla z21.s, z3.h, z6.h\n"
      ".inst 0x6466e4bd  // bfmmla z29.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #5, MUL VL]\n"
      ".inst 0x6467e42a  // bfmmla z10.s, z1.h, z7.h\n"
      ".inst 0x6467e472  // bfmmla z18.s, z3.h, z7.h\n"
      ".inst 0x6467e4ba  // bfmmla z26.s, z5.h, z7.h\n"
      "ld1h { z7.h }, p5/Z, [x11, #6, MUL VL]\n"
      ".inst 0x6466e42e  // bfmmla z14.s, z1.h, z6.h\n"
      ".inst 0x6466e476  // bfmmla z22.s, z3.h, z6.h\n"
      ".inst 0x6466e4be  // bfmmla z30.s, z5.h, z6.h\n"
      "ld1h { z6.h }, p5/Z, [x11, #7, MUL VL]\n"
      "addvl x11, x11, #8\n"
      ".inst 0x6467e42b  // bfmmla z11.s, z1.h, z7.h\n"
      ".inst 0x6467e473  // bfmmla z19.s, z3.h, z7.h\n"
      ".inst 0x6467e4bb  // bfmmla z27.s, z5.h, z7.h\n"
      ".inst 0x6466e42f  // bfmmla z15.s, z1.h, z6.h\n"
      ".inst 0x6466e477  // bfmmla z23.s, z3.h, z6.h\n"
      ".inst 0x6466e4bf  // bfmmla z31.s, z5.h, z6.h\n"
      "76:"  // Height 6: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 71b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 z7.d, z8.d, z12.d\n"
      "uzp2 z8.d, z8.d, z12.d\n"
      "uzp1 z12.d, z9.d, z13.d\n"
      "uzp2 z9.d, z9.d, z13.d\n"
      "uzp1 z13.d, z10.d, z14.d\n"
      "uzp2 z10.d, z10.d, z14.d\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "uzp1 z14.d, z11.d, z15.d\n"
      "uzp2 z11.d, z11.d, z15.d\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 z15.d, z16.d, z20.d\n"
      "uzp2 z16.d, z16.d, z20.d\n"
      "add x22, x23, x20, LSL #2\n"
      "uzp1 z20.d, z17.d, z21.d\n"
      "uzp2 z17.d, z17.d, z21.d\n"
      "uzp1 z21.d, z18.d, z22.d\n"
      "uzp2 z18.d, z18.d, z22.d\n"
      "uzp1 z22.d, z19.d, z23.d\n"
      "uzp2 z19.d, z19.d, z23.d\n"
      "uzp1 z23.d, z24.d, z28.d\n"
      "uzp2 z24.d, z24.d, z28.d\n"
      "uzp1 z28.d, z25.d, z29.d\n"
      "uzp2 z25.d, z25.d, z29.d\n"
      "uzp1 z29.d, z26.d, z30.d\n"
      "uzp2 z26.d, z26.d, z30.d\n"
      "uzp1 z30.d, z27.d, z31.d\n"
      "uzp2 z27.d, z27.d, z31.d\n"
      "tbz %x[flags], #1, 77f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rw { z0.s }, p5/Z, [x21]\n"
      "ld1rw { z31.s }, p5/Z, [x20]\n"
      "fmin z7.s, p5/M, z7.s, z0.s\n"
      "fmin z12.s, p5/M, z12.s, z0.s\n"
      "fmin z13.s, p5/M, z13.s, z0.s\n"
      "fmin z14.s, p5/M, z14.s, z0.s\n"
      "fmin z8.s, p5/M, z8.s, z0.s\n"
      "fmin z9.s, p5/M, z9.s, z0.s\n"
      "fmin z10.s, p5/M, z10.s, z0.s\n"
      "fmin z11.s, p5/M, z11.s, z0.s\n"
      "fmin z15.s, p5/M, z15.s, z0.s\n"
      "fmin z20.s, p5/M, z20.s, z0.s\n"
      "fmin z21.s, p5/M, z21.s, z0.s\n"
      "fmin z22.s, p5/M, z22.s, z0.s\n"
      "fmin z16.s, p5/M, z16.s, z0.s\n"
      "fmin z17.s, p5/M, z17.s, z0.s\n"
      "fmin z18.s, p5/M, z18.s, z0.s\n"
      "fmin z19.s, p5/M, z19.s, z0.s\n"
      "fmin z23.s, p5/M, z23.s, z0.s\n"
      "fmin z28.s, p5/M, z28.s, z0.s\n"
      "fmin z29.s, p5/M, z29.s, z0.s\n"
      "fmin z30.s, p5/M, z30.s, z0.s\n"
      "fmin z24.s, p5/M, z24.s, z0.s\n"
      "fmin z25.s, p5/M, z25.s, z0.s\n"
      "fmin z26.s, p5/M, z26.s, z0.s\n"
      "fmin z27.s, p5/M, z27.s, z0.s\n"
      "fmax z7.s, p5/M, z7.s, z31.s\n"
      "fmax z12.s, p5/M, z12.s, z31.s\n"
      "fmax z13.s, p5/M, z13.s, z31.s\n"
      "fmax z14.s, p5/M, z14.s, z31.s\n"
      "fmax z8.s, p5/M, z8.s, z31.s\n"
      "fmax z9.s, p5/M, z9.s, z31.s\n"
      "fmax z10.s, p5/M, z10.s, z31.s\n"
      "fmax z11.s, p5/M, z11.s, z31.s\n"
      "fmax z15.s, p5/M, z15.s, z31.s\n"
      "fmax z20.s, p5/M, z20.s, z31.s\n"
      "fmax z21.s, p5/M, z21.s, z31.s\n"
      "fmax z22.s, p5/M, z22.s, z31.s\n"
      "fmax z16.s, p5/M, z16.s, z31.s\n"
      "fmax z17.s, p5/M, z17.s, z31.s\n"
      "fmax z18.s, p5/M, z18.s, z31.s\n"
      "fmax z19.s, p5/M, z19.s, z31.s\n"
      "fmax z23.s, p5/M, z23.s, z31.s\n"
      "fmax z28.s, p5/M, z28.s, z31.s\n"
      "fmax z29.s, p5/M, z29.s, z31.s\n"
      "fmax z30.s, p5/M, z30.s, z31.s\n"
      "fmax z24.s, p5/M, z24.s, z31.s\n"
      "fmax z25.s, p5/M, z25.s, z31.s\n"
      "fmax z26.s, p5/M, z26.s, z31.s\n"
      "fmax z27.s, p5/M, z27.s, z31.s\n"
      "77:"  // Height 6: No activation
      "st1w { z7.s }, p4, [x9]\n"
      "st1w { z12.s }, p3, [x9, #1, MUL VL]\n"
      "st1w { z13.s }, p2, [x9, #2, MUL VL]\n"
      "st1w { z14.s }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z8.s }, p4, [x26]\n"
      "st1w { z9.s }, p3, [x26, #1, MUL VL]\n"
      "st1w { z10.s }, p2, [x26, #2, MUL VL]\n"
      "st1w { z11.s }, p1, [x26, #3, MUL VL]\n"
      "st1w { z15.s }, p4, [x25]\n"
      "st1w { z20.s }, p3, [x25, #1, MUL VL]\n"
      "st1w { z21.s }, p2, [x25, #2, MUL VL]\n"
      "st1w { z22.s }, p1, [x25, #3, MUL VL]\n"
      "st1w { z16.s }, p4, [x24]\n"
      "st1w { z17.s }, p3, [x24, #1, MUL VL]\n"
      "st1w { z18.s }, p2, [x24, #2, MUL VL]\n"
      "st1w { z19.s }, p1, [x24, #3, MUL VL]\n"
      "st1w { z23.s }, p4, [x23]\n"
      "st1w { z28.s }, p3, [x23, #1, MUL VL]\n"
      "st1w { z29.s }, p2, [x23, #2, MUL VL]\n"
      "st1w { z30.s }, p1, [x23, #3, MUL VL]\n"
      "st1w { z24.s }, p4, [x22]\n"
      "st1w { z25.s }, p3, [x22, #1, MUL VL]\n"
      "st1w { z26.s }, p2, [x22, #2, MUL VL]\n"
      "st1w { z27.s }, p1, [x22, #3, MUL VL]\n"
      "decw x12, ALL, MUL #4\n"
      "cmp x12, XZR\n"
      "bgt 67b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 80f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 79f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "79:"  // Update direct input
      "mov x20, #0xc\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "80:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_maxval] "I" (offsetof(KernelArgs, maxval)), [offsetof_minval] "I" (offsetof(KernelArgs, minval)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "x9", "x10", "x11", "x12", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm

#endif // defined(ARM_COMPUTE_ENABLE_BF16) && defined(ARM_COMPUTE_ENABLE_SVE) && defined(__aarch64__)

