/*
 * Copyright (c) 2025-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(ARM_COMPUTE_ENABLE_SME) && defined(__aarch64__)

#include "arm_gemm/arm_gemm.hpp"


#include "arm_common/internal/utils.hpp"

namespace arm_gemm {

void sme_interleaved_nomerge_fp16fp32_mopa_4VLx1VL(const __fp16 *const A, const __fp16 *const B, float *const C, int ldc, const int M, const int N, const int K, const float *const bias, const Activation act, bool accumulate, float *const accumulator_buffer)
{
  struct KernelArgs
  {
    KernelArgs(
      const __fp16 *const A,
      const __fp16 *const B,
      float *const C, const int ldc,
      const int M, const int N, const int K,
      const float *const bias,
      const Activation act,
      bool accumulate,
      float *const accumulator_buffer
    ) : A(A),
        B(B), kstride_bytes(roundup(K, 2) * sizeof(__fp16)),
        C(C), ldcb(ldc * sizeof(float)),
        M(M), N(N), K(K),
        min(-std::numeric_limits<float>::infinity()),
        max(std::numeric_limits<float>::infinity()),
        bias(bias),
        accumulator_buffer(accumulator_buffer),
        flags(0x0)
    {
      if (accumulate)
      {
        flags |= 1 << 0;  // FILL_ACCUMULATORS_FROM_BUFFER
      }
      if (C == nullptr)
      {
        flags |= 1 << 1;  // STORE_ACCUMULATORS_TO_BUFFER
      }
      if (act.type == Activation::Type::None)
      {
        flags |= 1 << 2;  // SKIP_ACTIVATION
      }

      // Initialise the activation values
      switch (act.type)
      {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            this->max = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            this->min = static_cast<float>(0);
            break;
      }
    }

    const __fp16 *const A;
    const __fp16 *const B;
    const long kstride_bytes;
    float *const C;
    const long ldcb;
    const long M, N, K;
    float min = -std::numeric_limits<float>::infinity();
    float max = std::numeric_limits<float>::infinity();

    const float *const bias;


    float *const accumulator_buffer;
    uint64_t flags;
  };

  // Construct arguments for this kernel
  KernelArgs args(A, B, C, ldc, M, N, K, bias, act, accumulate, accumulator_buffer);

  __asm__ __volatile__(
      "ldr x5, [%x[args], %[offsetof_flags]]\n"
      ".inst 0xd503477f  // SMSTART ZA\n"
      "ptrue p1.b\n"
      "ldr x6, [%x[args], %[offsetof_accumulator_buffer]]\n"
      "ldr x7, [%x[args], %[offsetof_accumulator_buffer]]\n"
      "tbz x5, #0, 2f\n"
      "mov x12, #0\n"
      "cntw x26\n"
      "cntw x25\n"
      "cntw x24, ALL, MUL #2\n"
      "cntw x23, ALL, MUL #3\n"
      "1:"  // Initial accumulator load from buffer: Loop
      "addvl x22, x6, #4\n"
      "addvl x21, x6, #8\n"
      ".inst 0xe09f04c0  // ld1w { za0h.s[x12] }, p1/Z, [x6, XZR, LSL #2]\n"
      "addvl x20, x6, #12\n"
      ".inst 0xe09f06c4  // ld1w { za1h.s[x12] }, p1/Z, [x22, XZR, LSL #2]\n"
      ".inst 0xe09f06a8  // ld1w { za2h.s[x12] }, p1/Z, [x21, XZR, LSL #2]\n"
      ".inst 0xe09f068c  // ld1w { za3h.s[x12] }, p1/Z, [x20, XZR, LSL #2]\n"
      ".inst 0xe09904c1  // ld1w { za0h.s[x12, #1] }, p1/Z, [x6, x25, LSL #2]\n"
      ".inst 0xe09906c5  // ld1w { za1h.s[x12, #1] }, p1/Z, [x22, x25, LSL #2]\n"
      ".inst 0xe09906a9  // ld1w { za2h.s[x12, #1] }, p1/Z, [x21, x25, LSL #2]\n"
      ".inst 0xe099068d  // ld1w { za3h.s[x12, #1] }, p1/Z, [x20, x25, LSL #2]\n"
      ".inst 0xe09804c2  // ld1w { za0h.s[x12, #2] }, p1/Z, [x6, x24, LSL #2]\n"
      ".inst 0xe09806c6  // ld1w { za1h.s[x12, #2] }, p1/Z, [x22, x24, LSL #2]\n"
      ".inst 0xe09806aa  // ld1w { za2h.s[x12, #2] }, p1/Z, [x21, x24, LSL #2]\n"
      ".inst 0xe098068e  // ld1w { za3h.s[x12, #2] }, p1/Z, [x20, x24, LSL #2]\n"
      ".inst 0xe09704c3  // ld1w { za0h.s[x12, #3] }, p1/Z, [x6, x23, LSL #2]\n"
      "addvl x6, x6, #16\n"
      ".inst 0xe09706c7  // ld1w { za1h.s[x12, #3] }, p1/Z, [x22, x23, LSL #2]\n"
      ".inst 0xe09706ab  // ld1w { za2h.s[x12, #3] }, p1/Z, [x21, x23, LSL #2]\n"
      ".inst 0xe097068f  // ld1w { za3h.s[x12, #3] }, p1/Z, [x20, x23, LSL #2]\n"
      "add x12, x12, #0x4\n"
      "cmp x12, x26\n"
      "blt 1b\n"
      "2:"  // Initial accumulator load from buffer: End
      "ldr x8, [%x[args], %[offsetof_K]]\n"
      "mov x17, #0\n"
      "mov x16, #0\n"
      "ldr w15, [%x[args], %[offsetof_M]]\n"
      "ldr w14, [%x[args], %[offsetof_N]]\n"
      "add x8, x8, #0x1\n"
      "ldr x13, [%x[args], %[offsetof_A]]\n"
      "lsr x8, x8, #0x1\n"
      "3:"  // M loop
      "ldr x11, [%x[args], %[offsetof_B]]\n"
      "4:"  // N loop
      "mov x10, x13\n"
      "whilelt p0.s, x16, x14\n"
      "tbnz x5, #0, 5f\n"
      "ldr x20, [%x[args], %[offsetof_bias]]\n"
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "cbz x20, 6f\n"
      "fmov z17.s, #1.0\n"
      "ld1w { z16.s }, p0/Z, [x20, x16, LSL #2]\n"
      ".inst 0x80902620  // fmopa za0.s, p1/M, p1/M, z17.s, z16.s\n"
      ".inst 0x80902621  // fmopa za1.s, p1/M, p1/M, z17.s, z16.s\n"
      ".inst 0x80902622  // fmopa za2.s, p1/M, p1/M, z17.s, z16.s\n"
      ".inst 0x80902623  // fmopa za3.s, p1/M, p1/M, z17.s, z16.s\n"
      "5:"  // Prepare accumulators: Test for last block
      "mov x20, x16\n"
      "mov x21, x17\n"
      "incw x20\n"
      "incw x21, ALL, MUL #4\n"
      "cmp x20, x14\n"
      "mov x20, x5\n"
      "csel x21, x17, x21, LT\n"
      "bfm x5, XZR, #0, #0  // bfc x5, #0, #0x1\n"
      "cmp x21, x15\n"
      "csel x5, x20, x5, LT\n"
      "6:"  // Prepare accumulators: End
      "lsr x23, x8, #0x2\n"
      "and x22, x8, #0x3\n"
      "cbz x23, 9f\n"
      "addvl x21, x10, #8\n"
      "addvl x20, x10, #12\n"
      "ld1h { z3.h }, p1/Z, [x10]\n"
      "subs x23, x23, #0x1\n"
      "ld1h { z2.h }, p1/Z, [x10, #1, MUL VL]\n"
      "ld1h { z1.h }, p1/Z, [x10, #2, MUL VL]\n"
      "ld1h { z0.h }, p1/Z, [x10, #3, MUL VL]\n"
      "ld1h { z31.h }, p1/Z, [x10, #4, MUL VL]\n"
      "ld1h { z30.h }, p1/Z, [x10, #5, MUL VL]\n"
      "ld1h { z29.h }, p1/Z, [x10, #6, MUL VL]\n"
      "ld1h { z28.h }, p1/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "ld1h { z27.h }, p1/Z, [x21]\n"
      "ld1h { z26.h }, p1/Z, [x21, #1, MUL VL]\n"
      "ld1h { z25.h }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1h { z24.h }, p1/Z, [x21, #3, MUL VL]\n"
      "ld1h { z23.h }, p1/Z, [x20]\n"
      "ld1h { z22.h }, p1/Z, [x20, #1, MUL VL]\n"
      "ld1h { z21.h }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1h { z20.h }, p1/Z, [x20, #3, MUL VL]\n"
      "ld1h { z19.h }, p1/Z, [x11]\n"
      "ld1h { z18.h }, p1/Z, [x11, #1, MUL VL]\n"
      "ld1h { z17.h }, p1/Z, [x11, #2, MUL VL]\n"
      "ld1h { z16.h }, p1/Z, [x11, #3, MUL VL]\n"
      "addvl x11, x11, #4\n"
      "ble 8f\n"
      "7:"  // K loop
      ".inst 0x81b32460  // fmopa za0.s, p1/M, p1/M, z3.h, z19.h\n"
      "addvl x21, x10, #8\n"
      "addvl x20, x10, #12\n"
      "ld1h { z3.h }, p1/Z, [x10]\n"
      ".inst 0x81b32441  // fmopa za1.s, p1/M, p1/M, z2.h, z19.h\n"
      "subs x23, x23, #0x1\n"
      "ld1h { z2.h }, p1/Z, [x10, #1, MUL VL]\n"
      ".inst 0x81b32422  // fmopa za2.s, p1/M, p1/M, z1.h, z19.h\n"
      "ld1h { z1.h }, p1/Z, [x10, #2, MUL VL]\n"
      ".inst 0x81b32403  // fmopa za3.s, p1/M, p1/M, z0.h, z19.h\n"
      "ld1h { z0.h }, p1/Z, [x10, #3, MUL VL]\n"
      ".inst 0x81b227e0  // fmopa za0.s, p1/M, p1/M, z31.h, z18.h\n"
      "ld1h { z31.h }, p1/Z, [x10, #4, MUL VL]\n"
      ".inst 0x81b227c1  // fmopa za1.s, p1/M, p1/M, z30.h, z18.h\n"
      "ld1h { z30.h }, p1/Z, [x10, #5, MUL VL]\n"
      ".inst 0x81b227a2  // fmopa za2.s, p1/M, p1/M, z29.h, z18.h\n"
      "ld1h { z29.h }, p1/Z, [x10, #6, MUL VL]\n"
      ".inst 0x81b22783  // fmopa za3.s, p1/M, p1/M, z28.h, z18.h\n"
      "ld1h { z28.h }, p1/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      ".inst 0x81b12760  // fmopa za0.s, p1/M, p1/M, z27.h, z17.h\n"
      "ld1h { z27.h }, p1/Z, [x21]\n"
      ".inst 0x81b12741  // fmopa za1.s, p1/M, p1/M, z26.h, z17.h\n"
      "ld1h { z26.h }, p1/Z, [x21, #1, MUL VL]\n"
      ".inst 0x81b12722  // fmopa za2.s, p1/M, p1/M, z25.h, z17.h\n"
      "ld1h { z25.h }, p1/Z, [x21, #2, MUL VL]\n"
      ".inst 0x81b12703  // fmopa za3.s, p1/M, p1/M, z24.h, z17.h\n"
      "ld1h { z24.h }, p1/Z, [x21, #3, MUL VL]\n"
      ".inst 0x81b026e0  // fmopa za0.s, p1/M, p1/M, z23.h, z16.h\n"
      "ld1h { z23.h }, p1/Z, [x20]\n"
      ".inst 0x81b026c1  // fmopa za1.s, p1/M, p1/M, z22.h, z16.h\n"
      "ld1h { z22.h }, p1/Z, [x20, #1, MUL VL]\n"
      ".inst 0x81b026a2  // fmopa za2.s, p1/M, p1/M, z21.h, z16.h\n"
      "ld1h { z21.h }, p1/Z, [x20, #2, MUL VL]\n"
      ".inst 0x81b02683  // fmopa za3.s, p1/M, p1/M, z20.h, z16.h\n"
      "ld1h { z20.h }, p1/Z, [x20, #3, MUL VL]\n"
      "ld1h { z19.h }, p1/Z, [x11]\n"
      "ld1h { z18.h }, p1/Z, [x11, #1, MUL VL]\n"
      "ld1h { z17.h }, p1/Z, [x11, #2, MUL VL]\n"
      "ld1h { z16.h }, p1/Z, [x11, #3, MUL VL]\n"
      "addvl x11, x11, #4\n"
      "bgt 7b\n"
      "8:"  // K loop tail
      ".inst 0x81b32460  // fmopa za0.s, p1/M, p1/M, z3.h, z19.h\n"
      ".inst 0x81b32441  // fmopa za1.s, p1/M, p1/M, z2.h, z19.h\n"
      ".inst 0x81b32422  // fmopa za2.s, p1/M, p1/M, z1.h, z19.h\n"
      ".inst 0x81b32403  // fmopa za3.s, p1/M, p1/M, z0.h, z19.h\n"
      ".inst 0x81b227e0  // fmopa za0.s, p1/M, p1/M, z31.h, z18.h\n"
      ".inst 0x81b227c1  // fmopa za1.s, p1/M, p1/M, z30.h, z18.h\n"
      ".inst 0x81b227a2  // fmopa za2.s, p1/M, p1/M, z29.h, z18.h\n"
      ".inst 0x81b22783  // fmopa za3.s, p1/M, p1/M, z28.h, z18.h\n"
      ".inst 0x81b12760  // fmopa za0.s, p1/M, p1/M, z27.h, z17.h\n"
      ".inst 0x81b12741  // fmopa za1.s, p1/M, p1/M, z26.h, z17.h\n"
      ".inst 0x81b12722  // fmopa za2.s, p1/M, p1/M, z25.h, z17.h\n"
      ".inst 0x81b12703  // fmopa za3.s, p1/M, p1/M, z24.h, z17.h\n"
      ".inst 0x81b026e0  // fmopa za0.s, p1/M, p1/M, z23.h, z16.h\n"
      ".inst 0x81b026c1  // fmopa za1.s, p1/M, p1/M, z22.h, z16.h\n"
      ".inst 0x81b026a2  // fmopa za2.s, p1/M, p1/M, z21.h, z16.h\n"
      ".inst 0x81b02683  // fmopa za3.s, p1/M, p1/M, z20.h, z16.h\n"
      "9:"  // K oddments
      "cbz x22, 11f\n"
      "10:"  // K oddments: Loop
      "ld1h { z20.h }, p1/Z, [x10]\n"
      "subs x22, x22, #0x1\n"
      "ld1h { z19.h }, p1/Z, [x10, #1, MUL VL]\n"
      "ld1h { z18.h }, p1/Z, [x10, #2, MUL VL]\n"
      "ld1h { z17.h }, p1/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "ld1h { z16.h }, p1/Z, [x11]\n"
      "addvl x11, x11, #1\n"
      ".inst 0x81b02680  // fmopa za0.s, p1/M, p1/M, z20.h, z16.h\n"
      ".inst 0x81b02661  // fmopa za1.s, p1/M, p1/M, z19.h, z16.h\n"
      ".inst 0x81b02642  // fmopa za2.s, p1/M, p1/M, z18.h, z16.h\n"
      ".inst 0x81b02623  // fmopa za3.s, p1/M, p1/M, z17.h, z16.h\n"
      "bgt 10b\n"
      "11:"  // K oddments: End
      "tbz x5, #1, 15f\n"
      "tbz x5, #0, 13f\n"
      "mov x12, #0\n"
      "cntw x9\n"
      "cntw x28\n"
      "cntw x27, ALL, MUL #2\n"
      "cntw x26, ALL, MUL #3\n"
      "12:"  // Store to partial result buffer: Store and refill: Loop
      ".inst 0xe0bf04e0  // st1w { za0h.s[x12] }, p1/Z, [x7, XZR, LSL #2]\n"
      ".inst 0xe09f04c0  // ld1w { za0h.s[x12] }, p1/Z, [x6, XZR, LSL #2]\n"
      "addvl x25, x7, #4\n"
      "addvl x24, x6, #4\n"
      ".inst 0xe0bc04e1  // st1w { za0h.s[x12, #1] }, p1/Z, [x7, x28, LSL #2]\n"
      ".inst 0xe09c04c1  // ld1w { za0h.s[x12, #1] }, p1/Z, [x6, x28, LSL #2]\n"
      "addvl x23, x7, #8\n"
      "addvl x22, x6, #8\n"
      ".inst 0xe0bb04e2  // st1w { za0h.s[x12, #2] }, p1/Z, [x7, x27, LSL #2]\n"
      ".inst 0xe09b04c2  // ld1w { za0h.s[x12, #2] }, p1/Z, [x6, x27, LSL #2]\n"
      "addvl x21, x7, #12\n"
      "addvl x20, x6, #12\n"
      ".inst 0xe0ba04e3  // st1w { za0h.s[x12, #3] }, p1/Z, [x7, x26, LSL #2]\n"
      ".inst 0xe09a04c3  // ld1w { za0h.s[x12, #3] }, p1/Z, [x6, x26, LSL #2]\n"
      "addvl x7, x7, #16\n"
      "addvl x6, x6, #16\n"
      ".inst 0xe0bf0724  // st1w { za1h.s[x12] }, p1/Z, [x25, XZR, LSL #2]\n"
      ".inst 0xe09f0704  // ld1w { za1h.s[x12] }, p1/Z, [x24, XZR, LSL #2]\n"
      ".inst 0xe0bc0725  // st1w { za1h.s[x12, #1] }, p1/Z, [x25, x28, LSL #2]\n"
      ".inst 0xe09c0705  // ld1w { za1h.s[x12, #1] }, p1/Z, [x24, x28, LSL #2]\n"
      ".inst 0xe0bb0726  // st1w { za1h.s[x12, #2] }, p1/Z, [x25, x27, LSL #2]\n"
      ".inst 0xe09b0706  // ld1w { za1h.s[x12, #2] }, p1/Z, [x24, x27, LSL #2]\n"
      ".inst 0xe0ba0727  // st1w { za1h.s[x12, #3] }, p1/Z, [x25, x26, LSL #2]\n"
      ".inst 0xe09a0707  // ld1w { za1h.s[x12, #3] }, p1/Z, [x24, x26, LSL #2]\n"
      ".inst 0xe0bf06e8  // st1w { za2h.s[x12] }, p1/Z, [x23, XZR, LSL #2]\n"
      ".inst 0xe09f06c8  // ld1w { za2h.s[x12] }, p1/Z, [x22, XZR, LSL #2]\n"
      ".inst 0xe0bc06e9  // st1w { za2h.s[x12, #1] }, p1/Z, [x23, x28, LSL #2]\n"
      ".inst 0xe09c06c9  // ld1w { za2h.s[x12, #1] }, p1/Z, [x22, x28, LSL #2]\n"
      ".inst 0xe0bb06ea  // st1w { za2h.s[x12, #2] }, p1/Z, [x23, x27, LSL #2]\n"
      ".inst 0xe09b06ca  // ld1w { za2h.s[x12, #2] }, p1/Z, [x22, x27, LSL #2]\n"
      ".inst 0xe0ba06eb  // st1w { za2h.s[x12, #3] }, p1/Z, [x23, x26, LSL #2]\n"
      ".inst 0xe09a06cb  // ld1w { za2h.s[x12, #3] }, p1/Z, [x22, x26, LSL #2]\n"
      ".inst 0xe0bf06ac  // st1w { za3h.s[x12] }, p1/Z, [x21, XZR, LSL #2]\n"
      ".inst 0xe09f068c  // ld1w { za3h.s[x12] }, p1/Z, [x20, XZR, LSL #2]\n"
      ".inst 0xe0bc06ad  // st1w { za3h.s[x12, #1] }, p1/Z, [x21, x28, LSL #2]\n"
      ".inst 0xe09c068d  // ld1w { za3h.s[x12, #1] }, p1/Z, [x20, x28, LSL #2]\n"
      ".inst 0xe0bb06ae  // st1w { za3h.s[x12, #2] }, p1/Z, [x21, x27, LSL #2]\n"
      ".inst 0xe09b068e  // ld1w { za3h.s[x12, #2] }, p1/Z, [x20, x27, LSL #2]\n"
      ".inst 0xe0ba06af  // st1w { za3h.s[x12, #3] }, p1/Z, [x21, x26, LSL #2]\n"
      ".inst 0xe09a068f  // ld1w { za3h.s[x12, #3] }, p1/Z, [x20, x26, LSL #2]\n"
      "add x12, x12, #0x4\n"
      "cmp x12, x9\n"
      "blt 12b\n"
      "b 43f\n"
      "13:"  // Store to partial result buffer: Store only
      "mov x12, #0\n"
      "cntw x26\n"
      "cntw x25\n"
      "cntw x24, ALL, MUL #2\n"
      "cntw x23, ALL, MUL #3\n"
      "14:"  // Store to partial result buffer: Store only: Loop
      ".inst 0xe0bf04e0  // st1w { za0h.s[x12] }, p1/Z, [x7, XZR, LSL #2]\n"
      "addvl x22, x7, #4\n"
      "addvl x21, x7, #8\n"
      ".inst 0xe0b904e1  // st1w { za0h.s[x12, #1] }, p1/Z, [x7, x25, LSL #2]\n"
      "addvl x20, x7, #12\n"
      ".inst 0xe0b804e2  // st1w { za0h.s[x12, #2] }, p1/Z, [x7, x24, LSL #2]\n"
      ".inst 0xe0b704e3  // st1w { za0h.s[x12, #3] }, p1/Z, [x7, x23, LSL #2]\n"
      "addvl x7, x7, #16\n"
      ".inst 0xe0bf06c4  // st1w { za1h.s[x12] }, p1/Z, [x22, XZR, LSL #2]\n"
      ".inst 0xe0b906c5  // st1w { za1h.s[x12, #1] }, p1/Z, [x22, x25, LSL #2]\n"
      ".inst 0xe0b806c6  // st1w { za1h.s[x12, #2] }, p1/Z, [x22, x24, LSL #2]\n"
      ".inst 0xe0b706c7  // st1w { za1h.s[x12, #3] }, p1/Z, [x22, x23, LSL #2]\n"
      ".inst 0xe0bf06a8  // st1w { za2h.s[x12] }, p1/Z, [x21, XZR, LSL #2]\n"
      ".inst 0xe0b906a9  // st1w { za2h.s[x12, #1] }, p1/Z, [x21, x25, LSL #2]\n"
      ".inst 0xe0b806aa  // st1w { za2h.s[x12, #2] }, p1/Z, [x21, x24, LSL #2]\n"
      ".inst 0xe0b706ab  // st1w { za2h.s[x12, #3] }, p1/Z, [x21, x23, LSL #2]\n"
      ".inst 0xe0bf068c  // st1w { za3h.s[x12] }, p1/Z, [x20, XZR, LSL #2]\n"
      ".inst 0xe0b9068d  // st1w { za3h.s[x12, #1] }, p1/Z, [x20, x25, LSL #2]\n"
      ".inst 0xe0b8068e  // st1w { za3h.s[x12, #2] }, p1/Z, [x20, x24, LSL #2]\n"
      ".inst 0xe0b7068f  // st1w { za3h.s[x12, #3] }, p1/Z, [x20, x23, LSL #2]\n"
      "add x12, x12, #0x4\n"
      "cmp x12, x26\n"
      "blt 14b\n"
      "b 43f\n"
      "15:"  // Store to output array
      "ldr x26, [%x[args], %[offsetof_C]]\n"
      "sub x25, x15, x17\n"
      "ldr x24, [%x[args], %[offsetof_ldcb]]\n"
      "add x26, x26, x16, LSL #2\n"  // C += n
      "madd x26, x17, x24, x26\n"  // C += m * ldc
      "tbz x5, #2, 28f\n"
      "cntw x23\n"
      "mov x12, #0\n"
      "cmp x25, x23\n"
      "csel x22, x25, x23, LT\n"
      "lsr x21, x22, #0x2\n"
      "and x20, x22, #0x3\n"
      "cbz x21, 17f\n"
      "16:"  // Store to output array: Skip activation: Accumulator row 0 loop
      ".inst 0xc0820413  // mova z19.s, p1/M, za0h.s[x12]\n"
      "st1w { z19.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc0820432  // mova z18.s, p1/M, za0h.s[x12, #1]\n"
      ".inst 0xc0820451  // mova z17.s, p1/M, za0h.s[x12, #2]\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc0820470  // mova z16.s, p1/M, za0h.s[x12, #3]\n"
      "add x12, x12, #0x4\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "cmp x12, x21, LSL #2\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "blt 16b\n"
      "17:"  // Store to output array: Skip activation: Accumulator row 0 oddments
      "cbz x20, 18f\n"
      "subs x20, x20, #0x1\n"
      ".inst 0xc0820412  // mova z18.s, p1/M, za0h.s[x12]\n"
      ".inst 0xc0820431  // mova z17.s, p1/M, za0h.s[x12, #1]\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc0820450  // mova z16.s, p1/M, za0h.s[x12, #2]\n"
      "beq 18f\n"
      "subs x20, x20, #0x1\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 18f\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "18:"  // Store to output array: Skip activation: Accumulator row 0 oddments: End
      "subs x25, x25, x22\n"
      "beq 28f\n"
      "cmp x25, x23\n"
      "mov x12, #0\n"
      "csel x22, x25, x23, LT\n"
      "lsr x21, x22, #0x2\n"
      "and x20, x22, #0x3\n"
      "cbz x21, 20f\n"
      "19:"  // Store to output array: Skip activation: Accumulator row 1 loop
      ".inst 0xc0820493  // mova z19.s, p1/M, za1h.s[x12]\n"
      "st1w { z19.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc08204b2  // mova z18.s, p1/M, za1h.s[x12, #1]\n"
      ".inst 0xc08204d1  // mova z17.s, p1/M, za1h.s[x12, #2]\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc08204f0  // mova z16.s, p1/M, za1h.s[x12, #3]\n"
      "add x12, x12, #0x4\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "cmp x12, x21, LSL #2\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "blt 19b\n"
      "20:"  // Store to output array: Skip activation: Accumulator row 1 oddments
      "cbz x20, 21f\n"
      "subs x20, x20, #0x1\n"
      ".inst 0xc0820492  // mova z18.s, p1/M, za1h.s[x12]\n"
      ".inst 0xc08204b1  // mova z17.s, p1/M, za1h.s[x12, #1]\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc08204d0  // mova z16.s, p1/M, za1h.s[x12, #2]\n"
      "beq 21f\n"
      "subs x20, x20, #0x1\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 21f\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "21:"  // Store to output array: Skip activation: Accumulator row 1 oddments: End
      "subs x25, x25, x22\n"
      "beq 28f\n"
      "cmp x25, x23\n"
      "mov x12, #0\n"
      "csel x22, x25, x23, LT\n"
      "lsr x21, x22, #0x2\n"
      "and x20, x22, #0x3\n"
      "cbz x21, 23f\n"
      "22:"  // Store to output array: Skip activation: Accumulator row 2 loop
      ".inst 0xc0820513  // mova z19.s, p1/M, za2h.s[x12]\n"
      "st1w { z19.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc0820532  // mova z18.s, p1/M, za2h.s[x12, #1]\n"
      ".inst 0xc0820551  // mova z17.s, p1/M, za2h.s[x12, #2]\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc0820570  // mova z16.s, p1/M, za2h.s[x12, #3]\n"
      "add x12, x12, #0x4\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "cmp x12, x21, LSL #2\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "blt 22b\n"
      "23:"  // Store to output array: Skip activation: Accumulator row 2 oddments
      "cbz x20, 24f\n"
      "subs x20, x20, #0x1\n"
      ".inst 0xc0820512  // mova z18.s, p1/M, za2h.s[x12]\n"
      ".inst 0xc0820531  // mova z17.s, p1/M, za2h.s[x12, #1]\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc0820550  // mova z16.s, p1/M, za2h.s[x12, #2]\n"
      "beq 24f\n"
      "subs x20, x20, #0x1\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 24f\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "24:"  // Store to output array: Skip activation: Accumulator row 2 oddments: End
      "subs x25, x25, x22\n"
      "beq 28f\n"
      "cmp x25, x23\n"
      "mov x12, #0\n"
      "csel x22, x25, x23, LT\n"
      "lsr x21, x22, #0x2\n"
      "and x20, x22, #0x3\n"
      "cbz x21, 26f\n"
      "25:"  // Store to output array: Skip activation: Accumulator row 3 loop
      ".inst 0xc0820593  // mova z19.s, p1/M, za3h.s[x12]\n"
      "st1w { z19.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc08205b2  // mova z18.s, p1/M, za3h.s[x12, #1]\n"
      ".inst 0xc08205d1  // mova z17.s, p1/M, za3h.s[x12, #2]\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc08205f0  // mova z16.s, p1/M, za3h.s[x12, #3]\n"
      "add x12, x12, #0x4\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "cmp x12, x21, LSL #2\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "blt 25b\n"
      "26:"  // Store to output array: Skip activation: Accumulator row 3 oddments
      "cbz x20, 27f\n"
      "subs x20, x20, #0x1\n"
      ".inst 0xc0820592  // mova z18.s, p1/M, za3h.s[x12]\n"
      ".inst 0xc08205b1  // mova z17.s, p1/M, za3h.s[x12, #1]\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      ".inst 0xc08205d0  // mova z16.s, p1/M, za3h.s[x12, #2]\n"
      "beq 27f\n"
      "subs x20, x20, #0x1\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 27f\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "27:"  // Store to output array: Skip activation: Accumulator row 3 oddments: End
      "subs x25, x25, x22\n"
      "beq 28f\n"
      "b 41f\n"
      "28:"  // Store to output array: Skip activation: End
      "cntw x23\n"
      "ld1rw { z21.s }, p1/Z, [%x[args], %[offsetof_KernelArgs_min]]\n"
      "mov x12, #0\n"
      "cmp x25, x23\n"
      "ld1rw { z20.s }, p1/Z, [%x[args], %[offsetof_KernelArgs_max]]\n"
      "csel x22, x25, x23, LT\n"
      "lsr x21, x22, #0x2\n"
      "and x20, x22, #0x3\n"
      "cbz x21, 30f\n"
      "29:"  // Store to output array: Accumulator row 0 loop
      ".inst 0xc0820413  // mova z19.s, p1/M, za0h.s[x12]\n"
      ".inst 0xc0820432  // mova z18.s, p1/M, za0h.s[x12, #1]\n"
      "fmin z19.s, p1/M, z19.s, z20.s\n"
      ".inst 0xc0820451  // mova z17.s, p1/M, za0h.s[x12, #2]\n"
      "fmin z18.s, p1/M, z18.s, z20.s\n"
      ".inst 0xc0820470  // mova z16.s, p1/M, za0h.s[x12, #3]\n"
      "fmin z17.s, p1/M, z17.s, z20.s\n"
      "add x12, x12, #0x4\n"
      "fmin z16.s, p1/M, z16.s, z20.s\n"
      "cmp x12, x21, LSL #2\n"
      "fmax z19.s, p1/M, z19.s, z21.s\n"
      "fmax z18.s, p1/M, z18.s, z21.s\n"
      "fmax z17.s, p1/M, z17.s, z21.s\n"
      "fmax z16.s, p1/M, z16.s, z21.s\n"
      "st1w { z19.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "blt 29b\n"
      "30:"  // Store to output array: Accumulator row 0 oddments
      "cbz x20, 31f\n"
      ".inst 0xc0820412  // mova z18.s, p1/M, za0h.s[x12]\n"
      "subs x20, x20, #0x1\n"
      ".inst 0xc0820431  // mova z17.s, p1/M, za0h.s[x12, #1]\n"
      "fmin z18.s, p1/M, z18.s, z20.s\n"
      ".inst 0xc0820450  // mova z16.s, p1/M, za0h.s[x12, #2]\n"
      "fmin z17.s, p1/M, z17.s, z20.s\n"
      "fmin z16.s, p1/M, z16.s, z20.s\n"
      "fmax z18.s, p1/M, z18.s, z21.s\n"
      "fmax z17.s, p1/M, z17.s, z21.s\n"
      "fmax z16.s, p1/M, z16.s, z21.s\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 31f\n"
      "subs x20, x20, #0x1\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 31f\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "31:"  // Store to output array: Accumulator row 0 oddments: End
      "subs x25, x25, x22\n"
      "beq 41f\n"
      "cmp x25, x23\n"
      "mov x12, #0\n"
      "csel x22, x25, x23, LT\n"
      "lsr x21, x22, #0x2\n"
      "and x20, x22, #0x3\n"
      "cbz x21, 33f\n"
      "32:"  // Store to output array: Accumulator row 1 loop
      ".inst 0xc0820493  // mova z19.s, p1/M, za1h.s[x12]\n"
      ".inst 0xc08204b2  // mova z18.s, p1/M, za1h.s[x12, #1]\n"
      "fmin z19.s, p1/M, z19.s, z20.s\n"
      ".inst 0xc08204d1  // mova z17.s, p1/M, za1h.s[x12, #2]\n"
      "fmin z18.s, p1/M, z18.s, z20.s\n"
      ".inst 0xc08204f0  // mova z16.s, p1/M, za1h.s[x12, #3]\n"
      "fmin z17.s, p1/M, z17.s, z20.s\n"
      "add x12, x12, #0x4\n"
      "fmin z16.s, p1/M, z16.s, z20.s\n"
      "cmp x12, x21, LSL #2\n"
      "fmax z19.s, p1/M, z19.s, z21.s\n"
      "fmax z18.s, p1/M, z18.s, z21.s\n"
      "fmax z17.s, p1/M, z17.s, z21.s\n"
      "fmax z16.s, p1/M, z16.s, z21.s\n"
      "st1w { z19.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "blt 32b\n"
      "33:"  // Store to output array: Accumulator row 1 oddments
      "cbz x20, 34f\n"
      ".inst 0xc0820492  // mova z18.s, p1/M, za1h.s[x12]\n"
      "subs x20, x20, #0x1\n"
      ".inst 0xc08204b1  // mova z17.s, p1/M, za1h.s[x12, #1]\n"
      "fmin z18.s, p1/M, z18.s, z20.s\n"
      ".inst 0xc08204d0  // mova z16.s, p1/M, za1h.s[x12, #2]\n"
      "fmin z17.s, p1/M, z17.s, z20.s\n"
      "fmin z16.s, p1/M, z16.s, z20.s\n"
      "fmax z18.s, p1/M, z18.s, z21.s\n"
      "fmax z17.s, p1/M, z17.s, z21.s\n"
      "fmax z16.s, p1/M, z16.s, z21.s\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 34f\n"
      "subs x20, x20, #0x1\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 34f\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "34:"  // Store to output array: Accumulator row 1 oddments: End
      "subs x25, x25, x22\n"
      "beq 41f\n"
      "cmp x25, x23\n"
      "mov x12, #0\n"
      "csel x22, x25, x23, LT\n"
      "lsr x21, x22, #0x2\n"
      "and x20, x22, #0x3\n"
      "cbz x21, 36f\n"
      "35:"  // Store to output array: Accumulator row 2 loop
      ".inst 0xc0820513  // mova z19.s, p1/M, za2h.s[x12]\n"
      ".inst 0xc0820532  // mova z18.s, p1/M, za2h.s[x12, #1]\n"
      "fmin z19.s, p1/M, z19.s, z20.s\n"
      ".inst 0xc0820551  // mova z17.s, p1/M, za2h.s[x12, #2]\n"
      "fmin z18.s, p1/M, z18.s, z20.s\n"
      ".inst 0xc0820570  // mova z16.s, p1/M, za2h.s[x12, #3]\n"
      "fmin z17.s, p1/M, z17.s, z20.s\n"
      "add x12, x12, #0x4\n"
      "fmin z16.s, p1/M, z16.s, z20.s\n"
      "cmp x12, x21, LSL #2\n"
      "fmax z19.s, p1/M, z19.s, z21.s\n"
      "fmax z18.s, p1/M, z18.s, z21.s\n"
      "fmax z17.s, p1/M, z17.s, z21.s\n"
      "fmax z16.s, p1/M, z16.s, z21.s\n"
      "st1w { z19.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "blt 35b\n"
      "36:"  // Store to output array: Accumulator row 2 oddments
      "cbz x20, 37f\n"
      ".inst 0xc0820512  // mova z18.s, p1/M, za2h.s[x12]\n"
      "subs x20, x20, #0x1\n"
      ".inst 0xc0820531  // mova z17.s, p1/M, za2h.s[x12, #1]\n"
      "fmin z18.s, p1/M, z18.s, z20.s\n"
      ".inst 0xc0820550  // mova z16.s, p1/M, za2h.s[x12, #2]\n"
      "fmin z17.s, p1/M, z17.s, z20.s\n"
      "fmin z16.s, p1/M, z16.s, z20.s\n"
      "fmax z18.s, p1/M, z18.s, z21.s\n"
      "fmax z17.s, p1/M, z17.s, z21.s\n"
      "fmax z16.s, p1/M, z16.s, z21.s\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 37f\n"
      "subs x20, x20, #0x1\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 37f\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "37:"  // Store to output array: Accumulator row 2 oddments: End
      "subs x25, x25, x22\n"
      "beq 41f\n"
      "cmp x25, x23\n"
      "mov x12, #0\n"
      "csel x20, x25, x23, LT\n"
      "lsr x21, x20, #0x2\n"
      "and x20, x20, #0x3\n"
      "cbz x21, 39f\n"
      "38:"  // Store to output array: Accumulator row 3 loop
      ".inst 0xc0820593  // mova z19.s, p1/M, za3h.s[x12]\n"
      ".inst 0xc08205b2  // mova z18.s, p1/M, za3h.s[x12, #1]\n"
      "fmin z19.s, p1/M, z19.s, z20.s\n"
      ".inst 0xc08205d1  // mova z17.s, p1/M, za3h.s[x12, #2]\n"
      "fmin z18.s, p1/M, z18.s, z20.s\n"
      ".inst 0xc08205f0  // mova z16.s, p1/M, za3h.s[x12, #3]\n"
      "fmin z17.s, p1/M, z17.s, z20.s\n"
      "add x12, x12, #0x4\n"
      "fmin z16.s, p1/M, z16.s, z20.s\n"
      "cmp x12, x21, LSL #2\n"
      "fmax z19.s, p1/M, z19.s, z21.s\n"
      "fmax z18.s, p1/M, z18.s, z21.s\n"
      "fmax z17.s, p1/M, z17.s, z21.s\n"
      "fmax z16.s, p1/M, z16.s, z21.s\n"
      "st1w { z19.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "st1w { z16.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "blt 38b\n"
      "39:"  // Store to output array: Accumulator row 3 oddments
      "cbz x20, 40f\n"
      ".inst 0xc0820592  // mova z18.s, p1/M, za3h.s[x12]\n"
      "subs x20, x20, #0x1\n"
      ".inst 0xc08205b1  // mova z17.s, p1/M, za3h.s[x12, #1]\n"
      "fmin z18.s, p1/M, z18.s, z20.s\n"
      ".inst 0xc08205d0  // mova z16.s, p1/M, za3h.s[x12, #2]\n"
      "fmin z17.s, p1/M, z17.s, z20.s\n"
      "fmin z16.s, p1/M, z16.s, z20.s\n"
      "fmax z18.s, p1/M, z18.s, z21.s\n"
      "fmax z17.s, p1/M, z17.s, z21.s\n"
      "fmax z16.s, p1/M, z16.s, z21.s\n"
      "st1w { z18.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 40f\n"
      "subs x20, x20, #0x1\n"
      "st1w { z17.s }, p0, [x26]\n"
      "add x26, x26, x24\n"
      "beq 40f\n"
      "st1w { z16.s }, p0, [x26]\n"
      "40:"  // Store to output array: Accumulator row 3 oddments: End
      "41:"  // Store to output array: End
      "tbz x5, #0, 43f\n"
      "mov x12, #0\n"
      "cntw x26\n"
      "cntw x25\n"
      "cntw x24, ALL, MUL #2\n"
      "cntw x23, ALL, MUL #3\n"
      "42:"  // Store to output array: Refill accumulators: Loop
      "addvl x22, x6, #4\n"
      "addvl x21, x6, #8\n"
      ".inst 0xe09f04c0  // ld1w { za0h.s[x12] }, p1/Z, [x6, XZR, LSL #2]\n"
      "addvl x20, x6, #12\n"
      ".inst 0xe09f06c4  // ld1w { za1h.s[x12] }, p1/Z, [x22, XZR, LSL #2]\n"
      ".inst 0xe09f06a8  // ld1w { za2h.s[x12] }, p1/Z, [x21, XZR, LSL #2]\n"
      ".inst 0xe09f068c  // ld1w { za3h.s[x12] }, p1/Z, [x20, XZR, LSL #2]\n"
      ".inst 0xe09904c1  // ld1w { za0h.s[x12, #1] }, p1/Z, [x6, x25, LSL #2]\n"
      ".inst 0xe09906c5  // ld1w { za1h.s[x12, #1] }, p1/Z, [x22, x25, LSL #2]\n"
      ".inst 0xe09906a9  // ld1w { za2h.s[x12, #1] }, p1/Z, [x21, x25, LSL #2]\n"
      ".inst 0xe099068d  // ld1w { za3h.s[x12, #1] }, p1/Z, [x20, x25, LSL #2]\n"
      ".inst 0xe09804c2  // ld1w { za0h.s[x12, #2] }, p1/Z, [x6, x24, LSL #2]\n"
      ".inst 0xe09806c6  // ld1w { za1h.s[x12, #2] }, p1/Z, [x22, x24, LSL #2]\n"
      ".inst 0xe09806aa  // ld1w { za2h.s[x12, #2] }, p1/Z, [x21, x24, LSL #2]\n"
      ".inst 0xe098068e  // ld1w { za3h.s[x12, #2] }, p1/Z, [x20, x24, LSL #2]\n"
      ".inst 0xe09704c3  // ld1w { za0h.s[x12, #3] }, p1/Z, [x6, x23, LSL #2]\n"
      "addvl x6, x6, #16\n"
      ".inst 0xe09706c7  // ld1w { za1h.s[x12, #3] }, p1/Z, [x22, x23, LSL #2]\n"
      ".inst 0xe09706ab  // ld1w { za2h.s[x12, #3] }, p1/Z, [x21, x23, LSL #2]\n"
      ".inst 0xe097068f  // ld1w { za3h.s[x12, #3] }, p1/Z, [x20, x23, LSL #2]\n"
      "add x12, x12, #0x4\n"
      "cmp x12, x26\n"
      "blt 42b\n"
      "43:"  // End block
      "incw x16\n"
      "cmp x16, x14\n"
      "blt 4b\n"
      "incw x17, ALL, MUL #4\n"
      "mov x16, #0\n"
      "cmp x17, x15\n"
      "mov x13, x10\n"
      "blt 3b\n"
      ".inst 0xd503467f  // SMSTOP\n"
      :
      : [args] "r" (&args), [offsetof_A] "I" (offsetof(KernelArgs, A)), [offsetof_B] "I" (offsetof(KernelArgs, B)), [offsetof_C] "I" (offsetof(KernelArgs, C)), [offsetof_K] "I" (offsetof(KernelArgs, K)), [offsetof_KernelArgs_max] "I" (offsetof(KernelArgs, max)), [offsetof_KernelArgs_min] "I" (offsetof(KernelArgs, min)), [offsetof_M] "I" (offsetof(KernelArgs, M)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_accumulator_buffer] "I" (offsetof(KernelArgs, accumulator_buffer)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_flags] "I" (offsetof(KernelArgs, flags)), [offsetof_ldcb] "I" (offsetof(KernelArgs, ldcb))
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

}  // namespace arm_gemm

#endif // (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(ARM_COMPUTE_ENABLE_SME) && defined(__aarch64__)

