/*
 * Copyright (c) 2025-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(ARM_COMPUTE_ENABLE_SME) && defined(__aarch64__)

#include "arm_gemm/arm_gemm.hpp"
#include "arm_common/internal/utils.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void sme_gemv_fp16fp32fp16_mla_8VL_rhs2VL (
    const __fp16 *A_ptr, const __fp16 *B_ptr, __fp16 *output_ptr,
    size_t N, size_t K,
    const __fp16 *bias, Activation act, bool
)
{
    struct KernelArgs {
        __fp16 maxval = static_cast<__fp16>(std::numeric_limits<float>::infinity());
        __fp16 minval = - static_cast<__fp16>(std::numeric_limits<float>::infinity());
        const __fp16 *B_ptr = {};
        size_t output_offset = {};
        unsigned int input_initial_col = {};
    } args;

    unsigned long flags=0;
    args.B_ptr = B_ptr;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            args.maxval = static_cast<__fp16>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            args.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      ".inst 0xd503477f  // SMSTART ZA\n"
      "add x10, %x[K], #0x1\n"
      "mov x9, %x[bias]\n"
      "cntw x28\n"
      "bic x10, x10, #0x1\n"
      "add x27, %x[N], x28\n"
      "lsl x10, x10, #0x1\n"
      "sub x27, x27, #0x1\n"
      "cntw x20, ALL, MUL #2\n"
      "ptrue p2.b\n"
      "udiv x27, x27, x28\n"
      "mul x10, x10, x20\n"
      "1:"  // Column loop
      "cmp x27, #0x8\n"
      "bge 50f\n"
      "cmp x27, #0x6\n"
      "bgt 43f\n"
      "beq 36f\n"
      "cmp x27, #0x4\n"
      "bgt 29f\n"
      "beq 22f\n"
      "cmp x27, #0x2\n"
      "bgt 15f\n"
      "beq 8f\n"
      "mov x26, %x[K]\n"
      "mov x25, %x[A_ptr]\n"
      "whilelt p1.h, XZR, %x[N]\n"
      "cbz x9, 2f\n"
      "ld1h { z24.s }, p2/Z, [x9]\n"
      "fcvt z24.s, p2/m, z24.h\n"
      "b 3f\n"
      "2:"  // Width 1: no bias
      "fmov z24.s, #0\n"
      "3:"  // Width 1: setup done
      "cmp x26, #0x8\n"
      "ble 5f\n"
      "4:"  // Width 1: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "sub x26, x26, #0x8\n"
      "add x25, x25, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "cmp x26, #0x8\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a04038  // fmlalb z24.s, z1.h, z0.h[0]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a04c38  // fmlalt z24.s, z1.h, z0.h[1]\n"
      ".inst 0x64a84058  // fmlalb z24.s, z2.h, z0.h[2]\n"
      ".inst 0x64a84c58  // fmlalt z24.s, z2.h, z0.h[3]\n"
      ".inst 0x64b04078  // fmlalb z24.s, z3.h, z0.h[4]\n"
      ".inst 0x64b04c78  // fmlalt z24.s, z3.h, z0.h[5]\n"
      ".inst 0x64b84098  // fmlalb z24.s, z4.h, z0.h[6]\n"
      ".inst 0x64b84c98  // fmlalt z24.s, z4.h, z0.h[7]\n"
      "bgt 4b\n"
      "5:"  // Width 1: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a040b8  // fmlalb z24.s, z5.h, z0.h[0]\n"
      ".inst 0x64a04cb8  // fmlalt z24.s, z5.h, z0.h[1]\n"
      "ble 6f\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a840d8  // fmlalb z24.s, z6.h, z0.h[2]\n"
      ".inst 0x64a84cd8  // fmlalt z24.s, z6.h, z0.h[3]\n"
      "ble 6f\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64b040f8  // fmlalb z24.s, z7.h, z0.h[4]\n"
      ".inst 0x64b04cf8  // fmlalt z24.s, z7.h, z0.h[5]\n"
      "ble 6f\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64b84118  // fmlalb z24.s, z8.h, z0.h[6]\n"
      ".inst 0x64b84d18  // fmlalt z24.s, z8.h, z0.h[7]\n"
      "6:"  // Width 1: Multiply loop: multiply skip
      "fcvt z24.h, p2/m, z24.s\n"
      "uzp1 z24.h, z24.h, z24.h\n"
      "tbz %x[flags], #1, 7f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "7:"  // Width 1: No activation
      "st1h { z24.h }, p1, [%x[output_ptr]]\n"
      "addvl %x[output_ptr], %x[output_ptr], #1\n"
      "b 57f\n"
      "8:"  // Width 2
      "mov x26, %x[K]\n"
      "mov x25, %x[A_ptr]\n"
      "whilelt p1.h, XZR, %x[N]\n"
      "cbz x9, 9f\n"
      "ld1h { z24.s }, p2/Z, [x9]\n"
      "ld1h { z25.s }, p2/Z, [x9, #1, MUL VL]\n"
      "fcvt z24.s, p2/m, z24.h\n"
      "fcvt z25.s, p2/m, z25.h\n"
      "b 10f\n"
      "9:"  // Width 2: no bias
      "fmov z24.s, #0\n"
      "fmov z25.s, #0\n"
      "10:"  // Width 2: setup done
      "cmp x26, #0x8\n"
      "ble 12f\n"
      "11:"  // Width 2: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x26, x26, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "cmp x26, #0x8\n"
      "add x25, x25, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a04038  // fmlalb z24.s, z1.h, z0.h[0]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a04059  // fmlalb z25.s, z2.h, z0.h[0]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a04c38  // fmlalt z24.s, z1.h, z0.h[1]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a04c59  // fmlalt z25.s, z2.h, z0.h[1]\n"
      ".inst 0x64a84078  // fmlalb z24.s, z3.h, z0.h[2]\n"
      ".inst 0x64a84099  // fmlalb z25.s, z4.h, z0.h[2]\n"
      ".inst 0x64a84c78  // fmlalt z24.s, z3.h, z0.h[3]\n"
      ".inst 0x64a84c99  // fmlalt z25.s, z4.h, z0.h[3]\n"
      ".inst 0x64b040b8  // fmlalb z24.s, z5.h, z0.h[4]\n"
      ".inst 0x64b040d9  // fmlalb z25.s, z6.h, z0.h[4]\n"
      ".inst 0x64b04cb8  // fmlalt z24.s, z5.h, z0.h[5]\n"
      ".inst 0x64b04cd9  // fmlalt z25.s, z6.h, z0.h[5]\n"
      ".inst 0x64b840f8  // fmlalb z24.s, z7.h, z0.h[6]\n"
      ".inst 0x64b84119  // fmlalb z25.s, z8.h, z0.h[6]\n"
      ".inst 0x64b84cf8  // fmlalt z24.s, z7.h, z0.h[7]\n"
      ".inst 0x64b84d19  // fmlalt z25.s, z8.h, z0.h[7]\n"
      "bgt 11b\n"
      "12:"  // Width 2: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a04138  // fmlalb z24.s, z9.h, z0.h[0]\n"
      ".inst 0x64a04159  // fmlalb z25.s, z10.h, z0.h[0]\n"
      ".inst 0x64a04d38  // fmlalt z24.s, z9.h, z0.h[1]\n"
      ".inst 0x64a04d59  // fmlalt z25.s, z10.h, z0.h[1]\n"
      "ble 13f\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a84178  // fmlalb z24.s, z11.h, z0.h[2]\n"
      ".inst 0x64a84199  // fmlalb z25.s, z12.h, z0.h[2]\n"
      ".inst 0x64a84d78  // fmlalt z24.s, z11.h, z0.h[3]\n"
      ".inst 0x64a84d99  // fmlalt z25.s, z12.h, z0.h[3]\n"
      "ble 13f\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64b041b8  // fmlalb z24.s, z13.h, z0.h[4]\n"
      ".inst 0x64b041d9  // fmlalb z25.s, z14.h, z0.h[4]\n"
      ".inst 0x64b04db8  // fmlalt z24.s, z13.h, z0.h[5]\n"
      ".inst 0x64b04dd9  // fmlalt z25.s, z14.h, z0.h[5]\n"
      "ble 13f\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64b841f8  // fmlalb z24.s, z15.h, z0.h[6]\n"
      ".inst 0x64b84219  // fmlalb z25.s, z16.h, z0.h[6]\n"
      ".inst 0x64b84df8  // fmlalt z24.s, z15.h, z0.h[7]\n"
      ".inst 0x64b84e19  // fmlalt z25.s, z16.h, z0.h[7]\n"
      "13:"  // Width 2: Multiply loop: multiply skip
      "fcvt z24.h, p2/m, z24.s\n"
      "fcvt z25.h, p2/m, z25.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "tbz %x[flags], #1, 14f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "14:"  // Width 2: No activation
      "st1h { z24.h }, p1, [%x[output_ptr]]\n"
      "addvl %x[output_ptr], %x[output_ptr], #1\n"
      "b 57f\n"
      "15:"  // Width 3
      "lsl x20, x28, #0x1\n"
      "mov x26, %x[K]\n"
      "sub x21, %x[N], x20\n"
      "mov x25, %x[A_ptr]\n"
      "add x20, %x[B_ptr], x10\n"
      "whilelt p1.h, XZR, x21\n"
      "cbz x9, 16f\n"
      "ld1h { z24.s }, p2/Z, [x9]\n"
      "ld1h { z25.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1h { z26.s }, p2/Z, [x9, #2, MUL VL]\n"
      "fcvt z24.s, p2/m, z24.h\n"
      "fcvt z25.s, p2/m, z25.h\n"
      "fcvt z26.s, p2/m, z26.h\n"
      "b 17f\n"
      "16:"  // Width 3: no bias
      "fmov z24.s, #0\n"
      "fmov z25.s, #0\n"
      "fmov z26.s, #0\n"
      "17:"  // Width 3: setup done
      "cmp x26, #0x8\n"
      "ble 19f\n"
      "18:"  // Width 3: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x26, x26, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "cmp x26, #0x8\n"
      "add x25, x25, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z3.h }, p2/Z, [x20]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a04038  // fmlalb z24.s, z1.h, z0.h[0]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a04059  // fmlalb z25.s, z2.h, z0.h[0]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a0407a  // fmlalb z26.s, z3.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [x20]\n"
      "addvl x20, x20, #2\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a04c38  // fmlalt z24.s, z1.h, z0.h[1]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a04c59  // fmlalt z25.s, z2.h, z0.h[1]\n"
      "ldnt1h { z9.h }, p2/Z, [x20]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a04c7a  // fmlalt z26.s, z3.h, z0.h[1]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a84098  // fmlalb z24.s, z4.h, z0.h[2]\n"
      "ldnt1h { z12.h }, p2/Z, [x20]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a840b9  // fmlalb z25.s, z5.h, z0.h[2]\n"
      ".inst 0x64a840da  // fmlalb z26.s, z6.h, z0.h[2]\n"
      ".inst 0x64a84c98  // fmlalt z24.s, z4.h, z0.h[3]\n"
      ".inst 0x64a84cb9  // fmlalt z25.s, z5.h, z0.h[3]\n"
      ".inst 0x64a84cda  // fmlalt z26.s, z6.h, z0.h[3]\n"
      ".inst 0x64b040f8  // fmlalb z24.s, z7.h, z0.h[4]\n"
      ".inst 0x64b04119  // fmlalb z25.s, z8.h, z0.h[4]\n"
      ".inst 0x64b0413a  // fmlalb z26.s, z9.h, z0.h[4]\n"
      ".inst 0x64b04cf8  // fmlalt z24.s, z7.h, z0.h[5]\n"
      ".inst 0x64b04d19  // fmlalt z25.s, z8.h, z0.h[5]\n"
      ".inst 0x64b04d3a  // fmlalt z26.s, z9.h, z0.h[5]\n"
      ".inst 0x64b84158  // fmlalb z24.s, z10.h, z0.h[6]\n"
      ".inst 0x64b84179  // fmlalb z25.s, z11.h, z0.h[6]\n"
      ".inst 0x64b8419a  // fmlalb z26.s, z12.h, z0.h[6]\n"
      ".inst 0x64b84d58  // fmlalt z24.s, z10.h, z0.h[7]\n"
      ".inst 0x64b84d79  // fmlalt z25.s, z11.h, z0.h[7]\n"
      ".inst 0x64b84d9a  // fmlalt z26.s, z12.h, z0.h[7]\n"
      "bgt 18b\n"
      "19:"  // Width 3: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z15.h }, p2/Z, [x20]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a041b8  // fmlalb z24.s, z13.h, z0.h[0]\n"
      ".inst 0x64a041d9  // fmlalb z25.s, z14.h, z0.h[0]\n"
      ".inst 0x64a041fa  // fmlalb z26.s, z15.h, z0.h[0]\n"
      ".inst 0x64a04db8  // fmlalt z24.s, z13.h, z0.h[1]\n"
      ".inst 0x64a04dd9  // fmlalt z25.s, z14.h, z0.h[1]\n"
      ".inst 0x64a04dfa  // fmlalt z26.s, z15.h, z0.h[1]\n"
      "ble 20f\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z18.h }, p2/Z, [x20]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a84218  // fmlalb z24.s, z16.h, z0.h[2]\n"
      ".inst 0x64a84239  // fmlalb z25.s, z17.h, z0.h[2]\n"
      ".inst 0x64a8425a  // fmlalb z26.s, z18.h, z0.h[2]\n"
      ".inst 0x64a84e18  // fmlalt z24.s, z16.h, z0.h[3]\n"
      ".inst 0x64a84e39  // fmlalt z25.s, z17.h, z0.h[3]\n"
      ".inst 0x64a84e5a  // fmlalt z26.s, z18.h, z0.h[3]\n"
      "ble 20f\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z21.h }, p2/Z, [x20]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64b04278  // fmlalb z24.s, z19.h, z0.h[4]\n"
      ".inst 0x64b04299  // fmlalb z25.s, z20.h, z0.h[4]\n"
      ".inst 0x64b042ba  // fmlalb z26.s, z21.h, z0.h[4]\n"
      ".inst 0x64b04e78  // fmlalt z24.s, z19.h, z0.h[5]\n"
      ".inst 0x64b04e99  // fmlalt z25.s, z20.h, z0.h[5]\n"
      ".inst 0x64b04eba  // fmlalt z26.s, z21.h, z0.h[5]\n"
      "ble 20f\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z1.h }, p2/Z, [x20]\n"
      ".inst 0x64b842d8  // fmlalb z24.s, z22.h, z0.h[6]\n"
      ".inst 0x64b842f9  // fmlalb z25.s, z23.h, z0.h[6]\n"
      ".inst 0x64b8403a  // fmlalb z26.s, z1.h, z0.h[6]\n"
      ".inst 0x64b84ed8  // fmlalt z24.s, z22.h, z0.h[7]\n"
      ".inst 0x64b84ef9  // fmlalt z25.s, z23.h, z0.h[7]\n"
      ".inst 0x64b84c3a  // fmlalt z26.s, z1.h, z0.h[7]\n"
      "20:"  // Width 3: Multiply loop: multiply skip
      "fcvt z24.h, p2/m, z24.s\n"
      "fcvt z25.h, p2/m, z25.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "fcvt z25.h, p2/m, z26.s\n"
      "uzp1 z25.h, z25.h, z25.h\n"
      "tbz %x[flags], #1, 21f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "21:"  // Width 3: No activation
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "st1h { z25.h }, p1, [%x[output_ptr], #1, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #2\n"
      "b 57f\n"
      "22:"  // Width 4
      "lsl x20, x28, #0x1\n"
      "mov x26, %x[K]\n"
      "sub x21, %x[N], x20\n"
      "mov x25, %x[A_ptr]\n"
      "add x20, %x[B_ptr], x10\n"
      "whilelt p1.h, XZR, x21\n"
      "cbz x9, 23f\n"
      "ld1h { z24.s }, p2/Z, [x9]\n"
      "ld1h { z25.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1h { z26.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1h { z27.s }, p2/Z, [x9, #3, MUL VL]\n"
      "fcvt z24.s, p2/m, z24.h\n"
      "fcvt z25.s, p2/m, z25.h\n"
      "fcvt z26.s, p2/m, z26.h\n"
      "fcvt z27.s, p2/m, z27.h\n"
      "b 24f\n"
      "23:"  // Width 4: no bias
      "fmov z24.s, #0\n"
      "fmov z25.s, #0\n"
      "fmov z26.s, #0\n"
      "fmov z27.s, #0\n"
      "24:"  // Width 4: setup done
      "cmp x26, #0x8\n"
      "ble 26f\n"
      "25:"  // Width 4: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x26, x26, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "cmp x26, #0x8\n"
      "add x25, x25, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z3.h }, p2/Z, [x20]\n"
      "ldnt1h { z4.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64a04038  // fmlalb z24.s, z1.h, z0.h[0]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a04059  // fmlalb z25.s, z2.h, z0.h[0]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a0407a  // fmlalb z26.s, z3.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a0409b  // fmlalb z27.s, z4.h, z0.h[0]\n"
      "ldnt1h { z7.h }, p2/Z, [x20]\n"
      ".inst 0x64a04c38  // fmlalt z24.s, z1.h, z0.h[1]\n"
      "ldnt1h { z8.h }, p2/Z, [x20, #1, MUL VL]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a04c59  // fmlalt z25.s, z2.h, z0.h[1]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a04c7a  // fmlalt z26.s, z3.h, z0.h[1]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a04c9b  // fmlalt z27.s, z4.h, z0.h[1]\n"
      "ldnt1h { z11.h }, p2/Z, [x20]\n"
      ".inst 0x64a840b8  // fmlalb z24.s, z5.h, z0.h[2]\n"
      "ldnt1h { z12.h }, p2/Z, [x20, #1, MUL VL]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a840d9  // fmlalb z25.s, z6.h, z0.h[2]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a840fa  // fmlalb z26.s, z7.h, z0.h[2]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a8411b  // fmlalb z27.s, z8.h, z0.h[2]\n"
      "ldnt1h { z15.h }, p2/Z, [x20]\n"
      ".inst 0x64a84cb8  // fmlalt z24.s, z5.h, z0.h[3]\n"
      "ldnt1h { z16.h }, p2/Z, [x20, #1, MUL VL]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a84cd9  // fmlalt z25.s, z6.h, z0.h[3]\n"
      ".inst 0x64a84cfa  // fmlalt z26.s, z7.h, z0.h[3]\n"
      ".inst 0x64a84d1b  // fmlalt z27.s, z8.h, z0.h[3]\n"
      ".inst 0x64b04138  // fmlalb z24.s, z9.h, z0.h[4]\n"
      ".inst 0x64b04159  // fmlalb z25.s, z10.h, z0.h[4]\n"
      ".inst 0x64b0417a  // fmlalb z26.s, z11.h, z0.h[4]\n"
      ".inst 0x64b0419b  // fmlalb z27.s, z12.h, z0.h[4]\n"
      ".inst 0x64b04d38  // fmlalt z24.s, z9.h, z0.h[5]\n"
      ".inst 0x64b04d59  // fmlalt z25.s, z10.h, z0.h[5]\n"
      ".inst 0x64b04d7a  // fmlalt z26.s, z11.h, z0.h[5]\n"
      ".inst 0x64b04d9b  // fmlalt z27.s, z12.h, z0.h[5]\n"
      ".inst 0x64b841b8  // fmlalb z24.s, z13.h, z0.h[6]\n"
      ".inst 0x64b841d9  // fmlalb z25.s, z14.h, z0.h[6]\n"
      ".inst 0x64b841fa  // fmlalb z26.s, z15.h, z0.h[6]\n"
      ".inst 0x64b8421b  // fmlalb z27.s, z16.h, z0.h[6]\n"
      ".inst 0x64b84db8  // fmlalt z24.s, z13.h, z0.h[7]\n"
      ".inst 0x64b84dd9  // fmlalt z25.s, z14.h, z0.h[7]\n"
      ".inst 0x64b84dfa  // fmlalt z26.s, z15.h, z0.h[7]\n"
      ".inst 0x64b84e1b  // fmlalt z27.s, z16.h, z0.h[7]\n"
      "bgt 25b\n"
      "26:"  // Width 4: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z19.h }, p2/Z, [x20]\n"
      "ldnt1h { z20.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64a04238  // fmlalb z24.s, z17.h, z0.h[0]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a04259  // fmlalb z25.s, z18.h, z0.h[0]\n"
      ".inst 0x64a0427a  // fmlalb z26.s, z19.h, z0.h[0]\n"
      ".inst 0x64a0429b  // fmlalb z27.s, z20.h, z0.h[0]\n"
      ".inst 0x64a04e38  // fmlalt z24.s, z17.h, z0.h[1]\n"
      ".inst 0x64a04e59  // fmlalt z25.s, z18.h, z0.h[1]\n"
      ".inst 0x64a04e7a  // fmlalt z26.s, z19.h, z0.h[1]\n"
      ".inst 0x64a04e9b  // fmlalt z27.s, z20.h, z0.h[1]\n"
      "ble 27f\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z23.h }, p2/Z, [x20]\n"
      "ldnt1h { z1.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64a842b8  // fmlalb z24.s, z21.h, z0.h[2]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a842d9  // fmlalb z25.s, z22.h, z0.h[2]\n"
      ".inst 0x64a842fa  // fmlalb z26.s, z23.h, z0.h[2]\n"
      ".inst 0x64a8403b  // fmlalb z27.s, z1.h, z0.h[2]\n"
      ".inst 0x64a84eb8  // fmlalt z24.s, z21.h, z0.h[3]\n"
      ".inst 0x64a84ed9  // fmlalt z25.s, z22.h, z0.h[3]\n"
      ".inst 0x64a84efa  // fmlalt z26.s, z23.h, z0.h[3]\n"
      ".inst 0x64a84c3b  // fmlalt z27.s, z1.h, z0.h[3]\n"
      "ble 27f\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z4.h }, p2/Z, [x20]\n"
      "ldnt1h { z5.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64b04058  // fmlalb z24.s, z2.h, z0.h[4]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64b04079  // fmlalb z25.s, z3.h, z0.h[4]\n"
      ".inst 0x64b0409a  // fmlalb z26.s, z4.h, z0.h[4]\n"
      ".inst 0x64b040bb  // fmlalb z27.s, z5.h, z0.h[4]\n"
      ".inst 0x64b04c58  // fmlalt z24.s, z2.h, z0.h[5]\n"
      ".inst 0x64b04c79  // fmlalt z25.s, z3.h, z0.h[5]\n"
      ".inst 0x64b04c9a  // fmlalt z26.s, z4.h, z0.h[5]\n"
      ".inst 0x64b04cbb  // fmlalt z27.s, z5.h, z0.h[5]\n"
      "ble 27f\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z8.h }, p2/Z, [x20]\n"
      "ldnt1h { z9.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64b840d8  // fmlalb z24.s, z6.h, z0.h[6]\n"
      ".inst 0x64b840f9  // fmlalb z25.s, z7.h, z0.h[6]\n"
      ".inst 0x64b8411a  // fmlalb z26.s, z8.h, z0.h[6]\n"
      ".inst 0x64b8413b  // fmlalb z27.s, z9.h, z0.h[6]\n"
      ".inst 0x64b84cd8  // fmlalt z24.s, z6.h, z0.h[7]\n"
      ".inst 0x64b84cf9  // fmlalt z25.s, z7.h, z0.h[7]\n"
      ".inst 0x64b84d1a  // fmlalt z26.s, z8.h, z0.h[7]\n"
      ".inst 0x64b84d3b  // fmlalt z27.s, z9.h, z0.h[7]\n"
      "27:"  // Width 4: Multiply loop: multiply skip
      "fcvt z24.h, p2/m, z24.s\n"
      "fcvt z25.h, p2/m, z25.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "fcvt z25.h, p2/m, z26.s\n"
      "fcvt z26.h, p2/m, z27.s\n"
      "uzp1 z25.h, z25.h, z26.h\n"
      "tbz %x[flags], #1, 28f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "28:"  // Width 4: No activation
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "st1h { z25.h }, p1, [%x[output_ptr], #1, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #2\n"
      "b 57f\n"
      "29:"  // Width 5
      "lsl x21, x28, #0x1\n"
      "mov x20, #0x2\n"
      "msub x22, x21, x20, %x[N]\n"
      "mov x26, %x[K]\n"
      "mov x25, %x[A_ptr]\n"
      "add x21, %x[B_ptr], x10\n"
      "add x20, %x[B_ptr], x10, LSL #1\n"
      "whilelt p1.h, XZR, x22\n"
      "cbz x9, 30f\n"
      "ld1h { z24.s }, p2/Z, [x9]\n"
      "ld1h { z25.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1h { z26.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1h { z27.s }, p2/Z, [x9, #3, MUL VL]\n"
      "fcvt z24.s, p2/m, z24.h\n"
      "ld1h { z28.s }, p2/Z, [x9, #4, MUL VL]\n"
      "fcvt z25.s, p2/m, z25.h\n"
      "fcvt z26.s, p2/m, z26.h\n"
      "fcvt z27.s, p2/m, z27.h\n"
      "fcvt z28.s, p2/m, z28.h\n"
      "b 31f\n"
      "30:"  // Width 5: no bias
      "fmov z24.s, #0\n"
      "fmov z25.s, #0\n"
      "fmov z26.s, #0\n"
      "fmov z27.s, #0\n"
      "fmov z28.s, #0\n"
      "31:"  // Width 5: setup done
      "cmp x26, #0x8\n"
      "ble 33f\n"
      "32:"  // Width 5: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x26, x26, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "cmp x26, #0x8\n"
      "add x25, x25, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z3.h }, p2/Z, [x21]\n"
      "ldnt1h { z4.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a04038  // fmlalb z24.s, z1.h, z0.h[0]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z5.h }, p2/Z, [x20]\n"
      ".inst 0x64a04059  // fmlalb z25.s, z2.h, z0.h[0]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a0407a  // fmlalb z26.s, z3.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a0409b  // fmlalb z27.s, z4.h, z0.h[0]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a040bc  // fmlalb z28.s, z5.h, z0.h[0]\n"
      ".inst 0x64a04c38  // fmlalt z24.s, z1.h, z0.h[1]\n"
      "ldnt1h { z8.h }, p2/Z, [x21]\n"
      ".inst 0x64a04c59  // fmlalt z25.s, z2.h, z0.h[1]\n"
      "ldnt1h { z9.h }, p2/Z, [x21, #1, MUL VL]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64a04c7a  // fmlalt z26.s, z3.h, z0.h[1]\n"
      "ldnt1h { z10.h }, p2/Z, [x20]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a04c9b  // fmlalt z27.s, z4.h, z0.h[1]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a04cbc  // fmlalt z28.s, z5.h, z0.h[1]\n"
      ".inst 0x64a840d8  // fmlalb z24.s, z6.h, z0.h[2]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a840f9  // fmlalb z25.s, z7.h, z0.h[2]\n"
      "ldnt1h { z13.h }, p2/Z, [x21]\n"
      ".inst 0x64a8411a  // fmlalb z26.s, z8.h, z0.h[2]\n"
      "ldnt1h { z14.h }, p2/Z, [x21, #1, MUL VL]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64a8413b  // fmlalb z27.s, z9.h, z0.h[2]\n"
      "ldnt1h { z15.h }, p2/Z, [x20]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a8415c  // fmlalb z28.s, z10.h, z0.h[2]\n"
      ".inst 0x64a84cd8  // fmlalt z24.s, z6.h, z0.h[3]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a84cf9  // fmlalt z25.s, z7.h, z0.h[3]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a84d1a  // fmlalt z26.s, z8.h, z0.h[3]\n"
      "ldnt1h { z18.h }, p2/Z, [x21]\n"
      ".inst 0x64a84d3b  // fmlalt z27.s, z9.h, z0.h[3]\n"
      "ldnt1h { z19.h }, p2/Z, [x21, #1, MUL VL]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64a84d5c  // fmlalt z28.s, z10.h, z0.h[3]\n"
      ".inst 0x64b04178  // fmlalb z24.s, z11.h, z0.h[4]\n"
      "ldnt1h { z20.h }, p2/Z, [x20]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64b04199  // fmlalb z25.s, z12.h, z0.h[4]\n"
      ".inst 0x64b041ba  // fmlalb z26.s, z13.h, z0.h[4]\n"
      ".inst 0x64b041db  // fmlalb z27.s, z14.h, z0.h[4]\n"
      ".inst 0x64b041fc  // fmlalb z28.s, z15.h, z0.h[4]\n"
      ".inst 0x64b04d78  // fmlalt z24.s, z11.h, z0.h[5]\n"
      ".inst 0x64b04d99  // fmlalt z25.s, z12.h, z0.h[5]\n"
      ".inst 0x64b04dba  // fmlalt z26.s, z13.h, z0.h[5]\n"
      ".inst 0x64b04ddb  // fmlalt z27.s, z14.h, z0.h[5]\n"
      ".inst 0x64b04dfc  // fmlalt z28.s, z15.h, z0.h[5]\n"
      ".inst 0x64b84218  // fmlalb z24.s, z16.h, z0.h[6]\n"
      ".inst 0x64b84239  // fmlalb z25.s, z17.h, z0.h[6]\n"
      ".inst 0x64b8425a  // fmlalb z26.s, z18.h, z0.h[6]\n"
      ".inst 0x64b8427b  // fmlalb z27.s, z19.h, z0.h[6]\n"
      ".inst 0x64b8429c  // fmlalb z28.s, z20.h, z0.h[6]\n"
      ".inst 0x64b84e18  // fmlalt z24.s, z16.h, z0.h[7]\n"
      ".inst 0x64b84e39  // fmlalt z25.s, z17.h, z0.h[7]\n"
      ".inst 0x64b84e5a  // fmlalt z26.s, z18.h, z0.h[7]\n"
      ".inst 0x64b84e7b  // fmlalt z27.s, z19.h, z0.h[7]\n"
      ".inst 0x64b84e9c  // fmlalt z28.s, z20.h, z0.h[7]\n"
      "bgt 32b\n"
      "33:"  // Width 5: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z23.h }, p2/Z, [x21]\n"
      "ldnt1h { z1.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a042b8  // fmlalb z24.s, z21.h, z0.h[0]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z2.h }, p2/Z, [x20]\n"
      ".inst 0x64a042d9  // fmlalb z25.s, z22.h, z0.h[0]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a042fa  // fmlalb z26.s, z23.h, z0.h[0]\n"
      ".inst 0x64a0403b  // fmlalb z27.s, z1.h, z0.h[0]\n"
      ".inst 0x64a0405c  // fmlalb z28.s, z2.h, z0.h[0]\n"
      ".inst 0x64a04eb8  // fmlalt z24.s, z21.h, z0.h[1]\n"
      ".inst 0x64a04ed9  // fmlalt z25.s, z22.h, z0.h[1]\n"
      ".inst 0x64a04efa  // fmlalt z26.s, z23.h, z0.h[1]\n"
      ".inst 0x64a04c3b  // fmlalt z27.s, z1.h, z0.h[1]\n"
      ".inst 0x64a04c5c  // fmlalt z28.s, z2.h, z0.h[1]\n"
      "ble 34f\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z5.h }, p2/Z, [x21]\n"
      "ldnt1h { z6.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a84078  // fmlalb z24.s, z3.h, z0.h[2]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z7.h }, p2/Z, [x20]\n"
      ".inst 0x64a84099  // fmlalb z25.s, z4.h, z0.h[2]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a840ba  // fmlalb z26.s, z5.h, z0.h[2]\n"
      ".inst 0x64a840db  // fmlalb z27.s, z6.h, z0.h[2]\n"
      ".inst 0x64a840fc  // fmlalb z28.s, z7.h, z0.h[2]\n"
      ".inst 0x64a84c78  // fmlalt z24.s, z3.h, z0.h[3]\n"
      ".inst 0x64a84c99  // fmlalt z25.s, z4.h, z0.h[3]\n"
      ".inst 0x64a84cba  // fmlalt z26.s, z5.h, z0.h[3]\n"
      ".inst 0x64a84cdb  // fmlalt z27.s, z6.h, z0.h[3]\n"
      ".inst 0x64a84cfc  // fmlalt z28.s, z7.h, z0.h[3]\n"
      "ble 34f\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z10.h }, p2/Z, [x21]\n"
      "ldnt1h { z11.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64b04118  // fmlalb z24.s, z8.h, z0.h[4]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z12.h }, p2/Z, [x20]\n"
      ".inst 0x64b04139  // fmlalb z25.s, z9.h, z0.h[4]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64b0415a  // fmlalb z26.s, z10.h, z0.h[4]\n"
      ".inst 0x64b0417b  // fmlalb z27.s, z11.h, z0.h[4]\n"
      ".inst 0x64b0419c  // fmlalb z28.s, z12.h, z0.h[4]\n"
      ".inst 0x64b04d18  // fmlalt z24.s, z8.h, z0.h[5]\n"
      ".inst 0x64b04d39  // fmlalt z25.s, z9.h, z0.h[5]\n"
      ".inst 0x64b04d5a  // fmlalt z26.s, z10.h, z0.h[5]\n"
      ".inst 0x64b04d7b  // fmlalt z27.s, z11.h, z0.h[5]\n"
      ".inst 0x64b04d9c  // fmlalt z28.s, z12.h, z0.h[5]\n"
      "ble 34f\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z15.h }, p2/Z, [x21]\n"
      "ldnt1h { z16.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64b841b8  // fmlalb z24.s, z13.h, z0.h[6]\n"
      "ldnt1h { z17.h }, p2/Z, [x20]\n"
      ".inst 0x64b841d9  // fmlalb z25.s, z14.h, z0.h[6]\n"
      ".inst 0x64b841fa  // fmlalb z26.s, z15.h, z0.h[6]\n"
      ".inst 0x64b8421b  // fmlalb z27.s, z16.h, z0.h[6]\n"
      ".inst 0x64b8423c  // fmlalb z28.s, z17.h, z0.h[6]\n"
      ".inst 0x64b84db8  // fmlalt z24.s, z13.h, z0.h[7]\n"
      ".inst 0x64b84dd9  // fmlalt z25.s, z14.h, z0.h[7]\n"
      ".inst 0x64b84dfa  // fmlalt z26.s, z15.h, z0.h[7]\n"
      ".inst 0x64b84e1b  // fmlalt z27.s, z16.h, z0.h[7]\n"
      ".inst 0x64b84e3c  // fmlalt z28.s, z17.h, z0.h[7]\n"
      "34:"  // Width 5: Multiply loop: multiply skip
      "fcvt z24.h, p2/m, z24.s\n"
      "fcvt z25.h, p2/m, z25.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "fcvt z25.h, p2/m, z26.s\n"
      "fcvt z26.h, p2/m, z27.s\n"
      "uzp1 z25.h, z25.h, z26.h\n"
      "fcvt z26.h, p2/m, z28.s\n"
      "uzp1 z26.h, z26.h, z26.h\n"
      "tbz %x[flags], #1, 35f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmin z26.h, p2/M, z26.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "fmax z26.h, p2/M, z26.h, z16.h\n"
      "35:"  // Width 5: No activation
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "st1h { z25.h }, p2, [%x[output_ptr], #1, MUL VL]\n"
      "st1h { z26.h }, p1, [%x[output_ptr], #2, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #3\n"
      "b 57f\n"
      "36:"  // Width 6
      "lsl x21, x28, #0x1\n"
      "mov x20, #0x2\n"
      "msub x22, x21, x20, %x[N]\n"
      "mov x26, %x[K]\n"
      "mov x25, %x[A_ptr]\n"
      "add x21, %x[B_ptr], x10\n"
      "add x20, %x[B_ptr], x10, LSL #1\n"
      "whilelt p1.h, XZR, x22\n"
      "cbz x9, 37f\n"
      "ld1h { z24.s }, p2/Z, [x9]\n"
      "ld1h { z25.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1h { z26.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1h { z27.s }, p2/Z, [x9, #3, MUL VL]\n"
      "fcvt z24.s, p2/m, z24.h\n"
      "ld1h { z28.s }, p2/Z, [x9, #4, MUL VL]\n"
      "fcvt z25.s, p2/m, z25.h\n"
      "ld1h { z29.s }, p2/Z, [x9, #5, MUL VL]\n"
      "fcvt z26.s, p2/m, z26.h\n"
      "fcvt z27.s, p2/m, z27.h\n"
      "fcvt z28.s, p2/m, z28.h\n"
      "fcvt z29.s, p2/m, z29.h\n"
      "b 38f\n"
      "37:"  // Width 6: no bias
      "fmov z24.s, #0\n"
      "fmov z25.s, #0\n"
      "fmov z26.s, #0\n"
      "fmov z27.s, #0\n"
      "fmov z28.s, #0\n"
      "fmov z29.s, #0\n"
      "38:"  // Width 6: setup done
      "cmp x26, #0x8\n"
      "ble 40f\n"
      "39:"  // Width 6: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x26, x26, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "cmp x26, #0x8\n"
      "add x25, x25, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z3.h }, p2/Z, [x21]\n"
      "ldnt1h { z4.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a04038  // fmlalb z24.s, z1.h, z0.h[0]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z5.h }, p2/Z, [x20]\n"
      ".inst 0x64a04059  // fmlalb z25.s, z2.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64a0407a  // fmlalb z26.s, z3.h, z0.h[0]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a0409b  // fmlalb z27.s, z4.h, z0.h[0]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a040bc  // fmlalb z28.s, z5.h, z0.h[0]\n"
      ".inst 0x64a04c38  // fmlalt z24.s, z1.h, z0.h[1]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a040dd  // fmlalb z29.s, z6.h, z0.h[0]\n"
      ".inst 0x64a04c59  // fmlalt z25.s, z2.h, z0.h[1]\n"
      "ldnt1h { z9.h }, p2/Z, [x21]\n"
      ".inst 0x64a04c7a  // fmlalt z26.s, z3.h, z0.h[1]\n"
      "ldnt1h { z10.h }, p2/Z, [x21, #1, MUL VL]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64a04c9b  // fmlalt z27.s, z4.h, z0.h[1]\n"
      "ldnt1h { z11.h }, p2/Z, [x20]\n"
      ".inst 0x64a04cbc  // fmlalt z28.s, z5.h, z0.h[1]\n"
      "ldnt1h { z12.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64a840f8  // fmlalb z24.s, z7.h, z0.h[2]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a04cdd  // fmlalt z29.s, z6.h, z0.h[1]\n"
      ".inst 0x64a84119  // fmlalb z25.s, z8.h, z0.h[2]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a8413a  // fmlalb z26.s, z9.h, z0.h[2]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a8415b  // fmlalb z27.s, z10.h, z0.h[2]\n"
      "ldnt1h { z15.h }, p2/Z, [x21]\n"
      ".inst 0x64a8417c  // fmlalb z28.s, z11.h, z0.h[2]\n"
      ".inst 0x64a84cf8  // fmlalt z24.s, z7.h, z0.h[3]\n"
      "ldnt1h { z16.h }, p2/Z, [x21, #1, MUL VL]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64a8419d  // fmlalb z29.s, z12.h, z0.h[2]\n"
      ".inst 0x64a84d19  // fmlalt z25.s, z8.h, z0.h[3]\n"
      "ldnt1h { z17.h }, p2/Z, [x20]\n"
      ".inst 0x64a84d3a  // fmlalt z26.s, z9.h, z0.h[3]\n"
      "ldnt1h { z18.h }, p2/Z, [x20, #1, MUL VL]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a84d5b  // fmlalt z27.s, z10.h, z0.h[3]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a84d7c  // fmlalt z28.s, z11.h, z0.h[3]\n"
      ".inst 0x64b041b8  // fmlalb z24.s, z13.h, z0.h[4]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a84d9d  // fmlalt z29.s, z12.h, z0.h[3]\n"
      ".inst 0x64b041d9  // fmlalb z25.s, z14.h, z0.h[4]\n"
      "ldnt1h { z21.h }, p2/Z, [x21]\n"
      ".inst 0x64b041fa  // fmlalb z26.s, z15.h, z0.h[4]\n"
      "ldnt1h { z22.h }, p2/Z, [x21, #1, MUL VL]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64b0421b  // fmlalb z27.s, z16.h, z0.h[4]\n"
      "ldnt1h { z23.h }, p2/Z, [x20]\n"
      ".inst 0x64b0423c  // fmlalb z28.s, z17.h, z0.h[4]\n"
      ".inst 0x64b04db8  // fmlalt z24.s, z13.h, z0.h[5]\n"
      "ldnt1h { z1.h }, p2/Z, [x20, #1, MUL VL]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64b0425d  // fmlalb z29.s, z18.h, z0.h[4]\n"
      ".inst 0x64b04dd9  // fmlalt z25.s, z14.h, z0.h[5]\n"
      ".inst 0x64b04dfa  // fmlalt z26.s, z15.h, z0.h[5]\n"
      ".inst 0x64b04e1b  // fmlalt z27.s, z16.h, z0.h[5]\n"
      ".inst 0x64b04e3c  // fmlalt z28.s, z17.h, z0.h[5]\n"
      ".inst 0x64b84278  // fmlalb z24.s, z19.h, z0.h[6]\n"
      ".inst 0x64b04e5d  // fmlalt z29.s, z18.h, z0.h[5]\n"
      ".inst 0x64b84299  // fmlalb z25.s, z20.h, z0.h[6]\n"
      ".inst 0x64b842ba  // fmlalb z26.s, z21.h, z0.h[6]\n"
      ".inst 0x64b842db  // fmlalb z27.s, z22.h, z0.h[6]\n"
      ".inst 0x64b842fc  // fmlalb z28.s, z23.h, z0.h[6]\n"
      ".inst 0x64b84e78  // fmlalt z24.s, z19.h, z0.h[7]\n"
      ".inst 0x64b8403d  // fmlalb z29.s, z1.h, z0.h[6]\n"
      ".inst 0x64b84e99  // fmlalt z25.s, z20.h, z0.h[7]\n"
      ".inst 0x64b84eba  // fmlalt z26.s, z21.h, z0.h[7]\n"
      ".inst 0x64b84edb  // fmlalt z27.s, z22.h, z0.h[7]\n"
      ".inst 0x64b84efc  // fmlalt z28.s, z23.h, z0.h[7]\n"
      ".inst 0x64b84c3d  // fmlalt z29.s, z1.h, z0.h[7]\n"
      "bgt 39b\n"
      "40:"  // Width 6: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z4.h }, p2/Z, [x21]\n"
      "ldnt1h { z5.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a04058  // fmlalb z24.s, z2.h, z0.h[0]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z6.h }, p2/Z, [x20]\n"
      ".inst 0x64a04079  // fmlalb z25.s, z3.h, z0.h[0]\n"
      "ldnt1h { z7.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64a0409a  // fmlalb z26.s, z4.h, z0.h[0]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a040bb  // fmlalb z27.s, z5.h, z0.h[0]\n"
      ".inst 0x64a040dc  // fmlalb z28.s, z6.h, z0.h[0]\n"
      ".inst 0x64a04c58  // fmlalt z24.s, z2.h, z0.h[1]\n"
      ".inst 0x64a040fd  // fmlalb z29.s, z7.h, z0.h[0]\n"
      ".inst 0x64a04c79  // fmlalt z25.s, z3.h, z0.h[1]\n"
      ".inst 0x64a04c9a  // fmlalt z26.s, z4.h, z0.h[1]\n"
      ".inst 0x64a04cbb  // fmlalt z27.s, z5.h, z0.h[1]\n"
      ".inst 0x64a04cdc  // fmlalt z28.s, z6.h, z0.h[1]\n"
      ".inst 0x64a04cfd  // fmlalt z29.s, z7.h, z0.h[1]\n"
      "ble 41f\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z10.h }, p2/Z, [x21]\n"
      "ldnt1h { z11.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a84118  // fmlalb z24.s, z8.h, z0.h[2]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z12.h }, p2/Z, [x20]\n"
      ".inst 0x64a84139  // fmlalb z25.s, z9.h, z0.h[2]\n"
      "ldnt1h { z13.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64a8415a  // fmlalb z26.s, z10.h, z0.h[2]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a8417b  // fmlalb z27.s, z11.h, z0.h[2]\n"
      ".inst 0x64a8419c  // fmlalb z28.s, z12.h, z0.h[2]\n"
      ".inst 0x64a84d18  // fmlalt z24.s, z8.h, z0.h[3]\n"
      ".inst 0x64a841bd  // fmlalb z29.s, z13.h, z0.h[2]\n"
      ".inst 0x64a84d39  // fmlalt z25.s, z9.h, z0.h[3]\n"
      ".inst 0x64a84d5a  // fmlalt z26.s, z10.h, z0.h[3]\n"
      ".inst 0x64a84d7b  // fmlalt z27.s, z11.h, z0.h[3]\n"
      ".inst 0x64a84d9c  // fmlalt z28.s, z12.h, z0.h[3]\n"
      ".inst 0x64a84dbd  // fmlalt z29.s, z13.h, z0.h[3]\n"
      "ble 41f\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z16.h }, p2/Z, [x21]\n"
      "ldnt1h { z17.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64b041d8  // fmlalb z24.s, z14.h, z0.h[4]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z18.h }, p2/Z, [x20]\n"
      ".inst 0x64b041f9  // fmlalb z25.s, z15.h, z0.h[4]\n"
      "ldnt1h { z19.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64b0421a  // fmlalb z26.s, z16.h, z0.h[4]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64b0423b  // fmlalb z27.s, z17.h, z0.h[4]\n"
      ".inst 0x64b0425c  // fmlalb z28.s, z18.h, z0.h[4]\n"
      ".inst 0x64b04dd8  // fmlalt z24.s, z14.h, z0.h[5]\n"
      ".inst 0x64b0427d  // fmlalb z29.s, z19.h, z0.h[4]\n"
      ".inst 0x64b04df9  // fmlalt z25.s, z15.h, z0.h[5]\n"
      ".inst 0x64b04e1a  // fmlalt z26.s, z16.h, z0.h[5]\n"
      ".inst 0x64b04e3b  // fmlalt z27.s, z17.h, z0.h[5]\n"
      ".inst 0x64b04e5c  // fmlalt z28.s, z18.h, z0.h[5]\n"
      ".inst 0x64b04e7d  // fmlalt z29.s, z19.h, z0.h[5]\n"
      "ble 41f\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z22.h }, p2/Z, [x21]\n"
      "ldnt1h { z23.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64b84298  // fmlalb z24.s, z20.h, z0.h[6]\n"
      "ldnt1h { z1.h }, p2/Z, [x20]\n"
      ".inst 0x64b842b9  // fmlalb z25.s, z21.h, z0.h[6]\n"
      "ldnt1h { z2.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64b842da  // fmlalb z26.s, z22.h, z0.h[6]\n"
      ".inst 0x64b842fb  // fmlalb z27.s, z23.h, z0.h[6]\n"
      ".inst 0x64b8403c  // fmlalb z28.s, z1.h, z0.h[6]\n"
      ".inst 0x64b84e98  // fmlalt z24.s, z20.h, z0.h[7]\n"
      ".inst 0x64b8405d  // fmlalb z29.s, z2.h, z0.h[6]\n"
      ".inst 0x64b84eb9  // fmlalt z25.s, z21.h, z0.h[7]\n"
      ".inst 0x64b84eda  // fmlalt z26.s, z22.h, z0.h[7]\n"
      ".inst 0x64b84efb  // fmlalt z27.s, z23.h, z0.h[7]\n"
      ".inst 0x64b84c3c  // fmlalt z28.s, z1.h, z0.h[7]\n"
      ".inst 0x64b84c5d  // fmlalt z29.s, z2.h, z0.h[7]\n"
      "41:"  // Width 6: Multiply loop: multiply skip
      "fcvt z24.h, p2/m, z24.s\n"
      "fcvt z25.h, p2/m, z25.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "fcvt z25.h, p2/m, z26.s\n"
      "fcvt z26.h, p2/m, z27.s\n"
      "fcvt z27.h, p2/m, z29.s\n"
      "uzp1 z25.h, z25.h, z26.h\n"
      "fcvt z26.h, p2/m, z28.s\n"
      "uzp1 z26.h, z26.h, z27.h\n"
      "tbz %x[flags], #1, 42f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmin z26.h, p2/M, z26.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "fmax z26.h, p2/M, z26.h, z16.h\n"
      "42:"  // Width 6: No activation
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "st1h { z25.h }, p2, [%x[output_ptr], #1, MUL VL]\n"
      "st1h { z26.h }, p1, [%x[output_ptr], #2, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #3\n"
      "b 57f\n"
      "43:"  // Width 7
      "lsl x21, x28, #0x1\n"
      "mov x20, #0x3\n"
      "add x23, %x[B_ptr], x10, LSL #1\n"
      "msub x22, x21, x20, %x[N]\n"
      "mov x26, %x[K]\n"
      "mov x25, %x[A_ptr]\n"
      "add x21, %x[B_ptr], x10\n"
      "add x20, x23, x10\n"
      "whilelt p1.h, XZR, x22\n"
      "cbz x9, 44f\n"
      "ld1h { z24.s }, p2/Z, [x9]\n"
      "ld1h { z25.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1h { z26.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1h { z27.s }, p2/Z, [x9, #3, MUL VL]\n"
      "fcvt z24.s, p2/m, z24.h\n"
      "ld1h { z28.s }, p2/Z, [x9, #4, MUL VL]\n"
      "fcvt z25.s, p2/m, z25.h\n"
      "ld1h { z29.s }, p2/Z, [x9, #5, MUL VL]\n"
      "fcvt z26.s, p2/m, z26.h\n"
      "ld1h { z30.s }, p2/Z, [x9, #6, MUL VL]\n"
      "fcvt z27.s, p2/m, z27.h\n"
      "fcvt z28.s, p2/m, z28.h\n"
      "fcvt z29.s, p2/m, z29.h\n"
      "fcvt z30.s, p2/m, z30.h\n"
      "b 45f\n"
      "44:"  // Width 7: no bias
      "fmov z24.s, #0\n"
      "fmov z25.s, #0\n"
      "fmov z26.s, #0\n"
      "fmov z27.s, #0\n"
      "fmov z28.s, #0\n"
      "fmov z29.s, #0\n"
      "fmov z30.s, #0\n"
      "45:"  // Width 7: setup done
      "cmp x26, #0x8\n"
      "ble 47f\n"
      "46:"  // Width 7: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x26, x26, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "cmp x26, #0x8\n"
      "add x25, x25, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z3.h }, p2/Z, [x21]\n"
      "ldnt1h { z4.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a04038  // fmlalb z24.s, z1.h, z0.h[0]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z5.h }, p2/Z, [x23]\n"
      ".inst 0x64a04059  // fmlalb z25.s, z2.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [x23, #1, MUL VL]\n"
      ".inst 0x64a0407a  // fmlalb z26.s, z3.h, z0.h[0]\n"
      "addvl x23, x23, #2\n"
      "ldnt1h { z7.h }, p2/Z, [x20]\n"
      ".inst 0x64a0409b  // fmlalb z27.s, z4.h, z0.h[0]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a040bc  // fmlalb z28.s, z5.h, z0.h[0]\n"
      ".inst 0x64a04c38  // fmlalt z24.s, z1.h, z0.h[1]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a040dd  // fmlalb z29.s, z6.h, z0.h[0]\n"
      ".inst 0x64a04c59  // fmlalt z25.s, z2.h, z0.h[1]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a040fe  // fmlalb z30.s, z7.h, z0.h[0]\n"
      ".inst 0x64a04c7a  // fmlalt z26.s, z3.h, z0.h[1]\n"
      "ldnt1h { z10.h }, p2/Z, [x21]\n"
      ".inst 0x64a04c9b  // fmlalt z27.s, z4.h, z0.h[1]\n"
      "ldnt1h { z11.h }, p2/Z, [x21, #1, MUL VL]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64a04cbc  // fmlalt z28.s, z5.h, z0.h[1]\n"
      "ldnt1h { z12.h }, p2/Z, [x23]\n"
      ".inst 0x64a84118  // fmlalb z24.s, z8.h, z0.h[2]\n"
      ".inst 0x64a04cdd  // fmlalt z29.s, z6.h, z0.h[1]\n"
      "ldnt1h { z13.h }, p2/Z, [x23, #1, MUL VL]\n"
      ".inst 0x64a84139  // fmlalb z25.s, z9.h, z0.h[2]\n"
      "addvl x23, x23, #2\n"
      ".inst 0x64a04cfe  // fmlalt z30.s, z7.h, z0.h[1]\n"
      "ldnt1h { z14.h }, p2/Z, [x20]\n"
      ".inst 0x64a8415a  // fmlalb z26.s, z10.h, z0.h[2]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a8417b  // fmlalb z27.s, z11.h, z0.h[2]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a8419c  // fmlalb z28.s, z12.h, z0.h[2]\n"
      ".inst 0x64a84d18  // fmlalt z24.s, z8.h, z0.h[3]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a841bd  // fmlalb z29.s, z13.h, z0.h[2]\n"
      ".inst 0x64a84d39  // fmlalt z25.s, z9.h, z0.h[3]\n"
      "ldnt1h { z17.h }, p2/Z, [x21]\n"
      ".inst 0x64a841de  // fmlalb z30.s, z14.h, z0.h[2]\n"
      ".inst 0x64a84d5a  // fmlalt z26.s, z10.h, z0.h[3]\n"
      "ldnt1h { z18.h }, p2/Z, [x21, #1, MUL VL]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64a84d7b  // fmlalt z27.s, z11.h, z0.h[3]\n"
      "ldnt1h { z19.h }, p2/Z, [x23]\n"
      ".inst 0x64a84d9c  // fmlalt z28.s, z12.h, z0.h[3]\n"
      "ldnt1h { z20.h }, p2/Z, [x23, #1, MUL VL]\n"
      ".inst 0x64b041f8  // fmlalb z24.s, z15.h, z0.h[4]\n"
      "addvl x23, x23, #2\n"
      ".inst 0x64a84dbd  // fmlalt z29.s, z13.h, z0.h[3]\n"
      "ldnt1h { z21.h }, p2/Z, [x20]\n"
      ".inst 0x64b04219  // fmlalb z25.s, z16.h, z0.h[4]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a84dde  // fmlalt z30.s, z14.h, z0.h[3]\n"
      ".inst 0x64b0423a  // fmlalb z26.s, z17.h, z0.h[4]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64b0425b  // fmlalb z27.s, z18.h, z0.h[4]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64b0427c  // fmlalb z28.s, z19.h, z0.h[4]\n"
      ".inst 0x64b04df8  // fmlalt z24.s, z15.h, z0.h[5]\n"
      "ldnt1h { z1.h }, p2/Z, [x21]\n"
      ".inst 0x64b0429d  // fmlalb z29.s, z20.h, z0.h[4]\n"
      ".inst 0x64b04e19  // fmlalt z25.s, z16.h, z0.h[5]\n"
      "ldnt1h { z2.h }, p2/Z, [x21, #1, MUL VL]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64b042be  // fmlalb z30.s, z21.h, z0.h[4]\n"
      ".inst 0x64b04e3a  // fmlalt z26.s, z17.h, z0.h[5]\n"
      "ldnt1h { z3.h }, p2/Z, [x23]\n"
      ".inst 0x64b04e5b  // fmlalt z27.s, z18.h, z0.h[5]\n"
      "ldnt1h { z4.h }, p2/Z, [x23, #1, MUL VL]\n"
      "addvl x23, x23, #2\n"
      ".inst 0x64b04e7c  // fmlalt z28.s, z19.h, z0.h[5]\n"
      "ldnt1h { z5.h }, p2/Z, [x20]\n"
      ".inst 0x64b842d8  // fmlalb z24.s, z22.h, z0.h[6]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64b04e9d  // fmlalt z29.s, z20.h, z0.h[5]\n"
      ".inst 0x64b842f9  // fmlalb z25.s, z23.h, z0.h[6]\n"
      ".inst 0x64b04ebe  // fmlalt z30.s, z21.h, z0.h[5]\n"
      ".inst 0x64b8403a  // fmlalb z26.s, z1.h, z0.h[6]\n"
      ".inst 0x64b8405b  // fmlalb z27.s, z2.h, z0.h[6]\n"
      ".inst 0x64b8407c  // fmlalb z28.s, z3.h, z0.h[6]\n"
      ".inst 0x64b84ed8  // fmlalt z24.s, z22.h, z0.h[7]\n"
      ".inst 0x64b8409d  // fmlalb z29.s, z4.h, z0.h[6]\n"
      ".inst 0x64b84ef9  // fmlalt z25.s, z23.h, z0.h[7]\n"
      ".inst 0x64b840be  // fmlalb z30.s, z5.h, z0.h[6]\n"
      ".inst 0x64b84c3a  // fmlalt z26.s, z1.h, z0.h[7]\n"
      ".inst 0x64b84c5b  // fmlalt z27.s, z2.h, z0.h[7]\n"
      ".inst 0x64b84c7c  // fmlalt z28.s, z3.h, z0.h[7]\n"
      ".inst 0x64b84c9d  // fmlalt z29.s, z4.h, z0.h[7]\n"
      ".inst 0x64b84cbe  // fmlalt z30.s, z5.h, z0.h[7]\n"
      "bgt 46b\n"
      "47:"  // Width 7: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z8.h }, p2/Z, [x21]\n"
      "ldnt1h { z9.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a040d8  // fmlalb z24.s, z6.h, z0.h[0]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z10.h }, p2/Z, [x23]\n"
      ".inst 0x64a040f9  // fmlalb z25.s, z7.h, z0.h[0]\n"
      "ldnt1h { z11.h }, p2/Z, [x23, #1, MUL VL]\n"
      ".inst 0x64a0411a  // fmlalb z26.s, z8.h, z0.h[0]\n"
      "addvl x23, x23, #2\n"
      "ldnt1h { z12.h }, p2/Z, [x20]\n"
      ".inst 0x64a0413b  // fmlalb z27.s, z9.h, z0.h[0]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a0415c  // fmlalb z28.s, z10.h, z0.h[0]\n"
      ".inst 0x64a04cd8  // fmlalt z24.s, z6.h, z0.h[1]\n"
      ".inst 0x64a0417d  // fmlalb z29.s, z11.h, z0.h[0]\n"
      ".inst 0x64a04cf9  // fmlalt z25.s, z7.h, z0.h[1]\n"
      ".inst 0x64a0419e  // fmlalb z30.s, z12.h, z0.h[0]\n"
      ".inst 0x64a04d1a  // fmlalt z26.s, z8.h, z0.h[1]\n"
      ".inst 0x64a04d3b  // fmlalt z27.s, z9.h, z0.h[1]\n"
      ".inst 0x64a04d5c  // fmlalt z28.s, z10.h, z0.h[1]\n"
      ".inst 0x64a04d7d  // fmlalt z29.s, z11.h, z0.h[1]\n"
      ".inst 0x64a04d9e  // fmlalt z30.s, z12.h, z0.h[1]\n"
      "ble 48f\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z15.h }, p2/Z, [x21]\n"
      "ldnt1h { z16.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a841b8  // fmlalb z24.s, z13.h, z0.h[2]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z17.h }, p2/Z, [x23]\n"
      ".inst 0x64a841d9  // fmlalb z25.s, z14.h, z0.h[2]\n"
      "ldnt1h { z18.h }, p2/Z, [x23, #1, MUL VL]\n"
      ".inst 0x64a841fa  // fmlalb z26.s, z15.h, z0.h[2]\n"
      "addvl x23, x23, #2\n"
      "ldnt1h { z19.h }, p2/Z, [x20]\n"
      ".inst 0x64a8421b  // fmlalb z27.s, z16.h, z0.h[2]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a8423c  // fmlalb z28.s, z17.h, z0.h[2]\n"
      ".inst 0x64a84db8  // fmlalt z24.s, z13.h, z0.h[3]\n"
      ".inst 0x64a8425d  // fmlalb z29.s, z18.h, z0.h[2]\n"
      ".inst 0x64a84dd9  // fmlalt z25.s, z14.h, z0.h[3]\n"
      ".inst 0x64a8427e  // fmlalb z30.s, z19.h, z0.h[2]\n"
      ".inst 0x64a84dfa  // fmlalt z26.s, z15.h, z0.h[3]\n"
      ".inst 0x64a84e1b  // fmlalt z27.s, z16.h, z0.h[3]\n"
      ".inst 0x64a84e3c  // fmlalt z28.s, z17.h, z0.h[3]\n"
      ".inst 0x64a84e5d  // fmlalt z29.s, z18.h, z0.h[3]\n"
      ".inst 0x64a84e7e  // fmlalt z30.s, z19.h, z0.h[3]\n"
      "ble 48f\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z22.h }, p2/Z, [x21]\n"
      "ldnt1h { z23.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64b04298  // fmlalb z24.s, z20.h, z0.h[4]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z1.h }, p2/Z, [x23]\n"
      ".inst 0x64b042b9  // fmlalb z25.s, z21.h, z0.h[4]\n"
      "ldnt1h { z2.h }, p2/Z, [x23, #1, MUL VL]\n"
      ".inst 0x64b042da  // fmlalb z26.s, z22.h, z0.h[4]\n"
      "addvl x23, x23, #2\n"
      "ldnt1h { z3.h }, p2/Z, [x20]\n"
      ".inst 0x64b042fb  // fmlalb z27.s, z23.h, z0.h[4]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64b0403c  // fmlalb z28.s, z1.h, z0.h[4]\n"
      ".inst 0x64b04e98  // fmlalt z24.s, z20.h, z0.h[5]\n"
      ".inst 0x64b0405d  // fmlalb z29.s, z2.h, z0.h[4]\n"
      ".inst 0x64b04eb9  // fmlalt z25.s, z21.h, z0.h[5]\n"
      ".inst 0x64b0407e  // fmlalb z30.s, z3.h, z0.h[4]\n"
      ".inst 0x64b04eda  // fmlalt z26.s, z22.h, z0.h[5]\n"
      ".inst 0x64b04efb  // fmlalt z27.s, z23.h, z0.h[5]\n"
      ".inst 0x64b04c3c  // fmlalt z28.s, z1.h, z0.h[5]\n"
      ".inst 0x64b04c5d  // fmlalt z29.s, z2.h, z0.h[5]\n"
      ".inst 0x64b04c7e  // fmlalt z30.s, z3.h, z0.h[5]\n"
      "ble 48f\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z6.h }, p2/Z, [x21]\n"
      "ldnt1h { z7.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64b84098  // fmlalb z24.s, z4.h, z0.h[6]\n"
      "ldnt1h { z8.h }, p2/Z, [x23]\n"
      ".inst 0x64b840b9  // fmlalb z25.s, z5.h, z0.h[6]\n"
      "ldnt1h { z9.h }, p2/Z, [x23, #1, MUL VL]\n"
      ".inst 0x64b840da  // fmlalb z26.s, z6.h, z0.h[6]\n"
      "ldnt1h { z10.h }, p2/Z, [x20]\n"
      ".inst 0x64b840fb  // fmlalb z27.s, z7.h, z0.h[6]\n"
      ".inst 0x64b8411c  // fmlalb z28.s, z8.h, z0.h[6]\n"
      ".inst 0x64b84c98  // fmlalt z24.s, z4.h, z0.h[7]\n"
      ".inst 0x64b8413d  // fmlalb z29.s, z9.h, z0.h[6]\n"
      ".inst 0x64b84cb9  // fmlalt z25.s, z5.h, z0.h[7]\n"
      ".inst 0x64b8415e  // fmlalb z30.s, z10.h, z0.h[6]\n"
      ".inst 0x64b84cda  // fmlalt z26.s, z6.h, z0.h[7]\n"
      ".inst 0x64b84cfb  // fmlalt z27.s, z7.h, z0.h[7]\n"
      ".inst 0x64b84d1c  // fmlalt z28.s, z8.h, z0.h[7]\n"
      ".inst 0x64b84d3d  // fmlalt z29.s, z9.h, z0.h[7]\n"
      ".inst 0x64b84d5e  // fmlalt z30.s, z10.h, z0.h[7]\n"
      "48:"  // Width 7: Multiply loop: multiply skip
      "fcvt z24.h, p2/m, z24.s\n"
      "fcvt z25.h, p2/m, z25.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "fcvt z25.h, p2/m, z26.s\n"
      "fcvt z26.h, p2/m, z27.s\n"
      "fcvt z27.h, p2/m, z29.s\n"
      "uzp1 z25.h, z25.h, z26.h\n"
      "fcvt z26.h, p2/m, z28.s\n"
      "uzp1 z26.h, z26.h, z27.h\n"
      "fcvt z27.h, p2/m, z30.s\n"
      "uzp1 z27.h, z27.h, z27.h\n"
      "tbz %x[flags], #1, 49f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmin z26.h, p2/M, z26.h, z17.h\n"
      "fmin z27.h, p2/M, z27.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "fmax z26.h, p2/M, z26.h, z16.h\n"
      "fmax z27.h, p2/M, z27.h, z16.h\n"
      "49:"  // Width 7: No activation
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "st1h { z25.h }, p2, [%x[output_ptr], #1, MUL VL]\n"
      "st1h { z26.h }, p2, [%x[output_ptr], #2, MUL VL]\n"
      "st1h { z27.h }, p1, [%x[output_ptr], #3, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #4\n"
      "b 57f\n"
      "50:"  // Width 8
      "lsl x21, x28, #0x1\n"
      "mov x20, #0x3\n"
      "add x24, %x[B_ptr], x10, LSL #1\n"
      "msub x23, x21, x20, %x[N]\n"
      "mov x26, %x[K]\n"
      "mov x25, %x[A_ptr]\n"
      "add x21, %x[B_ptr], x10\n"
      "add x20, x24, x10\n"
      "add x22, %x[B_ptr], x10, LSL #2\n"
      "whilelt p1.h, XZR, x23\n"
      "cbz x9, 51f\n"
      "ld1h { z24.s }, p2/Z, [x9]\n"
      "ld1h { z25.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1h { z26.s }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1h { z27.s }, p2/Z, [x9, #3, MUL VL]\n"
      "fcvt z24.s, p2/m, z24.h\n"
      "ld1h { z28.s }, p2/Z, [x9, #4, MUL VL]\n"
      "fcvt z25.s, p2/m, z25.h\n"
      "ld1h { z29.s }, p2/Z, [x9, #5, MUL VL]\n"
      "fcvt z26.s, p2/m, z26.h\n"
      "ld1h { z30.s }, p2/Z, [x9, #6, MUL VL]\n"
      "fcvt z27.s, p2/m, z27.h\n"
      "ld1h { z31.s }, p2/Z, [x9, #7, MUL VL]\n"
      "inch x9, ALL, MUL #8\n"
      "fcvt z28.s, p2/m, z28.h\n"
      "fcvt z29.s, p2/m, z29.h\n"
      "fcvt z30.s, p2/m, z30.h\n"
      "fcvt z31.s, p2/m, z31.h\n"
      "b 52f\n"
      "51:"  // Width 8: no bias
      "fmov z24.s, #0\n"
      "fmov z25.s, #0\n"
      "fmov z26.s, #0\n"
      "fmov z27.s, #0\n"
      "fmov z28.s, #0\n"
      "fmov z29.s, #0\n"
      "fmov z30.s, #0\n"
      "fmov z31.s, #0\n"
      "52:"  // Width 8: setup done
      "cmp x26, #0x8\n"
      "ble 54f\n"
      "53:"  // Width 8: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x26, x26, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "cmp x26, #0x8\n"
      "add x25, x25, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z3.h }, p2/Z, [x21]\n"
      "ldnt1h { z4.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a04038  // fmlalb z24.s, z1.h, z0.h[0]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z5.h }, p2/Z, [x24]\n"
      ".inst 0x64a04059  // fmlalb z25.s, z2.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [x24, #1, MUL VL]\n"
      ".inst 0x64a0407a  // fmlalb z26.s, z3.h, z0.h[0]\n"
      "addvl x24, x24, #2\n"
      "ldnt1h { z7.h }, p2/Z, [x20]\n"
      ".inst 0x64a0409b  // fmlalb z27.s, z4.h, z0.h[0]\n"
      "ldnt1h { z8.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64a040bc  // fmlalb z28.s, z5.h, z0.h[0]\n"
      ".inst 0x64a04c38  // fmlalt z24.s, z1.h, z0.h[1]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a040dd  // fmlalb z29.s, z6.h, z0.h[0]\n"
      ".inst 0x64a04c59  // fmlalt z25.s, z2.h, z0.h[1]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a040fe  // fmlalb z30.s, z7.h, z0.h[0]\n"
      ".inst 0x64a04c7a  // fmlalt z26.s, z3.h, z0.h[1]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a0411f  // fmlalb z31.s, z8.h, z0.h[0]\n"
      ".inst 0x64a04c9b  // fmlalt z27.s, z4.h, z0.h[1]\n"
      "ldnt1h { z11.h }, p2/Z, [x21]\n"
      ".inst 0x64a04cbc  // fmlalt z28.s, z5.h, z0.h[1]\n"
      "ldnt1h { z12.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a84138  // fmlalb z24.s, z9.h, z0.h[2]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64a04cdd  // fmlalt z29.s, z6.h, z0.h[1]\n"
      "ldnt1h { z13.h }, p2/Z, [x24]\n"
      ".inst 0x64a84159  // fmlalb z25.s, z10.h, z0.h[2]\n"
      ".inst 0x64a04cfe  // fmlalt z30.s, z7.h, z0.h[1]\n"
      "ldnt1h { z14.h }, p2/Z, [x24, #1, MUL VL]\n"
      ".inst 0x64a8417a  // fmlalb z26.s, z11.h, z0.h[2]\n"
      "addvl x24, x24, #2\n"
      ".inst 0x64a04d1f  // fmlalt z31.s, z8.h, z0.h[1]\n"
      "ldnt1h { z15.h }, p2/Z, [x20]\n"
      ".inst 0x64a8419b  // fmlalb z27.s, z12.h, z0.h[2]\n"
      "ldnt1h { z16.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64a841bc  // fmlalb z28.s, z13.h, z0.h[2]\n"
      ".inst 0x64a84d38  // fmlalt z24.s, z9.h, z0.h[3]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a841dd  // fmlalb z29.s, z14.h, z0.h[2]\n"
      ".inst 0x64a84d59  // fmlalt z25.s, z10.h, z0.h[3]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64a841fe  // fmlalb z30.s, z15.h, z0.h[2]\n"
      ".inst 0x64a84d7a  // fmlalt z26.s, z11.h, z0.h[3]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64a8421f  // fmlalb z31.s, z16.h, z0.h[2]\n"
      ".inst 0x64a84d9b  // fmlalt z27.s, z12.h, z0.h[3]\n"
      "ldnt1h { z19.h }, p2/Z, [x21]\n"
      ".inst 0x64a84dbc  // fmlalt z28.s, z13.h, z0.h[3]\n"
      "ldnt1h { z20.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64b04238  // fmlalb z24.s, z17.h, z0.h[4]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64a84ddd  // fmlalt z29.s, z14.h, z0.h[3]\n"
      "ldnt1h { z21.h }, p2/Z, [x24]\n"
      ".inst 0x64b04259  // fmlalb z25.s, z18.h, z0.h[4]\n"
      ".inst 0x64a84dfe  // fmlalt z30.s, z15.h, z0.h[3]\n"
      "ldnt1h { z22.h }, p2/Z, [x24, #1, MUL VL]\n"
      ".inst 0x64b0427a  // fmlalb z26.s, z19.h, z0.h[4]\n"
      "addvl x24, x24, #2\n"
      ".inst 0x64a84e1f  // fmlalt z31.s, z16.h, z0.h[3]\n"
      "ldnt1h { z23.h }, p2/Z, [x20]\n"
      ".inst 0x64b0429b  // fmlalb z27.s, z20.h, z0.h[4]\n"
      "ldnt1h { z1.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64b042bc  // fmlalb z28.s, z21.h, z0.h[4]\n"
      ".inst 0x64b04e38  // fmlalt z24.s, z17.h, z0.h[5]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64b042dd  // fmlalb z29.s, z22.h, z0.h[4]\n"
      ".inst 0x64b04e59  // fmlalt z25.s, z18.h, z0.h[5]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      ".inst 0x64b042fe  // fmlalb z30.s, z23.h, z0.h[4]\n"
      ".inst 0x64b04e7a  // fmlalt z26.s, z19.h, z0.h[5]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      ".inst 0x64b0403f  // fmlalb z31.s, z1.h, z0.h[4]\n"
      ".inst 0x64b04e9b  // fmlalt z27.s, z20.h, z0.h[5]\n"
      "ldnt1h { z4.h }, p2/Z, [x21]\n"
      ".inst 0x64b04ebc  // fmlalt z28.s, z21.h, z0.h[5]\n"
      "ldnt1h { z5.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64b84058  // fmlalb z24.s, z2.h, z0.h[6]\n"
      "addvl x21, x21, #2\n"
      ".inst 0x64b04edd  // fmlalt z29.s, z22.h, z0.h[5]\n"
      "ldnt1h { z6.h }, p2/Z, [x24]\n"
      ".inst 0x64b84079  // fmlalb z25.s, z3.h, z0.h[6]\n"
      ".inst 0x64b04efe  // fmlalt z30.s, z23.h, z0.h[5]\n"
      "ldnt1h { z7.h }, p2/Z, [x24, #1, MUL VL]\n"
      ".inst 0x64b8409a  // fmlalb z26.s, z4.h, z0.h[6]\n"
      "addvl x24, x24, #2\n"
      ".inst 0x64b04c3f  // fmlalt z31.s, z1.h, z0.h[5]\n"
      "ldnt1h { z8.h }, p2/Z, [x20]\n"
      ".inst 0x64b840bb  // fmlalb z27.s, z5.h, z0.h[6]\n"
      "ldnt1h { z9.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64b840dc  // fmlalb z28.s, z6.h, z0.h[6]\n"
      ".inst 0x64b84c58  // fmlalt z24.s, z2.h, z0.h[7]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64b840fd  // fmlalb z29.s, z7.h, z0.h[6]\n"
      ".inst 0x64b84c79  // fmlalt z25.s, z3.h, z0.h[7]\n"
      ".inst 0x64b8411e  // fmlalb z30.s, z8.h, z0.h[6]\n"
      ".inst 0x64b84c9a  // fmlalt z26.s, z4.h, z0.h[7]\n"
      ".inst 0x64b8413f  // fmlalb z31.s, z9.h, z0.h[6]\n"
      ".inst 0x64b84cbb  // fmlalt z27.s, z5.h, z0.h[7]\n"
      ".inst 0x64b84cdc  // fmlalt z28.s, z6.h, z0.h[7]\n"
      ".inst 0x64b84cfd  // fmlalt z29.s, z7.h, z0.h[7]\n"
      ".inst 0x64b84d1e  // fmlalt z30.s, z8.h, z0.h[7]\n"
      ".inst 0x64b84d3f  // fmlalt z31.s, z9.h, z0.h[7]\n"
      "bgt 53b\n"
      "54:"  // Width 8: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x26\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z12.h }, p2/Z, [x21]\n"
      "ldnt1h { z13.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a04158  // fmlalb z24.s, z10.h, z0.h[0]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z14.h }, p2/Z, [x24]\n"
      ".inst 0x64a04179  // fmlalb z25.s, z11.h, z0.h[0]\n"
      "ldnt1h { z15.h }, p2/Z, [x24, #1, MUL VL]\n"
      ".inst 0x64a0419a  // fmlalb z26.s, z12.h, z0.h[0]\n"
      "addvl x24, x24, #2\n"
      "ldnt1h { z16.h }, p2/Z, [x20]\n"
      ".inst 0x64a041bb  // fmlalb z27.s, z13.h, z0.h[0]\n"
      "ldnt1h { z17.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64a041dc  // fmlalb z28.s, z14.h, z0.h[0]\n"
      ".inst 0x64a04d58  // fmlalt z24.s, z10.h, z0.h[1]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a041fd  // fmlalb z29.s, z15.h, z0.h[0]\n"
      ".inst 0x64a04d79  // fmlalt z25.s, z11.h, z0.h[1]\n"
      ".inst 0x64a0421e  // fmlalb z30.s, z16.h, z0.h[0]\n"
      ".inst 0x64a04d9a  // fmlalt z26.s, z12.h, z0.h[1]\n"
      ".inst 0x64a0423f  // fmlalb z31.s, z17.h, z0.h[0]\n"
      ".inst 0x64a04dbb  // fmlalt z27.s, z13.h, z0.h[1]\n"
      ".inst 0x64a04ddc  // fmlalt z28.s, z14.h, z0.h[1]\n"
      ".inst 0x64a04dfd  // fmlalt z29.s, z15.h, z0.h[1]\n"
      ".inst 0x64a04e1e  // fmlalt z30.s, z16.h, z0.h[1]\n"
      ".inst 0x64a04e3f  // fmlalt z31.s, z17.h, z0.h[1]\n"
      "ble 55f\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z20.h }, p2/Z, [x21]\n"
      "ldnt1h { z21.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64a84258  // fmlalb z24.s, z18.h, z0.h[2]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z22.h }, p2/Z, [x24]\n"
      ".inst 0x64a84279  // fmlalb z25.s, z19.h, z0.h[2]\n"
      "ldnt1h { z23.h }, p2/Z, [x24, #1, MUL VL]\n"
      ".inst 0x64a8429a  // fmlalb z26.s, z20.h, z0.h[2]\n"
      "addvl x24, x24, #2\n"
      "ldnt1h { z1.h }, p2/Z, [x20]\n"
      ".inst 0x64a842bb  // fmlalb z27.s, z21.h, z0.h[2]\n"
      "ldnt1h { z2.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64a842dc  // fmlalb z28.s, z22.h, z0.h[2]\n"
      ".inst 0x64a84e58  // fmlalt z24.s, z18.h, z0.h[3]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64a842fd  // fmlalb z29.s, z23.h, z0.h[2]\n"
      ".inst 0x64a84e79  // fmlalt z25.s, z19.h, z0.h[3]\n"
      ".inst 0x64a8403e  // fmlalb z30.s, z1.h, z0.h[2]\n"
      ".inst 0x64a84e9a  // fmlalt z26.s, z20.h, z0.h[3]\n"
      ".inst 0x64a8405f  // fmlalb z31.s, z2.h, z0.h[2]\n"
      ".inst 0x64a84ebb  // fmlalt z27.s, z21.h, z0.h[3]\n"
      ".inst 0x64a84edc  // fmlalt z28.s, z22.h, z0.h[3]\n"
      ".inst 0x64a84efd  // fmlalt z29.s, z23.h, z0.h[3]\n"
      ".inst 0x64a84c3e  // fmlalt z30.s, z1.h, z0.h[3]\n"
      ".inst 0x64a84c5f  // fmlalt z31.s, z2.h, z0.h[3]\n"
      "ble 55f\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x26, x26, #0x2\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z5.h }, p2/Z, [x21]\n"
      "ldnt1h { z6.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64b04078  // fmlalb z24.s, z3.h, z0.h[4]\n"
      "addvl x21, x21, #2\n"
      "ldnt1h { z7.h }, p2/Z, [x24]\n"
      ".inst 0x64b04099  // fmlalb z25.s, z4.h, z0.h[4]\n"
      "ldnt1h { z8.h }, p2/Z, [x24, #1, MUL VL]\n"
      ".inst 0x64b040ba  // fmlalb z26.s, z5.h, z0.h[4]\n"
      "addvl x24, x24, #2\n"
      "ldnt1h { z9.h }, p2/Z, [x20]\n"
      ".inst 0x64b040db  // fmlalb z27.s, z6.h, z0.h[4]\n"
      "ldnt1h { z10.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64b040fc  // fmlalb z28.s, z7.h, z0.h[4]\n"
      ".inst 0x64b04c78  // fmlalt z24.s, z3.h, z0.h[5]\n"
      "addvl x20, x20, #2\n"
      ".inst 0x64b0411d  // fmlalb z29.s, z8.h, z0.h[4]\n"
      ".inst 0x64b04c99  // fmlalt z25.s, z4.h, z0.h[5]\n"
      ".inst 0x64b0413e  // fmlalb z30.s, z9.h, z0.h[4]\n"
      ".inst 0x64b04cba  // fmlalt z26.s, z5.h, z0.h[5]\n"
      ".inst 0x64b0415f  // fmlalb z31.s, z10.h, z0.h[4]\n"
      ".inst 0x64b04cdb  // fmlalt z27.s, z6.h, z0.h[5]\n"
      ".inst 0x64b04cfc  // fmlalt z28.s, z7.h, z0.h[5]\n"
      ".inst 0x64b04d1d  // fmlalt z29.s, z8.h, z0.h[5]\n"
      ".inst 0x64b04d3e  // fmlalt z30.s, z9.h, z0.h[5]\n"
      ".inst 0x64b04d5f  // fmlalt z31.s, z10.h, z0.h[5]\n"
      "ble 55f\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #2\n"
      "ldnt1h { z13.h }, p2/Z, [x21]\n"
      "ldnt1h { z14.h }, p2/Z, [x21, #1, MUL VL]\n"
      ".inst 0x64b84178  // fmlalb z24.s, z11.h, z0.h[6]\n"
      "ldnt1h { z15.h }, p2/Z, [x24]\n"
      ".inst 0x64b84199  // fmlalb z25.s, z12.h, z0.h[6]\n"
      "ldnt1h { z16.h }, p2/Z, [x24, #1, MUL VL]\n"
      ".inst 0x64b841ba  // fmlalb z26.s, z13.h, z0.h[6]\n"
      "ldnt1h { z17.h }, p2/Z, [x20]\n"
      ".inst 0x64b841db  // fmlalb z27.s, z14.h, z0.h[6]\n"
      "ldnt1h { z18.h }, p2/Z, [x20, #1, MUL VL]\n"
      ".inst 0x64b841fc  // fmlalb z28.s, z15.h, z0.h[6]\n"
      ".inst 0x64b84d78  // fmlalt z24.s, z11.h, z0.h[7]\n"
      ".inst 0x64b8421d  // fmlalb z29.s, z16.h, z0.h[6]\n"
      ".inst 0x64b84d99  // fmlalt z25.s, z12.h, z0.h[7]\n"
      ".inst 0x64b8423e  // fmlalb z30.s, z17.h, z0.h[6]\n"
      ".inst 0x64b84dba  // fmlalt z26.s, z13.h, z0.h[7]\n"
      ".inst 0x64b8425f  // fmlalb z31.s, z18.h, z0.h[6]\n"
      ".inst 0x64b84ddb  // fmlalt z27.s, z14.h, z0.h[7]\n"
      ".inst 0x64b84dfc  // fmlalt z28.s, z15.h, z0.h[7]\n"
      ".inst 0x64b84e1d  // fmlalt z29.s, z16.h, z0.h[7]\n"
      ".inst 0x64b84e3e  // fmlalt z30.s, z17.h, z0.h[7]\n"
      ".inst 0x64b84e5f  // fmlalt z31.s, z18.h, z0.h[7]\n"
      "55:"  // Width 8: Multiply loop: multiply skip
      "fcvt z24.h, p2/m, z24.s\n"
      "fcvt z25.h, p2/m, z25.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "fcvt z25.h, p2/m, z26.s\n"
      "fcvt z26.h, p2/m, z27.s\n"
      "fcvt z27.h, p2/m, z29.s\n"
      "uzp1 z25.h, z25.h, z26.h\n"
      "fcvt z26.h, p2/m, z28.s\n"
      "fcvt z28.h, p2/m, z31.s\n"
      "uzp1 z26.h, z26.h, z27.h\n"
      "fcvt z27.h, p2/m, z30.s\n"
      "uzp1 z27.h, z27.h, z28.h\n"
      "tbz %x[flags], #1, 56f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmin z26.h, p2/M, z26.h, z17.h\n"
      "fmin z27.h, p2/M, z27.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "fmax z26.h, p2/M, z26.h, z16.h\n"
      "fmax z27.h, p2/M, z27.h, z16.h\n"
      "56:"  // Width 8: No activation
      "subs x27, x27, #0x8\n"
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "mov %x[B_ptr], x22\n"
      "st1h { z25.h }, p2, [%x[output_ptr], #1, MUL VL]\n"
      "sub %x[N], %x[N], x28, LSL #3\n"
      "st1h { z26.h }, p2, [%x[output_ptr], #2, MUL VL]\n"
      "st1h { z27.h }, p1, [%x[output_ptr], #3, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #4\n"
      "bgt 1b\n"
      "57:"  // Exit
      ".inst 0xd503467f  // SMSTOP\n"
      : [B_ptr] "+&r" (B_ptr), [N] "+&r" (N), [output_ptr] "+&r" (output_ptr)
      : [A_ptr] "r" (A_ptr), [K] "r" (K), [args_ptr] "r" (&args), [bias] "r" (bias), [flags] "r" (flags), [offsetof_maxval] "I" (offsetof(KernelArgs, maxval)), [offsetof_minval] "I" (offsetof(KernelArgs, minval))
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x9", "x10", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm

#endif // (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(ARM_COMPUTE_ENABLE_SME) && defined(__aarch64__)

