/*
 * Copyright (c) 2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(ARM_COMPUTE_ENABLE_SME) && defined(__aarch64__)

#include "arm_gemm/arm_gemm.hpp"
#include "arm_common/internal/utils.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void sme_gemv_fp16_mla_8VL (
    const __fp16 *A_ptr, const __fp16 *B_ptr, __fp16 *output_ptr,
    size_t N, size_t K,
    const __fp16 *bias, Activation act, bool
)
{
    struct KernelArgs {
        __fp16 maxval = static_cast<__fp16>(std::numeric_limits<float>::infinity());
        __fp16 minval = - static_cast<__fp16>(std::numeric_limits<float>::infinity());
        const __fp16 *B_ptr = {};
        size_t output_offset = {};
        unsigned int input_initial_col = {};
    } args;

    unsigned long flags=0;
    args.B_ptr = B_ptr;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            args.maxval = static_cast<__fp16>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            args.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      ".inst 0xd503477f  // SMSTART ZA\n"
      "mov x25, %x[bias]\n"
      "cnth x24\n"
      "ptrue p2.b\n"
      "add x23, %x[N], x24\n"
      "sub x23, x23, #0x1\n"
      "udiv x23, x23, x24\n"
      "1:"  // Column loop
      "cmp x23, #0x8\n"
      "bge 50f\n"
      "cmp x23, #0x6\n"
      "bgt 43f\n"
      "beq 36f\n"
      "cmp x23, #0x4\n"
      "bgt 29f\n"
      "beq 22f\n"
      "cmp x23, #0x2\n"
      "bgt 15f\n"
      "beq 8f\n"
      "mov x22, %x[K]\n"
      "mov x21, %x[A_ptr]\n"
      "whilelt p1.h, XZR, %x[N]\n"
      "cbz x25, 2f\n"
      "ld1h { z24.h }, p2/Z, [x25]\n"
      "b 3f\n"
      "2:"  // Width 1: no bias
      "fmov z24.h, #0\n"
      "3:"  // Width 1: setup done
      "cmp x22, #0x8\n"
      "ble 5f\n"
      "4:"  // Width 1: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "sub x22, x22, #0x8\n"
      "add x21, x21, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "cmp x22, #0x8\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z1.h, z0.h[0]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z2.h, z0.h[1]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z3.h, z0.h[2]\n"
      "fmla z24.h, z4.h, z0.h[3]\n"
      "fmla z24.h, z5.h, z0.h[4]\n"
      "fmla z24.h, z6.h, z0.h[5]\n"
      "fmla z24.h, z7.h, z0.h[6]\n"
      "fmla z24.h, z8.h, z0.h[7]\n"
      "bgt 4b\n"
      "5:"  // Width 1: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z9.h, z0.h[0]\n"
      "ble 6f\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z10.h, z0.h[1]\n"
      "ble 6f\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z11.h, z0.h[2]\n"
      "ble 6f\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z12.h, z0.h[3]\n"
      "ble 6f\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z13.h, z0.h[4]\n"
      "ble 6f\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z14.h, z0.h[5]\n"
      "ble 6f\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z15.h, z0.h[6]\n"
      "ble 6f\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr]]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z16.h, z0.h[7]\n"
      "6:"  // Width 1: Multiply loop: multiply skip
      "tbz %x[flags], #1, 7f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "7:"  // Width 1: No activation
      "st1h { z24.h }, p1, [%x[output_ptr]]\n"
      "addvl %x[output_ptr], %x[output_ptr], #1\n"
      "b 57f\n"
      "8:"  // Width 2
      "sub x20, %x[N], x24\n"
      "mov x22, %x[K]\n"
      "mov x21, %x[A_ptr]\n"
      "whilelt p1.h, XZR, x20\n"
      "cbz x25, 9f\n"
      "ld1h { z24.h }, p2/Z, [x25]\n"
      "ld1h { z25.h }, p2/Z, [x25, #1, MUL VL]\n"
      "b 10f\n"
      "9:"  // Width 2: no bias
      "fmov z24.h, #0\n"
      "fmov z25.h, #0\n"
      "10:"  // Width 2: setup done
      "cmp x22, #0x8\n"
      "ble 12f\n"
      "11:"  // Width 2: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "cmp x22, #0x8\n"
      "add x21, x21, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z24.h, z1.h, z0.h[0]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z2.h, z0.h[0]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z24.h, z3.h, z0.h[1]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z4.h, z0.h[1]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z24.h, z5.h, z0.h[2]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z6.h, z0.h[2]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z24.h, z7.h, z0.h[3]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z8.h, z0.h[3]\n"
      "fmla z24.h, z9.h, z0.h[4]\n"
      "fmla z25.h, z10.h, z0.h[4]\n"
      "fmla z24.h, z11.h, z0.h[5]\n"
      "fmla z25.h, z12.h, z0.h[5]\n"
      "fmla z24.h, z13.h, z0.h[6]\n"
      "fmla z25.h, z14.h, z0.h[6]\n"
      "fmla z24.h, z15.h, z0.h[7]\n"
      "fmla z25.h, z16.h, z0.h[7]\n"
      "bgt 11b\n"
      "12:"  // Width 2: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z17.h, z0.h[0]\n"
      "fmla z25.h, z18.h, z0.h[0]\n"
      "ble 13f\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z19.h, z0.h[1]\n"
      "fmla z25.h, z20.h, z0.h[1]\n"
      "ble 13f\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z21.h, z0.h[2]\n"
      "fmla z25.h, z22.h, z0.h[2]\n"
      "ble 13f\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z23.h, z0.h[3]\n"
      "fmla z25.h, z1.h, z0.h[3]\n"
      "ble 13f\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z2.h, z0.h[4]\n"
      "fmla z25.h, z3.h, z0.h[4]\n"
      "ble 13f\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z4.h, z0.h[5]\n"
      "fmla z25.h, z5.h, z0.h[5]\n"
      "ble 13f\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z6.h, z0.h[6]\n"
      "fmla z25.h, z7.h, z0.h[6]\n"
      "ble 13f\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z8.h, z0.h[7]\n"
      "fmla z25.h, z9.h, z0.h[7]\n"
      "13:"  // Width 2: Multiply loop: multiply skip
      "tbz %x[flags], #1, 14f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "14:"  // Width 2: No activation
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "st1h { z25.h }, p1, [%x[output_ptr], #1, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #2\n"
      "b 57f\n"
      "15:"  // Width 3
      "mov x20, #0x2\n"
      "mov x22, %x[K]\n"
      "msub x20, x24, x20, %x[N]\n"
      "mov x21, %x[A_ptr]\n"
      "whilelt p1.h, XZR, x20\n"
      "cbz x25, 16f\n"
      "ld1h { z24.h }, p2/Z, [x25]\n"
      "ld1h { z25.h }, p2/Z, [x25, #1, MUL VL]\n"
      "ld1h { z26.h }, p2/Z, [x25, #2, MUL VL]\n"
      "b 17f\n"
      "16:"  // Width 3: no bias
      "fmov z24.h, #0\n"
      "fmov z25.h, #0\n"
      "fmov z26.h, #0\n"
      "17:"  // Width 3: setup done
      "cmp x22, #0x8\n"
      "ble 19f\n"
      "18:"  // Width 3: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "cmp x22, #0x8\n"
      "add x21, x21, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z1.h, z0.h[0]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z25.h, z2.h, z0.h[0]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z26.h, z3.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z24.h, z4.h, z0.h[1]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z25.h, z5.h, z0.h[1]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z6.h, z0.h[1]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z24.h, z7.h, z0.h[2]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z8.h, z0.h[2]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z26.h, z9.h, z0.h[2]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z10.h, z0.h[3]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z25.h, z11.h, z0.h[3]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z26.h, z12.h, z0.h[3]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z24.h, z13.h, z0.h[4]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z25.h, z14.h, z0.h[4]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z15.h, z0.h[4]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z24.h, z16.h, z0.h[5]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z17.h, z0.h[5]\n"
      "fmla z26.h, z18.h, z0.h[5]\n"
      "fmla z24.h, z19.h, z0.h[6]\n"
      "fmla z25.h, z20.h, z0.h[6]\n"
      "fmla z26.h, z21.h, z0.h[6]\n"
      "fmla z24.h, z22.h, z0.h[7]\n"
      "fmla z25.h, z23.h, z0.h[7]\n"
      "fmla z26.h, z1.h, z0.h[7]\n"
      "bgt 18b\n"
      "19:"  // Width 3: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z2.h, z0.h[0]\n"
      "fmla z25.h, z3.h, z0.h[0]\n"
      "fmla z26.h, z4.h, z0.h[0]\n"
      "ble 20f\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z5.h, z0.h[1]\n"
      "fmla z25.h, z6.h, z0.h[1]\n"
      "fmla z26.h, z7.h, z0.h[1]\n"
      "ble 20f\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z8.h, z0.h[2]\n"
      "fmla z25.h, z9.h, z0.h[2]\n"
      "fmla z26.h, z10.h, z0.h[2]\n"
      "ble 20f\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z11.h, z0.h[3]\n"
      "fmla z25.h, z12.h, z0.h[3]\n"
      "fmla z26.h, z13.h, z0.h[3]\n"
      "ble 20f\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z14.h, z0.h[4]\n"
      "fmla z25.h, z15.h, z0.h[4]\n"
      "fmla z26.h, z16.h, z0.h[4]\n"
      "ble 20f\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z17.h, z0.h[5]\n"
      "fmla z25.h, z18.h, z0.h[5]\n"
      "fmla z26.h, z19.h, z0.h[5]\n"
      "ble 20f\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z20.h, z0.h[6]\n"
      "fmla z25.h, z21.h, z0.h[6]\n"
      "fmla z26.h, z22.h, z0.h[6]\n"
      "ble 20f\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z24.h, z23.h, z0.h[7]\n"
      "fmla z25.h, z1.h, z0.h[7]\n"
      "fmla z26.h, z2.h, z0.h[7]\n"
      "20:"  // Width 3: Multiply loop: multiply skip
      "tbz %x[flags], #1, 21f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmin z26.h, p2/M, z26.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "fmax z26.h, p2/M, z26.h, z16.h\n"
      "21:"  // Width 3: No activation
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "st1h { z25.h }, p2, [%x[output_ptr], #1, MUL VL]\n"
      "st1h { z26.h }, p1, [%x[output_ptr], #2, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #3\n"
      "b 57f\n"
      "22:"  // Width 4
      "mov x20, #0x3\n"
      "mov x22, %x[K]\n"
      "msub x20, x24, x20, %x[N]\n"
      "mov x21, %x[A_ptr]\n"
      "whilelt p1.h, XZR, x20\n"
      "cbz x25, 23f\n"
      "ld1h { z24.h }, p2/Z, [x25]\n"
      "ld1h { z25.h }, p2/Z, [x25, #1, MUL VL]\n"
      "ld1h { z26.h }, p2/Z, [x25, #2, MUL VL]\n"
      "ld1h { z27.h }, p2/Z, [x25, #3, MUL VL]\n"
      "b 24f\n"
      "23:"  // Width 4: no bias
      "fmov z24.h, #0\n"
      "fmov z25.h, #0\n"
      "fmov z26.h, #0\n"
      "fmov z27.h, #0\n"
      "24:"  // Width 4: setup done
      "cmp x22, #0x8\n"
      "ble 26f\n"
      "25:"  // Width 4: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "cmp x22, #0x8\n"
      "add x21, x21, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z1.h, z0.h[0]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z2.h, z0.h[0]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z26.h, z3.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z27.h, z4.h, z0.h[0]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z5.h, z0.h[1]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z6.h, z0.h[1]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z26.h, z7.h, z0.h[1]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z27.h, z8.h, z0.h[1]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z9.h, z0.h[2]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z10.h, z0.h[2]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z26.h, z11.h, z0.h[2]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z27.h, z12.h, z0.h[2]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z13.h, z0.h[3]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z14.h, z0.h[3]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z26.h, z15.h, z0.h[3]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z27.h, z16.h, z0.h[3]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z17.h, z0.h[4]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z18.h, z0.h[4]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z26.h, z19.h, z0.h[4]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z27.h, z20.h, z0.h[4]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z21.h, z0.h[5]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z22.h, z0.h[5]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z26.h, z23.h, z0.h[5]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z27.h, z1.h, z0.h[5]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z2.h, z0.h[6]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z3.h, z0.h[6]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z26.h, z4.h, z0.h[6]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z27.h, z5.h, z0.h[6]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z6.h, z0.h[7]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z7.h, z0.h[7]\n"
      "fmla z26.h, z8.h, z0.h[7]\n"
      "fmla z27.h, z9.h, z0.h[7]\n"
      "bgt 25b\n"
      "26:"  // Width 4: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z10.h, z0.h[0]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z11.h, z0.h[0]\n"
      "fmla z26.h, z12.h, z0.h[0]\n"
      "fmla z27.h, z13.h, z0.h[0]\n"
      "ble 27f\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z14.h, z0.h[1]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z15.h, z0.h[1]\n"
      "fmla z26.h, z16.h, z0.h[1]\n"
      "fmla z27.h, z17.h, z0.h[1]\n"
      "ble 27f\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z18.h, z0.h[2]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z19.h, z0.h[2]\n"
      "fmla z26.h, z20.h, z0.h[2]\n"
      "fmla z27.h, z21.h, z0.h[2]\n"
      "ble 27f\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z22.h, z0.h[3]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z23.h, z0.h[3]\n"
      "fmla z26.h, z1.h, z0.h[3]\n"
      "fmla z27.h, z2.h, z0.h[3]\n"
      "ble 27f\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z3.h, z0.h[4]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z4.h, z0.h[4]\n"
      "fmla z26.h, z5.h, z0.h[4]\n"
      "fmla z27.h, z6.h, z0.h[4]\n"
      "ble 27f\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z7.h, z0.h[5]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z8.h, z0.h[5]\n"
      "fmla z26.h, z9.h, z0.h[5]\n"
      "fmla z27.h, z10.h, z0.h[5]\n"
      "ble 27f\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z11.h, z0.h[6]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z12.h, z0.h[6]\n"
      "fmla z26.h, z13.h, z0.h[6]\n"
      "fmla z27.h, z14.h, z0.h[6]\n"
      "ble 27f\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z15.h, z0.h[7]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z25.h, z16.h, z0.h[7]\n"
      "fmla z26.h, z17.h, z0.h[7]\n"
      "fmla z27.h, z18.h, z0.h[7]\n"
      "27:"  // Width 4: Multiply loop: multiply skip
      "tbz %x[flags], #1, 28f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmin z26.h, p2/M, z26.h, z17.h\n"
      "fmin z27.h, p2/M, z27.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "fmax z26.h, p2/M, z26.h, z16.h\n"
      "fmax z27.h, p2/M, z27.h, z16.h\n"
      "28:"  // Width 4: No activation
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "st1h { z25.h }, p2, [%x[output_ptr], #1, MUL VL]\n"
      "st1h { z26.h }, p2, [%x[output_ptr], #2, MUL VL]\n"
      "st1h { z27.h }, p1, [%x[output_ptr], #3, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #4\n"
      "b 57f\n"
      "29:"  // Width 5
      "mov x20, #0x4\n"
      "mov x22, %x[K]\n"
      "msub x20, x24, x20, %x[N]\n"
      "mov x21, %x[A_ptr]\n"
      "whilelt p1.h, XZR, x20\n"
      "cbz x25, 30f\n"
      "ld1h { z24.h }, p2/Z, [x25]\n"
      "ld1h { z25.h }, p2/Z, [x25, #1, MUL VL]\n"
      "ld1h { z26.h }, p2/Z, [x25, #2, MUL VL]\n"
      "ld1h { z27.h }, p2/Z, [x25, #3, MUL VL]\n"
      "ld1h { z28.h }, p2/Z, [x25, #4, MUL VL]\n"
      "b 31f\n"
      "30:"  // Width 5: no bias
      "fmov z24.h, #0\n"
      "fmov z25.h, #0\n"
      "fmov z26.h, #0\n"
      "fmov z27.h, #0\n"
      "fmov z28.h, #0\n"
      "31:"  // Width 5: setup done
      "cmp x22, #0x8\n"
      "ble 33f\n"
      "32:"  // Width 5: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "cmp x22, #0x8\n"
      "add x21, x21, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z1.h, z0.h[0]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z2.h, z0.h[0]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z3.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z27.h, z4.h, z0.h[0]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z28.h, z5.h, z0.h[0]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z6.h, z0.h[1]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z7.h, z0.h[1]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z8.h, z0.h[1]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z27.h, z9.h, z0.h[1]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z28.h, z10.h, z0.h[1]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z11.h, z0.h[2]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z12.h, z0.h[2]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z13.h, z0.h[2]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z27.h, z14.h, z0.h[2]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z28.h, z15.h, z0.h[2]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z16.h, z0.h[3]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z17.h, z0.h[3]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z18.h, z0.h[3]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z27.h, z19.h, z0.h[3]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z28.h, z20.h, z0.h[3]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z21.h, z0.h[4]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z22.h, z0.h[4]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z23.h, z0.h[4]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z27.h, z1.h, z0.h[4]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z28.h, z2.h, z0.h[4]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z3.h, z0.h[5]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z4.h, z0.h[5]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z5.h, z0.h[5]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z27.h, z6.h, z0.h[5]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z28.h, z7.h, z0.h[5]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z8.h, z0.h[6]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z9.h, z0.h[6]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z10.h, z0.h[6]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z27.h, z11.h, z0.h[6]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z28.h, z12.h, z0.h[6]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z13.h, z0.h[7]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z14.h, z0.h[7]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z15.h, z0.h[7]\n"
      "fmla z27.h, z16.h, z0.h[7]\n"
      "fmla z28.h, z17.h, z0.h[7]\n"
      "bgt 32b\n"
      "33:"  // Width 5: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z18.h, z0.h[0]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z19.h, z0.h[0]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z20.h, z0.h[0]\n"
      "fmla z27.h, z21.h, z0.h[0]\n"
      "fmla z28.h, z22.h, z0.h[0]\n"
      "ble 34f\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z23.h, z0.h[1]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z1.h, z0.h[1]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z2.h, z0.h[1]\n"
      "fmla z27.h, z3.h, z0.h[1]\n"
      "fmla z28.h, z4.h, z0.h[1]\n"
      "ble 34f\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z5.h, z0.h[2]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z6.h, z0.h[2]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z7.h, z0.h[2]\n"
      "fmla z27.h, z8.h, z0.h[2]\n"
      "fmla z28.h, z9.h, z0.h[2]\n"
      "ble 34f\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z10.h, z0.h[3]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z11.h, z0.h[3]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z12.h, z0.h[3]\n"
      "fmla z27.h, z13.h, z0.h[3]\n"
      "fmla z28.h, z14.h, z0.h[3]\n"
      "ble 34f\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z15.h, z0.h[4]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z16.h, z0.h[4]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z17.h, z0.h[4]\n"
      "fmla z27.h, z18.h, z0.h[4]\n"
      "fmla z28.h, z19.h, z0.h[4]\n"
      "ble 34f\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z20.h, z0.h[5]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z21.h, z0.h[5]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z22.h, z0.h[5]\n"
      "fmla z27.h, z23.h, z0.h[5]\n"
      "fmla z28.h, z1.h, z0.h[5]\n"
      "ble 34f\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z2.h, z0.h[6]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z3.h, z0.h[6]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z4.h, z0.h[6]\n"
      "fmla z27.h, z5.h, z0.h[6]\n"
      "fmla z28.h, z6.h, z0.h[6]\n"
      "ble 34f\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z7.h, z0.h[7]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z8.h, z0.h[7]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z26.h, z9.h, z0.h[7]\n"
      "fmla z27.h, z10.h, z0.h[7]\n"
      "fmla z28.h, z11.h, z0.h[7]\n"
      "34:"  // Width 5: Multiply loop: multiply skip
      "tbz %x[flags], #1, 35f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmin z26.h, p2/M, z26.h, z17.h\n"
      "fmin z27.h, p2/M, z27.h, z17.h\n"
      "fmin z28.h, p2/M, z28.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "fmax z26.h, p2/M, z26.h, z16.h\n"
      "fmax z27.h, p2/M, z27.h, z16.h\n"
      "fmax z28.h, p2/M, z28.h, z16.h\n"
      "35:"  // Width 5: No activation
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "st1h { z25.h }, p2, [%x[output_ptr], #1, MUL VL]\n"
      "st1h { z26.h }, p2, [%x[output_ptr], #2, MUL VL]\n"
      "st1h { z27.h }, p2, [%x[output_ptr], #3, MUL VL]\n"
      "st1h { z28.h }, p1, [%x[output_ptr], #4, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #5\n"
      "b 57f\n"
      "36:"  // Width 6
      "mov x20, #0x5\n"
      "mov x22, %x[K]\n"
      "msub x20, x24, x20, %x[N]\n"
      "mov x21, %x[A_ptr]\n"
      "whilelt p1.h, XZR, x20\n"
      "cbz x25, 37f\n"
      "ld1h { z24.h }, p2/Z, [x25]\n"
      "ld1h { z25.h }, p2/Z, [x25, #1, MUL VL]\n"
      "ld1h { z26.h }, p2/Z, [x25, #2, MUL VL]\n"
      "ld1h { z27.h }, p2/Z, [x25, #3, MUL VL]\n"
      "ld1h { z28.h }, p2/Z, [x25, #4, MUL VL]\n"
      "ld1h { z29.h }, p2/Z, [x25, #5, MUL VL]\n"
      "b 38f\n"
      "37:"  // Width 6: no bias
      "fmov z24.h, #0\n"
      "fmov z25.h, #0\n"
      "fmov z26.h, #0\n"
      "fmov z27.h, #0\n"
      "fmov z28.h, #0\n"
      "fmov z29.h, #0\n"
      "38:"  // Width 6: setup done
      "cmp x22, #0x8\n"
      "ble 40f\n"
      "39:"  // Width 6: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "cmp x22, #0x8\n"
      "add x21, x21, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z1.h, z0.h[0]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z2.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z3.h, z0.h[0]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z4.h, z0.h[0]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z28.h, z5.h, z0.h[0]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z29.h, z6.h, z0.h[0]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z7.h, z0.h[1]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z8.h, z0.h[1]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z9.h, z0.h[1]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z10.h, z0.h[1]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z28.h, z11.h, z0.h[1]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z29.h, z12.h, z0.h[1]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z13.h, z0.h[2]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z14.h, z0.h[2]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z15.h, z0.h[2]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z16.h, z0.h[2]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z28.h, z17.h, z0.h[2]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z29.h, z18.h, z0.h[2]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z19.h, z0.h[3]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z20.h, z0.h[3]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z21.h, z0.h[3]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z22.h, z0.h[3]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z28.h, z23.h, z0.h[3]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z29.h, z1.h, z0.h[3]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z2.h, z0.h[4]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z3.h, z0.h[4]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z4.h, z0.h[4]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z5.h, z0.h[4]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z28.h, z6.h, z0.h[4]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z29.h, z7.h, z0.h[4]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z8.h, z0.h[5]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z9.h, z0.h[5]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z10.h, z0.h[5]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z11.h, z0.h[5]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z28.h, z12.h, z0.h[5]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z29.h, z13.h, z0.h[5]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z14.h, z0.h[6]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z15.h, z0.h[6]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z16.h, z0.h[6]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z17.h, z0.h[6]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z28.h, z18.h, z0.h[6]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z29.h, z19.h, z0.h[6]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z20.h, z0.h[7]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z21.h, z0.h[7]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z22.h, z0.h[7]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z23.h, z0.h[7]\n"
      "fmla z28.h, z1.h, z0.h[7]\n"
      "fmla z29.h, z2.h, z0.h[7]\n"
      "bgt 39b\n"
      "40:"  // Width 6: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z3.h, z0.h[0]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z4.h, z0.h[0]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z5.h, z0.h[0]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z6.h, z0.h[0]\n"
      "fmla z28.h, z7.h, z0.h[0]\n"
      "fmla z29.h, z8.h, z0.h[0]\n"
      "ble 41f\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z9.h, z0.h[1]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z10.h, z0.h[1]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z11.h, z0.h[1]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z12.h, z0.h[1]\n"
      "fmla z28.h, z13.h, z0.h[1]\n"
      "fmla z29.h, z14.h, z0.h[1]\n"
      "ble 41f\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z15.h, z0.h[2]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z16.h, z0.h[2]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z17.h, z0.h[2]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z18.h, z0.h[2]\n"
      "fmla z28.h, z19.h, z0.h[2]\n"
      "fmla z29.h, z20.h, z0.h[2]\n"
      "ble 41f\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z21.h, z0.h[3]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z22.h, z0.h[3]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z23.h, z0.h[3]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z1.h, z0.h[3]\n"
      "fmla z28.h, z2.h, z0.h[3]\n"
      "fmla z29.h, z3.h, z0.h[3]\n"
      "ble 41f\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z4.h, z0.h[4]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z5.h, z0.h[4]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z6.h, z0.h[4]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z7.h, z0.h[4]\n"
      "fmla z28.h, z8.h, z0.h[4]\n"
      "fmla z29.h, z9.h, z0.h[4]\n"
      "ble 41f\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z10.h, z0.h[5]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z11.h, z0.h[5]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z12.h, z0.h[5]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z13.h, z0.h[5]\n"
      "fmla z28.h, z14.h, z0.h[5]\n"
      "fmla z29.h, z15.h, z0.h[5]\n"
      "ble 41f\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z16.h, z0.h[6]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z17.h, z0.h[6]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z18.h, z0.h[6]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z19.h, z0.h[6]\n"
      "fmla z28.h, z20.h, z0.h[6]\n"
      "fmla z29.h, z21.h, z0.h[6]\n"
      "ble 41f\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z22.h, z0.h[7]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z23.h, z0.h[7]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z1.h, z0.h[7]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z27.h, z2.h, z0.h[7]\n"
      "fmla z28.h, z3.h, z0.h[7]\n"
      "fmla z29.h, z4.h, z0.h[7]\n"
      "41:"  // Width 6: Multiply loop: multiply skip
      "tbz %x[flags], #1, 42f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmin z26.h, p2/M, z26.h, z17.h\n"
      "fmin z27.h, p2/M, z27.h, z17.h\n"
      "fmin z28.h, p2/M, z28.h, z17.h\n"
      "fmin z29.h, p2/M, z29.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "fmax z26.h, p2/M, z26.h, z16.h\n"
      "fmax z27.h, p2/M, z27.h, z16.h\n"
      "fmax z28.h, p2/M, z28.h, z16.h\n"
      "fmax z29.h, p2/M, z29.h, z16.h\n"
      "42:"  // Width 6: No activation
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "st1h { z25.h }, p2, [%x[output_ptr], #1, MUL VL]\n"
      "st1h { z26.h }, p2, [%x[output_ptr], #2, MUL VL]\n"
      "st1h { z27.h }, p2, [%x[output_ptr], #3, MUL VL]\n"
      "st1h { z28.h }, p2, [%x[output_ptr], #4, MUL VL]\n"
      "st1h { z29.h }, p1, [%x[output_ptr], #5, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #6\n"
      "b 57f\n"
      "43:"  // Width 7
      "mov x20, #0x6\n"
      "mov x22, %x[K]\n"
      "msub x20, x24, x20, %x[N]\n"
      "mov x21, %x[A_ptr]\n"
      "whilelt p1.h, XZR, x20\n"
      "cbz x25, 44f\n"
      "ld1h { z24.h }, p2/Z, [x25]\n"
      "ld1h { z25.h }, p2/Z, [x25, #1, MUL VL]\n"
      "ld1h { z26.h }, p2/Z, [x25, #2, MUL VL]\n"
      "ld1h { z27.h }, p2/Z, [x25, #3, MUL VL]\n"
      "ld1h { z28.h }, p2/Z, [x25, #4, MUL VL]\n"
      "ld1h { z29.h }, p2/Z, [x25, #5, MUL VL]\n"
      "ld1h { z30.h }, p2/Z, [x25, #6, MUL VL]\n"
      "b 45f\n"
      "44:"  // Width 7: no bias
      "fmov z24.h, #0\n"
      "fmov z25.h, #0\n"
      "fmov z26.h, #0\n"
      "fmov z27.h, #0\n"
      "fmov z28.h, #0\n"
      "fmov z29.h, #0\n"
      "fmov z30.h, #0\n"
      "45:"  // Width 7: setup done
      "cmp x22, #0x8\n"
      "ble 47f\n"
      "46:"  // Width 7: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "cmp x22, #0x8\n"
      "add x21, x21, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z1.h, z0.h[0]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z2.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z3.h, z0.h[0]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z4.h, z0.h[0]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z5.h, z0.h[0]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z29.h, z6.h, z0.h[0]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z30.h, z7.h, z0.h[0]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z8.h, z0.h[1]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z9.h, z0.h[1]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z10.h, z0.h[1]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z11.h, z0.h[1]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z12.h, z0.h[1]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z29.h, z13.h, z0.h[1]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z30.h, z14.h, z0.h[1]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z15.h, z0.h[2]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z16.h, z0.h[2]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z17.h, z0.h[2]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z18.h, z0.h[2]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z19.h, z0.h[2]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z29.h, z20.h, z0.h[2]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z30.h, z21.h, z0.h[2]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z22.h, z0.h[3]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z23.h, z0.h[3]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z1.h, z0.h[3]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z2.h, z0.h[3]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z3.h, z0.h[3]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z29.h, z4.h, z0.h[3]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z30.h, z5.h, z0.h[3]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z6.h, z0.h[4]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z7.h, z0.h[4]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z8.h, z0.h[4]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z9.h, z0.h[4]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z10.h, z0.h[4]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z29.h, z11.h, z0.h[4]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z30.h, z12.h, z0.h[4]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z13.h, z0.h[5]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z14.h, z0.h[5]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z15.h, z0.h[5]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z16.h, z0.h[5]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z17.h, z0.h[5]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z29.h, z18.h, z0.h[5]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z30.h, z19.h, z0.h[5]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z20.h, z0.h[6]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z21.h, z0.h[6]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z22.h, z0.h[6]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z23.h, z0.h[6]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z1.h, z0.h[6]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z29.h, z2.h, z0.h[6]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z30.h, z3.h, z0.h[6]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z4.h, z0.h[7]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z5.h, z0.h[7]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z6.h, z0.h[7]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z7.h, z0.h[7]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z8.h, z0.h[7]\n"
      "fmla z29.h, z9.h, z0.h[7]\n"
      "fmla z30.h, z10.h, z0.h[7]\n"
      "bgt 46b\n"
      "47:"  // Width 7: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z11.h, z0.h[0]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z12.h, z0.h[0]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z13.h, z0.h[0]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z14.h, z0.h[0]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z15.h, z0.h[0]\n"
      "fmla z29.h, z16.h, z0.h[0]\n"
      "fmla z30.h, z17.h, z0.h[0]\n"
      "ble 48f\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z18.h, z0.h[1]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z19.h, z0.h[1]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z20.h, z0.h[1]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z21.h, z0.h[1]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z22.h, z0.h[1]\n"
      "fmla z29.h, z23.h, z0.h[1]\n"
      "fmla z30.h, z1.h, z0.h[1]\n"
      "ble 48f\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z2.h, z0.h[2]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z3.h, z0.h[2]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z4.h, z0.h[2]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z5.h, z0.h[2]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z6.h, z0.h[2]\n"
      "fmla z29.h, z7.h, z0.h[2]\n"
      "fmla z30.h, z8.h, z0.h[2]\n"
      "ble 48f\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z9.h, z0.h[3]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z10.h, z0.h[3]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z11.h, z0.h[3]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z12.h, z0.h[3]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z13.h, z0.h[3]\n"
      "fmla z29.h, z14.h, z0.h[3]\n"
      "fmla z30.h, z15.h, z0.h[3]\n"
      "ble 48f\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z16.h, z0.h[4]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z17.h, z0.h[4]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z18.h, z0.h[4]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z19.h, z0.h[4]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z20.h, z0.h[4]\n"
      "fmla z29.h, z21.h, z0.h[4]\n"
      "fmla z30.h, z22.h, z0.h[4]\n"
      "ble 48f\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z23.h, z0.h[5]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z1.h, z0.h[5]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z2.h, z0.h[5]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z3.h, z0.h[5]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z4.h, z0.h[5]\n"
      "fmla z29.h, z5.h, z0.h[5]\n"
      "fmla z30.h, z6.h, z0.h[5]\n"
      "ble 48f\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z7.h, z0.h[6]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z8.h, z0.h[6]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z9.h, z0.h[6]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z10.h, z0.h[6]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z11.h, z0.h[6]\n"
      "fmla z29.h, z12.h, z0.h[6]\n"
      "fmla z30.h, z13.h, z0.h[6]\n"
      "ble 48f\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z14.h, z0.h[7]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z15.h, z0.h[7]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z16.h, z0.h[7]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z17.h, z0.h[7]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z28.h, z18.h, z0.h[7]\n"
      "fmla z29.h, z19.h, z0.h[7]\n"
      "fmla z30.h, z20.h, z0.h[7]\n"
      "48:"  // Width 7: Multiply loop: multiply skip
      "tbz %x[flags], #1, 49f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmin z26.h, p2/M, z26.h, z17.h\n"
      "fmin z27.h, p2/M, z27.h, z17.h\n"
      "fmin z28.h, p2/M, z28.h, z17.h\n"
      "fmin z29.h, p2/M, z29.h, z17.h\n"
      "fmin z30.h, p2/M, z30.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "fmax z26.h, p2/M, z26.h, z16.h\n"
      "fmax z27.h, p2/M, z27.h, z16.h\n"
      "fmax z28.h, p2/M, z28.h, z16.h\n"
      "fmax z29.h, p2/M, z29.h, z16.h\n"
      "fmax z30.h, p2/M, z30.h, z16.h\n"
      "49:"  // Width 7: No activation
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "st1h { z25.h }, p2, [%x[output_ptr], #1, MUL VL]\n"
      "st1h { z26.h }, p2, [%x[output_ptr], #2, MUL VL]\n"
      "st1h { z27.h }, p2, [%x[output_ptr], #3, MUL VL]\n"
      "st1h { z28.h }, p2, [%x[output_ptr], #4, MUL VL]\n"
      "st1h { z29.h }, p2, [%x[output_ptr], #5, MUL VL]\n"
      "st1h { z30.h }, p1, [%x[output_ptr], #6, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #7\n"
      "b 57f\n"
      "50:"  // Width 8
      "mov x20, #0x7\n"
      "mov x22, %x[K]\n"
      "msub x20, x24, x20, %x[N]\n"
      "mov x21, %x[A_ptr]\n"
      "whilelt p1.h, XZR, x20\n"
      "cbz x25, 51f\n"
      "ld1h { z24.h }, p2/Z, [x25]\n"
      "ld1h { z25.h }, p2/Z, [x25, #1, MUL VL]\n"
      "ld1h { z26.h }, p2/Z, [x25, #2, MUL VL]\n"
      "ld1h { z27.h }, p2/Z, [x25, #3, MUL VL]\n"
      "ld1h { z28.h }, p2/Z, [x25, #4, MUL VL]\n"
      "ld1h { z29.h }, p2/Z, [x25, #5, MUL VL]\n"
      "ld1h { z30.h }, p2/Z, [x25, #6, MUL VL]\n"
      "ld1h { z31.h }, p2/Z, [x25, #7, MUL VL]\n"
      "incb x25, ALL, MUL #8\n"
      "b 52f\n"
      "51:"  // Width 8: no bias
      "fmov z24.h, #0\n"
      "fmov z25.h, #0\n"
      "fmov z26.h, #0\n"
      "fmov z27.h, #0\n"
      "fmov z28.h, #0\n"
      "fmov z29.h, #0\n"
      "fmov z30.h, #0\n"
      "fmov z31.h, #0\n"
      "52:"  // Width 8: setup done
      "cmp x22, #0x8\n"
      "ble 54f\n"
      "53:"  // Width 8: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr]]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "cmp x22, #0x8\n"
      "add x21, x21, #0x10\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z1.h, z0.h[0]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z2.h, z0.h[0]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z3.h, z0.h[0]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z4.h, z0.h[0]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z5.h, z0.h[0]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z6.h, z0.h[0]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z30.h, z7.h, z0.h[0]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z31.h, z8.h, z0.h[0]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z9.h, z0.h[1]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z10.h, z0.h[1]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z11.h, z0.h[1]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z12.h, z0.h[1]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z13.h, z0.h[1]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z14.h, z0.h[1]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z30.h, z15.h, z0.h[1]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z31.h, z16.h, z0.h[1]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z17.h, z0.h[2]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z18.h, z0.h[2]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z19.h, z0.h[2]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z20.h, z0.h[2]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z21.h, z0.h[2]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z22.h, z0.h[2]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z30.h, z23.h, z0.h[2]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z31.h, z1.h, z0.h[2]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z2.h, z0.h[3]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z3.h, z0.h[3]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z4.h, z0.h[3]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z5.h, z0.h[3]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z6.h, z0.h[3]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z7.h, z0.h[3]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z30.h, z8.h, z0.h[3]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z31.h, z9.h, z0.h[3]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z10.h, z0.h[4]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z11.h, z0.h[4]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z12.h, z0.h[4]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z13.h, z0.h[4]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z14.h, z0.h[4]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z15.h, z0.h[4]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z30.h, z16.h, z0.h[4]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z31.h, z17.h, z0.h[4]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z18.h, z0.h[5]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z19.h, z0.h[5]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z20.h, z0.h[5]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z21.h, z0.h[5]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z22.h, z0.h[5]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z23.h, z0.h[5]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z30.h, z1.h, z0.h[5]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z31.h, z2.h, z0.h[5]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z3.h, z0.h[6]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z4.h, z0.h[6]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z5.h, z0.h[6]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z6.h, z0.h[6]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z7.h, z0.h[6]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z8.h, z0.h[6]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr]]\n"
      "fmla z30.h, z9.h, z0.h[6]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "fmla z31.h, z10.h, z0.h[6]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z11.h, z0.h[7]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z12.h, z0.h[7]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z13.h, z0.h[7]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z14.h, z0.h[7]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z15.h, z0.h[7]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z16.h, z0.h[7]\n"
      "fmla z30.h, z17.h, z0.h[7]\n"
      "fmla z31.h, z18.h, z0.h[7]\n"
      "bgt 53b\n"
      "54:"  // Width 8: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x22\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ld1rqh { z0.h }, p0/Z, [x21]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z19.h, z0.h[0]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z20.h, z0.h[0]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z21.h, z0.h[0]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z22.h, z0.h[0]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z23.h, z0.h[0]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z1.h, z0.h[0]\n"
      "fmla z30.h, z2.h, z0.h[0]\n"
      "fmla z31.h, z3.h, z0.h[0]\n"
      "ble 55f\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z4.h, z0.h[1]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z5.h, z0.h[1]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z6.h, z0.h[1]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z7.h, z0.h[1]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z8.h, z0.h[1]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z9.h, z0.h[1]\n"
      "fmla z30.h, z10.h, z0.h[1]\n"
      "fmla z31.h, z11.h, z0.h[1]\n"
      "ble 55f\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z12.h, z0.h[2]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z13.h, z0.h[2]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z14.h, z0.h[2]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z15.h, z0.h[2]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z16.h, z0.h[2]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z17.h, z0.h[2]\n"
      "fmla z30.h, z18.h, z0.h[2]\n"
      "fmla z31.h, z19.h, z0.h[2]\n"
      "ble 55f\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z20.h, z0.h[3]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z21.h, z0.h[3]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z22.h, z0.h[3]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z23.h, z0.h[3]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z1.h, z0.h[3]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z2.h, z0.h[3]\n"
      "fmla z30.h, z3.h, z0.h[3]\n"
      "fmla z31.h, z4.h, z0.h[3]\n"
      "ble 55f\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z5.h, z0.h[4]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z6.h, z0.h[4]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z7.h, z0.h[4]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z8.h, z0.h[4]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z9.h, z0.h[4]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z10.h, z0.h[4]\n"
      "fmla z30.h, z11.h, z0.h[4]\n"
      "fmla z31.h, z12.h, z0.h[4]\n"
      "ble 55f\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z14.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z15.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z16.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z13.h, z0.h[5]\n"
      "ldnt1h { z17.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z14.h, z0.h[5]\n"
      "ldnt1h { z18.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z15.h, z0.h[5]\n"
      "ldnt1h { z19.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z16.h, z0.h[5]\n"
      "ldnt1h { z20.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z17.h, z0.h[5]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z18.h, z0.h[5]\n"
      "fmla z30.h, z19.h, z0.h[5]\n"
      "fmla z31.h, z20.h, z0.h[5]\n"
      "ble 55f\n"
      "ldnt1h { z21.h }, p2/Z, [%x[B_ptr]]\n"
      "subs x22, x22, #0x1\n"
      "ldnt1h { z22.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z23.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z1.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z21.h, z0.h[6]\n"
      "ldnt1h { z2.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z22.h, z0.h[6]\n"
      "ldnt1h { z3.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z23.h, z0.h[6]\n"
      "ldnt1h { z4.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z1.h, z0.h[6]\n"
      "ldnt1h { z5.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z2.h, z0.h[6]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z3.h, z0.h[6]\n"
      "fmla z30.h, z4.h, z0.h[6]\n"
      "fmla z31.h, z5.h, z0.h[6]\n"
      "ble 55f\n"
      "ldnt1h { z6.h }, p2/Z, [%x[B_ptr]]\n"
      "ldnt1h { z7.h }, p2/Z, [%x[B_ptr], #1, MUL VL]\n"
      "ldnt1h { z8.h }, p2/Z, [%x[B_ptr], #2, MUL VL]\n"
      "ldnt1h { z9.h }, p2/Z, [%x[B_ptr], #3, MUL VL]\n"
      "fmla z24.h, z6.h, z0.h[7]\n"
      "ldnt1h { z10.h }, p2/Z, [%x[B_ptr], #4, MUL VL]\n"
      "fmla z25.h, z7.h, z0.h[7]\n"
      "ldnt1h { z11.h }, p2/Z, [%x[B_ptr], #5, MUL VL]\n"
      "fmla z26.h, z8.h, z0.h[7]\n"
      "ldnt1h { z12.h }, p2/Z, [%x[B_ptr], #6, MUL VL]\n"
      "fmla z27.h, z9.h, z0.h[7]\n"
      "ldnt1h { z13.h }, p2/Z, [%x[B_ptr], #7, MUL VL]\n"
      "fmla z28.h, z10.h, z0.h[7]\n"
      "addvl %x[B_ptr], %x[B_ptr], #8\n"
      "fmla z29.h, z11.h, z0.h[7]\n"
      "fmla z30.h, z12.h, z0.h[7]\n"
      "fmla z31.h, z13.h, z0.h[7]\n"
      "55:"  // Width 8: Multiply loop: multiply skip
      "tbz %x[flags], #1, 56f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1rh { z17.h }, p2/Z, [x21]\n"
      "ld1rh { z16.h }, p2/Z, [x20]\n"
      "fmin z24.h, p2/M, z24.h, z17.h\n"
      "fmin z25.h, p2/M, z25.h, z17.h\n"
      "fmin z26.h, p2/M, z26.h, z17.h\n"
      "fmin z27.h, p2/M, z27.h, z17.h\n"
      "fmin z28.h, p2/M, z28.h, z17.h\n"
      "fmin z29.h, p2/M, z29.h, z17.h\n"
      "fmin z30.h, p2/M, z30.h, z17.h\n"
      "fmin z31.h, p2/M, z31.h, z17.h\n"
      "fmax z24.h, p2/M, z24.h, z16.h\n"
      "fmax z25.h, p2/M, z25.h, z16.h\n"
      "fmax z26.h, p2/M, z26.h, z16.h\n"
      "fmax z27.h, p2/M, z27.h, z16.h\n"
      "fmax z28.h, p2/M, z28.h, z16.h\n"
      "fmax z29.h, p2/M, z29.h, z16.h\n"
      "fmax z30.h, p2/M, z30.h, z16.h\n"
      "fmax z31.h, p2/M, z31.h, z16.h\n"
      "56:"  // Width 8: No activation
      "subs x23, x23, #0x8\n"
      "st1h { z24.h }, p2, [%x[output_ptr]]\n"
      "sub %x[N], %x[N], x24, LSL #3\n"
      "st1h { z25.h }, p2, [%x[output_ptr], #1, MUL VL]\n"
      "st1h { z26.h }, p2, [%x[output_ptr], #2, MUL VL]\n"
      "st1h { z27.h }, p2, [%x[output_ptr], #3, MUL VL]\n"
      "st1h { z28.h }, p2, [%x[output_ptr], #4, MUL VL]\n"
      "st1h { z29.h }, p2, [%x[output_ptr], #5, MUL VL]\n"
      "st1h { z30.h }, p2, [%x[output_ptr], #6, MUL VL]\n"
      "st1h { z31.h }, p1, [%x[output_ptr], #7, MUL VL]\n"
      "addvl %x[output_ptr], %x[output_ptr], #8\n"
      "bgt 1b\n"
      "57:"  // Exit
      ".inst 0xd503467f  // SMSTOP\n"
      : [B_ptr] "+&r" (B_ptr), [N] "+&r" (N), [output_ptr] "+&r" (output_ptr)
      : [A_ptr] "r" (A_ptr), [K] "r" (K), [args_ptr] "r" (&args), [bias] "r" (bias), [flags] "r" (flags), [offsetof_maxval] "I" (offsetof(KernelArgs, maxval)), [offsetof_minval] "I" (offsetof(KernelArgs, minval))
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x20", "x21", "x22", "x23", "x24", "x25", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm

#endif // (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(ARM_COMPUTE_ENABLE_SME) && defined(__aarch64__)

