/*
 * Copyright (c) 2021, 2023-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifdef __aarch64__

#include "arm_gemm/arm_gemm.hpp"
#include "arm_common/internal/utils.hpp"
#include <cassert>

namespace arm_gemm {

void a64_hybrid_u8u32_mmla_6x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<uint8_t> A_arg,
    size_t M, size_t N, const uint8_t *B_ptr, IndirectOutputArg<uint32_t> output_arg,
    const uint32_t *bias, Activation, bool accumulate
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const uint8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const uint32_t *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 191f\n"
      "cmp %x[M], #0x4\n"
      "bgt 153f\n"
      "beq 115f\n"
      "cmp %x[M], #0x2\n"
      "bgt 77f\n"
      "beq 39f\n"
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "cbz x10, 3f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "b 15f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 14f\n"
      "cmp x12, #0x10\n"
      "bge 12f\n"
      "tbz x12, #3, 7f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "tbz x12, #2, 5f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "tbz x12, #1, 4f\n"
      "ldr d16, [x9], #0x8\n"
      "mov x25, #0x38\n"
      "tbz x12, #0, 11f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "b 11f\n"
      "4:"  // Height 1: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x12, #0, 11f\n"
      "ldr s16, [x9, #0]\n"
      "b 11f\n"
      "5:"  // Height 1: Partial accumulate: partial_2_8
      "tbz x12, #1, 6f\n"
      "ldr d11, [x9], #0x8\n"
      "mov x25, #0x28\n"
      "tbz x12, #0, 11f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "b 11f\n"
      "6:"  // Height 1: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x12, #0, 11f\n"
      "ldr s11, [x9, #0]\n"
      "b 11f\n"
      "7:"  // Height 1: Partial accumulate: partial_4_0
      "tbz x12, #2, 9f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "tbz x12, #1, 8f\n"
      "ldr d10, [x9], #0x8\n"
      "mov x25, #0x18\n"
      "tbz x12, #0, 11f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "b 11f\n"
      "8:"  // Height 1: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x12, #0, 11f\n"
      "ldr s10, [x9, #0]\n"
      "b 11f\n"
      "9:"  // Height 1: Partial accumulate: partial_2_0
      "tbz x12, #1, 10f\n"
      "ldr d9, [x9], #0x8\n"
      "mov x25, #0x8\n"
      "tbz x12, #0, 11f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "b 11f\n"
      "10:"  // Height 1: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0]\n"
      "mov x25, #0\n"
      "11:"  // Height 1: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 13f\n"
      "12:"  // Height 1: full accumulate
      "ldr q9, [x9, #0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "13:"  // Height 1: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "b 15f\n"
      "14:"  // Height 1: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "15:"  // Height 1: setup done
      "mov x28, #0\n"
      "16:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 17f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "cbnz x28, 18f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "b 18f\n"
      "17:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "18:"  // Height 1: input setup done
      "cmp x27, #0x10\n"
      "blt 21f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q7, [x11, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q6, [x11, #0x10]\n"
      "blt 20f\n"
      "19:"  // Height 1: Multiply loop: Main loop head
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0]\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      "ldr q6, [x11, #0x10]\n"
      "bge 19b\n"
      "20:"  // Height 1: Multiply loop: Single iteration only
      "add x26, x26, #0x10\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      "21:"  // Height 1: Multiply loop: Main loop skip
      "cbz x27, 28f\n"
      "cmp x27, #0x8\n"
      "blt 23f\n"
      "22:"  // Height 1: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr q6, [x11, #0]\n"
      "sub x27, x27, #0x8\n"
      "ldr q7, [x11, #0x10]\n"
      "cmp x27, #0x8\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      ".inst 0x6e86a408  // ummla v8.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x20]\n"
      ".inst 0x6e87a40c  // ummla v12.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x6e86a409  // ummla v9.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x6e87a40d  // ummla v13.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x6e86a40a  // ummla v10.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x6e87a40e  // ummla v14.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e86a40b  // ummla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x6e87a40f  // ummla v15.4s, v0.16b, v7.16b\n"
      "bge 22b\n"
      "23:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x27, 28f\n"
      "tbz x27, #2, 25f\n"
      "ldr s1, [x26], #0x4\n"
      "tbz x27, #1, 24f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "tbz x27, #0, 27f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "b 27f\n"
      "24:"  // Height 1: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 27f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "b 27f\n"
      "25:"  // Height 1: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 26f\n"
      "ldr h1, [x26], #0x2\n"
      "tbz x27, #0, 27f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "b 27f\n"
      "26:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "27:"  // Height 1: Multiply loop: Ragged operand read: Done
      "ldr q7, [x11, #0]\n"
      "ldr q6, [x11, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      "28:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 16b\n"
      "cmp x12, #0x10\n"
      "uzp1 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v9.2d, v9.2d, v13.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "uzp1 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v11.2d, v11.2d, v15.2d\n"
      "bge 37f\n"
      "tbz x12, #3, 32f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "tbz x12, #2, 30f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "tbz x12, #1, 29f\n"
      "str d11, [x9], #0x8\n"
      "tbz x12, #0, 36f\n"
      "st1 { v11.s }[2], [x9]\n"
      "b 36f\n"
      "29:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x12, #0, 36f\n"
      "str s11, [x9, #0]\n"
      "b 36f\n"
      "30:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x12, #1, 31f\n"
      "str d10, [x9], #0x8\n"
      "tbz x12, #0, 36f\n"
      "st1 { v10.s }[2], [x9]\n"
      "b 36f\n"
      "31:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x12, #0, 36f\n"
      "str s10, [x9, #0]\n"
      "b 36f\n"
      "32:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x12, #2, 34f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "tbz x12, #1, 33f\n"
      "str d9, [x9], #0x8\n"
      "tbz x12, #0, 36f\n"
      "st1 { v9.s }[2], [x9]\n"
      "b 36f\n"
      "33:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x12, #0, 36f\n"
      "str s9, [x9, #0]\n"
      "b 36f\n"
      "34:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x12, #1, 35f\n"
      "str d8, [x9], #0x8\n"
      "tbz x12, #0, 36f\n"
      "st1 { v8.s }[2], [x9]\n"
      "b 36f\n"
      "35:"  // Height 1: Partial direct writeback: partial_1_0
      "str s8, [x9, #0]\n"
      "36:"  // Height 1: Partial direct writeback: Done
      "b 38f\n"
      "37:"  // Height 1: Full writeback
      "str q8, [x9, #0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "38:"  // Height 1: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 2b\n"
      "b 230f\n"
      "39:"  // Height 2
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "40:"  // Height 2: Column loop
      "cbz x10, 41f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "b 53f\n"
      "41:"  // Height 2: no bias
      "tbz %x[flags], #0, 52f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "add x24, x9, x20, LSL #2\n"
      "bge 50f\n"
      "tbz x12, #3, 45f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "tbz x12, #2, 43f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "tbz x12, #1, 42f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "tbz x12, #0, 49f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "b 49f\n"
      "42:"  // Height 2: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x12, #0, 49f\n"
      "ldr s16, [x9, #0]\n"
      "ldr s15, [x24, #0]\n"
      "b 49f\n"
      "43:"  // Height 2: Partial accumulate: partial_2_8
      "tbz x12, #1, 44f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "tbz x12, #0, 49f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "b 49f\n"
      "44:"  // Height 2: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x12, #0, 49f\n"
      "ldr s11, [x9, #0]\n"
      "ldr s14, [x24, #0]\n"
      "b 49f\n"
      "45:"  // Height 2: Partial accumulate: partial_4_0
      "tbz x12, #2, 47f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "tbz x12, #1, 46f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "tbz x12, #0, 49f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "b 49f\n"
      "46:"  // Height 2: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x12, #0, 49f\n"
      "ldr s10, [x9, #0]\n"
      "ldr s13, [x24, #0]\n"
      "b 49f\n"
      "47:"  // Height 2: Partial accumulate: partial_2_0
      "tbz x12, #1, 48f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "tbz x12, #0, 49f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "b 49f\n"
      "48:"  // Height 2: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0]\n"
      "ldr s12, [x24, #0]\n"
      "mov x25, #0\n"
      "49:"  // Height 2: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 51f\n"
      "50:"  // Height 2: full accumulate
      "ldr q9, [x9, #0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x24, #0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "51:"  // Height 2: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "b 53f\n"
      "52:"  // Height 2: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "53:"  // Height 2: setup done
      "mov x28, #0\n"
      "54:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 55f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 56f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "b 56f\n"
      "55:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "56:"  // Height 2: input setup done
      "cmp x27, #0x10\n"
      "blt 59f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q2, [x25, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q7, [x11, #0]\n"
      "ldr q6, [x11, #0x10]\n"
      "blt 58f\n"
      "57:"  // Height 2: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "ldr q2, [x25, #0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0]\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      "ldr q6, [x11, #0x10]\n"
      "bge 57b\n"
      "58:"  // Height 2: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      "59:"  // Height 2: Multiply loop: Main loop skip
      "cbz x27, 66f\n"
      "cmp x27, #0x8\n"
      "blt 61f\n"
      "60:"  // Height 2: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d2, [x25], #0x8\n"
      "sub x27, x27, #0x8\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "cmp x27, #0x8\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      ".inst 0x6e86a408  // ummla v8.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x20]\n"
      ".inst 0x6e87a40c  // ummla v12.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x6e86a409  // ummla v9.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x6e87a40d  // ummla v13.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x6e86a40a  // ummla v10.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x6e87a40e  // ummla v14.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e86a40b  // ummla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x6e87a40f  // ummla v15.4s, v0.16b, v7.16b\n"
      "bge 60b\n"
      "61:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x27, 66f\n"
      "tbz x27, #2, 63f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "tbz x27, #1, 62f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "tbz x27, #0, 65f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "b 65f\n"
      "62:"  // Height 2: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 65f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "b 65f\n"
      "63:"  // Height 2: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 64f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "tbz x27, #0, 65f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "b 65f\n"
      "64:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "ldr b2, [x25, #0]\n"
      "65:"  // Height 2: Multiply loop: Ragged operand read: Done
      "ldr q7, [x11, #0]\n"
      "ldr q6, [x11, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      "66:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 54b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x24, x9, x20, LSL #2\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x24, #0]\n"
      "bge 75f\n"
      "tbz x12, #3, 70f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v9.4s }, [x24], #0x10\n"
      "tbz x12, #2, 68f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x24], #0x10\n"
      "tbz x12, #1, 67f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x24], #0x8\n"
      "tbz x12, #0, 74f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x24]\n"
      "b 74f\n"
      "67:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x12, #0, 74f\n"
      "str s14, [x9, #0]\n"
      "str s11, [x24, #0]\n"
      "b 74f\n"
      "68:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x12, #1, 69f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x24], #0x8\n"
      "tbz x12, #0, 74f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x24]\n"
      "b 74f\n"
      "69:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x12, #0, 74f\n"
      "str s13, [x9, #0]\n"
      "str s10, [x24, #0]\n"
      "b 74f\n"
      "70:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x12, #2, 72f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "tbz x12, #1, 71f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x24], #0x8\n"
      "tbz x12, #0, 74f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x24]\n"
      "b 74f\n"
      "71:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x12, #0, 74f\n"
      "str s12, [x9, #0]\n"
      "str s9, [x24, #0]\n"
      "b 74f\n"
      "72:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x12, #1, 73f\n"
      "str d7, [x9], #0x8\n"
      "str d8, [x24], #0x8\n"
      "tbz x12, #0, 74f\n"
      "st1 { v7.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x24]\n"
      "b 74f\n"
      "73:"  // Height 2: Partial direct writeback: partial_1_0
      "str s7, [x9, #0]\n"
      "str s8, [x24, #0]\n"
      "74:"  // Height 2: Partial direct writeback: Done
      "b 76f\n"
      "75:"  // Height 2: Full writeback
      "str q7, [x9, #0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x24, #0]\n"
      "str q9, [x24, #0x10]\n"
      "str q10, [x24, #0x20]\n"
      "str q11, [x24, #0x30]\n"
      "76:"  // Height 2: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 40b\n"
      "b 230f\n"
      "77:"  // Height 3
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "78:"  // Height 3: Column loop
      "cbz x10, 79f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "mov v16.16b, v8.16b\n"
      "mov v20.16b, v12.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v21.16b, v13.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v22.16b, v14.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v23.16b, v15.16b\n"
      "b 91f\n"
      "79:"  // Height 3: no bias
      "tbz %x[flags], #0, 90f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "bge 88f\n"
      "tbz x12, #3, 83f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "tbz x12, #2, 81f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v19.4s }, [x23], #0x10\n"
      "tbz x12, #1, 80f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d24, [x23], #0x8\n"
      "tbz x12, #0, 87f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v24.s }[2], [x23]\n"
      "b 87f\n"
      "80:"  // Height 3: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x12, #0, 87f\n"
      "ldr s16, [x9, #0]\n"
      "ldr s15, [x24, #0]\n"
      "ldr s24, [x23, #0]\n"
      "b 87f\n"
      "81:"  // Height 3: Partial accumulate: partial_2_8
      "tbz x12, #1, 82f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d19, [x23], #0x8\n"
      "tbz x12, #0, 87f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "b 87f\n"
      "82:"  // Height 3: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x12, #0, 87f\n"
      "ldr s11, [x9, #0]\n"
      "ldr s14, [x24, #0]\n"
      "ldr s19, [x23, #0]\n"
      "b 87f\n"
      "83:"  // Height 3: Partial accumulate: partial_4_0
      "tbz x12, #2, 85f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "tbz x12, #1, 84f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d18, [x23], #0x8\n"
      "tbz x12, #0, 87f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "b 87f\n"
      "84:"  // Height 3: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x12, #0, 87f\n"
      "ldr s10, [x9, #0]\n"
      "ldr s13, [x24, #0]\n"
      "ldr s18, [x23, #0]\n"
      "b 87f\n"
      "85:"  // Height 3: Partial accumulate: partial_2_0
      "tbz x12, #1, 86f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "tbz x12, #0, 87f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "b 87f\n"
      "86:"  // Height 3: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0]\n"
      "ldr s12, [x24, #0]\n"
      "mov x25, #0\n"
      "ldr s17, [x23, #0]\n"
      "87:"  // Height 3: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 89f\n"
      "88:"  // Height 3: full accumulate
      "ldr q9, [x9, #0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x24, #0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q17, [x23, #0]\n"
      "ldr q18, [x23, #0x10]\n"
      "ldr q19, [x23, #0x20]\n"
      "ldr q24, [x23, #0x30]\n"
      "89:"  // Height 3: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "b 91f\n"
      "90:"  // Height 3: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "movi v20.4s, #0\n"
      "movi v21.4s, #0\n"
      "movi v22.4s, #0\n"
      "movi v23.4s, #0\n"
      "91:"  // Height 3: setup done
      "mov x28, #0\n"
      "92:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 93f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 94f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "b 94f\n"
      "93:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "94:"  // Height 3: input setup done
      "cmp x27, #0x10\n"
      "blt 97f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q2, [x25, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0]\n"
      "ldr q7, [x11, #0]\n"
      "ldr q6, [x11, #0x10]\n"
      "blt 96f\n"
      "95:"  // Height 3: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      "ldr q2, [x25, #0]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a470  // ummla v16.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a474  // ummla v20.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a471  // ummla v17.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a475  // ummla v21.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a472  // ummla v18.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a476  // ummla v22.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a473  // ummla v19.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0]\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      ".inst 0x6e86a477  // ummla v23.4s, v3.16b, v6.16b\n"
      "ldr q3, [x24, #0]\n"
      "ldr q6, [x11, #0x10]\n"
      "bge 95b\n"
      "96:"  // Height 3: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a470  // ummla v16.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a474  // ummla v20.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a471  // ummla v17.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a475  // ummla v21.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a472  // ummla v18.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a476  // ummla v22.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a473  // ummla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a477  // ummla v23.4s, v3.16b, v6.16b\n"
      "97:"  // Height 3: Multiply loop: Main loop skip
      "cbz x27, 104f\n"
      "cmp x27, #0x8\n"
      "blt 99f\n"
      "98:"  // Height 3: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d2, [x25], #0x8\n"
      "sub x27, x27, #0x8\n"
      "ldr d3, [x24], #0x8\n"
      "ldr q6, [x11, #0]\n"
      "cmp x27, #0x8\n"
      "ldr q7, [x11, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x6e86a408  // ummla v8.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a450  // ummla v16.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x20]\n"
      ".inst 0x6e87a40c  // ummla v12.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a454  // ummla v20.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x6e86a409  // ummla v9.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a451  // ummla v17.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x6e87a40d  // ummla v13.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a455  // ummla v21.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x6e86a40a  // ummla v10.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a452  // ummla v18.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x6e87a40e  // ummla v14.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a456  // ummla v22.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e86a40b  // ummla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a453  // ummla v19.4s, v2.16b, v6.16b\n"
      ".inst 0x6e87a40f  // ummla v15.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a457  // ummla v23.4s, v2.16b, v7.16b\n"
      "bge 98b\n"
      "99:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x27, 104f\n"
      "tbz x27, #2, 101f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "tbz x27, #1, 100f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "tbz x27, #0, 103f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "b 103f\n"
      "100:"  // Height 3: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 103f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "b 103f\n"
      "101:"  // Height 3: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 102f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "tbz x27, #0, 103f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "b 103f\n"
      "102:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "ldr b2, [x25, #0]\n"
      "ldr b3, [x24, #0]\n"
      "103:"  // Height 3: Multiply loop: Ragged operand read: Done
      "ldr q7, [x11, #0]\n"
      "ldr q6, [x11, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      "104:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 92b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x24, #0]\n"
      "prfm pstl1keep, [x23, #0]\n"
      "uzp1 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v19.2d, v19.2d, v23.2d\n"
      "bge 113f\n"
      "tbz x12, #3, 108f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v9.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "tbz x12, #2, 106f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "tbz x12, #1, 105f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "tbz x12, #0, 112f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "b 112f\n"
      "105:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x12, #0, 112f\n"
      "str s14, [x9, #0]\n"
      "str s11, [x24, #0]\n"
      "str s19, [x23, #0]\n"
      "b 112f\n"
      "106:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x12, #1, 107f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "tbz x12, #0, 112f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "b 112f\n"
      "107:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x12, #0, 112f\n"
      "str s13, [x9, #0]\n"
      "str s10, [x24, #0]\n"
      "str s18, [x23, #0]\n"
      "b 112f\n"
      "108:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x12, #2, 110f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "tbz x12, #1, 109f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "tbz x12, #0, 112f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "b 112f\n"
      "109:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x12, #0, 112f\n"
      "str s12, [x9, #0]\n"
      "str s9, [x24, #0]\n"
      "str s17, [x23, #0]\n"
      "b 112f\n"
      "110:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x12, #1, 111f\n"
      "str d7, [x9], #0x8\n"
      "str d8, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "tbz x12, #0, 112f\n"
      "st1 { v7.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "b 112f\n"
      "111:"  // Height 3: Partial direct writeback: partial_1_0
      "str s7, [x9, #0]\n"
      "str s8, [x24, #0]\n"
      "str s16, [x23, #0]\n"
      "112:"  // Height 3: Partial direct writeback: Done
      "b 114f\n"
      "113:"  // Height 3: Full writeback
      "str q7, [x9, #0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x24, #0]\n"
      "str q9, [x24, #0x10]\n"
      "str q10, [x24, #0x20]\n"
      "str q11, [x24, #0x30]\n"
      "str q16, [x23, #0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "114:"  // Height 3: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 78b\n"
      "b 230f\n"
      "115:"  // Height 4
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "116:"  // Height 4: Column loop
      "cbz x10, 117f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "mov v16.16b, v8.16b\n"
      "mov v20.16b, v12.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v21.16b, v13.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v22.16b, v14.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v23.16b, v15.16b\n"
      "b 129f\n"
      "117:"  // Height 4: no bias
      "tbz %x[flags], #0, 128f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "bge 126f\n"
      "tbz x12, #3, 121f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "tbz x12, #2, 119f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v19.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "tbz x12, #1, 118f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d24, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "tbz x12, #0, 125f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v24.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "b 125f\n"
      "118:"  // Height 4: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x12, #0, 125f\n"
      "ldr s16, [x9, #0]\n"
      "ldr s15, [x24, #0]\n"
      "ldr s24, [x23, #0]\n"
      "ldr s23, [x22, #0]\n"
      "b 125f\n"
      "119:"  // Height 4: Partial accumulate: partial_2_8
      "tbz x12, #1, 120f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "tbz x12, #0, 125f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "b 125f\n"
      "120:"  // Height 4: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x12, #0, 125f\n"
      "ldr s11, [x9, #0]\n"
      "ldr s14, [x24, #0]\n"
      "ldr s19, [x23, #0]\n"
      "ldr s22, [x22, #0]\n"
      "b 125f\n"
      "121:"  // Height 4: Partial accumulate: partial_4_0
      "tbz x12, #2, 123f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "tbz x12, #1, 122f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "tbz x12, #0, 125f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "b 125f\n"
      "122:"  // Height 4: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x12, #0, 125f\n"
      "ldr s10, [x9, #0]\n"
      "ldr s13, [x24, #0]\n"
      "ldr s18, [x23, #0]\n"
      "ldr s21, [x22, #0]\n"
      "b 125f\n"
      "123:"  // Height 4: Partial accumulate: partial_2_0
      "tbz x12, #1, 124f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "tbz x12, #0, 125f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "b 125f\n"
      "124:"  // Height 4: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0]\n"
      "ldr s12, [x24, #0]\n"
      "mov x25, #0\n"
      "ldr s17, [x23, #0]\n"
      "ldr s20, [x22, #0]\n"
      "125:"  // Height 4: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 127f\n"
      "126:"  // Height 4: full accumulate
      "ldr q9, [x9, #0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x24, #0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q17, [x23, #0]\n"
      "ldr q18, [x23, #0x10]\n"
      "ldr q19, [x23, #0x20]\n"
      "ldr q24, [x23, #0x30]\n"
      "ldr q20, [x22, #0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "127:"  // Height 4: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "b 129f\n"
      "128:"  // Height 4: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "movi v20.4s, #0\n"
      "movi v21.4s, #0\n"
      "movi v22.4s, #0\n"
      "movi v23.4s, #0\n"
      "129:"  // Height 4: setup done
      "mov x28, #0\n"
      "130:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 131f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 132f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 132f\n"
      "131:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "132:"  // Height 4: input setup done
      "cmp x27, #0x10\n"
      "blt 135f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q2, [x25, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0]\n"
      "ldr q4, [x23, #0]\n"
      "ldr q7, [x11, #0]\n"
      "ldr q6, [x11, #0x10]\n"
      "blt 134f\n"
      "133:"  // Height 4: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "ldr q4, [x23, #0]\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      "ldr q2, [x25, #0]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a470  // ummla v16.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a474  // ummla v20.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a471  // ummla v17.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a475  // ummla v21.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a472  // ummla v18.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a476  // ummla v22.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a473  // ummla v19.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0]\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      ".inst 0x6e86a477  // ummla v23.4s, v3.16b, v6.16b\n"
      "ldr q3, [x24, #0]\n"
      "ldr q6, [x11, #0x10]\n"
      "bge 133b\n"
      "134:"  // Height 4: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a470  // ummla v16.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a474  // ummla v20.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a471  // ummla v17.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a475  // ummla v21.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a472  // ummla v18.4s, v3.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a476  // ummla v22.4s, v3.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a473  // ummla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a477  // ummla v23.4s, v3.16b, v6.16b\n"
      "135:"  // Height 4: Multiply loop: Main loop skip
      "cbz x27, 142f\n"
      "cmp x27, #0x8\n"
      "blt 137f\n"
      "136:"  // Height 4: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d2, [x25], #0x8\n"
      "sub x27, x27, #0x8\n"
      "ldr d3, [x24], #0x8\n"
      "ldr d4, [x23], #0x8\n"
      "cmp x27, #0x8\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x6e86a408  // ummla v8.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a450  // ummla v16.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x20]\n"
      ".inst 0x6e87a40c  // ummla v12.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a454  // ummla v20.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x6e86a409  // ummla v9.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a451  // ummla v17.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x6e87a40d  // ummla v13.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a455  // ummla v21.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x6e86a40a  // ummla v10.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a452  // ummla v18.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x6e87a40e  // ummla v14.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a456  // ummla v22.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e86a40b  // ummla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a453  // ummla v19.4s, v2.16b, v6.16b\n"
      ".inst 0x6e87a40f  // ummla v15.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a457  // ummla v23.4s, v2.16b, v7.16b\n"
      "bge 136b\n"
      "137:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x27, 142f\n"
      "tbz x27, #2, 139f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "tbz x27, #1, 138f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "tbz x27, #0, 141f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "b 141f\n"
      "138:"  // Height 4: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 141f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "b 141f\n"
      "139:"  // Height 4: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 140f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "tbz x27, #0, 141f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "b 141f\n"
      "140:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "ldr b2, [x25, #0]\n"
      "ldr b3, [x24, #0]\n"
      "ldr b4, [x23, #0]\n"
      "141:"  // Height 4: Multiply loop: Ragged operand read: Done
      "ldr q7, [x11, #0]\n"
      "ldr q6, [x11, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      "142:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 130b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x24, #0]\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x23, #0]\n"
      "prfm pstl1keep, [x22, #0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "bge 151f\n"
      "tbz x12, #3, 146f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v9.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "st1 { v17.4s }, [x22], #0x10\n"
      "tbz x12, #2, 144f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x24], #0x10\n"
      "st1 { v21.4s }, [x23], #0x10\n"
      "st1 { v18.4s }, [x22], #0x10\n"
      "tbz x12, #1, 143f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x24], #0x8\n"
      "str d22, [x23], #0x8\n"
      "str d19, [x22], #0x8\n"
      "tbz x12, #0, 150f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x24]\n"
      "st1 { v22.s }[2], [x23]\n"
      "st1 { v19.s }[2], [x22]\n"
      "b 150f\n"
      "143:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x12, #0, 150f\n"
      "str s14, [x9, #0]\n"
      "str s11, [x24, #0]\n"
      "str s22, [x23, #0]\n"
      "str s19, [x22, #0]\n"
      "b 150f\n"
      "144:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x12, #1, 145f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x24], #0x8\n"
      "str d21, [x23], #0x8\n"
      "str d18, [x22], #0x8\n"
      "tbz x12, #0, 150f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x24]\n"
      "st1 { v21.s }[2], [x23]\n"
      "st1 { v18.s }[2], [x22]\n"
      "b 150f\n"
      "145:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x12, #0, 150f\n"
      "str s13, [x9, #0]\n"
      "str s10, [x24, #0]\n"
      "str s21, [x23, #0]\n"
      "str s18, [x22, #0]\n"
      "b 150f\n"
      "146:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x12, #2, 148f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "tbz x12, #1, 147f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x24], #0x8\n"
      "str d20, [x23], #0x8\n"
      "str d17, [x22], #0x8\n"
      "tbz x12, #0, 150f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x24]\n"
      "st1 { v20.s }[2], [x23]\n"
      "st1 { v17.s }[2], [x22]\n"
      "b 150f\n"
      "147:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x12, #0, 150f\n"
      "str s12, [x9, #0]\n"
      "str s9, [x24, #0]\n"
      "str s20, [x23, #0]\n"
      "str s17, [x22, #0]\n"
      "b 150f\n"
      "148:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x12, #1, 149f\n"
      "str d7, [x9], #0x8\n"
      "str d8, [x24], #0x8\n"
      "str d15, [x23], #0x8\n"
      "str d16, [x22], #0x8\n"
      "tbz x12, #0, 150f\n"
      "st1 { v7.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x24]\n"
      "st1 { v15.s }[2], [x23]\n"
      "st1 { v16.s }[2], [x22]\n"
      "b 150f\n"
      "149:"  // Height 4: Partial direct writeback: partial_1_0
      "str s7, [x9, #0]\n"
      "str s8, [x24, #0]\n"
      "str s15, [x23, #0]\n"
      "str s16, [x22, #0]\n"
      "150:"  // Height 4: Partial direct writeback: Done
      "b 152f\n"
      "151:"  // Height 4: Full writeback
      "str q7, [x9, #0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x24, #0]\n"
      "str q9, [x24, #0x10]\n"
      "str q10, [x24, #0x20]\n"
      "str q11, [x24, #0x30]\n"
      "str q15, [x23, #0]\n"
      "str q20, [x23, #0x10]\n"
      "str q21, [x23, #0x20]\n"
      "str q22, [x23, #0x30]\n"
      "str q16, [x22, #0]\n"
      "str q17, [x22, #0x10]\n"
      "str q18, [x22, #0x20]\n"
      "str q19, [x22, #0x30]\n"
      "152:"  // Height 4: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 116b\n"
      "b 230f\n"
      "153:"  // Height 5
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "154:"  // Height 5: Column loop
      "cbz x10, 155f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "mov v16.16b, v8.16b\n"
      "mov v20.16b, v12.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v21.16b, v13.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v22.16b, v14.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v23.16b, v15.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v28.16b, v12.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v29.16b, v13.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v30.16b, v14.16b\n"
      "mov v27.16b, v11.16b\n"
      "mov v31.16b, v15.16b\n"
      "b 167f\n"
      "155:"  // Height 5: no bias
      "tbz %x[flags], #0, 166f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "bge 164f\n"
      "tbz x12, #3, 159f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "tbz x12, #2, 157f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v19.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v27.4s }, [x21], #0x10\n"
      "tbz x12, #1, 156f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d24, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d6, [x21], #0x8\n"
      "tbz x12, #0, 163f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v24.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v6.s }[2], [x21]\n"
      "b 163f\n"
      "156:"  // Height 5: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x12, #0, 163f\n"
      "ldr s16, [x9, #0]\n"
      "ldr s15, [x24, #0]\n"
      "ldr s24, [x23, #0]\n"
      "ldr s23, [x22, #0]\n"
      "ldr s6, [x21, #0]\n"
      "b 163f\n"
      "157:"  // Height 5: Partial accumulate: partial_2_8
      "tbz x12, #1, 158f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "tbz x12, #0, 163f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "b 163f\n"
      "158:"  // Height 5: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x12, #0, 163f\n"
      "ldr s11, [x9, #0]\n"
      "ldr s14, [x24, #0]\n"
      "ldr s19, [x23, #0]\n"
      "ldr s22, [x22, #0]\n"
      "ldr s27, [x21, #0]\n"
      "b 163f\n"
      "159:"  // Height 5: Partial accumulate: partial_4_0
      "tbz x12, #2, 161f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "tbz x12, #1, 160f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "tbz x12, #0, 163f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "b 163f\n"
      "160:"  // Height 5: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x12, #0, 163f\n"
      "ldr s10, [x9, #0]\n"
      "ldr s13, [x24, #0]\n"
      "ldr s18, [x23, #0]\n"
      "ldr s21, [x22, #0]\n"
      "ldr s26, [x21, #0]\n"
      "b 163f\n"
      "161:"  // Height 5: Partial accumulate: partial_2_0
      "tbz x12, #1, 162f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "ldr d25, [x21], #0x8\n"
      "tbz x12, #0, 163f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "ld1 { v25.s }[2], [x21]\n"
      "b 163f\n"
      "162:"  // Height 5: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0]\n"
      "ldr s12, [x24, #0]\n"
      "mov x25, #0\n"
      "ldr s17, [x23, #0]\n"
      "ldr s20, [x22, #0]\n"
      "ldr s25, [x21, #0]\n"
      "163:"  // Height 5: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 165f\n"
      "164:"  // Height 5: full accumulate
      "ldr q9, [x9, #0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x24, #0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q17, [x23, #0]\n"
      "ldr q18, [x23, #0x10]\n"
      "ldr q19, [x23, #0x20]\n"
      "ldr q24, [x23, #0x30]\n"
      "ldr q20, [x22, #0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q25, [x21, #0]\n"
      "ldr q26, [x21, #0x10]\n"
      "ldr q27, [x21, #0x20]\n"
      "ldr q6, [x21, #0x30]\n"
      "165:"  // Height 5: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "zip1 v24.2d, v25.2d, v28.2d\n"
      "zip2 v28.2d, v25.2d, v28.2d\n"
      "zip1 v25.2d, v26.2d, v29.2d\n"
      "zip2 v29.2d, v26.2d, v29.2d\n"
      "zip1 v26.2d, v27.2d, v30.2d\n"
      "zip2 v30.2d, v27.2d, v30.2d\n"
      "zip1 v27.2d, v6.2d, v31.2d\n"
      "zip2 v31.2d, v6.2d, v31.2d\n"
      "b 167f\n"
      "166:"  // Height 5: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "movi v20.4s, #0\n"
      "movi v21.4s, #0\n"
      "movi v22.4s, #0\n"
      "movi v23.4s, #0\n"
      "movi v24.4s, #0\n"
      "movi v25.4s, #0\n"
      "movi v26.4s, #0\n"
      "movi v27.4s, #0\n"
      "movi v28.4s, #0\n"
      "movi v29.4s, #0\n"
      "movi v30.4s, #0\n"
      "movi v31.4s, #0\n"
      "167:"  // Height 5: setup done
      "mov x28, #0\n"
      "168:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 169f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 170f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 170f\n"
      "169:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "170:"  // Height 5: input setup done
      "cmp x27, #0x10\n"
      "blt 173f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q2, [x25, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0]\n"
      "ldr q4, [x23, #0]\n"
      "ldr q5, [x22, #0]\n"
      "ldr q7, [x11, #0]\n"
      "blt 172f\n"
      "171:"  // Height 5: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x27, x27, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x11, #0x10]\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      "add x22, x22, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x6e87a498  // ummla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49c  // ummla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a499  // ummla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49d  // ummla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49a  // ummla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49e  // ummla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49b  // ummla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      "ldr q2, [x25, #0]\n"
      ".inst 0x6e86a49f  // ummla v31.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      "ldr q4, [x23, #0]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a470  // ummla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4b8  // ummla v24.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a474  // ummla v20.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4bc  // ummla v28.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a471  // ummla v17.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4b9  // ummla v25.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a475  // ummla v21.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4bd  // ummla v29.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a472  // ummla v18.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4ba  // ummla v26.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a476  // ummla v22.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4be  // ummla v30.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a473  // ummla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4bb  // ummla v27.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0]\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      ".inst 0x6e86a477  // ummla v23.4s, v3.16b, v6.16b\n"
      "ldr q3, [x24, #0]\n"
      ".inst 0x6e86a4bf  // ummla v31.4s, v5.16b, v6.16b\n"
      "ldr q5, [x22, #0]\n"
      "bge 171b\n"
      "172:"  // Height 5: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x26, x26, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x11, #0x10]\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x6e87a498  // ummla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49c  // ummla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a499  // ummla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49d  // ummla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49a  // ummla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49e  // ummla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49b  // ummla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49f  // ummla v31.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a470  // ummla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4b8  // ummla v24.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a474  // ummla v20.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4bc  // ummla v28.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a471  // ummla v17.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4b9  // ummla v25.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a475  // ummla v21.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4bd  // ummla v29.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a472  // ummla v18.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4ba  // ummla v26.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a476  // ummla v22.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4be  // ummla v30.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a473  // ummla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4bb  // ummla v27.4s, v5.16b, v7.16b\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a477  // ummla v23.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4bf  // ummla v31.4s, v5.16b, v6.16b\n"
      "173:"  // Height 5: Multiply loop: Main loop skip
      "cbz x27, 180f\n"
      "cmp x27, #0x8\n"
      "blt 175f\n"
      "174:"  // Height 5: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d2, [x25], #0x8\n"
      "sub x27, x27, #0x8\n"
      "ldr d3, [x24], #0x8\n"
      "ldr d4, [x23], #0x8\n"
      "cmp x27, #0x8\n"
      "ldr d5, [x22], #0x8\n"
      "ldr q6, [x11, #0]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn1 v4.2d, v5.2d, v7.2d\n"
      "ldr q7, [x11, #0x10]\n"
      ".inst 0x6e86a408  // ummla v8.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a450  // ummla v16.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a498  // ummla v24.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x20]\n"
      ".inst 0x6e87a40c  // ummla v12.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a454  // ummla v20.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49c  // ummla v28.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x6e86a409  // ummla v9.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a451  // ummla v17.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a499  // ummla v25.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x6e87a40d  // ummla v13.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a455  // ummla v21.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49d  // ummla v29.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x6e86a40a  // ummla v10.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a452  // ummla v18.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49a  // ummla v26.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x6e87a40e  // ummla v14.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a456  // ummla v22.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49e  // ummla v30.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e86a40b  // ummla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a453  // ummla v19.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49b  // ummla v27.4s, v4.16b, v6.16b\n"
      ".inst 0x6e87a40f  // ummla v15.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a457  // ummla v23.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49f  // ummla v31.4s, v4.16b, v7.16b\n"
      "bge 174b\n"
      "175:"  // Height 5: Multiply loop: Skip odd blocks
      "cbz x27, 180f\n"
      "tbz x27, #2, 177f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "ldr s5, [x22], #0x4\n"
      "tbz x27, #1, 176f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "ld1 { v5.h }[2], [x22], #0x2\n"
      "tbz x27, #0, 179f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "ld1 { v5.b }[6], [x22]\n"
      "b 179f\n"
      "176:"  // Height 5: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 179f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "ld1 { v5.b }[4], [x22]\n"
      "b 179f\n"
      "177:"  // Height 5: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 178f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "ldr h5, [x22], #0x2\n"
      "tbz x27, #0, 179f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "ld1 { v5.b }[2], [x22]\n"
      "b 179f\n"
      "178:"  // Height 5: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "ldr b2, [x25, #0]\n"
      "ldr b3, [x24, #0]\n"
      "ldr b4, [x23, #0]\n"
      "ldr b5, [x22, #0]\n"
      "179:"  // Height 5: Multiply loop: Ragged operand read: Done
      "ldr q7, [x11, #0]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "ldr q6, [x11, #0x10]\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a498  // ummla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49c  // ummla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a499  // ummla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49d  // ummla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49a  // ummla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49e  // ummla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49b  // ummla v27.4s, v4.16b, v7.16b\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49f  // ummla v31.4s, v4.16b, v6.16b\n"
      "180:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 168b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x24, #0]\n"
      "add x21, x22, x20, LSL #2\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x23, #0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "prfm pstl1keep, [x22, #0]\n"
      "prfm pstl1keep, [x21, #0]\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "uzp1 v24.2d, v24.2d, v28.2d\n"
      "uzp1 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v27.2d, v27.2d, v31.2d\n"
      "bge 189f\n"
      "tbz x12, #3, 184f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v9.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "st1 { v17.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "st1 { v25.4s }, [x21], #0x10\n"
      "tbz x12, #2, 182f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x24], #0x10\n"
      "st1 { v21.4s }, [x23], #0x10\n"
      "st1 { v18.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x21], #0x10\n"
      "tbz x12, #1, 181f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x24], #0x8\n"
      "str d22, [x23], #0x8\n"
      "str d19, [x22], #0x8\n"
      "str d27, [x21], #0x8\n"
      "tbz x12, #0, 188f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x24]\n"
      "st1 { v22.s }[2], [x23]\n"
      "st1 { v19.s }[2], [x22]\n"
      "st1 { v27.s }[2], [x21]\n"
      "b 188f\n"
      "181:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x12, #0, 188f\n"
      "str s14, [x9, #0]\n"
      "str s11, [x24, #0]\n"
      "str s22, [x23, #0]\n"
      "str s19, [x22, #0]\n"
      "str s27, [x21, #0]\n"
      "b 188f\n"
      "182:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x12, #1, 183f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x24], #0x8\n"
      "str d21, [x23], #0x8\n"
      "str d18, [x22], #0x8\n"
      "str d26, [x21], #0x8\n"
      "tbz x12, #0, 188f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x24]\n"
      "st1 { v21.s }[2], [x23]\n"
      "st1 { v18.s }[2], [x22]\n"
      "st1 { v26.s }[2], [x21]\n"
      "b 188f\n"
      "183:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x12, #0, 188f\n"
      "str s13, [x9, #0]\n"
      "str s10, [x24, #0]\n"
      "str s21, [x23, #0]\n"
      "str s18, [x22, #0]\n"
      "str s26, [x21, #0]\n"
      "b 188f\n"
      "184:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x12, #2, 186f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "tbz x12, #1, 185f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x24], #0x8\n"
      "str d20, [x23], #0x8\n"
      "str d17, [x22], #0x8\n"
      "str d25, [x21], #0x8\n"
      "tbz x12, #0, 188f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x24]\n"
      "st1 { v20.s }[2], [x23]\n"
      "st1 { v17.s }[2], [x22]\n"
      "st1 { v25.s }[2], [x21]\n"
      "b 188f\n"
      "185:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x12, #0, 188f\n"
      "str s12, [x9, #0]\n"
      "str s9, [x24, #0]\n"
      "str s20, [x23, #0]\n"
      "str s17, [x22, #0]\n"
      "str s25, [x21, #0]\n"
      "b 188f\n"
      "186:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x12, #1, 187f\n"
      "str d7, [x9], #0x8\n"
      "str d8, [x24], #0x8\n"
      "str d15, [x23], #0x8\n"
      "str d16, [x22], #0x8\n"
      "str d24, [x21], #0x8\n"
      "tbz x12, #0, 188f\n"
      "st1 { v7.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x24]\n"
      "st1 { v15.s }[2], [x23]\n"
      "st1 { v16.s }[2], [x22]\n"
      "st1 { v24.s }[2], [x21]\n"
      "b 188f\n"
      "187:"  // Height 5: Partial direct writeback: partial_1_0
      "str s7, [x9, #0]\n"
      "str s8, [x24, #0]\n"
      "str s15, [x23, #0]\n"
      "str s16, [x22, #0]\n"
      "str s24, [x21, #0]\n"
      "188:"  // Height 5: Partial direct writeback: Done
      "b 190f\n"
      "189:"  // Height 5: Full writeback
      "str q7, [x9, #0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x24, #0]\n"
      "str q9, [x24, #0x10]\n"
      "str q10, [x24, #0x20]\n"
      "str q11, [x24, #0x30]\n"
      "str q15, [x23, #0]\n"
      "str q20, [x23, #0x10]\n"
      "str q21, [x23, #0x20]\n"
      "str q22, [x23, #0x30]\n"
      "str q16, [x22, #0]\n"
      "str q17, [x22, #0x10]\n"
      "str q18, [x22, #0x20]\n"
      "str q19, [x22, #0x30]\n"
      "str q24, [x21, #0]\n"
      "str q25, [x21, #0x10]\n"
      "str q26, [x21, #0x20]\n"
      "str q27, [x21, #0x30]\n"
      "190:"  // Height 5: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 154b\n"
      "b 230f\n"
      "191:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x18\n"
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "madd x20, x21, x20, x9\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "192:"  // Height 6: Column loop
      "cbz x10, 193f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "mov v16.16b, v8.16b\n"
      "mov v20.16b, v12.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v21.16b, v13.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v22.16b, v14.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v23.16b, v15.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v28.16b, v12.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v29.16b, v13.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v30.16b, v14.16b\n"
      "mov v27.16b, v11.16b\n"
      "mov v31.16b, v15.16b\n"
      "b 205f\n"
      "193:"  // Height 6: no bias
      "tbz %x[flags], #0, 204f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "bge 202f\n"
      "tbz x12, #3, 197f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "ld1 { v28.4s }, [x20], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "ld1 { v29.4s }, [x20], #0x10\n"
      "tbz x12, #2, 195f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v19.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v27.4s }, [x21], #0x10\n"
      "ld1 { v30.4s }, [x20], #0x10\n"
      "tbz x12, #1, 194f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x24], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d24, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d6, [x21], #0x8\n"
      "ldr d31, [x20], #0x8\n"
      "tbz x12, #0, 201f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v24.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v6.s }[2], [x21]\n"
      "ld1 { v31.s }[2], [x20]\n"
      "b 201f\n"
      "194:"  // Height 6: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x12, #0, 201f\n"
      "ldr s16, [x9, #0]\n"
      "ldr s15, [x24, #0]\n"
      "ldr s24, [x23, #0]\n"
      "ldr s23, [x22, #0]\n"
      "ldr s6, [x21, #0]\n"
      "ldr s31, [x20, #0]\n"
      "b 201f\n"
      "195:"  // Height 6: Partial accumulate: partial_2_8
      "tbz x12, #1, 196f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x24], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "ldr d30, [x20], #0x8\n"
      "tbz x12, #0, 201f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "ld1 { v30.s }[2], [x20]\n"
      "b 201f\n"
      "196:"  // Height 6: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x12, #0, 201f\n"
      "ldr s11, [x9, #0]\n"
      "ldr s14, [x24, #0]\n"
      "ldr s19, [x23, #0]\n"
      "ldr s22, [x22, #0]\n"
      "ldr s27, [x21, #0]\n"
      "ldr s30, [x20, #0]\n"
      "b 201f\n"
      "197:"  // Height 6: Partial accumulate: partial_4_0
      "tbz x12, #2, 199f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "ld1 { v28.4s }, [x20], #0x10\n"
      "tbz x12, #1, 198f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x24], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "ldr d29, [x20], #0x8\n"
      "tbz x12, #0, 201f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "ld1 { v29.s }[2], [x20]\n"
      "b 201f\n"
      "198:"  // Height 6: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x12, #0, 201f\n"
      "ldr s10, [x9, #0]\n"
      "ldr s13, [x24, #0]\n"
      "ldr s18, [x23, #0]\n"
      "ldr s21, [x22, #0]\n"
      "ldr s26, [x21, #0]\n"
      "ldr s29, [x20, #0]\n"
      "b 201f\n"
      "199:"  // Height 6: Partial accumulate: partial_2_0
      "tbz x12, #1, 200f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "ldr d25, [x21], #0x8\n"
      "ldr d28, [x20], #0x8\n"
      "tbz x12, #0, 201f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "ld1 { v25.s }[2], [x21]\n"
      "ld1 { v28.s }[2], [x20]\n"
      "b 201f\n"
      "200:"  // Height 6: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0]\n"
      "ldr s12, [x24, #0]\n"
      "mov x25, #0\n"
      "ldr s17, [x23, #0]\n"
      "ldr s20, [x22, #0]\n"
      "ldr s25, [x21, #0]\n"
      "ldr s28, [x20, #0]\n"
      "201:"  // Height 6: Partial accumulate: Done
      "sub x9, x9, x25\n"
      "b 203f\n"
      "202:"  // Height 6: full accumulate
      "ldr q9, [x9, #0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x24, #0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q17, [x23, #0]\n"
      "ldr q18, [x23, #0x10]\n"
      "ldr q19, [x23, #0x20]\n"
      "ldr q24, [x23, #0x30]\n"
      "ldr q20, [x22, #0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q25, [x21, #0]\n"
      "ldr q26, [x21, #0x10]\n"
      "ldr q27, [x21, #0x20]\n"
      "ldr q6, [x21, #0x30]\n"
      "ldr q28, [x20, #0]\n"
      "ldr q29, [x20, #0x10]\n"
      "ldr q30, [x20, #0x20]\n"
      "ldr q31, [x20, #0x30]\n"
      "203:"  // Height 6: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "zip1 v24.2d, v25.2d, v28.2d\n"
      "zip2 v28.2d, v25.2d, v28.2d\n"
      "zip1 v25.2d, v26.2d, v29.2d\n"
      "zip2 v29.2d, v26.2d, v29.2d\n"
      "zip1 v26.2d, v27.2d, v30.2d\n"
      "zip2 v30.2d, v27.2d, v30.2d\n"
      "zip1 v27.2d, v6.2d, v31.2d\n"
      "zip2 v31.2d, v6.2d, v31.2d\n"
      "b 205f\n"
      "204:"  // Height 6: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "movi v20.4s, #0\n"
      "movi v21.4s, #0\n"
      "movi v22.4s, #0\n"
      "movi v23.4s, #0\n"
      "movi v24.4s, #0\n"
      "movi v25.4s, #0\n"
      "movi v26.4s, #0\n"
      "movi v27.4s, #0\n"
      "movi v28.4s, #0\n"
      "movi v29.4s, #0\n"
      "movi v30.4s, #0\n"
      "movi v31.4s, #0\n"
      "205:"  // Height 6: setup done
      "mov x28, #0\n"
      "206:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 207f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 208f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 208f\n"
      "207:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "208:"  // Height 6: input setup done
      "cmp x27, #0x10\n"
      "blt 211f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q2, [x25, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0]\n"
      "ldr q4, [x23, #0]\n"
      "ldr q5, [x22, #0]\n"
      "ldr q6, [x21, #0]\n"
      "ldr q7, [x11, #0]\n"
      "blt 210f\n"
      "209:"  // Height 6: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x11, #0x10]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x6e87a498  // ummla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49c  // ummla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a499  // ummla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49d  // ummla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49a  // ummla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49e  // ummla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49b  // ummla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      "ldr q2, [x25, #0]\n"
      ".inst 0x6e86a49f  // ummla v31.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      "ldr q4, [x23, #0]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a470  // ummla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4b8  // ummla v24.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a474  // ummla v20.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4bc  // ummla v28.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a471  // ummla v17.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4b9  // ummla v25.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a475  // ummla v21.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4bd  // ummla v29.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a472  // ummla v18.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4ba  // ummla v26.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a476  // ummla v22.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4be  // ummla v30.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a473  // ummla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4bb  // ummla v27.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0]\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      ".inst 0x6e86a477  // ummla v23.4s, v3.16b, v6.16b\n"
      "ldr q3, [x24, #0]\n"
      ".inst 0x6e86a4bf  // ummla v31.4s, v5.16b, v6.16b\n"
      "ldr q5, [x22, #0]\n"
      "ldr q6, [x21, #0]\n"
      "bge 209b\n"
      "210:"  // Height 6: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x11, #0x10]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      "add x21, x21, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x6e87a498  // ummla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49c  // ummla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a499  // ummla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49d  // ummla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49a  // ummla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49e  // ummla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49b  // ummla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x80]\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49f  // ummla v31.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x90]\n"
      ".inst 0x6e87a428  // ummla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a470  // ummla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4b8  // ummla v24.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xa0]\n"
      ".inst 0x6e86a42c  // ummla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a474  // ummla v20.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4bc  // ummla v28.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xb0]\n"
      ".inst 0x6e87a429  // ummla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a471  // ummla v17.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4b9  // ummla v25.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xc0]\n"
      ".inst 0x6e86a42d  // ummla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a475  // ummla v21.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4bd  // ummla v29.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xd0]\n"
      ".inst 0x6e87a42a  // ummla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a472  // ummla v18.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4ba  // ummla v26.4s, v5.16b, v7.16b\n"
      "ldr q7, [x11, #0xe0]\n"
      ".inst 0x6e86a42e  // ummla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a476  // ummla v22.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4be  // ummla v30.4s, v5.16b, v6.16b\n"
      "ldr q6, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x6e87a42b  // ummla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x6e87a473  // ummla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x6e87a4bb  // ummla v27.4s, v5.16b, v7.16b\n"
      ".inst 0x6e86a42f  // ummla v15.4s, v1.16b, v6.16b\n"
      ".inst 0x6e86a477  // ummla v23.4s, v3.16b, v6.16b\n"
      ".inst 0x6e86a4bf  // ummla v31.4s, v5.16b, v6.16b\n"
      "211:"  // Height 6: Multiply loop: Main loop skip
      "cbz x27, 218f\n"
      "cmp x27, #0x8\n"
      "blt 213f\n"
      "212:"  // Height 6: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d2, [x25], #0x8\n"
      "sub x27, x27, #0x8\n"
      "ldr d3, [x24], #0x8\n"
      "ldr d4, [x23], #0x8\n"
      "cmp x27, #0x8\n"
      "ldr d5, [x22], #0x8\n"
      "ldr d7, [x21], #0x8\n"
      "ldr q6, [x11, #0]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn1 v4.2d, v5.2d, v7.2d\n"
      "ldr q7, [x11, #0x10]\n"
      ".inst 0x6e86a408  // ummla v8.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a450  // ummla v16.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a498  // ummla v24.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x20]\n"
      ".inst 0x6e87a40c  // ummla v12.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a454  // ummla v20.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49c  // ummla v28.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x6e86a409  // ummla v9.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a451  // ummla v17.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a499  // ummla v25.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x6e87a40d  // ummla v13.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a455  // ummla v21.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49d  // ummla v29.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x6e86a40a  // ummla v10.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a452  // ummla v18.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49a  // ummla v26.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x6e87a40e  // ummla v14.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a456  // ummla v22.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49e  // ummla v30.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e86a40b  // ummla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a453  // ummla v19.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49b  // ummla v27.4s, v4.16b, v6.16b\n"
      ".inst 0x6e87a40f  // ummla v15.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a457  // ummla v23.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49f  // ummla v31.4s, v4.16b, v7.16b\n"
      "bge 212b\n"
      "213:"  // Height 6: Multiply loop: Skip odd blocks
      "cbz x27, 218f\n"
      "tbz x27, #2, 215f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "ldr s5, [x22], #0x4\n"
      "ldr s6, [x21], #0x4\n"
      "tbz x27, #1, 214f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "ld1 { v5.h }[2], [x22], #0x2\n"
      "ld1 { v6.h }[2], [x21], #0x2\n"
      "tbz x27, #0, 217f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "ld1 { v5.b }[6], [x22]\n"
      "ld1 { v6.b }[6], [x21]\n"
      "b 217f\n"
      "214:"  // Height 6: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 217f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "ld1 { v5.b }[4], [x22]\n"
      "ld1 { v6.b }[4], [x21]\n"
      "b 217f\n"
      "215:"  // Height 6: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 216f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "ldr h5, [x22], #0x2\n"
      "ldr h6, [x21], #0x2\n"
      "tbz x27, #0, 217f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "ld1 { v5.b }[2], [x22]\n"
      "ld1 { v6.b }[2], [x21]\n"
      "b 217f\n"
      "216:"  // Height 6: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "ldr b2, [x25, #0]\n"
      "ldr b3, [x24, #0]\n"
      "ldr b4, [x23, #0]\n"
      "ldr b5, [x22, #0]\n"
      "ldr b6, [x21, #0]\n"
      "217:"  // Height 6: Multiply loop: Ragged operand read: Done
      "ldr q7, [x11, #0]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "ldr q6, [x11, #0x10]\n"
      ".inst 0x6e87a408  // ummla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a450  // ummla v16.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a498  // ummla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x20]\n"
      ".inst 0x6e86a40c  // ummla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a454  // ummla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49c  // ummla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x30]\n"
      ".inst 0x6e87a409  // ummla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a451  // ummla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a499  // ummla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x40]\n"
      ".inst 0x6e86a40d  // ummla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a455  // ummla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49d  // ummla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x50]\n"
      ".inst 0x6e87a40a  // ummla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a452  // ummla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49a  // ummla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x11, #0x60]\n"
      ".inst 0x6e86a40e  // ummla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a456  // ummla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49e  // ummla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x11, #0x70]\n"
      "add x11, x11, #0x80\n"
      ".inst 0x6e87a40b  // ummla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x6e87a453  // ummla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x6e87a49b  // ummla v27.4s, v4.16b, v7.16b\n"
      ".inst 0x6e86a40f  // ummla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x6e86a457  // ummla v23.4s, v2.16b, v6.16b\n"
      ".inst 0x6e86a49f  // ummla v31.4s, v4.16b, v6.16b\n"
      "218:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 206b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x24, #0]\n"
      "add x21, x22, x20, LSL #2\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x23, #0]\n"
      "add x20, x21, x20, LSL #2\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "prfm pstl1keep, [x22, #0]\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "prfm pstl1keep, [x21, #0]\n"
      "prfm pstl1keep, [x20, #0]\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "uzp1 v23.2d, v24.2d, v28.2d\n"
      "uzp2 v24.2d, v24.2d, v28.2d\n"
      "uzp1 v28.2d, v25.2d, v29.2d\n"
      "uzp2 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v29.2d, v26.2d, v30.2d\n"
      "uzp2 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v30.2d, v27.2d, v31.2d\n"
      "uzp2 v27.2d, v27.2d, v31.2d\n"
      "bge 227f\n"
      "tbz x12, #3, 222f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v9.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "st1 { v17.4s }, [x22], #0x10\n"
      "st1 { v23.4s }, [x21], #0x10\n"
      "st1 { v28.4s }, [x21], #0x10\n"
      "st1 { v24.4s }, [x20], #0x10\n"
      "st1 { v25.4s }, [x20], #0x10\n"
      "tbz x12, #2, 220f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x24], #0x10\n"
      "st1 { v21.4s }, [x23], #0x10\n"
      "st1 { v18.4s }, [x22], #0x10\n"
      "st1 { v29.4s }, [x21], #0x10\n"
      "st1 { v26.4s }, [x20], #0x10\n"
      "tbz x12, #1, 219f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x24], #0x8\n"
      "str d22, [x23], #0x8\n"
      "str d19, [x22], #0x8\n"
      "str d30, [x21], #0x8\n"
      "str d27, [x20], #0x8\n"
      "tbz x12, #0, 226f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x24]\n"
      "st1 { v22.s }[2], [x23]\n"
      "st1 { v19.s }[2], [x22]\n"
      "st1 { v30.s }[2], [x21]\n"
      "st1 { v27.s }[2], [x20]\n"
      "b 226f\n"
      "219:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x12, #0, 226f\n"
      "str s14, [x9, #0]\n"
      "str s11, [x24, #0]\n"
      "str s22, [x23, #0]\n"
      "str s19, [x22, #0]\n"
      "str s30, [x21, #0]\n"
      "str s27, [x20, #0]\n"
      "b 226f\n"
      "220:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x12, #1, 221f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x24], #0x8\n"
      "str d21, [x23], #0x8\n"
      "str d18, [x22], #0x8\n"
      "str d29, [x21], #0x8\n"
      "str d26, [x20], #0x8\n"
      "tbz x12, #0, 226f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x24]\n"
      "st1 { v21.s }[2], [x23]\n"
      "st1 { v18.s }[2], [x22]\n"
      "st1 { v29.s }[2], [x21]\n"
      "st1 { v26.s }[2], [x20]\n"
      "b 226f\n"
      "221:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x12, #0, 226f\n"
      "str s13, [x9, #0]\n"
      "str s10, [x24, #0]\n"
      "str s21, [x23, #0]\n"
      "str s18, [x22, #0]\n"
      "str s29, [x21, #0]\n"
      "str s26, [x20, #0]\n"
      "b 226f\n"
      "222:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x12, #2, 224f\n"
      "st1 { v7.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x24], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x22], #0x10\n"
      "st1 { v23.4s }, [x21], #0x10\n"
      "st1 { v24.4s }, [x20], #0x10\n"
      "tbz x12, #1, 223f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x24], #0x8\n"
      "str d20, [x23], #0x8\n"
      "str d17, [x22], #0x8\n"
      "str d28, [x21], #0x8\n"
      "str d25, [x20], #0x8\n"
      "tbz x12, #0, 226f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x24]\n"
      "st1 { v20.s }[2], [x23]\n"
      "st1 { v17.s }[2], [x22]\n"
      "st1 { v28.s }[2], [x21]\n"
      "st1 { v25.s }[2], [x20]\n"
      "b 226f\n"
      "223:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x12, #0, 226f\n"
      "str s12, [x9, #0]\n"
      "str s9, [x24, #0]\n"
      "str s20, [x23, #0]\n"
      "str s17, [x22, #0]\n"
      "str s28, [x21, #0]\n"
      "str s25, [x20, #0]\n"
      "b 226f\n"
      "224:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x12, #1, 225f\n"
      "str d7, [x9], #0x8\n"
      "str d8, [x24], #0x8\n"
      "str d15, [x23], #0x8\n"
      "str d16, [x22], #0x8\n"
      "str d23, [x21], #0x8\n"
      "str d24, [x20], #0x8\n"
      "tbz x12, #0, 226f\n"
      "st1 { v7.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x24]\n"
      "st1 { v15.s }[2], [x23]\n"
      "st1 { v16.s }[2], [x22]\n"
      "st1 { v23.s }[2], [x21]\n"
      "st1 { v24.s }[2], [x20]\n"
      "b 226f\n"
      "225:"  // Height 6: Partial direct writeback: partial_1_0
      "str s7, [x9, #0]\n"
      "str s8, [x24, #0]\n"
      "str s15, [x23, #0]\n"
      "str s16, [x22, #0]\n"
      "str s23, [x21, #0]\n"
      "str s24, [x20, #0]\n"
      "226:"  // Height 6: Partial direct writeback: Done
      "b 228f\n"
      "227:"  // Height 6: Full writeback
      "str q7, [x9, #0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x24, #0]\n"
      "str q9, [x24, #0x10]\n"
      "str q10, [x24, #0x20]\n"
      "str q11, [x24, #0x30]\n"
      "str q15, [x23, #0]\n"
      "str q20, [x23, #0x10]\n"
      "str q21, [x23, #0x20]\n"
      "str q22, [x23, #0x30]\n"
      "str q16, [x22, #0]\n"
      "str q17, [x22, #0x10]\n"
      "str q18, [x22, #0x20]\n"
      "str q19, [x22, #0x30]\n"
      "str q23, [x21, #0]\n"
      "str q28, [x21, #0x10]\n"
      "str q29, [x21, #0x20]\n"
      "str q30, [x21, #0x30]\n"
      "str q24, [x20, #0]\n"
      "str q25, [x20, #0x10]\n"
      "str q26, [x20, #0x20]\n"
      "str q27, [x20, #0x30]\n"
      "228:"  // Height 6: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 192b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 230f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 229f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "229:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "230:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm

#endif // __aarch64__

