/*
 * Copyright (c) 2019-2021, 2023-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifdef __aarch64__

#include "arm_gemm/arm_gemm.hpp"
#include "arm_common/internal/utils.hpp"
#include <cassert>

namespace arm_gemm {

void a64_hybrid_s8s32_dot_6x16_a55 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int32_t> output_arg,
    const int32_t *bias, Activation, bool accumulate
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const int32_t *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 176f\n"
      "cmp %x[M], #0x4\n"
      "bgt 141f\n"
      "beq 106f\n"
      "cmp %x[M], #0x2\n"
      "bgt 71f\n"
      "beq 36f\n"
      "ldr x7, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x8, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x17, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "cbz x17, 3f\n"
      "ldr q8, [x17, #0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "add x17, x17, #0x40\n"
      "b 14f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 13f\n"
      "cmp x7, #0x10\n"
      "bge 12f\n"
      "tbz x7, #3, 7f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "ld1 { v9.4s }, [x16], #0x10\n"
      "tbz x7, #2, 5f\n"
      "ld1 { v10.4s }, [x16], #0x10\n"
      "tbz x7, #1, 4f\n"
      "ldr d11, [x16], #0x8\n"
      "mov x25, #0x38\n"
      "tbz x7, #0, 11f\n"
      "ld1 { v11.s }[2], [x16]\n"
      "b 11f\n"
      "4:"  // Height 1: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x7, #0, 11f\n"
      "ldr s11, [x16, #0]\n"
      "b 11f\n"
      "5:"  // Height 1: Partial accumulate: partial_2_8
      "tbz x7, #1, 6f\n"
      "ldr d10, [x16], #0x8\n"
      "mov x25, #0x28\n"
      "tbz x7, #0, 11f\n"
      "ld1 { v10.s }[2], [x16]\n"
      "b 11f\n"
      "6:"  // Height 1: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x7, #0, 11f\n"
      "ldr s10, [x16, #0]\n"
      "b 11f\n"
      "7:"  // Height 1: Partial accumulate: partial_4_0
      "tbz x7, #2, 9f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "tbz x7, #1, 8f\n"
      "ldr d9, [x16], #0x8\n"
      "mov x25, #0x18\n"
      "tbz x7, #0, 11f\n"
      "ld1 { v9.s }[2], [x16]\n"
      "b 11f\n"
      "8:"  // Height 1: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x7, #0, 11f\n"
      "ldr s9, [x16, #0]\n"
      "b 11f\n"
      "9:"  // Height 1: Partial accumulate: partial_2_0
      "tbz x7, #1, 10f\n"
      "ldr d8, [x16], #0x8\n"
      "mov x25, #0x8\n"
      "tbz x7, #0, 11f\n"
      "ld1 { v8.s }[2], [x16]\n"
      "b 11f\n"
      "10:"  // Height 1: Partial accumulate: partial_1_0
      "ldr s8, [x16, #0]\n"
      "mov x25, #0\n"
      "11:"  // Height 1: Partial accumulate: Done
      "sub x16, x16, x25\n"
      "b 14f\n"
      "12:"  // Height 1: full accumulate
      "ldr q8, [x16, #0]\n"
      "ldr q9, [x16, #0x10]\n"
      "ldr q10, [x16, #0x20]\n"
      "ldr q11, [x16, #0x30]\n"
      "b 14f\n"
      "13:"  // Height 1: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "14:"  // Height 1: setup done
      "mov x15, #0\n"
      "15:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "tbz %x[flags], #3, 16f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0]\n"
      "cbnz x15, 17f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "b 17f\n"
      "16:"  // Height 1: setup direct input
      "mov x13, %x[input_ptr]\n"
      "17:"  // Height 1: input setup done
      "cmp x14, #0x10\n"
      "blt 20f\n"
      "ldr q0, [x13, #0]\n"
      "cmp x14, #0x20\n"
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      "blt 19f\n"
      "18:"  // Height 1: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr d6, [x8, #0x20]\n"
      "ldr x21, [x8, #0x28]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr d7, [x8, #0x30]\n"
      "add x13, x13, #0x10\n"
      "ldr x20, [x8, #0x38]\n"
      "sub x14, x14, #0x10\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0x48]\n"
      "ldr x22, [x13, #0x8]\n"
      "cmp x14, #0x20\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "ldr d6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "ldr d7, [x8, #0x50]\n"
      "ldr x20, [x8, #0x58]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0x68]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      "ldr d6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      "ldr d7, [x8, #0x70]\n"
      "ldr x20, [x8, #0x78]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0x88]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      "ldr d6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      "ldr d7, [x8, #0x90]\n"
      "ldr x20, [x8, #0x98]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0xa8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      "ldr d6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      "ldr d7, [x8, #0xb0]\n"
      "ldr x20, [x8, #0xb8]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0xc8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      "ldr d6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      "ldr d7, [x8, #0xd0]\n"
      "ldr x20, [x8, #0xd8]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0xe8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      "ldr d6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      "ldr d7, [x8, #0xf0]\n"
      "ldr x20, [x8, #0xf8]\n"
      "add x8, x8, #0x100\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0x8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      "ldr d6, [x8, #0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0]\n"
      "ldr d7, [x8, #0x10]\n"
      "ldr x20, [x8, #0x18]\n"
      "mov v6.d[1], x21\n"
      "mov v0.d[1], x22\n"
      "mov v7.d[1], x20\n"
      "bge 18b\n"
      "19:"  // Height 1: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      "add x13, x13, #0x10\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "ldr q6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "ldr q7, [x8, #0x50]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      "ldr q6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      "ldr q7, [x8, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      "ldr q6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      "ldr q7, [x8, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      "ldr q6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      "ldr q7, [x8, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      "ldr q6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      "ldr q7, [x8, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      "ldr q6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      "ldr q7, [x8, #0xf0]\n"
      "add x8, x8, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "20:"  // Height 1: Multiply loop: Main loop skip
      "cbz x14, 25f\n"
      "cmp x14, #0x4\n"
      "blt 22f\n"
      "21:"  // Height 1: Multiply loop: Odd block loop
      "ldr s0, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr q6, [x8, #0]\n"
      "cmp x14, #0x4\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "bge 21b\n"
      "22:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x14, 25f\n"
      "tbz x14, #1, 23f\n"
      "ldr h0, [x13], #0x2\n"
      "tbz x14, #0, 24f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "b 24f\n"
      "23:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0]\n"
      "24:"  // Height 1: Multiply loop: Ragged operand read: Done
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "25:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 15b\n"
      "cmp x7, #0x10\n"
      "prfm pstl1keep, [x16, #0]\n"
      "bge 34f\n"
      "tbz x7, #3, 29f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "st1 { v9.4s }, [x16], #0x10\n"
      "tbz x7, #2, 27f\n"
      "st1 { v10.4s }, [x16], #0x10\n"
      "tbz x7, #1, 26f\n"
      "str d11, [x16], #0x8\n"
      "tbz x7, #0, 33f\n"
      "st1 { v11.s }[2], [x16]\n"
      "b 33f\n"
      "26:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x7, #0, 33f\n"
      "str s11, [x16, #0]\n"
      "b 33f\n"
      "27:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x7, #1, 28f\n"
      "str d10, [x16], #0x8\n"
      "tbz x7, #0, 33f\n"
      "st1 { v10.s }[2], [x16]\n"
      "b 33f\n"
      "28:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x7, #0, 33f\n"
      "str s10, [x16, #0]\n"
      "b 33f\n"
      "29:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x7, #2, 31f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "tbz x7, #1, 30f\n"
      "str d9, [x16], #0x8\n"
      "tbz x7, #0, 33f\n"
      "st1 { v9.s }[2], [x16]\n"
      "b 33f\n"
      "30:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x7, #0, 33f\n"
      "str s9, [x16, #0]\n"
      "b 33f\n"
      "31:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x7, #1, 32f\n"
      "str d8, [x16], #0x8\n"
      "tbz x7, #0, 33f\n"
      "st1 { v8.s }[2], [x16]\n"
      "b 33f\n"
      "32:"  // Height 1: Partial direct writeback: partial_1_0
      "str s8, [x16, #0]\n"
      "33:"  // Height 1: Partial direct writeback: Done
      "b 35f\n"
      "34:"  // Height 1: Full writeback
      "str q8, [x16, #0]\n"
      "str q9, [x16, #0x10]\n"
      "str q10, [x16, #0x20]\n"
      "str q11, [x16, #0x30]\n"
      "add x16, x16, #0x40\n"
      "35:"  // Height 1: Writeback done
      "subs x7, x7, #0x10\n"
      "bgt 2b\n"
      "b 212f\n"
      "36:"  // Height 2
      "ldr x7, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x8, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x17, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "37:"  // Height 2: Column loop
      "cbz x17, 38f\n"
      "ldr q8, [x17, #0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "add x17, x17, #0x40\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "b 49f\n"
      "38:"  // Height 2: no bias
      "tbz %x[flags], #0, 48f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x7, #0x10\n"
      "add x24, x16, x20, LSL #2\n"
      "bge 47f\n"
      "tbz x7, #3, 42f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v9.4s }, [x16], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "tbz x7, #2, 40f\n"
      "ld1 { v10.4s }, [x16], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "tbz x7, #1, 39f\n"
      "ldr d11, [x16], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d15, [x24], #0x8\n"
      "tbz x7, #0, 46f\n"
      "ld1 { v11.s }[2], [x16]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "b 46f\n"
      "39:"  // Height 2: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x7, #0, 46f\n"
      "ldr s11, [x16, #0]\n"
      "ldr s15, [x24, #0]\n"
      "b 46f\n"
      "40:"  // Height 2: Partial accumulate: partial_2_8
      "tbz x7, #1, 41f\n"
      "ldr d10, [x16], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d14, [x24], #0x8\n"
      "tbz x7, #0, 46f\n"
      "ld1 { v10.s }[2], [x16]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "b 46f\n"
      "41:"  // Height 2: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x7, #0, 46f\n"
      "ldr s10, [x16, #0]\n"
      "ldr s14, [x24, #0]\n"
      "b 46f\n"
      "42:"  // Height 2: Partial accumulate: partial_4_0
      "tbz x7, #2, 44f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "tbz x7, #1, 43f\n"
      "ldr d9, [x16], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d13, [x24], #0x8\n"
      "tbz x7, #0, 46f\n"
      "ld1 { v9.s }[2], [x16]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "b 46f\n"
      "43:"  // Height 2: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x7, #0, 46f\n"
      "ldr s9, [x16, #0]\n"
      "ldr s13, [x24, #0]\n"
      "b 46f\n"
      "44:"  // Height 2: Partial accumulate: partial_2_0
      "tbz x7, #1, 45f\n"
      "ldr d8, [x16], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "tbz x7, #0, 46f\n"
      "ld1 { v8.s }[2], [x16]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "b 46f\n"
      "45:"  // Height 2: Partial accumulate: partial_1_0
      "ldr s8, [x16, #0]\n"
      "mov x25, #0\n"
      "ldr s12, [x24, #0]\n"
      "46:"  // Height 2: Partial accumulate: Done
      "sub x16, x16, x25\n"
      "b 49f\n"
      "47:"  // Height 2: full accumulate
      "ldr q8, [x16, #0]\n"
      "ldr q9, [x16, #0x10]\n"
      "ldr q10, [x16, #0x20]\n"
      "ldr q11, [x16, #0x30]\n"
      "ldr q12, [x24, #0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "b 49f\n"
      "48:"  // Height 2: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "49:"  // Height 2: setup done
      "mov x15, #0\n"
      "50:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "tbz %x[flags], #3, 51f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0]\n"
      "ldr x12, [x20, #0x8]\n"
      "cbnz x15, 52f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "add x12, x12, x20\n"
      "b 52f\n"
      "51:"  // Height 2: setup direct input
      "mov x13, %x[input_ptr]\n"
      "add x12, x13, x21\n"
      "52:"  // Height 2: input setup done
      "cmp x14, #0x10\n"
      "blt 55f\n"
      "ldr q0, [x13, #0]\n"
      "cmp x14, #0x20\n"
      "ldr q1, [x12, #0]\n"
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      "blt 54f\n"
      "53:"  // Height 2: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr x20, [x8, #0x28]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr d6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "ldr x21, [x8, #0x38]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr d7, [x8, #0x30]\n"
      "mov v6.d[1], x20\n"
      "ldr x20, [x8, #0x48]\n"
      "add x13, x13, #0x10\n"
      "add x12, x12, #0x10\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "ldr d6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "ldr x21, [x8, #0x58]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "ldr d7, [x8, #0x50]\n"
      "mov v6.d[1], x20\n"
      "ldr x20, [x8, #0x68]\n"
      "ldr x23, [x13, #0x8]\n"
      "sub x14, x14, #0x10\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      "ldr d6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      "ldr x21, [x8, #0x78]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      "ldr d7, [x8, #0x70]\n"
      "mov v6.d[1], x20\n"
      "ldr x20, [x8, #0x88]\n"
      "ldr x22, [x12, #0x8]\n"
      "cmp x14, #0x20\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      "ldr d6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      "ldr x21, [x8, #0x98]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      "ldr d7, [x8, #0x90]\n"
      "mov v6.d[1], x20\n"
      "ldr x20, [x8, #0xa8]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      "ldr d6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      "ldr x21, [x8, #0xb8]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      "ldr d7, [x8, #0xb0]\n"
      "mov v6.d[1], x20\n"
      "ldr x20, [x8, #0xc8]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      "ldr d6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      "ldr x21, [x8, #0xd8]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      "ldr d7, [x8, #0xd0]\n"
      "mov v6.d[1], x20\n"
      "ldr x20, [x8, #0xe8]\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      "ldr d6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      "ldr x21, [x8, #0xf8]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      "ldr d7, [x8, #0xf0]\n"
      "mov v6.d[1], x20\n"
      "add x8, x8, #0x100\n"
      "ldr x20, [x8, #0x8]\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      "ldr d6, [x8, #0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr d1, [x12, #0]\n"
      "ldr d7, [x8, #0x10]\n"
      "mov v6.d[1], x20\n"
      "ldr x20, [x8, #0x18]\n"
      "mov v0.d[1], x23\n"
      "mov v1.d[1], x22\n"
      "mov v7.d[1], x20\n"
      "bge 53b\n"
      "54:"  // Height 2: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "ldr q6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "ldr q7, [x8, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      "ldr q6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      "ldr q7, [x8, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      "ldr q6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      "ldr q7, [x8, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      "ldr q6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      "ldr q7, [x8, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      "ldr q6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      "ldr q7, [x8, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      "ldr q6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      "ldr q7, [x8, #0xf0]\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      "add x8, x8, #0x100\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "55:"  // Height 2: Multiply loop: Main loop skip
      "cbz x14, 60f\n"
      "cmp x14, #0x4\n"
      "blt 57f\n"
      "56:"  // Height 2: Multiply loop: Odd block loop
      "ldr s0, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr s1, [x12], #0x4\n"
      "cmp x14, #0x4\n"
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "bge 56b\n"
      "57:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x14, 60f\n"
      "tbz x14, #1, 58f\n"
      "ldr h0, [x13], #0x2\n"
      "ldr h1, [x12], #0x2\n"
      "tbz x14, #0, 59f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "ld1 { v1.b }[2], [x12]\n"
      "b 59f\n"
      "58:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0]\n"
      "ldr b1, [x12, #0]\n"
      "59:"  // Height 2: Multiply loop: Ragged operand read: Done
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "60:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 50b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x7, #0x10\n"
      "prfm pstl1keep, [x16, #0]\n"
      "add x24, x16, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0]\n"
      "bge 69f\n"
      "tbz x7, #3, 64f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "st1 { v9.4s }, [x16], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "tbz x7, #2, 62f\n"
      "st1 { v10.4s }, [x16], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "tbz x7, #1, 61f\n"
      "str d11, [x16], #0x8\n"
      "str d15, [x24], #0x8\n"
      "tbz x7, #0, 68f\n"
      "st1 { v11.s }[2], [x16]\n"
      "st1 { v15.s }[2], [x24]\n"
      "b 68f\n"
      "61:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x7, #0, 68f\n"
      "str s11, [x16, #0]\n"
      "str s15, [x24, #0]\n"
      "b 68f\n"
      "62:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x7, #1, 63f\n"
      "str d10, [x16], #0x8\n"
      "str d14, [x24], #0x8\n"
      "tbz x7, #0, 68f\n"
      "st1 { v10.s }[2], [x16]\n"
      "st1 { v14.s }[2], [x24]\n"
      "b 68f\n"
      "63:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x7, #0, 68f\n"
      "str s10, [x16, #0]\n"
      "str s14, [x24, #0]\n"
      "b 68f\n"
      "64:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x7, #2, 66f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "tbz x7, #1, 65f\n"
      "str d9, [x16], #0x8\n"
      "str d13, [x24], #0x8\n"
      "tbz x7, #0, 68f\n"
      "st1 { v9.s }[2], [x16]\n"
      "st1 { v13.s }[2], [x24]\n"
      "b 68f\n"
      "65:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x7, #0, 68f\n"
      "str s9, [x16, #0]\n"
      "str s13, [x24, #0]\n"
      "b 68f\n"
      "66:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x7, #1, 67f\n"
      "str d8, [x16], #0x8\n"
      "str d12, [x24], #0x8\n"
      "tbz x7, #0, 68f\n"
      "st1 { v8.s }[2], [x16]\n"
      "st1 { v12.s }[2], [x24]\n"
      "b 68f\n"
      "67:"  // Height 2: Partial direct writeback: partial_1_0
      "str s8, [x16, #0]\n"
      "str s12, [x24, #0]\n"
      "68:"  // Height 2: Partial direct writeback: Done
      "b 70f\n"
      "69:"  // Height 2: Full writeback
      "str q8, [x16, #0]\n"
      "str q9, [x16, #0x10]\n"
      "str q10, [x16, #0x20]\n"
      "str q11, [x16, #0x30]\n"
      "add x16, x16, #0x40\n"
      "str q12, [x24, #0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "70:"  // Height 2: Writeback done
      "subs x7, x7, #0x10\n"
      "bgt 37b\n"
      "b 212f\n"
      "71:"  // Height 3
      "ldr x7, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x8, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x17, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "72:"  // Height 3: Column loop
      "cbz x17, 73f\n"
      "ldr q8, [x17, #0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "add x17, x17, #0x40\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "b 84f\n"
      "73:"  // Height 3: no bias
      "tbz %x[flags], #0, 83f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x7, #0x10\n"
      "add x24, x16, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "bge 82f\n"
      "tbz x7, #3, 77f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v9.4s }, [x16], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "tbz x7, #2, 75f\n"
      "ld1 { v10.4s }, [x16], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "tbz x7, #1, 74f\n"
      "ldr d11, [x16], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d15, [x24], #0x8\n"
      "ldr d19, [x23], #0x8\n"
      "tbz x7, #0, 81f\n"
      "ld1 { v11.s }[2], [x16]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "b 81f\n"
      "74:"  // Height 3: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x7, #0, 81f\n"
      "ldr s11, [x16, #0]\n"
      "ldr s15, [x24, #0]\n"
      "ldr s19, [x23, #0]\n"
      "b 81f\n"
      "75:"  // Height 3: Partial accumulate: partial_2_8
      "tbz x7, #1, 76f\n"
      "ldr d10, [x16], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d14, [x24], #0x8\n"
      "ldr d18, [x23], #0x8\n"
      "tbz x7, #0, 81f\n"
      "ld1 { v10.s }[2], [x16]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "b 81f\n"
      "76:"  // Height 3: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x7, #0, 81f\n"
      "ldr s10, [x16, #0]\n"
      "ldr s14, [x24, #0]\n"
      "ldr s18, [x23, #0]\n"
      "b 81f\n"
      "77:"  // Height 3: Partial accumulate: partial_4_0
      "tbz x7, #2, 79f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "tbz x7, #1, 78f\n"
      "ldr d9, [x16], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d13, [x24], #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "tbz x7, #0, 81f\n"
      "ld1 { v9.s }[2], [x16]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "b 81f\n"
      "78:"  // Height 3: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x7, #0, 81f\n"
      "ldr s9, [x16, #0]\n"
      "ldr s13, [x24, #0]\n"
      "ldr s17, [x23, #0]\n"
      "b 81f\n"
      "79:"  // Height 3: Partial accumulate: partial_2_0
      "tbz x7, #1, 80f\n"
      "ldr d8, [x16], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "tbz x7, #0, 81f\n"
      "ld1 { v8.s }[2], [x16]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "b 81f\n"
      "80:"  // Height 3: Partial accumulate: partial_1_0
      "ldr s8, [x16, #0]\n"
      "mov x25, #0\n"
      "ldr s12, [x24, #0]\n"
      "ldr s16, [x23, #0]\n"
      "81:"  // Height 3: Partial accumulate: Done
      "sub x16, x16, x25\n"
      "b 84f\n"
      "82:"  // Height 3: full accumulate
      "ldr q8, [x16, #0]\n"
      "ldr q9, [x16, #0x10]\n"
      "ldr q10, [x16, #0x20]\n"
      "ldr q11, [x16, #0x30]\n"
      "ldr q12, [x24, #0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "b 84f\n"
      "83:"  // Height 3: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "84:"  // Height 3: setup done
      "mov x15, #0\n"
      "85:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "tbz %x[flags], #3, 86f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0]\n"
      "ldr x12, [x20, #0x8]\n"
      "ldr x11, [x20, #0x10]\n"
      "cbnz x15, 87f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "add x12, x12, x20\n"
      "add x11, x11, x20\n"
      "b 87f\n"
      "86:"  // Height 3: setup direct input
      "mov x13, %x[input_ptr]\n"
      "add x12, x13, x21\n"
      "add x11, x12, x21\n"
      "87:"  // Height 3: input setup done
      "cmp x14, #0x10\n"
      "blt 90f\n"
      "ldr q0, [x13, #0]\n"
      "cmp x14, #0x20\n"
      "ldr q1, [x12, #0]\n"
      "ldr q2, [x11, #0]\n"
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      "blt 89f\n"
      "88:"  // Height 3: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr x21, [x8, #0x28]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr x20, [x8, #0x38]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr d6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr d7, [x8, #0x30]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0x48]\n"
      "add x11, x11, #0x10\n"
      "ldr x24, [x13, #0x8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "ldr x20, [x8, #0x58]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      "ldr d6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "ldr x23, [x12, #0x8]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "ldr x22, [x11, #0x8]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      "ldr d7, [x8, #0x50]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0x68]\n"
      "sub x14, x14, #0x10\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      "ldr x20, [x8, #0x78]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      "ldr d6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      "cmp x14, #0x20\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      "ldr d7, [x8, #0x70]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0x88]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      "ldr x20, [x8, #0x98]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      "ldr d6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      "ldr d7, [x8, #0x90]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0xa8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      "ldr x20, [x8, #0xb8]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      "ldr d6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      "ldr d7, [x8, #0xb0]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0xc8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      "ldr x20, [x8, #0xd8]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      "ldr d6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      "ldr d7, [x8, #0xd0]\n"
      "mov v6.d[1], x21\n"
      "ldr x21, [x8, #0xe8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      "ldr x20, [x8, #0xf8]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      "ldr d6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      "ldr d7, [x8, #0xf0]\n"
      "mov v6.d[1], x21\n"
      "add x8, x8, #0x100\n"
      "ldr x21, [x8, #0x8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      "ldr x20, [x8, #0x18]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      "ldr d6, [x8, #0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr d1, [x12, #0]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      "ldr d2, [x11, #0]\n"
      "ldr d7, [x8, #0x10]\n"
      "mov v6.d[1], x21\n"
      "mov v0.d[1], x24\n"
      "mov v1.d[1], x23\n"
      "mov v2.d[1], x22\n"
      "mov v7.d[1], x20\n"
      "bge 88b\n"
      "89:"  // Height 3: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      "ldr q6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      "ldr q7, [x8, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      "ldr q6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      "ldr q7, [x8, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      "ldr q6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      "ldr q7, [x8, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      "ldr q6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      "ldr q7, [x8, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      "ldr q6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      "ldr q7, [x8, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      "ldr q6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      "ldr q7, [x8, #0xf0]\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      "add x8, x8, #0x100\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      "90:"  // Height 3: Multiply loop: Main loop skip
      "cbz x14, 95f\n"
      "cmp x14, #0x4\n"
      "blt 92f\n"
      "91:"  // Height 3: Multiply loop: Odd block loop
      "ldr s0, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr s1, [x12], #0x4\n"
      "cmp x14, #0x4\n"
      "ldr s2, [x11], #0x4\n"
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      "bge 91b\n"
      "92:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x14, 95f\n"
      "tbz x14, #1, 93f\n"
      "ldr h0, [x13], #0x2\n"
      "ldr h1, [x12], #0x2\n"
      "ldr h2, [x11], #0x2\n"
      "tbz x14, #0, 94f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "ld1 { v1.b }[2], [x12]\n"
      "ld1 { v2.b }[2], [x11]\n"
      "b 94f\n"
      "93:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0]\n"
      "ldr b1, [x12, #0]\n"
      "ldr b2, [x11, #0]\n"
      "94:"  // Height 3: Multiply loop: Ragged operand read: Done
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      "95:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 85b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x7, #0x10\n"
      "prfm pstl1keep, [x16, #0]\n"
      "add x24, x16, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0]\n"
      "prfm pstl1keep, [x23, #0]\n"
      "bge 104f\n"
      "tbz x7, #3, 99f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "st1 { v9.4s }, [x16], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "tbz x7, #2, 97f\n"
      "st1 { v10.4s }, [x16], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "tbz x7, #1, 96f\n"
      "str d11, [x16], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "tbz x7, #0, 103f\n"
      "st1 { v11.s }[2], [x16]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "b 103f\n"
      "96:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x7, #0, 103f\n"
      "str s11, [x16, #0]\n"
      "str s15, [x24, #0]\n"
      "str s19, [x23, #0]\n"
      "b 103f\n"
      "97:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x7, #1, 98f\n"
      "str d10, [x16], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "tbz x7, #0, 103f\n"
      "st1 { v10.s }[2], [x16]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "b 103f\n"
      "98:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x7, #0, 103f\n"
      "str s10, [x16, #0]\n"
      "str s14, [x24, #0]\n"
      "str s18, [x23, #0]\n"
      "b 103f\n"
      "99:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x7, #2, 101f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "tbz x7, #1, 100f\n"
      "str d9, [x16], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "tbz x7, #0, 103f\n"
      "st1 { v9.s }[2], [x16]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "b 103f\n"
      "100:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x7, #0, 103f\n"
      "str s9, [x16, #0]\n"
      "str s13, [x24, #0]\n"
      "str s17, [x23, #0]\n"
      "b 103f\n"
      "101:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x7, #1, 102f\n"
      "str d8, [x16], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "tbz x7, #0, 103f\n"
      "st1 { v8.s }[2], [x16]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "b 103f\n"
      "102:"  // Height 3: Partial direct writeback: partial_1_0
      "str s8, [x16, #0]\n"
      "str s12, [x24, #0]\n"
      "str s16, [x23, #0]\n"
      "103:"  // Height 3: Partial direct writeback: Done
      "b 105f\n"
      "104:"  // Height 3: Full writeback
      "str q8, [x16, #0]\n"
      "str q9, [x16, #0x10]\n"
      "str q10, [x16, #0x20]\n"
      "str q11, [x16, #0x30]\n"
      "add x16, x16, #0x40\n"
      "str q12, [x24, #0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "105:"  // Height 3: Writeback done
      "subs x7, x7, #0x10\n"
      "bgt 72b\n"
      "b 212f\n"
      "106:"  // Height 4
      "ldr x7, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x8, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x17, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "107:"  // Height 4: Column loop
      "cbz x17, 108f\n"
      "ldr q8, [x17, #0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "add x17, x17, #0x40\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "b 119f\n"
      "108:"  // Height 4: no bias
      "tbz %x[flags], #0, 118f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x7, #0x10\n"
      "add x24, x16, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "bge 117f\n"
      "tbz x7, #3, 112f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v9.4s }, [x16], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "tbz x7, #2, 110f\n"
      "ld1 { v10.4s }, [x16], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "tbz x7, #1, 109f\n"
      "ldr d11, [x16], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d15, [x24], #0x8\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "tbz x7, #0, 116f\n"
      "ld1 { v11.s }[2], [x16]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "b 116f\n"
      "109:"  // Height 4: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x7, #0, 116f\n"
      "ldr s11, [x16, #0]\n"
      "ldr s15, [x24, #0]\n"
      "ldr s19, [x23, #0]\n"
      "ldr s23, [x22, #0]\n"
      "b 116f\n"
      "110:"  // Height 4: Partial accumulate: partial_2_8
      "tbz x7, #1, 111f\n"
      "ldr d10, [x16], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d14, [x24], #0x8\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "tbz x7, #0, 116f\n"
      "ld1 { v10.s }[2], [x16]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "b 116f\n"
      "111:"  // Height 4: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x7, #0, 116f\n"
      "ldr s10, [x16, #0]\n"
      "ldr s14, [x24, #0]\n"
      "ldr s18, [x23, #0]\n"
      "ldr s22, [x22, #0]\n"
      "b 116f\n"
      "112:"  // Height 4: Partial accumulate: partial_4_0
      "tbz x7, #2, 114f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "tbz x7, #1, 113f\n"
      "ldr d9, [x16], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d13, [x24], #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "tbz x7, #0, 116f\n"
      "ld1 { v9.s }[2], [x16]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "b 116f\n"
      "113:"  // Height 4: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x7, #0, 116f\n"
      "ldr s9, [x16, #0]\n"
      "ldr s13, [x24, #0]\n"
      "ldr s17, [x23, #0]\n"
      "ldr s21, [x22, #0]\n"
      "b 116f\n"
      "114:"  // Height 4: Partial accumulate: partial_2_0
      "tbz x7, #1, 115f\n"
      "ldr d8, [x16], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "tbz x7, #0, 116f\n"
      "ld1 { v8.s }[2], [x16]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "b 116f\n"
      "115:"  // Height 4: Partial accumulate: partial_1_0
      "ldr s8, [x16, #0]\n"
      "mov x25, #0\n"
      "ldr s12, [x24, #0]\n"
      "ldr s16, [x23, #0]\n"
      "ldr s20, [x22, #0]\n"
      "116:"  // Height 4: Partial accumulate: Done
      "sub x16, x16, x25\n"
      "b 119f\n"
      "117:"  // Height 4: full accumulate
      "ldr q8, [x16, #0]\n"
      "ldr q9, [x16, #0x10]\n"
      "ldr q10, [x16, #0x20]\n"
      "ldr q11, [x16, #0x30]\n"
      "ldr q12, [x24, #0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "ldr q20, [x22, #0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "b 119f\n"
      "118:"  // Height 4: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "movi v20.4s, #0\n"
      "movi v21.4s, #0\n"
      "movi v22.4s, #0\n"
      "movi v23.4s, #0\n"
      "119:"  // Height 4: setup done
      "mov x15, #0\n"
      "120:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "tbz %x[flags], #3, 121f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0]\n"
      "ldr x12, [x20, #0x8]\n"
      "ldr x11, [x20, #0x10]\n"
      "ldr x10, [x20, #0x18]\n"
      "cbnz x15, 122f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "add x12, x12, x20\n"
      "add x11, x11, x20\n"
      "add x10, x10, x20\n"
      "b 122f\n"
      "121:"  // Height 4: setup direct input
      "mov x13, %x[input_ptr]\n"
      "add x12, x13, x21\n"
      "add x11, x12, x21\n"
      "add x10, x11, x21\n"
      "122:"  // Height 4: input setup done
      "cmp x14, #0x10\n"
      "blt 125f\n"
      "ldr q0, [x13, #0]\n"
      "cmp x14, #0x20\n"
      "ldr q1, [x12, #0]\n"
      "ldr q2, [x11, #0]\n"
      "ldr q3, [x10, #0]\n"
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      "blt 124f\n"
      "123:"  // Height 4: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr x20, [x8, #0x28]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr x21, [x8, #0x38]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr d6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "mov v6.d[1], x20\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr d7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "ldr x20, [x8, #0x48]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "add x10, x10, #0x10\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "ldr d6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "ldr x21, [x8, #0x58]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "ldr x25, [x13, #0x8]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      "mov v6.d[1], x20\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      "ldr d7, [x8, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      "ldr x20, [x8, #0x68]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      "ldr x24, [x12, #0x8]\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      "ldr d6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      "ldr x21, [x8, #0x78]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      "ldr x23, [x11, #0x8]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      "mov v6.d[1], x20\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      "ldr d7, [x8, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      "ldr x20, [x8, #0x88]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      "ldr x22, [x10, #0x8]\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      "ldr d6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      "ldr x21, [x8, #0x98]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      "mov v6.d[1], x20\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      "ldr d7, [x8, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      "ldr x20, [x8, #0xa8]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      "cmp x14, #0x20\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      "ldr d6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      "ldr x21, [x8, #0xb8]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      "mov v6.d[1], x20\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      "ldr d7, [x8, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      "ldr x20, [x8, #0xc8]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      "ldr d6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      "ldr x21, [x8, #0xd8]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      "mov v6.d[1], x20\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      "ldr d7, [x8, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      "ldr x20, [x8, #0xe8]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      "ldr d6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      "ldr x21, [x8, #0xf8]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      "mov v6.d[1], x20\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      "ldr d7, [x8, #0xf0]\n"
      "add x8, x8, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      "ldr x20, [x8, #0x8]\n"
      "mov v7.d[1], x21\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      "ldr d6, [x8, #0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr d1, [x12, #0]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      "ldr d2, [x11, #0]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      "ldr d3, [x10, #0]\n"
      "ldr d7, [x8, #0x10]\n"
      "mov v6.d[1], x20\n"
      "ldr x20, [x8, #0x18]\n"
      "mov v0.d[1], x25\n"
      "mov v1.d[1], x24\n"
      "mov v2.d[1], x23\n"
      "mov v3.d[1], x22\n"
      "mov v7.d[1], x20\n"
      "bge 123b\n"
      "124:"  // Height 4: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x8, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      "ldr q6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      "ldr q7, [x8, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      "ldr q6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      "ldr q7, [x8, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      "ldr q6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      "ldr q7, [x8, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      "ldr q6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      "ldr q7, [x8, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      "ldr q6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      "ldr q7, [x8, #0xf0]\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      "add x8, x8, #0x100\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      "125:"  // Height 4: Multiply loop: Main loop skip
      "cbz x14, 130f\n"
      "cmp x14, #0x4\n"
      "blt 127f\n"
      "126:"  // Height 4: Multiply loop: Odd block loop
      "ldr s0, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr s1, [x12], #0x4\n"
      "cmp x14, #0x4\n"
      "ldr s2, [x11], #0x4\n"
      "ldr s3, [x10], #0x4\n"
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      "bge 126b\n"
      "127:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x14, 130f\n"
      "tbz x14, #1, 128f\n"
      "ldr h0, [x13], #0x2\n"
      "ldr h1, [x12], #0x2\n"
      "ldr h2, [x11], #0x2\n"
      "ldr h3, [x10], #0x2\n"
      "tbz x14, #0, 129f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "ld1 { v1.b }[2], [x12]\n"
      "ld1 { v2.b }[2], [x11]\n"
      "ld1 { v3.b }[2], [x10]\n"
      "b 129f\n"
      "128:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0]\n"
      "ldr b1, [x12, #0]\n"
      "ldr b2, [x11, #0]\n"
      "ldr b3, [x10, #0]\n"
      "129:"  // Height 4: Multiply loop: Ragged operand read: Done
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      "130:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 120b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x7, #0x10\n"
      "prfm pstl1keep, [x16, #0]\n"
      "add x24, x16, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0]\n"
      "add x22, x23, x20, LSL #2\n"
      "prfm pstl1keep, [x23, #0]\n"
      "prfm pstl1keep, [x22, #0]\n"
      "bge 139f\n"
      "tbz x7, #3, 134f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "st1 { v9.4s }, [x16], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "tbz x7, #2, 132f\n"
      "st1 { v10.4s }, [x16], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "tbz x7, #1, 131f\n"
      "str d11, [x16], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "tbz x7, #0, 138f\n"
      "st1 { v11.s }[2], [x16]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "b 138f\n"
      "131:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x7, #0, 138f\n"
      "str s11, [x16, #0]\n"
      "str s15, [x24, #0]\n"
      "str s19, [x23, #0]\n"
      "str s23, [x22, #0]\n"
      "b 138f\n"
      "132:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x7, #1, 133f\n"
      "str d10, [x16], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "tbz x7, #0, 138f\n"
      "st1 { v10.s }[2], [x16]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "b 138f\n"
      "133:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x7, #0, 138f\n"
      "str s10, [x16, #0]\n"
      "str s14, [x24, #0]\n"
      "str s18, [x23, #0]\n"
      "str s22, [x22, #0]\n"
      "b 138f\n"
      "134:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x7, #2, 136f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "tbz x7, #1, 135f\n"
      "str d9, [x16], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "tbz x7, #0, 138f\n"
      "st1 { v9.s }[2], [x16]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "b 138f\n"
      "135:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x7, #0, 138f\n"
      "str s9, [x16, #0]\n"
      "str s13, [x24, #0]\n"
      "str s17, [x23, #0]\n"
      "str s21, [x22, #0]\n"
      "b 138f\n"
      "136:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x7, #1, 137f\n"
      "str d8, [x16], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "tbz x7, #0, 138f\n"
      "st1 { v8.s }[2], [x16]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "b 138f\n"
      "137:"  // Height 4: Partial direct writeback: partial_1_0
      "str s8, [x16, #0]\n"
      "str s12, [x24, #0]\n"
      "str s16, [x23, #0]\n"
      "str s20, [x22, #0]\n"
      "138:"  // Height 4: Partial direct writeback: Done
      "b 140f\n"
      "139:"  // Height 4: Full writeback
      "str q8, [x16, #0]\n"
      "str q9, [x16, #0x10]\n"
      "str q10, [x16, #0x20]\n"
      "str q11, [x16, #0x30]\n"
      "add x16, x16, #0x40\n"
      "str q12, [x24, #0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "str q20, [x22, #0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "140:"  // Height 4: Writeback done
      "subs x7, x7, #0x10\n"
      "bgt 107b\n"
      "b 212f\n"
      "141:"  // Height 5
      "ldr x7, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x8, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x17, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "142:"  // Height 5: Column loop
      "cbz x17, 143f\n"
      "ldr q8, [x17, #0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "add x17, x17, #0x40\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v27.16b, v11.16b\n"
      "b 154f\n"
      "143:"  // Height 5: no bias
      "tbz %x[flags], #0, 153f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x7, #0x10\n"
      "add x24, x16, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "bge 152f\n"
      "tbz x7, #3, 147f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "ld1 { v9.4s }, [x16], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "tbz x7, #2, 145f\n"
      "ld1 { v10.4s }, [x16], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "tbz x7, #1, 144f\n"
      "ldr d11, [x16], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d15, [x24], #0x8\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "tbz x7, #0, 151f\n"
      "ld1 { v11.s }[2], [x16]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "b 151f\n"
      "144:"  // Height 5: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x7, #0, 151f\n"
      "ldr s11, [x16, #0]\n"
      "ldr s15, [x24, #0]\n"
      "ldr s19, [x23, #0]\n"
      "ldr s23, [x22, #0]\n"
      "ldr s27, [x21, #0]\n"
      "b 151f\n"
      "145:"  // Height 5: Partial accumulate: partial_2_8
      "tbz x7, #1, 146f\n"
      "ldr d10, [x16], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d14, [x24], #0x8\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "tbz x7, #0, 151f\n"
      "ld1 { v10.s }[2], [x16]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "b 151f\n"
      "146:"  // Height 5: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x7, #0, 151f\n"
      "ldr s10, [x16, #0]\n"
      "ldr s14, [x24, #0]\n"
      "ldr s18, [x23, #0]\n"
      "ldr s22, [x22, #0]\n"
      "ldr s26, [x21, #0]\n"
      "b 151f\n"
      "147:"  // Height 5: Partial accumulate: partial_4_0
      "tbz x7, #2, 149f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "tbz x7, #1, 148f\n"
      "ldr d9, [x16], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d13, [x24], #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d25, [x21], #0x8\n"
      "tbz x7, #0, 151f\n"
      "ld1 { v9.s }[2], [x16]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v25.s }[2], [x21]\n"
      "b 151f\n"
      "148:"  // Height 5: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x7, #0, 151f\n"
      "ldr s9, [x16, #0]\n"
      "ldr s13, [x24, #0]\n"
      "ldr s17, [x23, #0]\n"
      "ldr s21, [x22, #0]\n"
      "ldr s25, [x21, #0]\n"
      "b 151f\n"
      "149:"  // Height 5: Partial accumulate: partial_2_0
      "tbz x7, #1, 150f\n"
      "ldr d8, [x16], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "ldr d24, [x21], #0x8\n"
      "tbz x7, #0, 151f\n"
      "ld1 { v8.s }[2], [x16]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "ld1 { v24.s }[2], [x21]\n"
      "b 151f\n"
      "150:"  // Height 5: Partial accumulate: partial_1_0
      "ldr s8, [x16, #0]\n"
      "mov x25, #0\n"
      "ldr s12, [x24, #0]\n"
      "ldr s16, [x23, #0]\n"
      "ldr s20, [x22, #0]\n"
      "ldr s24, [x21, #0]\n"
      "151:"  // Height 5: Partial accumulate: Done
      "sub x16, x16, x25\n"
      "b 154f\n"
      "152:"  // Height 5: full accumulate
      "ldr q8, [x16, #0]\n"
      "ldr q9, [x16, #0x10]\n"
      "ldr q10, [x16, #0x20]\n"
      "ldr q11, [x16, #0x30]\n"
      "ldr q12, [x24, #0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "ldr q20, [x22, #0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q24, [x21, #0]\n"
      "ldr q25, [x21, #0x10]\n"
      "ldr q26, [x21, #0x20]\n"
      "ldr q27, [x21, #0x30]\n"
      "b 154f\n"
      "153:"  // Height 5: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "movi v20.4s, #0\n"
      "movi v21.4s, #0\n"
      "movi v22.4s, #0\n"
      "movi v23.4s, #0\n"
      "movi v24.4s, #0\n"
      "movi v25.4s, #0\n"
      "movi v26.4s, #0\n"
      "movi v27.4s, #0\n"
      "154:"  // Height 5: setup done
      "mov x15, #0\n"
      "155:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "tbz %x[flags], #3, 156f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0]\n"
      "ldr x12, [x20, #0x8]\n"
      "ldr x11, [x20, #0x10]\n"
      "ldr x10, [x20, #0x18]\n"
      "ldr x9, [x20, #0x20]\n"
      "cbnz x15, 157f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "add x12, x12, x20\n"
      "add x11, x11, x20\n"
      "add x10, x10, x20\n"
      "add x9, x9, x20\n"
      "b 157f\n"
      "156:"  // Height 5: setup direct input
      "mov x13, %x[input_ptr]\n"
      "add x12, x13, x21\n"
      "add x11, x12, x21\n"
      "add x10, x11, x21\n"
      "add x9, x10, x21\n"
      "157:"  // Height 5: input setup done
      "cmp x14, #0x10\n"
      "blt 160f\n"
      "ldr q0, [x13, #0]\n"
      "cmp x14, #0x20\n"
      "ldr q1, [x12, #0]\n"
      "ldr q2, [x11, #0]\n"
      "ldr q3, [x10, #0]\n"
      "ldr q4, [x9, #0]\n"
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      "blt 159f\n"
      "158:"  // Height 5: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr x21, [x8, #0x28]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr x20, [x8, #0x38]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr d6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr x21, [x8, #0x48]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr d7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "ldr x26, [x13, #0x8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "ldr x20, [x8, #0x58]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      "ldr d6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "ldr x25, [x12, #0x8]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "ldr x24, [x11, #0x8]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      "ldr x21, [x8, #0x68]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      "ldr d7, [x8, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      "ldr x23, [x10, #0x8]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      "ldr x22, [x9, #0x8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      "ldr x20, [x8, #0x78]\n"
      ".inst 0x4fa4e0d8  // sdot v24.4s, v6.16b, v4.4b[1]\n"
      "ldr d6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      "cmp x14, #0x20\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      "ldr x21, [x8, #0x88]\n"
      ".inst 0x4fa4e0f9  // sdot v25.4s, v7.16b, v4.4b[1]\n"
      "ldr d7, [x8, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      "ldr x20, [x8, #0x98]\n"
      ".inst 0x4fa4e0da  // sdot v26.4s, v6.16b, v4.4b[1]\n"
      "ldr d6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      "ldr x21, [x8, #0xa8]\n"
      ".inst 0x4fa4e0fb  // sdot v27.4s, v7.16b, v4.4b[1]\n"
      "ldr d7, [x8, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      "ldr x20, [x8, #0xb8]\n"
      ".inst 0x4f84e8d8  // sdot v24.4s, v6.16b, v4.4b[2]\n"
      "ldr d6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      "ldr x21, [x8, #0xc8]\n"
      ".inst 0x4f84e8f9  // sdot v25.4s, v7.16b, v4.4b[2]\n"
      "ldr d7, [x8, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      "ldr x20, [x8, #0xd8]\n"
      ".inst 0x4f84e8da  // sdot v26.4s, v6.16b, v4.4b[2]\n"
      "ldr d6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      "ldr x21, [x8, #0xe8]\n"
      ".inst 0x4f84e8fb  // sdot v27.4s, v7.16b, v4.4b[2]\n"
      "ldr d7, [x8, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      "ldr x20, [x8, #0xf8]\n"
      ".inst 0x4fa4e8d8  // sdot v24.4s, v6.16b, v4.4b[3]\n"
      "ldr d6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8f9  // sdot v25.4s, v7.16b, v4.4b[3]\n"
      "ldr d7, [x8, #0xf0]\n"
      "add x8, x8, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      "ldr x21, [x8, #0x8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      "ldr x20, [x8, #0x18]\n"
      ".inst 0x4fa4e8da  // sdot v26.4s, v6.16b, v4.4b[3]\n"
      "ldr d6, [x8, #0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr d1, [x12, #0]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      "ldr d2, [x11, #0]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      "ldr d3, [x10, #0]\n"
      ".inst 0x4fa4e8fb  // sdot v27.4s, v7.16b, v4.4b[3]\n"
      "ldr d4, [x9, #0]\n"
      "ldr d7, [x8, #0x10]\n"
      "mov v6.d[1], x21\n"
      "mov v0.d[1], x26\n"
      "mov v1.d[1], x25\n"
      "mov v2.d[1], x24\n"
      "mov v3.d[1], x23\n"
      "mov v4.d[1], x22\n"
      "mov v7.d[1], x20\n"
      "bge 158b\n"
      "159:"  // Height 5: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      "ldr q6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      "ldr q7, [x8, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0d8  // sdot v24.4s, v6.16b, v4.4b[1]\n"
      "ldr q6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0f9  // sdot v25.4s, v7.16b, v4.4b[1]\n"
      "ldr q7, [x8, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0da  // sdot v26.4s, v6.16b, v4.4b[1]\n"
      "ldr q6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0fb  // sdot v27.4s, v7.16b, v4.4b[1]\n"
      "ldr q7, [x8, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8d8  // sdot v24.4s, v6.16b, v4.4b[2]\n"
      "ldr q6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8f9  // sdot v25.4s, v7.16b, v4.4b[2]\n"
      "ldr q7, [x8, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8da  // sdot v26.4s, v6.16b, v4.4b[2]\n"
      "ldr q6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8fb  // sdot v27.4s, v7.16b, v4.4b[2]\n"
      "ldr q7, [x8, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8d8  // sdot v24.4s, v6.16b, v4.4b[3]\n"
      "ldr q6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8f9  // sdot v25.4s, v7.16b, v4.4b[3]\n"
      "ldr q7, [x8, #0xf0]\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      "add x8, x8, #0x100\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8da  // sdot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8fb  // sdot v27.4s, v7.16b, v4.4b[3]\n"
      "160:"  // Height 5: Multiply loop: Main loop skip
      "cbz x14, 165f\n"
      "cmp x14, #0x4\n"
      "blt 162f\n"
      "161:"  // Height 5: Multiply loop: Odd block loop
      "ldr s0, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr s1, [x12], #0x4\n"
      "cmp x14, #0x4\n"
      "ldr s2, [x11], #0x4\n"
      "ldr s3, [x10], #0x4\n"
      "ldr s4, [x9], #0x4\n"
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      "bge 161b\n"
      "162:"  // Height 5: Multiply loop: Skip odd blocks
      "cbz x14, 165f\n"
      "tbz x14, #1, 163f\n"
      "ldr h0, [x13], #0x2\n"
      "ldr h1, [x12], #0x2\n"
      "ldr h2, [x11], #0x2\n"
      "ldr h3, [x10], #0x2\n"
      "ldr h4, [x9], #0x2\n"
      "tbz x14, #0, 164f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "ld1 { v1.b }[2], [x12]\n"
      "ld1 { v2.b }[2], [x11]\n"
      "ld1 { v3.b }[2], [x10]\n"
      "ld1 { v4.b }[2], [x9]\n"
      "b 164f\n"
      "163:"  // Height 5: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0]\n"
      "ldr b1, [x12, #0]\n"
      "ldr b2, [x11, #0]\n"
      "ldr b3, [x10, #0]\n"
      "ldr b4, [x9, #0]\n"
      "164:"  // Height 5: Multiply loop: Ragged operand read: Done
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      "165:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 155b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x7, #0x10\n"
      "prfm pstl1keep, [x16, #0]\n"
      "add x24, x16, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0]\n"
      "add x22, x23, x20, LSL #2\n"
      "prfm pstl1keep, [x23, #0]\n"
      "add x21, x22, x20, LSL #2\n"
      "prfm pstl1keep, [x22, #0]\n"
      "prfm pstl1keep, [x21, #0]\n"
      "bge 174f\n"
      "tbz x7, #3, 169f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "st1 { v9.4s }, [x16], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "st1 { v25.4s }, [x21], #0x10\n"
      "tbz x7, #2, 167f\n"
      "st1 { v10.4s }, [x16], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x21], #0x10\n"
      "tbz x7, #1, 166f\n"
      "str d11, [x16], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "str d27, [x21], #0x8\n"
      "tbz x7, #0, 173f\n"
      "st1 { v11.s }[2], [x16]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "st1 { v27.s }[2], [x21]\n"
      "b 173f\n"
      "166:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x7, #0, 173f\n"
      "str s11, [x16, #0]\n"
      "str s15, [x24, #0]\n"
      "str s19, [x23, #0]\n"
      "str s23, [x22, #0]\n"
      "str s27, [x21, #0]\n"
      "b 173f\n"
      "167:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x7, #1, 168f\n"
      "str d10, [x16], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "str d26, [x21], #0x8\n"
      "tbz x7, #0, 173f\n"
      "st1 { v10.s }[2], [x16]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "st1 { v26.s }[2], [x21]\n"
      "b 173f\n"
      "168:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x7, #0, 173f\n"
      "str s10, [x16, #0]\n"
      "str s14, [x24, #0]\n"
      "str s18, [x23, #0]\n"
      "str s22, [x22, #0]\n"
      "str s26, [x21, #0]\n"
      "b 173f\n"
      "169:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x7, #2, 171f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "tbz x7, #1, 170f\n"
      "str d9, [x16], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "str d25, [x21], #0x8\n"
      "tbz x7, #0, 173f\n"
      "st1 { v9.s }[2], [x16]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "st1 { v25.s }[2], [x21]\n"
      "b 173f\n"
      "170:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x7, #0, 173f\n"
      "str s9, [x16, #0]\n"
      "str s13, [x24, #0]\n"
      "str s17, [x23, #0]\n"
      "str s21, [x22, #0]\n"
      "str s25, [x21, #0]\n"
      "b 173f\n"
      "171:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x7, #1, 172f\n"
      "str d8, [x16], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "str d24, [x21], #0x8\n"
      "tbz x7, #0, 173f\n"
      "st1 { v8.s }[2], [x16]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "st1 { v24.s }[2], [x21]\n"
      "b 173f\n"
      "172:"  // Height 5: Partial direct writeback: partial_1_0
      "str s8, [x16, #0]\n"
      "str s12, [x24, #0]\n"
      "str s16, [x23, #0]\n"
      "str s20, [x22, #0]\n"
      "str s24, [x21, #0]\n"
      "173:"  // Height 5: Partial direct writeback: Done
      "b 175f\n"
      "174:"  // Height 5: Full writeback
      "str q8, [x16, #0]\n"
      "str q9, [x16, #0x10]\n"
      "str q10, [x16, #0x20]\n"
      "str q11, [x16, #0x30]\n"
      "add x16, x16, #0x40\n"
      "str q12, [x24, #0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "str q20, [x22, #0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "str q24, [x21, #0]\n"
      "str q25, [x21, #0x10]\n"
      "str q26, [x21, #0x20]\n"
      "str q27, [x21, #0x30]\n"
      "175:"  // Height 5: Writeback done
      "subs x7, x7, #0x10\n"
      "bgt 142b\n"
      "b 212f\n"
      "176:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x18\n"
      "ldr x16, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "ldr x7, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x8, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd x20, x21, x20, x16\n"
      "ldr x17, [%x[args_ptr], %[offsetof_bias]]\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "177:"  // Height 6: Column loop
      "cbz x17, 178f\n"
      "ldr q8, [x17, #0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "add x17, x17, #0x40\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v27.16b, v11.16b\n"
      "mov v28.16b, v8.16b\n"
      "mov v29.16b, v9.16b\n"
      "mov v30.16b, v10.16b\n"
      "mov v31.16b, v11.16b\n"
      "b 189f\n"
      "178:"  // Height 6: no bias
      "tbz %x[flags], #0, 188f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x7, #0x10\n"
      "add x24, x16, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "bge 187f\n"
      "tbz x7, #3, 182f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "ld1 { v28.4s }, [x20], #0x10\n"
      "ld1 { v9.4s }, [x16], #0x10\n"
      "ld1 { v13.4s }, [x24], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v25.4s }, [x21], #0x10\n"
      "ld1 { v29.4s }, [x20], #0x10\n"
      "tbz x7, #2, 180f\n"
      "ld1 { v10.4s }, [x16], #0x10\n"
      "ld1 { v14.4s }, [x24], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "ld1 { v30.4s }, [x20], #0x10\n"
      "tbz x7, #1, 179f\n"
      "ldr d11, [x16], #0x8\n"
      "mov x25, #0x38\n"
      "ldr d15, [x24], #0x8\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "ldr d31, [x20], #0x8\n"
      "tbz x7, #0, 186f\n"
      "ld1 { v11.s }[2], [x16]\n"
      "ld1 { v15.s }[2], [x24]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "ld1 { v31.s }[2], [x20]\n"
      "b 186f\n"
      "179:"  // Height 6: Partial accumulate: partial_1_12
      "mov x25, #0x30\n"
      "tbz x7, #0, 186f\n"
      "ldr s11, [x16, #0]\n"
      "ldr s15, [x24, #0]\n"
      "ldr s19, [x23, #0]\n"
      "ldr s23, [x22, #0]\n"
      "ldr s27, [x21, #0]\n"
      "ldr s31, [x20, #0]\n"
      "b 186f\n"
      "180:"  // Height 6: Partial accumulate: partial_2_8
      "tbz x7, #1, 181f\n"
      "ldr d10, [x16], #0x8\n"
      "mov x25, #0x28\n"
      "ldr d14, [x24], #0x8\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "ldr d30, [x20], #0x8\n"
      "tbz x7, #0, 186f\n"
      "ld1 { v10.s }[2], [x16]\n"
      "ld1 { v14.s }[2], [x24]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "ld1 { v30.s }[2], [x20]\n"
      "b 186f\n"
      "181:"  // Height 6: Partial accumulate: partial_1_8
      "mov x25, #0x20\n"
      "tbz x7, #0, 186f\n"
      "ldr s10, [x16, #0]\n"
      "ldr s14, [x24, #0]\n"
      "ldr s18, [x23, #0]\n"
      "ldr s22, [x22, #0]\n"
      "ldr s26, [x21, #0]\n"
      "ldr s30, [x20, #0]\n"
      "b 186f\n"
      "182:"  // Height 6: Partial accumulate: partial_4_0
      "tbz x7, #2, 184f\n"
      "ld1 { v8.4s }, [x16], #0x10\n"
      "ld1 { v12.4s }, [x24], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v24.4s }, [x21], #0x10\n"
      "ld1 { v28.4s }, [x20], #0x10\n"
      "tbz x7, #1, 183f\n"
      "ldr d9, [x16], #0x8\n"
      "mov x25, #0x18\n"
      "ldr d13, [x24], #0x8\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d25, [x21], #0x8\n"
      "ldr d29, [x20], #0x8\n"
      "tbz x7, #0, 186f\n"
      "ld1 { v9.s }[2], [x16]\n"
      "ld1 { v13.s }[2], [x24]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v25.s }[2], [x21]\n"
      "ld1 { v29.s }[2], [x20]\n"
      "b 186f\n"
      "183:"  // Height 6: Partial accumulate: partial_1_4
      "mov x25, #0x10\n"
      "tbz x7, #0, 186f\n"
      "ldr s9, [x16, #0]\n"
      "ldr s13, [x24, #0]\n"
      "ldr s17, [x23, #0]\n"
      "ldr s21, [x22, #0]\n"
      "ldr s25, [x21, #0]\n"
      "ldr s29, [x20, #0]\n"
      "b 186f\n"
      "184:"  // Height 6: Partial accumulate: partial_2_0
      "tbz x7, #1, 185f\n"
      "ldr d8, [x16], #0x8\n"
      "mov x25, #0x8\n"
      "ldr d12, [x24], #0x8\n"
      "ldr d16, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "ldr d24, [x21], #0x8\n"
      "ldr d28, [x20], #0x8\n"
      "tbz x7, #0, 186f\n"
      "ld1 { v8.s }[2], [x16]\n"
      "ld1 { v12.s }[2], [x24]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "ld1 { v24.s }[2], [x21]\n"
      "ld1 { v28.s }[2], [x20]\n"
      "b 186f\n"
      "185:"  // Height 6: Partial accumulate: partial_1_0
      "ldr s8, [x16, #0]\n"
      "mov x25, #0\n"
      "ldr s12, [x24, #0]\n"
      "ldr s16, [x23, #0]\n"
      "ldr s20, [x22, #0]\n"
      "ldr s24, [x21, #0]\n"
      "ldr s28, [x20, #0]\n"
      "186:"  // Height 6: Partial accumulate: Done
      "sub x16, x16, x25\n"
      "b 189f\n"
      "187:"  // Height 6: full accumulate
      "ldr q8, [x16, #0]\n"
      "ldr q9, [x16, #0x10]\n"
      "ldr q10, [x16, #0x20]\n"
      "ldr q11, [x16, #0x30]\n"
      "ldr q12, [x24, #0]\n"
      "ldr q13, [x24, #0x10]\n"
      "ldr q14, [x24, #0x20]\n"
      "ldr q15, [x24, #0x30]\n"
      "ldr q16, [x23, #0]\n"
      "ldr q17, [x23, #0x10]\n"
      "ldr q18, [x23, #0x20]\n"
      "ldr q19, [x23, #0x30]\n"
      "ldr q20, [x22, #0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q24, [x21, #0]\n"
      "ldr q25, [x21, #0x10]\n"
      "ldr q26, [x21, #0x20]\n"
      "ldr q27, [x21, #0x30]\n"
      "ldr q28, [x20, #0]\n"
      "ldr q29, [x20, #0x10]\n"
      "ldr q30, [x20, #0x20]\n"
      "ldr q31, [x20, #0x30]\n"
      "b 189f\n"
      "188:"  // Height 6: no accumulate
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "movi v20.4s, #0\n"
      "movi v21.4s, #0\n"
      "movi v22.4s, #0\n"
      "movi v23.4s, #0\n"
      "movi v24.4s, #0\n"
      "movi v25.4s, #0\n"
      "movi v26.4s, #0\n"
      "movi v27.4s, #0\n"
      "movi v28.4s, #0\n"
      "movi v29.4s, #0\n"
      "movi v30.4s, #0\n"
      "movi v31.4s, #0\n"
      "189:"  // Height 6: setup done
      "mov x15, #0\n"
      "190:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w14, [x20, x15, LSL #0x2]\n"
      "tbz %x[flags], #3, 191f\n"
      "ldr x20, [%x[input_ptr], x15, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x13, [x20, #0]\n"
      "ldr x12, [x20, #0x8]\n"
      "ldr x11, [x20, #0x10]\n"
      "ldr x10, [x20, #0x18]\n"
      "ldr x9, [x20, #0x20]\n"
      "ldr x28, [x20, #0x28]\n"
      "cbnz x15, 192f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x13, x13, x20\n"
      "add x12, x12, x20\n"
      "add x11, x11, x20\n"
      "add x10, x10, x20\n"
      "add x9, x9, x20\n"
      "add x28, x28, x20\n"
      "b 192f\n"
      "191:"  // Height 6: setup direct input
      "mov x13, %x[input_ptr]\n"
      "add x12, x13, x21\n"
      "add x11, x12, x21\n"
      "add x10, x11, x21\n"
      "add x9, x10, x21\n"
      "add x28, x9, x21\n"
      "192:"  // Height 6: input setup done
      "cmp x14, #0x10\n"
      "blt 195f\n"
      "ldr q0, [x13, #0]\n"
      "cmp x14, #0x20\n"
      "ldr q1, [x12, #0]\n"
      "ldr q2, [x11, #0]\n"
      "ldr q3, [x10, #0]\n"
      "ldr q4, [x9, #0]\n"
      "ldr q5, [x28, #0]\n"
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      "blt 194f\n"
      "193:"  // Height 6: Multiply loop: Main loop head
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "ldr x21, [x8, #0x28]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "ldr x20, [x8, #0x38]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x4f85e0dc  // sdot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr d6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "ldr x21, [x8, #0x48]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "add x28, x28, #0x10\n"
      ".inst 0x4f85e0fd  // sdot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr d7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "ldr x27, [x13, #0x8]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "ldr x26, [x12, #0x8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      "ldr x20, [x8, #0x58]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      "ldr x25, [x11, #0x8]\n"
      ".inst 0x4f85e0de  // sdot v30.4s, v6.16b, v5.4b[0]\n"
      "ldr d6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      "ldr x24, [x10, #0x8]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      "ldr x23, [x9, #0x8]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      "ldr x21, [x8, #0x68]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      "ldr x22, [x28, #0x8]\n"
      ".inst 0x4f85e0ff  // sdot v31.4s, v7.16b, v5.4b[0]\n"
      "ldr d7, [x8, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      "cmp x14, #0x20\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      "ldr x20, [x8, #0x78]\n"
      ".inst 0x4fa4e0d8  // sdot v24.4s, v6.16b, v4.4b[1]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x4fa5e0dc  // sdot v28.4s, v6.16b, v5.4b[1]\n"
      "ldr d6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      "ldr x21, [x8, #0x88]\n"
      ".inst 0x4fa4e0f9  // sdot v25.4s, v7.16b, v4.4b[1]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      ".inst 0x4fa5e0fd  // sdot v29.4s, v7.16b, v5.4b[1]\n"
      "ldr d7, [x8, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      "ldr x20, [x8, #0x98]\n"
      ".inst 0x4fa4e0da  // sdot v26.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0de  // sdot v30.4s, v6.16b, v5.4b[1]\n"
      "ldr d6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      "ldr x21, [x8, #0xa8]\n"
      ".inst 0x4fa4e0fb  // sdot v27.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0ff  // sdot v31.4s, v7.16b, v5.4b[1]\n"
      "ldr d7, [x8, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      "ldr x20, [x8, #0xb8]\n"
      ".inst 0x4f84e8d8  // sdot v24.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8dc  // sdot v28.4s, v6.16b, v5.4b[2]\n"
      "ldr d6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      "ldr x21, [x8, #0xc8]\n"
      ".inst 0x4f84e8f9  // sdot v25.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8fd  // sdot v29.4s, v7.16b, v5.4b[2]\n"
      "ldr d7, [x8, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      "ldr x20, [x8, #0xd8]\n"
      ".inst 0x4f84e8da  // sdot v26.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8de  // sdot v30.4s, v6.16b, v5.4b[2]\n"
      "ldr d6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      "ldr x21, [x8, #0xe8]\n"
      ".inst 0x4f84e8fb  // sdot v27.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8ff  // sdot v31.4s, v7.16b, v5.4b[2]\n"
      "ldr d7, [x8, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      "ldr x20, [x8, #0xf8]\n"
      ".inst 0x4fa4e8d8  // sdot v24.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8dc  // sdot v28.4s, v6.16b, v5.4b[3]\n"
      "ldr d6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      "mov v6.d[1], x21\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8f9  // sdot v25.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8fd  // sdot v29.4s, v7.16b, v5.4b[3]\n"
      "ldr d7, [x8, #0xf0]\n"
      "add x8, x8, #0x100\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      "ldr x21, [x8, #0x8]\n"
      "mov v7.d[1], x20\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      "ldr x20, [x8, #0x18]\n"
      ".inst 0x4fa4e8da  // sdot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8de  // sdot v30.4s, v6.16b, v5.4b[3]\n"
      "ldr d6, [x8, #0]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      "ldr d0, [x13, #0]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      "ldr d1, [x12, #0]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      "ldr d2, [x11, #0]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      "ldr d3, [x10, #0]\n"
      ".inst 0x4fa4e8fb  // sdot v27.4s, v7.16b, v4.4b[3]\n"
      "ldr d4, [x9, #0]\n"
      ".inst 0x4fa5e8ff  // sdot v31.4s, v7.16b, v5.4b[3]\n"
      "ldr d5, [x28, #0]\n"
      "ldr d7, [x8, #0x10]\n"
      "mov v6.d[1], x21\n"
      "mov v0.d[1], x27\n"
      "mov v1.d[1], x26\n"
      "mov v2.d[1], x25\n"
      "mov v3.d[1], x24\n"
      "mov v4.d[1], x23\n"
      "mov v5.d[1], x22\n"
      "mov v7.d[1], x20\n"
      "bge 193b\n"
      "194:"  // Height 6: Multiply loop: Single iteration only
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      "add x13, x13, #0x10\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      "add x12, x12, #0x10\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      "add x11, x11, #0x10\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      "add x10, x10, #0x10\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f85e0dc  // sdot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      "add x28, x28, #0x10\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      "sub x14, x14, #0x10\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x13, #0x80]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      "prfm pldl1keep, [x12, #0x80]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      ".inst 0x4f85e0fd  // sdot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0de  // sdot v30.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x8, #0x40]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0ff  // sdot v31.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x8, #0x50]\n"
      ".inst 0x4fa0e0c8  // sdot v8.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0cc  // sdot v12.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d0  // sdot v16.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d4  // sdot v20.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0d8  // sdot v24.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0dc  // sdot v28.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x8, #0x60]\n"
      ".inst 0x4fa0e0e9  // sdot v9.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ed  // sdot v13.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f1  // sdot v17.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f5  // sdot v21.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0f9  // sdot v25.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0fd  // sdot v29.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x8, #0x70]\n"
      ".inst 0x4fa0e0ca  // sdot v10.4s, v6.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ce  // sdot v14.4s, v6.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0d2  // sdot v18.4s, v6.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0d6  // sdot v22.4s, v6.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0da  // sdot v26.4s, v6.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0de  // sdot v30.4s, v6.16b, v5.4b[1]\n"
      "ldr q6, [x8, #0x80]\n"
      ".inst 0x4fa0e0eb  // sdot v11.4s, v7.16b, v0.4b[1]\n"
      ".inst 0x4fa1e0ef  // sdot v15.4s, v7.16b, v1.4b[1]\n"
      ".inst 0x4fa2e0f3  // sdot v19.4s, v7.16b, v2.4b[1]\n"
      ".inst 0x4fa3e0f7  // sdot v23.4s, v7.16b, v3.4b[1]\n"
      ".inst 0x4fa4e0fb  // sdot v27.4s, v7.16b, v4.4b[1]\n"
      ".inst 0x4fa5e0ff  // sdot v31.4s, v7.16b, v5.4b[1]\n"
      "ldr q7, [x8, #0x90]\n"
      ".inst 0x4f80e8c8  // sdot v8.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8cc  // sdot v12.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d0  // sdot v16.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d4  // sdot v20.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8d8  // sdot v24.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8dc  // sdot v28.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x8, #0xa0]\n"
      ".inst 0x4f80e8e9  // sdot v9.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ed  // sdot v13.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f1  // sdot v17.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f5  // sdot v21.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8f9  // sdot v25.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8fd  // sdot v29.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x8, #0xb0]\n"
      ".inst 0x4f80e8ca  // sdot v10.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ce  // sdot v14.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d2  // sdot v18.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8d6  // sdot v22.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f84e8da  // sdot v26.4s, v6.16b, v4.4b[2]\n"
      ".inst 0x4f85e8de  // sdot v30.4s, v6.16b, v5.4b[2]\n"
      "ldr q6, [x8, #0xc0]\n"
      ".inst 0x4f80e8eb  // sdot v11.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8ef  // sdot v15.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8f3  // sdot v19.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8f7  // sdot v23.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f84e8fb  // sdot v27.4s, v7.16b, v4.4b[2]\n"
      ".inst 0x4f85e8ff  // sdot v31.4s, v7.16b, v5.4b[2]\n"
      "ldr q7, [x8, #0xd0]\n"
      ".inst 0x4fa0e8c8  // sdot v8.4s, v6.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8cc  // sdot v12.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d0  // sdot v16.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d4  // sdot v20.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8d8  // sdot v24.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8dc  // sdot v28.4s, v6.16b, v5.4b[3]\n"
      "ldr q6, [x8, #0xe0]\n"
      ".inst 0x4fa0e8e9  // sdot v9.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ed  // sdot v13.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f1  // sdot v17.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f5  // sdot v21.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8f9  // sdot v25.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8fd  // sdot v29.4s, v7.16b, v5.4b[3]\n"
      "ldr q7, [x8, #0xf0]\n"
      ".inst 0x4fa0e8ca  // sdot v10.4s, v6.16b, v0.4b[3]\n"
      "add x8, x8, #0x100\n"
      ".inst 0x4fa1e8ce  // sdot v14.4s, v6.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8d2  // sdot v18.4s, v6.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8d6  // sdot v22.4s, v6.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8da  // sdot v26.4s, v6.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8de  // sdot v30.4s, v6.16b, v5.4b[3]\n"
      ".inst 0x4fa0e8eb  // sdot v11.4s, v7.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8ef  // sdot v15.4s, v7.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8f3  // sdot v19.4s, v7.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8f7  // sdot v23.4s, v7.16b, v3.4b[3]\n"
      ".inst 0x4fa4e8fb  // sdot v27.4s, v7.16b, v4.4b[3]\n"
      ".inst 0x4fa5e8ff  // sdot v31.4s, v7.16b, v5.4b[3]\n"
      "195:"  // Height 6: Multiply loop: Main loop skip
      "cbz x14, 200f\n"
      "cmp x14, #0x4\n"
      "blt 197f\n"
      "196:"  // Height 6: Multiply loop: Odd block loop
      "ldr s0, [x13], #0x4\n"
      "sub x14, x14, #0x4\n"
      "ldr s1, [x12], #0x4\n"
      "cmp x14, #0x4\n"
      "ldr s2, [x11], #0x4\n"
      "ldr s3, [x10], #0x4\n"
      "ldr s4, [x9], #0x4\n"
      "ldr s5, [x28], #0x4\n"
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0dc  // sdot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0fd  // sdot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0de  // sdot v30.4s, v6.16b, v5.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0ff  // sdot v31.4s, v7.16b, v5.4b[0]\n"
      "bge 196b\n"
      "197:"  // Height 6: Multiply loop: Skip odd blocks
      "cbz x14, 200f\n"
      "tbz x14, #1, 198f\n"
      "ldr h0, [x13], #0x2\n"
      "ldr h1, [x12], #0x2\n"
      "ldr h2, [x11], #0x2\n"
      "ldr h3, [x10], #0x2\n"
      "ldr h4, [x9], #0x2\n"
      "ldr h5, [x28], #0x2\n"
      "tbz x14, #0, 199f\n"
      "ld1 { v0.b }[2], [x13]\n"
      "ld1 { v1.b }[2], [x12]\n"
      "ld1 { v2.b }[2], [x11]\n"
      "ld1 { v3.b }[2], [x10]\n"
      "ld1 { v4.b }[2], [x9]\n"
      "ld1 { v5.b }[2], [x28]\n"
      "b 199f\n"
      "198:"  // Height 6: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x13, #0]\n"
      "ldr b1, [x12, #0]\n"
      "ldr b2, [x11, #0]\n"
      "ldr b3, [x10, #0]\n"
      "ldr b4, [x9, #0]\n"
      "ldr b5, [x28, #0]\n"
      "199:"  // Height 6: Multiply loop: Ragged operand read: Done
      "ldr q6, [x8, #0]\n"
      "ldr q7, [x8, #0x10]\n"
      ".inst 0x4f80e0c8  // sdot v8.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0cc  // sdot v12.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d0  // sdot v16.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d4  // sdot v20.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0d8  // sdot v24.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0dc  // sdot v28.4s, v6.16b, v5.4b[0]\n"
      "ldr q6, [x8, #0x20]\n"
      ".inst 0x4f80e0e9  // sdot v9.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ed  // sdot v13.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f1  // sdot v17.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f5  // sdot v21.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0f9  // sdot v25.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0fd  // sdot v29.4s, v7.16b, v5.4b[0]\n"
      "ldr q7, [x8, #0x30]\n"
      ".inst 0x4f80e0ca  // sdot v10.4s, v6.16b, v0.4b[0]\n"
      "add x8, x8, #0x40\n"
      ".inst 0x4f81e0ce  // sdot v14.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d2  // sdot v18.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0d6  // sdot v22.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f84e0da  // sdot v26.4s, v6.16b, v4.4b[0]\n"
      ".inst 0x4f85e0de  // sdot v30.4s, v6.16b, v5.4b[0]\n"
      ".inst 0x4f80e0eb  // sdot v11.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0ef  // sdot v15.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f3  // sdot v19.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0f7  // sdot v23.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f84e0fb  // sdot v27.4s, v7.16b, v4.4b[0]\n"
      ".inst 0x4f85e0ff  // sdot v31.4s, v7.16b, v5.4b[0]\n"
      "200:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x15, x15, #0x1\n"
      "cmp x15, x20\n"
      "bne 190b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x7, #0x10\n"
      "prfm pstl1keep, [x16, #0]\n"
      "add x24, x16, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0]\n"
      "add x22, x23, x20, LSL #2\n"
      "prfm pstl1keep, [x23, #0]\n"
      "add x21, x22, x20, LSL #2\n"
      "prfm pstl1keep, [x22, #0]\n"
      "add x20, x21, x20, LSL #2\n"
      "prfm pstl1keep, [x21, #0]\n"
      "prfm pstl1keep, [x20, #0]\n"
      "bge 209f\n"
      "tbz x7, #3, 204f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "st1 { v9.4s }, [x16], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v13.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "st1 { v25.4s }, [x21], #0x10\n"
      "st1 { v28.4s }, [x20], #0x10\n"
      "st1 { v29.4s }, [x20], #0x10\n"
      "tbz x7, #2, 202f\n"
      "st1 { v10.4s }, [x16], #0x10\n"
      "st1 { v14.4s }, [x24], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x21], #0x10\n"
      "st1 { v30.4s }, [x20], #0x10\n"
      "tbz x7, #1, 201f\n"
      "str d11, [x16], #0x8\n"
      "str d15, [x24], #0x8\n"
      "str d19, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "str d27, [x21], #0x8\n"
      "str d31, [x20], #0x8\n"
      "tbz x7, #0, 208f\n"
      "st1 { v11.s }[2], [x16]\n"
      "st1 { v15.s }[2], [x24]\n"
      "st1 { v19.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "st1 { v27.s }[2], [x21]\n"
      "st1 { v31.s }[2], [x20]\n"
      "b 208f\n"
      "201:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x7, #0, 208f\n"
      "str s11, [x16, #0]\n"
      "str s15, [x24, #0]\n"
      "str s19, [x23, #0]\n"
      "str s23, [x22, #0]\n"
      "str s27, [x21, #0]\n"
      "str s31, [x20, #0]\n"
      "b 208f\n"
      "202:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x7, #1, 203f\n"
      "str d10, [x16], #0x8\n"
      "str d14, [x24], #0x8\n"
      "str d18, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "str d26, [x21], #0x8\n"
      "str d30, [x20], #0x8\n"
      "tbz x7, #0, 208f\n"
      "st1 { v10.s }[2], [x16]\n"
      "st1 { v14.s }[2], [x24]\n"
      "st1 { v18.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "st1 { v26.s }[2], [x21]\n"
      "st1 { v30.s }[2], [x20]\n"
      "b 208f\n"
      "203:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x7, #0, 208f\n"
      "str s10, [x16, #0]\n"
      "str s14, [x24, #0]\n"
      "str s18, [x23, #0]\n"
      "str s22, [x22, #0]\n"
      "str s26, [x21, #0]\n"
      "str s30, [x20, #0]\n"
      "b 208f\n"
      "204:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x7, #2, 206f\n"
      "st1 { v8.4s }, [x16], #0x10\n"
      "st1 { v12.4s }, [x24], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v24.4s }, [x21], #0x10\n"
      "st1 { v28.4s }, [x20], #0x10\n"
      "tbz x7, #1, 205f\n"
      "str d9, [x16], #0x8\n"
      "str d13, [x24], #0x8\n"
      "str d17, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "str d25, [x21], #0x8\n"
      "str d29, [x20], #0x8\n"
      "tbz x7, #0, 208f\n"
      "st1 { v9.s }[2], [x16]\n"
      "st1 { v13.s }[2], [x24]\n"
      "st1 { v17.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "st1 { v25.s }[2], [x21]\n"
      "st1 { v29.s }[2], [x20]\n"
      "b 208f\n"
      "205:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x7, #0, 208f\n"
      "str s9, [x16, #0]\n"
      "str s13, [x24, #0]\n"
      "str s17, [x23, #0]\n"
      "str s21, [x22, #0]\n"
      "str s25, [x21, #0]\n"
      "str s29, [x20, #0]\n"
      "b 208f\n"
      "206:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x7, #1, 207f\n"
      "str d8, [x16], #0x8\n"
      "str d12, [x24], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "str d24, [x21], #0x8\n"
      "str d28, [x20], #0x8\n"
      "tbz x7, #0, 208f\n"
      "st1 { v8.s }[2], [x16]\n"
      "st1 { v12.s }[2], [x24]\n"
      "st1 { v16.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "st1 { v24.s }[2], [x21]\n"
      "st1 { v28.s }[2], [x20]\n"
      "b 208f\n"
      "207:"  // Height 6: Partial direct writeback: partial_1_0
      "str s8, [x16, #0]\n"
      "str s12, [x24, #0]\n"
      "str s16, [x23, #0]\n"
      "str s20, [x22, #0]\n"
      "str s24, [x21, #0]\n"
      "str s28, [x20, #0]\n"
      "208:"  // Height 6: Partial direct writeback: Done
      "b 210f\n"
      "209:"  // Height 6: Full writeback
      "str q8, [x16, #0]\n"
      "str q9, [x16, #0x10]\n"
      "str q10, [x16, #0x20]\n"
      "str q11, [x16, #0x30]\n"
      "add x16, x16, #0x40\n"
      "str q12, [x24, #0]\n"
      "str q13, [x24, #0x10]\n"
      "str q14, [x24, #0x20]\n"
      "str q15, [x24, #0x30]\n"
      "str q16, [x23, #0]\n"
      "str q17, [x23, #0x10]\n"
      "str q18, [x23, #0x20]\n"
      "str q19, [x23, #0x30]\n"
      "str q20, [x22, #0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "str q24, [x21, #0]\n"
      "str q25, [x21, #0x10]\n"
      "str q26, [x21, #0x20]\n"
      "str q27, [x21, #0x30]\n"
      "str q28, [x20, #0]\n"
      "str q29, [x20, #0x10]\n"
      "str q30, [x20, #0x20]\n"
      "str q31, [x20, #0x30]\n"
      "210:"  // Height 6: Writeback done
      "subs x7, x7, #0x10\n"
      "bgt 177b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 212f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 211f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "211:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "212:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm

#endif // __aarch64__

