/*
 * Copyright (c) 2021,2023-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifdef __aarch64__

#include "arm_gemm/arm_gemm.hpp"
#include "arm_common/internal/utils.hpp"
#include <cassert>
#include <cstdint>

namespace arm_gemm {

void a64_hybrid_s8qs_mmla_6x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int col_base
)
{
    struct KernelArgs {
        const int32_t *multiplier_ptr = {};
        const int32_t *shift_ptr = {};
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    if (qp->per_channel_requant) {
        flags |= 0x10;
        ka.multiplier_ptr=qp->per_channel_muls + col_base;
        ka.shift_ptr=qp->per_channel_right_shifts + col_base;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 141f\n"
      "cmp %x[M], #0x4\n"
      "bgt 113f\n"
      "beq 85f\n"
      "cmp %x[M], #0x2\n"
      "bgt 57f\n"
      "beq 29f\n"
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "mov x28, #0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "cbnz x28, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x27, #0x10\n"
      "blt 9f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q7, [x10, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q6, [x10, #0x10]\n"
      "blt 8f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0]\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      "ldr q6, [x10, #0x10]\n"
      "bge 7b\n"
      "8:"  // Height 1: Multiply loop: Single iteration only
      "add x26, x26, #0x10\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      "9:"  // Height 1: Multiply loop: Main loop skip
      "cbz x27, 16f\n"
      "cmp x27, #0x8\n"
      "blt 11f\n"
      "10:"  // Height 1: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr q6, [x10, #0]\n"
      "sub x27, x27, #0x8\n"
      "ldr q7, [x10, #0x10]\n"
      "cmp x27, #0x8\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      ".inst 0x4e86a408  // smmla v8.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4e87a40c  // smmla v12.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x30]\n"
      ".inst 0x4e86a409  // smmla v9.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4e87a40d  // smmla v13.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4e86a40a  // smmla v10.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4e87a40e  // smmla v14.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e86a40b  // smmla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87a40f  // smmla v15.4s, v0.16b, v7.16b\n"
      "bge 10b\n"
      "11:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x27, 16f\n"
      "tbz x27, #2, 13f\n"
      "ldr s1, [x26], #0x4\n"
      "tbz x27, #1, 12f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "tbz x27, #0, 15f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "b 15f\n"
      "12:"  // Height 1: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 15f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "b 15f\n"
      "13:"  // Height 1: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 14f\n"
      "ldr h1, [x26], #0x2\n"
      "tbz x27, #0, 15f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "b 15f\n"
      "14:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "15:"  // Height 1: Multiply loop: Ragged operand read: Done
      "ldr q7, [x10, #0]\n"
      "ldr q6, [x10, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      "16:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 4b\n"
      "ldr q0, [x12, #0]\n"
      "ldr q1, [x12, #0x10]\n"
      "uzp1 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v9.2d, v9.2d, v13.2d\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "uzp1 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "add x12, x12, #0x40\n"
      "mov v15.16b, v8.16b\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v15.4s, v15.4s, v0.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "tbz %x[flags], #4, 17f\n"
      "ldr q0, [x13, #0]\n"
      "ldr q4, [x14, #0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 18f\n"
      "17:"  // Height 1: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "18:"  // Height 1: parameters loaded
      "sqdmulh v15.4s, v15.4s, v4.4s\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "add x21, %x[qp], %[c_offset]\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v4.4s }, [x21]\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "ld1r { v6.4s }, [x20]\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "ld1r { v5.4s }, [x20]\n"
      "srshl v15.4s, v15.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "smin v15.4s, v15.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smax v15.4s, v15.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "uzp1 v15.8h, v15.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v15.16b, v15.16b, v9.16b\n"
      "bge 27f\n"
      "tbz x11, #3, 22f\n"
      "str d15, [x9], #0x8\n"
      "tbz x11, #2, 20f\n"
      "st1 { v15.s }[2], [x9], #0x4\n"
      "tbz x11, #1, 19f\n"
      "st1 { v15.h }[6], [x9], #0x2\n"
      "tbz x11, #0, 26f\n"
      "st1 { v15.b }[14], [x9]\n"
      "b 26f\n"
      "19:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x11, #0, 26f\n"
      "st1 { v15.b }[12], [x9]\n"
      "b 26f\n"
      "20:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x11, #1, 21f\n"
      "st1 { v15.h }[4], [x9], #0x2\n"
      "tbz x11, #0, 26f\n"
      "st1 { v15.b }[10], [x9]\n"
      "b 26f\n"
      "21:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x11, #0, 26f\n"
      "st1 { v15.b }[8], [x9]\n"
      "b 26f\n"
      "22:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x11, #2, 24f\n"
      "str s15, [x9], #0x4\n"
      "tbz x11, #1, 23f\n"
      "st1 { v15.h }[2], [x9], #0x2\n"
      "tbz x11, #0, 26f\n"
      "st1 { v15.b }[6], [x9]\n"
      "b 26f\n"
      "23:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x11, #0, 26f\n"
      "st1 { v15.b }[4], [x9]\n"
      "b 26f\n"
      "24:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x11, #1, 25f\n"
      "str h15, [x9], #0x2\n"
      "tbz x11, #0, 26f\n"
      "st1 { v15.b }[2], [x9]\n"
      "b 26f\n"
      "25:"  // Height 1: Partial direct writeback: partial_1_0
      "str b15, [x9, #0]\n"
      "26:"  // Height 1: Partial direct writeback: Done
      "b 28f\n"
      "27:"  // Height 1: Full writeback
      "str q15, [x9, #0]\n"
      "add x9, x9, #0x10\n"
      "28:"  // Height 1: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 2b\n"
      "b 170f\n"
      "29:"  // Height 2
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "30:"  // Height 2: Column loop
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "mov x28, #0\n"
      "32:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 33f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 34f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "b 34f\n"
      "33:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "34:"  // Height 2: input setup done
      "cmp x27, #0x10\n"
      "blt 37f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q2, [x25, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q7, [x10, #0]\n"
      "ldr q6, [x10, #0x10]\n"
      "blt 36f\n"
      "35:"  // Height 2: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "ldr q2, [x25, #0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0]\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      "ldr q6, [x10, #0x10]\n"
      "bge 35b\n"
      "36:"  // Height 2: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      "37:"  // Height 2: Multiply loop: Main loop skip
      "cbz x27, 44f\n"
      "cmp x27, #0x8\n"
      "blt 39f\n"
      "38:"  // Height 2: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d2, [x25], #0x8\n"
      "sub x27, x27, #0x8\n"
      "ldr q6, [x10, #0]\n"
      "ldr q7, [x10, #0x10]\n"
      "cmp x27, #0x8\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      ".inst 0x4e86a408  // smmla v8.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4e87a40c  // smmla v12.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x30]\n"
      ".inst 0x4e86a409  // smmla v9.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4e87a40d  // smmla v13.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4e86a40a  // smmla v10.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4e87a40e  // smmla v14.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e86a40b  // smmla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87a40f  // smmla v15.4s, v0.16b, v7.16b\n"
      "bge 38b\n"
      "39:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x27, 44f\n"
      "tbz x27, #2, 41f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "tbz x27, #1, 40f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "tbz x27, #0, 43f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "b 43f\n"
      "40:"  // Height 2: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 43f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "b 43f\n"
      "41:"  // Height 2: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 42f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "tbz x27, #0, 43f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "b 43f\n"
      "42:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "ldr b2, [x25, #0]\n"
      "43:"  // Height 2: Multiply loop: Ragged operand read: Done
      "ldr q7, [x10, #0]\n"
      "ldr q6, [x10, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      "44:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 32b\n"
      "ldr q0, [x12, #0]\n"
      "ldr q1, [x12, #0x10]\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "add x12, x12, #0x40\n"
      "mov v15.16b, v7.16b\n"
      "add v12.4s, v12.4s, v1.4s\n"
      "add x27, x9, x20\n"
      "prfm pstl1keep, [x27, #0]\n"
      "add v13.4s, v13.4s, v2.4s\n"
      "add v8.4s, v8.4s, v0.4s\n"
      "add v15.4s, v15.4s, v0.4s\n"
      "add v14.4s, v14.4s, v3.4s\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "tbz %x[flags], #4, 45f\n"
      "ldr q0, [x13, #0]\n"
      "ldr q4, [x14, #0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 46f\n"
      "45:"  // Height 2: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "46:"  // Height 2: parameters loaded
      "sqdmulh v15.4s, v15.4s, v4.4s\n"
      "sqdmulh v12.4s, v12.4s, v5.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      "sqdmulh v13.4s, v13.4s, v6.4s\n"
      "sqdmulh v14.4s, v14.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "sqdmulh v8.4s, v8.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "ld1r { v6.4s }, [x21]\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "srshl v15.4s, v15.4s, v0.4s\n"
      "srshl v12.4s, v12.4s, v1.4s\n"
      "srshl v13.4s, v13.4s, v2.4s\n"
      "srshl v14.4s, v14.4s, v3.4s\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "add v12.4s, v12.4s, v4.4s\n"
      "add v13.4s, v13.4s, v4.4s\n"
      "add v14.4s, v14.4s, v4.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "smin v15.4s, v15.4s, v6.4s\n"
      "smin v12.4s, v12.4s, v6.4s\n"
      "smin v13.4s, v13.4s, v6.4s\n"
      "smin v14.4s, v14.4s, v6.4s\n"
      "smin v8.4s, v8.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smax v15.4s, v15.4s, v5.4s\n"
      "smax v12.4s, v12.4s, v5.4s\n"
      "smax v13.4s, v13.4s, v5.4s\n"
      "smax v14.4s, v14.4s, v5.4s\n"
      "smax v8.4s, v8.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "uzp1 v15.8h, v15.8h, v12.8h\n"
      "uzp1 v12.8h, v13.8h, v14.8h\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v15.16b, v15.16b, v12.16b\n"
      "uzp1 v8.16b, v8.16b, v9.16b\n"
      "bge 55f\n"
      "tbz x11, #3, 50f\n"
      "str d15, [x9], #0x8\n"
      "str d8, [x27], #0x8\n"
      "tbz x11, #2, 48f\n"
      "st1 { v15.s }[2], [x9], #0x4\n"
      "st1 { v8.s }[2], [x27], #0x4\n"
      "tbz x11, #1, 47f\n"
      "st1 { v15.h }[6], [x9], #0x2\n"
      "st1 { v8.h }[6], [x27], #0x2\n"
      "tbz x11, #0, 54f\n"
      "st1 { v15.b }[14], [x9]\n"
      "st1 { v8.b }[14], [x27]\n"
      "b 54f\n"
      "47:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x11, #0, 54f\n"
      "st1 { v15.b }[12], [x9]\n"
      "st1 { v8.b }[12], [x27]\n"
      "b 54f\n"
      "48:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x11, #1, 49f\n"
      "st1 { v15.h }[4], [x9], #0x2\n"
      "st1 { v8.h }[4], [x27], #0x2\n"
      "tbz x11, #0, 54f\n"
      "st1 { v15.b }[10], [x9]\n"
      "st1 { v8.b }[10], [x27]\n"
      "b 54f\n"
      "49:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x11, #0, 54f\n"
      "st1 { v15.b }[8], [x9]\n"
      "st1 { v8.b }[8], [x27]\n"
      "b 54f\n"
      "50:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x11, #2, 52f\n"
      "str s15, [x9], #0x4\n"
      "str s8, [x27], #0x4\n"
      "tbz x11, #1, 51f\n"
      "st1 { v15.h }[2], [x9], #0x2\n"
      "st1 { v8.h }[2], [x27], #0x2\n"
      "tbz x11, #0, 54f\n"
      "st1 { v15.b }[6], [x9]\n"
      "st1 { v8.b }[6], [x27]\n"
      "b 54f\n"
      "51:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x11, #0, 54f\n"
      "st1 { v15.b }[4], [x9]\n"
      "st1 { v8.b }[4], [x27]\n"
      "b 54f\n"
      "52:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x11, #1, 53f\n"
      "str h15, [x9], #0x2\n"
      "str h8, [x27], #0x2\n"
      "tbz x11, #0, 54f\n"
      "st1 { v15.b }[2], [x9]\n"
      "st1 { v8.b }[2], [x27]\n"
      "b 54f\n"
      "53:"  // Height 2: Partial direct writeback: partial_1_0
      "str b15, [x9, #0]\n"
      "str b8, [x27, #0]\n"
      "54:"  // Height 2: Partial direct writeback: Done
      "b 56f\n"
      "55:"  // Height 2: Full writeback
      "str q15, [x9, #0]\n"
      "add x9, x9, #0x10\n"
      "str q8, [x27, #0]\n"
      "56:"  // Height 2: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 30b\n"
      "b 170f\n"
      "57:"  // Height 3
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "58:"  // Height 3: Column loop
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "movi v20.4s, #0\n"
      "movi v21.4s, #0\n"
      "movi v22.4s, #0\n"
      "movi v23.4s, #0\n"
      "mov x28, #0\n"
      "60:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 61f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 62f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "b 62f\n"
      "61:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "62:"  // Height 3: input setup done
      "cmp x27, #0x10\n"
      "blt 65f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q2, [x25, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0]\n"
      "ldr q7, [x10, #0]\n"
      "ldr q6, [x10, #0x10]\n"
      "blt 64f\n"
      "63:"  // Height 3: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      "ldr q2, [x25, #0]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a474  // smmla v20.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a471  // smmla v17.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a475  // smmla v21.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a472  // smmla v18.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a476  // smmla v22.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a473  // smmla v19.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0]\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      "ldr q3, [x24, #0]\n"
      "ldr q6, [x10, #0x10]\n"
      "bge 63b\n"
      "64:"  // Height 3: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a474  // smmla v20.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a471  // smmla v17.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a475  // smmla v21.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a472  // smmla v18.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a476  // smmla v22.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a473  // smmla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      "65:"  // Height 3: Multiply loop: Main loop skip
      "cbz x27, 72f\n"
      "cmp x27, #0x8\n"
      "blt 67f\n"
      "66:"  // Height 3: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d2, [x25], #0x8\n"
      "sub x27, x27, #0x8\n"
      "ldr d3, [x24], #0x8\n"
      "ldr q6, [x10, #0]\n"
      "cmp x27, #0x8\n"
      "ldr q7, [x10, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x4e86a408  // smmla v8.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a450  // smmla v16.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4e87a40c  // smmla v12.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a454  // smmla v20.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x30]\n"
      ".inst 0x4e86a409  // smmla v9.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a451  // smmla v17.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4e87a40d  // smmla v13.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a455  // smmla v21.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4e86a40a  // smmla v10.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a452  // smmla v18.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4e87a40e  // smmla v14.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a456  // smmla v22.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e86a40b  // smmla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a453  // smmla v19.4s, v2.16b, v6.16b\n"
      ".inst 0x4e87a40f  // smmla v15.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a457  // smmla v23.4s, v2.16b, v7.16b\n"
      "bge 66b\n"
      "67:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x27, 72f\n"
      "tbz x27, #2, 69f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "tbz x27, #1, 68f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "tbz x27, #0, 71f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "b 71f\n"
      "68:"  // Height 3: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 71f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "b 71f\n"
      "69:"  // Height 3: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 70f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "tbz x27, #0, 71f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "b 71f\n"
      "70:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "ldr b2, [x25, #0]\n"
      "ldr b3, [x24, #0]\n"
      "71:"  // Height 3: Multiply loop: Ragged operand read: Done
      "ldr q7, [x10, #0]\n"
      "ldr q6, [x10, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      "72:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 60b\n"
      "ldr q0, [x12, #0]\n"
      "ldr q1, [x12, #0x10]\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "add x12, x12, #0x40\n"
      "uzp1 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v19.2d, v19.2d, v23.2d\n"
      "add x27, x9, x20\n"
      "mov v23.16b, v7.16b\n"
      "add x26, x27, x20\n"
      "prfm pstl1keep, [x27, #0]\n"
      "add v12.4s, v12.4s, v1.4s\n"
      "prfm pstl1keep, [x26, #0]\n"
      "add v13.4s, v13.4s, v2.4s\n"
      "add v14.4s, v14.4s, v3.4s\n"
      "add v23.4s, v23.4s, v0.4s\n"
      "add v8.4s, v8.4s, v0.4s\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "tbz %x[flags], #4, 73f\n"
      "ldr q0, [x13, #0]\n"
      "ldr q4, [x14, #0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 74f\n"
      "73:"  // Height 3: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "74:"  // Height 3: parameters loaded
      "sqdmulh v23.4s, v23.4s, v4.4s\n"
      "sqdmulh v12.4s, v12.4s, v5.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      "sqdmulh v13.4s, v13.4s, v6.4s\n"
      "sqdmulh v14.4s, v14.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "sqdmulh v8.4s, v8.4s, v4.4s\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "sqdmulh v17.4s, v17.4s, v5.4s\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v18.4s, v18.4s, v6.4s\n"
      "ld1r { v6.4s }, [x21]\n"
      "sqdmulh v19.4s, v19.4s, v7.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v12.4s, v12.4s, v1.4s\n"
      "srshl v13.4s, v13.4s, v2.4s\n"
      "srshl v14.4s, v14.4s, v3.4s\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "add v12.4s, v12.4s, v4.4s\n"
      "add v13.4s, v13.4s, v4.4s\n"
      "add v14.4s, v14.4s, v4.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smin v12.4s, v12.4s, v6.4s\n"
      "smin v13.4s, v13.4s, v6.4s\n"
      "smin v14.4s, v14.4s, v6.4s\n"
      "smin v8.4s, v8.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "smax v12.4s, v12.4s, v5.4s\n"
      "smax v13.4s, v13.4s, v5.4s\n"
      "smax v14.4s, v14.4s, v5.4s\n"
      "smax v8.4s, v8.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "uzp1 v23.8h, v23.8h, v12.8h\n"
      "uzp1 v12.8h, v13.8h, v14.8h\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v23.16b, v23.16b, v12.16b\n"
      "uzp1 v8.16b, v8.16b, v9.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "bge 83f\n"
      "tbz x11, #3, 78f\n"
      "str d23, [x9], #0x8\n"
      "str d8, [x27], #0x8\n"
      "str d16, [x26], #0x8\n"
      "tbz x11, #2, 76f\n"
      "st1 { v23.s }[2], [x9], #0x4\n"
      "st1 { v8.s }[2], [x27], #0x4\n"
      "st1 { v16.s }[2], [x26], #0x4\n"
      "tbz x11, #1, 75f\n"
      "st1 { v23.h }[6], [x9], #0x2\n"
      "st1 { v8.h }[6], [x27], #0x2\n"
      "st1 { v16.h }[6], [x26], #0x2\n"
      "tbz x11, #0, 82f\n"
      "st1 { v23.b }[14], [x9]\n"
      "st1 { v8.b }[14], [x27]\n"
      "st1 { v16.b }[14], [x26]\n"
      "b 82f\n"
      "75:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x11, #0, 82f\n"
      "st1 { v23.b }[12], [x9]\n"
      "st1 { v8.b }[12], [x27]\n"
      "st1 { v16.b }[12], [x26]\n"
      "b 82f\n"
      "76:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x11, #1, 77f\n"
      "st1 { v23.h }[4], [x9], #0x2\n"
      "st1 { v8.h }[4], [x27], #0x2\n"
      "st1 { v16.h }[4], [x26], #0x2\n"
      "tbz x11, #0, 82f\n"
      "st1 { v23.b }[10], [x9]\n"
      "st1 { v8.b }[10], [x27]\n"
      "st1 { v16.b }[10], [x26]\n"
      "b 82f\n"
      "77:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x11, #0, 82f\n"
      "st1 { v23.b }[8], [x9]\n"
      "st1 { v8.b }[8], [x27]\n"
      "st1 { v16.b }[8], [x26]\n"
      "b 82f\n"
      "78:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x11, #2, 80f\n"
      "str s23, [x9], #0x4\n"
      "str s8, [x27], #0x4\n"
      "str s16, [x26], #0x4\n"
      "tbz x11, #1, 79f\n"
      "st1 { v23.h }[2], [x9], #0x2\n"
      "st1 { v8.h }[2], [x27], #0x2\n"
      "st1 { v16.h }[2], [x26], #0x2\n"
      "tbz x11, #0, 82f\n"
      "st1 { v23.b }[6], [x9]\n"
      "st1 { v8.b }[6], [x27]\n"
      "st1 { v16.b }[6], [x26]\n"
      "b 82f\n"
      "79:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x11, #0, 82f\n"
      "st1 { v23.b }[4], [x9]\n"
      "st1 { v8.b }[4], [x27]\n"
      "st1 { v16.b }[4], [x26]\n"
      "b 82f\n"
      "80:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x11, #1, 81f\n"
      "str h23, [x9], #0x2\n"
      "str h8, [x27], #0x2\n"
      "str h16, [x26], #0x2\n"
      "tbz x11, #0, 82f\n"
      "st1 { v23.b }[2], [x9]\n"
      "st1 { v8.b }[2], [x27]\n"
      "st1 { v16.b }[2], [x26]\n"
      "b 82f\n"
      "81:"  // Height 3: Partial direct writeback: partial_1_0
      "str b23, [x9, #0]\n"
      "str b8, [x27, #0]\n"
      "str b16, [x26, #0]\n"
      "82:"  // Height 3: Partial direct writeback: Done
      "b 84f\n"
      "83:"  // Height 3: Full writeback
      "str q23, [x9, #0]\n"
      "add x9, x9, #0x10\n"
      "str q8, [x27, #0]\n"
      "str q16, [x26, #0]\n"
      "84:"  // Height 3: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 58b\n"
      "b 170f\n"
      "85:"  // Height 4
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "86:"  // Height 4: Column loop
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "movi v20.4s, #0\n"
      "movi v21.4s, #0\n"
      "movi v22.4s, #0\n"
      "movi v23.4s, #0\n"
      "mov x28, #0\n"
      "88:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 89f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 90f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 90f\n"
      "89:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "90:"  // Height 4: input setup done
      "cmp x27, #0x10\n"
      "blt 93f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q2, [x25, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0]\n"
      "ldr q4, [x23, #0]\n"
      "ldr q7, [x10, #0]\n"
      "ldr q6, [x10, #0x10]\n"
      "blt 92f\n"
      "91:"  // Height 4: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "ldr q4, [x23, #0]\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      "ldr q2, [x25, #0]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a474  // smmla v20.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a471  // smmla v17.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a475  // smmla v21.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a472  // smmla v18.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a476  // smmla v22.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a473  // smmla v19.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0]\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      "ldr q3, [x24, #0]\n"
      "ldr q6, [x10, #0x10]\n"
      "bge 91b\n"
      "92:"  // Height 4: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "sub x27, x27, #0x10\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a474  // smmla v20.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a471  // smmla v17.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a475  // smmla v21.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a472  // smmla v18.4s, v3.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a476  // smmla v22.4s, v3.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a473  // smmla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      "93:"  // Height 4: Multiply loop: Main loop skip
      "cbz x27, 100f\n"
      "cmp x27, #0x8\n"
      "blt 95f\n"
      "94:"  // Height 4: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d2, [x25], #0x8\n"
      "sub x27, x27, #0x8\n"
      "ldr d3, [x24], #0x8\n"
      "ldr d4, [x23], #0x8\n"
      "cmp x27, #0x8\n"
      "ldr q6, [x10, #0]\n"
      "ldr q7, [x10, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x4e86a408  // smmla v8.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a450  // smmla v16.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4e87a40c  // smmla v12.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a454  // smmla v20.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x30]\n"
      ".inst 0x4e86a409  // smmla v9.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a451  // smmla v17.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4e87a40d  // smmla v13.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a455  // smmla v21.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4e86a40a  // smmla v10.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a452  // smmla v18.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4e87a40e  // smmla v14.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a456  // smmla v22.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e86a40b  // smmla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a453  // smmla v19.4s, v2.16b, v6.16b\n"
      ".inst 0x4e87a40f  // smmla v15.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a457  // smmla v23.4s, v2.16b, v7.16b\n"
      "bge 94b\n"
      "95:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x27, 100f\n"
      "tbz x27, #2, 97f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "tbz x27, #1, 96f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "tbz x27, #0, 99f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "b 99f\n"
      "96:"  // Height 4: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 99f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "b 99f\n"
      "97:"  // Height 4: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 98f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "tbz x27, #0, 99f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "b 99f\n"
      "98:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "ldr b2, [x25, #0]\n"
      "ldr b3, [x24, #0]\n"
      "ldr b4, [x23, #0]\n"
      "99:"  // Height 4: Multiply loop: Ragged operand read: Done
      "ldr q7, [x10, #0]\n"
      "ldr q6, [x10, #0x10]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      "100:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 88b\n"
      "ldr q0, [x12, #0]\n"
      "ldr q1, [x12, #0x10]\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "add x12, x12, #0x40\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "add x27, x9, x20\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "add x26, x27, x20\n"
      "prfm pstl1keep, [x27, #0]\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "add x25, x26, x20\n"
      "prfm pstl1keep, [x26, #0]\n"
      "mov v23.16b, v7.16b\n"
      "prfm pstl1keep, [x25, #0]\n"
      "add v12.4s, v12.4s, v1.4s\n"
      "add v13.4s, v13.4s, v2.4s\n"
      "add v14.4s, v14.4s, v3.4s\n"
      "add v23.4s, v23.4s, v0.4s\n"
      "add v8.4s, v8.4s, v0.4s\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "add v15.4s, v15.4s, v0.4s\n"
      "add v20.4s, v20.4s, v1.4s\n"
      "add v21.4s, v21.4s, v2.4s\n"
      "add v22.4s, v22.4s, v3.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "tbz %x[flags], #4, 101f\n"
      "ldr q0, [x13, #0]\n"
      "ldr q4, [x14, #0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 102f\n"
      "101:"  // Height 4: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "102:"  // Height 4: parameters loaded
      "sqdmulh v23.4s, v23.4s, v4.4s\n"
      "sqdmulh v12.4s, v12.4s, v5.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      "sqdmulh v13.4s, v13.4s, v6.4s\n"
      "sqdmulh v14.4s, v14.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "sqdmulh v8.4s, v8.4s, v4.4s\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "sqdmulh v15.4s, v15.4s, v4.4s\n"
      "sqdmulh v20.4s, v20.4s, v5.4s\n"
      "sqdmulh v21.4s, v21.4s, v6.4s\n"
      "sqdmulh v22.4s, v22.4s, v7.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "sqdmulh v17.4s, v17.4s, v5.4s\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v18.4s, v18.4s, v6.4s\n"
      "ld1r { v6.4s }, [x21]\n"
      "sqdmulh v19.4s, v19.4s, v7.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v12.4s, v12.4s, v1.4s\n"
      "srshl v13.4s, v13.4s, v2.4s\n"
      "srshl v14.4s, v14.4s, v3.4s\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "srshl v15.4s, v15.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v1.4s\n"
      "srshl v21.4s, v21.4s, v2.4s\n"
      "srshl v22.4s, v22.4s, v3.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "add v12.4s, v12.4s, v4.4s\n"
      "add v13.4s, v13.4s, v4.4s\n"
      "add v14.4s, v14.4s, v4.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smin v12.4s, v12.4s, v6.4s\n"
      "smin v13.4s, v13.4s, v6.4s\n"
      "smin v14.4s, v14.4s, v6.4s\n"
      "smin v8.4s, v8.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smin v15.4s, v15.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "smax v12.4s, v12.4s, v5.4s\n"
      "smax v13.4s, v13.4s, v5.4s\n"
      "smax v14.4s, v14.4s, v5.4s\n"
      "smax v8.4s, v8.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "smax v15.4s, v15.4s, v5.4s\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "uzp1 v23.8h, v23.8h, v12.8h\n"
      "uzp1 v12.8h, v13.8h, v14.8h\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v15.8h, v15.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v23.16b, v23.16b, v12.16b\n"
      "uzp1 v8.16b, v8.16b, v9.16b\n"
      "uzp1 v15.16b, v15.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "bge 111f\n"
      "tbz x11, #3, 106f\n"
      "str d23, [x9], #0x8\n"
      "str d8, [x27], #0x8\n"
      "str d15, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "tbz x11, #2, 104f\n"
      "st1 { v23.s }[2], [x9], #0x4\n"
      "st1 { v8.s }[2], [x27], #0x4\n"
      "st1 { v15.s }[2], [x26], #0x4\n"
      "st1 { v16.s }[2], [x25], #0x4\n"
      "tbz x11, #1, 103f\n"
      "st1 { v23.h }[6], [x9], #0x2\n"
      "st1 { v8.h }[6], [x27], #0x2\n"
      "st1 { v15.h }[6], [x26], #0x2\n"
      "st1 { v16.h }[6], [x25], #0x2\n"
      "tbz x11, #0, 110f\n"
      "st1 { v23.b }[14], [x9]\n"
      "st1 { v8.b }[14], [x27]\n"
      "st1 { v15.b }[14], [x26]\n"
      "st1 { v16.b }[14], [x25]\n"
      "b 110f\n"
      "103:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x11, #0, 110f\n"
      "st1 { v23.b }[12], [x9]\n"
      "st1 { v8.b }[12], [x27]\n"
      "st1 { v15.b }[12], [x26]\n"
      "st1 { v16.b }[12], [x25]\n"
      "b 110f\n"
      "104:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x11, #1, 105f\n"
      "st1 { v23.h }[4], [x9], #0x2\n"
      "st1 { v8.h }[4], [x27], #0x2\n"
      "st1 { v15.h }[4], [x26], #0x2\n"
      "st1 { v16.h }[4], [x25], #0x2\n"
      "tbz x11, #0, 110f\n"
      "st1 { v23.b }[10], [x9]\n"
      "st1 { v8.b }[10], [x27]\n"
      "st1 { v15.b }[10], [x26]\n"
      "st1 { v16.b }[10], [x25]\n"
      "b 110f\n"
      "105:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x11, #0, 110f\n"
      "st1 { v23.b }[8], [x9]\n"
      "st1 { v8.b }[8], [x27]\n"
      "st1 { v15.b }[8], [x26]\n"
      "st1 { v16.b }[8], [x25]\n"
      "b 110f\n"
      "106:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x11, #2, 108f\n"
      "str s23, [x9], #0x4\n"
      "str s8, [x27], #0x4\n"
      "str s15, [x26], #0x4\n"
      "str s16, [x25], #0x4\n"
      "tbz x11, #1, 107f\n"
      "st1 { v23.h }[2], [x9], #0x2\n"
      "st1 { v8.h }[2], [x27], #0x2\n"
      "st1 { v15.h }[2], [x26], #0x2\n"
      "st1 { v16.h }[2], [x25], #0x2\n"
      "tbz x11, #0, 110f\n"
      "st1 { v23.b }[6], [x9]\n"
      "st1 { v8.b }[6], [x27]\n"
      "st1 { v15.b }[6], [x26]\n"
      "st1 { v16.b }[6], [x25]\n"
      "b 110f\n"
      "107:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x11, #0, 110f\n"
      "st1 { v23.b }[4], [x9]\n"
      "st1 { v8.b }[4], [x27]\n"
      "st1 { v15.b }[4], [x26]\n"
      "st1 { v16.b }[4], [x25]\n"
      "b 110f\n"
      "108:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x11, #1, 109f\n"
      "str h23, [x9], #0x2\n"
      "str h8, [x27], #0x2\n"
      "str h15, [x26], #0x2\n"
      "str h16, [x25], #0x2\n"
      "tbz x11, #0, 110f\n"
      "st1 { v23.b }[2], [x9]\n"
      "st1 { v8.b }[2], [x27]\n"
      "st1 { v15.b }[2], [x26]\n"
      "st1 { v16.b }[2], [x25]\n"
      "b 110f\n"
      "109:"  // Height 4: Partial direct writeback: partial_1_0
      "str b23, [x9, #0]\n"
      "str b8, [x27, #0]\n"
      "str b15, [x26, #0]\n"
      "str b16, [x25, #0]\n"
      "110:"  // Height 4: Partial direct writeback: Done
      "b 112f\n"
      "111:"  // Height 4: Full writeback
      "str q23, [x9, #0]\n"
      "add x9, x9, #0x10\n"
      "str q8, [x27, #0]\n"
      "str q15, [x26, #0]\n"
      "str q16, [x25, #0]\n"
      "112:"  // Height 4: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 86b\n"
      "b 170f\n"
      "113:"  // Height 5
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "mov x12, %x[col_bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "114:"  // Height 5: Column loop
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "movi v20.4s, #0\n"
      "movi v21.4s, #0\n"
      "movi v22.4s, #0\n"
      "movi v23.4s, #0\n"
      "movi v24.4s, #0\n"
      "movi v25.4s, #0\n"
      "movi v26.4s, #0\n"
      "movi v27.4s, #0\n"
      "movi v28.4s, #0\n"
      "movi v29.4s, #0\n"
      "movi v30.4s, #0\n"
      "movi v31.4s, #0\n"
      "mov x28, #0\n"
      "116:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 117f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 118f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 118f\n"
      "117:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "118:"  // Height 5: input setup done
      "cmp x27, #0x10\n"
      "blt 121f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q2, [x25, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0]\n"
      "ldr q4, [x23, #0]\n"
      "ldr q5, [x22, #0]\n"
      "ldr q7, [x10, #0]\n"
      "blt 120f\n"
      "119:"  // Height 5: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x27, x27, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "add x22, x22, #0x10\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      "ldr q2, [x25, #0]\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      "ldr q4, [x23, #0]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a474  // smmla v20.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bc  // smmla v28.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a471  // smmla v17.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b9  // smmla v25.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a475  // smmla v21.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bd  // smmla v29.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a472  // smmla v18.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4ba  // smmla v26.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a476  // smmla v22.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4be  // smmla v30.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a473  // smmla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4bb  // smmla v27.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0]\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      "ldr q3, [x24, #0]\n"
      ".inst 0x4e86a4bf  // smmla v31.4s, v5.16b, v6.16b\n"
      "ldr q5, [x22, #0]\n"
      "bge 119b\n"
      "120:"  // Height 5: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x26, x26, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a474  // smmla v20.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bc  // smmla v28.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a471  // smmla v17.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b9  // smmla v25.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a475  // smmla v21.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bd  // smmla v29.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a472  // smmla v18.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4ba  // smmla v26.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a476  // smmla v22.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4be  // smmla v30.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a473  // smmla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4bb  // smmla v27.4s, v5.16b, v7.16b\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bf  // smmla v31.4s, v5.16b, v6.16b\n"
      "121:"  // Height 5: Multiply loop: Main loop skip
      "cbz x27, 128f\n"
      "cmp x27, #0x8\n"
      "blt 123f\n"
      "122:"  // Height 5: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d2, [x25], #0x8\n"
      "sub x27, x27, #0x8\n"
      "ldr d3, [x24], #0x8\n"
      "ldr d4, [x23], #0x8\n"
      "cmp x27, #0x8\n"
      "ldr d5, [x22], #0x8\n"
      "ldr q6, [x10, #0]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn1 v4.2d, v5.2d, v7.2d\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4e86a408  // smmla v8.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a450  // smmla v16.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a498  // smmla v24.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4e87a40c  // smmla v12.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a454  // smmla v20.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49c  // smmla v28.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x30]\n"
      ".inst 0x4e86a409  // smmla v9.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a451  // smmla v17.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a499  // smmla v25.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4e87a40d  // smmla v13.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a455  // smmla v21.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49d  // smmla v29.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4e86a40a  // smmla v10.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a452  // smmla v18.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49a  // smmla v26.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4e87a40e  // smmla v14.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a456  // smmla v22.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49e  // smmla v30.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e86a40b  // smmla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a453  // smmla v19.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49b  // smmla v27.4s, v4.16b, v6.16b\n"
      ".inst 0x4e87a40f  // smmla v15.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a457  // smmla v23.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49f  // smmla v31.4s, v4.16b, v7.16b\n"
      "bge 122b\n"
      "123:"  // Height 5: Multiply loop: Skip odd blocks
      "cbz x27, 128f\n"
      "tbz x27, #2, 125f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "ldr s5, [x22], #0x4\n"
      "tbz x27, #1, 124f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "ld1 { v5.h }[2], [x22], #0x2\n"
      "tbz x27, #0, 127f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "ld1 { v5.b }[6], [x22]\n"
      "b 127f\n"
      "124:"  // Height 5: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 127f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "ld1 { v5.b }[4], [x22]\n"
      "b 127f\n"
      "125:"  // Height 5: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 126f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "ldr h5, [x22], #0x2\n"
      "tbz x27, #0, 127f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "ld1 { v5.b }[2], [x22]\n"
      "b 127f\n"
      "126:"  // Height 5: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "ldr b2, [x25, #0]\n"
      "ldr b3, [x24, #0]\n"
      "ldr b4, [x23, #0]\n"
      "ldr b5, [x22, #0]\n"
      "127:"  // Height 5: Multiply loop: Ragged operand read: Done
      "ldr q7, [x10, #0]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "128:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 116b\n"
      "ldr q0, [x12, #0]\n"
      "ldr q1, [x12, #0x10]\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "add x12, x12, #0x40\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "add x27, x9, x20\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "add x26, x27, x20\n"
      "prfm pstl1keep, [x27, #0]\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "add x25, x26, x20\n"
      "prfm pstl1keep, [x26, #0]\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "add x24, x25, x20\n"
      "prfm pstl1keep, [x25, #0]\n"
      "uzp1 v24.2d, v24.2d, v28.2d\n"
      "uzp1 v25.2d, v25.2d, v29.2d\n"
      "prfm pstl1keep, [x24, #0]\n"
      "uzp1 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v27.2d, v27.2d, v31.2d\n"
      "mov v31.16b, v7.16b\n"
      "add v12.4s, v12.4s, v1.4s\n"
      "add v13.4s, v13.4s, v2.4s\n"
      "add v14.4s, v14.4s, v3.4s\n"
      "add v31.4s, v31.4s, v0.4s\n"
      "add v8.4s, v8.4s, v0.4s\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "add v15.4s, v15.4s, v0.4s\n"
      "add v20.4s, v20.4s, v1.4s\n"
      "add v21.4s, v21.4s, v2.4s\n"
      "add v22.4s, v22.4s, v3.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "add v25.4s, v25.4s, v1.4s\n"
      "add v26.4s, v26.4s, v2.4s\n"
      "add v27.4s, v27.4s, v3.4s\n"
      "tbz %x[flags], #4, 129f\n"
      "ldr q0, [x13, #0]\n"
      "ldr q4, [x14, #0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 130f\n"
      "129:"  // Height 5: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "130:"  // Height 5: parameters loaded
      "sqdmulh v31.4s, v31.4s, v4.4s\n"
      "sqdmulh v12.4s, v12.4s, v5.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      "sqdmulh v13.4s, v13.4s, v6.4s\n"
      "sqdmulh v14.4s, v14.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "sqdmulh v8.4s, v8.4s, v4.4s\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "sqdmulh v15.4s, v15.4s, v4.4s\n"
      "sqdmulh v20.4s, v20.4s, v5.4s\n"
      "sqdmulh v21.4s, v21.4s, v6.4s\n"
      "sqdmulh v22.4s, v22.4s, v7.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v5.4s\n"
      "sqdmulh v18.4s, v18.4s, v6.4s\n"
      "sqdmulh v19.4s, v19.4s, v7.4s\n"
      "sqdmulh v24.4s, v24.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "sqdmulh v25.4s, v25.4s, v5.4s\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v26.4s, v26.4s, v6.4s\n"
      "ld1r { v6.4s }, [x21]\n"
      "sqdmulh v27.4s, v27.4s, v7.4s\n"
      "srshl v31.4s, v31.4s, v0.4s\n"
      "srshl v12.4s, v12.4s, v1.4s\n"
      "srshl v13.4s, v13.4s, v2.4s\n"
      "srshl v14.4s, v14.4s, v3.4s\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "srshl v15.4s, v15.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v1.4s\n"
      "srshl v21.4s, v21.4s, v2.4s\n"
      "srshl v22.4s, v22.4s, v3.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v1.4s\n"
      "srshl v26.4s, v26.4s, v2.4s\n"
      "srshl v27.4s, v27.4s, v3.4s\n"
      "add v31.4s, v31.4s, v4.4s\n"
      "add v12.4s, v12.4s, v4.4s\n"
      "add v13.4s, v13.4s, v4.4s\n"
      "add v14.4s, v14.4s, v4.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "add v24.4s, v24.4s, v4.4s\n"
      "add v25.4s, v25.4s, v4.4s\n"
      "add v26.4s, v26.4s, v4.4s\n"
      "add v27.4s, v27.4s, v4.4s\n"
      "smin v31.4s, v31.4s, v6.4s\n"
      "smin v12.4s, v12.4s, v6.4s\n"
      "smin v13.4s, v13.4s, v6.4s\n"
      "smin v14.4s, v14.4s, v6.4s\n"
      "smin v8.4s, v8.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smin v15.4s, v15.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smin v24.4s, v24.4s, v6.4s\n"
      "smin v25.4s, v25.4s, v6.4s\n"
      "smin v26.4s, v26.4s, v6.4s\n"
      "smin v27.4s, v27.4s, v6.4s\n"
      "smax v31.4s, v31.4s, v5.4s\n"
      "smax v12.4s, v12.4s, v5.4s\n"
      "smax v13.4s, v13.4s, v5.4s\n"
      "smax v14.4s, v14.4s, v5.4s\n"
      "smax v8.4s, v8.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "smax v15.4s, v15.4s, v5.4s\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "smax v24.4s, v24.4s, v5.4s\n"
      "smax v25.4s, v25.4s, v5.4s\n"
      "smax v26.4s, v26.4s, v5.4s\n"
      "smax v27.4s, v27.4s, v5.4s\n"
      "uzp1 v31.8h, v31.8h, v12.8h\n"
      "uzp1 v12.8h, v13.8h, v14.8h\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v15.8h, v15.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v25.8h, v26.8h, v27.8h\n"
      "uzp1 v31.16b, v31.16b, v12.16b\n"
      "uzp1 v8.16b, v8.16b, v9.16b\n"
      "uzp1 v15.16b, v15.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "uzp1 v24.16b, v24.16b, v25.16b\n"
      "bge 139f\n"
      "tbz x11, #3, 134f\n"
      "str d31, [x9], #0x8\n"
      "str d8, [x27], #0x8\n"
      "str d15, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "str d24, [x24], #0x8\n"
      "tbz x11, #2, 132f\n"
      "st1 { v31.s }[2], [x9], #0x4\n"
      "st1 { v8.s }[2], [x27], #0x4\n"
      "st1 { v15.s }[2], [x26], #0x4\n"
      "st1 { v16.s }[2], [x25], #0x4\n"
      "st1 { v24.s }[2], [x24], #0x4\n"
      "tbz x11, #1, 131f\n"
      "st1 { v31.h }[6], [x9], #0x2\n"
      "st1 { v8.h }[6], [x27], #0x2\n"
      "st1 { v15.h }[6], [x26], #0x2\n"
      "st1 { v16.h }[6], [x25], #0x2\n"
      "st1 { v24.h }[6], [x24], #0x2\n"
      "tbz x11, #0, 138f\n"
      "st1 { v31.b }[14], [x9]\n"
      "st1 { v8.b }[14], [x27]\n"
      "st1 { v15.b }[14], [x26]\n"
      "st1 { v16.b }[14], [x25]\n"
      "st1 { v24.b }[14], [x24]\n"
      "b 138f\n"
      "131:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x11, #0, 138f\n"
      "st1 { v31.b }[12], [x9]\n"
      "st1 { v8.b }[12], [x27]\n"
      "st1 { v15.b }[12], [x26]\n"
      "st1 { v16.b }[12], [x25]\n"
      "st1 { v24.b }[12], [x24]\n"
      "b 138f\n"
      "132:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x11, #1, 133f\n"
      "st1 { v31.h }[4], [x9], #0x2\n"
      "st1 { v8.h }[4], [x27], #0x2\n"
      "st1 { v15.h }[4], [x26], #0x2\n"
      "st1 { v16.h }[4], [x25], #0x2\n"
      "st1 { v24.h }[4], [x24], #0x2\n"
      "tbz x11, #0, 138f\n"
      "st1 { v31.b }[10], [x9]\n"
      "st1 { v8.b }[10], [x27]\n"
      "st1 { v15.b }[10], [x26]\n"
      "st1 { v16.b }[10], [x25]\n"
      "st1 { v24.b }[10], [x24]\n"
      "b 138f\n"
      "133:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x11, #0, 138f\n"
      "st1 { v31.b }[8], [x9]\n"
      "st1 { v8.b }[8], [x27]\n"
      "st1 { v15.b }[8], [x26]\n"
      "st1 { v16.b }[8], [x25]\n"
      "st1 { v24.b }[8], [x24]\n"
      "b 138f\n"
      "134:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x11, #2, 136f\n"
      "str s31, [x9], #0x4\n"
      "str s8, [x27], #0x4\n"
      "str s15, [x26], #0x4\n"
      "str s16, [x25], #0x4\n"
      "str s24, [x24], #0x4\n"
      "tbz x11, #1, 135f\n"
      "st1 { v31.h }[2], [x9], #0x2\n"
      "st1 { v8.h }[2], [x27], #0x2\n"
      "st1 { v15.h }[2], [x26], #0x2\n"
      "st1 { v16.h }[2], [x25], #0x2\n"
      "st1 { v24.h }[2], [x24], #0x2\n"
      "tbz x11, #0, 138f\n"
      "st1 { v31.b }[6], [x9]\n"
      "st1 { v8.b }[6], [x27]\n"
      "st1 { v15.b }[6], [x26]\n"
      "st1 { v16.b }[6], [x25]\n"
      "st1 { v24.b }[6], [x24]\n"
      "b 138f\n"
      "135:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x11, #0, 138f\n"
      "st1 { v31.b }[4], [x9]\n"
      "st1 { v8.b }[4], [x27]\n"
      "st1 { v15.b }[4], [x26]\n"
      "st1 { v16.b }[4], [x25]\n"
      "st1 { v24.b }[4], [x24]\n"
      "b 138f\n"
      "136:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x11, #1, 137f\n"
      "str h31, [x9], #0x2\n"
      "str h8, [x27], #0x2\n"
      "str h15, [x26], #0x2\n"
      "str h16, [x25], #0x2\n"
      "str h24, [x24], #0x2\n"
      "tbz x11, #0, 138f\n"
      "st1 { v31.b }[2], [x9]\n"
      "st1 { v8.b }[2], [x27]\n"
      "st1 { v15.b }[2], [x26]\n"
      "st1 { v16.b }[2], [x25]\n"
      "st1 { v24.b }[2], [x24]\n"
      "b 138f\n"
      "137:"  // Height 5: Partial direct writeback: partial_1_0
      "str b31, [x9, #0]\n"
      "str b8, [x27, #0]\n"
      "str b15, [x26, #0]\n"
      "str b16, [x25, #0]\n"
      "str b24, [x24, #0]\n"
      "138:"  // Height 5: Partial direct writeback: Done
      "b 140f\n"
      "139:"  // Height 5: Full writeback
      "str q31, [x9, #0]\n"
      "add x9, x9, #0x10\n"
      "str q8, [x27, #0]\n"
      "str q15, [x26, #0]\n"
      "str q16, [x25, #0]\n"
      "str q24, [x24, #0]\n"
      "140:"  // Height 5: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 114b\n"
      "b 170f\n"
      "141:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x6\n"
      "mov x12, %x[col_bias]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd x20, x21, x20, x9\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "142:"  // Height 6: Column loop
      "movi v8.4s, #0\n"
      "movi v9.4s, #0\n"
      "movi v10.4s, #0\n"
      "movi v11.4s, #0\n"
      "movi v12.4s, #0\n"
      "movi v13.4s, #0\n"
      "movi v14.4s, #0\n"
      "movi v15.4s, #0\n"
      "movi v16.4s, #0\n"
      "movi v17.4s, #0\n"
      "movi v18.4s, #0\n"
      "movi v19.4s, #0\n"
      "movi v20.4s, #0\n"
      "movi v21.4s, #0\n"
      "movi v22.4s, #0\n"
      "movi v23.4s, #0\n"
      "movi v24.4s, #0\n"
      "movi v25.4s, #0\n"
      "movi v26.4s, #0\n"
      "movi v27.4s, #0\n"
      "movi v28.4s, #0\n"
      "movi v29.4s, #0\n"
      "movi v30.4s, #0\n"
      "movi v31.4s, #0\n"
      "mov x28, #0\n"
      "144:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 145f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 146f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 146f\n"
      "145:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "146:"  // Height 6: input setup done
      "cmp x27, #0x10\n"
      "blt 149f\n"
      "ldr q1, [x26, #0]\n"
      "ldr q2, [x25, #0]\n"
      "cmp x27, #0x20\n"
      "ldr q3, [x24, #0]\n"
      "ldr q4, [x23, #0]\n"
      "ldr q5, [x22, #0]\n"
      "ldr q6, [x21, #0]\n"
      "ldr q7, [x10, #0]\n"
      "blt 148f\n"
      "147:"  // Height 6: Multiply loop: Main loop head
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "sub x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x10, #0x10]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "cmp x27, #0x20\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      "ldr q2, [x25, #0]\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      "ldr q4, [x23, #0]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a474  // smmla v20.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bc  // smmla v28.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a471  // smmla v17.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b9  // smmla v25.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a475  // smmla v21.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bd  // smmla v29.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a472  // smmla v18.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4ba  // smmla v26.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a476  // smmla v22.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4be  // smmla v30.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a473  // smmla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4bb  // smmla v27.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0]\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      "ldr q1, [x26, #0]\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      "ldr q3, [x24, #0]\n"
      ".inst 0x4e86a4bf  // smmla v31.4s, v5.16b, v6.16b\n"
      "ldr q5, [x22, #0]\n"
      "ldr q6, [x21, #0]\n"
      "bge 147b\n"
      "148:"  // Height 6: Multiply loop: Single iteration only
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn2 v1.2d, v1.2d, v2.2d\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn2 v3.2d, v3.2d, v4.2d\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "trn2 v5.2d, v5.2d, v6.2d\n"
      "ldr q6, [x10, #0x10]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      "add x21, x21, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "sub x27, x27, #0x10\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x80]\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x90]\n"
      ".inst 0x4e87a428  // smmla v8.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a470  // smmla v16.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b8  // smmla v24.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xa0]\n"
      ".inst 0x4e86a42c  // smmla v12.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a474  // smmla v20.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bc  // smmla v28.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xb0]\n"
      ".inst 0x4e87a429  // smmla v9.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a471  // smmla v17.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4b9  // smmla v25.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xc0]\n"
      ".inst 0x4e86a42d  // smmla v13.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a475  // smmla v21.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bd  // smmla v29.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xd0]\n"
      ".inst 0x4e87a42a  // smmla v10.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a472  // smmla v18.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4ba  // smmla v26.4s, v5.16b, v7.16b\n"
      "ldr q7, [x10, #0xe0]\n"
      ".inst 0x4e86a42e  // smmla v14.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a476  // smmla v22.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4be  // smmla v30.4s, v5.16b, v6.16b\n"
      "ldr q6, [x10, #0xf0]\n"
      "add x10, x10, #0x100\n"
      ".inst 0x4e87a42b  // smmla v11.4s, v1.16b, v7.16b\n"
      ".inst 0x4e87a473  // smmla v19.4s, v3.16b, v7.16b\n"
      ".inst 0x4e87a4bb  // smmla v27.4s, v5.16b, v7.16b\n"
      ".inst 0x4e86a42f  // smmla v15.4s, v1.16b, v6.16b\n"
      ".inst 0x4e86a477  // smmla v23.4s, v3.16b, v6.16b\n"
      ".inst 0x4e86a4bf  // smmla v31.4s, v5.16b, v6.16b\n"
      "149:"  // Height 6: Multiply loop: Main loop skip
      "cbz x27, 156f\n"
      "cmp x27, #0x8\n"
      "blt 151f\n"
      "150:"  // Height 6: Multiply loop: Odd block loop
      "ldr d1, [x26], #0x8\n"
      "ldr d2, [x25], #0x8\n"
      "sub x27, x27, #0x8\n"
      "ldr d3, [x24], #0x8\n"
      "ldr d4, [x23], #0x8\n"
      "cmp x27, #0x8\n"
      "ldr d5, [x22], #0x8\n"
      "ldr d7, [x21], #0x8\n"
      "ldr q6, [x10, #0]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn1 v4.2d, v5.2d, v7.2d\n"
      "ldr q7, [x10, #0x10]\n"
      ".inst 0x4e86a408  // smmla v8.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a450  // smmla v16.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a498  // smmla v24.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4e87a40c  // smmla v12.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a454  // smmla v20.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49c  // smmla v28.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x30]\n"
      ".inst 0x4e86a409  // smmla v9.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a451  // smmla v17.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a499  // smmla v25.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4e87a40d  // smmla v13.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a455  // smmla v21.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49d  // smmla v29.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x4e86a40a  // smmla v10.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a452  // smmla v18.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49a  // smmla v26.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4e87a40e  // smmla v14.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a456  // smmla v22.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49e  // smmla v30.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e86a40b  // smmla v11.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a453  // smmla v19.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49b  // smmla v27.4s, v4.16b, v6.16b\n"
      ".inst 0x4e87a40f  // smmla v15.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a457  // smmla v23.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49f  // smmla v31.4s, v4.16b, v7.16b\n"
      "bge 150b\n"
      "151:"  // Height 6: Multiply loop: Skip odd blocks
      "cbz x27, 156f\n"
      "tbz x27, #2, 153f\n"
      "ldr s1, [x26], #0x4\n"
      "ldr s2, [x25], #0x4\n"
      "ldr s3, [x24], #0x4\n"
      "ldr s4, [x23], #0x4\n"
      "ldr s5, [x22], #0x4\n"
      "ldr s6, [x21], #0x4\n"
      "tbz x27, #1, 152f\n"
      "ld1 { v1.h }[2], [x26], #0x2\n"
      "ld1 { v2.h }[2], [x25], #0x2\n"
      "ld1 { v3.h }[2], [x24], #0x2\n"
      "ld1 { v4.h }[2], [x23], #0x2\n"
      "ld1 { v5.h }[2], [x22], #0x2\n"
      "ld1 { v6.h }[2], [x21], #0x2\n"
      "tbz x27, #0, 155f\n"
      "ld1 { v1.b }[6], [x26]\n"
      "ld1 { v2.b }[6], [x25]\n"
      "ld1 { v3.b }[6], [x24]\n"
      "ld1 { v4.b }[6], [x23]\n"
      "ld1 { v5.b }[6], [x22]\n"
      "ld1 { v6.b }[6], [x21]\n"
      "b 155f\n"
      "152:"  // Height 6: Multiply loop: Ragged operand read: partial_1_4
      "tbz x27, #0, 155f\n"
      "ld1 { v1.b }[4], [x26]\n"
      "ld1 { v2.b }[4], [x25]\n"
      "ld1 { v3.b }[4], [x24]\n"
      "ld1 { v4.b }[4], [x23]\n"
      "ld1 { v5.b }[4], [x22]\n"
      "ld1 { v6.b }[4], [x21]\n"
      "b 155f\n"
      "153:"  // Height 6: Multiply loop: Ragged operand read: partial_2_0
      "tbz x27, #1, 154f\n"
      "ldr h1, [x26], #0x2\n"
      "ldr h2, [x25], #0x2\n"
      "ldr h3, [x24], #0x2\n"
      "ldr h4, [x23], #0x2\n"
      "ldr h5, [x22], #0x2\n"
      "ldr h6, [x21], #0x2\n"
      "tbz x27, #0, 155f\n"
      "ld1 { v1.b }[2], [x26]\n"
      "ld1 { v2.b }[2], [x25]\n"
      "ld1 { v3.b }[2], [x24]\n"
      "ld1 { v4.b }[2], [x23]\n"
      "ld1 { v5.b }[2], [x22]\n"
      "ld1 { v6.b }[2], [x21]\n"
      "b 155f\n"
      "154:"  // Height 6: Multiply loop: Ragged operand read: partial_1_0
      "ldr b1, [x26, #0]\n"
      "ldr b2, [x25, #0]\n"
      "ldr b3, [x24, #0]\n"
      "ldr b4, [x23, #0]\n"
      "ldr b5, [x22, #0]\n"
      "ldr b6, [x21, #0]\n"
      "155:"  // Height 6: Multiply loop: Ragged operand read: Done
      "ldr q7, [x10, #0]\n"
      "trn1 v0.2d, v1.2d, v2.2d\n"
      "trn1 v2.2d, v3.2d, v4.2d\n"
      "trn1 v4.2d, v5.2d, v6.2d\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4e87a408  // smmla v8.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a450  // smmla v16.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a498  // smmla v24.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x20]\n"
      ".inst 0x4e86a40c  // smmla v12.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a454  // smmla v20.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49c  // smmla v28.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4e87a409  // smmla v9.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a451  // smmla v17.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a499  // smmla v25.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x40]\n"
      ".inst 0x4e86a40d  // smmla v13.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a455  // smmla v21.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49d  // smmla v29.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4e87a40a  // smmla v10.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a452  // smmla v18.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49a  // smmla v26.4s, v4.16b, v7.16b\n"
      "ldr q7, [x10, #0x60]\n"
      ".inst 0x4e86a40e  // smmla v14.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a456  // smmla v22.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49e  // smmla v30.4s, v4.16b, v6.16b\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4e87a40b  // smmla v11.4s, v0.16b, v7.16b\n"
      ".inst 0x4e87a453  // smmla v19.4s, v2.16b, v7.16b\n"
      ".inst 0x4e87a49b  // smmla v27.4s, v4.16b, v7.16b\n"
      ".inst 0x4e86a40f  // smmla v15.4s, v0.16b, v6.16b\n"
      ".inst 0x4e86a457  // smmla v23.4s, v2.16b, v6.16b\n"
      ".inst 0x4e86a49f  // smmla v31.4s, v4.16b, v6.16b\n"
      "156:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 144b\n"
      "ldr q0, [x12, #0]\n"
      "ldr q1, [x12, #0x10]\n"
      "uzp1 v7.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "ldr q2, [x12, #0x20]\n"
      "ldr q3, [x12, #0x30]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "prfm pstl1keep, [x9, #0]\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "add x12, x12, #0x40\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "add x27, x9, x20\n"
      "add x26, x27, x20\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "add x25, x26, x20\n"
      "prfm pstl1keep, [x27, #0]\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "add x24, x25, x20\n"
      "prfm pstl1keep, [x26, #0]\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "add x23, x24, x20\n"
      "prfm pstl1keep, [x25, #0]\n"
      "uzp1 v23.2d, v24.2d, v28.2d\n"
      "uzp2 v24.2d, v24.2d, v28.2d\n"
      "prfm pstl1keep, [x24, #0]\n"
      "prfm pstl1keep, [x23, #0]\n"
      "uzp1 v28.2d, v25.2d, v29.2d\n"
      "uzp2 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v29.2d, v26.2d, v30.2d\n"
      "uzp2 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v30.2d, v27.2d, v31.2d\n"
      "uzp2 v27.2d, v27.2d, v31.2d\n"
      "mov v31.16b, v7.16b\n"
      "add v12.4s, v12.4s, v1.4s\n"
      "add v13.4s, v13.4s, v2.4s\n"
      "add v14.4s, v14.4s, v3.4s\n"
      "add v31.4s, v31.4s, v0.4s\n"
      "add v8.4s, v8.4s, v0.4s\n"
      "add v9.4s, v9.4s, v1.4s\n"
      "add v10.4s, v10.4s, v2.4s\n"
      "add v11.4s, v11.4s, v3.4s\n"
      "add v15.4s, v15.4s, v0.4s\n"
      "add v20.4s, v20.4s, v1.4s\n"
      "add v21.4s, v21.4s, v2.4s\n"
      "add v22.4s, v22.4s, v3.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v1.4s\n"
      "add v18.4s, v18.4s, v2.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v23.4s, v23.4s, v0.4s\n"
      "add v28.4s, v28.4s, v1.4s\n"
      "add v29.4s, v29.4s, v2.4s\n"
      "add v30.4s, v30.4s, v3.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "add v25.4s, v25.4s, v1.4s\n"
      "add v26.4s, v26.4s, v2.4s\n"
      "add v27.4s, v27.4s, v3.4s\n"
      "tbz %x[flags], #4, 157f\n"
      "ldr q0, [x13, #0]\n"
      "ldr q4, [x14, #0]\n"
      "ldr q1, [x13, #0x10]\n"
      "ldr q5, [x14, #0x10]\n"
      "ldr q2, [x13, #0x20]\n"
      "ldr q6, [x14, #0x20]\n"
      "ldr q3, [x13, #0x30]\n"
      "ldr q7, [x14, #0x30]\n"
      "add x13, x13, #0x40\n"
      "add x14, x14, #0x40\n"
      "b 158f\n"
      "157:"  // Height 6: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1r { v0.4s }, [x21]\n"
      "ld1r { v4.4s }, [x20]\n"
      "mov v1.16b, v0.16b\n"
      "mov v5.16b, v4.16b\n"
      "mov v2.16b, v0.16b\n"
      "mov v6.16b, v4.16b\n"
      "mov v3.16b, v0.16b\n"
      "mov v7.16b, v4.16b\n"
      "158:"  // Height 6: parameters loaded
      "sqdmulh v31.4s, v31.4s, v4.4s\n"
      "sqdmulh v12.4s, v12.4s, v5.4s\n"
      "add x22, %x[qp], %[c_offset]\n"
      "add x21, %x[qp], %[maxval]\n"
      "sqdmulh v13.4s, v13.4s, v6.4s\n"
      "sqdmulh v14.4s, v14.4s, v7.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x11, #0x10\n"
      "sqdmulh v8.4s, v8.4s, v4.4s\n"
      "sqdmulh v9.4s, v9.4s, v5.4s\n"
      "sqdmulh v10.4s, v10.4s, v6.4s\n"
      "sqdmulh v11.4s, v11.4s, v7.4s\n"
      "sqdmulh v15.4s, v15.4s, v4.4s\n"
      "sqdmulh v20.4s, v20.4s, v5.4s\n"
      "sqdmulh v21.4s, v21.4s, v6.4s\n"
      "sqdmulh v22.4s, v22.4s, v7.4s\n"
      "sqdmulh v16.4s, v16.4s, v4.4s\n"
      "sqdmulh v17.4s, v17.4s, v5.4s\n"
      "sqdmulh v18.4s, v18.4s, v6.4s\n"
      "sqdmulh v19.4s, v19.4s, v7.4s\n"
      "sqdmulh v23.4s, v23.4s, v4.4s\n"
      "sqdmulh v28.4s, v28.4s, v5.4s\n"
      "sqdmulh v29.4s, v29.4s, v6.4s\n"
      "sqdmulh v30.4s, v30.4s, v7.4s\n"
      "sqdmulh v24.4s, v24.4s, v4.4s\n"
      "ld1r { v4.4s }, [x22]\n"
      "sqdmulh v25.4s, v25.4s, v5.4s\n"
      "ld1r { v5.4s }, [x20]\n"
      "sqdmulh v26.4s, v26.4s, v6.4s\n"
      "ld1r { v6.4s }, [x21]\n"
      "sqdmulh v27.4s, v27.4s, v7.4s\n"
      "srshl v31.4s, v31.4s, v0.4s\n"
      "srshl v12.4s, v12.4s, v1.4s\n"
      "srshl v13.4s, v13.4s, v2.4s\n"
      "srshl v14.4s, v14.4s, v3.4s\n"
      "srshl v8.4s, v8.4s, v0.4s\n"
      "srshl v9.4s, v9.4s, v1.4s\n"
      "srshl v10.4s, v10.4s, v2.4s\n"
      "srshl v11.4s, v11.4s, v3.4s\n"
      "srshl v15.4s, v15.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v1.4s\n"
      "srshl v21.4s, v21.4s, v2.4s\n"
      "srshl v22.4s, v22.4s, v3.4s\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v1.4s\n"
      "srshl v18.4s, v18.4s, v2.4s\n"
      "srshl v19.4s, v19.4s, v3.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v28.4s, v28.4s, v1.4s\n"
      "srshl v29.4s, v29.4s, v2.4s\n"
      "srshl v30.4s, v30.4s, v3.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v1.4s\n"
      "srshl v26.4s, v26.4s, v2.4s\n"
      "srshl v27.4s, v27.4s, v3.4s\n"
      "add v31.4s, v31.4s, v4.4s\n"
      "add v12.4s, v12.4s, v4.4s\n"
      "add v13.4s, v13.4s, v4.4s\n"
      "add v14.4s, v14.4s, v4.4s\n"
      "add v8.4s, v8.4s, v4.4s\n"
      "add v9.4s, v9.4s, v4.4s\n"
      "add v10.4s, v10.4s, v4.4s\n"
      "add v11.4s, v11.4s, v4.4s\n"
      "add v15.4s, v15.4s, v4.4s\n"
      "add v20.4s, v20.4s, v4.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v4.4s\n"
      "add v16.4s, v16.4s, v4.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v4.4s\n"
      "add v19.4s, v19.4s, v4.4s\n"
      "add v23.4s, v23.4s, v4.4s\n"
      "add v28.4s, v28.4s, v4.4s\n"
      "add v29.4s, v29.4s, v4.4s\n"
      "add v30.4s, v30.4s, v4.4s\n"
      "add v24.4s, v24.4s, v4.4s\n"
      "add v25.4s, v25.4s, v4.4s\n"
      "add v26.4s, v26.4s, v4.4s\n"
      "add v27.4s, v27.4s, v4.4s\n"
      "smin v31.4s, v31.4s, v6.4s\n"
      "smin v12.4s, v12.4s, v6.4s\n"
      "smin v13.4s, v13.4s, v6.4s\n"
      "smin v14.4s, v14.4s, v6.4s\n"
      "smin v8.4s, v8.4s, v6.4s\n"
      "smin v9.4s, v9.4s, v6.4s\n"
      "smin v10.4s, v10.4s, v6.4s\n"
      "smin v11.4s, v11.4s, v6.4s\n"
      "smin v15.4s, v15.4s, v6.4s\n"
      "smin v20.4s, v20.4s, v6.4s\n"
      "smin v21.4s, v21.4s, v6.4s\n"
      "smin v22.4s, v22.4s, v6.4s\n"
      "smin v16.4s, v16.4s, v6.4s\n"
      "smin v17.4s, v17.4s, v6.4s\n"
      "smin v18.4s, v18.4s, v6.4s\n"
      "smin v19.4s, v19.4s, v6.4s\n"
      "smin v23.4s, v23.4s, v6.4s\n"
      "smin v28.4s, v28.4s, v6.4s\n"
      "smin v29.4s, v29.4s, v6.4s\n"
      "smin v30.4s, v30.4s, v6.4s\n"
      "smin v24.4s, v24.4s, v6.4s\n"
      "smin v25.4s, v25.4s, v6.4s\n"
      "smin v26.4s, v26.4s, v6.4s\n"
      "smin v27.4s, v27.4s, v6.4s\n"
      "smax v31.4s, v31.4s, v5.4s\n"
      "smax v12.4s, v12.4s, v5.4s\n"
      "smax v13.4s, v13.4s, v5.4s\n"
      "smax v14.4s, v14.4s, v5.4s\n"
      "smax v8.4s, v8.4s, v5.4s\n"
      "smax v9.4s, v9.4s, v5.4s\n"
      "smax v10.4s, v10.4s, v5.4s\n"
      "smax v11.4s, v11.4s, v5.4s\n"
      "smax v15.4s, v15.4s, v5.4s\n"
      "smax v20.4s, v20.4s, v5.4s\n"
      "smax v21.4s, v21.4s, v5.4s\n"
      "smax v22.4s, v22.4s, v5.4s\n"
      "smax v16.4s, v16.4s, v5.4s\n"
      "smax v17.4s, v17.4s, v5.4s\n"
      "smax v18.4s, v18.4s, v5.4s\n"
      "smax v19.4s, v19.4s, v5.4s\n"
      "smax v23.4s, v23.4s, v5.4s\n"
      "smax v28.4s, v28.4s, v5.4s\n"
      "smax v29.4s, v29.4s, v5.4s\n"
      "smax v30.4s, v30.4s, v5.4s\n"
      "smax v24.4s, v24.4s, v5.4s\n"
      "smax v25.4s, v25.4s, v5.4s\n"
      "smax v26.4s, v26.4s, v5.4s\n"
      "smax v27.4s, v27.4s, v5.4s\n"
      "uzp1 v31.8h, v31.8h, v12.8h\n"
      "uzp1 v12.8h, v13.8h, v14.8h\n"
      "uzp1 v8.8h, v8.8h, v9.8h\n"
      "uzp1 v9.8h, v10.8h, v11.8h\n"
      "uzp1 v15.8h, v15.8h, v20.8h\n"
      "uzp1 v20.8h, v21.8h, v22.8h\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v23.8h, v23.8h, v28.8h\n"
      "uzp1 v28.8h, v29.8h, v30.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v25.8h, v26.8h, v27.8h\n"
      "uzp1 v31.16b, v31.16b, v12.16b\n"
      "uzp1 v8.16b, v8.16b, v9.16b\n"
      "uzp1 v15.16b, v15.16b, v20.16b\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "uzp1 v23.16b, v23.16b, v28.16b\n"
      "uzp1 v24.16b, v24.16b, v25.16b\n"
      "bge 167f\n"
      "tbz x11, #3, 162f\n"
      "str d31, [x9], #0x8\n"
      "str d8, [x27], #0x8\n"
      "str d15, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "str d23, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "tbz x11, #2, 160f\n"
      "st1 { v31.s }[2], [x9], #0x4\n"
      "st1 { v8.s }[2], [x27], #0x4\n"
      "st1 { v15.s }[2], [x26], #0x4\n"
      "st1 { v16.s }[2], [x25], #0x4\n"
      "st1 { v23.s }[2], [x24], #0x4\n"
      "st1 { v24.s }[2], [x23], #0x4\n"
      "tbz x11, #1, 159f\n"
      "st1 { v31.h }[6], [x9], #0x2\n"
      "st1 { v8.h }[6], [x27], #0x2\n"
      "st1 { v15.h }[6], [x26], #0x2\n"
      "st1 { v16.h }[6], [x25], #0x2\n"
      "st1 { v23.h }[6], [x24], #0x2\n"
      "st1 { v24.h }[6], [x23], #0x2\n"
      "tbz x11, #0, 166f\n"
      "st1 { v31.b }[14], [x9]\n"
      "st1 { v8.b }[14], [x27]\n"
      "st1 { v15.b }[14], [x26]\n"
      "st1 { v16.b }[14], [x25]\n"
      "st1 { v23.b }[14], [x24]\n"
      "st1 { v24.b }[14], [x23]\n"
      "b 166f\n"
      "159:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x11, #0, 166f\n"
      "st1 { v31.b }[12], [x9]\n"
      "st1 { v8.b }[12], [x27]\n"
      "st1 { v15.b }[12], [x26]\n"
      "st1 { v16.b }[12], [x25]\n"
      "st1 { v23.b }[12], [x24]\n"
      "st1 { v24.b }[12], [x23]\n"
      "b 166f\n"
      "160:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x11, #1, 161f\n"
      "st1 { v31.h }[4], [x9], #0x2\n"
      "st1 { v8.h }[4], [x27], #0x2\n"
      "st1 { v15.h }[4], [x26], #0x2\n"
      "st1 { v16.h }[4], [x25], #0x2\n"
      "st1 { v23.h }[4], [x24], #0x2\n"
      "st1 { v24.h }[4], [x23], #0x2\n"
      "tbz x11, #0, 166f\n"
      "st1 { v31.b }[10], [x9]\n"
      "st1 { v8.b }[10], [x27]\n"
      "st1 { v15.b }[10], [x26]\n"
      "st1 { v16.b }[10], [x25]\n"
      "st1 { v23.b }[10], [x24]\n"
      "st1 { v24.b }[10], [x23]\n"
      "b 166f\n"
      "161:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x11, #0, 166f\n"
      "st1 { v31.b }[8], [x9]\n"
      "st1 { v8.b }[8], [x27]\n"
      "st1 { v15.b }[8], [x26]\n"
      "st1 { v16.b }[8], [x25]\n"
      "st1 { v23.b }[8], [x24]\n"
      "st1 { v24.b }[8], [x23]\n"
      "b 166f\n"
      "162:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x11, #2, 164f\n"
      "str s31, [x9], #0x4\n"
      "str s8, [x27], #0x4\n"
      "str s15, [x26], #0x4\n"
      "str s16, [x25], #0x4\n"
      "str s23, [x24], #0x4\n"
      "str s24, [x23], #0x4\n"
      "tbz x11, #1, 163f\n"
      "st1 { v31.h }[2], [x9], #0x2\n"
      "st1 { v8.h }[2], [x27], #0x2\n"
      "st1 { v15.h }[2], [x26], #0x2\n"
      "st1 { v16.h }[2], [x25], #0x2\n"
      "st1 { v23.h }[2], [x24], #0x2\n"
      "st1 { v24.h }[2], [x23], #0x2\n"
      "tbz x11, #0, 166f\n"
      "st1 { v31.b }[6], [x9]\n"
      "st1 { v8.b }[6], [x27]\n"
      "st1 { v15.b }[6], [x26]\n"
      "st1 { v16.b }[6], [x25]\n"
      "st1 { v23.b }[6], [x24]\n"
      "st1 { v24.b }[6], [x23]\n"
      "b 166f\n"
      "163:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x11, #0, 166f\n"
      "st1 { v31.b }[4], [x9]\n"
      "st1 { v8.b }[4], [x27]\n"
      "st1 { v15.b }[4], [x26]\n"
      "st1 { v16.b }[4], [x25]\n"
      "st1 { v23.b }[4], [x24]\n"
      "st1 { v24.b }[4], [x23]\n"
      "b 166f\n"
      "164:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x11, #1, 165f\n"
      "str h31, [x9], #0x2\n"
      "str h8, [x27], #0x2\n"
      "str h15, [x26], #0x2\n"
      "str h16, [x25], #0x2\n"
      "str h23, [x24], #0x2\n"
      "str h24, [x23], #0x2\n"
      "tbz x11, #0, 166f\n"
      "st1 { v31.b }[2], [x9]\n"
      "st1 { v8.b }[2], [x27]\n"
      "st1 { v15.b }[2], [x26]\n"
      "st1 { v16.b }[2], [x25]\n"
      "st1 { v23.b }[2], [x24]\n"
      "st1 { v24.b }[2], [x23]\n"
      "b 166f\n"
      "165:"  // Height 6: Partial direct writeback: partial_1_0
      "str b31, [x9, #0]\n"
      "str b8, [x27, #0]\n"
      "str b15, [x26, #0]\n"
      "str b16, [x25, #0]\n"
      "str b23, [x24, #0]\n"
      "str b24, [x23, #0]\n"
      "166:"  // Height 6: Partial direct writeback: Done
      "b 168f\n"
      "167:"  // Height 6: Full writeback
      "str q31, [x9, #0]\n"
      "add x9, x9, #0x10\n"
      "str q8, [x27, #0]\n"
      "str q15, [x26, #0]\n"
      "str q16, [x25, #0]\n"
      "str q23, [x24, #0]\n"
      "str q24, [x23, #0]\n"
      "168:"  // Height 6: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 142b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 170f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 169f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "169:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "170:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [flags] "r" (flags), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_multiplier_ptr] "I" (offsetof(KernelArgs, multiplier_ptr)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_shift_ptr] "I" (offsetof(KernelArgs, shift_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x13", "x14", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm

#endif // __aarch64__

