/*
 * Copyright (c) 2025-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(__aarch64__)

#include "arm_gemm/arm_gemm.hpp"
#include "arm_common/internal/utils.hpp"
#include <cassert>
#include <limits>

namespace arm_gemm {

void a64_hybrid_fp16fp32_mla_6x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<__fp16> A_arg,
    size_t M, size_t N, const __fp16 *B_ptr, IndirectOutputArg<float> output_arg,
    const float *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const __fp16 *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const float *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 166f\n"
      "cmp %x[M], #0x4\n"
      "bgt 133f\n"
      "beq 100f\n"
      "cmp %x[M], #0x2\n"
      "bgt 67f\n"
      "beq 34f\n"
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "cbz x10, 3f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "b 14f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 13f\n"
      "cmp x12, #0x10\n"
      "bge 12f\n"
      "tbz x12, #3, 7f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "tbz x12, #2, 5f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "tbz x12, #1, 4f\n"
      "ldr d11, [x9], #0x8\n"
      "mov x20, #0x38\n"
      "tbz x12, #0, 11f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "b 11f\n"
      "4:"  // Height 1: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x12, #0, 11f\n"
      "ldr s11, [x9, #0]\n"
      "b 11f\n"
      "5:"  // Height 1: Partial accumulate: partial_2_8
      "tbz x12, #1, 6f\n"
      "ldr d10, [x9], #0x8\n"
      "mov x20, #0x28\n"
      "tbz x12, #0, 11f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "b 11f\n"
      "6:"  // Height 1: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x12, #0, 11f\n"
      "ldr s10, [x9, #0]\n"
      "b 11f\n"
      "7:"  // Height 1: Partial accumulate: partial_4_0
      "tbz x12, #2, 9f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "tbz x12, #1, 8f\n"
      "ldr d9, [x9], #0x8\n"
      "mov x20, #0x18\n"
      "tbz x12, #0, 11f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "b 11f\n"
      "8:"  // Height 1: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x12, #0, 11f\n"
      "ldr s9, [x9, #0]\n"
      "b 11f\n"
      "9:"  // Height 1: Partial accumulate: partial_2_0
      "tbz x12, #1, 10f\n"
      "ldr d8, [x9], #0x8\n"
      "mov x20, #0x8\n"
      "tbz x12, #0, 11f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "b 11f\n"
      "10:"  // Height 1: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0]\n"
      "mov x20, #0\n"
      "11:"  // Height 1: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 14f\n"
      "12:"  // Height 1: full accumulate
      "ldr q8, [x9, #0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "b 14f\n"
      "13:"  // Height 1: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "14:"  // Height 1: setup done
      "mov x28, #0\n"
      "15:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 16f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "cbnz x28, 17f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "b 17f\n"
      "16:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "17:"  // Height 1: input setup done
      "cmp x27, #0x8\n"
      "blt 20f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q6, [x11, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q7, [x11, #0x10]\n"
      "blt 19f\n"
      "18:"  // Height 1: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "sub x27, x27, #0x8\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      "add x26, x26, #0x10\n"
      "cmp x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      "ldr q6, [x11, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "bge 18b\n"
      "19:"  // Height 1: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      "sub x27, x27, #0x8\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "20:"  // Height 1: Multiply loop: Main loop skip
      "cbz x27, 22f\n"
      "21:"  // Height 1: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr q6, [x11, #0]\n"
      "sub x27, x27, #0x1\n"
      "ldr q7, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      "cbnz x27, 21b\n"
      "22:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 15b\n"
      "prfm pstl1keep, [x9, #0]\n"
      "tbz %x[flags], #1, 23f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v17.4s }, [x21]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v17.4s\n"
      "fmin v9.4s, v9.4s, v17.4s\n"
      "fmin v10.4s, v10.4s, v17.4s\n"
      "fmin v11.4s, v11.4s, v17.4s\n"
      "fmax v8.4s, v8.4s, v16.4s\n"
      "fmax v9.4s, v9.4s, v16.4s\n"
      "fmax v10.4s, v10.4s, v16.4s\n"
      "fmax v11.4s, v11.4s, v16.4s\n"
      "23:"  // Height 1: No activation
      "cmp x12, #0x10\n"
      "bge 32f\n"
      "tbz x12, #3, 27f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "tbz x12, #2, 25f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "tbz x12, #1, 24f\n"
      "str d11, [x9], #0x8\n"
      "tbz x12, #0, 31f\n"
      "st1 { v11.s }[2], [x9]\n"
      "b 31f\n"
      "24:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x12, #0, 31f\n"
      "str s11, [x9, #0]\n"
      "b 31f\n"
      "25:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x12, #1, 26f\n"
      "str d10, [x9], #0x8\n"
      "tbz x12, #0, 31f\n"
      "st1 { v10.s }[2], [x9]\n"
      "b 31f\n"
      "26:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x12, #0, 31f\n"
      "str s10, [x9, #0]\n"
      "b 31f\n"
      "27:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x12, #2, 29f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "tbz x12, #1, 28f\n"
      "str d9, [x9], #0x8\n"
      "tbz x12, #0, 31f\n"
      "st1 { v9.s }[2], [x9]\n"
      "b 31f\n"
      "28:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x12, #0, 31f\n"
      "str s9, [x9, #0]\n"
      "b 31f\n"
      "29:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x12, #1, 30f\n"
      "str d8, [x9], #0x8\n"
      "tbz x12, #0, 31f\n"
      "st1 { v8.s }[2], [x9]\n"
      "b 31f\n"
      "30:"  // Height 1: Partial direct writeback: partial_1_0
      "str s8, [x9, #0]\n"
      "31:"  // Height 1: Partial direct writeback: Done
      "b 33f\n"
      "32:"  // Height 1: Full writeback
      "str q8, [x9, #0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "33:"  // Height 1: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 2b\n"
      "b 200f\n"
      "34:"  // Height 2
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "35:"  // Height 2: Column loop
      "cbz x10, 36f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "b 47f\n"
      "36:"  // Height 2: no bias
      "tbz %x[flags], #0, 46f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "add x26, x9, x20, LSL #2\n"
      "bge 45f\n"
      "tbz x12, #3, 40f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x26], #0x10\n"
      "tbz x12, #2, 38f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x26], #0x10\n"
      "tbz x12, #1, 37f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "mov x20, #0x38\n"
      "tbz x12, #0, 44f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x26]\n"
      "b 44f\n"
      "37:"  // Height 2: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x12, #0, 44f\n"
      "ldr s11, [x9, #0]\n"
      "ldr s15, [x26, #0]\n"
      "b 44f\n"
      "38:"  // Height 2: Partial accumulate: partial_2_8
      "tbz x12, #1, 39f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "mov x20, #0x28\n"
      "tbz x12, #0, 44f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x26]\n"
      "b 44f\n"
      "39:"  // Height 2: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x12, #0, 44f\n"
      "ldr s10, [x9, #0]\n"
      "ldr s14, [x26, #0]\n"
      "b 44f\n"
      "40:"  // Height 2: Partial accumulate: partial_4_0
      "tbz x12, #2, 42f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "tbz x12, #1, 41f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "mov x20, #0x18\n"
      "tbz x12, #0, 44f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x26]\n"
      "b 44f\n"
      "41:"  // Height 2: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x12, #0, 44f\n"
      "ldr s9, [x9, #0]\n"
      "ldr s13, [x26, #0]\n"
      "b 44f\n"
      "42:"  // Height 2: Partial accumulate: partial_2_0
      "tbz x12, #1, 43f\n"
      "ldr d8, [x9], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "mov x20, #0x8\n"
      "tbz x12, #0, 44f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x26]\n"
      "b 44f\n"
      "43:"  // Height 2: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0]\n"
      "ldr s12, [x26, #0]\n"
      "mov x20, #0\n"
      "44:"  // Height 2: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 47f\n"
      "45:"  // Height 2: full accumulate
      "ldr q8, [x9, #0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "ldr q12, [x26, #0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "b 47f\n"
      "46:"  // Height 2: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "movi v12.16b, #0\n"
      "movi v13.16b, #0\n"
      "movi v14.16b, #0\n"
      "movi v15.16b, #0\n"
      "47:"  // Height 2: setup done
      "mov x28, #0\n"
      "48:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 49f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 50f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "b 50f\n"
      "49:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "50:"  // Height 2: input setup done
      "cmp x27, #0x8\n"
      "blt 53f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q1, [x25, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "blt 52f\n"
      "51:"  // Height 2: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      "cmp x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      "ldr q6, [x11, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "ldr q1, [x25, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "bge 51b\n"
      "52:"  // Height 2: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "sub x27, x27, #0x8\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "53:"  // Height 2: Multiply loop: Main loop skip
      "cbz x27, 55f\n"
      "54:"  // Height 2: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      "cbnz x27, 54b\n"
      "55:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 48b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x9, #0]\n"
      "add x26, x9, x20, LSL #2\n"
      "prfm pstl1keep, [x26, #0]\n"
      "tbz %x[flags], #1, 56f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v17.4s }, [x21]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v17.4s\n"
      "fmin v9.4s, v9.4s, v17.4s\n"
      "fmin v10.4s, v10.4s, v17.4s\n"
      "fmin v11.4s, v11.4s, v17.4s\n"
      "fmin v12.4s, v12.4s, v17.4s\n"
      "fmin v13.4s, v13.4s, v17.4s\n"
      "fmin v14.4s, v14.4s, v17.4s\n"
      "fmin v15.4s, v15.4s, v17.4s\n"
      "fmax v8.4s, v8.4s, v16.4s\n"
      "fmax v9.4s, v9.4s, v16.4s\n"
      "fmax v10.4s, v10.4s, v16.4s\n"
      "fmax v11.4s, v11.4s, v16.4s\n"
      "fmax v12.4s, v12.4s, v16.4s\n"
      "fmax v13.4s, v13.4s, v16.4s\n"
      "fmax v14.4s, v14.4s, v16.4s\n"
      "fmax v15.4s, v15.4s, v16.4s\n"
      "56:"  // Height 2: No activation
      "cmp x12, #0x10\n"
      "bge 65f\n"
      "tbz x12, #3, 60f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v13.4s }, [x26], #0x10\n"
      "tbz x12, #2, 58f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "st1 { v14.4s }, [x26], #0x10\n"
      "tbz x12, #1, 57f\n"
      "str d11, [x9], #0x8\n"
      "str d15, [x26], #0x8\n"
      "tbz x12, #0, 64f\n"
      "st1 { v11.s }[2], [x9]\n"
      "st1 { v15.s }[2], [x26]\n"
      "b 64f\n"
      "57:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x12, #0, 64f\n"
      "str s11, [x9, #0]\n"
      "str s15, [x26, #0]\n"
      "b 64f\n"
      "58:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x12, #1, 59f\n"
      "str d10, [x9], #0x8\n"
      "str d14, [x26], #0x8\n"
      "tbz x12, #0, 64f\n"
      "st1 { v10.s }[2], [x9]\n"
      "st1 { v14.s }[2], [x26]\n"
      "b 64f\n"
      "59:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x12, #0, 64f\n"
      "str s10, [x9, #0]\n"
      "str s14, [x26, #0]\n"
      "b 64f\n"
      "60:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x12, #2, 62f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "tbz x12, #1, 61f\n"
      "str d9, [x9], #0x8\n"
      "str d13, [x26], #0x8\n"
      "tbz x12, #0, 64f\n"
      "st1 { v9.s }[2], [x9]\n"
      "st1 { v13.s }[2], [x26]\n"
      "b 64f\n"
      "61:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x12, #0, 64f\n"
      "str s9, [x9, #0]\n"
      "str s13, [x26, #0]\n"
      "b 64f\n"
      "62:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x12, #1, 63f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x26], #0x8\n"
      "tbz x12, #0, 64f\n"
      "st1 { v8.s }[2], [x9]\n"
      "st1 { v12.s }[2], [x26]\n"
      "b 64f\n"
      "63:"  // Height 2: Partial direct writeback: partial_1_0
      "str s8, [x9, #0]\n"
      "str s12, [x26, #0]\n"
      "64:"  // Height 2: Partial direct writeback: Done
      "b 66f\n"
      "65:"  // Height 2: Full writeback
      "str q8, [x9, #0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q12, [x26, #0]\n"
      "str q13, [x26, #0x10]\n"
      "str q14, [x26, #0x20]\n"
      "str q15, [x26, #0x30]\n"
      "66:"  // Height 2: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 35b\n"
      "b 200f\n"
      "67:"  // Height 3
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "68:"  // Height 3: Column loop
      "cbz x10, 69f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "b 80f\n"
      "69:"  // Height 3: no bias
      "tbz %x[flags], #0, 79f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "bge 78f\n"
      "tbz x12, #3, 73f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "tbz x12, #2, 71f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x26], #0x10\n"
      "ld1 { v18.4s }, [x25], #0x10\n"
      "tbz x12, #1, 70f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d19, [x25], #0x8\n"
      "tbz x12, #0, 77f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x26]\n"
      "ld1 { v19.s }[2], [x25]\n"
      "b 77f\n"
      "70:"  // Height 3: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x12, #0, 77f\n"
      "ldr s11, [x9, #0]\n"
      "ldr s15, [x26, #0]\n"
      "ldr s19, [x25, #0]\n"
      "b 77f\n"
      "71:"  // Height 3: Partial accumulate: partial_2_8
      "tbz x12, #1, 72f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d18, [x25], #0x8\n"
      "tbz x12, #0, 77f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x26]\n"
      "ld1 { v18.s }[2], [x25]\n"
      "b 77f\n"
      "72:"  // Height 3: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x12, #0, 77f\n"
      "ldr s10, [x9, #0]\n"
      "ldr s14, [x26, #0]\n"
      "ldr s18, [x25, #0]\n"
      "b 77f\n"
      "73:"  // Height 3: Partial accumulate: partial_4_0
      "tbz x12, #2, 75f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "tbz x12, #1, 74f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d17, [x25], #0x8\n"
      "tbz x12, #0, 77f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x26]\n"
      "ld1 { v17.s }[2], [x25]\n"
      "b 77f\n"
      "74:"  // Height 3: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x12, #0, 77f\n"
      "ldr s9, [x9, #0]\n"
      "ldr s13, [x26, #0]\n"
      "ldr s17, [x25, #0]\n"
      "b 77f\n"
      "75:"  // Height 3: Partial accumulate: partial_2_0
      "tbz x12, #1, 76f\n"
      "ldr d8, [x9], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "tbz x12, #0, 77f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x26]\n"
      "ld1 { v16.s }[2], [x25]\n"
      "b 77f\n"
      "76:"  // Height 3: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0]\n"
      "ldr s12, [x26, #0]\n"
      "mov x20, #0\n"
      "ldr s16, [x25, #0]\n"
      "77:"  // Height 3: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 80f\n"
      "78:"  // Height 3: full accumulate
      "ldr q8, [x9, #0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "ldr q12, [x26, #0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q16, [x25, #0]\n"
      "ldr q17, [x25, #0x10]\n"
      "ldr q18, [x25, #0x20]\n"
      "ldr q19, [x25, #0x30]\n"
      "b 80f\n"
      "79:"  // Height 3: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "movi v12.16b, #0\n"
      "movi v13.16b, #0\n"
      "movi v14.16b, #0\n"
      "movi v15.16b, #0\n"
      "movi v16.16b, #0\n"
      "movi v17.16b, #0\n"
      "movi v18.16b, #0\n"
      "movi v19.16b, #0\n"
      "80:"  // Height 3: setup done
      "mov x28, #0\n"
      "81:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 82f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 83f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "b 83f\n"
      "82:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "83:"  // Height 3: input setup done
      "cmp x27, #0x8\n"
      "blt 86f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q1, [x25, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0]\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "blt 85f\n"
      "84:"  // Height 3: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "cmp x27, #0x10\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      "ldr q6, [x11, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "ldr q1, [x25, #0]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      "ldr q2, [x24, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "bge 84b\n"
      "85:"  // Height 3: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "add x24, x24, #0x10\n"
      "sub x27, x27, #0x8\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      "86:"  // Height 3: Multiply loop: Main loop skip
      "cbz x27, 88f\n"
      "87:"  // Height 3: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h2, [x24], #0x2\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      "cbnz x27, 87b\n"
      "88:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 81b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x9, #0]\n"
      "add x26, x9, x20, LSL #2\n"
      "prfm pstl1keep, [x26, #0]\n"
      "add x25, x26, x20, LSL #2\n"
      "prfm pstl1keep, [x25, #0]\n"
      "tbz %x[flags], #1, 89f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v21.4s }, [x21]\n"
      "ld1r { v20.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v21.4s\n"
      "fmin v9.4s, v9.4s, v21.4s\n"
      "fmin v10.4s, v10.4s, v21.4s\n"
      "fmin v11.4s, v11.4s, v21.4s\n"
      "fmin v12.4s, v12.4s, v21.4s\n"
      "fmin v13.4s, v13.4s, v21.4s\n"
      "fmin v14.4s, v14.4s, v21.4s\n"
      "fmin v15.4s, v15.4s, v21.4s\n"
      "fmin v16.4s, v16.4s, v21.4s\n"
      "fmin v17.4s, v17.4s, v21.4s\n"
      "fmin v18.4s, v18.4s, v21.4s\n"
      "fmin v19.4s, v19.4s, v21.4s\n"
      "fmax v8.4s, v8.4s, v20.4s\n"
      "fmax v9.4s, v9.4s, v20.4s\n"
      "fmax v10.4s, v10.4s, v20.4s\n"
      "fmax v11.4s, v11.4s, v20.4s\n"
      "fmax v12.4s, v12.4s, v20.4s\n"
      "fmax v13.4s, v13.4s, v20.4s\n"
      "fmax v14.4s, v14.4s, v20.4s\n"
      "fmax v15.4s, v15.4s, v20.4s\n"
      "fmax v16.4s, v16.4s, v20.4s\n"
      "fmax v17.4s, v17.4s, v20.4s\n"
      "fmax v18.4s, v18.4s, v20.4s\n"
      "fmax v19.4s, v19.4s, v20.4s\n"
      "89:"  // Height 3: No activation
      "cmp x12, #0x10\n"
      "bge 98f\n"
      "tbz x12, #3, 93f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v13.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "st1 { v17.4s }, [x25], #0x10\n"
      "tbz x12, #2, 91f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "st1 { v14.4s }, [x26], #0x10\n"
      "st1 { v18.4s }, [x25], #0x10\n"
      "tbz x12, #1, 90f\n"
      "str d11, [x9], #0x8\n"
      "str d15, [x26], #0x8\n"
      "str d19, [x25], #0x8\n"
      "tbz x12, #0, 97f\n"
      "st1 { v11.s }[2], [x9]\n"
      "st1 { v15.s }[2], [x26]\n"
      "st1 { v19.s }[2], [x25]\n"
      "b 97f\n"
      "90:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x12, #0, 97f\n"
      "str s11, [x9, #0]\n"
      "str s15, [x26, #0]\n"
      "str s19, [x25, #0]\n"
      "b 97f\n"
      "91:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x12, #1, 92f\n"
      "str d10, [x9], #0x8\n"
      "str d14, [x26], #0x8\n"
      "str d18, [x25], #0x8\n"
      "tbz x12, #0, 97f\n"
      "st1 { v10.s }[2], [x9]\n"
      "st1 { v14.s }[2], [x26]\n"
      "st1 { v18.s }[2], [x25]\n"
      "b 97f\n"
      "92:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x12, #0, 97f\n"
      "str s10, [x9, #0]\n"
      "str s14, [x26, #0]\n"
      "str s18, [x25, #0]\n"
      "b 97f\n"
      "93:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x12, #2, 95f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "tbz x12, #1, 94f\n"
      "str d9, [x9], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "tbz x12, #0, 97f\n"
      "st1 { v9.s }[2], [x9]\n"
      "st1 { v13.s }[2], [x26]\n"
      "st1 { v17.s }[2], [x25]\n"
      "b 97f\n"
      "94:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x12, #0, 97f\n"
      "str s9, [x9, #0]\n"
      "str s13, [x26, #0]\n"
      "str s17, [x25, #0]\n"
      "b 97f\n"
      "95:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x12, #1, 96f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "tbz x12, #0, 97f\n"
      "st1 { v8.s }[2], [x9]\n"
      "st1 { v12.s }[2], [x26]\n"
      "st1 { v16.s }[2], [x25]\n"
      "b 97f\n"
      "96:"  // Height 3: Partial direct writeback: partial_1_0
      "str s8, [x9, #0]\n"
      "str s12, [x26, #0]\n"
      "str s16, [x25, #0]\n"
      "97:"  // Height 3: Partial direct writeback: Done
      "b 99f\n"
      "98:"  // Height 3: Full writeback
      "str q8, [x9, #0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q12, [x26, #0]\n"
      "str q13, [x26, #0x10]\n"
      "str q14, [x26, #0x20]\n"
      "str q15, [x26, #0x30]\n"
      "str q16, [x25, #0]\n"
      "str q17, [x25, #0x10]\n"
      "str q18, [x25, #0x20]\n"
      "str q19, [x25, #0x30]\n"
      "99:"  // Height 3: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 68b\n"
      "b 200f\n"
      "100:"  // Height 4
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "101:"  // Height 4: Column loop
      "cbz x10, 102f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "b 113f\n"
      "102:"  // Height 4: no bias
      "tbz %x[flags], #0, 112f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "bge 111f\n"
      "tbz x12, #3, 106f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "ld1 { v21.4s }, [x24], #0x10\n"
      "tbz x12, #2, 104f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x26], #0x10\n"
      "ld1 { v18.4s }, [x25], #0x10\n"
      "ld1 { v22.4s }, [x24], #0x10\n"
      "tbz x12, #1, 103f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d19, [x25], #0x8\n"
      "ldr d23, [x24], #0x8\n"
      "tbz x12, #0, 110f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x26]\n"
      "ld1 { v19.s }[2], [x25]\n"
      "ld1 { v23.s }[2], [x24]\n"
      "b 110f\n"
      "103:"  // Height 4: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x12, #0, 110f\n"
      "ldr s11, [x9, #0]\n"
      "ldr s15, [x26, #0]\n"
      "ldr s19, [x25, #0]\n"
      "ldr s23, [x24, #0]\n"
      "b 110f\n"
      "104:"  // Height 4: Partial accumulate: partial_2_8
      "tbz x12, #1, 105f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d18, [x25], #0x8\n"
      "ldr d22, [x24], #0x8\n"
      "tbz x12, #0, 110f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x26]\n"
      "ld1 { v18.s }[2], [x25]\n"
      "ld1 { v22.s }[2], [x24]\n"
      "b 110f\n"
      "105:"  // Height 4: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x12, #0, 110f\n"
      "ldr s10, [x9, #0]\n"
      "ldr s14, [x26, #0]\n"
      "ldr s18, [x25, #0]\n"
      "ldr s22, [x24, #0]\n"
      "b 110f\n"
      "106:"  // Height 4: Partial accumulate: partial_4_0
      "tbz x12, #2, 108f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "tbz x12, #1, 107f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "tbz x12, #0, 110f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x26]\n"
      "ld1 { v17.s }[2], [x25]\n"
      "ld1 { v21.s }[2], [x24]\n"
      "b 110f\n"
      "107:"  // Height 4: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x12, #0, 110f\n"
      "ldr s9, [x9, #0]\n"
      "ldr s13, [x26, #0]\n"
      "ldr s17, [x25, #0]\n"
      "ldr s21, [x24, #0]\n"
      "b 110f\n"
      "108:"  // Height 4: Partial accumulate: partial_2_0
      "tbz x12, #1, 109f\n"
      "ldr d8, [x9], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "tbz x12, #0, 110f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x26]\n"
      "ld1 { v16.s }[2], [x25]\n"
      "ld1 { v20.s }[2], [x24]\n"
      "b 110f\n"
      "109:"  // Height 4: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0]\n"
      "ldr s12, [x26, #0]\n"
      "mov x20, #0\n"
      "ldr s16, [x25, #0]\n"
      "ldr s20, [x24, #0]\n"
      "110:"  // Height 4: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 113f\n"
      "111:"  // Height 4: full accumulate
      "ldr q8, [x9, #0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "ldr q12, [x26, #0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q16, [x25, #0]\n"
      "ldr q17, [x25, #0x10]\n"
      "ldr q18, [x25, #0x20]\n"
      "ldr q19, [x25, #0x30]\n"
      "ldr q20, [x24, #0]\n"
      "ldr q21, [x24, #0x10]\n"
      "ldr q22, [x24, #0x20]\n"
      "ldr q23, [x24, #0x30]\n"
      "b 113f\n"
      "112:"  // Height 4: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "movi v12.16b, #0\n"
      "movi v13.16b, #0\n"
      "movi v14.16b, #0\n"
      "movi v15.16b, #0\n"
      "movi v16.16b, #0\n"
      "movi v17.16b, #0\n"
      "movi v18.16b, #0\n"
      "movi v19.16b, #0\n"
      "movi v20.16b, #0\n"
      "movi v21.16b, #0\n"
      "movi v22.16b, #0\n"
      "movi v23.16b, #0\n"
      "113:"  // Height 4: setup done
      "mov x28, #0\n"
      "114:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 115f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 116f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "b 116f\n"
      "115:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "116:"  // Height 4: input setup done
      "cmp x27, #0x8\n"
      "blt 119f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q1, [x25, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0]\n"
      "ldr q3, [x23, #0]\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "blt 118f\n"
      "117:"  // Height 4: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "add x23, x23, #0x10\n"
      "cmp x27, #0x10\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      "ldr q6, [x11, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "ldr q1, [x25, #0]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      "ldr q2, [x24, #0]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      "ldr q3, [x23, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "bge 117b\n"
      "118:"  // Height 4: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "sub x27, x27, #0x8\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      "119:"  // Height 4: Multiply loop: Main loop skip
      "cbz x27, 121f\n"
      "120:"  // Height 4: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      "cbnz x27, 120b\n"
      "121:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 114b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x9, #0]\n"
      "add x26, x9, x20, LSL #2\n"
      "prfm pstl1keep, [x26, #0]\n"
      "add x25, x26, x20, LSL #2\n"
      "prfm pstl1keep, [x25, #0]\n"
      "add x24, x25, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0]\n"
      "tbz %x[flags], #1, 122f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v25.4s }, [x21]\n"
      "ld1r { v24.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v25.4s\n"
      "fmin v9.4s, v9.4s, v25.4s\n"
      "fmin v10.4s, v10.4s, v25.4s\n"
      "fmin v11.4s, v11.4s, v25.4s\n"
      "fmin v12.4s, v12.4s, v25.4s\n"
      "fmin v13.4s, v13.4s, v25.4s\n"
      "fmin v14.4s, v14.4s, v25.4s\n"
      "fmin v15.4s, v15.4s, v25.4s\n"
      "fmin v16.4s, v16.4s, v25.4s\n"
      "fmin v17.4s, v17.4s, v25.4s\n"
      "fmin v18.4s, v18.4s, v25.4s\n"
      "fmin v19.4s, v19.4s, v25.4s\n"
      "fmin v20.4s, v20.4s, v25.4s\n"
      "fmin v21.4s, v21.4s, v25.4s\n"
      "fmin v22.4s, v22.4s, v25.4s\n"
      "fmin v23.4s, v23.4s, v25.4s\n"
      "fmax v8.4s, v8.4s, v24.4s\n"
      "fmax v9.4s, v9.4s, v24.4s\n"
      "fmax v10.4s, v10.4s, v24.4s\n"
      "fmax v11.4s, v11.4s, v24.4s\n"
      "fmax v12.4s, v12.4s, v24.4s\n"
      "fmax v13.4s, v13.4s, v24.4s\n"
      "fmax v14.4s, v14.4s, v24.4s\n"
      "fmax v15.4s, v15.4s, v24.4s\n"
      "fmax v16.4s, v16.4s, v24.4s\n"
      "fmax v17.4s, v17.4s, v24.4s\n"
      "fmax v18.4s, v18.4s, v24.4s\n"
      "fmax v19.4s, v19.4s, v24.4s\n"
      "fmax v20.4s, v20.4s, v24.4s\n"
      "fmax v21.4s, v21.4s, v24.4s\n"
      "fmax v22.4s, v22.4s, v24.4s\n"
      "fmax v23.4s, v23.4s, v24.4s\n"
      "122:"  // Height 4: No activation
      "cmp x12, #0x10\n"
      "bge 131f\n"
      "tbz x12, #3, 126f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v13.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "st1 { v17.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x24], #0x10\n"
      "st1 { v21.4s }, [x24], #0x10\n"
      "tbz x12, #2, 124f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "st1 { v14.4s }, [x26], #0x10\n"
      "st1 { v18.4s }, [x25], #0x10\n"
      "st1 { v22.4s }, [x24], #0x10\n"
      "tbz x12, #1, 123f\n"
      "str d11, [x9], #0x8\n"
      "str d15, [x26], #0x8\n"
      "str d19, [x25], #0x8\n"
      "str d23, [x24], #0x8\n"
      "tbz x12, #0, 130f\n"
      "st1 { v11.s }[2], [x9]\n"
      "st1 { v15.s }[2], [x26]\n"
      "st1 { v19.s }[2], [x25]\n"
      "st1 { v23.s }[2], [x24]\n"
      "b 130f\n"
      "123:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x12, #0, 130f\n"
      "str s11, [x9, #0]\n"
      "str s15, [x26, #0]\n"
      "str s19, [x25, #0]\n"
      "str s23, [x24, #0]\n"
      "b 130f\n"
      "124:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x12, #1, 125f\n"
      "str d10, [x9], #0x8\n"
      "str d14, [x26], #0x8\n"
      "str d18, [x25], #0x8\n"
      "str d22, [x24], #0x8\n"
      "tbz x12, #0, 130f\n"
      "st1 { v10.s }[2], [x9]\n"
      "st1 { v14.s }[2], [x26]\n"
      "st1 { v18.s }[2], [x25]\n"
      "st1 { v22.s }[2], [x24]\n"
      "b 130f\n"
      "125:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x12, #0, 130f\n"
      "str s10, [x9, #0]\n"
      "str s14, [x26, #0]\n"
      "str s18, [x25, #0]\n"
      "str s22, [x24, #0]\n"
      "b 130f\n"
      "126:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x12, #2, 128f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x24], #0x10\n"
      "tbz x12, #1, 127f\n"
      "str d9, [x9], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "str d21, [x24], #0x8\n"
      "tbz x12, #0, 130f\n"
      "st1 { v9.s }[2], [x9]\n"
      "st1 { v13.s }[2], [x26]\n"
      "st1 { v17.s }[2], [x25]\n"
      "st1 { v21.s }[2], [x24]\n"
      "b 130f\n"
      "127:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x12, #0, 130f\n"
      "str s9, [x9, #0]\n"
      "str s13, [x26, #0]\n"
      "str s17, [x25, #0]\n"
      "str s21, [x24, #0]\n"
      "b 130f\n"
      "128:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x12, #1, 129f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "str d20, [x24], #0x8\n"
      "tbz x12, #0, 130f\n"
      "st1 { v8.s }[2], [x9]\n"
      "st1 { v12.s }[2], [x26]\n"
      "st1 { v16.s }[2], [x25]\n"
      "st1 { v20.s }[2], [x24]\n"
      "b 130f\n"
      "129:"  // Height 4: Partial direct writeback: partial_1_0
      "str s8, [x9, #0]\n"
      "str s12, [x26, #0]\n"
      "str s16, [x25, #0]\n"
      "str s20, [x24, #0]\n"
      "130:"  // Height 4: Partial direct writeback: Done
      "b 132f\n"
      "131:"  // Height 4: Full writeback
      "str q8, [x9, #0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q12, [x26, #0]\n"
      "str q13, [x26, #0x10]\n"
      "str q14, [x26, #0x20]\n"
      "str q15, [x26, #0x30]\n"
      "str q16, [x25, #0]\n"
      "str q17, [x25, #0x10]\n"
      "str q18, [x25, #0x20]\n"
      "str q19, [x25, #0x30]\n"
      "str q20, [x24, #0]\n"
      "str q21, [x24, #0x10]\n"
      "str q22, [x24, #0x20]\n"
      "str q23, [x24, #0x30]\n"
      "132:"  // Height 4: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 101b\n"
      "b 200f\n"
      "133:"  // Height 5
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "134:"  // Height 5: Column loop
      "cbz x10, 135f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v27.16b, v11.16b\n"
      "b 146f\n"
      "135:"  // Height 5: no bias
      "tbz %x[flags], #0, 145f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "bge 144f\n"
      "tbz x12, #3, 139f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v24.4s }, [x23], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "ld1 { v21.4s }, [x24], #0x10\n"
      "ld1 { v25.4s }, [x23], #0x10\n"
      "tbz x12, #2, 137f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x26], #0x10\n"
      "ld1 { v18.4s }, [x25], #0x10\n"
      "ld1 { v22.4s }, [x24], #0x10\n"
      "ld1 { v26.4s }, [x23], #0x10\n"
      "tbz x12, #1, 136f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d19, [x25], #0x8\n"
      "ldr d23, [x24], #0x8\n"
      "ldr d27, [x23], #0x8\n"
      "tbz x12, #0, 143f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x26]\n"
      "ld1 { v19.s }[2], [x25]\n"
      "ld1 { v23.s }[2], [x24]\n"
      "ld1 { v27.s }[2], [x23]\n"
      "b 143f\n"
      "136:"  // Height 5: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x12, #0, 143f\n"
      "ldr s11, [x9, #0]\n"
      "ldr s15, [x26, #0]\n"
      "ldr s19, [x25, #0]\n"
      "ldr s23, [x24, #0]\n"
      "ldr s27, [x23, #0]\n"
      "b 143f\n"
      "137:"  // Height 5: Partial accumulate: partial_2_8
      "tbz x12, #1, 138f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d18, [x25], #0x8\n"
      "ldr d22, [x24], #0x8\n"
      "ldr d26, [x23], #0x8\n"
      "tbz x12, #0, 143f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x26]\n"
      "ld1 { v18.s }[2], [x25]\n"
      "ld1 { v22.s }[2], [x24]\n"
      "ld1 { v26.s }[2], [x23]\n"
      "b 143f\n"
      "138:"  // Height 5: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x12, #0, 143f\n"
      "ldr s10, [x9, #0]\n"
      "ldr s14, [x26, #0]\n"
      "ldr s18, [x25, #0]\n"
      "ldr s22, [x24, #0]\n"
      "ldr s26, [x23, #0]\n"
      "b 143f\n"
      "139:"  // Height 5: Partial accumulate: partial_4_0
      "tbz x12, #2, 141f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v24.4s }, [x23], #0x10\n"
      "tbz x12, #1, 140f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "ldr d25, [x23], #0x8\n"
      "tbz x12, #0, 143f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x26]\n"
      "ld1 { v17.s }[2], [x25]\n"
      "ld1 { v21.s }[2], [x24]\n"
      "ld1 { v25.s }[2], [x23]\n"
      "b 143f\n"
      "140:"  // Height 5: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x12, #0, 143f\n"
      "ldr s9, [x9, #0]\n"
      "ldr s13, [x26, #0]\n"
      "ldr s17, [x25, #0]\n"
      "ldr s21, [x24, #0]\n"
      "ldr s25, [x23, #0]\n"
      "b 143f\n"
      "141:"  // Height 5: Partial accumulate: partial_2_0
      "tbz x12, #1, 142f\n"
      "ldr d8, [x9], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "ldr d24, [x23], #0x8\n"
      "tbz x12, #0, 143f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x26]\n"
      "ld1 { v16.s }[2], [x25]\n"
      "ld1 { v20.s }[2], [x24]\n"
      "ld1 { v24.s }[2], [x23]\n"
      "b 143f\n"
      "142:"  // Height 5: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0]\n"
      "ldr s12, [x26, #0]\n"
      "mov x20, #0\n"
      "ldr s16, [x25, #0]\n"
      "ldr s20, [x24, #0]\n"
      "ldr s24, [x23, #0]\n"
      "143:"  // Height 5: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 146f\n"
      "144:"  // Height 5: full accumulate
      "ldr q8, [x9, #0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "ldr q12, [x26, #0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q16, [x25, #0]\n"
      "ldr q17, [x25, #0x10]\n"
      "ldr q18, [x25, #0x20]\n"
      "ldr q19, [x25, #0x30]\n"
      "ldr q20, [x24, #0]\n"
      "ldr q21, [x24, #0x10]\n"
      "ldr q22, [x24, #0x20]\n"
      "ldr q23, [x24, #0x30]\n"
      "ldr q24, [x23, #0]\n"
      "ldr q25, [x23, #0x10]\n"
      "ldr q26, [x23, #0x20]\n"
      "ldr q27, [x23, #0x30]\n"
      "b 146f\n"
      "145:"  // Height 5: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "movi v12.16b, #0\n"
      "movi v13.16b, #0\n"
      "movi v14.16b, #0\n"
      "movi v15.16b, #0\n"
      "movi v16.16b, #0\n"
      "movi v17.16b, #0\n"
      "movi v18.16b, #0\n"
      "movi v19.16b, #0\n"
      "movi v20.16b, #0\n"
      "movi v21.16b, #0\n"
      "movi v22.16b, #0\n"
      "movi v23.16b, #0\n"
      "movi v24.16b, #0\n"
      "movi v25.16b, #0\n"
      "movi v26.16b, #0\n"
      "movi v27.16b, #0\n"
      "146:"  // Height 5: setup done
      "mov x28, #0\n"
      "147:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 148f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 149f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "b 149f\n"
      "148:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "149:"  // Height 5: input setup done
      "cmp x27, #0x8\n"
      "blt 152f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q1, [x25, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0]\n"
      "ldr q3, [x23, #0]\n"
      "ldr q4, [x22, #0]\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "blt 151f\n"
      "150:"  // Height 5: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      "cmp x27, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x4f9400d8  // fmlal v24.4s, v6.4h, v4.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9480d9  // fmlal2 v25.4s, v6.4h, v4.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x4f9400fa  // fmlal v26.4s, v7.4h, v4.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9480fb  // fmlal2 v27.4s, v7.4h, v4.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x4fa400d8  // fmlal v24.4s, v6.4h, v4.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa480d9  // fmlal2 v25.4s, v6.4h, v4.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x4fa400fa  // fmlal v26.4s, v7.4h, v4.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa480fb  // fmlal2 v27.4s, v7.4h, v4.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x4fb400d8  // fmlal v24.4s, v6.4h, v4.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb480d9  // fmlal2 v25.4s, v6.4h, v4.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x4fb400fa  // fmlal v26.4s, v7.4h, v4.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb480fb  // fmlal2 v27.4s, v7.4h, v4.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x4f8408d8  // fmlal v24.4s, v6.4h, v4.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8488d9  // fmlal2 v25.4s, v6.4h, v4.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x4f8408fa  // fmlal v26.4s, v7.4h, v4.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8488fb  // fmlal2 v27.4s, v7.4h, v4.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x4f9408d8  // fmlal v24.4s, v6.4h, v4.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9488d9  // fmlal2 v25.4s, v6.4h, v4.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x4f9408fa  // fmlal v26.4s, v7.4h, v4.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9488fb  // fmlal2 v27.4s, v7.4h, v4.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x4fa408d8  // fmlal v24.4s, v6.4h, v4.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa488d9  // fmlal2 v25.4s, v6.4h, v4.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x4fa408fa  // fmlal v26.4s, v7.4h, v4.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa488fb  // fmlal2 v27.4s, v7.4h, v4.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x4fb408d8  // fmlal v24.4s, v6.4h, v4.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb488d9  // fmlal2 v25.4s, v6.4h, v4.h[7]\n"
      "ldr q6, [x11, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x4fb408fa  // fmlal v26.4s, v7.4h, v4.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "ldr q1, [x25, #0]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      "ldr q2, [x24, #0]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      "ldr q3, [x23, #0]\n"
      ".inst 0x6fb488fb  // fmlal2 v27.4s, v7.4h, v4.h[7]\n"
      "ldr q4, [x22, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "bge 150b\n"
      "151:"  // Height 5: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "add x22, x22, #0x10\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "sub x27, x27, #0x8\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x4f9400d8  // fmlal v24.4s, v6.4h, v4.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9480d9  // fmlal2 v25.4s, v6.4h, v4.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x4f9400fa  // fmlal v26.4s, v7.4h, v4.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9480fb  // fmlal2 v27.4s, v7.4h, v4.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x4fa400d8  // fmlal v24.4s, v6.4h, v4.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa480d9  // fmlal2 v25.4s, v6.4h, v4.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x4fa400fa  // fmlal v26.4s, v7.4h, v4.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa480fb  // fmlal2 v27.4s, v7.4h, v4.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x4fb400d8  // fmlal v24.4s, v6.4h, v4.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb480d9  // fmlal2 v25.4s, v6.4h, v4.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x4fb400fa  // fmlal v26.4s, v7.4h, v4.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb480fb  // fmlal2 v27.4s, v7.4h, v4.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x4f8408d8  // fmlal v24.4s, v6.4h, v4.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8488d9  // fmlal2 v25.4s, v6.4h, v4.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x4f8408fa  // fmlal v26.4s, v7.4h, v4.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8488fb  // fmlal2 v27.4s, v7.4h, v4.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x4f9408d8  // fmlal v24.4s, v6.4h, v4.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9488d9  // fmlal2 v25.4s, v6.4h, v4.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x4f9408fa  // fmlal v26.4s, v7.4h, v4.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9488fb  // fmlal2 v27.4s, v7.4h, v4.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x4fa408d8  // fmlal v24.4s, v6.4h, v4.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa488d9  // fmlal2 v25.4s, v6.4h, v4.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x4fa408fa  // fmlal v26.4s, v7.4h, v4.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa488fb  // fmlal2 v27.4s, v7.4h, v4.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x4fb408d8  // fmlal v24.4s, v6.4h, v4.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb488d9  // fmlal2 v25.4s, v6.4h, v4.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x4fb408fa  // fmlal v26.4s, v7.4h, v4.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      ".inst 0x6fb488fb  // fmlal2 v27.4s, v7.4h, v4.h[7]\n"
      "152:"  // Height 5: Multiply loop: Main loop skip
      "cbz x27, 154f\n"
      "153:"  // Height 5: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr h4, [x22], #0x2\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      "cbnz x27, 153b\n"
      "154:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 147b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x9, #0]\n"
      "add x26, x9, x20, LSL #2\n"
      "prfm pstl1keep, [x26, #0]\n"
      "add x25, x26, x20, LSL #2\n"
      "prfm pstl1keep, [x25, #0]\n"
      "add x24, x25, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0]\n"
      "add x23, x24, x20, LSL #2\n"
      "prfm pstl1keep, [x23, #0]\n"
      "tbz %x[flags], #1, 155f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v29.4s }, [x21]\n"
      "ld1r { v28.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v29.4s\n"
      "fmin v9.4s, v9.4s, v29.4s\n"
      "fmin v10.4s, v10.4s, v29.4s\n"
      "fmin v11.4s, v11.4s, v29.4s\n"
      "fmin v12.4s, v12.4s, v29.4s\n"
      "fmin v13.4s, v13.4s, v29.4s\n"
      "fmin v14.4s, v14.4s, v29.4s\n"
      "fmin v15.4s, v15.4s, v29.4s\n"
      "fmin v16.4s, v16.4s, v29.4s\n"
      "fmin v17.4s, v17.4s, v29.4s\n"
      "fmin v18.4s, v18.4s, v29.4s\n"
      "fmin v19.4s, v19.4s, v29.4s\n"
      "fmin v20.4s, v20.4s, v29.4s\n"
      "fmin v21.4s, v21.4s, v29.4s\n"
      "fmin v22.4s, v22.4s, v29.4s\n"
      "fmin v23.4s, v23.4s, v29.4s\n"
      "fmin v24.4s, v24.4s, v29.4s\n"
      "fmin v25.4s, v25.4s, v29.4s\n"
      "fmin v26.4s, v26.4s, v29.4s\n"
      "fmin v27.4s, v27.4s, v29.4s\n"
      "fmax v8.4s, v8.4s, v28.4s\n"
      "fmax v9.4s, v9.4s, v28.4s\n"
      "fmax v10.4s, v10.4s, v28.4s\n"
      "fmax v11.4s, v11.4s, v28.4s\n"
      "fmax v12.4s, v12.4s, v28.4s\n"
      "fmax v13.4s, v13.4s, v28.4s\n"
      "fmax v14.4s, v14.4s, v28.4s\n"
      "fmax v15.4s, v15.4s, v28.4s\n"
      "fmax v16.4s, v16.4s, v28.4s\n"
      "fmax v17.4s, v17.4s, v28.4s\n"
      "fmax v18.4s, v18.4s, v28.4s\n"
      "fmax v19.4s, v19.4s, v28.4s\n"
      "fmax v20.4s, v20.4s, v28.4s\n"
      "fmax v21.4s, v21.4s, v28.4s\n"
      "fmax v22.4s, v22.4s, v28.4s\n"
      "fmax v23.4s, v23.4s, v28.4s\n"
      "fmax v24.4s, v24.4s, v28.4s\n"
      "fmax v25.4s, v25.4s, v28.4s\n"
      "fmax v26.4s, v26.4s, v28.4s\n"
      "fmax v27.4s, v27.4s, v28.4s\n"
      "155:"  // Height 5: No activation
      "cmp x12, #0x10\n"
      "bge 164f\n"
      "tbz x12, #3, 159f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v13.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "st1 { v17.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x24], #0x10\n"
      "st1 { v21.4s }, [x24], #0x10\n"
      "st1 { v24.4s }, [x23], #0x10\n"
      "st1 { v25.4s }, [x23], #0x10\n"
      "tbz x12, #2, 157f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "st1 { v14.4s }, [x26], #0x10\n"
      "st1 { v18.4s }, [x25], #0x10\n"
      "st1 { v22.4s }, [x24], #0x10\n"
      "st1 { v26.4s }, [x23], #0x10\n"
      "tbz x12, #1, 156f\n"
      "str d11, [x9], #0x8\n"
      "str d15, [x26], #0x8\n"
      "str d19, [x25], #0x8\n"
      "str d23, [x24], #0x8\n"
      "str d27, [x23], #0x8\n"
      "tbz x12, #0, 163f\n"
      "st1 { v11.s }[2], [x9]\n"
      "st1 { v15.s }[2], [x26]\n"
      "st1 { v19.s }[2], [x25]\n"
      "st1 { v23.s }[2], [x24]\n"
      "st1 { v27.s }[2], [x23]\n"
      "b 163f\n"
      "156:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x12, #0, 163f\n"
      "str s11, [x9, #0]\n"
      "str s15, [x26, #0]\n"
      "str s19, [x25, #0]\n"
      "str s23, [x24, #0]\n"
      "str s27, [x23, #0]\n"
      "b 163f\n"
      "157:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x12, #1, 158f\n"
      "str d10, [x9], #0x8\n"
      "str d14, [x26], #0x8\n"
      "str d18, [x25], #0x8\n"
      "str d22, [x24], #0x8\n"
      "str d26, [x23], #0x8\n"
      "tbz x12, #0, 163f\n"
      "st1 { v10.s }[2], [x9]\n"
      "st1 { v14.s }[2], [x26]\n"
      "st1 { v18.s }[2], [x25]\n"
      "st1 { v22.s }[2], [x24]\n"
      "st1 { v26.s }[2], [x23]\n"
      "b 163f\n"
      "158:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x12, #0, 163f\n"
      "str s10, [x9, #0]\n"
      "str s14, [x26, #0]\n"
      "str s18, [x25, #0]\n"
      "str s22, [x24, #0]\n"
      "str s26, [x23, #0]\n"
      "b 163f\n"
      "159:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x12, #2, 161f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x24], #0x10\n"
      "st1 { v24.4s }, [x23], #0x10\n"
      "tbz x12, #1, 160f\n"
      "str d9, [x9], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "str d21, [x24], #0x8\n"
      "str d25, [x23], #0x8\n"
      "tbz x12, #0, 163f\n"
      "st1 { v9.s }[2], [x9]\n"
      "st1 { v13.s }[2], [x26]\n"
      "st1 { v17.s }[2], [x25]\n"
      "st1 { v21.s }[2], [x24]\n"
      "st1 { v25.s }[2], [x23]\n"
      "b 163f\n"
      "160:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x12, #0, 163f\n"
      "str s9, [x9, #0]\n"
      "str s13, [x26, #0]\n"
      "str s17, [x25, #0]\n"
      "str s21, [x24, #0]\n"
      "str s25, [x23, #0]\n"
      "b 163f\n"
      "161:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x12, #1, 162f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "str d20, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "tbz x12, #0, 163f\n"
      "st1 { v8.s }[2], [x9]\n"
      "st1 { v12.s }[2], [x26]\n"
      "st1 { v16.s }[2], [x25]\n"
      "st1 { v20.s }[2], [x24]\n"
      "st1 { v24.s }[2], [x23]\n"
      "b 163f\n"
      "162:"  // Height 5: Partial direct writeback: partial_1_0
      "str s8, [x9, #0]\n"
      "str s12, [x26, #0]\n"
      "str s16, [x25, #0]\n"
      "str s20, [x24, #0]\n"
      "str s24, [x23, #0]\n"
      "163:"  // Height 5: Partial direct writeback: Done
      "b 165f\n"
      "164:"  // Height 5: Full writeback
      "str q8, [x9, #0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q12, [x26, #0]\n"
      "str q13, [x26, #0x10]\n"
      "str q14, [x26, #0x20]\n"
      "str q15, [x26, #0x30]\n"
      "str q16, [x25, #0]\n"
      "str q17, [x25, #0x10]\n"
      "str q18, [x25, #0x20]\n"
      "str q19, [x25, #0x30]\n"
      "str q20, [x24, #0]\n"
      "str q21, [x24, #0x10]\n"
      "str q22, [x24, #0x20]\n"
      "str q23, [x24, #0x30]\n"
      "str q24, [x23, #0]\n"
      "str q25, [x23, #0x10]\n"
      "str q26, [x23, #0x20]\n"
      "str q27, [x23, #0x30]\n"
      "165:"  // Height 5: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 134b\n"
      "b 200f\n"
      "166:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x18\n"
      "ldr x12, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_bias]]\n"
      "madd x20, x21, x20, x9\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "167:"  // Height 6: Column loop
      "cbz x10, 168f\n"
      "ldr q8, [x10, #0]\n"
      "ldr q9, [x10, #0x10]\n"
      "ldr q10, [x10, #0x20]\n"
      "ldr q11, [x10, #0x30]\n"
      "add x10, x10, #0x40\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v27.16b, v11.16b\n"
      "mov v28.16b, v8.16b\n"
      "mov v29.16b, v9.16b\n"
      "mov v30.16b, v10.16b\n"
      "mov v31.16b, v11.16b\n"
      "b 179f\n"
      "168:"  // Height 6: no bias
      "tbz %x[flags], #0, 178f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x12, #0x10\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "bge 177f\n"
      "tbz x12, #3, 172f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v24.4s }, [x23], #0x10\n"
      "ld1 { v28.4s }, [x22], #0x10\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "ld1 { v21.4s }, [x24], #0x10\n"
      "ld1 { v25.4s }, [x23], #0x10\n"
      "ld1 { v29.4s }, [x22], #0x10\n"
      "tbz x12, #2, 170f\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x26], #0x10\n"
      "ld1 { v18.4s }, [x25], #0x10\n"
      "ld1 { v22.4s }, [x24], #0x10\n"
      "ld1 { v26.4s }, [x23], #0x10\n"
      "ld1 { v30.4s }, [x22], #0x10\n"
      "tbz x12, #1, 169f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d19, [x25], #0x8\n"
      "ldr d23, [x24], #0x8\n"
      "ldr d27, [x23], #0x8\n"
      "ldr d31, [x22], #0x8\n"
      "tbz x12, #0, 176f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x26]\n"
      "ld1 { v19.s }[2], [x25]\n"
      "ld1 { v23.s }[2], [x24]\n"
      "ld1 { v27.s }[2], [x23]\n"
      "ld1 { v31.s }[2], [x22]\n"
      "b 176f\n"
      "169:"  // Height 6: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x12, #0, 176f\n"
      "ldr s11, [x9, #0]\n"
      "ldr s15, [x26, #0]\n"
      "ldr s19, [x25, #0]\n"
      "ldr s23, [x24, #0]\n"
      "ldr s27, [x23, #0]\n"
      "ldr s31, [x22, #0]\n"
      "b 176f\n"
      "170:"  // Height 6: Partial accumulate: partial_2_8
      "tbz x12, #1, 171f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d18, [x25], #0x8\n"
      "ldr d22, [x24], #0x8\n"
      "ldr d26, [x23], #0x8\n"
      "ldr d30, [x22], #0x8\n"
      "tbz x12, #0, 176f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x26]\n"
      "ld1 { v18.s }[2], [x25]\n"
      "ld1 { v22.s }[2], [x24]\n"
      "ld1 { v26.s }[2], [x23]\n"
      "ld1 { v30.s }[2], [x22]\n"
      "b 176f\n"
      "171:"  // Height 6: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x12, #0, 176f\n"
      "ldr s10, [x9, #0]\n"
      "ldr s14, [x26, #0]\n"
      "ldr s18, [x25, #0]\n"
      "ldr s22, [x24, #0]\n"
      "ldr s26, [x23, #0]\n"
      "ldr s30, [x22, #0]\n"
      "b 176f\n"
      "172:"  // Height 6: Partial accumulate: partial_4_0
      "tbz x12, #2, 174f\n"
      "ld1 { v8.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v24.4s }, [x23], #0x10\n"
      "ld1 { v28.4s }, [x22], #0x10\n"
      "tbz x12, #1, 173f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "ldr d25, [x23], #0x8\n"
      "ldr d29, [x22], #0x8\n"
      "tbz x12, #0, 176f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x26]\n"
      "ld1 { v17.s }[2], [x25]\n"
      "ld1 { v21.s }[2], [x24]\n"
      "ld1 { v25.s }[2], [x23]\n"
      "ld1 { v29.s }[2], [x22]\n"
      "b 176f\n"
      "173:"  // Height 6: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x12, #0, 176f\n"
      "ldr s9, [x9, #0]\n"
      "ldr s13, [x26, #0]\n"
      "ldr s17, [x25, #0]\n"
      "ldr s21, [x24, #0]\n"
      "ldr s25, [x23, #0]\n"
      "ldr s29, [x22, #0]\n"
      "b 176f\n"
      "174:"  // Height 6: Partial accumulate: partial_2_0
      "tbz x12, #1, 175f\n"
      "ldr d8, [x9], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "ldr d24, [x23], #0x8\n"
      "ldr d28, [x22], #0x8\n"
      "tbz x12, #0, 176f\n"
      "ld1 { v8.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x26]\n"
      "ld1 { v16.s }[2], [x25]\n"
      "ld1 { v20.s }[2], [x24]\n"
      "ld1 { v24.s }[2], [x23]\n"
      "ld1 { v28.s }[2], [x22]\n"
      "b 176f\n"
      "175:"  // Height 6: Partial accumulate: partial_1_0
      "ldr s8, [x9, #0]\n"
      "ldr s12, [x26, #0]\n"
      "mov x20, #0\n"
      "ldr s16, [x25, #0]\n"
      "ldr s20, [x24, #0]\n"
      "ldr s24, [x23, #0]\n"
      "ldr s28, [x22, #0]\n"
      "176:"  // Height 6: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 179f\n"
      "177:"  // Height 6: full accumulate
      "ldr q8, [x9, #0]\n"
      "ldr q9, [x9, #0x10]\n"
      "ldr q10, [x9, #0x20]\n"
      "ldr q11, [x9, #0x30]\n"
      "ldr q12, [x26, #0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q16, [x25, #0]\n"
      "ldr q17, [x25, #0x10]\n"
      "ldr q18, [x25, #0x20]\n"
      "ldr q19, [x25, #0x30]\n"
      "ldr q20, [x24, #0]\n"
      "ldr q21, [x24, #0x10]\n"
      "ldr q22, [x24, #0x20]\n"
      "ldr q23, [x24, #0x30]\n"
      "ldr q24, [x23, #0]\n"
      "ldr q25, [x23, #0x10]\n"
      "ldr q26, [x23, #0x20]\n"
      "ldr q27, [x23, #0x30]\n"
      "ldr q28, [x22, #0]\n"
      "ldr q29, [x22, #0x10]\n"
      "ldr q30, [x22, #0x20]\n"
      "ldr q31, [x22, #0x30]\n"
      "b 179f\n"
      "178:"  // Height 6: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "movi v12.16b, #0\n"
      "movi v13.16b, #0\n"
      "movi v14.16b, #0\n"
      "movi v15.16b, #0\n"
      "movi v16.16b, #0\n"
      "movi v17.16b, #0\n"
      "movi v18.16b, #0\n"
      "movi v19.16b, #0\n"
      "movi v20.16b, #0\n"
      "movi v21.16b, #0\n"
      "movi v22.16b, #0\n"
      "movi v23.16b, #0\n"
      "movi v24.16b, #0\n"
      "movi v25.16b, #0\n"
      "movi v26.16b, #0\n"
      "movi v27.16b, #0\n"
      "movi v28.16b, #0\n"
      "movi v29.16b, #0\n"
      "movi v30.16b, #0\n"
      "movi v31.16b, #0\n"
      "179:"  // Height 6: setup done
      "mov x28, #0\n"
      "180:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 181f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 182f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "add x21, x21, x20, LSL #1\n"
      "b 182f\n"
      "181:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "add x21, x22, x21, LSL #1\n"
      "182:"  // Height 6: input setup done
      "cmp x27, #0x8\n"
      "blt 185f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q1, [x25, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0]\n"
      "ldr q3, [x23, #0]\n"
      "ldr q4, [x22, #0]\n"
      "ldr q5, [x21, #0]\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "blt 184f\n"
      "183:"  // Height 6: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x4f8500dc  // fmlal v28.4s, v6.4h, v5.h[0]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "add x21, x21, #0x10\n"
      "cmp x27, #0x10\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8580dd  // fmlal2 v29.4s, v6.4h, v5.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x4f8500fe  // fmlal v30.4s, v7.4h, v5.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8580ff  // fmlal2 v31.4s, v7.4h, v5.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x4f9400d8  // fmlal v24.4s, v6.4h, v4.h[1]\n"
      ".inst 0x4f9500dc  // fmlal v28.4s, v6.4h, v5.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9480d9  // fmlal2 v25.4s, v6.4h, v4.h[1]\n"
      ".inst 0x6f9580dd  // fmlal2 v29.4s, v6.4h, v5.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x4f9400fa  // fmlal v26.4s, v7.4h, v4.h[1]\n"
      ".inst 0x4f9500fe  // fmlal v30.4s, v7.4h, v5.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9480fb  // fmlal2 v27.4s, v7.4h, v4.h[1]\n"
      ".inst 0x6f9580ff  // fmlal2 v31.4s, v7.4h, v5.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x4fa400d8  // fmlal v24.4s, v6.4h, v4.h[2]\n"
      ".inst 0x4fa500dc  // fmlal v28.4s, v6.4h, v5.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa480d9  // fmlal2 v25.4s, v6.4h, v4.h[2]\n"
      ".inst 0x6fa580dd  // fmlal2 v29.4s, v6.4h, v5.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x4fa400fa  // fmlal v26.4s, v7.4h, v4.h[2]\n"
      ".inst 0x4fa500fe  // fmlal v30.4s, v7.4h, v5.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa480fb  // fmlal2 v27.4s, v7.4h, v4.h[2]\n"
      ".inst 0x6fa580ff  // fmlal2 v31.4s, v7.4h, v5.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x4fb400d8  // fmlal v24.4s, v6.4h, v4.h[3]\n"
      ".inst 0x4fb500dc  // fmlal v28.4s, v6.4h, v5.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb480d9  // fmlal2 v25.4s, v6.4h, v4.h[3]\n"
      ".inst 0x6fb580dd  // fmlal2 v29.4s, v6.4h, v5.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x4fb400fa  // fmlal v26.4s, v7.4h, v4.h[3]\n"
      ".inst 0x4fb500fe  // fmlal v30.4s, v7.4h, v5.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb480fb  // fmlal2 v27.4s, v7.4h, v4.h[3]\n"
      ".inst 0x6fb580ff  // fmlal2 v31.4s, v7.4h, v5.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x4f8408d8  // fmlal v24.4s, v6.4h, v4.h[4]\n"
      ".inst 0x4f8508dc  // fmlal v28.4s, v6.4h, v5.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8488d9  // fmlal2 v25.4s, v6.4h, v4.h[4]\n"
      ".inst 0x6f8588dd  // fmlal2 v29.4s, v6.4h, v5.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x4f8408fa  // fmlal v26.4s, v7.4h, v4.h[4]\n"
      ".inst 0x4f8508fe  // fmlal v30.4s, v7.4h, v5.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8488fb  // fmlal2 v27.4s, v7.4h, v4.h[4]\n"
      ".inst 0x6f8588ff  // fmlal2 v31.4s, v7.4h, v5.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x4f9408d8  // fmlal v24.4s, v6.4h, v4.h[5]\n"
      ".inst 0x4f9508dc  // fmlal v28.4s, v6.4h, v5.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9488d9  // fmlal2 v25.4s, v6.4h, v4.h[5]\n"
      ".inst 0x6f9588dd  // fmlal2 v29.4s, v6.4h, v5.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x4f9408fa  // fmlal v26.4s, v7.4h, v4.h[5]\n"
      ".inst 0x4f9508fe  // fmlal v30.4s, v7.4h, v5.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9488fb  // fmlal2 v27.4s, v7.4h, v4.h[5]\n"
      ".inst 0x6f9588ff  // fmlal2 v31.4s, v7.4h, v5.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x4fa408d8  // fmlal v24.4s, v6.4h, v4.h[6]\n"
      ".inst 0x4fa508dc  // fmlal v28.4s, v6.4h, v5.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa488d9  // fmlal2 v25.4s, v6.4h, v4.h[6]\n"
      ".inst 0x6fa588dd  // fmlal2 v29.4s, v6.4h, v5.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x4fa408fa  // fmlal v26.4s, v7.4h, v4.h[6]\n"
      ".inst 0x4fa508fe  // fmlal v30.4s, v7.4h, v5.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa488fb  // fmlal2 v27.4s, v7.4h, v4.h[6]\n"
      ".inst 0x6fa588ff  // fmlal2 v31.4s, v7.4h, v5.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x4fb408d8  // fmlal v24.4s, v6.4h, v4.h[7]\n"
      ".inst 0x4fb508dc  // fmlal v28.4s, v6.4h, v5.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb488d9  // fmlal2 v25.4s, v6.4h, v4.h[7]\n"
      ".inst 0x6fb588dd  // fmlal2 v29.4s, v6.4h, v5.h[7]\n"
      "ldr q6, [x11, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x4fb408fa  // fmlal v26.4s, v7.4h, v4.h[7]\n"
      ".inst 0x4fb508fe  // fmlal v30.4s, v7.4h, v5.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "ldr q1, [x25, #0]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      "ldr q2, [x24, #0]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      "ldr q3, [x23, #0]\n"
      ".inst 0x6fb488fb  // fmlal2 v27.4s, v7.4h, v4.h[7]\n"
      "ldr q4, [x22, #0]\n"
      ".inst 0x6fb588ff  // fmlal2 v31.4s, v7.4h, v5.h[7]\n"
      "ldr q5, [x21, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "bge 183b\n"
      "184:"  // Height 6: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x4f8500dc  // fmlal v28.4s, v6.4h, v5.h[0]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8580dd  // fmlal2 v29.4s, v6.4h, v5.h[0]\n"
      "ldr q6, [x11, #0x20]\n"
      "sub x27, x27, #0x8\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x4f8500fe  // fmlal v30.4s, v7.4h, v5.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8580ff  // fmlal2 v31.4s, v7.4h, v5.h[0]\n"
      "ldr q7, [x11, #0x30]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x4f9400d8  // fmlal v24.4s, v6.4h, v4.h[1]\n"
      ".inst 0x4f9500dc  // fmlal v28.4s, v6.4h, v5.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9480d9  // fmlal2 v25.4s, v6.4h, v4.h[1]\n"
      ".inst 0x6f9580dd  // fmlal2 v29.4s, v6.4h, v5.h[1]\n"
      "ldr q6, [x11, #0x40]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x4f9400fa  // fmlal v26.4s, v7.4h, v4.h[1]\n"
      ".inst 0x4f9500fe  // fmlal v30.4s, v7.4h, v5.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9480fb  // fmlal2 v27.4s, v7.4h, v4.h[1]\n"
      ".inst 0x6f9580ff  // fmlal2 v31.4s, v7.4h, v5.h[1]\n"
      "ldr q7, [x11, #0x50]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x4fa400d8  // fmlal v24.4s, v6.4h, v4.h[2]\n"
      ".inst 0x4fa500dc  // fmlal v28.4s, v6.4h, v5.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa480d9  // fmlal2 v25.4s, v6.4h, v4.h[2]\n"
      ".inst 0x6fa580dd  // fmlal2 v29.4s, v6.4h, v5.h[2]\n"
      "ldr q6, [x11, #0x60]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x4fa400fa  // fmlal v26.4s, v7.4h, v4.h[2]\n"
      ".inst 0x4fa500fe  // fmlal v30.4s, v7.4h, v5.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa480fb  // fmlal2 v27.4s, v7.4h, v4.h[2]\n"
      ".inst 0x6fa580ff  // fmlal2 v31.4s, v7.4h, v5.h[2]\n"
      "ldr q7, [x11, #0x70]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x4fb400d8  // fmlal v24.4s, v6.4h, v4.h[3]\n"
      ".inst 0x4fb500dc  // fmlal v28.4s, v6.4h, v5.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb480d9  // fmlal2 v25.4s, v6.4h, v4.h[3]\n"
      ".inst 0x6fb580dd  // fmlal2 v29.4s, v6.4h, v5.h[3]\n"
      "ldr q6, [x11, #0x80]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x4fb400fa  // fmlal v26.4s, v7.4h, v4.h[3]\n"
      ".inst 0x4fb500fe  // fmlal v30.4s, v7.4h, v5.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb480fb  // fmlal2 v27.4s, v7.4h, v4.h[3]\n"
      ".inst 0x6fb580ff  // fmlal2 v31.4s, v7.4h, v5.h[3]\n"
      "ldr q7, [x11, #0x90]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x4f8408d8  // fmlal v24.4s, v6.4h, v4.h[4]\n"
      ".inst 0x4f8508dc  // fmlal v28.4s, v6.4h, v5.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8488d9  // fmlal2 v25.4s, v6.4h, v4.h[4]\n"
      ".inst 0x6f8588dd  // fmlal2 v29.4s, v6.4h, v5.h[4]\n"
      "ldr q6, [x11, #0xa0]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x4f8408fa  // fmlal v26.4s, v7.4h, v4.h[4]\n"
      ".inst 0x4f8508fe  // fmlal v30.4s, v7.4h, v5.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8488fb  // fmlal2 v27.4s, v7.4h, v4.h[4]\n"
      ".inst 0x6f8588ff  // fmlal2 v31.4s, v7.4h, v5.h[4]\n"
      "ldr q7, [x11, #0xb0]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x4f9408d8  // fmlal v24.4s, v6.4h, v4.h[5]\n"
      ".inst 0x4f9508dc  // fmlal v28.4s, v6.4h, v5.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9488d9  // fmlal2 v25.4s, v6.4h, v4.h[5]\n"
      ".inst 0x6f9588dd  // fmlal2 v29.4s, v6.4h, v5.h[5]\n"
      "ldr q6, [x11, #0xc0]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x4f9408fa  // fmlal v26.4s, v7.4h, v4.h[5]\n"
      ".inst 0x4f9508fe  // fmlal v30.4s, v7.4h, v5.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9488fb  // fmlal2 v27.4s, v7.4h, v4.h[5]\n"
      ".inst 0x6f9588ff  // fmlal2 v31.4s, v7.4h, v5.h[5]\n"
      "ldr q7, [x11, #0xd0]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x4fa408d8  // fmlal v24.4s, v6.4h, v4.h[6]\n"
      ".inst 0x4fa508dc  // fmlal v28.4s, v6.4h, v5.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa488d9  // fmlal2 v25.4s, v6.4h, v4.h[6]\n"
      ".inst 0x6fa588dd  // fmlal2 v29.4s, v6.4h, v5.h[6]\n"
      "ldr q6, [x11, #0xe0]\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x4fa408fa  // fmlal v26.4s, v7.4h, v4.h[6]\n"
      ".inst 0x4fa508fe  // fmlal v30.4s, v7.4h, v5.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa488fb  // fmlal2 v27.4s, v7.4h, v4.h[6]\n"
      ".inst 0x6fa588ff  // fmlal2 v31.4s, v7.4h, v5.h[6]\n"
      "ldr q7, [x11, #0xf0]\n"
      "add x11, x11, #0x100\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x4fb408d8  // fmlal v24.4s, v6.4h, v4.h[7]\n"
      ".inst 0x4fb508dc  // fmlal v28.4s, v6.4h, v5.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb488d9  // fmlal2 v25.4s, v6.4h, v4.h[7]\n"
      ".inst 0x6fb588dd  // fmlal2 v29.4s, v6.4h, v5.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x4fb408fa  // fmlal v26.4s, v7.4h, v4.h[7]\n"
      ".inst 0x4fb508fe  // fmlal v30.4s, v7.4h, v5.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      ".inst 0x6fb488fb  // fmlal2 v27.4s, v7.4h, v4.h[7]\n"
      ".inst 0x6fb588ff  // fmlal2 v31.4s, v7.4h, v5.h[7]\n"
      "185:"  // Height 6: Multiply loop: Main loop skip
      "cbz x27, 187f\n"
      "186:"  // Height 6: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr h4, [x22], #0x2\n"
      "ldr h5, [x21], #0x2\n"
      "ldr q6, [x11, #0]\n"
      "ldr q7, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x4f8500dc  // fmlal v28.4s, v6.4h, v5.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8580dd  // fmlal2 v29.4s, v6.4h, v5.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x4f8500fe  // fmlal v30.4s, v7.4h, v5.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8580ff  // fmlal2 v31.4s, v7.4h, v5.h[0]\n"
      "cbnz x27, 186b\n"
      "187:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 180b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x9, #0]\n"
      "add x26, x9, x20, LSL #2\n"
      "prfm pstl1keep, [x26, #0]\n"
      "add x25, x26, x20, LSL #2\n"
      "prfm pstl1keep, [x25, #0]\n"
      "add x24, x25, x20, LSL #2\n"
      "prfm pstl1keep, [x24, #0]\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "prfm pstl1keep, [x23, #0]\n"
      "prfm pstl1keep, [x22, #0]\n"
      "tbz %x[flags], #1, 188f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v1.4s }, [x21]\n"
      "ld1r { v0.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v1.4s\n"
      "fmin v9.4s, v9.4s, v1.4s\n"
      "fmin v10.4s, v10.4s, v1.4s\n"
      "fmin v11.4s, v11.4s, v1.4s\n"
      "fmin v12.4s, v12.4s, v1.4s\n"
      "fmin v13.4s, v13.4s, v1.4s\n"
      "fmin v14.4s, v14.4s, v1.4s\n"
      "fmin v15.4s, v15.4s, v1.4s\n"
      "fmin v16.4s, v16.4s, v1.4s\n"
      "fmin v17.4s, v17.4s, v1.4s\n"
      "fmin v18.4s, v18.4s, v1.4s\n"
      "fmin v19.4s, v19.4s, v1.4s\n"
      "fmin v20.4s, v20.4s, v1.4s\n"
      "fmin v21.4s, v21.4s, v1.4s\n"
      "fmin v22.4s, v22.4s, v1.4s\n"
      "fmin v23.4s, v23.4s, v1.4s\n"
      "fmin v24.4s, v24.4s, v1.4s\n"
      "fmin v25.4s, v25.4s, v1.4s\n"
      "fmin v26.4s, v26.4s, v1.4s\n"
      "fmin v27.4s, v27.4s, v1.4s\n"
      "fmin v28.4s, v28.4s, v1.4s\n"
      "fmin v29.4s, v29.4s, v1.4s\n"
      "fmin v30.4s, v30.4s, v1.4s\n"
      "fmin v31.4s, v31.4s, v1.4s\n"
      "fmax v8.4s, v8.4s, v0.4s\n"
      "fmax v9.4s, v9.4s, v0.4s\n"
      "fmax v10.4s, v10.4s, v0.4s\n"
      "fmax v11.4s, v11.4s, v0.4s\n"
      "fmax v12.4s, v12.4s, v0.4s\n"
      "fmax v13.4s, v13.4s, v0.4s\n"
      "fmax v14.4s, v14.4s, v0.4s\n"
      "fmax v15.4s, v15.4s, v0.4s\n"
      "fmax v16.4s, v16.4s, v0.4s\n"
      "fmax v17.4s, v17.4s, v0.4s\n"
      "fmax v18.4s, v18.4s, v0.4s\n"
      "fmax v19.4s, v19.4s, v0.4s\n"
      "fmax v20.4s, v20.4s, v0.4s\n"
      "fmax v21.4s, v21.4s, v0.4s\n"
      "fmax v22.4s, v22.4s, v0.4s\n"
      "fmax v23.4s, v23.4s, v0.4s\n"
      "fmax v24.4s, v24.4s, v0.4s\n"
      "fmax v25.4s, v25.4s, v0.4s\n"
      "fmax v26.4s, v26.4s, v0.4s\n"
      "fmax v27.4s, v27.4s, v0.4s\n"
      "fmax v28.4s, v28.4s, v0.4s\n"
      "fmax v29.4s, v29.4s, v0.4s\n"
      "fmax v30.4s, v30.4s, v0.4s\n"
      "fmax v31.4s, v31.4s, v0.4s\n"
      "188:"  // Height 6: No activation
      "cmp x12, #0x10\n"
      "bge 197f\n"
      "tbz x12, #3, 192f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v13.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "st1 { v17.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x24], #0x10\n"
      "st1 { v21.4s }, [x24], #0x10\n"
      "st1 { v24.4s }, [x23], #0x10\n"
      "st1 { v25.4s }, [x23], #0x10\n"
      "st1 { v28.4s }, [x22], #0x10\n"
      "st1 { v29.4s }, [x22], #0x10\n"
      "tbz x12, #2, 190f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "st1 { v14.4s }, [x26], #0x10\n"
      "st1 { v18.4s }, [x25], #0x10\n"
      "st1 { v22.4s }, [x24], #0x10\n"
      "st1 { v26.4s }, [x23], #0x10\n"
      "st1 { v30.4s }, [x22], #0x10\n"
      "tbz x12, #1, 189f\n"
      "str d11, [x9], #0x8\n"
      "str d15, [x26], #0x8\n"
      "str d19, [x25], #0x8\n"
      "str d23, [x24], #0x8\n"
      "str d27, [x23], #0x8\n"
      "str d31, [x22], #0x8\n"
      "tbz x12, #0, 196f\n"
      "st1 { v11.s }[2], [x9]\n"
      "st1 { v15.s }[2], [x26]\n"
      "st1 { v19.s }[2], [x25]\n"
      "st1 { v23.s }[2], [x24]\n"
      "st1 { v27.s }[2], [x23]\n"
      "st1 { v31.s }[2], [x22]\n"
      "b 196f\n"
      "189:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x12, #0, 196f\n"
      "str s11, [x9, #0]\n"
      "str s15, [x26, #0]\n"
      "str s19, [x25, #0]\n"
      "str s23, [x24, #0]\n"
      "str s27, [x23, #0]\n"
      "str s31, [x22, #0]\n"
      "b 196f\n"
      "190:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x12, #1, 191f\n"
      "str d10, [x9], #0x8\n"
      "str d14, [x26], #0x8\n"
      "str d18, [x25], #0x8\n"
      "str d22, [x24], #0x8\n"
      "str d26, [x23], #0x8\n"
      "str d30, [x22], #0x8\n"
      "tbz x12, #0, 196f\n"
      "st1 { v10.s }[2], [x9]\n"
      "st1 { v14.s }[2], [x26]\n"
      "st1 { v18.s }[2], [x25]\n"
      "st1 { v22.s }[2], [x24]\n"
      "st1 { v26.s }[2], [x23]\n"
      "st1 { v30.s }[2], [x22]\n"
      "b 196f\n"
      "191:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x12, #0, 196f\n"
      "str s10, [x9, #0]\n"
      "str s14, [x26, #0]\n"
      "str s18, [x25, #0]\n"
      "str s22, [x24, #0]\n"
      "str s26, [x23, #0]\n"
      "str s30, [x22, #0]\n"
      "b 196f\n"
      "192:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x12, #2, 194f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x24], #0x10\n"
      "st1 { v24.4s }, [x23], #0x10\n"
      "st1 { v28.4s }, [x22], #0x10\n"
      "tbz x12, #1, 193f\n"
      "str d9, [x9], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "str d21, [x24], #0x8\n"
      "str d25, [x23], #0x8\n"
      "str d29, [x22], #0x8\n"
      "tbz x12, #0, 196f\n"
      "st1 { v9.s }[2], [x9]\n"
      "st1 { v13.s }[2], [x26]\n"
      "st1 { v17.s }[2], [x25]\n"
      "st1 { v21.s }[2], [x24]\n"
      "st1 { v25.s }[2], [x23]\n"
      "st1 { v29.s }[2], [x22]\n"
      "b 196f\n"
      "193:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x12, #0, 196f\n"
      "str s9, [x9, #0]\n"
      "str s13, [x26, #0]\n"
      "str s17, [x25, #0]\n"
      "str s21, [x24, #0]\n"
      "str s25, [x23, #0]\n"
      "str s29, [x22, #0]\n"
      "b 196f\n"
      "194:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x12, #1, 195f\n"
      "str d8, [x9], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "str d20, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "str d28, [x22], #0x8\n"
      "tbz x12, #0, 196f\n"
      "st1 { v8.s }[2], [x9]\n"
      "st1 { v12.s }[2], [x26]\n"
      "st1 { v16.s }[2], [x25]\n"
      "st1 { v20.s }[2], [x24]\n"
      "st1 { v24.s }[2], [x23]\n"
      "st1 { v28.s }[2], [x22]\n"
      "b 196f\n"
      "195:"  // Height 6: Partial direct writeback: partial_1_0
      "str s8, [x9, #0]\n"
      "str s12, [x26, #0]\n"
      "str s16, [x25, #0]\n"
      "str s20, [x24, #0]\n"
      "str s24, [x23, #0]\n"
      "str s28, [x22, #0]\n"
      "196:"  // Height 6: Partial direct writeback: Done
      "b 198f\n"
      "197:"  // Height 6: Full writeback
      "str q8, [x9, #0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q12, [x26, #0]\n"
      "str q13, [x26, #0x10]\n"
      "str q14, [x26, #0x20]\n"
      "str q15, [x26, #0x30]\n"
      "str q16, [x25, #0]\n"
      "str q17, [x25, #0x10]\n"
      "str q18, [x25, #0x20]\n"
      "str q19, [x25, #0x30]\n"
      "str q20, [x24, #0]\n"
      "str q21, [x24, #0x10]\n"
      "str q22, [x24, #0x20]\n"
      "str q23, [x24, #0x30]\n"
      "str q24, [x23, #0]\n"
      "str q25, [x23, #0x10]\n"
      "str q26, [x23, #0x20]\n"
      "str q27, [x23, #0x30]\n"
      "str q28, [x22, #0]\n"
      "str q29, [x22, #0x10]\n"
      "str q30, [x22, #0x20]\n"
      "str q31, [x22, #0x30]\n"
      "198:"  // Height 6: Writeback done
      "subs x12, x12, #0x10\n"
      "bgt 167b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 200f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 199f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "199:"  // Update direct input
      "mov x20, #0xc\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "200:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_maxval] "I" (offsetof(KernelArgs, maxval)), [offsetof_minval] "I" (offsetof(KernelArgs, minval)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm

#endif // (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(__aarch64__)

