/*
 * Copyright (c) 2025-2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(ARM_COMPUTE_ENABLE_FIXED_FORMAT_KERNELS) && defined(__aarch64__)

#include "arm_gemm/arm_gemm.hpp"
#include "arm_common/internal/utils.hpp"
#include <cassert>
#include <limits>

namespace arm_gemm {

void a64_ffhybrid_fp16fp32fp16_mla_6x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<__fp16> A_arg,
    size_t M, size_t N, const __fp16 *B_ptr, size_t B_stride, IndirectOutputArg<__fp16> output_arg,
    const __fp16 *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const __fp16 *B_ptr = {};
        const __fp16 *cur_B_ptr = {};
        size_t B_stride = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const __fp16 *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    ka.B_stride = B_stride;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 171f\n"
      "cmp %x[M], #0x4\n"
      "bgt 137f\n"
      "beq 103f\n"
      "cmp %x[M], #0x2\n"
      "bgt 69f\n"
      "beq 35f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "ldr x10, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x13, #0x8\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 3f\n"
      "mov x9, x10\n"
      "3:"  // Height 1: B setup done
      "cbz x12, 4f\n"
      "ldr q17, [x12, #0]\n"
      "ldr q16, [x12, #0x10]\n"
      "add x12, x12, #0x20\n"
      "fcvtl v8.4s, v17.4h\n"
      "fcvtl2 v9.4s, v17.8h\n"
      "fcvtl v10.4s, v16.4h\n"
      "fcvtl2 v11.4s, v16.8h\n"
      "b 15f\n"
      "4:"  // Height 1: no bias
      "tbz %x[flags], #0, 14f\n"
      "cmp x13, #0x10\n"
      "bge 13f\n"
      "tbz x13, #3, 8f\n"
      "ld1 { v8.4s }, [x11], #0x10\n"
      "tbz x13, #2, 6f\n"
      "ldr d9, [x11], #0x8\n"
      "tbz x13, #1, 5f\n"
      "ld1 { v9.s }[2], [x11], #0x4\n"
      "mov x20, #0x1c\n"
      "tbz x13, #0, 12f\n"
      "ld1 { v9.h }[6], [x11]\n"
      "b 12f\n"
      "5:"  // Height 1: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x13, #0, 12f\n"
      "ld1 { v9.h }[4], [x11]\n"
      "b 12f\n"
      "6:"  // Height 1: Partial accumulate: partial_2_8
      "tbz x13, #1, 7f\n"
      "ldr s9, [x11], #0x4\n"
      "mov x20, #0x14\n"
      "tbz x13, #0, 12f\n"
      "ld1 { v9.h }[2], [x11]\n"
      "b 12f\n"
      "7:"  // Height 1: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x13, #0, 12f\n"
      "ldr h9, [x11, #0]\n"
      "b 12f\n"
      "8:"  // Height 1: Partial accumulate: partial_4_0
      "tbz x13, #2, 10f\n"
      "ldr d8, [x11], #0x8\n"
      "tbz x13, #1, 9f\n"
      "ld1 { v8.s }[2], [x11], #0x4\n"
      "mov x20, #0xc\n"
      "tbz x13, #0, 12f\n"
      "ld1 { v8.h }[6], [x11]\n"
      "b 12f\n"
      "9:"  // Height 1: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x13, #0, 12f\n"
      "ld1 { v8.h }[4], [x11]\n"
      "b 12f\n"
      "10:"  // Height 1: Partial accumulate: partial_2_0
      "tbz x13, #1, 11f\n"
      "ldr s8, [x11], #0x4\n"
      "mov x20, #0x4\n"
      "tbz x13, #0, 12f\n"
      "ld1 { v8.h }[2], [x11]\n"
      "b 12f\n"
      "11:"  // Height 1: Partial accumulate: partial_1_0
      "ldr h8, [x11, #0]\n"
      "mov x20, #0\n"
      "12:"  // Height 1: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "fcvtl2 v11.4s, v9.8h\n"
      "fcvtl v10.4s, v9.4h\n"
      "fcvtl2 v9.4s, v8.8h\n"
      "fcvtl v8.4s, v8.4h\n"
      "b 15f\n"
      "13:"  // Height 1: full accumulate
      "ldr q17, [x11, #0]\n"
      "ldr q16, [x11, #0x10]\n"
      "fcvtl v8.4s, v17.4h\n"
      "fcvtl2 v9.4s, v17.8h\n"
      "fcvtl v10.4s, v16.4h\n"
      "fcvtl2 v11.4s, v16.8h\n"
      "b 15f\n"
      "14:"  // Height 1: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "15:"  // Height 1: setup done
      "mov x28, #0\n"
      "16:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 17f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "cbnz x28, 18f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "b 18f\n"
      "17:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "18:"  // Height 1: input setup done
      "cmp x27, #0x8\n"
      "blt 21f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q6, [x10, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q7, [x9, #0]\n"
      "blt 20f\n"
      "19:"  // Height 1: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      "sub x27, x27, #0x8\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      "add x26, x26, #0x10\n"
      "cmp x27, #0x10\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      "ldr q6, [x10, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      "ldr q7, [x9, #0]\n"
      "bge 19b\n"
      "20:"  // Height 1: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      "sub x27, x27, #0x8\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "21:"  // Height 1: Multiply loop: Main loop skip
      "cbz x27, 23f\n"
      "22:"  // Height 1: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr q6, [x10, #0]\n"
      "sub x27, x27, #0x1\n"
      "add x10, x10, #0x10\n"
      "ldr q7, [x9, #0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      "cbnz x27, 22b\n"
      "23:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 16b\n"
      "tbz %x[flags], #1, 24f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v17.4s }, [x21]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v17.4s\n"
      "fmin v9.4s, v9.4s, v17.4s\n"
      "fmin v10.4s, v10.4s, v17.4s\n"
      "fmin v11.4s, v11.4s, v17.4s\n"
      "fmax v8.4s, v8.4s, v16.4s\n"
      "fmax v9.4s, v9.4s, v16.4s\n"
      "fmax v10.4s, v10.4s, v16.4s\n"
      "fmax v11.4s, v11.4s, v16.4s\n"
      "24:"  // Height 1: No activation
      "fcvtn v8.4h, v8.4s\n"
      "cmp x13, #0x10\n"
      "fcvtn2 v8.8h, v9.4s\n"
      "fcvtn v9.4h, v10.4s\n"
      "fcvtn2 v9.8h, v11.4s\n"
      "bge 33f\n"
      "tbz x13, #3, 28f\n"
      "st1 { v8.4s }, [x11], #0x10\n"
      "tbz x13, #2, 26f\n"
      "str d9, [x11], #0x8\n"
      "tbz x13, #1, 25f\n"
      "st1 { v9.s }[2], [x11], #0x4\n"
      "tbz x13, #0, 32f\n"
      "st1 { v9.h }[6], [x11]\n"
      "b 32f\n"
      "25:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x13, #0, 32f\n"
      "st1 { v9.h }[4], [x11]\n"
      "b 32f\n"
      "26:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x13, #1, 27f\n"
      "str s9, [x11], #0x4\n"
      "tbz x13, #0, 32f\n"
      "st1 { v9.h }[2], [x11]\n"
      "b 32f\n"
      "27:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x13, #0, 32f\n"
      "str h9, [x11, #0]\n"
      "b 32f\n"
      "28:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x13, #2, 30f\n"
      "str d8, [x11], #0x8\n"
      "tbz x13, #1, 29f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "tbz x13, #0, 32f\n"
      "st1 { v8.h }[6], [x11]\n"
      "b 32f\n"
      "29:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x13, #0, 32f\n"
      "st1 { v8.h }[4], [x11]\n"
      "b 32f\n"
      "30:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x13, #1, 31f\n"
      "str s8, [x11], #0x4\n"
      "tbz x13, #0, 32f\n"
      "st1 { v8.h }[2], [x11]\n"
      "b 32f\n"
      "31:"  // Height 1: Partial direct writeback: partial_1_0
      "str h8, [x11, #0]\n"
      "32:"  // Height 1: Partial direct writeback: Done
      "b 34f\n"
      "33:"  // Height 1: Full writeback
      "str q8, [x11, #0]\n"
      "str q9, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      "34:"  // Height 1: Writeback done
      "subs x13, x13, #0x10\n"
      "bgt 2b\n"
      "b 206f\n"
      "35:"  // Height 2
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "36:"  // Height 2: Column loop
      "ldr x10, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x13, #0x8\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 37f\n"
      "mov x9, x10\n"
      "37:"  // Height 2: B setup done
      "cbz x12, 38f\n"
      "ldr q17, [x12, #0]\n"
      "ldr q16, [x12, #0x10]\n"
      "add x12, x12, #0x20\n"
      "fcvtl v8.4s, v17.4h\n"
      "fcvtl2 v9.4s, v17.8h\n"
      "fcvtl v10.4s, v16.4h\n"
      "fcvtl2 v11.4s, v16.8h\n"
      "mov v12.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "b 49f\n"
      "38:"  // Height 2: no bias
      "tbz %x[flags], #0, 48f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x13, #0x10\n"
      "add x26, x11, x20, LSL #1\n"
      "bge 47f\n"
      "tbz x13, #3, 42f\n"
      "ld1 { v8.4s }, [x11], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "tbz x13, #2, 40f\n"
      "ldr d9, [x11], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "tbz x13, #1, 39f\n"
      "ld1 { v9.s }[2], [x11], #0x4\n"
      "ld1 { v13.s }[2], [x26], #0x4\n"
      "mov x20, #0x1c\n"
      "tbz x13, #0, 46f\n"
      "ld1 { v9.h }[6], [x11]\n"
      "ld1 { v13.h }[6], [x26]\n"
      "b 46f\n"
      "39:"  // Height 2: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x13, #0, 46f\n"
      "ld1 { v9.h }[4], [x11]\n"
      "ld1 { v13.h }[4], [x26]\n"
      "b 46f\n"
      "40:"  // Height 2: Partial accumulate: partial_2_8
      "tbz x13, #1, 41f\n"
      "ldr s9, [x11], #0x4\n"
      "ldr s13, [x26], #0x4\n"
      "mov x20, #0x14\n"
      "tbz x13, #0, 46f\n"
      "ld1 { v9.h }[2], [x11]\n"
      "ld1 { v13.h }[2], [x26]\n"
      "b 46f\n"
      "41:"  // Height 2: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x13, #0, 46f\n"
      "ldr h9, [x11, #0]\n"
      "ldr h13, [x26, #0]\n"
      "b 46f\n"
      "42:"  // Height 2: Partial accumulate: partial_4_0
      "tbz x13, #2, 44f\n"
      "ldr d8, [x11], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "tbz x13, #1, 43f\n"
      "ld1 { v8.s }[2], [x11], #0x4\n"
      "ld1 { v12.s }[2], [x26], #0x4\n"
      "mov x20, #0xc\n"
      "tbz x13, #0, 46f\n"
      "ld1 { v8.h }[6], [x11]\n"
      "ld1 { v12.h }[6], [x26]\n"
      "b 46f\n"
      "43:"  // Height 2: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x13, #0, 46f\n"
      "ld1 { v8.h }[4], [x11]\n"
      "ld1 { v12.h }[4], [x26]\n"
      "b 46f\n"
      "44:"  // Height 2: Partial accumulate: partial_2_0
      "tbz x13, #1, 45f\n"
      "ldr s8, [x11], #0x4\n"
      "ldr s12, [x26], #0x4\n"
      "mov x20, #0x4\n"
      "tbz x13, #0, 46f\n"
      "ld1 { v8.h }[2], [x11]\n"
      "ld1 { v12.h }[2], [x26]\n"
      "b 46f\n"
      "45:"  // Height 2: Partial accumulate: partial_1_0
      "ldr h8, [x11, #0]\n"
      "ldr h12, [x26, #0]\n"
      "mov x20, #0\n"
      "46:"  // Height 2: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "fcvtl2 v11.4s, v9.8h\n"
      "fcvtl v10.4s, v9.4h\n"
      "fcvtl2 v9.4s, v8.8h\n"
      "fcvtl v8.4s, v8.4h\n"
      "fcvtl2 v15.4s, v13.8h\n"
      "fcvtl v14.4s, v13.4h\n"
      "fcvtl2 v13.4s, v12.8h\n"
      "fcvtl v12.4s, v12.4h\n"
      "b 49f\n"
      "47:"  // Height 2: full accumulate
      "ldr q19, [x11, #0]\n"
      "ldr q18, [x11, #0x10]\n"
      "ldr q17, [x26, #0]\n"
      "ldr q16, [x26, #0x10]\n"
      "fcvtl v8.4s, v19.4h\n"
      "fcvtl2 v9.4s, v19.8h\n"
      "fcvtl v10.4s, v18.4h\n"
      "fcvtl2 v11.4s, v18.8h\n"
      "fcvtl v12.4s, v17.4h\n"
      "fcvtl2 v13.4s, v17.8h\n"
      "fcvtl v14.4s, v16.4h\n"
      "fcvtl2 v15.4s, v16.8h\n"
      "b 49f\n"
      "48:"  // Height 2: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "movi v12.16b, #0\n"
      "movi v13.16b, #0\n"
      "movi v14.16b, #0\n"
      "movi v15.16b, #0\n"
      "49:"  // Height 2: setup done
      "mov x28, #0\n"
      "50:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 51f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 52f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "b 52f\n"
      "51:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "52:"  // Height 2: input setup done
      "cmp x27, #0x8\n"
      "blt 55f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q1, [x25, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q6, [x10, #0]\n"
      "ldr q7, [x9, #0]\n"
      "blt 54f\n"
      "53:"  // Height 2: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      "cmp x27, #0x10\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      "ldr q6, [x10, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "ldr q1, [x25, #0]\n"
      "ldr q7, [x9, #0]\n"
      "bge 53b\n"
      "54:"  // Height 2: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "55:"  // Height 2: Multiply loop: Main loop skip
      "cbz x27, 57f\n"
      "56:"  // Height 2: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr q6, [x10, #0]\n"
      "ldr q7, [x9, #0]\n"
      "add x10, x10, #0x10\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      "cbnz x27, 56b\n"
      "57:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 50b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x11, x20, LSL #1\n"
      "tbz %x[flags], #1, 58f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v17.4s }, [x21]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v17.4s\n"
      "fmin v9.4s, v9.4s, v17.4s\n"
      "fmin v10.4s, v10.4s, v17.4s\n"
      "fmin v11.4s, v11.4s, v17.4s\n"
      "fmin v12.4s, v12.4s, v17.4s\n"
      "fmin v13.4s, v13.4s, v17.4s\n"
      "fmin v14.4s, v14.4s, v17.4s\n"
      "fmin v15.4s, v15.4s, v17.4s\n"
      "fmax v8.4s, v8.4s, v16.4s\n"
      "fmax v9.4s, v9.4s, v16.4s\n"
      "fmax v10.4s, v10.4s, v16.4s\n"
      "fmax v11.4s, v11.4s, v16.4s\n"
      "fmax v12.4s, v12.4s, v16.4s\n"
      "fmax v13.4s, v13.4s, v16.4s\n"
      "fmax v14.4s, v14.4s, v16.4s\n"
      "fmax v15.4s, v15.4s, v16.4s\n"
      "58:"  // Height 2: No activation
      "fcvtn v8.4h, v8.4s\n"
      "fcvtn v12.4h, v12.4s\n"
      "cmp x13, #0x10\n"
      "fcvtn2 v8.8h, v9.4s\n"
      "fcvtn v9.4h, v10.4s\n"
      "fcvtn2 v12.8h, v13.4s\n"
      "fcvtn v13.4h, v14.4s\n"
      "fcvtn2 v9.8h, v11.4s\n"
      "fcvtn2 v13.8h, v15.4s\n"
      "bge 67f\n"
      "tbz x13, #3, 62f\n"
      "st1 { v8.4s }, [x11], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "tbz x13, #2, 60f\n"
      "str d9, [x11], #0x8\n"
      "str d13, [x26], #0x8\n"
      "tbz x13, #1, 59f\n"
      "st1 { v9.s }[2], [x11], #0x4\n"
      "st1 { v13.s }[2], [x26], #0x4\n"
      "tbz x13, #0, 66f\n"
      "st1 { v9.h }[6], [x11]\n"
      "st1 { v13.h }[6], [x26]\n"
      "b 66f\n"
      "59:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x13, #0, 66f\n"
      "st1 { v9.h }[4], [x11]\n"
      "st1 { v13.h }[4], [x26]\n"
      "b 66f\n"
      "60:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x13, #1, 61f\n"
      "str s9, [x11], #0x4\n"
      "str s13, [x26], #0x4\n"
      "tbz x13, #0, 66f\n"
      "st1 { v9.h }[2], [x11]\n"
      "st1 { v13.h }[2], [x26]\n"
      "b 66f\n"
      "61:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x13, #0, 66f\n"
      "str h9, [x11, #0]\n"
      "str h13, [x26, #0]\n"
      "b 66f\n"
      "62:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x13, #2, 64f\n"
      "str d8, [x11], #0x8\n"
      "str d12, [x26], #0x8\n"
      "tbz x13, #1, 63f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "st1 { v12.s }[2], [x26], #0x4\n"
      "tbz x13, #0, 66f\n"
      "st1 { v8.h }[6], [x11]\n"
      "st1 { v12.h }[6], [x26]\n"
      "b 66f\n"
      "63:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x13, #0, 66f\n"
      "st1 { v8.h }[4], [x11]\n"
      "st1 { v12.h }[4], [x26]\n"
      "b 66f\n"
      "64:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x13, #1, 65f\n"
      "str s8, [x11], #0x4\n"
      "str s12, [x26], #0x4\n"
      "tbz x13, #0, 66f\n"
      "st1 { v8.h }[2], [x11]\n"
      "st1 { v12.h }[2], [x26]\n"
      "b 66f\n"
      "65:"  // Height 2: Partial direct writeback: partial_1_0
      "str h8, [x11, #0]\n"
      "str h12, [x26, #0]\n"
      "66:"  // Height 2: Partial direct writeback: Done
      "b 68f\n"
      "67:"  // Height 2: Full writeback
      "str q8, [x11, #0]\n"
      "str q9, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      "str q12, [x26, #0]\n"
      "str q13, [x26, #0x10]\n"
      "68:"  // Height 2: Writeback done
      "subs x13, x13, #0x10\n"
      "bgt 36b\n"
      "b 206f\n"
      "69:"  // Height 3
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "70:"  // Height 3: Column loop
      "ldr x10, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x13, #0x8\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 71f\n"
      "mov x9, x10\n"
      "71:"  // Height 3: B setup done
      "cbz x12, 72f\n"
      "ldr q17, [x12, #0]\n"
      "ldr q16, [x12, #0x10]\n"
      "add x12, x12, #0x20\n"
      "fcvtl v8.4s, v17.4h\n"
      "fcvtl2 v9.4s, v17.8h\n"
      "fcvtl v10.4s, v16.4h\n"
      "fcvtl2 v11.4s, v16.8h\n"
      "mov v12.16b, v8.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "b 83f\n"
      "72:"  // Height 3: no bias
      "tbz %x[flags], #0, 82f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x13, #0x10\n"
      "add x26, x11, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "bge 81f\n"
      "tbz x13, #3, 76f\n"
      "ld1 { v8.4s }, [x11], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "tbz x13, #2, 74f\n"
      "ldr d9, [x11], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "tbz x13, #1, 73f\n"
      "ld1 { v9.s }[2], [x11], #0x4\n"
      "ld1 { v13.s }[2], [x26], #0x4\n"
      "mov x20, #0x1c\n"
      "ld1 { v17.s }[2], [x25], #0x4\n"
      "tbz x13, #0, 80f\n"
      "ld1 { v9.h }[6], [x11]\n"
      "ld1 { v13.h }[6], [x26]\n"
      "ld1 { v17.h }[6], [x25]\n"
      "b 80f\n"
      "73:"  // Height 3: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x13, #0, 80f\n"
      "ld1 { v9.h }[4], [x11]\n"
      "ld1 { v13.h }[4], [x26]\n"
      "ld1 { v17.h }[4], [x25]\n"
      "b 80f\n"
      "74:"  // Height 3: Partial accumulate: partial_2_8
      "tbz x13, #1, 75f\n"
      "ldr s9, [x11], #0x4\n"
      "ldr s13, [x26], #0x4\n"
      "mov x20, #0x14\n"
      "ldr s17, [x25], #0x4\n"
      "tbz x13, #0, 80f\n"
      "ld1 { v9.h }[2], [x11]\n"
      "ld1 { v13.h }[2], [x26]\n"
      "ld1 { v17.h }[2], [x25]\n"
      "b 80f\n"
      "75:"  // Height 3: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x13, #0, 80f\n"
      "ldr h9, [x11, #0]\n"
      "ldr h13, [x26, #0]\n"
      "ldr h17, [x25, #0]\n"
      "b 80f\n"
      "76:"  // Height 3: Partial accumulate: partial_4_0
      "tbz x13, #2, 78f\n"
      "ldr d8, [x11], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "tbz x13, #1, 77f\n"
      "ld1 { v8.s }[2], [x11], #0x4\n"
      "ld1 { v12.s }[2], [x26], #0x4\n"
      "mov x20, #0xc\n"
      "ld1 { v16.s }[2], [x25], #0x4\n"
      "tbz x13, #0, 80f\n"
      "ld1 { v8.h }[6], [x11]\n"
      "ld1 { v12.h }[6], [x26]\n"
      "ld1 { v16.h }[6], [x25]\n"
      "b 80f\n"
      "77:"  // Height 3: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x13, #0, 80f\n"
      "ld1 { v8.h }[4], [x11]\n"
      "ld1 { v12.h }[4], [x26]\n"
      "ld1 { v16.h }[4], [x25]\n"
      "b 80f\n"
      "78:"  // Height 3: Partial accumulate: partial_2_0
      "tbz x13, #1, 79f\n"
      "ldr s8, [x11], #0x4\n"
      "ldr s12, [x26], #0x4\n"
      "mov x20, #0x4\n"
      "ldr s16, [x25], #0x4\n"
      "tbz x13, #0, 80f\n"
      "ld1 { v8.h }[2], [x11]\n"
      "ld1 { v12.h }[2], [x26]\n"
      "ld1 { v16.h }[2], [x25]\n"
      "b 80f\n"
      "79:"  // Height 3: Partial accumulate: partial_1_0
      "ldr h8, [x11, #0]\n"
      "ldr h12, [x26, #0]\n"
      "mov x20, #0\n"
      "ldr h16, [x25, #0]\n"
      "80:"  // Height 3: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "fcvtl2 v11.4s, v9.8h\n"
      "fcvtl v10.4s, v9.4h\n"
      "fcvtl2 v9.4s, v8.8h\n"
      "fcvtl v8.4s, v8.4h\n"
      "fcvtl2 v15.4s, v13.8h\n"
      "fcvtl v14.4s, v13.4h\n"
      "fcvtl2 v13.4s, v12.8h\n"
      "fcvtl v12.4s, v12.4h\n"
      "fcvtl2 v19.4s, v17.8h\n"
      "fcvtl v18.4s, v17.4h\n"
      "fcvtl2 v17.4s, v16.8h\n"
      "fcvtl v16.4s, v16.4h\n"
      "b 83f\n"
      "81:"  // Height 3: full accumulate
      "ldr q21, [x11, #0]\n"
      "ldr q20, [x11, #0x10]\n"
      "ldr q18, [x26, #0]\n"
      "ldr q16, [x26, #0x10]\n"
      "ldr q17, [x25, #0]\n"
      "ldr q19, [x25, #0x10]\n"
      "fcvtl v8.4s, v21.4h\n"
      "fcvtl2 v9.4s, v21.8h\n"
      "fcvtl v10.4s, v20.4h\n"
      "fcvtl2 v11.4s, v20.8h\n"
      "fcvtl v12.4s, v18.4h\n"
      "fcvtl2 v13.4s, v18.8h\n"
      "fcvtl v14.4s, v16.4h\n"
      "fcvtl2 v15.4s, v16.8h\n"
      "fcvtl v16.4s, v17.4h\n"
      "fcvtl2 v17.4s, v17.8h\n"
      "fcvtl v18.4s, v19.4h\n"
      "fcvtl2 v19.4s, v19.8h\n"
      "b 83f\n"
      "82:"  // Height 3: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "movi v12.16b, #0\n"
      "movi v13.16b, #0\n"
      "movi v14.16b, #0\n"
      "movi v15.16b, #0\n"
      "movi v16.16b, #0\n"
      "movi v17.16b, #0\n"
      "movi v18.16b, #0\n"
      "movi v19.16b, #0\n"
      "83:"  // Height 3: setup done
      "mov x28, #0\n"
      "84:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 85f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 86f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "b 86f\n"
      "85:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "86:"  // Height 3: input setup done
      "cmp x27, #0x8\n"
      "blt 89f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q1, [x25, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0]\n"
      "ldr q6, [x10, #0]\n"
      "ldr q7, [x9, #0]\n"
      "blt 88f\n"
      "87:"  // Height 3: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "cmp x27, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      "ldr q6, [x10, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "ldr q1, [x25, #0]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      "ldr q2, [x24, #0]\n"
      "ldr q7, [x9, #0]\n"
      "bge 87b\n"
      "88:"  // Height 3: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      "89:"  // Height 3: Multiply loop: Main loop skip
      "cbz x27, 91f\n"
      "90:"  // Height 3: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h2, [x24], #0x2\n"
      "ldr q6, [x10, #0]\n"
      "add x10, x10, #0x10\n"
      "ldr q7, [x9, #0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      "cbnz x27, 90b\n"
      "91:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 84b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x11, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "tbz %x[flags], #1, 92f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v21.4s }, [x21]\n"
      "ld1r { v20.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v21.4s\n"
      "fmin v9.4s, v9.4s, v21.4s\n"
      "fmin v10.4s, v10.4s, v21.4s\n"
      "fmin v11.4s, v11.4s, v21.4s\n"
      "fmin v12.4s, v12.4s, v21.4s\n"
      "fmin v13.4s, v13.4s, v21.4s\n"
      "fmin v14.4s, v14.4s, v21.4s\n"
      "fmin v15.4s, v15.4s, v21.4s\n"
      "fmin v16.4s, v16.4s, v21.4s\n"
      "fmin v17.4s, v17.4s, v21.4s\n"
      "fmin v18.4s, v18.4s, v21.4s\n"
      "fmin v19.4s, v19.4s, v21.4s\n"
      "fmax v8.4s, v8.4s, v20.4s\n"
      "fmax v9.4s, v9.4s, v20.4s\n"
      "fmax v10.4s, v10.4s, v20.4s\n"
      "fmax v11.4s, v11.4s, v20.4s\n"
      "fmax v12.4s, v12.4s, v20.4s\n"
      "fmax v13.4s, v13.4s, v20.4s\n"
      "fmax v14.4s, v14.4s, v20.4s\n"
      "fmax v15.4s, v15.4s, v20.4s\n"
      "fmax v16.4s, v16.4s, v20.4s\n"
      "fmax v17.4s, v17.4s, v20.4s\n"
      "fmax v18.4s, v18.4s, v20.4s\n"
      "fmax v19.4s, v19.4s, v20.4s\n"
      "92:"  // Height 3: No activation
      "fcvtn v8.4h, v8.4s\n"
      "fcvtn v12.4h, v12.4s\n"
      "cmp x13, #0x10\n"
      "fcvtn v16.4h, v16.4s\n"
      "fcvtn2 v8.8h, v9.4s\n"
      "fcvtn v9.4h, v10.4s\n"
      "fcvtn2 v12.8h, v13.4s\n"
      "fcvtn v13.4h, v14.4s\n"
      "fcvtn2 v16.8h, v17.4s\n"
      "fcvtn v17.4h, v18.4s\n"
      "fcvtn2 v9.8h, v11.4s\n"
      "fcvtn2 v13.8h, v15.4s\n"
      "fcvtn2 v17.8h, v19.4s\n"
      "bge 101f\n"
      "tbz x13, #3, 96f\n"
      "st1 { v8.4s }, [x11], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "tbz x13, #2, 94f\n"
      "str d9, [x11], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "tbz x13, #1, 93f\n"
      "st1 { v9.s }[2], [x11], #0x4\n"
      "st1 { v13.s }[2], [x26], #0x4\n"
      "st1 { v17.s }[2], [x25], #0x4\n"
      "tbz x13, #0, 100f\n"
      "st1 { v9.h }[6], [x11]\n"
      "st1 { v13.h }[6], [x26]\n"
      "st1 { v17.h }[6], [x25]\n"
      "b 100f\n"
      "93:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x13, #0, 100f\n"
      "st1 { v9.h }[4], [x11]\n"
      "st1 { v13.h }[4], [x26]\n"
      "st1 { v17.h }[4], [x25]\n"
      "b 100f\n"
      "94:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x13, #1, 95f\n"
      "str s9, [x11], #0x4\n"
      "str s13, [x26], #0x4\n"
      "str s17, [x25], #0x4\n"
      "tbz x13, #0, 100f\n"
      "st1 { v9.h }[2], [x11]\n"
      "st1 { v13.h }[2], [x26]\n"
      "st1 { v17.h }[2], [x25]\n"
      "b 100f\n"
      "95:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x13, #0, 100f\n"
      "str h9, [x11, #0]\n"
      "str h13, [x26, #0]\n"
      "str h17, [x25, #0]\n"
      "b 100f\n"
      "96:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x13, #2, 98f\n"
      "str d8, [x11], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "tbz x13, #1, 97f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "st1 { v12.s }[2], [x26], #0x4\n"
      "st1 { v16.s }[2], [x25], #0x4\n"
      "tbz x13, #0, 100f\n"
      "st1 { v8.h }[6], [x11]\n"
      "st1 { v12.h }[6], [x26]\n"
      "st1 { v16.h }[6], [x25]\n"
      "b 100f\n"
      "97:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x13, #0, 100f\n"
      "st1 { v8.h }[4], [x11]\n"
      "st1 { v12.h }[4], [x26]\n"
      "st1 { v16.h }[4], [x25]\n"
      "b 100f\n"
      "98:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x13, #1, 99f\n"
      "str s8, [x11], #0x4\n"
      "str s12, [x26], #0x4\n"
      "str s16, [x25], #0x4\n"
      "tbz x13, #0, 100f\n"
      "st1 { v8.h }[2], [x11]\n"
      "st1 { v12.h }[2], [x26]\n"
      "st1 { v16.h }[2], [x25]\n"
      "b 100f\n"
      "99:"  // Height 3: Partial direct writeback: partial_1_0
      "str h8, [x11, #0]\n"
      "str h12, [x26, #0]\n"
      "str h16, [x25, #0]\n"
      "100:"  // Height 3: Partial direct writeback: Done
      "b 102f\n"
      "101:"  // Height 3: Full writeback
      "str q8, [x11, #0]\n"
      "str q9, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      "str q12, [x26, #0]\n"
      "str q13, [x26, #0x10]\n"
      "str q16, [x25, #0]\n"
      "str q17, [x25, #0x10]\n"
      "102:"  // Height 3: Writeback done
      "subs x13, x13, #0x10\n"
      "bgt 70b\n"
      "b 206f\n"
      "103:"  // Height 4
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "104:"  // Height 4: Column loop
      "ldr x10, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x13, #0x8\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 105f\n"
      "mov x9, x10\n"
      "105:"  // Height 4: B setup done
      "cbz x12, 106f\n"
      "ldr q17, [x12, #0]\n"
      "ldr q16, [x12, #0x10]\n"
      "add x12, x12, #0x20\n"
      "fcvtl v8.4s, v17.4h\n"
      "fcvtl2 v9.4s, v17.8h\n"
      "fcvtl v10.4s, v16.4h\n"
      "fcvtl2 v11.4s, v16.8h\n"
      "mov v12.16b, v8.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "b 117f\n"
      "106:"  // Height 4: no bias
      "tbz %x[flags], #0, 116f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x13, #0x10\n"
      "add x26, x11, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "bge 115f\n"
      "tbz x13, #3, 110f\n"
      "ld1 { v8.4s }, [x11], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "tbz x13, #2, 108f\n"
      "ldr d9, [x11], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "tbz x13, #1, 107f\n"
      "ld1 { v9.s }[2], [x11], #0x4\n"
      "ld1 { v13.s }[2], [x26], #0x4\n"
      "mov x20, #0x1c\n"
      "ld1 { v17.s }[2], [x25], #0x4\n"
      "ld1 { v21.s }[2], [x24], #0x4\n"
      "tbz x13, #0, 114f\n"
      "ld1 { v9.h }[6], [x11]\n"
      "ld1 { v13.h }[6], [x26]\n"
      "ld1 { v17.h }[6], [x25]\n"
      "ld1 { v21.h }[6], [x24]\n"
      "b 114f\n"
      "107:"  // Height 4: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x13, #0, 114f\n"
      "ld1 { v9.h }[4], [x11]\n"
      "ld1 { v13.h }[4], [x26]\n"
      "ld1 { v17.h }[4], [x25]\n"
      "ld1 { v21.h }[4], [x24]\n"
      "b 114f\n"
      "108:"  // Height 4: Partial accumulate: partial_2_8
      "tbz x13, #1, 109f\n"
      "ldr s9, [x11], #0x4\n"
      "ldr s13, [x26], #0x4\n"
      "mov x20, #0x14\n"
      "ldr s17, [x25], #0x4\n"
      "ldr s21, [x24], #0x4\n"
      "tbz x13, #0, 114f\n"
      "ld1 { v9.h }[2], [x11]\n"
      "ld1 { v13.h }[2], [x26]\n"
      "ld1 { v17.h }[2], [x25]\n"
      "ld1 { v21.h }[2], [x24]\n"
      "b 114f\n"
      "109:"  // Height 4: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x13, #0, 114f\n"
      "ldr h9, [x11, #0]\n"
      "ldr h13, [x26, #0]\n"
      "ldr h17, [x25, #0]\n"
      "ldr h21, [x24, #0]\n"
      "b 114f\n"
      "110:"  // Height 4: Partial accumulate: partial_4_0
      "tbz x13, #2, 112f\n"
      "ldr d8, [x11], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "tbz x13, #1, 111f\n"
      "ld1 { v8.s }[2], [x11], #0x4\n"
      "ld1 { v12.s }[2], [x26], #0x4\n"
      "mov x20, #0xc\n"
      "ld1 { v16.s }[2], [x25], #0x4\n"
      "ld1 { v20.s }[2], [x24], #0x4\n"
      "tbz x13, #0, 114f\n"
      "ld1 { v8.h }[6], [x11]\n"
      "ld1 { v12.h }[6], [x26]\n"
      "ld1 { v16.h }[6], [x25]\n"
      "ld1 { v20.h }[6], [x24]\n"
      "b 114f\n"
      "111:"  // Height 4: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x13, #0, 114f\n"
      "ld1 { v8.h }[4], [x11]\n"
      "ld1 { v12.h }[4], [x26]\n"
      "ld1 { v16.h }[4], [x25]\n"
      "ld1 { v20.h }[4], [x24]\n"
      "b 114f\n"
      "112:"  // Height 4: Partial accumulate: partial_2_0
      "tbz x13, #1, 113f\n"
      "ldr s8, [x11], #0x4\n"
      "ldr s12, [x26], #0x4\n"
      "mov x20, #0x4\n"
      "ldr s16, [x25], #0x4\n"
      "ldr s20, [x24], #0x4\n"
      "tbz x13, #0, 114f\n"
      "ld1 { v8.h }[2], [x11]\n"
      "ld1 { v12.h }[2], [x26]\n"
      "ld1 { v16.h }[2], [x25]\n"
      "ld1 { v20.h }[2], [x24]\n"
      "b 114f\n"
      "113:"  // Height 4: Partial accumulate: partial_1_0
      "ldr h8, [x11, #0]\n"
      "ldr h12, [x26, #0]\n"
      "mov x20, #0\n"
      "ldr h16, [x25, #0]\n"
      "ldr h20, [x24, #0]\n"
      "114:"  // Height 4: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "fcvtl2 v11.4s, v9.8h\n"
      "fcvtl v10.4s, v9.4h\n"
      "fcvtl2 v9.4s, v8.8h\n"
      "fcvtl v8.4s, v8.4h\n"
      "fcvtl2 v15.4s, v13.8h\n"
      "fcvtl v14.4s, v13.4h\n"
      "fcvtl2 v13.4s, v12.8h\n"
      "fcvtl v12.4s, v12.4h\n"
      "fcvtl2 v19.4s, v17.8h\n"
      "fcvtl v18.4s, v17.4h\n"
      "fcvtl2 v17.4s, v16.8h\n"
      "fcvtl v16.4s, v16.4h\n"
      "fcvtl2 v23.4s, v21.8h\n"
      "fcvtl v22.4s, v21.4h\n"
      "fcvtl2 v21.4s, v20.8h\n"
      "fcvtl v20.4s, v20.4h\n"
      "b 117f\n"
      "115:"  // Height 4: full accumulate
      "ldr q22, [x11, #0]\n"
      "ldr q20, [x11, #0x10]\n"
      "ldr q18, [x26, #0]\n"
      "ldr q16, [x26, #0x10]\n"
      "ldr q17, [x25, #0]\n"
      "ldr q19, [x25, #0x10]\n"
      "ldr q21, [x24, #0]\n"
      "ldr q23, [x24, #0x10]\n"
      "fcvtl v8.4s, v22.4h\n"
      "fcvtl2 v9.4s, v22.8h\n"
      "fcvtl v10.4s, v20.4h\n"
      "fcvtl2 v11.4s, v20.8h\n"
      "fcvtl v12.4s, v18.4h\n"
      "fcvtl2 v13.4s, v18.8h\n"
      "fcvtl v14.4s, v16.4h\n"
      "fcvtl2 v15.4s, v16.8h\n"
      "fcvtl v16.4s, v17.4h\n"
      "fcvtl2 v17.4s, v17.8h\n"
      "fcvtl v18.4s, v19.4h\n"
      "fcvtl2 v19.4s, v19.8h\n"
      "fcvtl v20.4s, v21.4h\n"
      "fcvtl2 v21.4s, v21.8h\n"
      "fcvtl v22.4s, v23.4h\n"
      "fcvtl2 v23.4s, v23.8h\n"
      "b 117f\n"
      "116:"  // Height 4: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "movi v12.16b, #0\n"
      "movi v13.16b, #0\n"
      "movi v14.16b, #0\n"
      "movi v15.16b, #0\n"
      "movi v16.16b, #0\n"
      "movi v17.16b, #0\n"
      "movi v18.16b, #0\n"
      "movi v19.16b, #0\n"
      "movi v20.16b, #0\n"
      "movi v21.16b, #0\n"
      "movi v22.16b, #0\n"
      "movi v23.16b, #0\n"
      "117:"  // Height 4: setup done
      "mov x28, #0\n"
      "118:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 119f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 120f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "b 120f\n"
      "119:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "120:"  // Height 4: input setup done
      "cmp x27, #0x8\n"
      "blt 123f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q1, [x25, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0]\n"
      "ldr q3, [x23, #0]\n"
      "ldr q6, [x10, #0]\n"
      "ldr q7, [x9, #0]\n"
      "blt 122f\n"
      "121:"  // Height 4: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "cmp x27, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      "ldr q6, [x10, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "ldr q1, [x25, #0]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      "ldr q2, [x24, #0]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      "ldr q3, [x23, #0]\n"
      "ldr q7, [x9, #0]\n"
      "bge 121b\n"
      "122:"  // Height 4: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      "123:"  // Height 4: Multiply loop: Main loop skip
      "cbz x27, 125f\n"
      "124:"  // Height 4: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr q6, [x10, #0]\n"
      "ldr q7, [x9, #0]\n"
      "add x10, x10, #0x10\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      "cbnz x27, 124b\n"
      "125:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 118b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x11, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "tbz %x[flags], #1, 126f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v25.4s }, [x21]\n"
      "ld1r { v24.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v25.4s\n"
      "fmin v9.4s, v9.4s, v25.4s\n"
      "fmin v10.4s, v10.4s, v25.4s\n"
      "fmin v11.4s, v11.4s, v25.4s\n"
      "fmin v12.4s, v12.4s, v25.4s\n"
      "fmin v13.4s, v13.4s, v25.4s\n"
      "fmin v14.4s, v14.4s, v25.4s\n"
      "fmin v15.4s, v15.4s, v25.4s\n"
      "fmin v16.4s, v16.4s, v25.4s\n"
      "fmin v17.4s, v17.4s, v25.4s\n"
      "fmin v18.4s, v18.4s, v25.4s\n"
      "fmin v19.4s, v19.4s, v25.4s\n"
      "fmin v20.4s, v20.4s, v25.4s\n"
      "fmin v21.4s, v21.4s, v25.4s\n"
      "fmin v22.4s, v22.4s, v25.4s\n"
      "fmin v23.4s, v23.4s, v25.4s\n"
      "fmax v8.4s, v8.4s, v24.4s\n"
      "fmax v9.4s, v9.4s, v24.4s\n"
      "fmax v10.4s, v10.4s, v24.4s\n"
      "fmax v11.4s, v11.4s, v24.4s\n"
      "fmax v12.4s, v12.4s, v24.4s\n"
      "fmax v13.4s, v13.4s, v24.4s\n"
      "fmax v14.4s, v14.4s, v24.4s\n"
      "fmax v15.4s, v15.4s, v24.4s\n"
      "fmax v16.4s, v16.4s, v24.4s\n"
      "fmax v17.4s, v17.4s, v24.4s\n"
      "fmax v18.4s, v18.4s, v24.4s\n"
      "fmax v19.4s, v19.4s, v24.4s\n"
      "fmax v20.4s, v20.4s, v24.4s\n"
      "fmax v21.4s, v21.4s, v24.4s\n"
      "fmax v22.4s, v22.4s, v24.4s\n"
      "fmax v23.4s, v23.4s, v24.4s\n"
      "126:"  // Height 4: No activation
      "fcvtn v8.4h, v8.4s\n"
      "fcvtn v12.4h, v12.4s\n"
      "cmp x13, #0x10\n"
      "fcvtn v16.4h, v16.4s\n"
      "fcvtn v20.4h, v20.4s\n"
      "fcvtn2 v8.8h, v9.4s\n"
      "fcvtn v9.4h, v10.4s\n"
      "fcvtn2 v12.8h, v13.4s\n"
      "fcvtn v13.4h, v14.4s\n"
      "fcvtn2 v16.8h, v17.4s\n"
      "fcvtn v17.4h, v18.4s\n"
      "fcvtn2 v20.8h, v21.4s\n"
      "fcvtn v21.4h, v22.4s\n"
      "fcvtn2 v9.8h, v11.4s\n"
      "fcvtn2 v13.8h, v15.4s\n"
      "fcvtn2 v17.8h, v19.4s\n"
      "fcvtn2 v21.8h, v23.4s\n"
      "bge 135f\n"
      "tbz x13, #3, 130f\n"
      "st1 { v8.4s }, [x11], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x24], #0x10\n"
      "tbz x13, #2, 128f\n"
      "str d9, [x11], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "str d21, [x24], #0x8\n"
      "tbz x13, #1, 127f\n"
      "st1 { v9.s }[2], [x11], #0x4\n"
      "st1 { v13.s }[2], [x26], #0x4\n"
      "st1 { v17.s }[2], [x25], #0x4\n"
      "st1 { v21.s }[2], [x24], #0x4\n"
      "tbz x13, #0, 134f\n"
      "st1 { v9.h }[6], [x11]\n"
      "st1 { v13.h }[6], [x26]\n"
      "st1 { v17.h }[6], [x25]\n"
      "st1 { v21.h }[6], [x24]\n"
      "b 134f\n"
      "127:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x13, #0, 134f\n"
      "st1 { v9.h }[4], [x11]\n"
      "st1 { v13.h }[4], [x26]\n"
      "st1 { v17.h }[4], [x25]\n"
      "st1 { v21.h }[4], [x24]\n"
      "b 134f\n"
      "128:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x13, #1, 129f\n"
      "str s9, [x11], #0x4\n"
      "str s13, [x26], #0x4\n"
      "str s17, [x25], #0x4\n"
      "str s21, [x24], #0x4\n"
      "tbz x13, #0, 134f\n"
      "st1 { v9.h }[2], [x11]\n"
      "st1 { v13.h }[2], [x26]\n"
      "st1 { v17.h }[2], [x25]\n"
      "st1 { v21.h }[2], [x24]\n"
      "b 134f\n"
      "129:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x13, #0, 134f\n"
      "str h9, [x11, #0]\n"
      "str h13, [x26, #0]\n"
      "str h17, [x25, #0]\n"
      "str h21, [x24, #0]\n"
      "b 134f\n"
      "130:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x13, #2, 132f\n"
      "str d8, [x11], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "str d20, [x24], #0x8\n"
      "tbz x13, #1, 131f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "st1 { v12.s }[2], [x26], #0x4\n"
      "st1 { v16.s }[2], [x25], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "tbz x13, #0, 134f\n"
      "st1 { v8.h }[6], [x11]\n"
      "st1 { v12.h }[6], [x26]\n"
      "st1 { v16.h }[6], [x25]\n"
      "st1 { v20.h }[6], [x24]\n"
      "b 134f\n"
      "131:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x13, #0, 134f\n"
      "st1 { v8.h }[4], [x11]\n"
      "st1 { v12.h }[4], [x26]\n"
      "st1 { v16.h }[4], [x25]\n"
      "st1 { v20.h }[4], [x24]\n"
      "b 134f\n"
      "132:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x13, #1, 133f\n"
      "str s8, [x11], #0x4\n"
      "str s12, [x26], #0x4\n"
      "str s16, [x25], #0x4\n"
      "str s20, [x24], #0x4\n"
      "tbz x13, #0, 134f\n"
      "st1 { v8.h }[2], [x11]\n"
      "st1 { v12.h }[2], [x26]\n"
      "st1 { v16.h }[2], [x25]\n"
      "st1 { v20.h }[2], [x24]\n"
      "b 134f\n"
      "133:"  // Height 4: Partial direct writeback: partial_1_0
      "str h8, [x11, #0]\n"
      "str h12, [x26, #0]\n"
      "str h16, [x25, #0]\n"
      "str h20, [x24, #0]\n"
      "134:"  // Height 4: Partial direct writeback: Done
      "b 136f\n"
      "135:"  // Height 4: Full writeback
      "str q8, [x11, #0]\n"
      "str q9, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      "str q12, [x26, #0]\n"
      "str q13, [x26, #0x10]\n"
      "str q16, [x25, #0]\n"
      "str q17, [x25, #0x10]\n"
      "str q20, [x24, #0]\n"
      "str q21, [x24, #0x10]\n"
      "136:"  // Height 4: Writeback done
      "subs x13, x13, #0x10\n"
      "bgt 104b\n"
      "b 206f\n"
      "137:"  // Height 5
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "138:"  // Height 5: Column loop
      "ldr x10, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x13, #0x8\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 139f\n"
      "mov x9, x10\n"
      "139:"  // Height 5: B setup done
      "cbz x12, 140f\n"
      "ldr q17, [x12, #0]\n"
      "ldr q16, [x12, #0x10]\n"
      "add x12, x12, #0x20\n"
      "fcvtl v8.4s, v17.4h\n"
      "fcvtl2 v9.4s, v17.8h\n"
      "fcvtl v10.4s, v16.4h\n"
      "fcvtl2 v11.4s, v16.8h\n"
      "mov v12.16b, v8.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v27.16b, v11.16b\n"
      "b 151f\n"
      "140:"  // Height 5: no bias
      "tbz %x[flags], #0, 150f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x13, #0x10\n"
      "add x26, x11, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "bge 149f\n"
      "tbz x13, #3, 144f\n"
      "ld1 { v8.4s }, [x11], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v24.4s }, [x23], #0x10\n"
      "tbz x13, #2, 142f\n"
      "ldr d9, [x11], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "ldr d25, [x23], #0x8\n"
      "tbz x13, #1, 141f\n"
      "ld1 { v9.s }[2], [x11], #0x4\n"
      "ld1 { v13.s }[2], [x26], #0x4\n"
      "mov x20, #0x1c\n"
      "ld1 { v17.s }[2], [x25], #0x4\n"
      "ld1 { v21.s }[2], [x24], #0x4\n"
      "ld1 { v25.s }[2], [x23], #0x4\n"
      "tbz x13, #0, 148f\n"
      "ld1 { v9.h }[6], [x11]\n"
      "ld1 { v13.h }[6], [x26]\n"
      "ld1 { v17.h }[6], [x25]\n"
      "ld1 { v21.h }[6], [x24]\n"
      "ld1 { v25.h }[6], [x23]\n"
      "b 148f\n"
      "141:"  // Height 5: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x13, #0, 148f\n"
      "ld1 { v9.h }[4], [x11]\n"
      "ld1 { v13.h }[4], [x26]\n"
      "ld1 { v17.h }[4], [x25]\n"
      "ld1 { v21.h }[4], [x24]\n"
      "ld1 { v25.h }[4], [x23]\n"
      "b 148f\n"
      "142:"  // Height 5: Partial accumulate: partial_2_8
      "tbz x13, #1, 143f\n"
      "ldr s9, [x11], #0x4\n"
      "ldr s13, [x26], #0x4\n"
      "mov x20, #0x14\n"
      "ldr s17, [x25], #0x4\n"
      "ldr s21, [x24], #0x4\n"
      "ldr s25, [x23], #0x4\n"
      "tbz x13, #0, 148f\n"
      "ld1 { v9.h }[2], [x11]\n"
      "ld1 { v13.h }[2], [x26]\n"
      "ld1 { v17.h }[2], [x25]\n"
      "ld1 { v21.h }[2], [x24]\n"
      "ld1 { v25.h }[2], [x23]\n"
      "b 148f\n"
      "143:"  // Height 5: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x13, #0, 148f\n"
      "ldr h9, [x11, #0]\n"
      "ldr h13, [x26, #0]\n"
      "ldr h17, [x25, #0]\n"
      "ldr h21, [x24, #0]\n"
      "ldr h25, [x23, #0]\n"
      "b 148f\n"
      "144:"  // Height 5: Partial accumulate: partial_4_0
      "tbz x13, #2, 146f\n"
      "ldr d8, [x11], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "ldr d24, [x23], #0x8\n"
      "tbz x13, #1, 145f\n"
      "ld1 { v8.s }[2], [x11], #0x4\n"
      "ld1 { v12.s }[2], [x26], #0x4\n"
      "mov x20, #0xc\n"
      "ld1 { v16.s }[2], [x25], #0x4\n"
      "ld1 { v20.s }[2], [x24], #0x4\n"
      "ld1 { v24.s }[2], [x23], #0x4\n"
      "tbz x13, #0, 148f\n"
      "ld1 { v8.h }[6], [x11]\n"
      "ld1 { v12.h }[6], [x26]\n"
      "ld1 { v16.h }[6], [x25]\n"
      "ld1 { v20.h }[6], [x24]\n"
      "ld1 { v24.h }[6], [x23]\n"
      "b 148f\n"
      "145:"  // Height 5: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x13, #0, 148f\n"
      "ld1 { v8.h }[4], [x11]\n"
      "ld1 { v12.h }[4], [x26]\n"
      "ld1 { v16.h }[4], [x25]\n"
      "ld1 { v20.h }[4], [x24]\n"
      "ld1 { v24.h }[4], [x23]\n"
      "b 148f\n"
      "146:"  // Height 5: Partial accumulate: partial_2_0
      "tbz x13, #1, 147f\n"
      "ldr s8, [x11], #0x4\n"
      "ldr s12, [x26], #0x4\n"
      "mov x20, #0x4\n"
      "ldr s16, [x25], #0x4\n"
      "ldr s20, [x24], #0x4\n"
      "ldr s24, [x23], #0x4\n"
      "tbz x13, #0, 148f\n"
      "ld1 { v8.h }[2], [x11]\n"
      "ld1 { v12.h }[2], [x26]\n"
      "ld1 { v16.h }[2], [x25]\n"
      "ld1 { v20.h }[2], [x24]\n"
      "ld1 { v24.h }[2], [x23]\n"
      "b 148f\n"
      "147:"  // Height 5: Partial accumulate: partial_1_0
      "ldr h8, [x11, #0]\n"
      "ldr h12, [x26, #0]\n"
      "mov x20, #0\n"
      "ldr h16, [x25, #0]\n"
      "ldr h20, [x24, #0]\n"
      "ldr h24, [x23, #0]\n"
      "148:"  // Height 5: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "fcvtl2 v11.4s, v9.8h\n"
      "fcvtl v10.4s, v9.4h\n"
      "fcvtl2 v9.4s, v8.8h\n"
      "fcvtl v8.4s, v8.4h\n"
      "fcvtl2 v15.4s, v13.8h\n"
      "fcvtl v14.4s, v13.4h\n"
      "fcvtl2 v13.4s, v12.8h\n"
      "fcvtl v12.4s, v12.4h\n"
      "fcvtl2 v19.4s, v17.8h\n"
      "fcvtl v18.4s, v17.4h\n"
      "fcvtl2 v17.4s, v16.8h\n"
      "fcvtl v16.4s, v16.4h\n"
      "fcvtl2 v23.4s, v21.8h\n"
      "fcvtl v22.4s, v21.4h\n"
      "fcvtl2 v21.4s, v20.8h\n"
      "fcvtl v20.4s, v20.4h\n"
      "fcvtl2 v27.4s, v25.8h\n"
      "fcvtl v26.4s, v25.4h\n"
      "fcvtl2 v25.4s, v24.8h\n"
      "fcvtl v24.4s, v24.4h\n"
      "b 151f\n"
      "149:"  // Height 5: full accumulate
      "ldr q22, [x11, #0]\n"
      "ldr q20, [x11, #0x10]\n"
      "ldr q18, [x26, #0]\n"
      "ldr q16, [x26, #0x10]\n"
      "ldr q17, [x25, #0]\n"
      "ldr q19, [x25, #0x10]\n"
      "ldr q21, [x24, #0]\n"
      "ldr q23, [x24, #0x10]\n"
      "fcvtl v8.4s, v22.4h\n"
      "fcvtl2 v9.4s, v22.8h\n"
      "ldr q25, [x23, #0]\n"
      "ldr q27, [x23, #0x10]\n"
      "fcvtl v10.4s, v20.4h\n"
      "fcvtl2 v11.4s, v20.8h\n"
      "fcvtl v12.4s, v18.4h\n"
      "fcvtl2 v13.4s, v18.8h\n"
      "fcvtl v14.4s, v16.4h\n"
      "fcvtl2 v15.4s, v16.8h\n"
      "fcvtl v16.4s, v17.4h\n"
      "fcvtl2 v17.4s, v17.8h\n"
      "fcvtl v18.4s, v19.4h\n"
      "fcvtl2 v19.4s, v19.8h\n"
      "fcvtl v20.4s, v21.4h\n"
      "fcvtl2 v21.4s, v21.8h\n"
      "fcvtl v22.4s, v23.4h\n"
      "fcvtl2 v23.4s, v23.8h\n"
      "fcvtl v24.4s, v25.4h\n"
      "fcvtl2 v25.4s, v25.8h\n"
      "fcvtl v26.4s, v27.4h\n"
      "fcvtl2 v27.4s, v27.8h\n"
      "b 151f\n"
      "150:"  // Height 5: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "movi v12.16b, #0\n"
      "movi v13.16b, #0\n"
      "movi v14.16b, #0\n"
      "movi v15.16b, #0\n"
      "movi v16.16b, #0\n"
      "movi v17.16b, #0\n"
      "movi v18.16b, #0\n"
      "movi v19.16b, #0\n"
      "movi v20.16b, #0\n"
      "movi v21.16b, #0\n"
      "movi v22.16b, #0\n"
      "movi v23.16b, #0\n"
      "movi v24.16b, #0\n"
      "movi v25.16b, #0\n"
      "movi v26.16b, #0\n"
      "movi v27.16b, #0\n"
      "151:"  // Height 5: setup done
      "mov x28, #0\n"
      "152:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 153f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 154f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "b 154f\n"
      "153:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "154:"  // Height 5: input setup done
      "cmp x27, #0x8\n"
      "blt 157f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q1, [x25, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0]\n"
      "ldr q3, [x23, #0]\n"
      "ldr q4, [x22, #0]\n"
      "ldr q6, [x10, #0]\n"
      "ldr q7, [x9, #0]\n"
      "blt 156f\n"
      "155:"  // Height 5: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "cmp x27, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x4f9400d8  // fmlal v24.4s, v6.4h, v4.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9480d9  // fmlal2 v25.4s, v6.4h, v4.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x4f9400fa  // fmlal v26.4s, v7.4h, v4.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9480fb  // fmlal2 v27.4s, v7.4h, v4.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x4fa400d8  // fmlal v24.4s, v6.4h, v4.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa480d9  // fmlal2 v25.4s, v6.4h, v4.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x4fa400fa  // fmlal v26.4s, v7.4h, v4.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa480fb  // fmlal2 v27.4s, v7.4h, v4.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x4fb400d8  // fmlal v24.4s, v6.4h, v4.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb480d9  // fmlal2 v25.4s, v6.4h, v4.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x4fb400fa  // fmlal v26.4s, v7.4h, v4.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb480fb  // fmlal2 v27.4s, v7.4h, v4.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x4f8408d8  // fmlal v24.4s, v6.4h, v4.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8488d9  // fmlal2 v25.4s, v6.4h, v4.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x4f8408fa  // fmlal v26.4s, v7.4h, v4.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8488fb  // fmlal2 v27.4s, v7.4h, v4.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x4f9408d8  // fmlal v24.4s, v6.4h, v4.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9488d9  // fmlal2 v25.4s, v6.4h, v4.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x4f9408fa  // fmlal v26.4s, v7.4h, v4.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9488fb  // fmlal2 v27.4s, v7.4h, v4.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x4fa408d8  // fmlal v24.4s, v6.4h, v4.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa488d9  // fmlal2 v25.4s, v6.4h, v4.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x4fa408fa  // fmlal v26.4s, v7.4h, v4.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa488fb  // fmlal2 v27.4s, v7.4h, v4.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x4fb408d8  // fmlal v24.4s, v6.4h, v4.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb488d9  // fmlal2 v25.4s, v6.4h, v4.h[7]\n"
      "ldr q6, [x10, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x4fb408fa  // fmlal v26.4s, v7.4h, v4.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "ldr q1, [x25, #0]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      "ldr q2, [x24, #0]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      "ldr q3, [x23, #0]\n"
      ".inst 0x6fb488fb  // fmlal2 v27.4s, v7.4h, v4.h[7]\n"
      "ldr q4, [x22, #0]\n"
      "ldr q7, [x9, #0]\n"
      "bge 155b\n"
      "156:"  // Height 5: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x4f9400d8  // fmlal v24.4s, v6.4h, v4.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9480d9  // fmlal2 v25.4s, v6.4h, v4.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x4f9400fa  // fmlal v26.4s, v7.4h, v4.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9480fb  // fmlal2 v27.4s, v7.4h, v4.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x4fa400d8  // fmlal v24.4s, v6.4h, v4.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa480d9  // fmlal2 v25.4s, v6.4h, v4.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x4fa400fa  // fmlal v26.4s, v7.4h, v4.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa480fb  // fmlal2 v27.4s, v7.4h, v4.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x4fb400d8  // fmlal v24.4s, v6.4h, v4.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb480d9  // fmlal2 v25.4s, v6.4h, v4.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x4fb400fa  // fmlal v26.4s, v7.4h, v4.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb480fb  // fmlal2 v27.4s, v7.4h, v4.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x4f8408d8  // fmlal v24.4s, v6.4h, v4.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8488d9  // fmlal2 v25.4s, v6.4h, v4.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x4f8408fa  // fmlal v26.4s, v7.4h, v4.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8488fb  // fmlal2 v27.4s, v7.4h, v4.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x4f9408d8  // fmlal v24.4s, v6.4h, v4.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9488d9  // fmlal2 v25.4s, v6.4h, v4.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x4f9408fa  // fmlal v26.4s, v7.4h, v4.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9488fb  // fmlal2 v27.4s, v7.4h, v4.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x4fa408d8  // fmlal v24.4s, v6.4h, v4.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa488d9  // fmlal2 v25.4s, v6.4h, v4.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x4fa408fa  // fmlal v26.4s, v7.4h, v4.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa488fb  // fmlal2 v27.4s, v7.4h, v4.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x4fb408d8  // fmlal v24.4s, v6.4h, v4.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb488d9  // fmlal2 v25.4s, v6.4h, v4.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x4fb408fa  // fmlal v26.4s, v7.4h, v4.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      ".inst 0x6fb488fb  // fmlal2 v27.4s, v7.4h, v4.h[7]\n"
      "157:"  // Height 5: Multiply loop: Main loop skip
      "cbz x27, 159f\n"
      "158:"  // Height 5: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr h4, [x22], #0x2\n"
      "ldr q6, [x10, #0]\n"
      "add x10, x10, #0x10\n"
      "ldr q7, [x9, #0]\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      "cbnz x27, 158b\n"
      "159:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 152b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x11, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "tbz %x[flags], #1, 160f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v29.4s }, [x21]\n"
      "ld1r { v28.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v29.4s\n"
      "fmin v9.4s, v9.4s, v29.4s\n"
      "fmin v10.4s, v10.4s, v29.4s\n"
      "fmin v11.4s, v11.4s, v29.4s\n"
      "fmin v12.4s, v12.4s, v29.4s\n"
      "fmin v13.4s, v13.4s, v29.4s\n"
      "fmin v14.4s, v14.4s, v29.4s\n"
      "fmin v15.4s, v15.4s, v29.4s\n"
      "fmin v16.4s, v16.4s, v29.4s\n"
      "fmin v17.4s, v17.4s, v29.4s\n"
      "fmin v18.4s, v18.4s, v29.4s\n"
      "fmin v19.4s, v19.4s, v29.4s\n"
      "fmin v20.4s, v20.4s, v29.4s\n"
      "fmin v21.4s, v21.4s, v29.4s\n"
      "fmin v22.4s, v22.4s, v29.4s\n"
      "fmin v23.4s, v23.4s, v29.4s\n"
      "fmin v24.4s, v24.4s, v29.4s\n"
      "fmin v25.4s, v25.4s, v29.4s\n"
      "fmin v26.4s, v26.4s, v29.4s\n"
      "fmin v27.4s, v27.4s, v29.4s\n"
      "fmax v8.4s, v8.4s, v28.4s\n"
      "fmax v9.4s, v9.4s, v28.4s\n"
      "fmax v10.4s, v10.4s, v28.4s\n"
      "fmax v11.4s, v11.4s, v28.4s\n"
      "fmax v12.4s, v12.4s, v28.4s\n"
      "fmax v13.4s, v13.4s, v28.4s\n"
      "fmax v14.4s, v14.4s, v28.4s\n"
      "fmax v15.4s, v15.4s, v28.4s\n"
      "fmax v16.4s, v16.4s, v28.4s\n"
      "fmax v17.4s, v17.4s, v28.4s\n"
      "fmax v18.4s, v18.4s, v28.4s\n"
      "fmax v19.4s, v19.4s, v28.4s\n"
      "fmax v20.4s, v20.4s, v28.4s\n"
      "fmax v21.4s, v21.4s, v28.4s\n"
      "fmax v22.4s, v22.4s, v28.4s\n"
      "fmax v23.4s, v23.4s, v28.4s\n"
      "fmax v24.4s, v24.4s, v28.4s\n"
      "fmax v25.4s, v25.4s, v28.4s\n"
      "fmax v26.4s, v26.4s, v28.4s\n"
      "fmax v27.4s, v27.4s, v28.4s\n"
      "160:"  // Height 5: No activation
      "fcvtn v8.4h, v8.4s\n"
      "fcvtn v12.4h, v12.4s\n"
      "cmp x13, #0x10\n"
      "fcvtn v16.4h, v16.4s\n"
      "fcvtn v20.4h, v20.4s\n"
      "fcvtn v24.4h, v24.4s\n"
      "fcvtn2 v8.8h, v9.4s\n"
      "fcvtn v9.4h, v10.4s\n"
      "fcvtn2 v12.8h, v13.4s\n"
      "fcvtn v13.4h, v14.4s\n"
      "fcvtn2 v16.8h, v17.4s\n"
      "fcvtn v17.4h, v18.4s\n"
      "fcvtn2 v20.8h, v21.4s\n"
      "fcvtn v21.4h, v22.4s\n"
      "fcvtn2 v24.8h, v25.4s\n"
      "fcvtn v25.4h, v26.4s\n"
      "fcvtn2 v9.8h, v11.4s\n"
      "fcvtn2 v13.8h, v15.4s\n"
      "fcvtn2 v17.8h, v19.4s\n"
      "fcvtn2 v21.8h, v23.4s\n"
      "fcvtn2 v25.8h, v27.4s\n"
      "bge 169f\n"
      "tbz x13, #3, 164f\n"
      "st1 { v8.4s }, [x11], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x24], #0x10\n"
      "st1 { v24.4s }, [x23], #0x10\n"
      "tbz x13, #2, 162f\n"
      "str d9, [x11], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "str d21, [x24], #0x8\n"
      "str d25, [x23], #0x8\n"
      "tbz x13, #1, 161f\n"
      "st1 { v9.s }[2], [x11], #0x4\n"
      "st1 { v13.s }[2], [x26], #0x4\n"
      "st1 { v17.s }[2], [x25], #0x4\n"
      "st1 { v21.s }[2], [x24], #0x4\n"
      "st1 { v25.s }[2], [x23], #0x4\n"
      "tbz x13, #0, 168f\n"
      "st1 { v9.h }[6], [x11]\n"
      "st1 { v13.h }[6], [x26]\n"
      "st1 { v17.h }[6], [x25]\n"
      "st1 { v21.h }[6], [x24]\n"
      "st1 { v25.h }[6], [x23]\n"
      "b 168f\n"
      "161:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x13, #0, 168f\n"
      "st1 { v9.h }[4], [x11]\n"
      "st1 { v13.h }[4], [x26]\n"
      "st1 { v17.h }[4], [x25]\n"
      "st1 { v21.h }[4], [x24]\n"
      "st1 { v25.h }[4], [x23]\n"
      "b 168f\n"
      "162:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x13, #1, 163f\n"
      "str s9, [x11], #0x4\n"
      "str s13, [x26], #0x4\n"
      "str s17, [x25], #0x4\n"
      "str s21, [x24], #0x4\n"
      "str s25, [x23], #0x4\n"
      "tbz x13, #0, 168f\n"
      "st1 { v9.h }[2], [x11]\n"
      "st1 { v13.h }[2], [x26]\n"
      "st1 { v17.h }[2], [x25]\n"
      "st1 { v21.h }[2], [x24]\n"
      "st1 { v25.h }[2], [x23]\n"
      "b 168f\n"
      "163:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x13, #0, 168f\n"
      "str h9, [x11, #0]\n"
      "str h13, [x26, #0]\n"
      "str h17, [x25, #0]\n"
      "str h21, [x24, #0]\n"
      "str h25, [x23, #0]\n"
      "b 168f\n"
      "164:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x13, #2, 166f\n"
      "str d8, [x11], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "str d20, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "tbz x13, #1, 165f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "st1 { v12.s }[2], [x26], #0x4\n"
      "st1 { v16.s }[2], [x25], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "st1 { v24.s }[2], [x23], #0x4\n"
      "tbz x13, #0, 168f\n"
      "st1 { v8.h }[6], [x11]\n"
      "st1 { v12.h }[6], [x26]\n"
      "st1 { v16.h }[6], [x25]\n"
      "st1 { v20.h }[6], [x24]\n"
      "st1 { v24.h }[6], [x23]\n"
      "b 168f\n"
      "165:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x13, #0, 168f\n"
      "st1 { v8.h }[4], [x11]\n"
      "st1 { v12.h }[4], [x26]\n"
      "st1 { v16.h }[4], [x25]\n"
      "st1 { v20.h }[4], [x24]\n"
      "st1 { v24.h }[4], [x23]\n"
      "b 168f\n"
      "166:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x13, #1, 167f\n"
      "str s8, [x11], #0x4\n"
      "str s12, [x26], #0x4\n"
      "str s16, [x25], #0x4\n"
      "str s20, [x24], #0x4\n"
      "str s24, [x23], #0x4\n"
      "tbz x13, #0, 168f\n"
      "st1 { v8.h }[2], [x11]\n"
      "st1 { v12.h }[2], [x26]\n"
      "st1 { v16.h }[2], [x25]\n"
      "st1 { v20.h }[2], [x24]\n"
      "st1 { v24.h }[2], [x23]\n"
      "b 168f\n"
      "167:"  // Height 5: Partial direct writeback: partial_1_0
      "str h8, [x11, #0]\n"
      "str h12, [x26, #0]\n"
      "str h16, [x25, #0]\n"
      "str h20, [x24, #0]\n"
      "str h24, [x23, #0]\n"
      "168:"  // Height 5: Partial direct writeback: Done
      "b 170f\n"
      "169:"  // Height 5: Full writeback
      "str q8, [x11, #0]\n"
      "str q9, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      "str q12, [x26, #0]\n"
      "str q13, [x26, #0x10]\n"
      "str q16, [x25, #0]\n"
      "str q17, [x25, #0x10]\n"
      "str q20, [x24, #0]\n"
      "str q21, [x24, #0x10]\n"
      "str q24, [x23, #0]\n"
      "str q25, [x23, #0x10]\n"
      "170:"  // Height 5: Writeback done
      "subs x13, x13, #0x10\n"
      "bgt 138b\n"
      "b 206f\n"
      "171:"  // Height 6
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "mov x21, #0xc\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "madd x21, x20, x21, x11\n"
      "str x21, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "172:"  // Height 6: Column loop
      "ldr x10, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cmp x13, #0x8\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 173f\n"
      "mov x9, x10\n"
      "173:"  // Height 6: B setup done
      "cbz x12, 174f\n"
      "ldr q17, [x12, #0]\n"
      "ldr q16, [x12, #0x10]\n"
      "add x12, x12, #0x20\n"
      "fcvtl v8.4s, v17.4h\n"
      "fcvtl2 v9.4s, v17.8h\n"
      "fcvtl v10.4s, v16.4h\n"
      "fcvtl2 v11.4s, v16.8h\n"
      "mov v12.16b, v8.16b\n"
      "mov v16.16b, v8.16b\n"
      "mov v20.16b, v8.16b\n"
      "mov v13.16b, v9.16b\n"
      "mov v14.16b, v10.16b\n"
      "mov v15.16b, v11.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v27.16b, v11.16b\n"
      "mov v28.16b, v8.16b\n"
      "mov v29.16b, v9.16b\n"
      "mov v30.16b, v10.16b\n"
      "mov v31.16b, v11.16b\n"
      "b 185f\n"
      "174:"  // Height 6: no bias
      "tbz %x[flags], #0, 184f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x13, #0x10\n"
      "add x26, x11, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "add x22, x23, x20, LSL #1\n"
      "bge 183f\n"
      "tbz x13, #3, 178f\n"
      "ld1 { v8.4s }, [x11], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v16.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v24.4s }, [x23], #0x10\n"
      "ld1 { v28.4s }, [x22], #0x10\n"
      "tbz x13, #2, 176f\n"
      "ldr d9, [x11], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "ldr d25, [x23], #0x8\n"
      "ldr d29, [x22], #0x8\n"
      "tbz x13, #1, 175f\n"
      "ld1 { v9.s }[2], [x11], #0x4\n"
      "ld1 { v13.s }[2], [x26], #0x4\n"
      "mov x20, #0x1c\n"
      "ld1 { v17.s }[2], [x25], #0x4\n"
      "ld1 { v21.s }[2], [x24], #0x4\n"
      "ld1 { v25.s }[2], [x23], #0x4\n"
      "ld1 { v29.s }[2], [x22], #0x4\n"
      "tbz x13, #0, 182f\n"
      "ld1 { v9.h }[6], [x11]\n"
      "ld1 { v13.h }[6], [x26]\n"
      "ld1 { v17.h }[6], [x25]\n"
      "ld1 { v21.h }[6], [x24]\n"
      "ld1 { v25.h }[6], [x23]\n"
      "ld1 { v29.h }[6], [x22]\n"
      "b 182f\n"
      "175:"  // Height 6: Partial accumulate: partial_1_12
      "mov x20, #0x18\n"
      "tbz x13, #0, 182f\n"
      "ld1 { v9.h }[4], [x11]\n"
      "ld1 { v13.h }[4], [x26]\n"
      "ld1 { v17.h }[4], [x25]\n"
      "ld1 { v21.h }[4], [x24]\n"
      "ld1 { v25.h }[4], [x23]\n"
      "ld1 { v29.h }[4], [x22]\n"
      "b 182f\n"
      "176:"  // Height 6: Partial accumulate: partial_2_8
      "tbz x13, #1, 177f\n"
      "ldr s9, [x11], #0x4\n"
      "ldr s13, [x26], #0x4\n"
      "mov x20, #0x14\n"
      "ldr s17, [x25], #0x4\n"
      "ldr s21, [x24], #0x4\n"
      "ldr s25, [x23], #0x4\n"
      "ldr s29, [x22], #0x4\n"
      "tbz x13, #0, 182f\n"
      "ld1 { v9.h }[2], [x11]\n"
      "ld1 { v13.h }[2], [x26]\n"
      "ld1 { v17.h }[2], [x25]\n"
      "ld1 { v21.h }[2], [x24]\n"
      "ld1 { v25.h }[2], [x23]\n"
      "ld1 { v29.h }[2], [x22]\n"
      "b 182f\n"
      "177:"  // Height 6: Partial accumulate: partial_1_8
      "mov x20, #0x10\n"
      "tbz x13, #0, 182f\n"
      "ldr h9, [x11, #0]\n"
      "ldr h13, [x26, #0]\n"
      "ldr h17, [x25, #0]\n"
      "ldr h21, [x24, #0]\n"
      "ldr h25, [x23, #0]\n"
      "ldr h29, [x22, #0]\n"
      "b 182f\n"
      "178:"  // Height 6: Partial accumulate: partial_4_0
      "tbz x13, #2, 180f\n"
      "ldr d8, [x11], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "ldr d16, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "ldr d24, [x23], #0x8\n"
      "ldr d28, [x22], #0x8\n"
      "tbz x13, #1, 179f\n"
      "ld1 { v8.s }[2], [x11], #0x4\n"
      "ld1 { v12.s }[2], [x26], #0x4\n"
      "mov x20, #0xc\n"
      "ld1 { v16.s }[2], [x25], #0x4\n"
      "ld1 { v20.s }[2], [x24], #0x4\n"
      "ld1 { v24.s }[2], [x23], #0x4\n"
      "ld1 { v28.s }[2], [x22], #0x4\n"
      "tbz x13, #0, 182f\n"
      "ld1 { v8.h }[6], [x11]\n"
      "ld1 { v12.h }[6], [x26]\n"
      "ld1 { v16.h }[6], [x25]\n"
      "ld1 { v20.h }[6], [x24]\n"
      "ld1 { v24.h }[6], [x23]\n"
      "ld1 { v28.h }[6], [x22]\n"
      "b 182f\n"
      "179:"  // Height 6: Partial accumulate: partial_1_4
      "mov x20, #0x8\n"
      "tbz x13, #0, 182f\n"
      "ld1 { v8.h }[4], [x11]\n"
      "ld1 { v12.h }[4], [x26]\n"
      "ld1 { v16.h }[4], [x25]\n"
      "ld1 { v20.h }[4], [x24]\n"
      "ld1 { v24.h }[4], [x23]\n"
      "ld1 { v28.h }[4], [x22]\n"
      "b 182f\n"
      "180:"  // Height 6: Partial accumulate: partial_2_0
      "tbz x13, #1, 181f\n"
      "ldr s8, [x11], #0x4\n"
      "ldr s12, [x26], #0x4\n"
      "mov x20, #0x4\n"
      "ldr s16, [x25], #0x4\n"
      "ldr s20, [x24], #0x4\n"
      "ldr s24, [x23], #0x4\n"
      "ldr s28, [x22], #0x4\n"
      "tbz x13, #0, 182f\n"
      "ld1 { v8.h }[2], [x11]\n"
      "ld1 { v12.h }[2], [x26]\n"
      "ld1 { v16.h }[2], [x25]\n"
      "ld1 { v20.h }[2], [x24]\n"
      "ld1 { v24.h }[2], [x23]\n"
      "ld1 { v28.h }[2], [x22]\n"
      "b 182f\n"
      "181:"  // Height 6: Partial accumulate: partial_1_0
      "ldr h8, [x11, #0]\n"
      "ldr h12, [x26, #0]\n"
      "mov x20, #0\n"
      "ldr h16, [x25, #0]\n"
      "ldr h20, [x24, #0]\n"
      "ldr h24, [x23, #0]\n"
      "ldr h28, [x22, #0]\n"
      "182:"  // Height 6: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "fcvtl2 v11.4s, v9.8h\n"
      "fcvtl v10.4s, v9.4h\n"
      "fcvtl2 v9.4s, v8.8h\n"
      "fcvtl v8.4s, v8.4h\n"
      "fcvtl2 v15.4s, v13.8h\n"
      "fcvtl v14.4s, v13.4h\n"
      "fcvtl2 v13.4s, v12.8h\n"
      "fcvtl v12.4s, v12.4h\n"
      "fcvtl2 v19.4s, v17.8h\n"
      "fcvtl v18.4s, v17.4h\n"
      "fcvtl2 v17.4s, v16.8h\n"
      "fcvtl v16.4s, v16.4h\n"
      "fcvtl2 v23.4s, v21.8h\n"
      "fcvtl v22.4s, v21.4h\n"
      "fcvtl2 v21.4s, v20.8h\n"
      "fcvtl v20.4s, v20.4h\n"
      "fcvtl2 v27.4s, v25.8h\n"
      "fcvtl v26.4s, v25.4h\n"
      "fcvtl2 v25.4s, v24.8h\n"
      "fcvtl v24.4s, v24.4h\n"
      "fcvtl2 v31.4s, v29.8h\n"
      "fcvtl v30.4s, v29.4h\n"
      "fcvtl2 v29.4s, v28.8h\n"
      "fcvtl v28.4s, v28.4h\n"
      "b 185f\n"
      "183:"  // Height 6: full accumulate
      "ldr q22, [x11, #0]\n"
      "ldr q20, [x11, #0x10]\n"
      "ldr q18, [x26, #0]\n"
      "ldr q16, [x26, #0x10]\n"
      "ldr q17, [x25, #0]\n"
      "ldr q19, [x25, #0x10]\n"
      "ldr q21, [x24, #0]\n"
      "ldr q23, [x24, #0x10]\n"
      "fcvtl v8.4s, v22.4h\n"
      "fcvtl2 v9.4s, v22.8h\n"
      "ldr q25, [x23, #0]\n"
      "ldr q27, [x23, #0x10]\n"
      "fcvtl v10.4s, v20.4h\n"
      "fcvtl2 v11.4s, v20.8h\n"
      "ldr q29, [x22, #0]\n"
      "ldr q31, [x22, #0x10]\n"
      "fcvtl v12.4s, v18.4h\n"
      "fcvtl2 v13.4s, v18.8h\n"
      "fcvtl v14.4s, v16.4h\n"
      "fcvtl2 v15.4s, v16.8h\n"
      "fcvtl v16.4s, v17.4h\n"
      "fcvtl2 v17.4s, v17.8h\n"
      "fcvtl v18.4s, v19.4h\n"
      "fcvtl2 v19.4s, v19.8h\n"
      "fcvtl v20.4s, v21.4h\n"
      "fcvtl2 v21.4s, v21.8h\n"
      "fcvtl v22.4s, v23.4h\n"
      "fcvtl2 v23.4s, v23.8h\n"
      "fcvtl v24.4s, v25.4h\n"
      "fcvtl2 v25.4s, v25.8h\n"
      "fcvtl v26.4s, v27.4h\n"
      "fcvtl2 v27.4s, v27.8h\n"
      "fcvtl v28.4s, v29.4h\n"
      "fcvtl2 v29.4s, v29.8h\n"
      "fcvtl v30.4s, v31.4h\n"
      "fcvtl2 v31.4s, v31.8h\n"
      "b 185f\n"
      "184:"  // Height 6: no accumulate
      "movi v8.16b, #0\n"
      "movi v9.16b, #0\n"
      "movi v10.16b, #0\n"
      "movi v11.16b, #0\n"
      "movi v12.16b, #0\n"
      "movi v13.16b, #0\n"
      "movi v14.16b, #0\n"
      "movi v15.16b, #0\n"
      "movi v16.16b, #0\n"
      "movi v17.16b, #0\n"
      "movi v18.16b, #0\n"
      "movi v19.16b, #0\n"
      "movi v20.16b, #0\n"
      "movi v21.16b, #0\n"
      "movi v22.16b, #0\n"
      "movi v23.16b, #0\n"
      "movi v24.16b, #0\n"
      "movi v25.16b, #0\n"
      "movi v26.16b, #0\n"
      "movi v27.16b, #0\n"
      "movi v28.16b, #0\n"
      "movi v29.16b, #0\n"
      "movi v30.16b, #0\n"
      "movi v31.16b, #0\n"
      "185:"  // Height 6: setup done
      "mov x28, #0\n"
      "186:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 187f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 188f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "add x21, x21, x20, LSL #1\n"
      "b 188f\n"
      "187:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "add x21, x22, x21, LSL #1\n"
      "188:"  // Height 6: input setup done
      "cmp x27, #0x8\n"
      "blt 191f\n"
      "ldr q0, [x26, #0]\n"
      "ldr q1, [x25, #0]\n"
      "cmp x27, #0x10\n"
      "ldr q2, [x24, #0]\n"
      "ldr q3, [x23, #0]\n"
      "ldr q4, [x22, #0]\n"
      "ldr q5, [x21, #0]\n"
      "ldr q6, [x10, #0]\n"
      "ldr q7, [x9, #0]\n"
      "blt 190f\n"
      "189:"  // Height 6: Multiply loop: Main loop head
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "cmp x27, #0x10\n"
      "add x25, x25, #0x10\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x4f8500dc  // fmlal v28.4s, v6.4h, v5.h[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8580dd  // fmlal2 v29.4s, v6.4h, v5.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x4f8500fe  // fmlal v30.4s, v7.4h, v5.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8580ff  // fmlal2 v31.4s, v7.4h, v5.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x4f9400d8  // fmlal v24.4s, v6.4h, v4.h[1]\n"
      ".inst 0x4f9500dc  // fmlal v28.4s, v6.4h, v5.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9480d9  // fmlal2 v25.4s, v6.4h, v4.h[1]\n"
      ".inst 0x6f9580dd  // fmlal2 v29.4s, v6.4h, v5.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x4f9400fa  // fmlal v26.4s, v7.4h, v4.h[1]\n"
      ".inst 0x4f9500fe  // fmlal v30.4s, v7.4h, v5.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9480fb  // fmlal2 v27.4s, v7.4h, v4.h[1]\n"
      ".inst 0x6f9580ff  // fmlal2 v31.4s, v7.4h, v5.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x4fa400d8  // fmlal v24.4s, v6.4h, v4.h[2]\n"
      ".inst 0x4fa500dc  // fmlal v28.4s, v6.4h, v5.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa480d9  // fmlal2 v25.4s, v6.4h, v4.h[2]\n"
      ".inst 0x6fa580dd  // fmlal2 v29.4s, v6.4h, v5.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x4fa400fa  // fmlal v26.4s, v7.4h, v4.h[2]\n"
      ".inst 0x4fa500fe  // fmlal v30.4s, v7.4h, v5.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa480fb  // fmlal2 v27.4s, v7.4h, v4.h[2]\n"
      ".inst 0x6fa580ff  // fmlal2 v31.4s, v7.4h, v5.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x4fb400d8  // fmlal v24.4s, v6.4h, v4.h[3]\n"
      ".inst 0x4fb500dc  // fmlal v28.4s, v6.4h, v5.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb480d9  // fmlal2 v25.4s, v6.4h, v4.h[3]\n"
      ".inst 0x6fb580dd  // fmlal2 v29.4s, v6.4h, v5.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x4fb400fa  // fmlal v26.4s, v7.4h, v4.h[3]\n"
      ".inst 0x4fb500fe  // fmlal v30.4s, v7.4h, v5.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb480fb  // fmlal2 v27.4s, v7.4h, v4.h[3]\n"
      ".inst 0x6fb580ff  // fmlal2 v31.4s, v7.4h, v5.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x4f8408d8  // fmlal v24.4s, v6.4h, v4.h[4]\n"
      ".inst 0x4f8508dc  // fmlal v28.4s, v6.4h, v5.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8488d9  // fmlal2 v25.4s, v6.4h, v4.h[4]\n"
      ".inst 0x6f8588dd  // fmlal2 v29.4s, v6.4h, v5.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x4f8408fa  // fmlal v26.4s, v7.4h, v4.h[4]\n"
      ".inst 0x4f8508fe  // fmlal v30.4s, v7.4h, v5.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8488fb  // fmlal2 v27.4s, v7.4h, v4.h[4]\n"
      ".inst 0x6f8588ff  // fmlal2 v31.4s, v7.4h, v5.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x4f9408d8  // fmlal v24.4s, v6.4h, v4.h[5]\n"
      ".inst 0x4f9508dc  // fmlal v28.4s, v6.4h, v5.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9488d9  // fmlal2 v25.4s, v6.4h, v4.h[5]\n"
      ".inst 0x6f9588dd  // fmlal2 v29.4s, v6.4h, v5.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x4f9408fa  // fmlal v26.4s, v7.4h, v4.h[5]\n"
      ".inst 0x4f9508fe  // fmlal v30.4s, v7.4h, v5.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9488fb  // fmlal2 v27.4s, v7.4h, v4.h[5]\n"
      ".inst 0x6f9588ff  // fmlal2 v31.4s, v7.4h, v5.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x4fa408d8  // fmlal v24.4s, v6.4h, v4.h[6]\n"
      ".inst 0x4fa508dc  // fmlal v28.4s, v6.4h, v5.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa488d9  // fmlal2 v25.4s, v6.4h, v4.h[6]\n"
      ".inst 0x6fa588dd  // fmlal2 v29.4s, v6.4h, v5.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x4fa408fa  // fmlal v26.4s, v7.4h, v4.h[6]\n"
      ".inst 0x4fa508fe  // fmlal v30.4s, v7.4h, v5.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa488fb  // fmlal2 v27.4s, v7.4h, v4.h[6]\n"
      ".inst 0x6fa588ff  // fmlal2 v31.4s, v7.4h, v5.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x4fb408d8  // fmlal v24.4s, v6.4h, v4.h[7]\n"
      ".inst 0x4fb508dc  // fmlal v28.4s, v6.4h, v5.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb488d9  // fmlal2 v25.4s, v6.4h, v4.h[7]\n"
      ".inst 0x6fb588dd  // fmlal2 v29.4s, v6.4h, v5.h[7]\n"
      "ldr q6, [x10, #0]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x4fb408fa  // fmlal v26.4s, v7.4h, v4.h[7]\n"
      ".inst 0x4fb508fe  // fmlal v30.4s, v7.4h, v5.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      "ldr q0, [x26, #0]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      "ldr q1, [x25, #0]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      "ldr q2, [x24, #0]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      "ldr q3, [x23, #0]\n"
      ".inst 0x6fb488fb  // fmlal2 v27.4s, v7.4h, v4.h[7]\n"
      "ldr q4, [x22, #0]\n"
      ".inst 0x6fb588ff  // fmlal2 v31.4s, v7.4h, v5.h[7]\n"
      "ldr q5, [x21, #0]\n"
      "ldr q7, [x9, #0]\n"
      "bge 189b\n"
      "190:"  // Height 6: Multiply loop: Single iteration only
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      "sub x27, x27, #0x8\n"
      "add x26, x26, #0x10\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x4f8500dc  // fmlal v28.4s, v6.4h, v5.h[0]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      "add x21, x21, #0x10\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8580dd  // fmlal2 v29.4s, v6.4h, v5.h[0]\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x4f8500fe  // fmlal v30.4s, v7.4h, v5.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8580ff  // fmlal2 v31.4s, v7.4h, v5.h[0]\n"
      "ldr q7, [x9, #0x10]\n"
      ".inst 0x4f9000c8  // fmlal v8.4s, v6.4h, v0.h[1]\n"
      ".inst 0x4f9100cc  // fmlal v12.4s, v6.4h, v1.h[1]\n"
      ".inst 0x4f9200d0  // fmlal v16.4s, v6.4h, v2.h[1]\n"
      ".inst 0x4f9300d4  // fmlal v20.4s, v6.4h, v3.h[1]\n"
      ".inst 0x4f9400d8  // fmlal v24.4s, v6.4h, v4.h[1]\n"
      ".inst 0x4f9500dc  // fmlal v28.4s, v6.4h, v5.h[1]\n"
      ".inst 0x6f9080c9  // fmlal2 v9.4s, v6.4h, v0.h[1]\n"
      ".inst 0x6f9180cd  // fmlal2 v13.4s, v6.4h, v1.h[1]\n"
      ".inst 0x6f9280d1  // fmlal2 v17.4s, v6.4h, v2.h[1]\n"
      ".inst 0x6f9380d5  // fmlal2 v21.4s, v6.4h, v3.h[1]\n"
      ".inst 0x6f9480d9  // fmlal2 v25.4s, v6.4h, v4.h[1]\n"
      ".inst 0x6f9580dd  // fmlal2 v29.4s, v6.4h, v5.h[1]\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x4f9000ea  // fmlal v10.4s, v7.4h, v0.h[1]\n"
      ".inst 0x4f9100ee  // fmlal v14.4s, v7.4h, v1.h[1]\n"
      ".inst 0x4f9200f2  // fmlal v18.4s, v7.4h, v2.h[1]\n"
      ".inst 0x4f9300f6  // fmlal v22.4s, v7.4h, v3.h[1]\n"
      ".inst 0x4f9400fa  // fmlal v26.4s, v7.4h, v4.h[1]\n"
      ".inst 0x4f9500fe  // fmlal v30.4s, v7.4h, v5.h[1]\n"
      ".inst 0x6f9080eb  // fmlal2 v11.4s, v7.4h, v0.h[1]\n"
      ".inst 0x6f9180ef  // fmlal2 v15.4s, v7.4h, v1.h[1]\n"
      ".inst 0x6f9280f3  // fmlal2 v19.4s, v7.4h, v2.h[1]\n"
      ".inst 0x6f9380f7  // fmlal2 v23.4s, v7.4h, v3.h[1]\n"
      ".inst 0x6f9480fb  // fmlal2 v27.4s, v7.4h, v4.h[1]\n"
      ".inst 0x6f9580ff  // fmlal2 v31.4s, v7.4h, v5.h[1]\n"
      "ldr q7, [x9, #0x20]\n"
      ".inst 0x4fa000c8  // fmlal v8.4s, v6.4h, v0.h[2]\n"
      ".inst 0x4fa100cc  // fmlal v12.4s, v6.4h, v1.h[2]\n"
      ".inst 0x4fa200d0  // fmlal v16.4s, v6.4h, v2.h[2]\n"
      ".inst 0x4fa300d4  // fmlal v20.4s, v6.4h, v3.h[2]\n"
      ".inst 0x4fa400d8  // fmlal v24.4s, v6.4h, v4.h[2]\n"
      ".inst 0x4fa500dc  // fmlal v28.4s, v6.4h, v5.h[2]\n"
      ".inst 0x6fa080c9  // fmlal2 v9.4s, v6.4h, v0.h[2]\n"
      ".inst 0x6fa180cd  // fmlal2 v13.4s, v6.4h, v1.h[2]\n"
      ".inst 0x6fa280d1  // fmlal2 v17.4s, v6.4h, v2.h[2]\n"
      ".inst 0x6fa380d5  // fmlal2 v21.4s, v6.4h, v3.h[2]\n"
      ".inst 0x6fa480d9  // fmlal2 v25.4s, v6.4h, v4.h[2]\n"
      ".inst 0x6fa580dd  // fmlal2 v29.4s, v6.4h, v5.h[2]\n"
      "ldr q6, [x10, #0x30]\n"
      ".inst 0x4fa000ea  // fmlal v10.4s, v7.4h, v0.h[2]\n"
      ".inst 0x4fa100ee  // fmlal v14.4s, v7.4h, v1.h[2]\n"
      ".inst 0x4fa200f2  // fmlal v18.4s, v7.4h, v2.h[2]\n"
      ".inst 0x4fa300f6  // fmlal v22.4s, v7.4h, v3.h[2]\n"
      ".inst 0x4fa400fa  // fmlal v26.4s, v7.4h, v4.h[2]\n"
      ".inst 0x4fa500fe  // fmlal v30.4s, v7.4h, v5.h[2]\n"
      ".inst 0x6fa080eb  // fmlal2 v11.4s, v7.4h, v0.h[2]\n"
      ".inst 0x6fa180ef  // fmlal2 v15.4s, v7.4h, v1.h[2]\n"
      ".inst 0x6fa280f3  // fmlal2 v19.4s, v7.4h, v2.h[2]\n"
      ".inst 0x6fa380f7  // fmlal2 v23.4s, v7.4h, v3.h[2]\n"
      ".inst 0x6fa480fb  // fmlal2 v27.4s, v7.4h, v4.h[2]\n"
      ".inst 0x6fa580ff  // fmlal2 v31.4s, v7.4h, v5.h[2]\n"
      "ldr q7, [x9, #0x30]\n"
      ".inst 0x4fb000c8  // fmlal v8.4s, v6.4h, v0.h[3]\n"
      ".inst 0x4fb100cc  // fmlal v12.4s, v6.4h, v1.h[3]\n"
      ".inst 0x4fb200d0  // fmlal v16.4s, v6.4h, v2.h[3]\n"
      ".inst 0x4fb300d4  // fmlal v20.4s, v6.4h, v3.h[3]\n"
      ".inst 0x4fb400d8  // fmlal v24.4s, v6.4h, v4.h[3]\n"
      ".inst 0x4fb500dc  // fmlal v28.4s, v6.4h, v5.h[3]\n"
      ".inst 0x6fb080c9  // fmlal2 v9.4s, v6.4h, v0.h[3]\n"
      ".inst 0x6fb180cd  // fmlal2 v13.4s, v6.4h, v1.h[3]\n"
      ".inst 0x6fb280d1  // fmlal2 v17.4s, v6.4h, v2.h[3]\n"
      ".inst 0x6fb380d5  // fmlal2 v21.4s, v6.4h, v3.h[3]\n"
      ".inst 0x6fb480d9  // fmlal2 v25.4s, v6.4h, v4.h[3]\n"
      ".inst 0x6fb580dd  // fmlal2 v29.4s, v6.4h, v5.h[3]\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x4fb000ea  // fmlal v10.4s, v7.4h, v0.h[3]\n"
      ".inst 0x4fb100ee  // fmlal v14.4s, v7.4h, v1.h[3]\n"
      ".inst 0x4fb200f2  // fmlal v18.4s, v7.4h, v2.h[3]\n"
      ".inst 0x4fb300f6  // fmlal v22.4s, v7.4h, v3.h[3]\n"
      ".inst 0x4fb400fa  // fmlal v26.4s, v7.4h, v4.h[3]\n"
      ".inst 0x4fb500fe  // fmlal v30.4s, v7.4h, v5.h[3]\n"
      ".inst 0x6fb080eb  // fmlal2 v11.4s, v7.4h, v0.h[3]\n"
      ".inst 0x6fb180ef  // fmlal2 v15.4s, v7.4h, v1.h[3]\n"
      ".inst 0x6fb280f3  // fmlal2 v19.4s, v7.4h, v2.h[3]\n"
      ".inst 0x6fb380f7  // fmlal2 v23.4s, v7.4h, v3.h[3]\n"
      ".inst 0x6fb480fb  // fmlal2 v27.4s, v7.4h, v4.h[3]\n"
      ".inst 0x6fb580ff  // fmlal2 v31.4s, v7.4h, v5.h[3]\n"
      "ldr q7, [x9, #0x40]\n"
      ".inst 0x4f8008c8  // fmlal v8.4s, v6.4h, v0.h[4]\n"
      ".inst 0x4f8108cc  // fmlal v12.4s, v6.4h, v1.h[4]\n"
      ".inst 0x4f8208d0  // fmlal v16.4s, v6.4h, v2.h[4]\n"
      ".inst 0x4f8308d4  // fmlal v20.4s, v6.4h, v3.h[4]\n"
      ".inst 0x4f8408d8  // fmlal v24.4s, v6.4h, v4.h[4]\n"
      ".inst 0x4f8508dc  // fmlal v28.4s, v6.4h, v5.h[4]\n"
      ".inst 0x6f8088c9  // fmlal2 v9.4s, v6.4h, v0.h[4]\n"
      ".inst 0x6f8188cd  // fmlal2 v13.4s, v6.4h, v1.h[4]\n"
      ".inst 0x6f8288d1  // fmlal2 v17.4s, v6.4h, v2.h[4]\n"
      ".inst 0x6f8388d5  // fmlal2 v21.4s, v6.4h, v3.h[4]\n"
      ".inst 0x6f8488d9  // fmlal2 v25.4s, v6.4h, v4.h[4]\n"
      ".inst 0x6f8588dd  // fmlal2 v29.4s, v6.4h, v5.h[4]\n"
      "ldr q6, [x10, #0x50]\n"
      ".inst 0x4f8008ea  // fmlal v10.4s, v7.4h, v0.h[4]\n"
      ".inst 0x4f8108ee  // fmlal v14.4s, v7.4h, v1.h[4]\n"
      ".inst 0x4f8208f2  // fmlal v18.4s, v7.4h, v2.h[4]\n"
      ".inst 0x4f8308f6  // fmlal v22.4s, v7.4h, v3.h[4]\n"
      ".inst 0x4f8408fa  // fmlal v26.4s, v7.4h, v4.h[4]\n"
      ".inst 0x4f8508fe  // fmlal v30.4s, v7.4h, v5.h[4]\n"
      ".inst 0x6f8088eb  // fmlal2 v11.4s, v7.4h, v0.h[4]\n"
      ".inst 0x6f8188ef  // fmlal2 v15.4s, v7.4h, v1.h[4]\n"
      ".inst 0x6f8288f3  // fmlal2 v19.4s, v7.4h, v2.h[4]\n"
      ".inst 0x6f8388f7  // fmlal2 v23.4s, v7.4h, v3.h[4]\n"
      ".inst 0x6f8488fb  // fmlal2 v27.4s, v7.4h, v4.h[4]\n"
      ".inst 0x6f8588ff  // fmlal2 v31.4s, v7.4h, v5.h[4]\n"
      "ldr q7, [x9, #0x50]\n"
      ".inst 0x4f9008c8  // fmlal v8.4s, v6.4h, v0.h[5]\n"
      ".inst 0x4f9108cc  // fmlal v12.4s, v6.4h, v1.h[5]\n"
      ".inst 0x4f9208d0  // fmlal v16.4s, v6.4h, v2.h[5]\n"
      ".inst 0x4f9308d4  // fmlal v20.4s, v6.4h, v3.h[5]\n"
      ".inst 0x4f9408d8  // fmlal v24.4s, v6.4h, v4.h[5]\n"
      ".inst 0x4f9508dc  // fmlal v28.4s, v6.4h, v5.h[5]\n"
      ".inst 0x6f9088c9  // fmlal2 v9.4s, v6.4h, v0.h[5]\n"
      ".inst 0x6f9188cd  // fmlal2 v13.4s, v6.4h, v1.h[5]\n"
      ".inst 0x6f9288d1  // fmlal2 v17.4s, v6.4h, v2.h[5]\n"
      ".inst 0x6f9388d5  // fmlal2 v21.4s, v6.4h, v3.h[5]\n"
      ".inst 0x6f9488d9  // fmlal2 v25.4s, v6.4h, v4.h[5]\n"
      ".inst 0x6f9588dd  // fmlal2 v29.4s, v6.4h, v5.h[5]\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x4f9008ea  // fmlal v10.4s, v7.4h, v0.h[5]\n"
      ".inst 0x4f9108ee  // fmlal v14.4s, v7.4h, v1.h[5]\n"
      ".inst 0x4f9208f2  // fmlal v18.4s, v7.4h, v2.h[5]\n"
      ".inst 0x4f9308f6  // fmlal v22.4s, v7.4h, v3.h[5]\n"
      ".inst 0x4f9408fa  // fmlal v26.4s, v7.4h, v4.h[5]\n"
      ".inst 0x4f9508fe  // fmlal v30.4s, v7.4h, v5.h[5]\n"
      ".inst 0x6f9088eb  // fmlal2 v11.4s, v7.4h, v0.h[5]\n"
      ".inst 0x6f9188ef  // fmlal2 v15.4s, v7.4h, v1.h[5]\n"
      ".inst 0x6f9288f3  // fmlal2 v19.4s, v7.4h, v2.h[5]\n"
      ".inst 0x6f9388f7  // fmlal2 v23.4s, v7.4h, v3.h[5]\n"
      ".inst 0x6f9488fb  // fmlal2 v27.4s, v7.4h, v4.h[5]\n"
      ".inst 0x6f9588ff  // fmlal2 v31.4s, v7.4h, v5.h[5]\n"
      "ldr q7, [x9, #0x60]\n"
      ".inst 0x4fa008c8  // fmlal v8.4s, v6.4h, v0.h[6]\n"
      ".inst 0x4fa108cc  // fmlal v12.4s, v6.4h, v1.h[6]\n"
      ".inst 0x4fa208d0  // fmlal v16.4s, v6.4h, v2.h[6]\n"
      ".inst 0x4fa308d4  // fmlal v20.4s, v6.4h, v3.h[6]\n"
      ".inst 0x4fa408d8  // fmlal v24.4s, v6.4h, v4.h[6]\n"
      ".inst 0x4fa508dc  // fmlal v28.4s, v6.4h, v5.h[6]\n"
      ".inst 0x6fa088c9  // fmlal2 v9.4s, v6.4h, v0.h[6]\n"
      ".inst 0x6fa188cd  // fmlal2 v13.4s, v6.4h, v1.h[6]\n"
      ".inst 0x6fa288d1  // fmlal2 v17.4s, v6.4h, v2.h[6]\n"
      ".inst 0x6fa388d5  // fmlal2 v21.4s, v6.4h, v3.h[6]\n"
      ".inst 0x6fa488d9  // fmlal2 v25.4s, v6.4h, v4.h[6]\n"
      ".inst 0x6fa588dd  // fmlal2 v29.4s, v6.4h, v5.h[6]\n"
      "ldr q6, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x4fa008ea  // fmlal v10.4s, v7.4h, v0.h[6]\n"
      ".inst 0x4fa108ee  // fmlal v14.4s, v7.4h, v1.h[6]\n"
      ".inst 0x4fa208f2  // fmlal v18.4s, v7.4h, v2.h[6]\n"
      ".inst 0x4fa308f6  // fmlal v22.4s, v7.4h, v3.h[6]\n"
      ".inst 0x4fa408fa  // fmlal v26.4s, v7.4h, v4.h[6]\n"
      ".inst 0x4fa508fe  // fmlal v30.4s, v7.4h, v5.h[6]\n"
      ".inst 0x6fa088eb  // fmlal2 v11.4s, v7.4h, v0.h[6]\n"
      ".inst 0x6fa188ef  // fmlal2 v15.4s, v7.4h, v1.h[6]\n"
      ".inst 0x6fa288f3  // fmlal2 v19.4s, v7.4h, v2.h[6]\n"
      ".inst 0x6fa388f7  // fmlal2 v23.4s, v7.4h, v3.h[6]\n"
      ".inst 0x6fa488fb  // fmlal2 v27.4s, v7.4h, v4.h[6]\n"
      ".inst 0x6fa588ff  // fmlal2 v31.4s, v7.4h, v5.h[6]\n"
      "ldr q7, [x9, #0x70]\n"
      "add x9, x9, #0x80\n"
      ".inst 0x4fb008c8  // fmlal v8.4s, v6.4h, v0.h[7]\n"
      ".inst 0x4fb108cc  // fmlal v12.4s, v6.4h, v1.h[7]\n"
      ".inst 0x4fb208d0  // fmlal v16.4s, v6.4h, v2.h[7]\n"
      ".inst 0x4fb308d4  // fmlal v20.4s, v6.4h, v3.h[7]\n"
      ".inst 0x4fb408d8  // fmlal v24.4s, v6.4h, v4.h[7]\n"
      ".inst 0x4fb508dc  // fmlal v28.4s, v6.4h, v5.h[7]\n"
      ".inst 0x6fb088c9  // fmlal2 v9.4s, v6.4h, v0.h[7]\n"
      ".inst 0x6fb188cd  // fmlal2 v13.4s, v6.4h, v1.h[7]\n"
      ".inst 0x6fb288d1  // fmlal2 v17.4s, v6.4h, v2.h[7]\n"
      ".inst 0x6fb388d5  // fmlal2 v21.4s, v6.4h, v3.h[7]\n"
      ".inst 0x6fb488d9  // fmlal2 v25.4s, v6.4h, v4.h[7]\n"
      ".inst 0x6fb588dd  // fmlal2 v29.4s, v6.4h, v5.h[7]\n"
      ".inst 0x4fb008ea  // fmlal v10.4s, v7.4h, v0.h[7]\n"
      ".inst 0x4fb108ee  // fmlal v14.4s, v7.4h, v1.h[7]\n"
      ".inst 0x4fb208f2  // fmlal v18.4s, v7.4h, v2.h[7]\n"
      ".inst 0x4fb308f6  // fmlal v22.4s, v7.4h, v3.h[7]\n"
      ".inst 0x4fb408fa  // fmlal v26.4s, v7.4h, v4.h[7]\n"
      ".inst 0x4fb508fe  // fmlal v30.4s, v7.4h, v5.h[7]\n"
      ".inst 0x6fb088eb  // fmlal2 v11.4s, v7.4h, v0.h[7]\n"
      ".inst 0x6fb188ef  // fmlal2 v15.4s, v7.4h, v1.h[7]\n"
      ".inst 0x6fb288f3  // fmlal2 v19.4s, v7.4h, v2.h[7]\n"
      ".inst 0x6fb388f7  // fmlal2 v23.4s, v7.4h, v3.h[7]\n"
      ".inst 0x6fb488fb  // fmlal2 v27.4s, v7.4h, v4.h[7]\n"
      ".inst 0x6fb588ff  // fmlal2 v31.4s, v7.4h, v5.h[7]\n"
      "191:"  // Height 6: Multiply loop: Main loop skip
      "cbz x27, 193f\n"
      "192:"  // Height 6: Multiply loop: Odd block loop
      "ldr h0, [x26], #0x2\n"
      "ldr h1, [x25], #0x2\n"
      "sub x27, x27, #0x1\n"
      "ldr h2, [x24], #0x2\n"
      "ldr h3, [x23], #0x2\n"
      "ldr h4, [x22], #0x2\n"
      "ldr h5, [x21], #0x2\n"
      "ldr q6, [x10, #0]\n"
      "ldr q7, [x9, #0]\n"
      "add x10, x10, #0x10\n"
      "add x9, x9, #0x10\n"
      ".inst 0x4f8000c8  // fmlal v8.4s, v6.4h, v0.h[0]\n"
      ".inst 0x4f8100cc  // fmlal v12.4s, v6.4h, v1.h[0]\n"
      ".inst 0x4f8200d0  // fmlal v16.4s, v6.4h, v2.h[0]\n"
      ".inst 0x4f8300d4  // fmlal v20.4s, v6.4h, v3.h[0]\n"
      ".inst 0x4f8400d8  // fmlal v24.4s, v6.4h, v4.h[0]\n"
      ".inst 0x4f8500dc  // fmlal v28.4s, v6.4h, v5.h[0]\n"
      ".inst 0x6f8080c9  // fmlal2 v9.4s, v6.4h, v0.h[0]\n"
      ".inst 0x6f8180cd  // fmlal2 v13.4s, v6.4h, v1.h[0]\n"
      ".inst 0x6f8280d1  // fmlal2 v17.4s, v6.4h, v2.h[0]\n"
      ".inst 0x6f8380d5  // fmlal2 v21.4s, v6.4h, v3.h[0]\n"
      ".inst 0x6f8480d9  // fmlal2 v25.4s, v6.4h, v4.h[0]\n"
      ".inst 0x6f8580dd  // fmlal2 v29.4s, v6.4h, v5.h[0]\n"
      ".inst 0x4f8000ea  // fmlal v10.4s, v7.4h, v0.h[0]\n"
      ".inst 0x4f8100ee  // fmlal v14.4s, v7.4h, v1.h[0]\n"
      ".inst 0x4f8200f2  // fmlal v18.4s, v7.4h, v2.h[0]\n"
      ".inst 0x4f8300f6  // fmlal v22.4s, v7.4h, v3.h[0]\n"
      ".inst 0x4f8400fa  // fmlal v26.4s, v7.4h, v4.h[0]\n"
      ".inst 0x4f8500fe  // fmlal v30.4s, v7.4h, v5.h[0]\n"
      ".inst 0x6f8080eb  // fmlal2 v11.4s, v7.4h, v0.h[0]\n"
      ".inst 0x6f8180ef  // fmlal2 v15.4s, v7.4h, v1.h[0]\n"
      ".inst 0x6f8280f3  // fmlal2 v19.4s, v7.4h, v2.h[0]\n"
      ".inst 0x6f8380f7  // fmlal2 v23.4s, v7.4h, v3.h[0]\n"
      ".inst 0x6f8480fb  // fmlal2 v27.4s, v7.4h, v4.h[0]\n"
      ".inst 0x6f8580ff  // fmlal2 v31.4s, v7.4h, v5.h[0]\n"
      "cbnz x27, 192b\n"
      "193:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 186b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x11, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "add x22, x23, x20, LSL #1\n"
      "tbz %x[flags], #1, 194f\n"
      "add x21, %x[args_ptr], %[offsetof_maxval]\n"
      "add x20, %x[args_ptr], %[offsetof_minval]\n"
      "ld1r { v1.4s }, [x21]\n"
      "ld1r { v0.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v1.4s\n"
      "fmin v9.4s, v9.4s, v1.4s\n"
      "fmin v10.4s, v10.4s, v1.4s\n"
      "fmin v11.4s, v11.4s, v1.4s\n"
      "fmin v12.4s, v12.4s, v1.4s\n"
      "fmin v13.4s, v13.4s, v1.4s\n"
      "fmin v14.4s, v14.4s, v1.4s\n"
      "fmin v15.4s, v15.4s, v1.4s\n"
      "fmin v16.4s, v16.4s, v1.4s\n"
      "fmin v17.4s, v17.4s, v1.4s\n"
      "fmin v18.4s, v18.4s, v1.4s\n"
      "fmin v19.4s, v19.4s, v1.4s\n"
      "fmin v20.4s, v20.4s, v1.4s\n"
      "fmin v21.4s, v21.4s, v1.4s\n"
      "fmin v22.4s, v22.4s, v1.4s\n"
      "fmin v23.4s, v23.4s, v1.4s\n"
      "fmin v24.4s, v24.4s, v1.4s\n"
      "fmin v25.4s, v25.4s, v1.4s\n"
      "fmin v26.4s, v26.4s, v1.4s\n"
      "fmin v27.4s, v27.4s, v1.4s\n"
      "fmin v28.4s, v28.4s, v1.4s\n"
      "fmin v29.4s, v29.4s, v1.4s\n"
      "fmin v30.4s, v30.4s, v1.4s\n"
      "fmin v31.4s, v31.4s, v1.4s\n"
      "fmax v8.4s, v8.4s, v0.4s\n"
      "fmax v9.4s, v9.4s, v0.4s\n"
      "fmax v10.4s, v10.4s, v0.4s\n"
      "fmax v11.4s, v11.4s, v0.4s\n"
      "fmax v12.4s, v12.4s, v0.4s\n"
      "fmax v13.4s, v13.4s, v0.4s\n"
      "fmax v14.4s, v14.4s, v0.4s\n"
      "fmax v15.4s, v15.4s, v0.4s\n"
      "fmax v16.4s, v16.4s, v0.4s\n"
      "fmax v17.4s, v17.4s, v0.4s\n"
      "fmax v18.4s, v18.4s, v0.4s\n"
      "fmax v19.4s, v19.4s, v0.4s\n"
      "fmax v20.4s, v20.4s, v0.4s\n"
      "fmax v21.4s, v21.4s, v0.4s\n"
      "fmax v22.4s, v22.4s, v0.4s\n"
      "fmax v23.4s, v23.4s, v0.4s\n"
      "fmax v24.4s, v24.4s, v0.4s\n"
      "fmax v25.4s, v25.4s, v0.4s\n"
      "fmax v26.4s, v26.4s, v0.4s\n"
      "fmax v27.4s, v27.4s, v0.4s\n"
      "fmax v28.4s, v28.4s, v0.4s\n"
      "fmax v29.4s, v29.4s, v0.4s\n"
      "fmax v30.4s, v30.4s, v0.4s\n"
      "fmax v31.4s, v31.4s, v0.4s\n"
      "194:"  // Height 6: No activation
      "fcvtn v8.4h, v8.4s\n"
      "fcvtn v12.4h, v12.4s\n"
      "cmp x13, #0x10\n"
      "fcvtn v16.4h, v16.4s\n"
      "fcvtn v20.4h, v20.4s\n"
      "fcvtn v24.4h, v24.4s\n"
      "fcvtn v28.4h, v28.4s\n"
      "fcvtn2 v8.8h, v9.4s\n"
      "fcvtn v9.4h, v10.4s\n"
      "fcvtn2 v12.8h, v13.4s\n"
      "fcvtn v13.4h, v14.4s\n"
      "fcvtn2 v16.8h, v17.4s\n"
      "fcvtn v17.4h, v18.4s\n"
      "fcvtn2 v20.8h, v21.4s\n"
      "fcvtn v21.4h, v22.4s\n"
      "fcvtn2 v24.8h, v25.4s\n"
      "fcvtn v25.4h, v26.4s\n"
      "fcvtn2 v28.8h, v29.4s\n"
      "fcvtn v29.4h, v30.4s\n"
      "fcvtn2 v9.8h, v11.4s\n"
      "fcvtn2 v13.8h, v15.4s\n"
      "fcvtn2 v17.8h, v19.4s\n"
      "fcvtn2 v21.8h, v23.4s\n"
      "fcvtn2 v25.8h, v27.4s\n"
      "fcvtn2 v29.8h, v31.4s\n"
      "bge 203f\n"
      "tbz x13, #3, 198f\n"
      "st1 { v8.4s }, [x11], #0x10\n"
      "st1 { v12.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x24], #0x10\n"
      "st1 { v24.4s }, [x23], #0x10\n"
      "st1 { v28.4s }, [x22], #0x10\n"
      "tbz x13, #2, 196f\n"
      "str d9, [x11], #0x8\n"
      "str d13, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "str d21, [x24], #0x8\n"
      "str d25, [x23], #0x8\n"
      "str d29, [x22], #0x8\n"
      "tbz x13, #1, 195f\n"
      "st1 { v9.s }[2], [x11], #0x4\n"
      "st1 { v13.s }[2], [x26], #0x4\n"
      "st1 { v17.s }[2], [x25], #0x4\n"
      "st1 { v21.s }[2], [x24], #0x4\n"
      "st1 { v25.s }[2], [x23], #0x4\n"
      "st1 { v29.s }[2], [x22], #0x4\n"
      "tbz x13, #0, 202f\n"
      "st1 { v9.h }[6], [x11]\n"
      "st1 { v13.h }[6], [x26]\n"
      "st1 { v17.h }[6], [x25]\n"
      "st1 { v21.h }[6], [x24]\n"
      "st1 { v25.h }[6], [x23]\n"
      "st1 { v29.h }[6], [x22]\n"
      "b 202f\n"
      "195:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x13, #0, 202f\n"
      "st1 { v9.h }[4], [x11]\n"
      "st1 { v13.h }[4], [x26]\n"
      "st1 { v17.h }[4], [x25]\n"
      "st1 { v21.h }[4], [x24]\n"
      "st1 { v25.h }[4], [x23]\n"
      "st1 { v29.h }[4], [x22]\n"
      "b 202f\n"
      "196:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x13, #1, 197f\n"
      "str s9, [x11], #0x4\n"
      "str s13, [x26], #0x4\n"
      "str s17, [x25], #0x4\n"
      "str s21, [x24], #0x4\n"
      "str s25, [x23], #0x4\n"
      "str s29, [x22], #0x4\n"
      "tbz x13, #0, 202f\n"
      "st1 { v9.h }[2], [x11]\n"
      "st1 { v13.h }[2], [x26]\n"
      "st1 { v17.h }[2], [x25]\n"
      "st1 { v21.h }[2], [x24]\n"
      "st1 { v25.h }[2], [x23]\n"
      "st1 { v29.h }[2], [x22]\n"
      "b 202f\n"
      "197:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x13, #0, 202f\n"
      "str h9, [x11, #0]\n"
      "str h13, [x26, #0]\n"
      "str h17, [x25, #0]\n"
      "str h21, [x24, #0]\n"
      "str h25, [x23, #0]\n"
      "str h29, [x22, #0]\n"
      "b 202f\n"
      "198:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x13, #2, 200f\n"
      "str d8, [x11], #0x8\n"
      "str d12, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "str d20, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "str d28, [x22], #0x8\n"
      "tbz x13, #1, 199f\n"
      "st1 { v8.s }[2], [x11], #0x4\n"
      "st1 { v12.s }[2], [x26], #0x4\n"
      "st1 { v16.s }[2], [x25], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "st1 { v24.s }[2], [x23], #0x4\n"
      "st1 { v28.s }[2], [x22], #0x4\n"
      "tbz x13, #0, 202f\n"
      "st1 { v8.h }[6], [x11]\n"
      "st1 { v12.h }[6], [x26]\n"
      "st1 { v16.h }[6], [x25]\n"
      "st1 { v20.h }[6], [x24]\n"
      "st1 { v24.h }[6], [x23]\n"
      "st1 { v28.h }[6], [x22]\n"
      "b 202f\n"
      "199:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x13, #0, 202f\n"
      "st1 { v8.h }[4], [x11]\n"
      "st1 { v12.h }[4], [x26]\n"
      "st1 { v16.h }[4], [x25]\n"
      "st1 { v20.h }[4], [x24]\n"
      "st1 { v24.h }[4], [x23]\n"
      "st1 { v28.h }[4], [x22]\n"
      "b 202f\n"
      "200:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x13, #1, 201f\n"
      "str s8, [x11], #0x4\n"
      "str s12, [x26], #0x4\n"
      "str s16, [x25], #0x4\n"
      "str s20, [x24], #0x4\n"
      "str s24, [x23], #0x4\n"
      "str s28, [x22], #0x4\n"
      "tbz x13, #0, 202f\n"
      "st1 { v8.h }[2], [x11]\n"
      "st1 { v12.h }[2], [x26]\n"
      "st1 { v16.h }[2], [x25]\n"
      "st1 { v20.h }[2], [x24]\n"
      "st1 { v24.h }[2], [x23]\n"
      "st1 { v28.h }[2], [x22]\n"
      "b 202f\n"
      "201:"  // Height 6: Partial direct writeback: partial_1_0
      "str h8, [x11, #0]\n"
      "str h12, [x26, #0]\n"
      "str h16, [x25, #0]\n"
      "str h20, [x24, #0]\n"
      "str h24, [x23, #0]\n"
      "str h28, [x22, #0]\n"
      "202:"  // Height 6: Partial direct writeback: Done
      "b 204f\n"
      "203:"  // Height 6: Full writeback
      "str q8, [x11, #0]\n"
      "str q9, [x11, #0x10]\n"
      "add x11, x11, #0x20\n"
      "str q12, [x26, #0]\n"
      "str q13, [x26, #0x10]\n"
      "str q16, [x25, #0]\n"
      "str q17, [x25, #0x10]\n"
      "str q20, [x24, #0]\n"
      "str q21, [x24, #0x10]\n"
      "str q24, [x23, #0]\n"
      "str q25, [x23, #0x10]\n"
      "str q28, [x22, #0]\n"
      "str q29, [x22, #0x10]\n"
      "204:"  // Height 6: Writeback done
      "subs x13, x13, #0x10\n"
      "bgt 172b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 206f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 205f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "205:"  // Update direct input
      "mov x20, #0xc\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "206:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_B_stride] "I" (offsetof(KernelArgs, B_stride)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_cur_B_ptr] "I" (offsetof(KernelArgs, cur_B_ptr)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_maxval] "I" (offsetof(KernelArgs, maxval)), [offsetof_minval] "I" (offsetof(KernelArgs, minval)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x13", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm

#endif // (defined(ENABLE_FP16_KERNELS) || defined(__ARM_FEATURE_FP16_VECTOR_ARITHMETIC)) && defined(ARM_COMPUTE_ENABLE_FIXED_FORMAT_KERNELS) && defined(__aarch64__)

