/*
 * Copyright (c) 2022-2024, 2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include <algorithm>
#include <cstddef>
#include "arm_gemm/arm_gemm.hpp"

using arm_gemm::Requantize32;

namespace arm_conv {
namespace depthwise {

void sme2_u8q_planar_3x3_s2_4rows_dot_za_impl(
  const uint8_t *inptr,
  size_t ld_in_row,
  size_t ld_in_col,
  size_t ld_in_vl,
  unsigned int pad_top,
  unsigned int valid_input_rows,
  unsigned int pad_left,
  unsigned int valid_input_cols,
  const uint8_t *weights,
  uint8_t **outptrs,
  const size_t *outlds,
  const size_t *outvllds,
  unsigned int output_cols,
  unsigned int start_channel,
  unsigned int valid_channels,
  const arm_gemm::Requantize32 &qp
)
{
  struct Args
  {
    const uint8_t *inptr;
    size_t ld_in_vl;
    long unsigned int pad_top, pad_bottom, pad_left;
    const uint8_t *weights;
    long unsigned int input_cols, output_cols;
    uint8_t **outptrs;
    const size_t *ld_out_cols;
    const size_t *ld_out_vls;
    long unsigned int current_channel, n_channels;
  };

  Args args = { inptr, ld_in_vl, pad_top, 9u - std::min(9u, pad_top + valid_input_rows), pad_left, weights, valid_input_cols, output_cols, outptrs, outlds, outvllds, start_channel, valid_channels };

  __asm__ __volatile__(
    ".inst 0xd503477f  // SMSTART ZA\n"
    "ldr x6, [%x[args], %[offsetof_Args_pad_bottom]]\n"
    "mov x22, SP\n"
    "mov x21, #0x9\n"
    "ptrue p2.b\n"
    "ldr x7, [%x[args], %[offsetof_Args_pad_top]]\n"
    "sub x20, x22, #0x8\n"
    "ldr x17, [%x[args], %[offsetof_Args_n_channels]]\n"
    "ld1rh { z29.h }, p2/Z, [%x[qp], %[offsetof_Requantize32_a_offset]]\n"
    "and x20, x20, #-0x400\n"
    ".inst 0x25207812  // ptrue pn10.b\n"
    "sub x21, x21, x6\n"
    "mov SP, x20\n"
    "ldr x16, [%x[args], %[offsetof_Args_current_channel]]\n"
    "ld1rw { z8.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_c_offset]]\n"
    "str x22, [SP]\n"
    "addvl SP, SP, #-6\n"
    "ld1rw { z1.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_per_layer_mul]]\n"
    "whilelt p1.s, XZR, x17\n"
    "whilelt p9.s, XZR, x21\n"
    "neg z29.h, p2/M, z29.h\n"
    "ld1rw { z0.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_per_layer_right_shift]]\n"
    "whilelt p8.s, XZR, x7\n"
    "ld1rw { z31.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_minval]]\n"
    "eor p8.b, p2/Z, p8.b, p9.b\n"
    "ld1rw { z28.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_maxval]]\n"
    "1:"  // Channel loop
    "ldr x20, [%x[qp], %[offsetof_Requantize32_bias]]\n"
    "mov z16.s, #0x0\n"
    "cbz x20, 2f\n"
    "ld1w { z16.s }, p1/Z, [x20, x16, LSL #2]\n"
    "2:"  // Load bias: Done
    "ldr x23, [%x[args], %[offsetof_Args_weights]]\n"
    "ld1rh { z22.h }, p2/Z, [%x[qp], %[offsetof_Requantize32_b_offset]]\n"
    "mov z27.h, #0x0\n"
    "addvl x22, SP, #6\n"
    "addvl x22, x22, #-2\n"
    "ldr x21, [%x[qp], %[offsetof_Requantize32_per_channel_muls]]\n"
    "mov z17.d, z16.d\n"
    "mov z18.d, z16.d\n"
    "mov z19.d, z16.d\n"
    "mov x20, x23\n"
    "incw x23\n"
    "ld1b { z25.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #3\n"
    "ld1b { z15.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #3\n"
    "ld1b { z9.s }, p2/Z, [x20]\n"
    "mov x20, x23\n"
    "incw x23\n"
    "sub z25.h, z25.h, z22.h\n"
    "sub z15.h, z15.h, z22.h\n"
    "sub z9.h, z9.h, z22.h\n"
    "trn1 z24.h, z25.h, z15.h\n"
    "ld1b { z12.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #3\n"
    "ld1b { z4.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #3\n"
    "trn1 z11.h, z9.h, z27.h\n"
    "ld1b { z15.s }, p2/Z, [x20]\n"
    "mov x20, x23\n"
    "sub z12.h, z12.h, z22.h\n"
    "sub z4.h, z4.h, z22.h\n"
    "st1h { z24.h }, p2, [x22]\n"
    "sub z15.h, z15.h, z22.h\n"
    "st1h { z11.h }, p2, [x22, #1, MUL VL]\n"
    "addvl x22, x22, #-2\n"
    "trn1 z9.h, z12.h, z4.h\n"
    "ld1b { z14.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #3\n"
    "ld1b { z10.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #3\n"
    "trn1 z21.h, z15.h, z27.h\n"
    "ld1b { z30.s }, p2/Z, [x20]\n"
    "sub z14.h, z14.h, z22.h\n"
    "sub z10.h, z10.h, z22.h\n"
    "st1h { z9.h }, p2, [x22]\n"
    "sub z30.h, z30.h, z22.h\n"
    "st1h { z21.h }, p2, [x22, #1, MUL VL]\n"
    "addvl x22, x22, #-2\n"
    "trn1 z15.h, z14.h, z10.h\n"
    "trn1 z25.h, z30.h, z27.h\n"
    "st1h { z15.h }, p2, [x22]\n"
    "st1h { z25.h }, p2, [x22, #1, MUL VL]\n"
    "cbz x21, 3f\n"
    "ld1w { z1.s }, p1/Z, [x21, x16, LSL #2]\n"
    "3:"  // Load mul: End
    "ldr x20, [%x[qp], %[offsetof_Requantize32_per_channel_right_shifts]]\n"
    "cbz x20, 4f\n"
    "ld1w { z0.s }, p1/Z, [x20, x16, LSL #2]\n"
    "4:"  // Load right_shift: End
    "ldr x15, [%x[args], %[offsetof_Args_input_cols]]\n"
    "mov x22, #0x9\n"
    "add x20, x7, x6\n"
    "ldr x14, [%x[args], %[offsetof_Args_inptr]]\n"
    "lsl x21, %x[ld_in_row], #0x0\n"
    "mov x8, #0x0\n"
    "ldr x13, [%x[args], %[offsetof_Args_output_cols]]\n"
    "sub x22, x22, x20\n"
    "sub x20, x15, #0x1\n"
    "orr x20, x20, %x[ld_in_col], LSL #16\n"
    "madd x21, x21, x7, x14\n"
    "orr x20, x17, x20, LSL #22\n"
    "lsl x20, x20, #0x0\n"
    "5:"  // Issue prefetches
    "subs x22, x22, #0x1\n"
    ".inst 0xf8b44abc  // rprfm pldstrm, x20, [x21]\n"
    "add x21, x21, %x[ld_in_col]\n"
    "bgt 5b\n"
    "ldr x23, [%x[args], %[offsetof_Args_outptrs]]\n"
    "lsl x21, %x[ld_in_row], #0x0\n"
    ".inst 0xc0040e00  // mova za.d[x8, #0], { z16.d-z19.d }\n"
    "mov x22, #0x2\n"
    "ldr x20, [%x[args], %[offsetof_Args_ld_out_cols]]\n"
    "msub x14, x7, x21, x14\n"
    ".inst 0xc0040e01  // mova za.d[x8, #1], { z16.d-z19.d }\n"
    "ldr x21, [%x[args], %[offsetof_Args_pad_left]]\n"
    ".inst 0xc0040e02  // mova za.d[x8, #2], { z16.d-z19.d }\n"
    "ldp x11, x10, [x23], #0x10\n"
    "ldp x9, x28, [x20], #0x10\n"
    "ldp x27, x26, [x23], #0x10\n"
    "ldp x25, x24, [x20], #0x10\n"
    "cbz x21, 7f\n"
    "cmp x21, x22\n"
    "csel x20, x21, x22, LT\n"
    "sub x21, x21, x20\n"
    "sub x22, x22, x20\n"
    "cbz x21, 7f\n"
    ".inst 0xc0060c0c  // mova { z12.d-z15.d }, za.d[x8, #0]\n"
    "and x22, x21, #0x1\n"
    "add x21, x21, #0x1\n"
    "lsr x21, x21, #0x1\n"
    "sub x13, x13, x21\n"
    ".inst 0xc1a1ac0c  // sqdmulh { z12.s-z15.s }, { z12.s-z15.s }, z1.s\n"
    ".inst 0xc1a0aa2c  // srshl { z12.s-z15.s }, { z12.s-z15.s }, z0.s\n"
    ".inst 0xc1a8ab0c  // add { z12.s-z15.s }, { z12.s-z15.s }, z8.s\n"
    ".inst 0xc1bccfec  // sclamp { z12.s-z15.s }, z31.s, z28.s\n"
    "6:"  // Left padding
    "subs x21, x21, #0x1\n"
    "st1b { z12.s }, p1, [x11]\n"
    "add x11, x11, x9\n"
    "st1b { z13.s }, p1, [x10]\n"
    "add x10, x10, x28\n"
    "st1b { z14.s }, p1, [x27]\n"
    "add x27, x27, x25\n"
    "st1b { z15.s }, p1, [x26]\n"
    "add x26, x26, x24\n"
    "bgt 6b\n"
    "7:"  // Left padding: End
    "adds XZR, x7, x6\n"
    "bne 12f\n"
    "cbz x22, 10f\n"
    "cmp x22, #0x1\n"
    "sub x15, x15, x22\n"
    "beq 9f\n"
    "8:"  // Unpadded: 2 priming loads
    "add x21, x14, %x[ld_in_row]\n"
    "ld1b { z23.s }, p1/Z, [x14]\n"
    "addvl x20, SP, #4\n"
    "ld1b { z4.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add x14, x14, %x[ld_in_col]\n"
    "ld1b { z24.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1b { z5.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1b { z25.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z23.h, z23.h, z4.h\n"
    "ld1b { z6.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1b { z26.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z24.h, z24.h, z5.h\n"
    "ld1b { z20.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z25.h, z25.h, z6.h\n"
    "ld1b { z10.s }, p1/Z, [x21]\n"
    "add z23.h, z23.h, z29.h\n"
    ".inst 0xa1402a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20]\n"
    "trn1 z26.h, z26.h, z20.h\n"
    "add z24.h, z24.h, z29.h\n"
    "mov z27.d, z10.d\n"
    "add z25.h, z25.h, z29.h\n"
    "add z26.h, z26.h, z29.h\n"
    "add z27.h, z27.h, z29.h\n"
    ".inst 0xc17616e8  // sdot za.s[x8, 0], { z23.h-z26.h }, z6.h\n"
    ".inst 0xc17e1708  // sdot za.s[x8, 0], { z24.h-z27.h }, z14.h\n"
    "9:"  // Unpadded: 1 priming loads
    "add x21, x14, %x[ld_in_row]\n"
    "ld1b { z20.s }, p1/Z, [x14]\n"
    "addvl x20, SP, #2\n"
    "ld1b { z2.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add x14, x14, %x[ld_in_col]\n"
    "ld1b { z21.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1b { z25.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1b { z22.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z20.h, z20.h, z2.h\n"
    "ld1b { z24.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1b { z23.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z21.h, z21.h, z25.h\n"
    "ld1b { z9.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z22.h, z22.h, z24.h\n"
    "ld1b { z3.s }, p1/Z, [x21]\n"
    "add z20.h, z20.h, z29.h\n"
    ".inst 0xa0402a86  // ld1h { z6.h-z7.h }, pn10.b/Z, [x20]\n"
    "trn1 z23.h, z23.h, z9.h\n"
    "add z21.h, z21.h, z29.h\n"
    "mov z24.d, z3.d\n"
    "add z22.h, z22.h, z29.h\n"
    "add z23.h, z23.h, z29.h\n"
    "add z24.h, z24.h, z29.h\n"
    ".inst 0xc1761688  // sdot za.s[x8, 0], { z20.h-z23.h }, z6.h\n"
    ".inst 0xc17716a8  // sdot za.s[x8, 0], { z21.h-z24.h }, z7.h\n"
    "10:"  // Unpadded: 0 priming loads
    "cmp x15, #0x2\n"
    ".inst 0xa0402be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP]\n"
    "blt 18f\n"
    "add x21, x14, %x[ld_in_row]\n"
    "ld1b { z10.s }, p1/Z, [x14]\n"
    "sub x15, x15, #0x2\n"
    "ld1b { z25.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "sub x13, x13, #0x1\n"
    "ld1b { z11.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "lsr x20, x15, #0x1\n"
    "ld1b { z24.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "cmp x20, x13\n"
    "ld1b { z12.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z10.h, z10.h, z25.h\n"
    "csel x23, x20, x13, LT\n"
    "ld1b { z30.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add x14, x14, %x[ld_in_col]\n"
    "ld1b { z13.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z11.h, z11.h, z24.h\n"
    "and x15, x15, #0x1\n"
    "ld1b { z20.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "sub x13, x13, x23\n"
    "ld1b { z26.s }, p1/Z, [x21]\n"
    "trn1 z12.h, z12.h, z30.h\n"
    "add z10.h, z10.h, z29.h\n"
    "trn1 z13.h, z13.h, z20.h\n"
    "add z11.h, z11.h, z29.h\n"
    "mov z14.d, z26.d\n"
    "add z12.h, z12.h, z29.h\n"
    "add z13.h, z13.h, z29.h\n"
    "add z14.h, z14.h, z29.h\n"
    "cbz x23, 17f\n"
    "11:"  // Unpadded: Main loop
    ".inst 0xc1721548  // sdot za.s[x8, 0], { z10.h-z13.h }, z2.h\n"
    "addvl x20, SP, #4\n"
    "add x22, x14, %x[ld_in_row]\n"
    "addvl x21, SP, #2\n"
    "subs x23, x23, #0x1\n"
    ".inst 0xc1731568  // sdot za.s[x8, 0], { z11.h-z14.h }, z3.h\n"
    ".inst 0xa1402a87  // ld1h { z7.h, z15.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc1771549  // sdot za.s[x8, 1], { z10.h-z13.h }, z7.h\n"
    "ld1b { z3.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col]\n"
    "ld1b { z9.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc0060c18  // mova { z24.d-z27.d }, za.d[x8, #0]\n"
    "add x20, x14, %x[ld_in_row]\n"
    ".inst 0xc17f1569  // sdot za.s[x8, 1], { z11.h-z14.h }, z15.h\n"
    "ld1b { z4.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z3.h, z3.h, z9.h\n"
    "ld1b { z15.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add x8, x8, #0x1\n"
    "ld1b { z5.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc1a1ac18  // sqdmulh { z24.s-z27.s }, { z24.s-z27.s }, z1.s\n"
    "ld1b { z10.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc0040e02  // mova za.d[x8, #2], { z16.d-z19.d }\n"
    "ld1b { z6.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z4.h, z4.h, z15.h\n"
    "add z3.h, z3.h, z29.h\n"
    "ld1b { z14.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z5.h, z5.h, z10.h\n"
    "ld1b { z21.s }, p1/Z, [x22]\n"
    ".inst 0xc1a0aa38  // srshl { z24.s-z27.s }, { z24.s-z27.s }, z0.s\n"
    ".inst 0xa0402aaa  // ld1h { z10.h-z11.h }, pn10.b/Z, [x21]\n"
    "trn1 z6.h, z6.h, z14.h\n"
    "add z4.h, z4.h, z29.h\n"
    "mov z7.d, z21.d\n"
    "add z5.h, z5.h, z29.h\n"
    ".inst 0xc1a8ab18  // add { z24.s-z27.s }, { z24.s-z27.s }, z8.s\n"
    "add z6.h, z6.h, z29.h\n"
    "add z7.h, z7.h, z29.h\n"
    ".inst 0xc1bccff8  // sclamp { z24.s-z27.s }, z31.s, z28.s\n"
    ".inst 0xc17a1468  // sdot za.s[x8, 0], { z3.h-z6.h }, z10.h\n"
    "ld1b { z10.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col]\n"
    "ld1b { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "st1b { z24.s }, p1, [x11]\n"
    "add x11, x11, x9\n"
    ".inst 0xc17b1488  // sdot za.s[x8, 0], { z4.h-z7.h }, z11.h\n"
    "ld1b { z11.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z10.h, z10.h, z22.h\n"
    "ld1b { z14.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "st1b { z25.s }, p1, [x10]\n"
    "add x10, x10, x28\n"
    "ld1b { z12.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "st1b { z26.s }, p1, [x27]\n"
    "add x27, x27, x25\n"
    "ld1b { z9.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "st1b { z27.s }, p1, [x26]\n"
    "add x26, x26, x24\n"
    "ld1b { z13.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z11.h, z11.h, z14.h\n"
    "add z10.h, z10.h, z29.h\n"
    "ld1b { z6.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "ld1b { z20.s }, p1/Z, [x20]\n"
    "trn1 z12.h, z12.h, z9.h\n"
    ".inst 0xa0402be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP]\n"
    "trn1 z13.h, z13.h, z6.h\n"
    "add z11.h, z11.h, z29.h\n"
    "mov z14.d, z20.d\n"
    "add z12.h, z12.h, z29.h\n"
    "add z13.h, z13.h, z29.h\n"
    "add z14.h, z14.h, z29.h\n"
    "bgt 11b\n"
    "b 17f\n"
    "12:"  // Padded
    "cbz x22, 15f\n"
    "cmp x22, #0x1\n"
    "sub x15, x15, x22\n"
    "beq 14f\n"
    "13:"  // Padded: 2 priming loads
    "mov x12, #0x0\n"
    "add x21, x14, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "addvl x20, SP, #4\n"
    ".inst 0xa1402a83  // ld1h { z3.h, z11.h }, pn10.b/Z, [x20]\n"
    "ld1b { z20.s }, p0/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col]\n"
    "add z20.h, p0/M, z20.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z22.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z22.h, p0/M, z22.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z21.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z20.h, z20.h, z22.h\n"
    "add z21.h, p0/M, z21.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1b { z23.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z23.h, p0/M, z23.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z22.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z21.h, z21.h, z23.h\n"
    "add z22.h, p0/M, z22.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z24.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z24.h, p0/M, z24.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z23.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z22.h, z22.h, z24.h\n"
    "add z23.h, p0/M, z23.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    "ld1b { z25.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z25.h, p0/M, z25.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z24.s }, p0/Z, [x21]\n"
    "trn1 z23.h, z23.h, z25.h\n"
    "add z24.h, p0/M, z24.h, z29.h\n"
    ".inst 0xc1731688  // sdot za.s[x8, 0], { z20.h-z23.h }, z3.h\n"
    "mov z24.d, z24.d\n"
    ".inst 0xc17b16a8  // sdot za.s[x8, 0], { z21.h-z24.h }, z11.h\n"
    "14:"  // Padded: 1 priming loads
    "mov x12, #0x0\n"
    "add x21, x14, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "addvl x20, SP, #2\n"
    ".inst 0xa1402a83  // ld1h { z3.h, z11.h }, pn10.b/Z, [x20]\n"
    "ld1b { z21.s }, p0/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col]\n"
    "add z21.h, p0/M, z21.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z20.h, p0/M, z20.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z22.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z21.h, z21.h, z20.h\n"
    "add z22.h, p0/M, z22.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z20.h, p0/M, z20.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z23.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z22.h, z22.h, z20.h\n"
    "add z23.h, p0/M, z23.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z20.h, p0/M, z20.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z24.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z23.h, z23.h, z20.h\n"
    "add z24.h, p0/M, z24.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    "ld1b { z25.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z25.h, p0/M, z25.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "trn1 z24.h, z24.h, z25.h\n"
    "add z20.h, p0/M, z20.h, z29.h\n"
    ".inst 0xc17316a8  // sdot za.s[x8, 0], { z21.h-z24.h }, z3.h\n"
    "mov z25.d, z20.d\n"
    ".inst 0xc17b16c8  // sdot za.s[x8, 0], { z22.h-z25.h }, z11.h\n"
    "15:"  // Padded: 0 priming loads
    "cmp x15, #0x2\n"
    ".inst 0xa0402be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP]\n"
    "blt 18f\n"
    "mov x12, #0x0\n"
    "add x21, x14, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "sub x15, x15, #0x2\n"
    "sub x13, x13, #0x1\n"
    "lsr x20, x15, #0x1\n"
    "cmp x20, x13\n"
    "and x15, x15, #0x1\n"
    "ld1b { z10.s }, p0/Z, [x14]\n"
    "csel x23, x20, x13, LT\n"
    "add x14, x14, %x[ld_in_col]\n"
    "sub x13, x13, x23\n"
    "add z10.h, p0/M, z10.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z20.h, p0/M, z20.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z11.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z10.h, z10.h, z20.h\n"
    "add z11.h, p0/M, z11.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z20.h, p0/M, z20.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z12.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z11.h, z11.h, z20.h\n"
    "add z12.h, p0/M, z12.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z20.h, p0/M, z20.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z13.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z12.h, z12.h, z20.h\n"
    "add z13.h, p0/M, z13.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    "ld1b { z21.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z21.h, p0/M, z21.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "trn1 z13.h, z13.h, z21.h\n"
    "add z20.h, p0/M, z20.h, z29.h\n"
    "mov z14.d, z20.d\n"
    "cbz x23, 17f\n"
    "16:"  // Padded: Main loop
    ".inst 0xc1721548  // sdot za.s[x8, 0], { z10.h-z13.h }, z2.h\n"
    "addvl x20, SP, #4\n"
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x22, x14, %x[ld_in_row]\n"
    "addvl x21, SP, #2\n"
    "subs x23, x23, #0x1\n"
    ".inst 0xc1731568  // sdot za.s[x8, 0], { z11.h-z14.h }, z3.h\n"
    ".inst 0xa0402a82  // ld1h { z2.h-z3.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc1721549  // sdot za.s[x8, 1], { z10.h-z13.h }, z2.h\n"
    "ld1b { z10.s }, p0/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col]\n"
    ".inst 0xc0060c04  // mova { z4.d-z7.d }, za.d[x8, #0]\n"
    "add x20, x14, %x[ld_in_row]\n"
    "add z10.h, p0/M, z10.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc1731569  // sdot za.s[x8, 1], { z11.h-z14.h }, z3.h\n"
    ".inst 0xa0402aa2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x21]\n"
    "add x8, x8, #0x1\n"
    ".inst 0xc1a1ac04  // sqdmulh { z4.s-z7.s }, { z4.s-z7.s }, z1.s\n"
    ".inst 0xc0040e02  // mova za.d[x8, #2], { z16.d-z19.d }\n"
    "ld1b { z26.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add z26.h, p0/M, z26.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc1a0aa24  // srshl { z4.s-z7.s }, { z4.s-z7.s }, z0.s\n"
    "ld1b { z11.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z10.h, z10.h, z26.h\n"
    ".inst 0xc1a8ab04  // add { z4.s-z7.s }, { z4.s-z7.s }, z8.s\n"
    "add z11.h, p0/M, z11.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1b { z9.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc1bccfe4  // sclamp { z4.s-z7.s }, z31.s, z28.s\n"
    "add z9.h, p0/M, z9.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "st1b { z4.s }, p1, [x11]\n"
    "add x11, x11, x9\n"
    "st1b { z5.s }, p1, [x10]\n"
    "add x10, x10, x28\n"
    "ld1b { z12.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "st1b { z6.s }, p1, [x27]\n"
    "add x27, x27, x25\n"
    "trn1 z11.h, z11.h, z9.h\n"
    "st1b { z7.s }, p1, [x26]\n"
    "add x26, x26, x24\n"
    "add z12.h, p0/M, z12.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z9.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add z9.h, p0/M, z9.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z13.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z12.h, z12.h, z9.h\n"
    "add z13.h, p0/M, z13.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    "ld1b { z20.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add z20.h, p0/M, z20.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "mov x12, #0x0\n"
    "ld1b { z9.s }, p0/Z, [x22]\n"
    "trn1 z13.h, z13.h, z20.h\n"
    "add z9.h, p0/M, z9.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc1721548  // sdot za.s[x8, 0], { z10.h-z13.h }, z2.h\n"
    "ld1b { z10.s }, p0/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col]\n"
    "mov z14.d, z9.d\n"
    "add z10.h, p0/M, z10.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc1731568  // sdot za.s[x8, 0], { z11.h-z14.h }, z3.h\n"
    ".inst 0xa0402be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP]\n"
    "ld1b { z25.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "add z25.h, p0/M, z25.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z11.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z10.h, z10.h, z25.h\n"
    "add z11.h, p0/M, z11.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1b { z15.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "add z15.h, p0/M, z15.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z12.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z11.h, z11.h, z15.h\n"
    "add z12.h, p0/M, z12.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z4.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "add z4.h, p0/M, z4.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z13.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z12.h, z12.h, z4.h\n"
    "add z13.h, p0/M, z13.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    "ld1b { z4.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "add z4.h, p0/M, z4.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z26.s }, p0/Z, [x20]\n"
    "trn1 z13.h, z13.h, z4.h\n"
    "add z26.h, p0/M, z26.h, z29.h\n"
    "mov z14.d, z26.d\n"
    "bgt 16b\n"
    "17:"  // Main loop tail
    ".inst 0xc1721548  // sdot za.s[x8, 0], { z10.h-z13.h }, z2.h\n"
    "addvl x22, SP, #4\n"
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x21, x14, %x[ld_in_row]\n"
    "addvl x20, SP, #2\n"
    ".inst 0xc1731568  // sdot za.s[x8, 0], { z11.h-z14.h }, z3.h\n"
    ".inst 0xa0402ac2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x22]\n"
    ".inst 0xc1721549  // sdot za.s[x8, 1], { z10.h-z13.h }, z2.h\n"
    "ld1b { z9.s }, p0/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col]\n"
    ".inst 0xc0060c04  // mova { z4.d-z7.d }, za.d[x8, #0]\n"
    "add z9.h, p0/M, z9.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc1731569  // sdot za.s[x8, 1], { z11.h-z14.h }, z3.h\n"
    ".inst 0xa0402a82  // ld1h { z2.h-z3.h }, pn10.b/Z, [x20]\n"
    "add x8, x8, #0x1\n"
    ".inst 0xc1a1ac04  // sqdmulh { z4.s-z7.s }, { z4.s-z7.s }, z1.s\n"
    ".inst 0xc0040e02  // mova za.d[x8, #2], { z16.d-z19.d }\n"
    "ld1b { z15.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z15.h, p0/M, z15.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc1a0aa24  // srshl { z4.s-z7.s }, { z4.s-z7.s }, z0.s\n"
    "ld1b { z10.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z9.h, z9.h, z15.h\n"
    ".inst 0xc1a8ab04  // add { z4.s-z7.s }, { z4.s-z7.s }, z8.s\n"
    "add z10.h, p0/M, z10.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1b { z15.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0xc1bccfe4  // sclamp { z4.s-z7.s }, z31.s, z28.s\n"
    "add z15.h, p0/M, z15.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "st1b { z4.s }, p1, [x11]\n"
    "add x11, x11, x9\n"
    "st1b { z5.s }, p1, [x10]\n"
    "add x10, x10, x28\n"
    "ld1b { z11.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "st1b { z6.s }, p1, [x27]\n"
    "add x27, x27, x25\n"
    "trn1 z10.h, z10.h, z15.h\n"
    "st1b { z7.s }, p1, [x26]\n"
    "add x26, x26, x24\n"
    "add z11.h, p0/M, z11.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z5.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z5.h, p0/M, z5.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z12.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z11.h, z11.h, z5.h\n"
    "add z12.h, p0/M, z12.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    "ld1b { z15.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z15.h, p0/M, z15.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z5.s }, p0/Z, [x21]\n"
    "trn1 z12.h, z12.h, z15.h\n"
    "add z5.h, p0/M, z5.h, z29.h\n"
    ".inst 0xc1721528  // sdot za.s[x8, 0], { z9.h-z12.h }, z2.h\n"
    "mov z13.d, z5.d\n"
    ".inst 0xc1731548  // sdot za.s[x8, 0], { z10.h-z13.h }, z3.h\n"
    ".inst 0xa0402be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP]\n"
    "18:"  // Main loop skip tail
    "cbz x15, 19f\n"  // Skip remainder inputs
    "mov x12, #0x0\n"
    "add x21, x14, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "addvl x20, SP, #4\n"
    "sub x13, x13, #0x1\n"
    "ld1b { z21.s }, p0/Z, [x14]\n"
    "add z21.h, p0/M, z21.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z15.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z15.h, p0/M, z15.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z22.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z21.h, z21.h, z15.h\n"
    "add z22.h, p0/M, z22.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1b { z12.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z12.h, p0/M, z12.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z23.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z22.h, z22.h, z12.h\n"
    "add z23.h, p0/M, z23.h, z29.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1b { z20.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z20.h, p0/M, z20.h, z29.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1b { z24.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z23.h, z23.h, z20.h\n"
    "add z24.h, p0/M, z24.h, z29.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x8\n"
    "ld1b { z30.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z30.h, p0/M, z30.h, z29.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1b { z6.s }, p0/Z, [x21]\n"
    "trn1 z24.h, z24.h, z30.h\n"
    "add z6.h, p0/M, z6.h, z29.h\n"
    ".inst 0xc17216a8  // sdot za.s[x8, 0], { z21.h-z24.h }, z2.h\n"
    "mov z25.d, z6.d\n"
    ".inst 0xc17316c8  // sdot za.s[x8, 0], { z22.h-z25.h }, z3.h\n"
    ".inst 0xa1402a85  // ld1h { z5.h, z13.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc17516a9  // sdot za.s[x8, 1], { z21.h-z24.h }, z5.h\n"
    ".inst 0xc0060c04  // mova { z4.d-z7.d }, za.d[x8, #0]\n"
    ".inst 0xc17d16c9  // sdot za.s[x8, 1], { z22.h-z25.h }, z13.h\n"
    "add x8, x8, #0x1\n"
    ".inst 0xc1a1ac04  // sqdmulh { z4.s-z7.s }, { z4.s-z7.s }, z1.s\n"
    ".inst 0xc0040e02  // mova za.d[x8, #2], { z16.d-z19.d }\n"
    ".inst 0xc1a0aa24  // srshl { z4.s-z7.s }, { z4.s-z7.s }, z0.s\n"
    ".inst 0xc1a8ab04  // add { z4.s-z7.s }, { z4.s-z7.s }, z8.s\n"
    ".inst 0xc1bccfe4  // sclamp { z4.s-z7.s }, z31.s, z28.s\n"
    "st1b { z4.s }, p1, [x11]\n"
    "add x11, x11, x9\n"
    "st1b { z5.s }, p1, [x10]\n"
    "add x10, x10, x28\n"
    "st1b { z6.s }, p1, [x27]\n"
    "add x27, x27, x25\n"
    "st1b { z7.s }, p1, [x26]\n"
    "add x26, x26, x24\n"
    "19:"  // Tail input: End
    "cbz x13, 21f\n"
    "20:"  // Right padding loop
    ".inst 0xc0060c04  // mova { z4.d-z7.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    "subs x13, x13, #0x1\n"
    ".inst 0xc0040e02  // mova za.d[x8, #2], { z16.d-z19.d }\n"
    ".inst 0xc1a1ac04  // sqdmulh { z4.s-z7.s }, { z4.s-z7.s }, z1.s\n"
    ".inst 0xc1a0aa24  // srshl { z4.s-z7.s }, { z4.s-z7.s }, z0.s\n"
    ".inst 0xc1a8ab04  // add { z4.s-z7.s }, { z4.s-z7.s }, z8.s\n"
    ".inst 0xc1bccfe4  // sclamp { z4.s-z7.s }, z31.s, z28.s\n"
    "st1b { z4.s }, p1, [x11]\n"
    "add x11, x11, x9\n"
    "st1b { z5.s }, p1, [x10]\n"
    "add x10, x10, x28\n"
    "st1b { z6.s }, p1, [x27]\n"
    "add x27, x27, x25\n"
    "st1b { z7.s }, p1, [x26]\n"
    "add x26, x26, x24\n"
    "bgt 20b\n"
    "21:"  // End
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    "incw x16\n"
    "whilelt p1.s, x16, x17\n"
    "incw x20, ALL, MUL #9\n"
    "str x20, [%x[args], %[offsetof_Args_weights]]\n"
    "ldr x21, [%x[args], %[offsetof_Args_ld_in_vl]]\n"
    "ldr x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "add x20, x20, x21\n"
    "str x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "ldr x25, [%x[args], %[offsetof_Args_outptrs]]\n"
    "ldr x24, [%x[args], %[offsetof_Args_ld_out_vls]]\n"
    "ldp x23, x22, [x25, #0x0]\n"
    "ldp x21, x20, [x24, #0x0]\n"
    "add x23, x23, x21\n"
    "add x22, x22, x20\n"
    "stp x23, x22, [x25, #0x0]\n"
    "ldp x23, x22, [x25, #0x10]\n"
    "ldp x21, x20, [x24, #0x10]\n"
    "add x23, x23, x21\n"
    "add x22, x22, x20\n"
    "stp x23, x22, [x25, #0x10]\n"
    "b.any 1b\n"
    "addvl SP, SP, #6\n"
    ".inst 0xd503467f  // SMSTOP\n"
    "ldr x20, [SP, #0x0]\n"
    "mov SP, x20\n"
    :
    : [args] "r" (&args), [ld_in_col] "r" (ld_in_col), [ld_in_row] "r" (ld_in_row), [offsetof_Args_current_channel] "I" (offsetof(Args, current_channel)), [offsetof_Args_inptr] "I" (offsetof(Args, inptr)), [offsetof_Args_input_cols] "I" (offsetof(Args, input_cols)), [offsetof_Args_ld_in_vl] "I" (offsetof(Args, ld_in_vl)), [offsetof_Args_ld_out_cols] "I" (offsetof(Args, ld_out_cols)), [offsetof_Args_ld_out_vls] "I" (offsetof(Args, ld_out_vls)), [offsetof_Args_n_channels] "I" (offsetof(Args, n_channels)), [offsetof_Args_outptrs] "I" (offsetof(Args, outptrs)), [offsetof_Args_output_cols] "I" (offsetof(Args, output_cols)), [offsetof_Args_pad_bottom] "I" (offsetof(Args, pad_bottom)), [offsetof_Args_pad_left] "I" (offsetof(Args, pad_left)), [offsetof_Args_pad_top] "I" (offsetof(Args, pad_top)), [offsetof_Args_weights] "I" (offsetof(Args, weights)), [offsetof_Requantize32_a_offset] "I" (offsetof(arm_gemm::Requantize32, a_offset)), [offsetof_Requantize32_b_offset] "I" (offsetof(arm_gemm::Requantize32, b_offset)), [offsetof_Requantize32_bias] "I" (offsetof(arm_gemm::Requantize32, bias)), [offsetof_Requantize32_c_offset] "I" (offsetof(arm_gemm::Requantize32, c_offset)), [offsetof_Requantize32_maxval] "I" (offsetof(arm_gemm::Requantize32, maxval)), [offsetof_Requantize32_minval] "I" (offsetof(arm_gemm::Requantize32, minval)), [offsetof_Requantize32_per_channel_muls] "I" (offsetof(arm_gemm::Requantize32, per_channel_muls)), [offsetof_Requantize32_per_channel_right_shifts] "I" (offsetof(arm_gemm::Requantize32, per_channel_right_shifts)), [offsetof_Requantize32_per_layer_mul] "I" (offsetof(arm_gemm::Requantize32, per_layer_mul)), [offsetof_Requantize32_per_layer_right_shift] "I" (offsetof(arm_gemm::Requantize32, per_layer_right_shift)), [qp] "r" (&qp)
    : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
  );
}

}  // namespace depthwise
}  // namespace arm_conv

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
