/*
 * Copyright (c) 2022-2024, 2026 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include <algorithm>
#include <cstddef>
#include "arm_gemm/arm_gemm.hpp"

using arm_gemm::Requantize32;

namespace arm_conv {
namespace depthwise {

void sme2_s8q_planar_5x5_s1_4rows_dot_za_impl(
  const int8_t *inptr,
  size_t ld_in_row,
  size_t ld_in_col,
  size_t ld_in_vl,
  unsigned int pad_top,
  unsigned int valid_input_rows,
  unsigned int pad_left,
  unsigned int valid_input_cols,
  const int8_t *weights,
  int8_t **outptrs,
  const size_t *outlds,
  const size_t *outvllds,
  unsigned int output_cols,
  unsigned int start_channel,
  unsigned int valid_channels,
  const arm_gemm::Requantize32 &qp
)
{
  struct Args
  {
    const int8_t *inptr;
    size_t ld_in_vl;
    long unsigned int pad_top, pad_bottom, pad_left;
    const int8_t *weights;
    long unsigned int input_cols, output_cols;
    int8_t **outptrs;
    const size_t *ld_out_cols;
    const size_t *ld_out_vls;
    long unsigned int current_channel, n_channels;
  };

  Args args = { inptr, ld_in_vl, pad_top, 8u - std::min(8u, pad_top + valid_input_rows), pad_left, weights, valid_input_cols, output_cols, outptrs, outlds, outvllds, start_channel, valid_channels };

  __asm__ __volatile__(
    ".inst 0xd503477f  // SMSTART ZA\n"
    "ldr x5, [%x[args], %[offsetof_Args_pad_bottom]]\n"
    "mov x22, SP\n"
    "mov x20, #0x8\n"
    "ptrue p2.b\n"
    "ldr x6, [%x[args], %[offsetof_Args_pad_top]]\n"
    "sub x21, x22, #0x8\n"
    "ldr x7, [%x[args], %[offsetof_Args_n_channels]]\n"
    "ld1rh { z15.h }, p2/Z, [%x[qp], %[offsetof_Requantize32_a_offset]]\n"
    "and x21, x21, #-0x400\n"
    ".inst 0x25207812  // ptrue pn10.b\n"
    "sub x20, x20, x5\n"
    "mov SP, x21\n"
    "ldr x17, [%x[args], %[offsetof_Args_current_channel]]\n"
    "ld1rw { z10.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_c_offset]]\n"
    "str x22, [SP]\n"
    "addvl SP, SP, #-30\n"
    "ld1rw { z8.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_per_layer_mul]]\n"
    "whilelt p1.s, XZR, x7\n"
    "whilelt p9.s, XZR, x20\n"
    "neg z15.h, p2/M, z15.h\n"
    "ld1rw { z11.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_per_layer_right_shift]]\n"
    "whilelt p8.s, XZR, x6\n"
    "ld1rw { z31.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_minval]]\n"
    "eor p8.b, p2/Z, p8.b, p9.b\n"
    "ld1rw { z30.s }, p2/Z, [%x[qp], %[offsetof_Requantize32_maxval]]\n"
    "1:"  // Channel loop
    "ldr x20, [%x[qp], %[offsetof_Requantize32_bias]]\n"
    "mov z28.s, #0x0\n"
    "cbz x20, 2f\n"
    "ld1w { z28.s }, p1/Z, [x20, x17, LSL #2]\n"
    "2:"  // Load bias: Done
    "ldr x24, [%x[args], %[offsetof_Args_weights]]\n"
    "ld1rh { z0.h }, p2/Z, [%x[qp], %[offsetof_Requantize32_b_offset]]\n"
    "mov z13.h, #0x0\n"
    "addvl x22, SP, #30\n"
    "addvl x22, x22, #-6\n"
    "ldr x21, [%x[qp], %[offsetof_Requantize32_per_channel_muls]]\n"
    "mov z29.d, z28.d\n"
    "mov x23, x24\n"
    "incw x24\n"
    "ld1sb { z22.s }, p2/Z, [x23]\n"
    "incw x23, ALL, MUL #5\n"
    "ld1sb { z21.s }, p2/Z, [x23]\n"
    "incw x23, ALL, MUL #5\n"
    "ld1sb { z19.s }, p2/Z, [x23]\n"
    "incw x23, ALL, MUL #5\n"
    "ld1sb { z25.s }, p2/Z, [x23]\n"
    "incw x23, ALL, MUL #5\n"
    "sub z22.h, z22.h, z0.h\n"
    "sub z21.h, z21.h, z0.h\n"
    "ld1sb { z5.s }, p2/Z, [x23]\n"
    "mov x20, x24\n"
    "incw x24\n"
    "sub z19.h, z19.h, z0.h\n"
    "sub z25.h, z25.h, z0.h\n"
    "sub z5.h, z5.h, z0.h\n"
    "trn1 z6.h, z13.h, z22.h\n"
    "trn1 z23.h, z22.h, z21.h\n"
    "ld1sb { z27.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z4.h, z21.h, z19.h\n"
    "ld1sb { z9.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z26.h, z19.h, z25.h\n"
    "ld1sb { z18.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z22.h, z25.h, z5.h\n"
    "ld1sb { z7.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z25.h, z5.h, z13.h\n"
    "sub z27.h, z27.h, z0.h\n"
    "sub z9.h, z9.h, z0.h\n"
    "ld1sb { z1.s }, p2/Z, [x20]\n"
    "mov x20, x24\n"
    "sub z18.h, z18.h, z0.h\n"
    "st1h { z6.h }, p2, [x22]\n"
    "incw x24\n"
    "sub z7.h, z7.h, z0.h\n"
    "st1h { z23.h }, p2, [x22, #1, MUL VL]\n"
    "sub z1.h, z1.h, z0.h\n"
    "st1h { z4.h }, p2, [x22, #2, MUL VL]\n"
    "trn1 z20.h, z13.h, z27.h\n"
    "trn1 z12.h, z27.h, z9.h\n"
    "ld1sb { z21.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z2.h, z9.h, z18.h\n"
    "ld1sb { z19.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z26.h }, p2, [x22, #3, MUL VL]\n"
    "trn1 z9.h, z18.h, z7.h\n"
    "ld1sb { z14.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z22.h }, p2, [x22, #4, MUL VL]\n"
    "trn1 z18.h, z7.h, z1.h\n"
    "ld1sb { z5.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z25.h }, p2, [x22, #5, MUL VL]\n"
    "trn1 z25.h, z1.h, z13.h\n"
    "sub z21.h, z21.h, z0.h\n"
    "sub z19.h, z19.h, z0.h\n"
    "addvl x22, x22, #-6\n"
    "ld1sb { z16.s }, p2/Z, [x20]\n"
    "sub z14.h, z14.h, z0.h\n"
    "mov x20, x24\n"
    "st1h { z20.h }, p2, [x22]\n"
    "sub z5.h, z5.h, z0.h\n"
    "st1h { z12.h }, p2, [x22, #1, MUL VL]\n"
    "incw x24\n"
    "st1h { z2.h }, p2, [x22, #2, MUL VL]\n"
    "sub z16.h, z16.h, z0.h\n"
    "trn1 z7.h, z13.h, z21.h\n"
    "trn1 z20.h, z21.h, z19.h\n"
    "ld1sb { z6.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z17.h, z19.h, z14.h\n"
    "st1h { z9.h }, p2, [x22, #3, MUL VL]\n"
    "ld1sb { z2.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z12.h, z14.h, z5.h\n"
    "st1h { z18.h }, p2, [x22, #4, MUL VL]\n"
    "ld1sb { z21.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z25.h }, p2, [x22, #5, MUL VL]\n"
    "addvl x22, x22, #-6\n"
    "trn1 z5.h, z5.h, z16.h\n"
    "ld1sb { z25.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z4.h, z16.h, z13.h\n"
    "sub z6.h, z6.h, z0.h\n"
    "sub z2.h, z2.h, z0.h\n"
    "ld1sb { z19.s }, p2/Z, [x20]\n"
    "mov x20, x24\n"
    "sub z21.h, z21.h, z0.h\n"
    "st1h { z7.h }, p2, [x22]\n"
    "sub z25.h, z25.h, z0.h\n"
    "st1h { z20.h }, p2, [x22, #1, MUL VL]\n"
    "sub z19.h, z19.h, z0.h\n"
    "st1h { z17.h }, p2, [x22, #2, MUL VL]\n"
    "trn1 z1.h, z13.h, z6.h\n"
    "trn1 z24.h, z6.h, z2.h\n"
    "ld1sb { z23.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "trn1 z16.h, z2.h, z21.h\n"
    "ld1sb { z6.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z12.h }, p2, [x22, #3, MUL VL]\n"
    "trn1 z20.h, z21.h, z25.h\n"
    "ld1sb { z14.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z5.h }, p2, [x22, #4, MUL VL]\n"
    "trn1 z17.h, z25.h, z19.h\n"
    "ld1sb { z22.s }, p2/Z, [x20]\n"
    "incw x20, ALL, MUL #5\n"
    "st1h { z4.h }, p2, [x22, #5, MUL VL]\n"
    "trn1 z19.h, z19.h, z13.h\n"
    "ld1sb { z27.s }, p2/Z, [x20]\n"
    "sub z23.h, z23.h, z0.h\n"
    "addvl x22, x22, #-6\n"
    "sub z6.h, z6.h, z0.h\n"
    "sub z14.h, z14.h, z0.h\n"
    "st1h { z1.h }, p2, [x22]\n"
    "sub z22.h, z22.h, z0.h\n"
    "st1h { z24.h }, p2, [x22, #1, MUL VL]\n"
    "sub z27.h, z27.h, z0.h\n"
    "st1h { z16.h }, p2, [x22, #2, MUL VL]\n"
    "st1h { z20.h }, p2, [x22, #3, MUL VL]\n"
    "trn1 z16.h, z13.h, z23.h\n"
    "st1h { z17.h }, p2, [x22, #4, MUL VL]\n"
    "trn1 z7.h, z23.h, z6.h\n"
    "trn1 z12.h, z6.h, z14.h\n"
    "st1h { z19.h }, p2, [x22, #5, MUL VL]\n"
    "addvl x22, x22, #-6\n"
    "trn1 z5.h, z14.h, z22.h\n"
    "trn1 z14.h, z22.h, z27.h\n"
    "trn1 z20.h, z27.h, z13.h\n"
    "st1h { z16.h }, p2, [x22]\n"
    "st1h { z7.h }, p2, [x22, #1, MUL VL]\n"
    "st1h { z12.h }, p2, [x22, #2, MUL VL]\n"
    "st1h { z5.h }, p2, [x22, #3, MUL VL]\n"
    "st1h { z14.h }, p2, [x22, #4, MUL VL]\n"
    "st1h { z20.h }, p2, [x22, #5, MUL VL]\n"
    "cbz x21, 3f\n"
    "ld1w { z8.s }, p1/Z, [x21, x17, LSL #2]\n"
    "3:"  // Load mul: End
    "ldr x20, [%x[qp], %[offsetof_Requantize32_per_channel_right_shifts]]\n"
    "cbz x20, 4f\n"
    "ld1w { z11.s }, p1/Z, [x20, x17, LSL #2]\n"
    "4:"  // Load right_shift: End
    "ldr x25, [%x[args], %[offsetof_Args_input_cols]]\n"
    "mov x22, #0x8\n"
    "add x20, x6, x5\n"
    "ldr x16, [%x[args], %[offsetof_Args_inptr]]\n"
    "lsl x21, %x[ld_in_row], #0x0\n"
    "mov x11, #0x0\n"
    "mov x8, #0x8\n"
    "ldr x15, [%x[args], %[offsetof_Args_output_cols]]\n"
    "sub x22, x22, x20\n"
    "sub x20, x25, #0x1\n"
    "orr x20, x20, %x[ld_in_col], LSL #16\n"
    "madd x21, x21, x6, x16\n"
    "orr x20, x7, x20, LSL #22\n"
    "lsl x20, x20, #0x0\n"
    "5:"  // Issue prefetches
    "subs x22, x22, #0x1\n"
    ".inst 0xf8b44abc  // rprfm pldstrm, x20, [x21]\n"
    "add x21, x21, %x[ld_in_col]\n"
    "bgt 5b\n"
    "ldr x23, [%x[args], %[offsetof_Args_outptrs]]\n"
    "lsl x21, %x[ld_in_row], #0x0\n"
    ".inst 0xc0046b80  // mova za.d[x11, #0], { z28.d-z29.d }\n"
    "mov x22, #0x4\n"
    "ldr x20, [%x[args], %[offsetof_Args_ld_out_cols]]\n"
    "msub x16, x6, x21, x16\n"
    ".inst 0xc0046b81  // mova za.d[x11, #1], { z28.d-z29.d }\n"
    "ldr x21, [%x[args], %[offsetof_Args_pad_left]]\n"
    ".inst 0xc0046b82  // mova za.d[x11, #2], { z28.d-z29.d }\n"
    "ldp x14, x13, [x23], #0x10\n"
    ".inst 0xc0046b83  // mova za.d[x11, #3], { z28.d-z29.d }\n"
    "ldp x4, x10, [x20], #0x10\n"
    ".inst 0xc0046b84  // mova za.d[x11, #4], { z28.d-z29.d }\n"
    ".inst 0xc0046b85  // mova za.d[x11, #5], { z28.d-z29.d }\n"
    "ldp x9, x28, [x23], #0x10\n"
    ".inst 0xc0046b86  // mova za.d[x11, #6], { z28.d-z29.d }\n"
    "ldp x27, x26, [x20], #0x10\n"
    ".inst 0xc0046b87  // mova za.d[x11, #7], { z28.d-z29.d }\n"
    ".inst 0xc0040b80  // mova za.d[x8, #0], { z28.d-z29.d }\n"
    ".inst 0xc0040b81  // mova za.d[x8, #1], { z28.d-z29.d }\n"
    "cbz x21, 7f\n"
    "cmp x21, x22\n"
    "csel x20, x21, x22, LT\n"
    "sub x21, x21, x20\n"
    "sub x22, x22, x20\n"
    "cbz x21, 7f\n"
    ".inst 0xc0066810  // mova { z16.d-z17.d }, za.d[x11, #0]\n"
    "sub x15, x15, x21\n"
    ".inst 0xc0066832  // mova { z18.d-z19.d }, za.d[x11, #1]\n"
    ".inst 0xc1a8ac10  // sqdmulh { z16.s-z19.s }, { z16.s-z19.s }, z8.s\n"
    ".inst 0xc1abaa30  // srshl { z16.s-z19.s }, { z16.s-z19.s }, z11.s\n"
    ".inst 0xc1aaab10  // add { z16.s-z19.s }, { z16.s-z19.s }, z10.s\n"
    ".inst 0xc1becff0  // sclamp { z16.s-z19.s }, z31.s, z30.s\n"
    "6:"  // Left padding
    "subs x21, x21, #0x1\n"
    "st1b { z16.s }, p1, [x14]\n"
    "add x14, x14, x4\n"
    "st1b { z18.s }, p1, [x13]\n"
    "add x13, x13, x10\n"
    "st1b { z17.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "st1b { z19.s }, p1, [x28]\n"
    "add x28, x28, x26\n"
    "bgt 6b\n"
    "7:"  // Left padding: End
    "adds XZR, x6, x5\n"
    "bne 14f\n"
    "cbz x22, 12f\n"
    "cmp x22, #0x1\n"
    "sub x25, x25, x22\n"
    "beq 11f\n"
    "cmp x22, #0x2\n"
    "beq 10f\n"
    "cmp x22, #0x3\n"
    "beq 9f\n"
    "8:"  // Unpadded: 4 priming loads
    "add x21, x16, %x[ld_in_row]\n"
    "ld1sb { z4.s }, p1/Z, [x16]\n"
    "addvl x20, SP, #24\n"
    "ld1sb { z13.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add x16, x16, %x[ld_in_col]\n"
    "ld1sb { z25.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1sb { z19.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "ld1sb { z14.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z22.h, z4.h, z13.h\n"
    "ld1sb { z27.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z23.h, z25.h, z19.h\n"
    "ld1sb { z21.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0xa0402a8c  // ld1h { z12.h-z13.h }, pn10.b/Z, [x20]\n"
    "add z22.h, z22.h, z15.h\n"
    "trn1 z24.h, z14.h, z27.h\n"
    "ld1sb { z20.s }, p1/Z, [x21]\n"
    ".inst 0xa1412a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "add z23.h, z23.h, z15.h\n"
    ".inst 0xa0422a80  // ld1h { z0.h-z1.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    "trn1 z25.h, z21.h, z20.h\n"
    "add z24.h, z24.h, z15.h\n"
    ".inst 0xc16d76c8  // sdot za.s[x11, 0], { z22.h-z23.h }, z13.h\n"
    ".inst 0xc16c76c9  // sdot za.s[x11, 1], { z22.h-z23.h }, z12.h\n"
    "add z25.h, z25.h, z15.h\n"
    ".inst 0xc16e76e8  // sdot za.s[x11, 0], { z23.h-z24.h }, z14.h\n"
    ".inst 0xc16676e9  // sdot za.s[x11, 1], { z23.h-z24.h }, z6.h\n"
    ".inst 0xc1617708  // sdot za.s[x11, 0], { z24.h-z25.h }, z1.h\n"
    ".inst 0xc1607709  // sdot za.s[x11, 1], { z24.h-z25.h }, z0.h\n"
    "9:"  // Unpadded: 3 priming loads
    "add x22, x16, %x[ld_in_row]\n"
    "ld1sb { z21.s }, p1/Z, [x16]\n"
    "addvl x21, SP, #18\n"
    "ld1sb { z18.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "addvl x20, SP, #24\n"
    "ld1sb { z17.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add x16, x16, %x[ld_in_col]\n"
    "ld1sb { z3.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "ld1sb { z27.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z24.h, z21.h, z18.h\n"
    "ld1sb { z7.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z25.h, z17.h, z3.h\n"
    "ld1sb { z17.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xa0402aa2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x21]\n"
    "add z24.h, z24.h, z15.h\n"
    "trn1 z26.h, z27.h, z7.h\n"
    "ld1sb { z16.s }, p1/Z, [x22]\n"
    ".inst 0xa1412aa1  // ld1h { z1.h, z9.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    "add z25.h, z25.h, z15.h\n"
    ".inst 0xa0422aa6  // ld1h { z6.h-z7.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    "trn1 z27.h, z17.h, z16.h\n"
    "add z26.h, z26.h, z15.h\n"
    ".inst 0xc1637708  // sdot za.s[x11, 0], { z24.h-z25.h }, z3.h\n"
    ".inst 0xc1627709  // sdot za.s[x11, 1], { z24.h-z25.h }, z2.h\n"
    ".inst 0xa0402a8c  // ld1h { z12.h-z13.h }, pn10.b/Z, [x20]\n"
    "add z27.h, z27.h, z15.h\n"
    ".inst 0xc16d770a  // sdot za.s[x11, 2], { z24.h-z25.h }, z13.h\n"
    ".inst 0xc16c770b  // sdot za.s[x11, 3], { z24.h-z25.h }, z12.h\n"
    ".inst 0xc1697728  // sdot za.s[x11, 0], { z25.h-z26.h }, z9.h\n"
    ".inst 0xc1617729  // sdot za.s[x11, 1], { z25.h-z26.h }, z1.h\n"
    ".inst 0xa0412a80  // ld1h { z0.h-z1.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc161772a  // sdot za.s[x11, 2], { z25.h-z26.h }, z1.h\n"
    ".inst 0xc160772b  // sdot za.s[x11, 3], { z25.h-z26.h }, z0.h\n"
    ".inst 0xc1677748  // sdot za.s[x11, 0], { z26.h-z27.h }, z7.h\n"
    ".inst 0xc1667749  // sdot za.s[x11, 1], { z26.h-z27.h }, z6.h\n"
    ".inst 0xa0422a8c  // ld1h { z12.h-z13.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16d774a  // sdot za.s[x11, 2], { z26.h-z27.h }, z13.h\n"
    ".inst 0xc16c774b  // sdot za.s[x11, 3], { z26.h-z27.h }, z12.h\n"
    "10:"  // Unpadded: 2 priming loads
    "add x23, x16, %x[ld_in_row]\n"
    "ld1sb { z0.s }, p1/Z, [x16]\n"
    "addvl x22, SP, #12\n"
    "ld1sb { z19.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "addvl x21, SP, #18\n"
    "ld1sb { z4.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "addvl x20, SP, #24\n"
    "ld1sb { z3.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "add x16, x16, %x[ld_in_col]\n"
    "ld1sb { z17.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z22.h, z0.h, z19.h\n"
    "ld1sb { z25.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z23.h, z4.h, z3.h\n"
    "ld1sb { z9.s }, p1/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    ".inst 0xa0402ac4  // ld1h { z4.h-z5.h }, pn10.b/Z, [x22]\n"
    "add z22.h, z22.h, z15.h\n"
    "trn1 z24.h, z17.h, z25.h\n"
    "ld1sb { z17.s }, p1/Z, [x23]\n"
    ".inst 0xa0412ac6  // ld1h { z6.h-z7.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    "add z23.h, z23.h, z15.h\n"
    ".inst 0xa0422ac0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    "trn1 z25.h, z9.h, z17.h\n"
    "add z24.h, z24.h, z15.h\n"
    ".inst 0xc16576c8  // sdot za.s[x11, 0], { z22.h-z23.h }, z5.h\n"
    ".inst 0xc16476c9  // sdot za.s[x11, 1], { z22.h-z23.h }, z4.h\n"
    ".inst 0xa0402aa4  // ld1h { z4.h-z5.h }, pn10.b/Z, [x21]\n"
    "add z25.h, z25.h, z15.h\n"
    ".inst 0xc16576ca  // sdot za.s[x11, 2], { z22.h-z23.h }, z5.h\n"
    ".inst 0xc16476cb  // sdot za.s[x11, 3], { z22.h-z23.h }, z4.h\n"
    ".inst 0xa0402a84  // ld1h { z4.h-z5.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc16776e8  // sdot za.s[x11, 0], { z23.h-z24.h }, z7.h\n"
    ".inst 0xc16676e9  // sdot za.s[x11, 1], { z23.h-z24.h }, z6.h\n"
    ".inst 0xa0412aa6  // ld1h { z6.h-z7.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc16576cc  // sdot za.s[x11, 4], { z22.h-z23.h }, z5.h\n"
    ".inst 0xc16476cd  // sdot za.s[x11, 5], { z22.h-z23.h }, z4.h\n"
    ".inst 0xc16776ea  // sdot za.s[x11, 2], { z23.h-z24.h }, z7.h\n"
    ".inst 0xc16676eb  // sdot za.s[x11, 3], { z23.h-z24.h }, z6.h\n"
    ".inst 0xa0412a84  // ld1h { z4.h-z5.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc1617708  // sdot za.s[x11, 0], { z24.h-z25.h }, z1.h\n"
    ".inst 0xc1607709  // sdot za.s[x11, 1], { z24.h-z25.h }, z0.h\n"
    ".inst 0xa0422aa6  // ld1h { z6.h-z7.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc16576ec  // sdot za.s[x11, 4], { z23.h-z24.h }, z5.h\n"
    ".inst 0xc16476ed  // sdot za.s[x11, 5], { z23.h-z24.h }, z4.h\n"
    ".inst 0xc167770a  // sdot za.s[x11, 2], { z24.h-z25.h }, z7.h\n"
    ".inst 0xc166770b  // sdot za.s[x11, 3], { z24.h-z25.h }, z6.h\n"
    ".inst 0xa1422a85  // ld1h { z5.h, z13.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16d770c  // sdot za.s[x11, 4], { z24.h-z25.h }, z13.h\n"
    ".inst 0xc165770d  // sdot za.s[x11, 5], { z24.h-z25.h }, z5.h\n"
    "11:"  // Unpadded: 1 priming loads
    "add x24, x16, %x[ld_in_row]\n"
    "ld1sb { z16.s }, p1/Z, [x16]\n"
    "addvl x23, SP, #6\n"
    "ld1sb { z22.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "addvl x22, SP, #12\n"
    "ld1sb { z19.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "addvl x21, SP, #18\n"
    "ld1sb { z25.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "addvl x20, SP, #24\n"
    "ld1sb { z6.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "trn1 z18.h, z16.h, z22.h\n"
    "add x16, x16, %x[ld_in_col]\n"
    "ld1sb { z4.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "trn1 z19.h, z19.h, z25.h\n"
    "ld1sb { z27.s }, p1/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    ".inst 0xa1402ae1  // ld1h { z1.h, z9.h }, pn10.b/Z, [x23]\n"
    "add z18.h, z18.h, z15.h\n"
    "trn1 z20.h, z6.h, z4.h\n"
    "ld1sb { z22.s }, p1/Z, [x24]\n"
    ".inst 0xa1412ae5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x23, #0x2, MUL VL]\n"
    "add z19.h, z19.h, z15.h\n"
    ".inst 0xa1422ae6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x23, #0x4, MUL VL]\n"
    "trn1 z21.h, z27.h, z22.h\n"
    "add z20.h, z20.h, z15.h\n"
    ".inst 0xc1697648  // sdot za.s[x11, 0], { z18.h-z19.h }, z9.h\n"
    ".inst 0xc1617649  // sdot za.s[x11, 1], { z18.h-z19.h }, z1.h\n"
    ".inst 0xa1402ac4  // ld1h { z4.h, z12.h }, pn10.b/Z, [x22]\n"
    "add z21.h, z21.h, z15.h\n"
    ".inst 0xc16c764a  // sdot za.s[x11, 2], { z18.h-z19.h }, z12.h\n"
    ".inst 0xc164764b  // sdot za.s[x11, 3], { z18.h-z19.h }, z4.h\n"
    ".inst 0xa0402aa2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc16d7668  // sdot za.s[x11, 0], { z19.h-z20.h }, z13.h\n"
    ".inst 0xc1657669  // sdot za.s[x11, 1], { z19.h-z20.h }, z5.h\n"
    ".inst 0xa1412ac4  // ld1h { z4.h, z12.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    ".inst 0xc163764c  // sdot za.s[x11, 4], { z18.h-z19.h }, z3.h\n"
    ".inst 0xc162764d  // sdot za.s[x11, 5], { z18.h-z19.h }, z2.h\n"
    ".inst 0xa1402a81  // ld1h { z1.h, z9.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc16c766a  // sdot za.s[x11, 2], { z19.h-z20.h }, z12.h\n"
    ".inst 0xc164766b  // sdot za.s[x11, 3], { z19.h-z20.h }, z4.h\n"
    ".inst 0xa0412aa2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc16e7688  // sdot za.s[x11, 0], { z20.h-z21.h }, z14.h\n"
    ".inst 0xc1667689  // sdot za.s[x11, 1], { z20.h-z21.h }, z6.h\n"
    ".inst 0xa1422ac4  // ld1h { z4.h, z12.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    ".inst 0xc169764e  // sdot za.s[x11, 6], { z18.h-z19.h }, z9.h\n"
    ".inst 0xc161764f  // sdot za.s[x11, 7], { z18.h-z19.h }, z1.h\n"
    ".inst 0xc163766c  // sdot za.s[x11, 4], { z19.h-z20.h }, z3.h\n"
    ".inst 0xc162766d  // sdot za.s[x11, 5], { z19.h-z20.h }, z2.h\n"
    ".inst 0xa1412a81  // ld1h { z1.h, z9.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc16c768a  // sdot za.s[x11, 2], { z20.h-z21.h }, z12.h\n"
    ".inst 0xc164768b  // sdot za.s[x11, 3], { z20.h-z21.h }, z4.h\n"
    ".inst 0xa1422aa4  // ld1h { z4.h, z12.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc169766e  // sdot za.s[x11, 6], { z19.h-z20.h }, z9.h\n"
    ".inst 0xc161766f  // sdot za.s[x11, 7], { z19.h-z20.h }, z1.h\n"
    ".inst 0xc16c768c  // sdot za.s[x11, 4], { z20.h-z21.h }, z12.h\n"
    ".inst 0xc164768d  // sdot za.s[x11, 5], { z20.h-z21.h }, z4.h\n"
    ".inst 0xa0422a8c  // ld1h { z12.h-z13.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16d768e  // sdot za.s[x11, 6], { z20.h-z21.h }, z13.h\n"
    ".inst 0xc16c768f  // sdot za.s[x11, 7], { z20.h-z21.h }, z12.h\n"
    "12:"  // Unpadded: 0 priming loads
    ".inst 0xa1402be1  // ld1h { z1.h, z9.h }, pn10.b/Z, [SP]\n"
    ".inst 0xa1412be4  // ld1h { z4.h, z12.h }, pn10.b/Z, [SP, #0x2, MUL VL]\n"
    ".inst 0xa0422be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP, #0x4, MUL VL]\n"
    "cbz x25, 22f\n"
    "add x20, x16, %x[ld_in_row]\n"
    "ld1sb { z6.s }, p1/Z, [x16]\n"
    "sub x25, x25, #0x1\n"
    "ld1sb { z13.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "sub x15, x15, #0x1\n"
    "ld1sb { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "cmp x25, x15\n"
    "ld1sb { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "csel x25, x25, x15, LT\n"
    "ld1sb { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z24.h, z6.h, z13.h\n"
    "add x16, x16, %x[ld_in_col]\n"
    "ld1sb { z13.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "sub x15, x15, x25\n"
    "ld1sb { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z25.h, z21.h, z19.h\n"
    "ld1sb { z16.s }, p1/Z, [x20]\n"
    "trn1 z26.h, z20.h, z13.h\n"
    "add z24.h, z24.h, z15.h\n"
    "trn1 z27.h, z22.h, z16.h\n"
    "add z25.h, z25.h, z15.h\n"
    "add z26.h, z26.h, z15.h\n"
    "add z27.h, z27.h, z15.h\n"
    "cbz x25, 21f\n"
    "13:"  // Unpadded: Main loop
    "addvl x24, SP, #6\n"
    ".inst 0xc1697708  // sdot za.s[x11, 0], { z24.h-z25.h }, z9.h\n"
    "addvl x23, SP, #12\n"
    "ld1sb { z23.s }, p1/Z, [x16]\n"
    ".inst 0xc1617709  // sdot za.s[x11, 1], { z24.h-z25.h }, z1.h\n"
    ".inst 0xa1402b01  // ld1h { z1.h, z9.h }, pn10.b/Z, [x24]\n"
    "addvl x22, SP, #18\n"
    "addvl x21, SP, #24\n"
    "add x20, x16, %x[ld_in_row]\n"
    "subs x25, x25, #0x1\n"
    "ld1sb { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xc169770a  // sdot za.s[x11, 2], { z24.h-z25.h }, z9.h\n"
    "ld1sb { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc161770b  // sdot za.s[x11, 3], { z24.h-z25.h }, z1.h\n"
    ".inst 0xa1402ae6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x23]\n"
    ".inst 0xc16c7728  // sdot za.s[x11, 0], { z25.h-z26.h }, z12.h\n"
    "ld1sb { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc1647729  // sdot za.s[x11, 1], { z25.h-z26.h }, z4.h\n"
    ".inst 0xa1412b04  // ld1h { z4.h, z12.h }, pn10.b/Z, [x24, #0x2, MUL VL]\n"
    ".inst 0xc16e770c  // sdot za.s[x11, 4], { z24.h-z25.h }, z14.h\n"
    "ld1sb { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc166770d  // sdot za.s[x11, 5], { z24.h-z25.h }, z6.h\n"
    ".inst 0xa1402ac5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x22]\n"
    ".inst 0xc16c772a  // sdot za.s[x11, 2], { z25.h-z26.h }, z12.h\n"
    "ld1sb { z18.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc164772b  // sdot za.s[x11, 3], { z25.h-z26.h }, z4.h\n"
    ".inst 0xa1412ae6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x23, #0x2, MUL VL]\n"
    ".inst 0xc1637748  // sdot za.s[x11, 0], { z26.h-z27.h }, z3.h\n"
    "ld1sb { z17.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    ".inst 0xc1627749  // sdot za.s[x11, 1], { z26.h-z27.h }, z2.h\n"
    ".inst 0xa0422b02  // ld1h { z2.h-z3.h }, pn10.b/Z, [x24, #0x4, MUL VL]\n"
    ".inst 0xc16d770e  // sdot za.s[x11, 6], { z24.h-z25.h }, z13.h\n"
    "ld1sb { z16.s }, p1/Z, [x20]\n"
    ".inst 0xc165770f  // sdot za.s[x11, 7], { z24.h-z25.h }, z5.h\n"
    ".inst 0xa1402aa4  // ld1h { z4.h, z12.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc16e772c  // sdot za.s[x11, 4], { z25.h-z26.h }, z14.h\n"
    ".inst 0xc166772d  // sdot za.s[x11, 5], { z25.h-z26.h }, z6.h\n"
    ".inst 0xa1412ac6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    ".inst 0xc163774a  // sdot za.s[x11, 2], { z26.h-z27.h }, z3.h\n"
    ".inst 0xc162774b  // sdot za.s[x11, 3], { z26.h-z27.h }, z2.h\n"
    ".inst 0xa0422ae0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x23, #0x4, MUL VL]\n"
    ".inst 0xc16e772e  // sdot za.s[x11, 6], { z25.h-z26.h }, z14.h\n"
    ".inst 0xc166772f  // sdot za.s[x11, 7], { z25.h-z26.h }, z6.h\n"
    ".inst 0xa1412aa5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc161774c  // sdot za.s[x11, 4], { z26.h-z27.h }, z1.h\n"
    ".inst 0xc160774d  // sdot za.s[x11, 5], { z26.h-z27.h }, z0.h\n"
    ".inst 0xa0422ac6  // ld1h { z6.h-z7.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    ".inst 0xc167774e  // sdot za.s[x11, 6], { z26.h-z27.h }, z7.h\n"
    ".inst 0xc166774f  // sdot za.s[x11, 7], { z26.h-z27.h }, z6.h\n"
    ".inst 0xa1422aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc16c1708  // sdot za.s[x8, 0], { z24.h-z25.h }, z12.h\n"
    ".inst 0xc1641709  // sdot za.s[x8, 1], { z24.h-z25.h }, z4.h\n"
    "trn1 z24.h, z23.h, z19.h\n"
    ".inst 0xa1402be1  // ld1h { z1.h, z9.h }, pn10.b/Z, [SP]\n"
    ".inst 0xc16d1728  // sdot za.s[x8, 0], { z25.h-z26.h }, z13.h\n"
    ".inst 0xc1651729  // sdot za.s[x8, 1], { z25.h-z26.h }, z5.h\n"
    "trn1 z25.h, z21.h, z20.h\n"
    ".inst 0xa1412be4  // ld1h { z4.h, z12.h }, pn10.b/Z, [SP, #0x2, MUL VL]\n"
    "add z24.h, z24.h, z15.h\n"
    ".inst 0xc16e1748  // sdot za.s[x8, 0], { z26.h-z27.h }, z14.h\n"
    ".inst 0xc1661749  // sdot za.s[x8, 1], { z26.h-z27.h }, z6.h\n"
    "trn1 z26.h, z22.h, z18.h\n"
    "trn1 z27.h, z17.h, z16.h\n"
    "add x8, x8, #0x2\n"
    ".inst 0xa0422be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP, #0x4, MUL VL]\n"
    "add z25.h, z25.h, z15.h\n"
    ".inst 0xc0066814  // mova { z20.d-z21.d }, za.d[x11, #0]\n"
    "add z26.h, z26.h, z15.h\n"
    "add z27.h, z27.h, z15.h\n"
    ".inst 0xc0066836  // mova { z22.d-z23.d }, za.d[x11, #1]\n"
    "add x11, x11, #0x2\n"
    ".inst 0xc0040b80  // mova za.d[x8, #0], { z28.d-z29.d }\n"
    ".inst 0xc0040b81  // mova za.d[x8, #1], { z28.d-z29.d }\n"
    ".inst 0xc1a8ac14  // sqdmulh { z20.s-z23.s }, { z20.s-z23.s }, z8.s\n"
    ".inst 0xc1abaa34  // srshl { z20.s-z23.s }, { z20.s-z23.s }, z11.s\n"
    ".inst 0xc1aaab14  // add { z20.s-z23.s }, { z20.s-z23.s }, z10.s\n"
    ".inst 0xc1becff4  // sclamp { z20.s-z23.s }, z31.s, z30.s\n"
    "st1b { z20.s }, p1, [x14]\n"
    "add x14, x14, x4\n"
    "st1b { z22.s }, p1, [x13]\n"
    "add x13, x13, x10\n"
    "st1b { z21.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "st1b { z23.s }, p1, [x28]\n"
    "add x28, x28, x26\n"
    "bgt 13b\n"
    "b 21f\n"
    "14:"  // Padded
    "cbz x22, 19f\n"
    "cmp x22, #0x1\n"
    "sub x25, x25, x22\n"
    "beq 18f\n"
    "cmp x22, #0x2\n"
    "beq 17f\n"
    "cmp x22, #0x3\n"
    "beq 16f\n"
    "15:"  // Padded: 4 priming loads
    "mov x12, #0x0\n"
    "add x21, x16, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "addvl x20, SP, #24\n"
    ".inst 0xa1402a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20]\n"
    ".inst 0xa0412a8c  // ld1h { z12.h-z13.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "ld1sb { z18.s }, p0/Z, [x16]\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xa0422a82  // ld1h { z2.h-z3.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z16.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z17.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z22.h, z18.h, z16.h\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1sb { z26.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "add z26.h, p0/M, z26.h, z15.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z18.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z23.h, z17.h, z26.h\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc16e76c8  // sdot za.s[x11, 0], { z22.h-z23.h }, z14.h\n"
    "ld1sb { z25.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    ".inst 0xc16676c9  // sdot za.s[x11, 1], { z22.h-z23.h }, z6.h\n"
    "add z25.h, p0/M, z25.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z1.s }, p0/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row]\n"
    "trn1 z24.h, z18.h, z25.h\n"
    "add z1.h, p0/M, z1.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc16d76e8  // sdot za.s[x11, 0], { z23.h-z24.h }, z13.h\n"
    "ld1sb { z16.s }, p0/Z, [x21]\n"
    ".inst 0xc16c76e9  // sdot za.s[x11, 1], { z23.h-z24.h }, z12.h\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    "trn1 z25.h, z1.h, z16.h\n"
    ".inst 0xc1637708  // sdot za.s[x11, 0], { z24.h-z25.h }, z3.h\n"
    ".inst 0xc1627709  // sdot za.s[x11, 1], { z24.h-z25.h }, z2.h\n"
    "16:"  // Padded: 3 priming loads
    "mov x12, #0x0\n"
    "add x22, x16, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "addvl x21, SP, #18\n"
    ".inst 0xa1402aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21]\n"
    "addvl x20, SP, #24\n"
    ".inst 0xa1412aa4  // ld1h { z4.h, z12.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    "ld1sb { z18.s }, p0/Z, [x16]\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xa1422aa5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z16.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z17.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z0.h, z18.h, z16.h\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1sb { z16.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z18.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z1.h, z17.h, z16.h\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc16e7408  // sdot za.s[x11, 0], { z0.h-z1.h }, z14.h\n"
    "ld1sb { z17.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc1667409  // sdot za.s[x11, 1], { z0.h-z1.h }, z6.h\n"
    ".inst 0xa1402a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20]\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc16e740a  // sdot za.s[x11, 2], { z0.h-z1.h }, z14.h\n"
    ".inst 0xc166740b  // sdot za.s[x11, 3], { z0.h-z1.h }, z6.h\n"
    "ld1sb { z0.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "trn1 z2.h, z18.h, z17.h\n"
    "add z0.h, p0/M, z0.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc16c7428  // sdot za.s[x11, 0], { z1.h-z2.h }, z12.h\n"
    "ld1sb { z17.s }, p0/Z, [x22]\n"
    ".inst 0xc1647429  // sdot za.s[x11, 1], { z1.h-z2.h }, z4.h\n"
    ".inst 0xa1412a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0xc16e742a  // sdot za.s[x11, 2], { z1.h-z2.h }, z14.h\n"
    ".inst 0xc166742b  // sdot za.s[x11, 3], { z1.h-z2.h }, z6.h\n"
    "trn1 z3.h, z0.h, z17.h\n"
    ".inst 0xc16d7448  // sdot za.s[x11, 0], { z2.h-z3.h }, z13.h\n"
    ".inst 0xc1657449  // sdot za.s[x11, 1], { z2.h-z3.h }, z5.h\n"
    ".inst 0xa1422a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16e744a  // sdot za.s[x11, 2], { z2.h-z3.h }, z14.h\n"
    ".inst 0xc166744b  // sdot za.s[x11, 3], { z2.h-z3.h }, z6.h\n"
    "17:"  // Padded: 2 priming loads
    "mov x12, #0x0\n"
    "add x23, x16, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "addvl x22, SP, #12\n"
    ".inst 0xa0402ac2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x22]\n"
    "addvl x21, SP, #18\n"
    "addvl x20, SP, #24\n"
    ".inst 0xa1412ac4  // ld1h { z4.h, z12.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    "ld1sb { z18.s }, p0/Z, [x16]\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xa1422ac5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z16.s }, p0/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z17.s }, p0/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z22.h, z18.h, z16.h\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1sb { z16.s }, p0/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z18.s }, p0/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z23.h, z17.h, z16.h\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc16376c8  // sdot za.s[x11, 0], { z22.h-z23.h }, z3.h\n"
    "ld1sb { z16.s }, p0/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    ".inst 0xc16276c9  // sdot za.s[x11, 1], { z22.h-z23.h }, z2.h\n"
    ".inst 0xa1402aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc16e76ca  // sdot za.s[x11, 2], { z22.h-z23.h }, z14.h\n"
    ".inst 0xc16676cb  // sdot za.s[x11, 3], { z22.h-z23.h }, z6.h\n"
    ".inst 0xa1402a81  // ld1h { z1.h, z9.h }, pn10.b/Z, [x20]\n"
    "ld1sb { z17.s }, p0/Z, [x23]\n"
    "add x23, x23, %x[ld_in_row]\n"
    "trn1 z24.h, z18.h, z16.h\n"
    ".inst 0xc16976cc  // sdot za.s[x11, 4], { z22.h-z23.h }, z9.h\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc16176cd  // sdot za.s[x11, 5], { z22.h-z23.h }, z1.h\n"
    ".inst 0xc16c76e8  // sdot za.s[x11, 0], { z23.h-z24.h }, z12.h\n"
    "ld1sb { z16.s }, p0/Z, [x23]\n"
    ".inst 0xc16476e9  // sdot za.s[x11, 1], { z23.h-z24.h }, z4.h\n"
    ".inst 0xa1412aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0xc16e76ea  // sdot za.s[x11, 2], { z23.h-z24.h }, z14.h\n"
    ".inst 0xc16676eb  // sdot za.s[x11, 3], { z23.h-z24.h }, z6.h\n"
    ".inst 0xa1412a81  // ld1h { z1.h, z9.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "trn1 z25.h, z17.h, z16.h\n"
    ".inst 0xc16976ec  // sdot za.s[x11, 4], { z23.h-z24.h }, z9.h\n"
    ".inst 0xc16176ed  // sdot za.s[x11, 5], { z23.h-z24.h }, z1.h\n"
    ".inst 0xc16d7708  // sdot za.s[x11, 0], { z24.h-z25.h }, z13.h\n"
    ".inst 0xc1657709  // sdot za.s[x11, 1], { z24.h-z25.h }, z5.h\n"
    ".inst 0xa0422aa4  // ld1h { z4.h-z5.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc165770a  // sdot za.s[x11, 2], { z24.h-z25.h }, z5.h\n"
    ".inst 0xc164770b  // sdot za.s[x11, 3], { z24.h-z25.h }, z4.h\n"
    ".inst 0xa1422a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16e770c  // sdot za.s[x11, 4], { z24.h-z25.h }, z14.h\n"
    ".inst 0xc166770d  // sdot za.s[x11, 5], { z24.h-z25.h }, z6.h\n"
    "18:"  // Padded: 1 priming loads
    "mov x12, #0x0\n"
    "add x24, x16, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "addvl x23, SP, #6\n"
    ".inst 0xa1402ae6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x23]\n"
    "addvl x22, SP, #12\n"
    "addvl x21, SP, #18\n"
    ".inst 0xa0412ae4  // ld1h { z4.h-z5.h }, pn10.b/Z, [x23, #0x2, MUL VL]\n"
    "addvl x20, SP, #24\n"
    "ld1sb { z18.s }, p0/Z, [x16]\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xa1422ae1  // ld1h { z1.h, z9.h }, pn10.b/Z, [x23, #0x4, MUL VL]\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z16.s }, p0/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z17.s }, p0/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "trn1 z21.h, z18.h, z16.h\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1sb { z16.s }, p0/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z18.s }, p0/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "trn1 z22.h, z17.h, z16.h\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc16e76a8  // sdot za.s[x11, 0], { z21.h-z22.h }, z14.h\n"
    "ld1sb { z16.s }, p0/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    ".inst 0xc16676a9  // sdot za.s[x11, 1], { z21.h-z22.h }, z6.h\n"
    ".inst 0xa0402ac6  // ld1h { z6.h-z7.h }, pn10.b/Z, [x22]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc16776aa  // sdot za.s[x11, 2], { z21.h-z22.h }, z7.h\n"
    ".inst 0xc16676ab  // sdot za.s[x11, 3], { z21.h-z22.h }, z6.h\n"
    ".inst 0xa0402aa6  // ld1h { z6.h-z7.h }, pn10.b/Z, [x21]\n"
    "ld1sb { z17.s }, p0/Z, [x24]\n"
    "add x24, x24, %x[ld_in_row]\n"
    "trn1 z23.h, z18.h, z16.h\n"
    ".inst 0xc16776ac  // sdot za.s[x11, 4], { z21.h-z22.h }, z7.h\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc16676ad  // sdot za.s[x11, 5], { z21.h-z22.h }, z6.h\n"
    ".inst 0xa0402a8c  // ld1h { z12.h-z13.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc16576c8  // sdot za.s[x11, 0], { z22.h-z23.h }, z5.h\n"
    "ld1sb { z16.s }, p0/Z, [x24]\n"
    ".inst 0xc16476c9  // sdot za.s[x11, 1], { z22.h-z23.h }, z4.h\n"
    ".inst 0xa1412ac6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    ".inst 0xc16d76ae  // sdot za.s[x11, 6], { z21.h-z22.h }, z13.h\n"
    ".inst 0xc16c76af  // sdot za.s[x11, 7], { z21.h-z22.h }, z12.h\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0xc16e76ca  // sdot za.s[x11, 2], { z22.h-z23.h }, z14.h\n"
    ".inst 0xc16676cb  // sdot za.s[x11, 3], { z22.h-z23.h }, z6.h\n"
    ".inst 0xa1412aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    "trn1 z24.h, z17.h, z16.h\n"
    ".inst 0xc16e76cc  // sdot za.s[x11, 4], { z22.h-z23.h }, z14.h\n"
    ".inst 0xc16676cd  // sdot za.s[x11, 5], { z22.h-z23.h }, z6.h\n"
    ".inst 0xa0412a86  // ld1h { z6.h-z7.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc16976e8  // sdot za.s[x11, 0], { z23.h-z24.h }, z9.h\n"
    ".inst 0xc16176e9  // sdot za.s[x11, 1], { z23.h-z24.h }, z1.h\n"
    ".inst 0xa0422ac0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    ".inst 0xc16776ce  // sdot za.s[x11, 6], { z22.h-z23.h }, z7.h\n"
    ".inst 0xc16676cf  // sdot za.s[x11, 7], { z22.h-z23.h }, z6.h\n"
    ".inst 0xc16176ea  // sdot za.s[x11, 2], { z23.h-z24.h }, z1.h\n"
    ".inst 0xc16076eb  // sdot za.s[x11, 3], { z23.h-z24.h }, z0.h\n"
    ".inst 0xa1422aa1  // ld1h { z1.h, z9.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc16976ec  // sdot za.s[x11, 4], { z23.h-z24.h }, z9.h\n"
    ".inst 0xc16176ed  // sdot za.s[x11, 5], { z23.h-z24.h }, z1.h\n"
    ".inst 0xa1422a84  // ld1h { z4.h, z12.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16c76ee  // sdot za.s[x11, 6], { z23.h-z24.h }, z12.h\n"
    ".inst 0xc16476ef  // sdot za.s[x11, 7], { z23.h-z24.h }, z4.h\n"
    "19:"  // Padded: 0 priming loads
    ".inst 0xa1402be1  // ld1h { z1.h, z9.h }, pn10.b/Z, [SP]\n"
    ".inst 0xa1412be4  // ld1h { z4.h, z12.h }, pn10.b/Z, [SP, #0x2, MUL VL]\n"
    ".inst 0xa0422be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP, #0x4, MUL VL]\n"
    "cbz x25, 22f\n"
    "mov x12, #0x0\n"
    "add x20, x16, %x[ld_in_row]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "sub x25, x25, #0x1\n"
    "sub x15, x15, #0x1\n"
    "cmp x25, x15\n"
    "ld1sb { z18.s }, p0/Z, [x16]\n"
    "csel x25, x25, x15, LT\n"
    "add x16, x16, %x[ld_in_col]\n"
    "sub x15, x15, x25\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z17.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z24.h, z18.h, z16.h\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1sb { z18.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z25.h, z17.h, z16.h\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1sb { z17.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row]\n"
    "trn1 z26.h, z18.h, z16.h\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1sb { z16.s }, p0/Z, [x20]\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    "trn1 z27.h, z17.h, z16.h\n"
    "cbz x25, 21f\n"
    "20:"  // Padded: Main loop
    "mov x12, #0x0\n"
    "addvl x24, SP, #6\n"
    ".inst 0xc1697708  // sdot za.s[x11, 0], { z24.h-z25.h }, z9.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc1617709  // sdot za.s[x11, 1], { z24.h-z25.h }, z1.h\n"
    ".inst 0xa1402b05  // ld1h { z5.h, z13.h }, pn10.b/Z, [x24]\n"
    "addvl x23, SP, #12\n"
    "add x22, x16, %x[ld_in_row]\n"
    "addvl x21, SP, #18\n"
    "addvl x20, SP, #24\n"
    "subs x25, x25, #0x1\n"
    "ld1sb { z16.s }, p0/Z, [x16]\n"
    ".inst 0xc16d770a  // sdot za.s[x11, 2], { z24.h-z25.h }, z13.h\n"
    "add x16, x16, %x[ld_in_col]\n"
    ".inst 0xc165770b  // sdot za.s[x11, 3], { z24.h-z25.h }, z5.h\n"
    ".inst 0xa1402ae6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x23]\n"
    ".inst 0xc16c7728  // sdot za.s[x11, 0], { z25.h-z26.h }, z12.h\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc1647729  // sdot za.s[x11, 1], { z25.h-z26.h }, z4.h\n"
    ".inst 0xa1412b01  // ld1h { z1.h, z9.h }, pn10.b/Z, [x24, #0x2, MUL VL]\n"
    ".inst 0xc16e770c  // sdot za.s[x11, 4], { z24.h-z25.h }, z14.h\n"
    ".inst 0xc166770d  // sdot za.s[x11, 5], { z24.h-z25.h }, z6.h\n"
    ".inst 0xa0402aa4  // ld1h { z4.h-z5.h }, pn10.b/Z, [x21]\n"
    "ld1sb { z19.s }, p0/Z, [x22]\n"
    ".inst 0xc169772a  // sdot za.s[x11, 2], { z25.h-z26.h }, z9.h\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc161772b  // sdot za.s[x11, 3], { z25.h-z26.h }, z1.h\n"
    ".inst 0xa1412ae6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x23, #0x2, MUL VL]\n"
    ".inst 0xc1637748  // sdot za.s[x11, 0], { z26.h-z27.h }, z3.h\n"
    "add z19.h, p0/M, z19.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc1627749  // sdot za.s[x11, 1], { z26.h-z27.h }, z2.h\n"
    ".inst 0xa0422b02  // ld1h { z2.h-z3.h }, pn10.b/Z, [x24, #0x4, MUL VL]\n"
    ".inst 0xc165770e  // sdot za.s[x11, 6], { z24.h-z25.h }, z5.h\n"
    ".inst 0xc164770f  // sdot za.s[x11, 7], { z24.h-z25.h }, z4.h\n"
    ".inst 0xa1402a84  // ld1h { z4.h, z12.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc16e772c  // sdot za.s[x11, 4], { z25.h-z26.h }, z14.h\n"
    "ld1sb { z23.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc166772d  // sdot za.s[x11, 5], { z25.h-z26.h }, z6.h\n"
    ".inst 0xa1412aa5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc163774a  // sdot za.s[x11, 2], { z26.h-z27.h }, z3.h\n"
    ".inst 0xc162774b  // sdot za.s[x11, 3], { z26.h-z27.h }, z2.h\n"
    ".inst 0xa0422ae6  // ld1h { z6.h-z7.h }, pn10.b/Z, [x23, #0x4, MUL VL]\n"
    "add z23.h, p0/M, z23.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc16d772e  // sdot za.s[x11, 6], { z25.h-z26.h }, z13.h\n"
    "mov x12, #0x4\n"
    ".inst 0xc165772f  // sdot za.s[x11, 7], { z25.h-z26.h }, z5.h\n"
    ".inst 0xa1412a85  // ld1h { z5.h, z13.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc167774c  // sdot za.s[x11, 4], { z26.h-z27.h }, z7.h\n"
    "ld1sb { z17.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc166774d  // sdot za.s[x11, 5], { z26.h-z27.h }, z6.h\n"
    ".inst 0xa0422aa2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc163774e  // sdot za.s[x11, 6], { z26.h-z27.h }, z3.h\n"
    ".inst 0xc162774f  // sdot za.s[x11, 7], { z26.h-z27.h }, z2.h\n"
    ".inst 0xa0422a82  // ld1h { z2.h-z3.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    "ld1sb { z18.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc16c1708  // sdot za.s[x8, 0], { z24.h-z25.h }, z12.h\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc1641709  // sdot za.s[x8, 1], { z24.h-z25.h }, z4.h\n"
    ".inst 0xa1402be1  // ld1h { z1.h, z9.h }, pn10.b/Z, [SP]\n"
    "trn1 z24.h, z16.h, z19.h\n"
    "ld1sb { z16.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    ".inst 0xc16d1728  // sdot za.s[x8, 0], { z25.h-z26.h }, z13.h\n"
    ".inst 0xc1651729  // sdot za.s[x8, 1], { z25.h-z26.h }, z5.h\n"
    ".inst 0xa1412be4  // ld1h { z4.h, z12.h }, pn10.b/Z, [SP, #0x2, MUL VL]\n"
    "trn1 z25.h, z23.h, z17.h\n"
    "add z16.h, p0/M, z16.h, z15.h\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc1631748  // sdot za.s[x8, 0], { z26.h-z27.h }, z3.h\n"
    ".inst 0xc1621749  // sdot za.s[x8, 1], { z26.h-z27.h }, z2.h\n"
    "ld1sb { z17.s }, p0/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row]\n"
    "add x8, x8, #0x2\n"
    ".inst 0xa0422be2  // ld1h { z2.h-z3.h }, pn10.b/Z, [SP, #0x4, MUL VL]\n"
    "trn1 z26.h, z18.h, z16.h\n"
    "add z17.h, p0/M, z17.h, z15.h\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc0066814  // mova { z20.d-z21.d }, za.d[x11, #0]\n"
    ".inst 0xc0066836  // mova { z22.d-z23.d }, za.d[x11, #1]\n"
    "add x11, x11, #0x2\n"
    "ld1sb { z18.s }, p0/Z, [x22]\n"
    ".inst 0xc0040b80  // mova za.d[x8, #0], { z28.d-z29.d }\n"
    ".inst 0xc0040b81  // mova za.d[x8, #1], { z28.d-z29.d }\n"
    ".inst 0xc1a8ac14  // sqdmulh { z20.s-z23.s }, { z20.s-z23.s }, z8.s\n"
    "add z18.h, p0/M, z18.h, z15.h\n"
    ".inst 0xc1abaa34  // srshl { z20.s-z23.s }, { z20.s-z23.s }, z11.s\n"
    "trn1 z27.h, z17.h, z18.h\n"
    ".inst 0xc1aaab14  // add { z20.s-z23.s }, { z20.s-z23.s }, z10.s\n"
    ".inst 0xc1becff4  // sclamp { z20.s-z23.s }, z31.s, z30.s\n"
    "st1b { z20.s }, p1, [x14]\n"
    "add x14, x14, x4\n"
    "st1b { z22.s }, p1, [x13]\n"
    "add x13, x13, x10\n"
    "st1b { z21.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "st1b { z23.s }, p1, [x28]\n"
    "add x28, x28, x26\n"
    "bgt 20b\n"
    "21:"  // Main loop tail
    "addvl x23, SP, #6\n"
    ".inst 0xc1697708  // sdot za.s[x11, 0], { z24.h-z25.h }, z9.h\n"
    "addvl x22, SP, #12\n"
    ".inst 0xc1617709  // sdot za.s[x11, 1], { z24.h-z25.h }, z1.h\n"
    ".inst 0xa1402ae6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x23]\n"
    "addvl x21, SP, #18\n"
    "addvl x20, SP, #24\n"
    ".inst 0xc16e770a  // sdot za.s[x11, 2], { z24.h-z25.h }, z14.h\n"
    ".inst 0xc166770b  // sdot za.s[x11, 3], { z24.h-z25.h }, z6.h\n"
    ".inst 0xa1402ac6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x22]\n"
    ".inst 0xc16c7728  // sdot za.s[x11, 0], { z25.h-z26.h }, z12.h\n"
    ".inst 0xc1647729  // sdot za.s[x11, 1], { z25.h-z26.h }, z4.h\n"
    ".inst 0xa1412ae5  // ld1h { z5.h, z13.h }, pn10.b/Z, [x23, #0x2, MUL VL]\n"
    ".inst 0xc16e770c  // sdot za.s[x11, 4], { z24.h-z25.h }, z14.h\n"
    ".inst 0xc166770d  // sdot za.s[x11, 5], { z24.h-z25.h }, z6.h\n"
    ".inst 0xa1402aa4  // ld1h { z4.h, z12.h }, pn10.b/Z, [x21]\n"
    ".inst 0xc16d772a  // sdot za.s[x11, 2], { z25.h-z26.h }, z13.h\n"
    ".inst 0xc165772b  // sdot za.s[x11, 3], { z25.h-z26.h }, z5.h\n"
    ".inst 0xa1412ac6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x22, #0x2, MUL VL]\n"
    ".inst 0xc1637748  // sdot za.s[x11, 0], { z26.h-z27.h }, z3.h\n"
    ".inst 0xc1627749  // sdot za.s[x11, 1], { z26.h-z27.h }, z2.h\n"
    ".inst 0xa0422ae2  // ld1h { z2.h-z3.h }, pn10.b/Z, [x23, #0x4, MUL VL]\n"
    ".inst 0xc16c770e  // sdot za.s[x11, 6], { z24.h-z25.h }, z12.h\n"
    ".inst 0xc164770f  // sdot za.s[x11, 7], { z24.h-z25.h }, z4.h\n"
    ".inst 0xa1402a85  // ld1h { z5.h, z13.h }, pn10.b/Z, [x20]\n"
    ".inst 0xc16e772c  // sdot za.s[x11, 4], { z25.h-z26.h }, z14.h\n"
    ".inst 0xc166772d  // sdot za.s[x11, 5], { z25.h-z26.h }, z6.h\n"
    ".inst 0xa1412aa6  // ld1h { z6.h, z14.h }, pn10.b/Z, [x21, #0x2, MUL VL]\n"
    ".inst 0xc163774a  // sdot za.s[x11, 2], { z26.h-z27.h }, z3.h\n"
    ".inst 0xc162774b  // sdot za.s[x11, 3], { z26.h-z27.h }, z2.h\n"
    ".inst 0xa1422ac4  // ld1h { z4.h, z12.h }, pn10.b/Z, [x22, #0x4, MUL VL]\n"
    ".inst 0xc16e772e  // sdot za.s[x11, 6], { z25.h-z26.h }, z14.h\n"
    ".inst 0xc166772f  // sdot za.s[x11, 7], { z25.h-z26.h }, z6.h\n"
    ".inst 0xa1412a86  // ld1h { z6.h, z14.h }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc16c774c  // sdot za.s[x11, 4], { z26.h-z27.h }, z12.h\n"
    ".inst 0xc164774d  // sdot za.s[x11, 5], { z26.h-z27.h }, z4.h\n"
    ".inst 0xa0422aa0  // ld1h { z0.h-z1.h }, pn10.b/Z, [x21, #0x4, MUL VL]\n"
    ".inst 0xc161774e  // sdot za.s[x11, 6], { z26.h-z27.h }, z1.h\n"
    ".inst 0xc160774f  // sdot za.s[x11, 7], { z26.h-z27.h }, z0.h\n"
    ".inst 0xa0422a80  // ld1h { z0.h-z1.h }, pn10.b/Z, [x20, #0x4, MUL VL]\n"
    ".inst 0xc16d1708  // sdot za.s[x8, 0], { z24.h-z25.h }, z13.h\n"
    ".inst 0xc1651709  // sdot za.s[x8, 1], { z24.h-z25.h }, z5.h\n"
    ".inst 0xc16e1728  // sdot za.s[x8, 0], { z25.h-z26.h }, z14.h\n"
    ".inst 0xc1661729  // sdot za.s[x8, 1], { z25.h-z26.h }, z6.h\n"
    ".inst 0xc1611748  // sdot za.s[x8, 0], { z26.h-z27.h }, z1.h\n"
    ".inst 0xc1601749  // sdot za.s[x8, 1], { z26.h-z27.h }, z0.h\n"
    "add x8, x8, #0x2\n"
    ".inst 0xc0066814  // mova { z20.d-z21.d }, za.d[x11, #0]\n"
    ".inst 0xc0066836  // mova { z22.d-z23.d }, za.d[x11, #1]\n"
    "add x11, x11, #0x2\n"
    ".inst 0xc0040b80  // mova za.d[x8, #0], { z28.d-z29.d }\n"
    ".inst 0xc0040b81  // mova za.d[x8, #1], { z28.d-z29.d }\n"
    ".inst 0xc1a8ac14  // sqdmulh { z20.s-z23.s }, { z20.s-z23.s }, z8.s\n"
    ".inst 0xc1abaa34  // srshl { z20.s-z23.s }, { z20.s-z23.s }, z11.s\n"
    ".inst 0xc1aaab14  // add { z20.s-z23.s }, { z20.s-z23.s }, z10.s\n"
    ".inst 0xc1becff4  // sclamp { z20.s-z23.s }, z31.s, z30.s\n"
    "st1b { z20.s }, p1, [x14]\n"
    "add x14, x14, x4\n"
    "st1b { z22.s }, p1, [x13]\n"
    "add x13, x13, x10\n"
    "st1b { z21.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "st1b { z23.s }, p1, [x28]\n"
    "add x28, x28, x26\n"
    "22:"  // Main loop skip tail
    "cbz x15, 24f\n"
    "23:"  // Right padding loop
    ".inst 0xc0066818  // mova { z24.d-z25.d }, za.d[x11, #0]\n"
    "add x8, x8, #0x2\n"
    "subs x15, x15, #0x1\n"
    ".inst 0xc006683a  // mova { z26.d-z27.d }, za.d[x11, #1]\n"
    "add x11, x11, #0x2\n"
    ".inst 0xc0040b80  // mova za.d[x8, #0], { z28.d-z29.d }\n"
    ".inst 0xc0040b81  // mova za.d[x8, #1], { z28.d-z29.d }\n"
    ".inst 0xc1a8ac18  // sqdmulh { z24.s-z27.s }, { z24.s-z27.s }, z8.s\n"
    ".inst 0xc1abaa38  // srshl { z24.s-z27.s }, { z24.s-z27.s }, z11.s\n"
    ".inst 0xc1aaab18  // add { z24.s-z27.s }, { z24.s-z27.s }, z10.s\n"
    ".inst 0xc1becff8  // sclamp { z24.s-z27.s }, z31.s, z30.s\n"
    "st1b { z24.s }, p1, [x14]\n"
    "add x14, x14, x4\n"
    "st1b { z26.s }, p1, [x13]\n"
    "add x13, x13, x10\n"
    "st1b { z25.s }, p1, [x9]\n"
    "add x9, x9, x27\n"
    "st1b { z27.s }, p1, [x28]\n"
    "add x28, x28, x26\n"
    "bgt 23b\n"
    "24:"  // End
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    "incw x17\n"
    "whilelt p1.s, x17, x7\n"
    "incw x20, ALL, MUL #16\n"
    "incw x20, ALL, MUL #9\n"
    "str x20, [%x[args], %[offsetof_Args_weights]]\n"
    "ldr x21, [%x[args], %[offsetof_Args_ld_in_vl]]\n"
    "ldr x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "add x20, x20, x21\n"
    "str x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "ldr x25, [%x[args], %[offsetof_Args_outptrs]]\n"
    "ldr x24, [%x[args], %[offsetof_Args_ld_out_vls]]\n"
    "ldp x23, x22, [x25, #0x0]\n"
    "ldp x21, x20, [x24, #0x0]\n"
    "add x23, x23, x21\n"
    "add x22, x22, x20\n"
    "stp x23, x22, [x25, #0x0]\n"
    "ldp x23, x22, [x25, #0x10]\n"
    "ldp x21, x20, [x24, #0x10]\n"
    "add x23, x23, x21\n"
    "add x22, x22, x20\n"
    "stp x23, x22, [x25, #0x10]\n"
    "b.any 1b\n"
    "addvl SP, SP, #30\n"
    ".inst 0xd503467f  // SMSTOP\n"
    "ldr x20, [SP, #0x0]\n"
    "mov SP, x20\n"
    :
    : [args] "r" (&args), [ld_in_col] "r" (ld_in_col), [ld_in_row] "r" (ld_in_row), [offsetof_Args_current_channel] "I" (offsetof(Args, current_channel)), [offsetof_Args_inptr] "I" (offsetof(Args, inptr)), [offsetof_Args_input_cols] "I" (offsetof(Args, input_cols)), [offsetof_Args_ld_in_vl] "I" (offsetof(Args, ld_in_vl)), [offsetof_Args_ld_out_cols] "I" (offsetof(Args, ld_out_cols)), [offsetof_Args_ld_out_vls] "I" (offsetof(Args, ld_out_vls)), [offsetof_Args_n_channels] "I" (offsetof(Args, n_channels)), [offsetof_Args_outptrs] "I" (offsetof(Args, outptrs)), [offsetof_Args_output_cols] "I" (offsetof(Args, output_cols)), [offsetof_Args_pad_bottom] "I" (offsetof(Args, pad_bottom)), [offsetof_Args_pad_left] "I" (offsetof(Args, pad_left)), [offsetof_Args_pad_top] "I" (offsetof(Args, pad_top)), [offsetof_Args_weights] "I" (offsetof(Args, weights)), [offsetof_Requantize32_a_offset] "I" (offsetof(arm_gemm::Requantize32, a_offset)), [offsetof_Requantize32_b_offset] "I" (offsetof(arm_gemm::Requantize32, b_offset)), [offsetof_Requantize32_bias] "I" (offsetof(arm_gemm::Requantize32, bias)), [offsetof_Requantize32_c_offset] "I" (offsetof(arm_gemm::Requantize32, c_offset)), [offsetof_Requantize32_maxval] "I" (offsetof(arm_gemm::Requantize32, maxval)), [offsetof_Requantize32_minval] "I" (offsetof(arm_gemm::Requantize32, minval)), [offsetof_Requantize32_per_channel_muls] "I" (offsetof(arm_gemm::Requantize32, per_channel_muls)), [offsetof_Requantize32_per_channel_right_shifts] "I" (offsetof(arm_gemm::Requantize32, per_channel_right_shifts)), [offsetof_Requantize32_per_layer_mul] "I" (offsetof(arm_gemm::Requantize32, per_layer_mul)), [offsetof_Requantize32_per_layer_right_shift] "I" (offsetof(arm_gemm::Requantize32, per_layer_right_shift)), [qp] "r" (&qp)
    : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
  );
}

}  // namespace depthwise
}  // namespace arm_conv

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
