.onLoad <- function(libname, pkgname){
  S7::methods_register()
  registerS3method("tidy", "sqlm::lm_sql_result", tidy.lm_sql_result, envir = asNamespace("broom"))
  registerS3method("glance", "sqlm::lm_sql_result", glance.lm_sql_result, envir = asNamespace("broom"))
  registerS3method("print", "sqlm::lm_sql_result", print.lm_sql_result)
  if (requireNamespace("orbital", quietly = TRUE)) {
    registerS3method("orbital", "sqlm::lm_sql_result", orbital.lm_sql_result, envir = asNamespace("orbital"))
  }
}


#' @import S7
#' @importFrom broom tidy glance
#' @importFrom tibble tibble
#' @importFrom MASS ginv
NULL

# -------------------------------------------------------------------------
# 1. Define S7 Class
# -------------------------------------------------------------------------

#' Result object for SQL-backed Linear Model
#'
#' @description An S7 class that stores the complete results of a SQL-backed
#'   linear regression fitted by [lm_sql()].
#'
#' @details This class is not called directly by users. It is created
#'   internally by [lm_sql()] and returned as the model object. It stores
#'   fitted coefficients, standard errors, t-statistics, p-values, and
#'   model-level summaries (R-squared, F-statistic, AIC, BIC, etc.). The
#'   `term_expressions` property holds named R expressions for each predictor,
#'   which are used by the [orbital.lm_sql_result()] method to generate
#'   in-database prediction expressions.
#'
#' @keywords internal
lm_sql_result <- S7::new_class("lm_sql_result",
                               package = "sqlm",
                               properties = list(
                                   coefficients = S7::class_numeric,
                                   std_error    = S7::class_numeric,
                                   sigma        = S7::class_double,
                                   r_squared    = S7::class_double,
                                   adj_r_squared = S7::class_double,
                                   f_statistic  = S7::class_numeric,
                                   f_p_value    = S7::class_numeric,
                                   logLik       = S7::class_double,
                                   AIC          = S7::class_double,
                                   BIC          = S7::class_double,
                                   nobs         = S7::class_double,
                                   df_residual  = S7::class_double,
                                   df_model     = S7::class_double,
                                   statistic    = S7::class_numeric,
                                   p_value      = S7::class_numeric,
                                   call         = S7::class_any,
                                   term_expressions = S7::class_any
                               )
)

# -------------------------------------------------------------------------
# 2. Helpers
# -------------------------------------------------------------------------

# Quote a column name for SQL safety using double quotes
bt <- function(x) paste0('"', x, '"')

# Internal: Scout for factor levels and generate CASE WHEN SQL + R expressions
get_dummy_map <- function(con, table_sql, col_name) {
    query <- glue::glue("SELECT DISTINCT {bt(col_name)} FROM ({table_sql}) AS z ORDER BY {bt(col_name)}")
    levels <- tryCatch(DBI::dbGetQuery(con, query)[[1]], error = function(e) NULL)

    if (is.null(levels)) return(NULL)
    levels <- stats::na.omit(levels)
    if (length(levels) < 2) return(NULL)
    active_levels <- levels[-1]

    map_sql <- purrr::map(active_levels, function(lvl) {
        safe_lvl <- gsub("'", "''", lvl)
        glue::glue("CASE WHEN {bt(col_name)} = '{safe_lvl}' THEN 1.0 ELSE 0.0 END")
    })

    map_r <- purrr::map(active_levels, function(lvl) {
        escaped_lvl <- gsub('"', '\\"', lvl, fixed = TRUE)
        glue::glue('ifelse({col_name} == "{escaped_lvl}", 1, 0)')
    })

    clean_lvl_names <- gsub("[^a-zA-Z0-9_]", "_", active_levels)
    nms <- paste0(col_name, "_", clean_lvl_names)
    names(map_sql) <- nms
    names(map_r) <- nms
    return(list(sql = map_sql, r = map_r))
}

# Translate R math expressions to SQL
expr_to_sql <- function(expr_str) {
    # Handle I() wrapper — strip and translate inner expression
    if (grepl("^I\\(", expr_str)) {
        inner <- sub("^I\\((.*)\\)$", "\\1", expr_str)
        return(translate_math(inner))
    }
    # Handle log(), sqrt() as bare terms
    if (grepl("^log\\(", expr_str) || grepl("^sqrt\\(", expr_str)) {
        return(translate_math(expr_str))
    }
    return(NULL) # not a transform
}

# Extract R expression for transforms (parallel to expr_to_sql)
expr_to_r <- function(expr_str) {
    if (grepl("^I\\(", expr_str)) {
        return(sub("^I\\((.*)\\)$", "\\1", expr_str))
    }
    if (grepl("^log\\(", expr_str) || grepl("^sqrt\\(", expr_str)) {
        return(expr_str)
    }
    return(NULL)
}

translate_math <- function(s) {
    # x^2 -> POWER(x, 2)
    s <- gsub("([a-zA-Z_][a-zA-Z0-9_.]*)\\s*\\^\\s*([0-9.]+)", "POWER(\\1, \\2)", s)
    # log(x) -> LN(x)
    s <- gsub("\\blog\\(", "LN(", s)
    # sqrt(x) -> SQRT(x) (already valid SQL)
    s
}

# -------------------------------------------------------------------------
# 3. Main Function
# -------------------------------------------------------------------------

#' SQL-Backed Linear Regression
#'
#' @description Fits a linear regression model using SQL aggregation on a
#'   remote database table. The data never leaves the database — only
#'   sufficient statistics (sums and cross-products) are returned to R.
#'
#' @details The function computes the \eqn{X^TX} and \eqn{X^Ty} matrices
#'   entirely inside the database engine via a single SQL aggregation query,
#'   then solves the normal equations in R using Cholesky decomposition
#'   (falling back to Moore-Penrose pseudoinverse for rank-deficient designs).
#'
#'   Supported formula features:
#'   \itemize{
#'     \item Numeric and categorical (character/factor) predictors with
#'       automatic dummy encoding via `CASE WHEN`.
#'     \item Interaction terms (`*` and `:`) including numeric × categorical
#'       and categorical × categorical cross-products.
#'     \item Dot expansion (`y ~ .`) to all non-response columns.
#'     \item Transforms: `I()`, `log()`, and `sqrt()` translated to SQL
#'       equivalents (`POWER`, `LN`, `SQRT`).
#'     \item Date and datetime predictors automatically cast to numeric in SQL.
#'     \item No-intercept models (`y ~ 0 + x`).
#'   }
#'
#'   For grouped data (via [dplyr::group_by()]), a single `GROUP BY` query is
#'   executed and one model per group is returned in a tibble with a `model`
#'   list-column.
#'
#'   NA handling uses listwise deletion: rows with `NULL` in any model variable
#'   are excluded via a `WHERE ... IS NOT NULL` clause.
#'
#' @param formula A formula object (e.g., \code{price ~ x + cut}).
#' @param data A \code{tbl_sql} object (from \pkg{dbplyr}).
#' @param tol Tolerance for detecting linear dependency.
#'
#' @return An S7 object of class \code{lm_sql_result}, or a tibble with a
#'   \code{model} list-column if the data is grouped.
#' @export
lm_sql <- function(formula, data, tol = 1e-7) {
    cl <- match.call()

    # 1. Setup & Connection
    if (is.data.frame(data)) stop("Requires a remote dbplyr 'tbl' object.")

    con <- data$src$con
    group_ids <- dplyr::group_vars(data)
    is_grouped <- length(group_ids) > 0

    # -- Dot expansion (must happen before terms() since terms() chokes on ".") --
    rhs_text <- as.character(formula)[3]
    response_var <- as.character(formula)[2]

    if (grepl("(^|[+*: ])\\.$|^\\.$", trimws(rhs_text))) {
        all_cols <- colnames(data)
        exclude <- c(response_var, group_ids)
        expand_cols <- setdiff(all_cols, exclude)
        has_intercept_orig <- !grepl("^0\\s*\\+|\\+\\s*0", rhs_text)
        formula <- stats::reformulate(expand_cols, response = response_var,
                                      intercept = has_intercept_orig)
    }

    term_obj <- stats::terms(formula)

    y <- response_var
    raw_xs <- attr(term_obj, "term.labels")
    has_intercept <- attr(term_obj, "intercept") == 1

    # 2. NA Handling (Listwise Deletion)
    # Figure out actual column names referenced (strip I(), log(), etc.)
    extract_colnames <- function(terms) {
        cols <- c()
        for (t in terms) {
            # strip I(), log(), sqrt()
            stripped <- t
            stripped <- gsub("^I\\((.*)\\)$", "\\1", stripped)
            stripped <- gsub("\\blog\\(([^)]+)\\)", "\\1", stripped)
            stripped <- gsub("\\bsqrt\\(([^)]+)\\)", "\\1", stripped)
            # split on :, *, +, -, ^, space, parens, commas
            parts <- unlist(strsplit(stripped, "[: *+\\-^(),/]+"))
            # keep only things that look like variable names
            parts <- parts[grepl("^[a-zA-Z._][a-zA-Z0-9._]*$", parts)]
            # remove numeric-only
            parts <- parts[!grepl("^[0-9.]+$", parts)]
            cols <- c(cols, parts)
        }
        unique(cols)
    }

    all_base_vars <- unique(c(y, extract_colnames(raw_xs)))

    data_clean <- data |>
        dplyr::filter(dplyr::if_all(dplyr::all_of(all_base_vars), ~ !is.na(.)))

    tbl_sql <- dbplyr::sql_render(data_clean)

    # 3. Categorical Scout & Pivot + Transform detection
    types <- dplyr::collect(utils::head(data_clean, 1))

    expanded_xs <- list()
    sql_definitions <- list()
    r_definitions <- list()
    # Track which raw categorical vars expanded to which dummies
    cat_expansion <- list()

    for (x in raw_xs) {
        # Skip interactions for now — handle after
        if (grepl(":", x)) {
            expanded_xs <- c(expanded_xs, x)
            next
        }

        # Check for I() or math transforms
        sql_trans <- expr_to_sql(x)
        if (!is.null(sql_trans)) {
            safe_name <- gsub("[^a-zA-Z0-9_]", "_", x)
            expanded_xs <- c(expanded_xs, safe_name)
            sql_definitions[[safe_name]] <- sql_trans
            r_definitions[[safe_name]] <- expr_to_r(x)
            next
        }

        # Check if variable is a date/datetime — cast to numeric
        # R's as.numeric(Date) returns days since 1970-01-01;
        # as.numeric(POSIXct) returns seconds since epoch.
        # Use SQL date subtraction to mirror this behavior.
        if (!is.null(types[[x]]) && inherits(types[[x]], c("Date", "POSIXct", "POSIXlt"))) {
            if (inherits(types[[x]], "Date")) {
                date_sql <- glue::glue("CAST(({bt(x)} - DATE '1970-01-01') AS DOUBLE PRECISION)")
            } else {
                date_sql <- glue::glue("EXTRACT(EPOCH FROM {bt(x)})")
            }
            expanded_xs <- c(expanded_xs, x)
            sql_definitions[[x]] <- date_sql
            r_definitions[[x]] <- paste0("as.numeric(", x, ")")
        } else
        # Check if variable is categorical
        if (!is.null(types[[x]]) && (is.character(types[[x]]) || is.factor(types[[x]]))) {
            dummy_map <- get_dummy_map(con, tbl_sql, x)
            if (!is.null(dummy_map)) {
                expanded_xs <- c(expanded_xs, names(dummy_map$sql))
                sql_definitions <- c(sql_definitions, dummy_map$sql)
                r_definitions <- c(r_definitions, dummy_map$r)
                cat_expansion[[x]] <- names(dummy_map$sql)
            }
        } else {
            expanded_xs <- c(expanded_xs, x)
            sql_definitions[[x]] <- bt(x)
            r_definitions[[x]] <- x
        }
    }

    # -- Post-process interactions with categorical variables --
    final_expanded <- list()
    for (x in expanded_xs) {
        if (!grepl(":", x)) {
            final_expanded <- c(final_expanded, x)
            next
        }
        # x is an interaction like "a:b" or "a:b:c"
        parts <- strsplit(x, ":")[[1]]
        # For each part, check if it was expanded to dummies
        part_expansions <- lapply(parts, function(p) {
            if (p %in% names(cat_expansion)) {
                cat_expansion[[p]]
            } else {
                p
            }
        })
        # Cartesian product of all expanded parts
        combos <- expand.grid(part_expansions, stringsAsFactors = FALSE)
        for (i in seq_len(nrow(combos))) {
            combo_parts <- as.character(combos[i, ])
            iname <- paste(combo_parts, collapse = ":")
            final_expanded <- c(final_expanded, iname)
            # Build SQL for this interaction
            sql_parts <- purrr::map_chr(combo_parts, function(cp) {
                if (cp %in% names(sql_definitions)) {
                    as.character(sql_definitions[[cp]])
                } else {
                    bt(cp)
                }
            })
            sql_definitions[[iname]] <- glue::glue("(1.0 * {paste(sql_parts, collapse = ' * ')})")
            # Build R expression for this interaction
            r_parts <- purrr::map_chr(combo_parts, function(cp) {
                if (cp %in% names(r_definitions)) {
                    as.character(r_definitions[[cp]])
                } else {
                    cp
                }
            })
            r_definitions[[iname]] <- paste(r_parts, collapse = " * ")
        }
    }
    xs <- unlist(final_expanded)

    # 4. Translation Layer
    term_to_sql <- function(t) {
        if (grepl(":", t)) {
            parts <- strsplit(t, ":")[[1]]
            trans <- purrr::map_chr(parts, term_to_sql)
            return(glue::glue("(1.0 * {paste(trans, collapse = ' * ')})"))
        }
        if (t %in% names(sql_definitions)) return(as.character(sql_definitions[[t]]))
        return(bt(t))
    }

    sql_exprs <- purrr::map_chr(xs, term_to_sql)
    names(sql_exprs) <- xs
    clean_name <- function(n) gsub("[^a-zA-Z0-9_]", "_", n)

    # 5. Construct Matrix Query
    all_r_terms <- c(y, xs)
    pairs <- expand.grid(v1 = all_r_terms, v2 = all_r_terms, stringsAsFactors = FALSE)
    pairs$key <- purrr::map2_chr(pairs$v1, pairs$v2, ~ paste(sort(c(.x, .y)), collapse = "-"))
    pairs <- pairs[!duplicated(pairs$key), ]

    sum_exprs <- purrr::map2_chr(pairs$v1, pairs$v2, function(v1, v2) {
        expr1 <- if (v1 == y) bt(y) else sql_exprs[v1]
        expr2 <- if (v2 == y) bt(y) else sql_exprs[v2]
        alias <- glue::glue("S_{clean_name(v1)}_{clean_name(v2)}")
        glue::glue("SUM(1.0 * ({expr1}) * ({expr2})) AS {alias}")
    })

    simple_sum_exprs <- purrr::map_chr(all_r_terms, function(t) {
        expr <- if (t == y) bt(y) else sql_exprs[t]
        glue::glue("SUM(1.0 * {expr}) AS S_{clean_name(t)}")
    })

    # -- Grouping --
    select_group <- if (is_grouped) paste0(paste(bt(group_ids), collapse = ", "), ", ") else ""
    group_clause <- if (is_grouped) paste("GROUP BY", paste(bt(group_ids), collapse = ", ")) else ""

    query <- glue::glue("
      SELECT
        {select_group}
        COUNT(*) as N,
        {paste(c(simple_sum_exprs, sum_exprs), collapse = ',\n        ')}
      FROM ({tbl_sql}) AS subquery
      {group_clause}
    ")

    # 6. Execute
    stats_df <- DBI::dbGetQuery(con, query)

    # Build term_expressions: named character vector mapping coefficient names to R expressions
    term_exprs <- purrr::map_chr(xs, function(t) {
        if (t %in% names(r_definitions)) {
            as.character(r_definitions[[t]])
        } else {
            t
        }
    })
    names(term_exprs) <- xs

    # 7. Internal Solver
    solve_one_row <- function(row_stats) {
        n <- as.numeric(row_stats[["N"]])
        p <- length(xs)
        dim <- if (has_intercept) p + 1 else p

        XtX <- matrix(0, nrow = dim, ncol = dim)
        XtY <- numeric(dim)
        names_vec <- if (has_intercept) c("(Intercept)", xs) else xs
        rownames(XtX) <- colnames(XtX) <- names_vec
        names(XtY) <- names_vec

        get_stat <- function(v1, v2 = NULL) {
            v1 <- clean_name(v1)
            if (!is.null(v2)) v2 <- clean_name(v2)
            if (is.null(v2)) return(as.numeric(row_stats[[paste0("S_", v1)]]))
            key1 <- paste0("S_", v1, "_", v2)
            if (key1 %in% names(row_stats)) return(as.numeric(row_stats[[key1]]))
            return(as.numeric(row_stats[[paste0("S_", v2, "_", v1)]]))
        }

        off <- if (has_intercept) 1 else 0

        if (has_intercept) {
            XtX[1, 1] <- n
            XtY[1] <- get_stat(y)
            for (i in 1:p) {
                val <- get_stat(xs[i])
                XtX[1, i + 1] <- val
                XtX[i + 1, 1] <- val
            }
        }

        for (i in 1:p) {
            XtY[i + off] <- get_stat(xs[i], y)
            for (j in 1:p) {
                val <- get_stat(xs[i], xs[j])
                XtX[i + off, j + off] <- val
            }
        }

        # Solve via Cholesky decomposition (XtX is positive definite for
        # full-rank designs); fall back to ginv for rank-deficient cases
        XtX_inv <- tryCatch(
            chol2inv(chol(XtX)),
            error = function(e) {
                if (requireNamespace("MASS", quietly = TRUE)) {
                    MASS::ginv(XtX, tol = tol)
                } else {
                    stop(e)
                }
            }
        )

        betas <- as.vector(XtX_inv %*% XtY)
        names(betas) <- names_vec

        # Statistics
        S_yy <- get_stat(y, y)
        RSS  <- S_yy - sum(betas * XtY)
        S_y  <- get_stat(y)

        if (has_intercept) {
            df_residual <- n - dim
            df_model    <- dim - 1
            TSS <- S_yy - (S_y^2 / n)
        } else {
            df_residual <- n - p
            df_model    <- p
            TSS <- S_yy  # not mean-corrected for no-intercept
        }

        sigma2 <- max(0, RSS / df_residual)
        std_errors <- sqrt(pmax(0, diag(XtX_inv) * sigma2))

        r_squared <- 1 - (RSS / TSS)
        adj_r_squared <- 1 - (1 - r_squared) * ((n - (if (has_intercept) 1 else 0)) / df_residual)

        if (df_model > 0 && sigma2 > 0) {
            mst <- (TSS - RSS) / df_model
            mse <- RSS / df_residual
            f_stat <- mst / mse
            f_pval <- 1 - stats::pf(f_stat, df_model, df_residual)
        } else {
            f_stat <- NA_real_; f_pval <- NA_real_
        }

        logLik_val <- -n / 2 * (log(2 * pi) + 1 + log(RSS / n))
        k_params <- dim + 1
        aic_val <- 2 * k_params - 2 * logLik_val
        bic_val <- log(n) * k_params - 2 * logLik_val

        statistic <- betas / std_errors
        p_val <- 2 * (1 - stats::pt(abs(statistic), df_residual))

        lm_sql_result(
            coefficients  = betas,
            std_error     = std_errors,
            sigma         = sqrt(sigma2),
            r_squared     = r_squared,
            adj_r_squared = adj_r_squared,
            f_statistic   = as.numeric(f_stat),
            f_p_value     = as.numeric(f_pval),
            logLik        = logLik_val,
            AIC           = aic_val,
            BIC           = bic_val,
            nobs          = as.numeric(n),
            df_residual   = as.numeric(df_residual),
            df_model      = as.numeric(df_model),
            statistic     = statistic,
            p_value       = p_val,
            call          = cl,
            term_expressions = term_exprs
        )
    }

    # 8. Return Results
    if (is_grouped) {
        model_list <- lapply(seq_len(nrow(stats_df)), function(i) {
            solve_one_row(as.list(stats_df[i, ]))
        })
        result_tbl <- stats_df[, group_ids, drop = FALSE]
        result_tbl$model <- model_list
        return(dplyr::as_tibble(result_tbl))
    } else {
        return(solve_one_row(stats_df[1, ]))
    }
}

# -------------------------------------------------------------------------
# 4. Broom Methods
# -------------------------------------------------------------------------

#' Tidy an lm_sql_result
#'
#' @description Extract a tidy tibble of per-term coefficient statistics from
#'   a fitted SQL linear model.
#'
#' @details Returns one row per model term with the estimate, standard error,
#'   t-statistic, and p-value. When `conf.int = TRUE`, confidence intervals
#'   are computed using the t-distribution with `df_residual` degrees of
#'   freedom.
#'
#' @param x An `lm_sql_result` object.
#' @param conf.int Logical. If `TRUE`, include confidence interval columns
#'   `conf.low` and `conf.high`. Defaults to `FALSE`.
#' @param conf.level Confidence level for the interval. Defaults to `0.95`.
#' @param ... Not used.
#' @return A tibble with columns `term`, `estimate`, `std.error`, `statistic`,
#'   and `p.value`. If `conf.int = TRUE`, also `conf.low` and `conf.high`.
#' @export
tidy.lm_sql_result <- function(x, conf.int = FALSE, conf.level = 0.95, ...) {
    ret <- tibble::tibble(
        term      = names(x@coefficients),
        estimate  = x@coefficients,
        std.error = x@std_error,
        statistic = x@statistic,
        p.value   = x@p_value
    )

    if (conf.int) {
        alpha <- (1 - conf.level) / 2
        ci_mult <- stats::qt(1 - alpha, x@df_residual)
        ret$conf.low  <- ret$estimate - ci_mult * ret$std.error
        ret$conf.high <- ret$estimate + ci_mult * ret$std.error
    }
    return(ret)
}

#' Glance at an lm_sql_result
#'
#' @description Extract a single-row tibble of model-level summary statistics
#'   from a fitted SQL linear model.
#'
#' @details Returns R-squared, adjusted R-squared, residual standard error,
#'   F-statistic and its p-value, model degrees of freedom, log-likelihood,
#'   AIC, BIC, number of observations, and residual degrees of freedom.
#'
#' @param x An `lm_sql_result` object.
#' @param ... Not used.
#' @return A single-row tibble with columns `r.squared`, `adj.r.squared`,
#'   `sigma`, `statistic`, `p.value`, `df`, `logLik`, `AIC`, `BIC`, `nobs`,
#'   and `df.residual`.
#' @export
glance.lm_sql_result <- function(x, ...) {
    tibble::tibble(
        r.squared     = x@r_squared,
        adj.r.squared = x@adj_r_squared,
        sigma         = x@sigma,
        statistic     = x@f_statistic,
        p.value       = x@f_p_value,
        df            = x@df_model,
        logLik        = x@logLik,
        AIC           = x@AIC,
        BIC           = x@BIC,
        nobs          = x@nobs,
        df.residual   = x@df_residual
    )
}

# -------------------------------------------------------------------------
# 5. Orbital Method
# -------------------------------------------------------------------------

#' Convert an lm_sql_result to an orbital object
#'
#' @description Creates an orbital object from a fitted SQL linear model,
#'   enabling in-database predictions without pulling data into R.
#'
#' @details Builds a single prediction expression by combining the fitted
#'   coefficients with the R expressions stored in `term_expressions`. For
#'   categorical predictors, the expression includes `ifelse()` calls that
#'   dbplyr translates to SQL `CASE WHEN`. The resulting `orbital_class`
#'   object can be used with [orbital::predict()] to get predictions or
#'   [orbital::augment()] to append a `.pred` column to a database table.
#'
#' @param x An `lm_sql_result` object.
#' @param ... Not used.
#' @param prefix Column name for predictions. Defaults to `".pred"`.
#' @return An `orbital_class` object.
#' @export
orbital.lm_sql_result <- function(x, ..., prefix = ".pred") {
    if (!requireNamespace("orbital", quietly = TRUE)) {
        stop("Package 'orbital' is required. Install it with install.packages('orbital').")
    }

    coefs <- x@coefficients
    exprs <- x@term_expressions
    terms_chr <- character(0)

    for (nm in names(coefs)) {
        val <- coefs[[nm]]
        if (nm == "(Intercept)") {
            terms_chr <- c(terms_chr, deparse(val))
        } else {
            terms_chr <- c(terms_chr, paste0(deparse(val), " * (", exprs[[nm]], ")"))
        }
    }

    pred_expr <- paste(terms_chr, collapse = " + ")
    obj <- stats::setNames(pred_expr, prefix)
    class(obj) <- "orbital_class"
    attr(obj, "pred_names") <- prefix
    obj
}

#' Print an lm_sql_result
#'
#' @description Display a concise summary of a fitted SQL linear model.
#'
#' @details Prints the original function call and the named coefficient vector.
#'
#' @param x An `lm_sql_result` object.
#' @param ... Not used.
#' @return Invisibly returns `x`.
#' @export
print.lm_sql_result <- function(x, ...) {
    cat("\nBig Data Linear Regression (S7)\n")
    cat("-------------------------------\n")
    cat("Call:\n")
    print(x@call)
    cat("\nCoefficients:\n")
    print(x@coefficients)
    cat("\n")
}
