% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/formants.R
\name{transplantFormants}
\alias{transplantFormants}
\title{Transplant formants}
\usage{
transplantFormants(
  donor,
  recipient,
  samplingRate = NULL,
  freqWindow = NULL,
  blur = NULL,
  dynamicRange = 80,
  windowLength = 50,
  step = NULL,
  overlap = 90,
  wn = "gaussian",
  zp = 0
)
}
\arguments{
\item{donor}{the sound that provides the formants (vector, Wave, or file) or
the desired spectral filter (matrix) as returned by
\code{\link{getSpectralEnvelope}}}

\item{recipient}{the sound that receives the formants (vector, Wave, or file)}

\item{samplingRate}{sampling rate of \code{x} (only needed if \code{x} is a
numeric vector)}

\item{freqWindow}{the width of smoothing window used to flatten the
recipient's spectrum per frame. Defaults to median pitch of the donor (or
of the recipient if donor is a filter matrix). If \code{blur} is NULL,
\code{freqWindow} also controls the amount of smoothing applied to the
donor's spectrogram}

\item{blur}{the amount of Gaussian blur applied to the donor's spectrogram as
a faster and more flexible alternative to smoothing it per bin with
\code{freqWindow}. Provide two numbers: frequency (Hz, normally
approximately equal to freqWindow), time (ms) (NA / NULL / 0 means no
blurring in that dimension). See examples and \code{\link{spectrogram}}}

\item{dynamicRange}{dynamic range, dB. All values more than one dynamicRange
under maximum are treated as zero}

\item{windowLength}{length of FFT window, ms (multiple values in a vector
produce a multi-resolution spectrogram)}

\item{step}{you can override \code{overlap} by specifying FFT step, ms - a
vector of the same length as windowLength (NB: because digital audio is
sampled at discrete time intervals of 1/samplingRate, the actual step and
thus the time stamps of STFT frames may be slightly different, eg 24.98866
instead of 25.0 ms)}

\item{overlap}{overlap between successive FFT frames, \%}

\item{wn}{window type accepted by \code{\link[seewave]{ftwindow}}, currently
gaussian, hanning, hamming, bartlett, blackman, flattop, rectangle}

\item{zp}{window length after zero padding, points}
}
\description{
Takes the general spectral envelope of one sound (\code{donor}) and
"transplants" it onto another sound (\code{recipient}). For biological sounds
like speech or animal vocalizations, this has the effect of replacing the
formants in the recipient sound while preserving the original intonation and
(to some extent) voice quality. Note that the amount of spectral smoothing
(specified with \code{freqWindow} or \code{blur}) is a crucial parameter: too
little smoothing, and noise between harmonics will be amplified, creasing
artifacts; too much, and formants may be missed. The default is to set
\code{freqWindow} to the estimated median pitch, but this is time-consuming
and error-prone, so set it to a reasonable value manually if possible. Also
ensure that both sounds have the same sampling rate.
}
\details{
Algorithm: makes spectrograms of both sounds, interpolates and smooths or
blurs the donor spectrogram, flattens the recipient spectrogram, multiplies
the spectrograms, and transforms back into time domain with inverse STFT.
}
\examples{
\dontrun{
# Objective: take formants from the bleating of a sheep and apply them to a
# synthetic sound with any arbitrary duration, intonation, nonlinearities etc
data(sheep, package = 'seewave')  # import a recording from seewave
playme(sheep)
spectrogram(sheep, osc = TRUE)

recipient = soundgen(
  sylLen = 1200,
  pitch = c(100, 300, 250, 200),
  vibratoFreq = 9, vibratoDep = 1,
  addSilence = 180,
  samplingRate = sheep@samp.rate,  # same as donor
  invalidArgAction = 'ignore')  # force to keep the low samplingRate
playme(recipient, sheep@samp.rate)
spectrogram(recipient, sheep@samp.rate, osc = TRUE)

s1 = transplantFormants(
  donor = sheep,
  recipient = recipient,
  samplingRate = sheep@samp.rate)
playme(s1, sheep@samp.rate)
spectrogram(s1, sheep@samp.rate, osc = TRUE)

# The spectral envelope of s1 will be similar to sheep's on a frequency scale
# determined by freqWindow. Compare the spectra:
par(mfrow = c(1, 2))
seewave::meanspec(sheep, dB = 'max0', alim = c(-50, 20), main = 'Donor')
seewave::meanspec(s1, f = sheep@samp.rate, dB = 'max0',
                  alim = c(-50, 20), main = 'Processed recipient')
par(mfrow = c(1, 1))

# if needed, transplant amplitude envelopes as well:
s2 = transplantEnv(donor = sheep, samplingRateD = sheep@samp.rate,
                   recipient = s1, windowLength = 10)
playme(s2, sheep@samp.rate)
spectrogram(s2, sheep@samp.rate, osc = TRUE)

# using "blur" to apply Gaussian blur to the donor's spectrogram instead of
# smoothing per frame with "freqWindow" (~2.5 times faster)
spectrogram(sheep, blur = c(150, 0))  # preview to select the amount of blur
s1b = transplantFormants(
  donor = sheep,
  recipient = recipient,
  samplingRate = sheep@samp.rate,
  freqWindow = 150,
  blur = c(150, 0))
  # blur: 150 = SD of 150 Hz along the frequency axis,
  #      0 = no smoothing along the time axis
playme(s1b, sheep@samp.rate)
spectrogram(s1b, sheep@samp.rate, osc = TRUE)

# Now we use human formants on sheep source: the sheep asks "why?"
s3 = transplantFormants(
  donor = getSpectralEnvelope(
            nr = 512, nc = 100,  # fairly arbitrary dimensions
            formants = 'uaaai',
            samplingRate = sheep@samp.rate),
  recipient = sheep,
  samplingRate = sheep@samp.rate)
playme(s3, sheep@samp.rate)
spectrogram(s3, sheep@samp.rate, osc = TRUE)
}
}
\seealso{
\code{\link{transplantEnv}} \code{\link{getSpectralEnvelope}}
  \code{\link{addFormants}} \code{\link{spectrogram}} \code{\link{soundgen}}
}
