% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detectNLP.R
\name{findJumps}
\alias{findJumps}
\title{Find frequency jumps}
\usage{
findJumps(
  pitch,
  step,
  jumpThres = 8,
  jumpWindow = 80,
  plot = FALSE,
  xlab = "Time, ms",
  ylab = "f0, Hz",
  ...
)
}
\arguments{
\item{pitch}{vector of frequencies per frame, Hz}

\item{step}{time step between frames, ms}

\item{jumpThres}{frames in which pitch changes by \code{jumpThres} octaves/s
more than in the surrounding frames are classified as containing "pitch
jumps". Note that this is the rate of frequency change PER SECOND, not from
one frame to the next}

\item{jumpWindow}{the window for calculating the median pitch slope around
the analyzed frame, ms}

\item{plot}{if TRUE, plots the pitch contour with putative frequency jumps
marked by arrows}

\item{xlab, ylab, ...}{graphical parameters passed to \code{plot}}
}
\value{
Returns a boolean vector of the same length as \code{pitch}, where
  TRUE values correspond to frames with detected pitch jumps.
}
\description{
This function flags frames with apparent pith jumps (frequency jumps, voice
breaks), defined as relatively large and sudden changes in voice pitch or
some other frequency measure (peak frequency, a formant frequency, etc). It
is called by \code{\link{detectNLP}}. Algorithm: a frame is considered to
contain a frequency jump if the absolute slope at this frame exceeds the
average slope over ±\code{jumpWindow} around it by more than
\code{jumpThres}. Note that the slope is considered per second rather than
per time step - that is, taking into account the sampling rate of the
frequency track. Thus, it's not just the change from frame to frame that
defines what is considered a jump, but a change that differs from the trend
in the surrounding frames (see examples). If several consecutive frames
contain apparent jumps, only the greatest of them is preserved.
}
\examples{
pitch = getSmoothContour(anchors = list(
  time = c(0, 350, 351, 890, 891, 1200),
  value = c(140, 230, 460, 330, 220, 200)), len = 40)
step = 25
pj = findJumps(pitch, step, plot = TRUE)

# convert frame indices to time in ms
step = 25
which(pj) * step
# or consider pj's to occur midway between the two frames
which(pj) * step - step / 2

# even very rapid changes are not considered jumps if they match
# the surrounding trend
pitch = getSmoothContour(anchors = list(
  time = c(0, 350, 351, 700),
  value = c(340, 710, 850, 1200)), len = 20)
findJumps(pitch, step, plot = TRUE)
diff(HzToSemitones(pitch)) * (1000 / step) / 12
# the slope at frame 10 (10.4 oct/s) exceeds the jumpThres (8 oct/s), but not
# 10.4 minus the average slope around frame 10 (~3 oct/s, so 10 - 3 < 8)
}
