% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gen_userp.R
\name{gen_userp}
\alias{gen_userp}
\alias{gen_sem_out_userp}
\title{Create a Wrapper To Be Used
in 'lavaan' Models}
\usage{
gen_userp(func, sem_out)

gen_sem_out_userp(
  userp,
  sem_out,
  userp_name = "semlbciuserp1234",
  fix = TRUE,
  control_args = list(),
  iter.max = 10000,
  max_attempts = 5
)
}
\arguments{
\item{func}{A function that receives a
\code{lavaan}-object and returns a scalar.
See Details on the restriction on this
function.}

\item{sem_out}{A \code{lavaan}-class object
to be modified.}

\item{userp}{A function that is
generated by \code{\link[=gen_userp]{gen_userp()}}.}

\item{userp_name}{The name of the
function \code{userp} to be used in the
\code{lavaan} model. It does not have to
be the name of the function in
\code{userp}. Should be changed only if it
conflicts with another object in the
parent environment, which should not
happen if the model is always fitted
in a clean R session.}

\item{fix}{If \code{TRUE}, the default,
the function generated is used to fix
the value of \code{userp} to a target
value using an equality constraint.
If \code{FALSE}, then the function simply
fits the model to the data.}

\item{control_args}{To be passed to
the argument of the same name in
\code{\link[lavaan:lavaan]{lavaan::lavaan()}}. Default is
\code{list()}. Can be used to set the
default values of this argument
in the generated function.}

\item{iter.max}{The maximum number of
iteration when the generated function
fit the model. Default is 10000.}

\item{max_attempts}{If the initial
fit with the equality constraint
fails, how many more attempts
will be made by the generated
function. Default is 5.}
}
\value{
\subsection{\code{gen_userp}}{

It returns a function that
accepts a numeric vector of length
equals to the number of free parameters
in \code{sem_out}, and returns a scalar
which is the output of \code{func}. If this
vector is not supplied, it will try to
find it in the \code{parent.frame()}. This
is how it works inside a \code{lavaan}
model.
}

\subsection{\code{gen_sem_out_userp}}{

If \code{fix} is \code{TRUE}, it returns a
function with these arguments:
\itemize{
\item \code{target}: The value to which the
user-defined parameter will be fixed
to.
\item \code{verbose}: If \code{TRUE}, additional
information will be printed when
fitting the model.
\item \code{control}: The values to be passed
as a list to the argument of the
same name in \code{\link[lavaan:lavaan]{lavaan::lavaan()}}.
\item \code{seed}: Numeric. If supplied, it
will be used in \code{\link[=set.seed]{set.seed()}} to
initialize the random number
generator. Necessary to reproduce
some results because random numbers
are used in some steps in \code{lavaan}.
If \code{NULL}, the default, \code{\link[=set.seed]{set.seed()}}
will not be called.
}

If \code{fix} is `FALSE, then it returns a
function with optional arguments that
will be ignored, Calling it will
simply fit the modified model to the
data. Useful for getting the value of
the user-defined parameter.
}
}
\description{
Make a function on
\code{lavaan} object usable in a \code{lavaan}
model syntax.
}
\details{
\subsection{\code{gen_userp}}{

There are cases in which we want to
create a user parameter which is a
function of other free parameters,
computed by a function. However such
a function may work only on a
\code{lavaan} object.

If the target function works by
extracting parameter estimates stored
in the \code{Model} slot and/or the
\code{implied} slot, then \code{\link[=gen_userp]{gen_userp()}}
can be used to convert it to a
function that retrieves the parameter
estimates when being called by
\code{\link[lavaan:lavaan]{lavaan::lavaan()}} or its wrappers,
modifies the stored
\code{lavaan} object using
\code{\link[lavaan:lav_model]{lavaan::lav_model_set_parameters()}}
and \code{\link[lavaan:lav_model]{lavaan::lav_model_implied()}} to
change the estimates, and call the
target function.

Note that this is an unconventional
way to define a user parameter and
the generated function should always
be checked to see whether it works as
expected.

As shown in the examples, the
parameter computed this may not
have standard error nor \emph{p}-value.

The main purpose is for the point
estimate, for searching the
likelihood-based confidence bound
using \code{\link[=ci_bound_ur]{ci_bound_ur()}} and
\code{\link[=ci_bound_ur_i]{ci_bound_ur_i()}}.

Note that the target function
specified in \code{func} should work
directly on the parameter estimates
stored in the \code{Model} slot and then
get the estimates using
\code{\link[lavaan:lav_model]{lavaan::lav_model_get_parameters()}}.
Functions that work on the unmodified
output generated by
\code{\link[lavaan:lavaan]{lavaan::lavaan()}} usually do not
work.

Users are not recommended to use
\code{\link[=gen_userp]{gen_userp()}} and \code{\link[=gen_sem_out_userp]{gen_sem_out_userp()}}
directly because they require
unconventional way to extract
parameter estimates from a lavaan
model. However, developers may use
them to include functions
they wrote in a lavaan model. This
is the technique used by
\code{\link[=ci_bound_ur_i]{ci_bound_ur_i()}} to constrain any
parameter in a model to an arbitrary
value.
}

\subsection{\code{gen_sem_out_userp}}{

The function \code{\link[=gen_sem_out_userp]{gen_sem_out_userp()}}
is to be used internally
for generating a function for searching
a likelihood-based confidence bound.
It is exported because it needs to
be run in an fresh external R process,
usually created by \code{callr} in other
internal functions.
}
}
\examples{

library(lavaan)

data(simple_med)
dat <- simple_med
mod <-
"
m ~ a*x
y ~ b*m
ab := a*b
"
fit_med <- sem(mod, simple_med, fixed.x = FALSE)
parameterEstimates(fit_med)

# A trivial example for verifying the results
my_ab <- function(object) {
    # Need to use lav_model_get_parameters()
    # because the object is only a modified
    # lavaan-object, not one directly
    # generated by lavaan function
    est <- lavaan::lav_model_get_parameters(object@Model, type = "user")
    unname(est[1] * est[2])
  }

# Check the function
my_ab(fit_med)
coef(fit_med, type = "user")["ab"]

# Create the function
my_userp <- gen_userp(func = my_ab,
                      sem_out = fit_med)
# Try it on the vector of free parameters
my_userp(coef(fit_med))

# Generate a modified lavaan model
fit_userp <- gen_sem_out_userp(userp = my_userp,
                               userp_name = "my_userp",
                               sem_out = fit_med)

# This function can then be used in the model syntax.

# Note that the following example only work when called inside the
# workspace or inside other functions such as ci_bound_ur()`
# and `ci_bound_ur_i()` because `lavaan::sem()` will
# search `my_userp()` in the global environment.

# Therefore, the following lines are commented out.
# They should be run only in a "TRUE" interactive
# session.

# mod2 <-
# "
# m ~ x
# y ~ m
# ab := my_userp()
# "
# fit_med2 <- sem(mod2, simple_med, fixed.x = FALSE)
# parameterEstimates(fit_med2)
#
# # Fit the model with the output of the function, a*b
# # fixed to .50
#
# fit_new <- fit_userp(.50)
#
# # Check if the parameter ab is fixed to .50
# parameterEstimates(fit_new)


}
