#' Get Screenshotbase status
#'
#' Returns quota information for the authenticated account.
#'
#' @param apikey Optional API key to override the session key.
#' @return A named list with status fields.
#' @examples
#' \dontrun{
#' set_api_key("YOUR-API-KEY")
#' screenshotbase_status()
#' }
#' @export
screenshotbase_status <- function(apikey = NULL) {
  req <- .sb_build_request("status", apikey = apikey)
  resp <- httr2::req_perform(req)
  httr2::resp_check_status(resp)
  httr2::resp_body_json(resp, simplifyVector = TRUE)
}

#' Take a website screenshot
#'
#' Renders a website and returns raw image bytes.
#'
#' @param url URL of the website to render (required).
#' @param format Image format: one of "png", "jpg", "jpeg", "gif".
#' @param quality Integer quality for jpg/jpeg (defaults to 80 on server if not set).
#' @param full_page Logical; capture full page.
#' @param viewport_width,viewport_height Integers for viewport size in pixels.
#' @param apikey Optional API key to override the session key.
#' @return A raw vector of the image content.
#' @examples
#' \dontrun{
#' set_api_key("YOUR-API-KEY")
#' bytes <- screenshotbase_take("https://bbc.com", format = "png", full_page = TRUE)
#' writeBin(bytes, "screenshot.png")
#' }
#' @export
screenshotbase_take <- function(
  url,
  format = NULL,
  quality = NULL,
  full_page = NULL,
  viewport_width = NULL,
  viewport_height = NULL,
  apikey = NULL
) {
  stopifnot(is.character(url), length(url) == 1L, nzchar(url))
  query <- list(url = url)
  if (!is.null(format)) {
    fmt <- tolower(as.character(format)[1L])
    if (!fmt %in% c("png", "jpg", "jpeg", "gif")) {
      stop("Invalid format: must be one of 'png','jpg','jpeg','gif'", call. = FALSE)
    }
    query$format <- fmt
  }
  if (!is.null(quality)) {
    if (!is.numeric(quality) || length(quality) != 1L) stop("quality must be a single number", call. = FALSE)
    query$quality <- as.integer(quality)
  }
  if (!is.null(full_page)) {
    stopifnot(is.logical(full_page), length(full_page) == 1L)
    query$full_page <- if (isTRUE(full_page)) 1L else 0L
  }
  if (!is.null(viewport_width)) {
    if (!is.numeric(viewport_width) || length(viewport_width) != 1L) stop("viewport_width must be a single number", call. = FALSE)
    query$viewport_width <- as.integer(viewport_width)
  }
  if (!is.null(viewport_height)) {
    if (!is.numeric(viewport_height) || length(viewport_height) != 1L) stop("viewport_height must be a single number", call. = FALSE)
    query$viewport_height <- as.integer(viewport_height)
  }

  req <- .sb_build_request("take", query = query, apikey = apikey)
  resp <- httr2::req_perform(req)
  httr2::resp_check_status(resp)
  httr2::resp_body_raw(resp)
}

