% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scf_glm.R
\name{scf_glm}
\alias{scf_glm}
\title{Estimate Generalized Linear Model from SCF Microdata}
\usage{
scf_glm(object, formula, family = binomial())
}
\arguments{
\item{object}{A \code{scf_mi_survey} object, typically created using \code{\link[=scf_load]{scf_load()}} and \code{\link[=scf_design]{scf_design()}}.}

\item{formula}{A valid model formula, e.g., \code{rich ~ age + factor(edcl)}.}

\item{family}{A GLM family object such as \code{\link[=binomial]{binomial()}}, \code{\link[=poisson]{poisson()}}, or \code{\link[=gaussian]{gaussian()}}. Defaults to \code{binomial()}.}
}
\value{
An object of class \code{"scf_glm"} and \code{"scf_model_result"} with:
\describe{
\item{results}{A data frame of pooled coefficients, standard errors, z-values, p-values, and significance stars.}
\item{fit}{A list of fit diagnostics including mean and SD of AIC; for binomial models, pseudo-R2 and its SD.}
\item{models}{A list of implicate-level \code{svyglm} model objects.}
\item{call}{The matched function call.}
}
}
\description{
Estimates generalized linear models (GLMs) with SCF public-use microdata.
Use this function when modeling outcomes that follow non-Gaussian
distributions (e.g., binary or count data). Rubin's Rules are used to combine
implicate-level coefficient and variance estimates.

GLMs are performed across SCF implicates using \code{svyglm()} and returns
pooled coefficients, standard errors, z-values, p-values, and fit diagnostics
including AIC and pseudo-R-Squared when applicable.
}
\section{Implementation}{

This function fits a GLM to each implicate in a \code{scf_mi_survey} object
using \code{survey::svyglm()}. The user specifies a model formula and a valid GLM
family (e.g., \code{binomial()}, \code{poisson()}, \code{gaussian()}). Coefficients and
variance-covariance matrices are extracted from each implicate and pooled
using Rubin's Rules.
}

\section{Details}{

Generalized linear models (GLMs) extend linear regression to accommodate
non-Gaussian outcome distributions. The choice of \code{family} determines the
link function and error distribution. For example:
\itemize{
\item \code{binomial()} fits logistic regression for binary outcomes
\item \code{poisson()} models count data
\item \code{gaussian()} recovers standard OLS behavior
}

Model estimation is performed independently on each implicate using
\code{svyglm()} with replicate weights. Rubin's Rules are used to pool coefficient
estimates and variance matrices. For the pooling procedure, see
\code{\link[=scf_MIcombine]{scf_MIcombine()}}.
}

\section{Internal Suppression}{


For CRAN compliance and to prevent diagnostic overload during package checks,
this function internally wraps each implicate-level model call in \code{suppressWarnings()}.
This suppresses the known benign warning:

\code{"non-integer #successes in a binomial glm!"}

which arises from using replicate weights with \code{family = binomial()}. This suppression
does not affect model validity or inference. Users wishing to inspect warnings can
run \code{survey::svyglm()} directly on individual implicates via \code{model$models[[i]]}.

For further background, see:
https://stackoverflow.com/questions/12953045/warning-non-integer-successes-in-a-binomial-glm-survey-packages
}

\examples{
\donttest{
# Do not implement these lines in real analysis:
# Use functions `scf_download()` and `scf_load()`
td <- tempfile("glm_")
dir.create(td)

src <- system.file("extdata", "scf2022_mock_raw.rds", package = "scf")
file.copy(src, file.path(td, "scf2022.rds"), overwrite = TRUE)
scf2022 <- scf_load(2022, data_directory = td)

# Example for real analysis: Run logistic regression
model <- scf_glm(scf2022, own ~ age + factor(edcl), family = binomial())
summary(model)

# Do not implement these lines in real analysis: Cleanup for package check
unlink(td, recursive = TRUE, force = TRUE)
}

}
\seealso{
\code{\link[=scf_ols]{scf_ols()}}, \code{\link[=scf_logit]{scf_logit()}}, \code{\link[=scf_regtable]{scf_regtable()}}
}
