% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weibull_model.R
\name{weibull_model}
\alias{weibull_model}
\title{Weibull Drug Release Kinetic Model}
\arguments{
\item{data}{A data frame containing experimental cumulative percent drug release data.}

\item{time_col}{Character string specifying the column name for time (minutes).}

\item{release_col}{Character string specifying the column name for cumulative percent drug release
or fraction released.}

\item{group_col}{Optional character string specifying a grouping variable
(e.g., formulation, batch).}

\item{pH_col}{Optional character string specifying a column containing pH values.}

\item{plot}{Logical; if TRUE, generates a plot with fitted Weibull release curves.}

\item{annotate}{Logical; if TRUE, annotates the plot with alpha, beta, R^2, and t50
(only if <= 2 groups).}

\item{normalize}{Logical; if TRUE (default), normalizes cumulative percent drug release
to fraction released (0-1). If FALSE, assumes the input release data are already
expressed as fraction released.}
}
\value{
A list containing:
\describe{
  \item{\code{fitted_parameters}}{Data frame with Weibull parameters (\code{alpha}, \code{beta}),
        R^2, and t50 for each group.}
  \item{\code{data}}{Processed data used for model fitting and plotting.}
}
}
\description{
Fits experimental cumulative drug release data to the Weibull model using a
linearized regression of log(-log(1 - Mt/MInf)) versus log(time).The function
automatically normalizes cumulative percent drug release to fraction (0-1) by default
and removes t = 0. In addition, the function supports optional grouping variables
(e.g., formulation, batch) and optional pH-dependent analysis. It generates
publication-quality plots with experimental curves, fitted Weibull straight lines,
a shape-parameter interpretation table, and annotations for the scale parameter
(alpha), shape parameter (beta), coefficient of determination (R^2), and the time
required for 50-percent drug release (t50).

Users can toggle 'normalize = TRUE/FALSE' to use fraction (0-1) or raw percent drug release.
Normalization is recommended (fraction release) because the model assumes
0 <= Mt/MInf <= 1.

Model:
log(-log(1 - Mt/MInf)) = beta * log(t) - beta * log(alpha)

The shape parameter beta indicates release kinetics:
  - beta = 1 : Exponential (first-order release)
  - beta < 1 : Parabolic (decelerating release)
  - beta > 1 : Sigmoidal (accelerating then decelerating)
}
\examples{
# Example I: Single formulation
df1 <- data.frame(
  time = c(0, 15, 30, 45, 60, 90, 120, 150, 180),
  release = c(0, 11.4, 20.8, 30.8, 39.8, 57.8, 72, 84.8, 93.5)
)
weibull_model(
  data = df1,
  time_col = "time",
  release_col = "release",
  normalize = TRUE
)

# Example II: Two formulations (grouped, not pH-dependent)
df2 <- data.frame(
  time = rep(c(0, 15, 30, 45, 60, 75, 90, 105, 120, 150), 2),
  release = c(
    0, 10, 22, 35, 48, 60, 72, 80, 86, 92,   # Formulation A
    0, 8, 18, 28, 38, 48, 58, 64, 68, 72     # Formulation B
  ),
  formulation = rep(c("Formulation A", "Formulation B"), each = 10)
)
weibull_model(
  data = df2,
  time_col = "time",
  release_col = "release",
  group_col = "formulation"
)

# Example III: pH-dependent release
df_pH <- data.frame(
  time = rep(c(0, 27, 60, 88, 95, 120, 138, 155, 175, 180), 2),
  release = c(
    0, 12, 25, 38, 52, 63, 72, 80, 88, 95,   # pH 7.4
    0, 10, 20, 30, 42, 53, 63, 70, 77, 85    # pH 4.5
  ),
  pH = rep(c(7.4, 4.5), each = 10)
)
weibull_model(
  data = df_pH,
  time_col = "time",
  release_col = "release",
  pH_col = "pH"
)

# Example IV: Two formulations under two pH conditions
df1 <- data.frame(
  time = rep(c(0, 20, 40, 60, 80, 100, 120, 140, 160, 180), 2),
  release = c(
    0, 12, 25, 38, 50, 62, 73, 82, 89, 95,   # pH 4.5
    0, 15, 30, 45, 59, 70, 79, 86, 91, 97    # pH 7.6
  ),
  pH = rep(c(4.5, 7.6), each = 10)
)
df2 <- data.frame(
  time = rep(c(0, 15, 30, 45, 60, 75, 90, 105, 120, 135), 2),
  release = c(
    0, 10, 22, 34, 46, 57, 67, 76, 84, 91,   # pH 4.5
    0, 13, 27, 41, 55, 67, 77, 85, 92, 98    # pH 7.6
  ),
  pH = rep(c(4.5, 7.6), each = 10)
)
df_all <- rbind(
  cbind(formulation = "Dataset 1", df1),
  cbind(formulation = "Dataset 2", df2)
)
weibull_model(
  data = df_all,
  time_col = "time",
  release_col = "release",
  group_col = "formulation",
  pH_col = "pH"
)
}
\references{
Weibull, W. (1951) <doi:10.1115/1.4010337> A statistical distribution
function of wide applicability. Journal of Applied Mechanics, 18(3), 293–297.
}
\author{
Paul Angelo C. Manlapaz
}
