#' Make a vertical vector from a matrix
#'
#' Creates a column-wise vector from a matrix. For every second column, the elements are reversed.
#'
#' @param mat A matrix to be converted into a vertical vector
#' @return A numeric vector
#' @importFrom stats cor
#' @export
make_vertical <- function(mat) {
  nr <- nrow(mat)
  nc <- ncol(mat)
  vec <- c()
  for (j in 1:nc) {
    col <- mat[, j]
    if (j %% 2 == 0) col <- rev(col)
    vec <- c(vec, col)
  }
  return(vec)
}

#' Make a horizontal vector from a matrix
#'
#' Creates a row-wise vector from a matrix. For every second row, the elements are reversed.
#'
#' @param mat A matrix to be converted into a horizontal vector
#' @return A numeric vector
#' @export
make_horizontal <- function(mat) {
  nr <- nrow(mat)
  nc <- ncol(mat)
  vec <- c()
  for (i in 1:nr) {
    row <- mat[i, ]
    if (i %% 2 == 0) row <- rev(row)
    vec <- c(vec, row)
  }
  return(vec)
}

#' Compute first-order serial correlation of a vector
#'
#' Computes the correlation between consecutive elements of a numeric vector.
#'
#' @param vec A numeric vector
#' @return Numeric value of the serial correlation
#' @export
serial_corr <- function(vec) {
  vec_num <- as.numeric(vec)
  if (any(is.na(vec_num))) {
    warning("Some values could not be converted to numeric")
  }
  vec_num <- na.omit(vec_num)

  if (length(vec_num) < 2) {
    return(NA)  # Not enough data to compute correlation
  }
  Xi <- vec_num[-length(vec_num)]
  Xip1 <- vec_num[-1]
  return(cor(Xi, Xip1))
}

#' computes the first-order serial correlation for both directions.
#'
#' @param df_mat A numeric matrix
#' @importFrom stats na.omit
#' @return A named list with two elements:
#'   - vertical: first-order serial correlation along vertical snake
#'   - horizontal: first-order serial correlation along horizontal snake
#' @export
serial_corrl <- function(df_mat) {
  # Convert to vertical and horizontal vectors
  vvec <- make_vertical(df_mat)
  hvec <- make_horizontal(df_mat)

  # Compute serial correlations
  vert_corr <- serial_corr(vvec)
  horiz_corr <- serial_corr(hvec)

  # Return as named list
  return(list(
    vertical = vert_corr,
    horizontal = horiz_corr
  ))
}

#' Compute 3x3 moving averages
#'
#' @param mat A numeric matrix (at least 3 rows and 3 columns)
#' @return A numeric matrix of 3x3 moving averages
#' @export
compute_moving_avg <- function(mat) {
  mat <- apply(mat, 2, function(x) as.numeric(as.character(x)))
  r <- nrow(mat)
  c <- ncol(mat)

  if (is.null(r) || is.null(c) || r < 3 || c < 3) {
    stop("Input matrix must be at least 3x3 and numeric.")
  }

  Pmat <- matrix(NA, nrow = r - 2, ncol = c - 2)

  for (i in 2:(r - 1)) {
    for (j in 2:(c - 1)) {
      block <- mat[(i - 1):(i + 1), (j - 1):(j + 1)]
      Pmat[i - 1, j - 1] <- mean(block, na.rm = TRUE)
    }
  }

  return(Pmat)
}

#' Fertility Classes heatmap with 3 * 3 moving average values ---
#' @param mat A matrix to be converted into a horizontal vector
#' @return Heatmap
#' @export
ferti_analysis <- function(mat) {
  # 1. Compute 3x3 moving averages
  Pmat <- compute_moving_avg(mat)

  # 2. Compute serial correlations on moving avg matrix
  serials <- serial_corrl(mat)

  # 3. Prepare heatmap data (categorical classes only)
  vals <- as.numeric(as.vector(Pmat))
  if (all(is.na(vals))) {
    stop("Pmat contains no numeric values to compute range")
  }
  range_vals <- range(vals, na.rm = TRUE)

  if (!all(is.finite(range_vals))) {
    stop("Range contains non-finite values cannot create breaks")
  }

  breaks <- seq(range_vals[1], range_vals[2], length.out = 7)
  labels <- paste0(sprintf("%.1f", breaks[-7]), "-", sprintf("%.1f", breaks[-1]))
  classes <- cut(vals, breaks = breaks, labels = labels, include.lowest = TRUE)
  Pclass <- matrix(classes, nrow = nrow(Pmat), ncol = ncol(Pmat), byrow = FALSE)

  df_heat <- as.data.frame.table(Pclass, responseName = "Class")
  colnames(df_heat) <- c("Row", "Col", "Class")
  df_heat$Row <- as.integer(df_heat$Row)
  df_heat$Col <- as.integer(df_heat$Col)

  # 4. Generate heatmap (without numeric values inside)
  p <- ggplot(df_heat, aes(x = Col, y = Row, fill = Class)) +
    geom_tile(color = "grey50") +
    scale_fill_brewer(palette = "YlOrRd", name = "Classes on 3 Moving Avg") +
    scale_y_reverse() +
    coord_fixed() +
    theme_minimal() +
    labs(title = paste0("Fertility Pattern Heatmap"),
         x = "Column", y = "Row") +
    theme(plot.title = element_text(hjust = 0.5, size = 12, face = "bold"))

  # 5. Return heatmap, serial correlation, and Pmat values separately
  return(list(
    heatmap = p,
    serial_correlation = serials,
    moving_avg_values = Pmat
  ))
}
