#' Issue an error message and stop
#'
#' @param format character, a sprintf style format
#' @param ... optional arguments to the format
#' @keywords internal
halt = function(format, ...) {
  msg = sprintf(format, ...)
  stop(msg, call.=FALSE)
}

#' Issue an error message and stop
#'
#' @param n integer, if n==1, format1 is used. Otherwise format2 is used
#' @param format1 character, the format to be used when n==1
#' @param format2 character, the format to be used when n!=1
#' @param ... optional arguments to the format
#' @keywords internal
halt.n = function(n, format1, format2, ...) {
  format = ifelse(n==1, format1, format2)
  msg = sprintf(format, ...)
  stop(msg, call.=FALSE)
}

#' Issue a warning
#'
#' @param format character, a sprintf style format
#' @param ... optional arguments to the format
#' @keywords internal
warn = function(format, ...) {
  msg = sprintf(format, ...)
  warning(msg, call.=FALSE)
}

#' Collapse a vector into a string
#'
#' Collapses a vector into a comma delimited string with the word 'and'
#' between the last two entries.
#'
#' @param x the vector
#' @param sep the separator for all but the last entry
#' @param maxItems If there are more than this many items, only the first
#'   maxItems are displayed.
#' @param last three sprintf formats (see details)
#' @param quote A function used to quote the elements (such as
#' sQuote or dQuote) or NULL
#'
#' @return a string
#' @keywords internal
#'
#' @details
#' The last= argument contains three sprintf formats. The first is used if
#' there are fewer than maxItems items, the second if there are maxItems+1
#' items, the first if there are more than maxItems+1 items.
#'
#' The arguments are passed as %1$: the string consisting of all items except
#' the last, %2$ the last item %3$ an integer indicating how many items are not
#' shown.
collapse = function(x,sep=', ',
                    maxItems=10,
                    last=c('%1$s and %2$s',
                           '%1$s, %2$s and %3$d more',
                           '%1$s, %2$s and %3$d more'),
                    quote=NULL) {
  if (is.function(quote)) x=quote(x)

  if (length(x)<=1) return(x)

  if (length(x)>maxItems) {
    leftOut = length(x)-maxItems
    x =x[1:maxItems]
  }
  else leftOut=NULL

  firsts  = x[1:(length(x)-1)]
  lastOne = x[length(x)]

  out = paste(firsts,collapse=sep)

  if (length(leftOut)==0) out = sprintf(gettext(last[1]),out,lastOne)
  else {
    fmt = ngettext(leftOut,last[2],last[3])
    out=sprintf(fmt,out,lastOne,leftOut)
  }
  return(out)
}


#' Collapse a vector into a string
#'
#' Collapses a vector into a comma delimited string with the word 'and'
#' between the last two entries.
#'
#' @param x the vector
#' @param sep the separator for all but the last entry
#' @param maxItems If there are more than this many items, only the first
#'   maxItems are displayed.
#' @param last three sprintf formats (see details)
#' @param quote A function used to quote the elements (such as
#' sQuote or dQuote) or NULL
#'
#' @return a string
#' @keywords internal
#'
#' @details
#' The last= argument contains three sprintf formats. The first is used if
#' there are fewer than maxItems items, the second if there are maxItems+1
#' items, the first if there are more than maxItems+1 items.
#'
#' The arguments are passed as %1$: the string consisting of all items except
#' the last, %2$ the last item %3$ an integer indicating how many items are not
#' shown.
#'
#' @examples
#' multiRec:::paste.and(1:4)  # "1, 2, 3 and 4"
#' multiRec:::paste.and(1:16, maxItems=4) # "1, 2, 3, 4 and 12 more"
paste.and = function(x,sep=', ',
                     maxItems=10,
                     last=c('%1$s and %2$s',
                            '%1$s, %2$s and %3$d more',
                            '%1$s, %2$s and %3$d more'),
                     quote=NULL) {
  collapse(x,sep=sep,maxItems=maxItems,last=last,quote=quote)
}
