/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5CXprivate.h" 
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5ESprivate.h" 
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5FOprivate.h" 
#include "H5Gprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  
#include "H5Tpkg.h"      
#include "H5VLprivate.h" 
#include "H5VMprivate.h" 

#include "H5Tconv_integer.h"
#include "H5Tconv_float.h"
#include "H5Tconv_string.h"
#include "H5Tconv_bitfield.h"
#include "H5Tconv_compound.h"
#include "H5Tconv_reference.h"
#include "H5Tconv_enum.h"
#include "H5Tconv_vlen.h"
#include "H5Tconv_array.h"
#include "H5Tconv_complex.h"

#define H5T_ENCODE_VERSION 0

#define H5T_DEF_CONV_TABLE_SLOTS 128

#define H5T_INIT_TYPE_BITFIELD_CORE                                                                          \
    {                                                                                                        \
        dt->shared->type = H5T_BITFIELD;                                                                     \
    }

#define H5T_INIT_TYPE_BITFIELD_COMMON(ENDIANNESS)                                                            \
    {                                                                                                        \
        H5T_INIT_TYPE_NUM_COMMON(ENDIANNESS)                                                                 \
        H5T_INIT_TYPE_BITFIELD_CORE;                                                                         \
    }

#define H5T_INIT_TYPE_BITFIELDLE_CORE                                                                        \
    {                                                                                                        \
        H5T_INIT_TYPE_BITFIELD_COMMON(H5T_ORDER_LE)                                                          \
    }

#define H5T_INIT_TYPE_BITFIELDBE_CORE                                                                        \
    {                                                                                                        \
        H5T_INIT_TYPE_BITFIELD_COMMON(H5T_ORDER_BE)                                                          \
    }

#define H5T_INIT_TYPE_TIME_CORE                                                                              \
    {                                                                                                        \
        dt->shared->type = H5T_TIME;                                                                         \
    }

#define H5T_INIT_TYPE_OFFSET_CORE                                                                            \
    {                                                                                                        \
        dt->shared->u.atomic.offset = 0;                                                                     \
    }

#define H5T_INIT_TYPE_NUM_COMMON(ENDIANNESS)                                                                 \
    {                                                                                                        \
        dt->shared->u.atomic.order   = ENDIANNESS;                                                           \
        dt->shared->u.atomic.offset  = 0;                                                                    \
        dt->shared->u.atomic.lsb_pad = H5T_PAD_ZERO;                                                         \
        dt->shared->u.atomic.msb_pad = H5T_PAD_ZERO;                                                         \
    }

#define H5T_INIT_TYPE_FLOAT16_COMMON(ENDIANNESS)                                                             \
    {                                                                                                        \
        H5T_INIT_TYPE_NUM_COMMON(ENDIANNESS)                                                                 \
        dt->shared->u.atomic.u.f.sign  = 15;                                                                 \
        dt->shared->u.atomic.u.f.epos  = 10;                                                                 \
        dt->shared->u.atomic.u.f.esize = 5;                                                                  \
        dt->shared->u.atomic.u.f.ebias = 0xf;                                                                \
        dt->shared->u.atomic.u.f.mpos  = 0;                                                                  \
        dt->shared->u.atomic.u.f.msize = 10;                                                                 \
        dt->shared->u.atomic.u.f.norm  = H5T_NORM_IMPLIED;                                                   \
        dt->shared->u.atomic.u.f.pad   = H5T_PAD_ZERO;                                                       \
    }

#define H5T_INIT_TYPE_FLOAT16LE_CORE                                                                         \
    {                                                                                                        \
        H5T_INIT_TYPE_FLOAT16_COMMON(H5T_ORDER_LE)                                                           \
    }

#define H5T_INIT_TYPE_FLOAT16BE_CORE                                                                         \
    {                                                                                                        \
        H5T_INIT_TYPE_FLOAT16_COMMON(H5T_ORDER_BE)                                                           \
    }

#define H5T_INIT_TYPE_BFLOAT16_COMMON(ENDIANNESS)                                                            \
    {                                                                                                        \
        H5T_INIT_TYPE_NUM_COMMON(ENDIANNESS)                                                                 \
        dt->shared->u.atomic.u.f.sign  = 15;                                                                 \
        dt->shared->u.atomic.u.f.epos  = 7;                                                                  \
        dt->shared->u.atomic.u.f.esize = 8;                                                                  \
        dt->shared->u.atomic.u.f.ebias = 0x7f;                                                               \
        dt->shared->u.atomic.u.f.mpos  = 0;                                                                  \
        dt->shared->u.atomic.u.f.msize = 7;                                                                  \
        dt->shared->u.atomic.u.f.norm  = H5T_NORM_IMPLIED;                                                   \
        dt->shared->u.atomic.u.f.pad   = H5T_PAD_ZERO;                                                       \
    }

#define H5T_INIT_TYPE_BFLOAT16LE_CORE                                                                        \
    {                                                                                                        \
        H5T_INIT_TYPE_BFLOAT16_COMMON(H5T_ORDER_LE)                                                          \
    }

#define H5T_INIT_TYPE_BFLOAT16BE_CORE                                                                        \
    {                                                                                                        \
        H5T_INIT_TYPE_BFLOAT16_COMMON(H5T_ORDER_BE)                                                          \
    }

#define H5T_INIT_TYPE_FLOAT8E4M3_CORE                                                                        \
    {                                                                                                        \
        H5T_INIT_TYPE_NUM_COMMON(H5T_ORDER_LE)                                      \
        dt->shared->u.atomic.u.f.sign  = 7;                                                                  \
        dt->shared->u.atomic.u.f.epos  = 3;                                                                  \
        dt->shared->u.atomic.u.f.esize = 4;                                                                  \
        dt->shared->u.atomic.u.f.ebias = 0x7;                                                                \
        dt->shared->u.atomic.u.f.mpos  = 0;                                                                  \
        dt->shared->u.atomic.u.f.msize = 3;                                                                  \
        dt->shared->u.atomic.u.f.norm  = H5T_NORM_IMPLIED;                                                   \
        dt->shared->u.atomic.u.f.pad   = H5T_PAD_ZERO;                                                       \
    }

#define H5T_INIT_TYPE_FLOAT8E5M2_CORE                                                                        \
    {                                                                                                        \
        H5T_INIT_TYPE_NUM_COMMON(H5T_ORDER_LE)                                      \
        dt->shared->u.atomic.u.f.sign  = 7;                                                                  \
        dt->shared->u.atomic.u.f.epos  = 2;                                                                  \
        dt->shared->u.atomic.u.f.esize = 5;                                                                  \
        dt->shared->u.atomic.u.f.ebias = 0xf;                                                                \
        dt->shared->u.atomic.u.f.mpos  = 0;                                                                  \
        dt->shared->u.atomic.u.f.msize = 2;                                                                  \
        dt->shared->u.atomic.u.f.norm  = H5T_NORM_IMPLIED;                                                   \
        dt->shared->u.atomic.u.f.pad   = H5T_PAD_ZERO;                                                       \
    }

#define H5T_INIT_TYPE_FLOAT_COMMON(ENDIANNESS)                                                               \
    {                                                                                                        \
        H5T_INIT_TYPE_NUM_COMMON(ENDIANNESS)                                                                 \
        dt->shared->u.atomic.u.f.sign  = 31;                                                                 \
        dt->shared->u.atomic.u.f.epos  = 23;                                                                 \
        dt->shared->u.atomic.u.f.esize = 8;                                                                  \
        dt->shared->u.atomic.u.f.ebias = 0x7f;                                                               \
        dt->shared->u.atomic.u.f.mpos  = 0;                                                                  \
        dt->shared->u.atomic.u.f.msize = 23;                                                                 \
        dt->shared->u.atomic.u.f.norm  = H5T_NORM_IMPLIED;                                                   \
        dt->shared->u.atomic.u.f.pad   = H5T_PAD_ZERO;                                                       \
    }

#define H5T_INIT_TYPE_FLOATLE_CORE                                                                           \
    {                                                                                                        \
        H5T_INIT_TYPE_FLOAT_COMMON(H5T_ORDER_LE)                                                             \
    }

#define H5T_INIT_TYPE_FLOATBE_CORE                                                                           \
    {                                                                                                        \
        H5T_INIT_TYPE_FLOAT_COMMON(H5T_ORDER_BE)                                                             \
    }

#define H5T_INIT_TYPE_DOUBLE_COMMON(ENDIANNESS)                                                              \
    {                                                                                                        \
        H5T_INIT_TYPE_NUM_COMMON(ENDIANNESS)                                                                 \
        dt->shared->u.atomic.u.f.sign  = 63;                                                                 \
        dt->shared->u.atomic.u.f.epos  = 52;                                                                 \
        dt->shared->u.atomic.u.f.esize = 11;                                                                 \
        dt->shared->u.atomic.u.f.ebias = 0x03ff;                                                             \
        dt->shared->u.atomic.u.f.mpos  = 0;                                                                  \
        dt->shared->u.atomic.u.f.msize = 52;                                                                 \
        dt->shared->u.atomic.u.f.norm  = H5T_NORM_IMPLIED;                                                   \
        dt->shared->u.atomic.u.f.pad   = H5T_PAD_ZERO;                                                       \
    }

#define H5T_INIT_TYPE_DOUBLELE_CORE                                                                          \
    {                                                                                                        \
        H5T_INIT_TYPE_DOUBLE_COMMON(H5T_ORDER_LE)                                                            \
    }

#define H5T_INIT_TYPE_DOUBLEBE_CORE                                                                          \
    {                                                                                                        \
        H5T_INIT_TYPE_DOUBLE_COMMON(H5T_ORDER_BE)                                                            \
    }

#define H5T_INIT_TYPE_FLOATVAX_CORE                                                                          \
    {                                                                                                        \
        H5T_INIT_TYPE_NUM_COMMON(H5T_ORDER_VAX)                                                              \
        dt->shared->u.atomic.u.f.sign  = 31;                                                                 \
        dt->shared->u.atomic.u.f.epos  = 23;                                                                 \
        dt->shared->u.atomic.u.f.esize = 8;                                                                  \
        dt->shared->u.atomic.u.f.ebias = 0x81;                                                               \
        dt->shared->u.atomic.u.f.mpos  = 0;                                                                  \
        dt->shared->u.atomic.u.f.msize = 23;                                                                 \
        dt->shared->u.atomic.u.f.norm  = H5T_NORM_IMPLIED;                                                   \
        dt->shared->u.atomic.u.f.pad   = H5T_PAD_ZERO;                                                       \
        dt->shared->version            = H5O_DTYPE_VERSION_3;                                                \
    }

#define H5T_INIT_TYPE_DOUBLEVAX_CORE                                                                         \
    {                                                                                                        \
        H5T_INIT_TYPE_NUM_COMMON(H5T_ORDER_VAX)                                                              \
        dt->shared->u.atomic.u.f.sign  = 63;                                                                 \
        dt->shared->u.atomic.u.f.epos  = 52;                                                                 \
        dt->shared->u.atomic.u.f.esize = 11;                                                                 \
        dt->shared->u.atomic.u.f.ebias = 0x0401;                                                             \
        dt->shared->u.atomic.u.f.mpos  = 0;                                                                  \
        dt->shared->u.atomic.u.f.msize = 52;                                                                 \
        dt->shared->u.atomic.u.f.norm  = H5T_NORM_IMPLIED;                                                   \
        dt->shared->u.atomic.u.f.pad   = H5T_PAD_ZERO;                                                       \
        dt->shared->version            = H5O_DTYPE_VERSION_3;                                                \
    }

#define H5T_INIT_TYPE_SINT_COMMON(ENDIANNESS)                                                                \
    {                                                                                                        \
        H5T_INIT_TYPE_NUM_COMMON(ENDIANNESS)                                                                 \
        dt->shared->u.atomic.u.i.sign = H5T_SGN_2;                                                           \
    }

#define H5T_INIT_TYPE_SINTLE_CORE                                                                            \
    {                                                                                                        \
        H5T_INIT_TYPE_SINT_COMMON(H5T_ORDER_LE)                                                              \
    }

#define H5T_INIT_TYPE_SINTBE_CORE                                                                            \
    {                                                                                                        \
        H5T_INIT_TYPE_SINT_COMMON(H5T_ORDER_BE)                                                              \
    }

#define H5T_INIT_TYPE_UINT_COMMON(ENDIANNESS)                                                                \
    {                                                                                                        \
        H5T_INIT_TYPE_NUM_COMMON(ENDIANNESS)                                                                 \
        dt->shared->u.atomic.u.i.sign = H5T_SGN_NONE;                                                        \
    }

#define H5T_INIT_TYPE_UINTLE_CORE                                                                            \
    {                                                                                                        \
        H5T_INIT_TYPE_UINT_COMMON(H5T_ORDER_LE)                                                              \
    }

#define H5T_INIT_TYPE_UINTBE_CORE                                                                            \
    {                                                                                                        \
        H5T_INIT_TYPE_UINT_COMMON(H5T_ORDER_BE)                                                              \
    }

#define H5T_INIT_TYPE_ALLOC_COMMON(TYPE)                                                                     \
    {                                                                                                        \
        dt->sh_loc.type  = H5O_SHARE_TYPE_UNSHARED;                                                          \
        dt->shared->type = TYPE;                                                                             \
    }

#define H5T_INIT_TYPE_OPAQ_CORE                                                                              \
    {                                                                                                        \
        H5T_INIT_TYPE_ALLOC_COMMON(H5T_OPAQUE)                                                               \
        dt->shared->u.opaque.tag = H5MM_xstrdup("");                                                         \
    }

#define H5T_INIT_TYPE_STRING_COMMON                                                                          \
    {                                                                                                        \
        H5T_INIT_TYPE_ALLOC_COMMON(H5T_STRING)                                                               \
        H5T_INIT_TYPE_NUM_COMMON(H5T_ORDER_NONE)                                                             \
        dt->shared->u.atomic.u.s.cset = H5F_DEFAULT_CSET;                                                    \
    }

#define H5T_INIT_TYPE_CSTRING_CORE                                                                           \
    {                                                                                                        \
        H5T_INIT_TYPE_STRING_COMMON                                                                          \
        dt->shared->u.atomic.u.s.pad = H5T_STR_NULLTERM;                                                     \
    }

#define H5T_INIT_TYPE_FORSTRING_CORE                                                                         \
    {                                                                                                        \
        H5T_INIT_TYPE_STRING_COMMON                                                                          \
        dt->shared->u.atomic.u.s.pad = H5T_STR_SPACEPAD;                                                     \
    }

#define H5T_INIT_TYPE_REF_COMMON                                                                             \
    {                                                                                                        \
        H5T_INIT_TYPE_ALLOC_COMMON(H5T_REFERENCE)                                                            \
        H5T_INIT_TYPE_NUM_COMMON(H5T_ORDER_NONE)                                                             \
        dt->shared->force_conv        = true;                                                                \
        dt->shared->u.atomic.u.r.file = NULL;                                                                \
        dt->shared->u.atomic.u.r.loc  = H5T_LOC_BADLOC;                                                      \
        dt->shared->u.atomic.u.r.cls  = NULL;                                                                \
    }

#define H5T_INIT_TYPE_OBJREF_CORE                                                                            \
    {                                                                                                        \
        H5T_INIT_TYPE_REF_COMMON                                                                             \
        dt->shared->u.atomic.u.r.rtype   = H5R_OBJECT1;                                                      \
        dt->shared->u.atomic.u.r.opaque  = false;                                                            \
        dt->shared->u.atomic.u.r.version = 0;                                                                \
    }

#define H5T_INIT_TYPE_REGREF_CORE                                                                            \
    {                                                                                                        \
        H5T_INIT_TYPE_REF_COMMON                                                                             \
        dt->shared->u.atomic.u.r.rtype   = H5R_DATASET_REGION1;                                              \
        dt->shared->u.atomic.u.r.opaque  = false;                                                            \
        dt->shared->u.atomic.u.r.version = 0;                                                                \
    }

#define H5T_INIT_TYPE_REF_CORE                                                                               \
    {                                                                                                        \
        H5T_INIT_TYPE_REF_COMMON                                                                             \
        dt->shared->u.atomic.u.r.rtype   = H5R_OBJECT2;                                                      \
        dt->shared->u.atomic.u.r.opaque  = true;                                                             \
        dt->shared->u.atomic.u.r.version = H5R_ENCODE_VERSION;                                               \
        dt->shared->version              = H5O_DTYPE_VERSION_4;                                              \
    }

#define H5T_INIT_TYPE_SET_SIZE(SIZE)                                                                         \
    {                                                                                                        \
        dt->shared->size          = SIZE;                                                                    \
        dt->shared->u.atomic.prec = 8 * (SIZE);                                                              \
    }

#define H5T_INIT_TYPE_NOSET_SIZE(SIZE)                                                                       \
    {                                                                                                        \
    }

#define H5T_INIT_TYPE_COPY_CREATE(BASE)                                                                      \
    {                                                                                                        \
                                                                          \
        if (NULL == (dt = H5T_copy(BASE, H5T_COPY_TRANSIENT)))                                               \
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "duplicating base type failed");                   \
    }

#define H5T_INIT_TYPE_ALLOC_CREATE(BASE)                                                                     \
    {                                                                                                        \
                                                                             \
        if (NULL == (dt = H5T__alloc()))                                                                     \
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "memory allocation failed");                      \
    }

#define H5T_INIT_TYPE(GUTS, GLOBAL, CRT_TMPL, BASE, SIZE_TMPL, SIZE)                                         \
    {                                                                                                        \
                                                                                \
        H5_GLUE3(H5T_INIT_TYPE_, CRT_TMPL, _CREATE)                                                          \
        (BASE)                                                                                               \
                                                                                                             \
                                                                       \
            dt->shared->state = H5T_STATE_IMMUTABLE;                                                         \
        H5_GLUE3(H5T_INIT_TYPE_, SIZE_TMPL, _SIZE)                                                           \
        (SIZE)                                                                                               \
                                                                                                             \
                                                                       \
            H5_GLUE3(H5T_INIT_TYPE_, GUTS, _CORE)                                                            \
                                                                                                             \
                                                                                        \
            if (((GLOBAL) = H5I_register(H5I_DATATYPE, dt, false)) < 0)                                      \
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL, "unable to register datatype atom");       \
    }

typedef H5T_t *(*H5T_copy_func_t)(H5T_t *old_dt);

static herr_t H5T__register_int(H5T_pers_t pers, const char *name, H5T_t *src, H5T_t *dst,
                                H5T_lib_conv_t func);
static herr_t H5T__register(H5T_pers_t pers, const char *name, H5T_t *src, H5T_t *dst, H5T_conv_func_t *conv);
static htri_t H5T__compiler_conv(H5T_t *src, H5T_t *dst);
static herr_t H5T__set_size(H5T_t *dt, size_t size);
static herr_t H5T__close_cb(void *dt, void **request);
static herr_t H5T__init_path_table(void);
static bool   H5T__path_table_search(const H5T_t *src, const H5T_t *dst, int *idx, int *last_cmp);
static H5T_path_t *H5T__path_find_real(const H5T_t *src, const H5T_t *dst, const char *name,
                                       H5T_conv_func_t *conv);
static herr_t      H5T__path_find_init_new_path(H5T_path_t *path, const H5T_t *src, const H5T_t *dst,
                                                H5T_conv_func_t *conv, H5T_conv_ctx_t *conv_ctx);
static herr_t      H5T__path_free(H5T_path_t *path, H5T_conv_ctx_t *conv_ctx);
static bool        H5T_path_match(H5T_path_t *path, H5T_pers_t pers, const char *name, H5T_t *src, H5T_t *dst,
                                  H5VL_object_t *owned_vol_obj, H5T_conv_t func);
static bool   H5T_path_match_find_type_with_volobj(const H5T_t *datatype, const H5VL_object_t *owned_vol_obj);
static bool   H5T__detect_vlen_ref(const H5T_t *dt);
static H5T_t *H5T__initiate_copy(const H5T_t *old_dt);
static H5T_t *H5T__copy_transient(H5T_t *old_dt);
static H5T_t *H5T__copy_all(H5T_t *old_dt);
static herr_t H5T__complete_copy(H5T_t *new_dt, const H5T_t *old_dt, H5T_shared_t *reopened_fo,
                                 bool set_memory_type, H5T_copy_func_t copyfn);

H5T_order_t H5T_native_order_g = H5T_ORDER_ERROR;

bool H5_PKG_INIT_VAR = false;

hid_t H5T_IEEE_F16BE_g = H5I_INVALID_HID;
hid_t H5T_IEEE_F16LE_g = H5I_INVALID_HID;
hid_t H5T_IEEE_F32BE_g = H5I_INVALID_HID;
hid_t H5T_IEEE_F32LE_g = H5I_INVALID_HID;
hid_t H5T_IEEE_F64BE_g = H5I_INVALID_HID;
hid_t H5T_IEEE_F64LE_g = H5I_INVALID_HID;

hid_t H5T_FLOAT_BFLOAT16BE_g = H5I_INVALID_HID;
hid_t H5T_FLOAT_BFLOAT16LE_g = H5I_INVALID_HID;
hid_t H5T_FLOAT_F8E4M3_g     = H5I_INVALID_HID;
hid_t H5T_FLOAT_F8E5M2_g     = H5I_INVALID_HID;

hid_t H5T_COMPLEX_IEEE_F16BE_g = H5I_INVALID_HID;
hid_t H5T_COMPLEX_IEEE_F16LE_g = H5I_INVALID_HID;
hid_t H5T_COMPLEX_IEEE_F32BE_g = H5I_INVALID_HID;
hid_t H5T_COMPLEX_IEEE_F32LE_g = H5I_INVALID_HID;
hid_t H5T_COMPLEX_IEEE_F64BE_g = H5I_INVALID_HID;
hid_t H5T_COMPLEX_IEEE_F64LE_g = H5I_INVALID_HID;

hid_t H5T_VAX_F32_g = H5I_INVALID_HID;
hid_t H5T_VAX_F64_g = H5I_INVALID_HID;

hid_t H5T_STD_I8BE_g        = H5I_INVALID_HID;
hid_t H5T_STD_I8LE_g        = H5I_INVALID_HID;
hid_t H5T_STD_I16BE_g       = H5I_INVALID_HID;
hid_t H5T_STD_I16LE_g       = H5I_INVALID_HID;
hid_t H5T_STD_I32BE_g       = H5I_INVALID_HID;
hid_t H5T_STD_I32LE_g       = H5I_INVALID_HID;
hid_t H5T_STD_I64BE_g       = H5I_INVALID_HID;
hid_t H5T_STD_I64LE_g       = H5I_INVALID_HID;
hid_t H5T_STD_U8BE_g        = H5I_INVALID_HID;
hid_t H5T_STD_U8LE_g        = H5I_INVALID_HID;
hid_t H5T_STD_U16BE_g       = H5I_INVALID_HID;
hid_t H5T_STD_U16LE_g       = H5I_INVALID_HID;
hid_t H5T_STD_U32BE_g       = H5I_INVALID_HID;
hid_t H5T_STD_U32LE_g       = H5I_INVALID_HID;
hid_t H5T_STD_U64BE_g       = H5I_INVALID_HID;
hid_t H5T_STD_U64LE_g       = H5I_INVALID_HID;
hid_t H5T_STD_B8BE_g        = H5I_INVALID_HID;
hid_t H5T_STD_B8LE_g        = H5I_INVALID_HID;
hid_t H5T_STD_B16BE_g       = H5I_INVALID_HID;
hid_t H5T_STD_B16LE_g       = H5I_INVALID_HID;
hid_t H5T_STD_B32BE_g       = H5I_INVALID_HID;
hid_t H5T_STD_B32LE_g       = H5I_INVALID_HID;
hid_t H5T_STD_B64BE_g       = H5I_INVALID_HID;
hid_t H5T_STD_B64LE_g       = H5I_INVALID_HID;
hid_t H5T_STD_REF_OBJ_g     = H5I_INVALID_HID;
hid_t H5T_STD_REF_DSETREG_g = H5I_INVALID_HID;
hid_t H5T_STD_REF_g         = H5I_INVALID_HID;

hid_t H5T_UNIX_D32BE_g = H5I_INVALID_HID;
hid_t H5T_UNIX_D32LE_g = H5I_INVALID_HID;
hid_t H5T_UNIX_D64BE_g = H5I_INVALID_HID;
hid_t H5T_UNIX_D64LE_g = H5I_INVALID_HID;

hid_t H5T_C_S1_g = H5I_INVALID_HID;

hid_t H5T_FORTRAN_S1_g = H5I_INVALID_HID;

hid_t H5T_NATIVE_SCHAR_g           = H5I_INVALID_HID;
hid_t H5T_NATIVE_UCHAR_g           = H5I_INVALID_HID;
hid_t H5T_NATIVE_SHORT_g           = H5I_INVALID_HID;
hid_t H5T_NATIVE_USHORT_g          = H5I_INVALID_HID;
hid_t H5T_NATIVE_INT_g             = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT_g            = H5I_INVALID_HID;
hid_t H5T_NATIVE_LONG_g            = H5I_INVALID_HID;
hid_t H5T_NATIVE_ULONG_g           = H5I_INVALID_HID;
hid_t H5T_NATIVE_LLONG_g           = H5I_INVALID_HID;
hid_t H5T_NATIVE_ULLONG_g          = H5I_INVALID_HID;
hid_t H5T_NATIVE_FLOAT16_g         = H5I_INVALID_HID;
hid_t H5T_NATIVE_FLOAT_g           = H5I_INVALID_HID;
hid_t H5T_NATIVE_DOUBLE_g          = H5I_INVALID_HID;
hid_t H5T_NATIVE_LDOUBLE_g         = H5I_INVALID_HID;
hid_t H5T_NATIVE_FLOAT_COMPLEX_g   = H5I_INVALID_HID;
hid_t H5T_NATIVE_DOUBLE_COMPLEX_g  = H5I_INVALID_HID;
hid_t H5T_NATIVE_LDOUBLE_COMPLEX_g = H5I_INVALID_HID;
hid_t H5T_NATIVE_B8_g              = H5I_INVALID_HID;
hid_t H5T_NATIVE_B16_g             = H5I_INVALID_HID;
hid_t H5T_NATIVE_B32_g             = H5I_INVALID_HID;
hid_t H5T_NATIVE_B64_g             = H5I_INVALID_HID;
hid_t H5T_NATIVE_OPAQUE_g          = H5I_INVALID_HID;
hid_t H5T_NATIVE_HADDR_g           = H5I_INVALID_HID;
hid_t H5T_NATIVE_HSIZE_g           = H5I_INVALID_HID;
hid_t H5T_NATIVE_HSSIZE_g          = H5I_INVALID_HID;
hid_t H5T_NATIVE_HERR_g            = H5I_INVALID_HID;
hid_t H5T_NATIVE_HBOOL_g           = H5I_INVALID_HID;

hid_t H5T_NATIVE_INT8_g        = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT8_g       = H5I_INVALID_HID;
hid_t H5T_NATIVE_INT_LEAST8_g  = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT_LEAST8_g = H5I_INVALID_HID;
hid_t H5T_NATIVE_INT_FAST8_g   = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT_FAST8_g  = H5I_INVALID_HID;

hid_t H5T_NATIVE_INT16_g        = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT16_g       = H5I_INVALID_HID;
hid_t H5T_NATIVE_INT_LEAST16_g  = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT_LEAST16_g = H5I_INVALID_HID;
hid_t H5T_NATIVE_INT_FAST16_g   = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT_FAST16_g  = H5I_INVALID_HID;

hid_t H5T_NATIVE_INT32_g        = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT32_g       = H5I_INVALID_HID;
hid_t H5T_NATIVE_INT_LEAST32_g  = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT_LEAST32_g = H5I_INVALID_HID;
hid_t H5T_NATIVE_INT_FAST32_g   = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT_FAST32_g  = H5I_INVALID_HID;

hid_t H5T_NATIVE_INT64_g        = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT64_g       = H5I_INVALID_HID;
hid_t H5T_NATIVE_INT_LEAST64_g  = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT_LEAST64_g = H5I_INVALID_HID;
hid_t H5T_NATIVE_INT_FAST64_g   = H5I_INVALID_HID;
hid_t H5T_NATIVE_UINT_FAST64_g  = H5I_INVALID_HID;

size_t H5T_POINTER_ALIGN_g     = 0;
size_t H5T_HVL_ALIGN_g         = 0;
size_t H5T_HOBJREF_ALIGN_g     = 0;
size_t H5T_HDSETREGREF_ALIGN_g = 0;
size_t H5T_REF_ALIGN_g         = 0;

size_t H5T_NATIVE_SCHAR_ALIGN_g           = 0;
size_t H5T_NATIVE_UCHAR_ALIGN_g           = 0;
size_t H5T_NATIVE_SHORT_ALIGN_g           = 0;
size_t H5T_NATIVE_USHORT_ALIGN_g          = 0;
size_t H5T_NATIVE_INT_ALIGN_g             = 0;
size_t H5T_NATIVE_UINT_ALIGN_g            = 0;
size_t H5T_NATIVE_LONG_ALIGN_g            = 0;
size_t H5T_NATIVE_ULONG_ALIGN_g           = 0;
size_t H5T_NATIVE_LLONG_ALIGN_g           = 0;
size_t H5T_NATIVE_ULLONG_ALIGN_g          = 0;
size_t H5T_NATIVE_FLOAT16_ALIGN_g         = 0;
size_t H5T_NATIVE_FLOAT_ALIGN_g           = 0;
size_t H5T_NATIVE_DOUBLE_ALIGN_g          = 0;
size_t H5T_NATIVE_LDOUBLE_ALIGN_g         = 0;
size_t H5T_NATIVE_FLOAT_COMPLEX_ALIGN_g   = 0;
size_t H5T_NATIVE_DOUBLE_COMPLEX_ALIGN_g  = 0;
size_t H5T_NATIVE_LDOUBLE_COMPLEX_ALIGN_g = 0;

size_t H5T_NATIVE_INT8_ALIGN_g        = 0;
size_t H5T_NATIVE_UINT8_ALIGN_g       = 0;
size_t H5T_NATIVE_INT_LEAST8_ALIGN_g  = 0;
size_t H5T_NATIVE_UINT_LEAST8_ALIGN_g = 0;
size_t H5T_NATIVE_INT_FAST8_ALIGN_g   = 0;
size_t H5T_NATIVE_UINT_FAST8_ALIGN_g  = 0;

size_t H5T_NATIVE_INT16_ALIGN_g        = 0;
size_t H5T_NATIVE_UINT16_ALIGN_g       = 0;
size_t H5T_NATIVE_INT_LEAST16_ALIGN_g  = 0;
size_t H5T_NATIVE_UINT_LEAST16_ALIGN_g = 0;
size_t H5T_NATIVE_INT_FAST16_ALIGN_g   = 0;
size_t H5T_NATIVE_UINT_FAST16_ALIGN_g  = 0;

size_t H5T_NATIVE_INT32_ALIGN_g        = 0;
size_t H5T_NATIVE_UINT32_ALIGN_g       = 0;
size_t H5T_NATIVE_INT_LEAST32_ALIGN_g  = 0;
size_t H5T_NATIVE_UINT_LEAST32_ALIGN_g = 0;
size_t H5T_NATIVE_INT_FAST32_ALIGN_g   = 0;
size_t H5T_NATIVE_UINT_FAST32_ALIGN_g  = 0;

size_t H5T_NATIVE_INT64_ALIGN_g        = 0;
size_t H5T_NATIVE_UINT64_ALIGN_g       = 0;
size_t H5T_NATIVE_INT_LEAST64_ALIGN_g  = 0;
size_t H5T_NATIVE_UINT_LEAST64_ALIGN_g = 0;
size_t H5T_NATIVE_INT_FAST64_ALIGN_g   = 0;
size_t H5T_NATIVE_UINT_FAST64_ALIGN_g  = 0;

#ifdef H5_HAVE__FLOAT16

H5__Float16 H5T_NATIVE_FLOAT16_POS_INF_g = 0.0f;
H5__Float16 H5T_NATIVE_FLOAT16_NEG_INF_g = 0.0f;
#endif
float       H5T_NATIVE_FLOAT_POS_INF_g   = 0.0F;
float       H5T_NATIVE_FLOAT_NEG_INF_g   = 0.0F;
double      H5T_NATIVE_DOUBLE_POS_INF_g  = 0.0;
double      H5T_NATIVE_DOUBLE_NEG_INF_g  = 0.0;
long double H5T_NATIVE_LDOUBLE_POS_INF_g = 0.0L;
long double H5T_NATIVE_LDOUBLE_NEG_INF_g = 0.0L;
#ifdef H5_HAVE_COMPLEX_NUMBERS
H5_float_complex   H5T_NATIVE_FLOAT_COMPLEX_POS_INF_g;
H5_float_complex   H5T_NATIVE_FLOAT_COMPLEX_NEG_INF_g;
H5_double_complex  H5T_NATIVE_DOUBLE_COMPLEX_POS_INF_g;
H5_double_complex  H5T_NATIVE_DOUBLE_COMPLEX_NEG_INF_g;
H5_ldouble_complex H5T_NATIVE_LDOUBLE_COMPLEX_POS_INF_g;
H5_ldouble_complex H5T_NATIVE_LDOUBLE_COMPLEX_NEG_INF_g;
#endif

H5FL_DEFINE(H5T_t);
H5FL_DEFINE(H5T_shared_t);

const unsigned H5O_dtype_ver_bounds[] = {
    H5O_DTYPE_VERSION_1,     
    H5O_DTYPE_VERSION_3,     
    H5O_DTYPE_VERSION_3,     
    H5O_DTYPE_VERSION_4,     
    H5O_DTYPE_VERSION_4,     
    H5O_DTYPE_VERSION_5,     
    H5O_DTYPE_VERSION_LATEST 
};

static struct {
    int          npaths; 
    size_t       apaths; 
    H5T_path_t **path;   
    int          nsoft;  
    size_t       asoft;  
    H5T_soft_t  *soft;   
} H5T_g;

H5FL_DEFINE_STATIC(H5T_path_t);

static const H5I_class_t H5I_DATATYPE_CLS[1] = {{
    H5I_DATATYPE,             
    0,                        
    8,                        
    (H5I_free_t)H5T__close_cb 
}};

static bool H5T_top_package_initialize_s = false;

herr_t
H5T_init(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)
    

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__init_inf(void)
{
    H5T_t        *dst_p;               
    H5T_atomic_t *dst;                 
    uint8_t      *d;                   
    size_t        half_size;           
    size_t        u;                   
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (dst_p = (H5T_t *)H5I_object(H5T_NATIVE_FLOAT_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    dst = &dst_p->shared->u.atomic;

    
    if (H5T_ORDER_LE != H5T_native_order_g && H5T_ORDER_BE != H5T_native_order_g)
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unsupported byte order");

    
    d = (uint8_t *)&H5T_NATIVE_FLOAT_POS_INF_g;
    H5T__bit_set(d, dst->u.f.sign, (size_t)1, false);
    H5T__bit_set(d, dst->u.f.epos, dst->u.f.esize, true);
    H5T__bit_set(d, dst->u.f.mpos, dst->u.f.msize, false);

    
    if (H5T_ORDER_BE == H5T_native_order_g) {
        half_size = dst_p->shared->size / 2;
        for (u = 0; u < half_size; u++) {
            uint8_t tmp                      = d[dst_p->shared->size - (u + 1)];
            d[dst_p->shared->size - (u + 1)] = d[u];
            d[u]                             = tmp;
        } 
    }     

    
    d = (uint8_t *)&H5T_NATIVE_FLOAT_NEG_INF_g;
    H5T__bit_set(d, dst->u.f.sign, (size_t)1, true);
    H5T__bit_set(d, dst->u.f.epos, dst->u.f.esize, true);
    H5T__bit_set(d, dst->u.f.mpos, dst->u.f.msize, false);

    
    if (H5T_ORDER_BE == H5T_native_order_g) {
        half_size = dst_p->shared->size / 2;
        for (u = 0; u < half_size; u++) {
            uint8_t tmp                      = d[dst_p->shared->size - (u + 1)];
            d[dst_p->shared->size - (u + 1)] = d[u];
            d[u]                             = tmp;
        } 
    }     

    
    if (NULL == (dst_p = (H5T_t *)H5I_object(H5T_NATIVE_DOUBLE_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    dst = &dst_p->shared->u.atomic;

    
    if (H5T_ORDER_LE != H5T_native_order_g && H5T_ORDER_BE != H5T_native_order_g)
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unsupported byte order");

    
    d = (uint8_t *)&H5T_NATIVE_DOUBLE_POS_INF_g;
    H5T__bit_set(d, dst->u.f.sign, (size_t)1, false);
    H5T__bit_set(d, dst->u.f.epos, dst->u.f.esize, true);
    H5T__bit_set(d, dst->u.f.mpos, dst->u.f.msize, false);

    
    if (H5T_ORDER_BE == H5T_native_order_g) {
        half_size = dst_p->shared->size / 2;
        for (u = 0; u < half_size; u++) {
            uint8_t tmp                      = d[dst_p->shared->size - (u + 1)];
            d[dst_p->shared->size - (u + 1)] = d[u];
            d[u]                             = tmp;
        } 
    }     

    
    d = (uint8_t *)&H5T_NATIVE_DOUBLE_NEG_INF_g;
    H5T__bit_set(d, dst->u.f.sign, (size_t)1, true);
    H5T__bit_set(d, dst->u.f.epos, dst->u.f.esize, true);
    H5T__bit_set(d, dst->u.f.mpos, dst->u.f.msize, false);

    
    if (H5T_ORDER_BE == H5T_native_order_g) {
        half_size = dst_p->shared->size / 2;
        for (u = 0; u < half_size; u++) {
            uint8_t tmp                      = d[dst_p->shared->size - (u + 1)];
            d[dst_p->shared->size - (u + 1)] = d[u];
            d[u]                             = tmp;
        } 
    }     

    
    if (NULL == (dst_p = (H5T_t *)H5I_object(H5T_NATIVE_LDOUBLE_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    dst = &dst_p->shared->u.atomic;

    
    if (H5T_ORDER_LE != H5T_native_order_g && H5T_ORDER_BE != H5T_native_order_g)
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unsupported byte order");

    
    d = (uint8_t *)&H5T_NATIVE_LDOUBLE_POS_INF_g;
    H5T__bit_set(d, dst->u.f.sign, (size_t)1, false);
    H5T__bit_set(d, dst->u.f.epos, dst->u.f.esize, true);
    H5T__bit_set(d, dst->u.f.mpos, dst->u.f.msize, false);

    
    if (H5T_ORDER_BE == H5T_native_order_g) {
        half_size = dst_p->shared->size / 2;
        for (u = 0; u < half_size; u++) {
            uint8_t tmp                      = d[dst_p->shared->size - (u + 1)];
            d[dst_p->shared->size - (u + 1)] = d[u];
            d[u]                             = tmp;
        } 
    }     

    
    d = (uint8_t *)&H5T_NATIVE_LDOUBLE_NEG_INF_g;
    H5T__bit_set(d, dst->u.f.sign, (size_t)1, true);
    H5T__bit_set(d, dst->u.f.epos, dst->u.f.esize, true);
    H5T__bit_set(d, dst->u.f.mpos, dst->u.f.msize, false);

    
    if (H5T_ORDER_BE == H5T_native_order_g) {
        half_size = dst_p->shared->size / 2;
        for (u = 0; u < half_size; u++) {
            uint8_t tmp                      = d[dst_p->shared->size - (u + 1)];
            d[dst_p->shared->size - (u + 1)] = d[u];
            d[u]                             = tmp;
        } 
    }     

#ifdef H5_HAVE__FLOAT16
    
    if (NULL == (dst_p = (H5T_t *)H5I_object(H5T_NATIVE_FLOAT16_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    dst = &dst_p->shared->u.atomic;

    
    if (H5T_ORDER_LE != H5T_native_order_g && H5T_ORDER_BE != H5T_native_order_g)
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unsupported byte order");

    
    d = (uint8_t *)&H5T_NATIVE_FLOAT16_POS_INF_g;
    H5T__bit_set(d, dst->u.f.sign, (size_t)1, false);
    H5T__bit_set(d, dst->u.f.epos, dst->u.f.esize, true);
    H5T__bit_set(d, dst->u.f.mpos, dst->u.f.msize, false);

    
    if (H5T_ORDER_BE == H5T_native_order_g) {
        half_size = dst_p->shared->size / 2;
        for (u = 0; u < half_size; u++) {
            uint8_t tmp                      = d[dst_p->shared->size - (u + 1)];
            d[dst_p->shared->size - (u + 1)] = d[u];
            d[u]                             = tmp;
        } 
    }     

    
    d = (uint8_t *)&H5T_NATIVE_FLOAT16_NEG_INF_g;
    H5T__bit_set(d, dst->u.f.sign, (size_t)1, true);
    H5T__bit_set(d, dst->u.f.epos, dst->u.f.esize, true);
    H5T__bit_set(d, dst->u.f.mpos, dst->u.f.msize, false);

    
    if (H5T_ORDER_BE == H5T_native_order_g) {
        half_size = dst_p->shared->size / 2;
        for (u = 0; u < half_size; u++) {
            uint8_t tmp                      = d[dst_p->shared->size - (u + 1)];
            d[dst_p->shared->size - (u + 1)] = d[u];
            d[u]                             = tmp;
        } 
    }     
#endif

#ifdef H5_HAVE_COMPLEX_NUMBERS
    
    memcpy(&H5T_NATIVE_FLOAT_COMPLEX_POS_INF_g, &H5T_NATIVE_FLOAT_POS_INF_g, sizeof(float));
    memcpy(&H5T_NATIVE_FLOAT_COMPLEX_NEG_INF_g, &H5T_NATIVE_FLOAT_NEG_INF_g, sizeof(float));
    memcpy(&H5T_NATIVE_DOUBLE_COMPLEX_POS_INF_g, &H5T_NATIVE_DOUBLE_POS_INF_g, sizeof(double));
    memcpy(&H5T_NATIVE_DOUBLE_COMPLEX_NEG_INF_g, &H5T_NATIVE_DOUBLE_NEG_INF_g, sizeof(double));
    memcpy(&H5T_NATIVE_LDOUBLE_COMPLEX_POS_INF_g, &H5T_NATIVE_LDOUBLE_POS_INF_g, sizeof(long double));
    memcpy(&H5T_NATIVE_LDOUBLE_COMPLEX_NEG_INF_g, &H5T_NATIVE_LDOUBLE_NEG_INF_g, sizeof(long double));
#endif

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__init_package(void)
{
    H5T_t  *native_schar   = NULL; 
    H5T_t  *native_uchar   = NULL; 
    H5T_t  *native_short   = NULL; 
    H5T_t  *native_ushort  = NULL; 
    H5T_t  *native_int     = NULL; 
    H5T_t  *native_uint    = NULL; 
    H5T_t  *native_long    = NULL; 
    H5T_t  *native_ulong   = NULL; 
    H5T_t  *native_llong   = NULL; 
    H5T_t  *native_ullong  = NULL; 
    H5T_t  *native_float   = NULL; 
    H5T_t  *native_double  = NULL; 
    H5T_t  *native_ldouble = NULL; 
    H5T_t  *std_u8le       = NULL; 
    H5T_t  *std_u8be       = NULL; 
    H5T_t  *std_u16le      = NULL; 
    H5T_t  *std_u16be      = NULL; 
    H5T_t  *std_u32le      = NULL; 
    H5T_t  *std_u32be      = NULL; 
    H5T_t  *std_u64le      = NULL; 
    H5T_t  *std_u64be      = NULL; 
    H5T_t  *dt             = NULL;
    H5T_t  *tmp_float      = NULL;
    H5T_t  *fixedpt        = NULL; 
    H5T_t  *floatpt        = NULL; 
    H5T_t  *string         = NULL; 
    H5T_t  *bitfield       = NULL; 
    H5T_t  *compound       = NULL; 
    H5T_t  *enum_type      = NULL; 
    H5T_t  *vlen           = NULL; 
    H5T_t  *array          = NULL; 
    H5T_t  *cplx           = NULL; 
    H5T_t  *objref         = NULL; 
    H5T_t  *regref         = NULL; 
    H5T_t  *ref            = NULL; 
    hsize_t dim[1]         = {1};  
    herr_t  status;
    bool    copied_dtype =
        true; 
#ifdef H5_HAVE__FLOAT16
    H5T_t *native_float16 = NULL; 
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    H5T_t *native_float_complex  = NULL; 
    H5T_t *native_double_complex = NULL; 
    H5T_t *native_ldouble_complex =
        NULL; 
#endif
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5I_register_type(H5I_DATATYPE_CLS) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to initialize interface");

    
    
    HDcompile_assert(H5T_NCLASSES < 16);

    
    if (H5T__init_native_float_types() < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to initialize floating-point types");

#ifdef H5_HAVE_COMPLEX_NUMBERS
    
    if (H5T__init_native_complex_types() < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to initialize complex number types");
#endif

    
    if (H5T__init_native_internal() < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to initialize integers");

    
    if (NULL == (native_schar = (H5T_t *)H5I_object(H5T_NATIVE_SCHAR_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_uchar = (H5T_t *)H5I_object(H5T_NATIVE_UCHAR_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_short = (H5T_t *)H5I_object(H5T_NATIVE_SHORT_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_ushort = (H5T_t *)H5I_object(H5T_NATIVE_USHORT_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_int = (H5T_t *)H5I_object(H5T_NATIVE_INT_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_uint = (H5T_t *)H5I_object(H5T_NATIVE_UINT_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_long = (H5T_t *)H5I_object(H5T_NATIVE_LONG_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_ulong = (H5T_t *)H5I_object(H5T_NATIVE_ULONG_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_llong = (H5T_t *)H5I_object(H5T_NATIVE_LLONG_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_ullong = (H5T_t *)H5I_object(H5T_NATIVE_ULLONG_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
#ifdef H5_HAVE__FLOAT16
    if (NULL == (native_float16 = (H5T_t *)H5I_object(H5T_NATIVE_FLOAT16_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
#endif
    if (NULL == (native_float = (H5T_t *)H5I_object(H5T_NATIVE_FLOAT_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_double = (H5T_t *)H5I_object(H5T_NATIVE_DOUBLE_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_ldouble = (H5T_t *)H5I_object(H5T_NATIVE_LDOUBLE_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
#ifdef H5_HAVE_COMPLEX_NUMBERS
    if (NULL == (native_float_complex = (H5T_t *)H5I_object(H5T_NATIVE_FLOAT_COMPLEX_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_double_complex = (H5T_t *)H5I_object(H5T_NATIVE_DOUBLE_COMPLEX_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (native_ldouble_complex = (H5T_t *)H5I_object(H5T_NATIVE_LDOUBLE_COMPLEX_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
#endif

    

    
    H5T_INIT_TYPE(BITFIELD, H5T_NATIVE_B8_g, COPY, native_uint, SET, 1)

    
    H5T_INIT_TYPE(BITFIELD, H5T_NATIVE_B16_g, COPY, native_uint, SET, 2)

    
    H5T_INIT_TYPE(BITFIELD, H5T_NATIVE_B32_g, COPY, native_uint, SET, 4)

    
    H5T_INIT_TYPE(BITFIELD, H5T_NATIVE_B64_g, COPY, native_uint, SET, 8)

    
    H5T_INIT_TYPE(OFFSET, H5T_NATIVE_HADDR_g, COPY, native_uint, SET, sizeof(haddr_t))

    
    H5T_INIT_TYPE(OFFSET, H5T_NATIVE_HSIZE_g, COPY, native_uint, SET, sizeof(hsize_t))

    
    H5T_INIT_TYPE(OFFSET, H5T_NATIVE_HSSIZE_g, COPY, native_int, SET, sizeof(hssize_t))

    
    H5T_INIT_TYPE(OFFSET, H5T_NATIVE_HERR_g, COPY, native_int, SET, sizeof(herr_t))

    
    H5T_INIT_TYPE(OFFSET, H5T_NATIVE_HBOOL_g, COPY, native_uint, SET, sizeof(bool))

    

    
    H5T_INIT_TYPE(FLOAT16LE, H5T_IEEE_F16LE_g, COPY, native_double, SET, 2)

    
    H5T_INIT_TYPE(FLOAT16BE, H5T_IEEE_F16BE_g, COPY, native_double, SET, 2)

    
    H5T_INIT_TYPE(FLOATLE, H5T_IEEE_F32LE_g, COPY, native_double, SET, 4)

    
    H5T_INIT_TYPE(FLOATBE, H5T_IEEE_F32BE_g, COPY, native_double, SET, 4)

    
    H5T_INIT_TYPE(DOUBLELE, H5T_IEEE_F64LE_g, COPY, native_double, SET, 8)

    
    H5T_INIT_TYPE(DOUBLEBE, H5T_IEEE_F64BE_g, COPY, native_double, SET, 8)

    

    
    H5T_INIT_TYPE(BFLOAT16LE, H5T_FLOAT_BFLOAT16LE_g, COPY, native_double, SET, 2)

    
    H5T_INIT_TYPE(BFLOAT16BE, H5T_FLOAT_BFLOAT16BE_g, COPY, native_double, SET, 2)

    
    H5T_INIT_TYPE(FLOAT8E4M3, H5T_FLOAT_F8E4M3_g, COPY, native_double, SET, 1)

    
    H5T_INIT_TYPE(FLOAT8E5M2, H5T_FLOAT_F8E5M2_g, COPY, native_double, SET, 1)

    

    
    H5T_INIT_TYPE(FLOATVAX, H5T_VAX_F32_g, COPY, native_double, SET, 4)

    
    H5T_INIT_TYPE(DOUBLEVAX, H5T_VAX_F64_g, COPY, native_double, SET, 8)

    

    
    H5T_INIT_TYPE(SINTLE, H5T_STD_I8LE_g, COPY, native_int, SET, 1)

    
    H5T_INIT_TYPE(SINTBE, H5T_STD_I8BE_g, COPY, native_int, SET, 1)

    
    H5T_INIT_TYPE(SINTLE, H5T_STD_I16LE_g, COPY, native_int, SET, 2)

    
    H5T_INIT_TYPE(SINTBE, H5T_STD_I16BE_g, COPY, native_int, SET, 2)

    
    H5T_INIT_TYPE(SINTLE, H5T_STD_I32LE_g, COPY, native_int, SET, 4)

    
    H5T_INIT_TYPE(SINTBE, H5T_STD_I32BE_g, COPY, native_int, SET, 4)

    
    H5T_INIT_TYPE(SINTLE, H5T_STD_I64LE_g, COPY, native_int, SET, 8)

    
    H5T_INIT_TYPE(SINTBE, H5T_STD_I64BE_g, COPY, native_int, SET, 8)

    
    H5T_INIT_TYPE(UINTLE, H5T_STD_U8LE_g, COPY, native_uint, SET, 1)
    std_u8le = dt; 

    
    H5T_INIT_TYPE(UINTBE, H5T_STD_U8BE_g, COPY, native_uint, SET, 1)
    std_u8be = dt; 

    
    H5T_INIT_TYPE(UINTLE, H5T_STD_U16LE_g, COPY, native_uint, SET, 2)
    std_u16le = dt; 

    
    H5T_INIT_TYPE(UINTBE, H5T_STD_U16BE_g, COPY, native_uint, SET, 2)
    std_u16be = dt; 

    
    H5T_INIT_TYPE(UINTLE, H5T_STD_U32LE_g, COPY, native_uint, SET, 4)
    std_u32le = dt; 

    
    H5T_INIT_TYPE(UINTBE, H5T_STD_U32BE_g, COPY, native_uint, SET, 4)
    std_u32be = dt; 

    
    H5T_INIT_TYPE(UINTLE, H5T_STD_U64LE_g, COPY, native_uint, SET, 8)
    std_u64le = dt; 

    
    H5T_INIT_TYPE(UINTBE, H5T_STD_U64BE_g, COPY, native_uint, SET, 8)
    std_u64be = dt; 

    

    
    H5T_INIT_TYPE(BITFIELDLE, H5T_STD_B8LE_g, COPY, std_u8le, NOSET, -)
    bitfield = dt; 

    
    H5T_INIT_TYPE(BITFIELDBE, H5T_STD_B8BE_g, COPY, std_u8be, NOSET, -)

    
    H5T_INIT_TYPE(BITFIELDLE, H5T_STD_B16LE_g, COPY, std_u16le, NOSET, -)

    
    H5T_INIT_TYPE(BITFIELDBE, H5T_STD_B16BE_g, COPY, std_u16be, NOSET, -)

    
    H5T_INIT_TYPE(BITFIELDLE, H5T_STD_B32LE_g, COPY, std_u32le, NOSET, -)

    
    H5T_INIT_TYPE(BITFIELDBE, H5T_STD_B32BE_g, COPY, std_u32be, NOSET, -)

    
    H5T_INIT_TYPE(BITFIELDLE, H5T_STD_B64LE_g, COPY, std_u64le, NOSET, -)

    
    H5T_INIT_TYPE(BITFIELDBE, H5T_STD_B64BE_g, COPY, std_u64be, NOSET, -)

    

    
    H5T_INIT_TYPE(TIME, H5T_UNIX_D32LE_g, COPY, std_u32le, NOSET, -)

    
    H5T_INIT_TYPE(TIME, H5T_UNIX_D32BE_g, COPY, std_u32be, NOSET, -)

    
    H5T_INIT_TYPE(TIME, H5T_UNIX_D64LE_g, COPY, std_u64le, NOSET, -)

    
    H5T_INIT_TYPE(TIME, H5T_UNIX_D64BE_g, COPY, std_u64be, NOSET, -)

    
    copied_dtype = false;

    
    H5T_INIT_TYPE(OPAQ, H5T_NATIVE_OPAQUE_g, ALLOC, -, SET, 1)

    

    
    H5T_INIT_TYPE(CSTRING, H5T_C_S1_g, ALLOC, -, SET, 1)
    string = dt; 

    

    
    H5T_INIT_TYPE(FORSTRING, H5T_FORTRAN_S1_g, ALLOC, -, SET, 1)

    

    
    H5T_INIT_TYPE(OBJREF, H5T_STD_REF_OBJ_g, ALLOC, -, NOSET, -)
    if (H5T_set_loc(dt, NULL, H5T_LOC_MEMORY) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "invalid datatype location");
    objref = dt; 

    
    H5T_INIT_TYPE(REGREF, H5T_STD_REF_DSETREG_g, ALLOC, -, NOSET, -)
    if (H5T_set_loc(dt, NULL, H5T_LOC_MEMORY) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "invalid datatype location");
    regref = dt; 

    
    H5T_INIT_TYPE(REF, H5T_STD_REF_g, ALLOC, -, NOSET, -)
    if (H5T_set_loc(dt, NULL, H5T_LOC_MEMORY) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "invalid datatype location");
    ref = dt; 

    

    
    if (NULL == (tmp_float = H5I_object(H5T_IEEE_F16LE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (dt = H5T__complex_create(tmp_float)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "couldn't create complex number datatype");
    if ((H5T_COMPLEX_IEEE_F16LE_g = H5I_register(H5I_DATATYPE, dt, false)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL, "unable to register datatype atom");

    
    if (NULL == (tmp_float = H5I_object(H5T_IEEE_F16BE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (dt = H5T__complex_create(tmp_float)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "couldn't create complex number datatype");
    if ((H5T_COMPLEX_IEEE_F16BE_g = H5I_register(H5I_DATATYPE, dt, false)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL, "unable to register datatype atom");

    
    if (NULL == (tmp_float = H5I_object(H5T_IEEE_F32LE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (dt = H5T__complex_create(tmp_float)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "couldn't create complex number datatype");
    if ((H5T_COMPLEX_IEEE_F32LE_g = H5I_register(H5I_DATATYPE, dt, false)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL, "unable to register datatype atom");

    
    if (NULL == (tmp_float = H5I_object(H5T_IEEE_F32BE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (dt = H5T__complex_create(tmp_float)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "couldn't create complex number datatype");
    if ((H5T_COMPLEX_IEEE_F32BE_g = H5I_register(H5I_DATATYPE, dt, false)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL, "unable to register datatype atom");

    
    if (NULL == (tmp_float = H5I_object(H5T_IEEE_F64LE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (dt = H5T__complex_create(tmp_float)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "couldn't create complex number datatype");
    if ((H5T_COMPLEX_IEEE_F64LE_g = H5I_register(H5I_DATATYPE, dt, false)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL, "unable to register datatype atom");

    
    if (NULL == (tmp_float = H5I_object(H5T_IEEE_F64BE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype object");
    if (NULL == (dt = H5T__complex_create(tmp_float)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "couldn't create complex number datatype");
    if ((H5T_COMPLEX_IEEE_F64BE_g = H5I_register(H5I_DATATYPE, dt, false)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL, "unable to register datatype atom");

    
    fixedpt = native_int;
    floatpt = native_float;
    if (NULL == (compound = H5T__create(H5T_COMPOUND, (size_t)1)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (NULL == (enum_type = H5T__create(H5T_ENUM, (size_t)1)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (NULL == (vlen = H5T__vlen_create(native_int)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (NULL == (array = H5T__array_create(native_int, 1, dim)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (NULL == (cplx = H5T__complex_create(native_float)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    status = 0;

    status |= H5T__register_int(H5T_PERS_SOFT, "i_i", fixedpt, fixedpt, H5T__conv_i_i);
    status |= H5T__register_int(H5T_PERS_SOFT, "i_f", fixedpt, floatpt, H5T__conv_i_f);
    status |= H5T__register_int(H5T_PERS_SOFT, "i_complex", fixedpt, cplx, H5T__conv_i_complex);
    status |= H5T__register_int(H5T_PERS_SOFT, "complex_i", cplx, fixedpt, H5T__conv_complex_i);
    status |= H5T__register_int(H5T_PERS_SOFT, "f_f", floatpt, floatpt, H5T__conv_f_f);
    status |= H5T__register_int(H5T_PERS_SOFT, "f_i", floatpt, fixedpt, H5T__conv_f_i);
    status |= H5T__register_int(H5T_PERS_SOFT, "f_complex", floatpt, cplx, H5T__conv_f_complex);
    status |= H5T__register_int(H5T_PERS_SOFT, "complex_f", cplx, floatpt, H5T__conv_complex_f);
    status |= H5T__register_int(H5T_PERS_SOFT, "complex_complex", cplx, cplx, H5T__conv_complex);
    status |= H5T__register_int(H5T_PERS_SOFT, "s_s", string, string, H5T__conv_s_s);
    status |= H5T__register_int(H5T_PERS_SOFT, "b_b", bitfield, bitfield, H5T__conv_b_b);
    status |= H5T__register_int(H5T_PERS_SOFT, "ibo", fixedpt, fixedpt, H5T__conv_order);
    status |= H5T__register_int(H5T_PERS_SOFT, "ibo(opt)", fixedpt, fixedpt, H5T__conv_order_opt);
    status |= H5T__register_int(H5T_PERS_SOFT, "fbo", floatpt, floatpt, H5T__conv_order);
    status |= H5T__register_int(H5T_PERS_SOFT, "fbo(opt)", floatpt, floatpt, H5T__conv_order_opt);
    status |= H5T__register_int(H5T_PERS_SOFT, "complexbo", cplx, cplx, H5T__conv_order);
    status |= H5T__register_int(H5T_PERS_SOFT, "complexbo(opt)", cplx, cplx, H5T__conv_order_opt);
    status |= H5T__register_int(H5T_PERS_SOFT, "struct(no-opt)", compound, compound, H5T__conv_struct);
    status |= H5T__register_int(H5T_PERS_SOFT, "struct(opt)", compound, compound, H5T__conv_struct_opt);
    status |= H5T__register_int(H5T_PERS_SOFT, "enum", enum_type, enum_type, H5T__conv_enum);
    status |= H5T__register_int(H5T_PERS_SOFT, "enum_i", enum_type, fixedpt, H5T__conv_enum_numeric);
    status |= H5T__register_int(H5T_PERS_SOFT, "enum_f", enum_type, floatpt, H5T__conv_enum_numeric);
    status |= H5T__register_int(H5T_PERS_SOFT, "vlen", vlen, vlen, H5T__conv_vlen);
    status |= H5T__register_int(H5T_PERS_SOFT, "array", array, array, H5T__conv_array);
    status |= H5T__register_int(H5T_PERS_SOFT, "objref", objref, objref, H5T__conv_noop);
    status |= H5T__register_int(H5T_PERS_SOFT, "regref", regref, regref, H5T__conv_noop);
    status |= H5T__register_int(H5T_PERS_SOFT, "ref", ref, ref, H5T__conv_ref);
    status |= H5T__register_int(H5T_PERS_SOFT, "objref_ref", objref, ref, H5T__conv_ref);
    status |= H5T__register_int(H5T_PERS_SOFT, "regref_ref", regref, ref, H5T__conv_ref);
    status |= H5T__register_int(H5T_PERS_SOFT, "complex_array_compat", cplx, array, H5T__conv_complex_compat);
    status |= H5T__register_int(H5T_PERS_SOFT, "array_complex_compat", array, cplx, H5T__conv_complex_compat);
    status |=
        H5T__register_int(H5T_PERS_SOFT, "complex_compound_compat", cplx, compound, H5T__conv_complex_compat);
    status |=
        H5T__register_int(H5T_PERS_SOFT, "compound_complex_compat", compound, cplx, H5T__conv_complex_compat);

    

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt_dbl", native_float, native_double, H5T__conv_float_double);
    status |=
        H5T__register_int(H5T_PERS_HARD, "dbl_flt", native_double, native_float, H5T__conv_double_float);
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt_ldbl", native_float, native_ldouble, H5T__conv_float_ldouble);
    status |=
        H5T__register_int(H5T_PERS_HARD, "dbl_ldbl", native_double, native_ldouble, H5T__conv_double_ldouble);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ldbl_flt", native_ldouble, native_float, H5T__conv_ldouble_float);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ldbl_dbl", native_ldouble, native_double, H5T__conv_ldouble_double);
#ifdef H5_HAVE__FLOAT16
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt16_flt", native_float16, native_float, H5T__conv__Float16_float);
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_dbl", native_float16, native_double,
                                H5T__conv__Float16_double);
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_ldbl", native_float16, native_ldouble,
                                H5T__conv__Float16_ldouble);
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt_flt16", native_float, native_float16, H5T__conv_float__Float16);
    status |= H5T__register_int(H5T_PERS_HARD, "dbl_flt16", native_double, native_float16,
                                H5T__conv_double__Float16);
#ifdef H5T_CONV_INTERNAL_LDOUBLE_FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "ldbl_flt16", native_ldouble, native_float16,
                                H5T__conv_ldouble__Float16);
#endif
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    status |= H5T__register_int(H5T_PERS_HARD, "flt_fcomplex", native_float, native_float_complex,
                                H5T__conv_float_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "flt_dcomplex", native_float, native_double_complex,
                                H5T__conv_float_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "flt_lcomplex", native_float, native_ldouble_complex,
                                H5T__conv_float_lcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "dbl_fcomplex", native_double, native_float_complex,
                                H5T__conv_double_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "dbl_dcomplex", native_double, native_double_complex,
                                H5T__conv_double_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "dbl_lcomplex", native_double, native_ldouble_complex,
                                H5T__conv_double_lcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "ldbl_fcomplex", native_ldouble, native_float_complex,
                                H5T__conv_ldouble_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "ldbl_dcomplex", native_ldouble, native_double_complex,
                                H5T__conv_ldouble_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "ldbl_lcomplex", native_ldouble, native_ldouble_complex,
                                H5T__conv_ldouble_lcomplex);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_fcomplex", native_float16, native_float_complex,
                                H5T__conv__Float16_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_dcomplex", native_float16, native_double_complex,
                                H5T__conv__Float16_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_lcomplex", native_float16, native_ldouble_complex,
                                H5T__conv__Float16_lcomplex);
#endif
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "llong_ullong", native_llong, native_ullong, H5T__conv_llong_ullong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ullong_llong", native_ullong, native_llong, H5T__conv_ullong_llong);
    status |= H5T__register_int(H5T_PERS_HARD, "llong_long", native_llong, native_long, H5T__conv_llong_long);
    status |=
        H5T__register_int(H5T_PERS_HARD, "llong_ulong", native_llong, native_ulong, H5T__conv_llong_ulong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ullong_long", native_ullong, native_long, H5T__conv_ullong_long);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ullong_ulong", native_ullong, native_ulong, H5T__conv_ullong_ulong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "llong_short", native_llong, native_short, H5T__conv_llong_short);
    status |=
        H5T__register_int(H5T_PERS_HARD, "llong_ushort", native_llong, native_ushort, H5T__conv_llong_ushort);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ullong_short", native_ullong, native_short, H5T__conv_ullong_short);
    status |= H5T__register_int(H5T_PERS_HARD, "ullong_ushort", native_ullong, native_ushort,
                                H5T__conv_ullong_ushort);
    status |= H5T__register_int(H5T_PERS_HARD, "llong_int", native_llong, native_int, H5T__conv_llong_int);
    status |= H5T__register_int(H5T_PERS_HARD, "llong_uint", native_llong, native_uint, H5T__conv_llong_uint);
    status |= H5T__register_int(H5T_PERS_HARD, "ullong_int", native_ullong, native_int, H5T__conv_ullong_int);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ullong_uint", native_ullong, native_uint, H5T__conv_ullong_uint);
    status |=
        H5T__register_int(H5T_PERS_HARD, "llong_schar", native_llong, native_schar, H5T__conv_llong_schar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "llong_uchar", native_llong, native_uchar, H5T__conv_llong_uchar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ullong_schar", native_ullong, native_schar, H5T__conv_ullong_schar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ullong_uchar", native_ullong, native_uchar, H5T__conv_ullong_uchar);

    
    status |= H5T__register_int(H5T_PERS_HARD, "long_llong", native_long, native_llong, H5T__conv_long_llong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "long_ullong", native_long, native_ullong, H5T__conv_long_ullong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ulong_llong", native_ulong, native_llong, H5T__conv_ulong_llong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ulong_ullong", native_ulong, native_ullong, H5T__conv_ulong_ullong);
    status |= H5T__register_int(H5T_PERS_HARD, "long_ulong", native_long, native_ulong, H5T__conv_long_ulong);
    status |= H5T__register_int(H5T_PERS_HARD, "ulong_long", native_ulong, native_long, H5T__conv_ulong_long);
    status |= H5T__register_int(H5T_PERS_HARD, "long_short", native_long, native_short, H5T__conv_long_short);
    status |=
        H5T__register_int(H5T_PERS_HARD, "long_ushort", native_long, native_ushort, H5T__conv_long_ushort);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ulong_short", native_ulong, native_short, H5T__conv_ulong_short);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ulong_ushort", native_ulong, native_ushort, H5T__conv_ulong_ushort);
    status |= H5T__register_int(H5T_PERS_HARD, "long_int", native_long, native_int, H5T__conv_long_int);
    status |= H5T__register_int(H5T_PERS_HARD, "long_uint", native_long, native_uint, H5T__conv_long_uint);
    status |= H5T__register_int(H5T_PERS_HARD, "ulong_int", native_ulong, native_int, H5T__conv_ulong_int);
    status |= H5T__register_int(H5T_PERS_HARD, "ulong_uint", native_ulong, native_uint, H5T__conv_ulong_uint);
    status |= H5T__register_int(H5T_PERS_HARD, "long_schar", native_long, native_schar, H5T__conv_long_schar);
    status |= H5T__register_int(H5T_PERS_HARD, "long_uchar", native_long, native_uchar, H5T__conv_long_uchar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ulong_schar", native_ulong, native_schar, H5T__conv_ulong_schar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ulong_uchar", native_ulong, native_uchar, H5T__conv_ulong_uchar);

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "short_llong", native_short, native_llong, H5T__conv_short_llong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "short_ullong", native_short, native_ullong, H5T__conv_short_ullong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ushort_llong", native_ushort, native_llong, H5T__conv_ushort_llong);
    status |= H5T__register_int(H5T_PERS_HARD, "ushort_ullong", native_ushort, native_ullong,
                                H5T__conv_ushort_ullong);
    status |= H5T__register_int(H5T_PERS_HARD, "short_long", native_short, native_long, H5T__conv_short_long);
    status |=
        H5T__register_int(H5T_PERS_HARD, "short_ulong", native_short, native_ulong, H5T__conv_short_ulong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ushort_long", native_ushort, native_long, H5T__conv_ushort_long);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ushort_ulong", native_ushort, native_ulong, H5T__conv_ushort_ulong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "short_ushort", native_short, native_ushort, H5T__conv_short_ushort);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ushort_short", native_ushort, native_short, H5T__conv_ushort_short);
    status |= H5T__register_int(H5T_PERS_HARD, "short_int", native_short, native_int, H5T__conv_short_int);
    status |= H5T__register_int(H5T_PERS_HARD, "short_uint", native_short, native_uint, H5T__conv_short_uint);
    status |= H5T__register_int(H5T_PERS_HARD, "ushort_int", native_ushort, native_int, H5T__conv_ushort_int);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ushort_uint", native_ushort, native_uint, H5T__conv_ushort_uint);
    status |=
        H5T__register_int(H5T_PERS_HARD, "short_schar", native_short, native_schar, H5T__conv_short_schar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "short_uchar", native_short, native_uchar, H5T__conv_short_uchar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ushort_schar", native_ushort, native_schar, H5T__conv_ushort_schar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ushort_uchar", native_ushort, native_uchar, H5T__conv_ushort_uchar);

    
    status |= H5T__register_int(H5T_PERS_HARD, "int_llong", native_int, native_llong, H5T__conv_int_llong);
    status |= H5T__register_int(H5T_PERS_HARD, "int_ullong", native_int, native_ullong, H5T__conv_int_ullong);
    status |= H5T__register_int(H5T_PERS_HARD, "uint_llong", native_uint, native_llong, H5T__conv_uint_llong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "uint_ullong", native_uint, native_ullong, H5T__conv_uint_ullong);
    status |= H5T__register_int(H5T_PERS_HARD, "int_long", native_int, native_long, H5T__conv_int_long);
    status |= H5T__register_int(H5T_PERS_HARD, "int_ulong", native_int, native_ulong, H5T__conv_int_ulong);
    status |= H5T__register_int(H5T_PERS_HARD, "uint_long", native_uint, native_long, H5T__conv_uint_long);
    status |= H5T__register_int(H5T_PERS_HARD, "uint_ulong", native_uint, native_ulong, H5T__conv_uint_ulong);
    status |= H5T__register_int(H5T_PERS_HARD, "int_short", native_int, native_short, H5T__conv_int_short);
    status |= H5T__register_int(H5T_PERS_HARD, "int_ushort", native_int, native_ushort, H5T__conv_int_ushort);
    status |= H5T__register_int(H5T_PERS_HARD, "uint_short", native_uint, native_short, H5T__conv_uint_short);
    status |=
        H5T__register_int(H5T_PERS_HARD, "uint_ushort", native_uint, native_ushort, H5T__conv_uint_ushort);
    status |= H5T__register_int(H5T_PERS_HARD, "int_uint", native_int, native_uint, H5T__conv_int_uint);
    status |= H5T__register_int(H5T_PERS_HARD, "uint_int", native_uint, native_int, H5T__conv_uint_int);
    status |= H5T__register_int(H5T_PERS_HARD, "int_schar", native_int, native_schar, H5T__conv_int_schar);
    status |= H5T__register_int(H5T_PERS_HARD, "int_uchar", native_int, native_uchar, H5T__conv_int_uchar);
    status |= H5T__register_int(H5T_PERS_HARD, "uint_schar", native_uint, native_schar, H5T__conv_uint_schar);
    status |= H5T__register_int(H5T_PERS_HARD, "uint_uchar", native_uint, native_uchar, H5T__conv_uint_uchar);

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "schar_llong", native_schar, native_llong, H5T__conv_schar_llong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "schar_ullong", native_schar, native_ullong, H5T__conv_schar_ullong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "uchar_llong", native_uchar, native_llong, H5T__conv_uchar_llong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "uchar_ullong", native_uchar, native_ullong, H5T__conv_uchar_ullong);
    status |= H5T__register_int(H5T_PERS_HARD, "schar_long", native_schar, native_long, H5T__conv_schar_long);
    status |=
        H5T__register_int(H5T_PERS_HARD, "schar_ulong", native_schar, native_ulong, H5T__conv_schar_ulong);
    status |= H5T__register_int(H5T_PERS_HARD, "uchar_long", native_uchar, native_long, H5T__conv_uchar_long);
    status |=
        H5T__register_int(H5T_PERS_HARD, "uchar_ulong", native_uchar, native_ulong, H5T__conv_uchar_ulong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "schar_short", native_schar, native_short, H5T__conv_schar_short);
    status |=
        H5T__register_int(H5T_PERS_HARD, "schar_ushort", native_schar, native_ushort, H5T__conv_schar_ushort);
    status |=
        H5T__register_int(H5T_PERS_HARD, "uchar_short", native_uchar, native_short, H5T__conv_uchar_short);
    status |=
        H5T__register_int(H5T_PERS_HARD, "uchar_ushort", native_uchar, native_ushort, H5T__conv_uchar_ushort);
    status |= H5T__register_int(H5T_PERS_HARD, "schar_int", native_schar, native_int, H5T__conv_schar_int);
    status |= H5T__register_int(H5T_PERS_HARD, "schar_uint", native_schar, native_uint, H5T__conv_schar_uint);
    status |= H5T__register_int(H5T_PERS_HARD, "uchar_int", native_uchar, native_int, H5T__conv_uchar_int);
    status |= H5T__register_int(H5T_PERS_HARD, "uchar_uint", native_uchar, native_uint, H5T__conv_uchar_uint);
    status |=
        H5T__register_int(H5T_PERS_HARD, "schar_uchar", native_schar, native_uchar, H5T__conv_schar_uchar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "uchar_schar", native_uchar, native_schar, H5T__conv_uchar_schar);

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "schar_flt", native_schar, native_float, H5T__conv_schar_float);
    status |=
        H5T__register_int(H5T_PERS_HARD, "schar_dbl", native_schar, native_double, H5T__conv_schar_double);
    status |=
        H5T__register_int(H5T_PERS_HARD, "schar_ldbl", native_schar, native_ldouble, H5T__conv_schar_ldouble);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "schar_flt16", native_schar, native_float16,
                                H5T__conv_schar__Float16);
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    status |= H5T__register_int(H5T_PERS_HARD, "schar_fcomplex", native_schar, native_float_complex,
                                H5T__conv_schar_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "schar_dcomplex", native_schar, native_double_complex,
                                H5T__conv_schar_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "schar_lcomplex", native_schar, native_ldouble_complex,
                                H5T__conv_schar_lcomplex);
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "uchar_flt", native_uchar, native_float, H5T__conv_uchar_float);
    status |=
        H5T__register_int(H5T_PERS_HARD, "uchar_dbl", native_uchar, native_double, H5T__conv_uchar_double);
    status |=
        H5T__register_int(H5T_PERS_HARD, "uchar_ldbl", native_uchar, native_ldouble, H5T__conv_uchar_ldouble);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "uchar_flt16", native_uchar, native_float16,
                                H5T__conv_uchar__Float16);
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    status |= H5T__register_int(H5T_PERS_HARD, "uchar_fcomplex", native_uchar, native_float_complex,
                                H5T__conv_uchar_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "uchar_dcomplex", native_uchar, native_double_complex,
                                H5T__conv_uchar_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "uchar_lcomplex", native_uchar, native_ldouble_complex,
                                H5T__conv_uchar_lcomplex);
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "short_flt", native_short, native_float, H5T__conv_short_float);
    status |=
        H5T__register_int(H5T_PERS_HARD, "short_dbl", native_short, native_double, H5T__conv_short_double);
    status |=
        H5T__register_int(H5T_PERS_HARD, "short_ldbl", native_short, native_ldouble, H5T__conv_short_ldouble);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "short_flt16", native_short, native_float16,
                                H5T__conv_short__Float16);
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    status |= H5T__register_int(H5T_PERS_HARD, "short_fcomplex", native_short, native_float_complex,
                                H5T__conv_short_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "short_dcomplex", native_short, native_double_complex,
                                H5T__conv_short_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "short_lcomplex", native_short, native_ldouble_complex,
                                H5T__conv_short_lcomplex);
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "ushort_flt", native_ushort, native_float, H5T__conv_ushort_float);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ushort_dbl", native_ushort, native_double, H5T__conv_ushort_double);
    status |= H5T__register_int(H5T_PERS_HARD, "ushort_ldbl", native_ushort, native_ldouble,
                                H5T__conv_ushort_ldouble);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "ushort_flt16", native_ushort, native_float16,
                                H5T__conv_ushort__Float16);
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    status |= H5T__register_int(H5T_PERS_HARD, "ushort_fcomplex", native_ushort, native_float_complex,
                                H5T__conv_ushort_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "ushort_dcomplex", native_ushort, native_double_complex,
                                H5T__conv_ushort_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "ushort_lcomplex", native_ushort, native_ldouble_complex,
                                H5T__conv_ushort_lcomplex);
#endif

    
    status |= H5T__register_int(H5T_PERS_HARD, "int_flt", native_int, native_float, H5T__conv_int_float);
    status |= H5T__register_int(H5T_PERS_HARD, "int_dbl", native_int, native_double, H5T__conv_int_double);
    status |= H5T__register_int(H5T_PERS_HARD, "int_ldbl", native_int, native_ldouble, H5T__conv_int_ldouble);
#ifdef H5_HAVE__FLOAT16
    status |=
        H5T__register_int(H5T_PERS_HARD, "int_flt16", native_int, native_float16, H5T__conv_int__Float16);
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    status |= H5T__register_int(H5T_PERS_HARD, "int_fcomplex", native_int, native_float_complex,
                                H5T__conv_int_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "int_dcomplex", native_int, native_double_complex,
                                H5T__conv_int_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "int_lcomplex", native_int, native_ldouble_complex,
                                H5T__conv_int_lcomplex);
#endif

    
    status |= H5T__register_int(H5T_PERS_HARD, "uint_flt", native_uint, native_float, H5T__conv_uint_float);
    status |= H5T__register_int(H5T_PERS_HARD, "uint_dbl", native_uint, native_double, H5T__conv_uint_double);
    status |=
        H5T__register_int(H5T_PERS_HARD, "uint_ldbl", native_uint, native_ldouble, H5T__conv_uint_ldouble);
#ifdef H5_HAVE__FLOAT16
    status |=
        H5T__register_int(H5T_PERS_HARD, "uint_flt16", native_uint, native_float16, H5T__conv_uint__Float16);
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    status |= H5T__register_int(H5T_PERS_HARD, "uint_fcomplex", native_uint, native_float_complex,
                                H5T__conv_uint_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "uint_dcomplex", native_uint, native_double_complex,
                                H5T__conv_uint_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "uint_lcomplex", native_uint, native_ldouble_complex,
                                H5T__conv_uint_lcomplex);
#endif

    
    status |= H5T__register_int(H5T_PERS_HARD, "long_flt", native_long, native_float, H5T__conv_long_float);
    status |= H5T__register_int(H5T_PERS_HARD, "long_dbl", native_long, native_double, H5T__conv_long_double);
    status |=
        H5T__register_int(H5T_PERS_HARD, "long_ldbl", native_long, native_ldouble, H5T__conv_long_ldouble);
#ifdef H5_HAVE__FLOAT16
    status |=
        H5T__register_int(H5T_PERS_HARD, "long_flt16", native_long, native_float16, H5T__conv_long__Float16);
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    status |= H5T__register_int(H5T_PERS_HARD, "long_fcomplex", native_long, native_float_complex,
                                H5T__conv_long_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "long_dcomplex", native_long, native_double_complex,
                                H5T__conv_long_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "long_lcomplex", native_long, native_ldouble_complex,
                                H5T__conv_long_lcomplex);
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "ulong_flt", native_ulong, native_float, H5T__conv_ulong_float);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ulong_dbl", native_ulong, native_double, H5T__conv_ulong_double);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ulong_ldbl", native_ulong, native_ldouble, H5T__conv_ulong_ldouble);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "ulong_flt16", native_ulong, native_float16,
                                H5T__conv_ulong__Float16);
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    status |= H5T__register_int(H5T_PERS_HARD, "ulong_fcomplex", native_ulong, native_float_complex,
                                H5T__conv_ulong_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "ulong_dcomplex", native_ulong, native_double_complex,
                                H5T__conv_ulong_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "ulong_lcomplex", native_ulong, native_ldouble_complex,
                                H5T__conv_ulong_lcomplex);
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "llong_flt", native_llong, native_float, H5T__conv_llong_float);
    status |=
        H5T__register_int(H5T_PERS_HARD, "llong_dbl", native_llong, native_double, H5T__conv_llong_double);
#ifdef H5T_CONV_INTERNAL_LLONG_LDOUBLE
    status |=
        H5T__register_int(H5T_PERS_HARD, "llong_ldbl", native_llong, native_ldouble, H5T__conv_llong_ldouble);
#endif 
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "llong_flt16", native_llong, native_float16,
                                H5T__conv_llong__Float16);
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    status |= H5T__register_int(H5T_PERS_HARD, "llong_fcomplex", native_llong, native_float_complex,
                                H5T__conv_llong_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "llong_dcomplex", native_llong, native_double_complex,
                                H5T__conv_llong_dcomplex);
#ifdef H5T_CONV_INTERNAL_LLONG_LDOUBLE
    status |= H5T__register_int(H5T_PERS_HARD, "llong_lcomplex", native_llong, native_ldouble_complex,
                                H5T__conv_llong_lcomplex);
#endif
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "ullong_flt", native_ullong, native_float, H5T__conv_ullong_float);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ullong_dbl", native_ullong, native_double, H5T__conv_ullong_double);
#ifdef H5T_CONV_INTERNAL_ULLONG_LDOUBLE
    status |= H5T__register_int(H5T_PERS_HARD, "ullong_ldbl", native_ullong, native_ldouble,
                                H5T__conv_ullong_ldouble);
#endif 
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "ullong_flt16", native_ullong, native_float16,
                                H5T__conv_ullong__Float16);
#endif
#ifdef H5_HAVE_COMPLEX_NUMBERS
    status |= H5T__register_int(H5T_PERS_HARD, "ullong_fcomplex", native_ullong, native_float_complex,
                                H5T__conv_ullong_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "ullong_dcomplex", native_ullong, native_double_complex,
                                H5T__conv_ullong_dcomplex);
#ifdef H5T_CONV_INTERNAL_ULLONG_LDOUBLE
    status |= H5T__register_int(H5T_PERS_HARD, "ullong_lcomplex", native_ullong, native_ldouble_complex,
                                H5T__conv_ullong_lcomplex);
#endif
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt_schar", native_float, native_schar, H5T__conv_float_schar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "dbl_schar", native_double, native_schar, H5T__conv_double_schar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ldbl_schar", native_ldouble, native_schar, H5T__conv_ldouble_schar);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_schar", native_float16, native_schar,
                                H5T__conv__Float16_schar);
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt_uchar", native_float, native_uchar, H5T__conv_float_uchar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "dbl_uchar", native_double, native_uchar, H5T__conv_double_uchar);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ldbl_uchar", native_ldouble, native_uchar, H5T__conv_ldouble_uchar);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_uchar", native_float16, native_uchar,
                                H5T__conv__Float16_uchar);
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt_short", native_float, native_short, H5T__conv_float_short);
    status |=
        H5T__register_int(H5T_PERS_HARD, "dbl_short", native_double, native_short, H5T__conv_double_short);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ldbl_short", native_ldouble, native_short, H5T__conv_ldouble_short);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_short", native_float16, native_short,
                                H5T__conv__Float16_short);
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt_ushort", native_float, native_ushort, H5T__conv_float_ushort);
    status |=
        H5T__register_int(H5T_PERS_HARD, "dbl_ushort", native_double, native_ushort, H5T__conv_double_ushort);
    status |= H5T__register_int(H5T_PERS_HARD, "ldbl_ushort", native_ldouble, native_ushort,
                                H5T__conv_ldouble_ushort);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_ushort", native_float16, native_ushort,
                                H5T__conv__Float16_ushort);
#endif

    
    status |= H5T__register_int(H5T_PERS_HARD, "flt_int", native_float, native_int, H5T__conv_float_int);
    status |= H5T__register_int(H5T_PERS_HARD, "dbl_int", native_double, native_int, H5T__conv_double_int);
    status |= H5T__register_int(H5T_PERS_HARD, "ldbl_int", native_ldouble, native_int, H5T__conv_ldouble_int);
#ifdef H5_HAVE__FLOAT16
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt16_int", native_float16, native_int, H5T__conv__Float16_int);
#endif

    
    status |= H5T__register_int(H5T_PERS_HARD, "flt_uint", native_float, native_uint, H5T__conv_float_uint);
    status |= H5T__register_int(H5T_PERS_HARD, "dbl_uint", native_double, native_uint, H5T__conv_double_uint);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ldbl_uint", native_ldouble, native_uint, H5T__conv_ldouble_uint);
#ifdef H5_HAVE__FLOAT16
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt16_uint", native_float16, native_uint, H5T__conv__Float16_uint);
#endif

    
    status |= H5T__register_int(H5T_PERS_HARD, "flt_long", native_float, native_long, H5T__conv_float_long);
    status |= H5T__register_int(H5T_PERS_HARD, "dbl_long", native_double, native_long, H5T__conv_double_long);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ldbl_long", native_ldouble, native_long, H5T__conv_ldouble_long);
#ifdef H5_HAVE__FLOAT16
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt16_long", native_float16, native_long, H5T__conv__Float16_long);
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt_ulong", native_float, native_ulong, H5T__conv_float_ulong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "dbl_ulong", native_double, native_ulong, H5T__conv_double_ulong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "ldbl_ulong", native_ldouble, native_ulong, H5T__conv_ldouble_ulong);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_ulong", native_float16, native_ulong,
                                H5T__conv__Float16_ulong);
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt_llong", native_float, native_llong, H5T__conv_float_llong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "dbl_llong", native_double, native_llong, H5T__conv_double_llong);
#ifdef H5T_CONV_INTERNAL_LDOUBLE_LLONG
    status |=
        H5T__register_int(H5T_PERS_HARD, "ldbl_llong", native_ldouble, native_llong, H5T__conv_ldouble_llong);
#endif 
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_llong", native_float16, native_llong,
                                H5T__conv__Float16_llong);
#endif

    
    status |=
        H5T__register_int(H5T_PERS_HARD, "flt_ullong", native_float, native_ullong, H5T__conv_float_ullong);
    status |=
        H5T__register_int(H5T_PERS_HARD, "dbl_ullong", native_double, native_ullong, H5T__conv_double_ullong);
#ifdef H5T_CONV_INTERNAL_LDOUBLE_ULLONG
    status |= H5T__register_int(H5T_PERS_HARD, "ldbl_ullong", native_ldouble, native_ullong,
                                H5T__conv_ldouble_ullong);
#endif 
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "flt16_ullong", native_float16, native_ullong,
                                H5T__conv__Float16_ullong);
#endif

#ifdef H5_HAVE_COMPLEX_NUMBERS
    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_schar", native_float_complex, native_schar,
                                H5T__conv_fcomplex_schar);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_schar", native_double_complex, native_schar,
                                H5T__conv_dcomplex_schar);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_schar", native_ldouble_complex, native_schar,
                                H5T__conv_lcomplex_schar);

    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_uchar", native_float_complex, native_uchar,
                                H5T__conv_fcomplex_uchar);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_uchar", native_double_complex, native_uchar,
                                H5T__conv_dcomplex_uchar);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_uchar", native_ldouble_complex, native_uchar,
                                H5T__conv_lcomplex_uchar);

    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_short", native_float_complex, native_short,
                                H5T__conv_fcomplex_short);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_short", native_double_complex, native_short,
                                H5T__conv_dcomplex_short);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_short", native_ldouble_complex, native_short,
                                H5T__conv_lcomplex_short);

    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_ushort", native_float_complex, native_ushort,
                                H5T__conv_fcomplex_ushort);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_ushort", native_double_complex, native_ushort,
                                H5T__conv_dcomplex_ushort);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_ushort", native_ldouble_complex, native_ushort,
                                H5T__conv_lcomplex_ushort);

    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_int", native_float_complex, native_int,
                                H5T__conv_fcomplex_int);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_int", native_double_complex, native_int,
                                H5T__conv_dcomplex_int);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_int", native_ldouble_complex, native_int,
                                H5T__conv_lcomplex_int);

    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_uint", native_float_complex, native_uint,
                                H5T__conv_fcomplex_uint);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_uint", native_double_complex, native_uint,
                                H5T__conv_dcomplex_uint);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_uint", native_ldouble_complex, native_uint,
                                H5T__conv_lcomplex_uint);

    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_long", native_float_complex, native_long,
                                H5T__conv_fcomplex_long);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_long", native_double_complex, native_long,
                                H5T__conv_dcomplex_long);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_long", native_ldouble_complex, native_long,
                                H5T__conv_lcomplex_long);

    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_ulong", native_float_complex, native_ulong,
                                H5T__conv_fcomplex_ulong);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_ulong", native_double_complex, native_ulong,
                                H5T__conv_dcomplex_ulong);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_ulong", native_ldouble_complex, native_ulong,
                                H5T__conv_lcomplex_ulong);

    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_llong", native_float_complex, native_llong,
                                H5T__conv_fcomplex_llong);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_llong", native_double_complex, native_llong,
                                H5T__conv_dcomplex_llong);
#ifdef H5T_CONV_INTERNAL_LDOUBLE_LLONG
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_llong", native_ldouble_complex, native_llong,
                                H5T__conv_lcomplex_llong);
#endif

    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_ullong", native_float_complex, native_ullong,
                                H5T__conv_fcomplex_ullong);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_ullong", native_double_complex, native_ullong,
                                H5T__conv_dcomplex_ullong);
#ifdef H5T_CONV_INTERNAL_LDOUBLE_ULLONG
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_ullong", native_ldouble_complex, native_ullong,
                                H5T__conv_lcomplex_ullong);
#endif

    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_flt", native_float_complex, native_float,
                                H5T__conv_fcomplex_float);
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_dbl", native_float_complex, native_double,
                                H5T__conv_fcomplex_double);
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_ldbl", native_float_complex, native_ldouble,
                                H5T__conv_fcomplex_ldouble);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_flt16", native_float_complex, native_float16,
                                H5T__conv_fcomplex__Float16);
#endif
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_flt", native_double_complex, native_float,
                                H5T__conv_dcomplex_float);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_dbl", native_double_complex, native_double,
                                H5T__conv_dcomplex_double);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_ldbl", native_double_complex, native_ldouble,
                                H5T__conv_dcomplex_ldouble);
#ifdef H5_HAVE__FLOAT16
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_flt16", native_double_complex, native_float16,
                                H5T__conv_dcomplex__Float16);
#endif
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_flt", native_ldouble_complex, native_float,
                                H5T__conv_lcomplex_float);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_dbl", native_ldouble_complex, native_double,
                                H5T__conv_lcomplex_double);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_ldbl", native_ldouble_complex, native_ldouble,
                                H5T__conv_lcomplex_ldouble);
#if defined(H5_HAVE__FLOAT16) && defined(H5T_CONV_INTERNAL_LDOUBLE_FLOAT16)
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_flt16", native_ldouble_complex, native_float16,
                                H5T__conv_lcomplex__Float16);
#endif

    
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_dcomplex", native_float_complex,
                                native_double_complex, H5T__conv_fcomplex_dcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "fcomplex_lcomplex", native_float_complex,
                                native_ldouble_complex, H5T__conv_fcomplex_lcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_fcomplex", native_double_complex,
                                native_float_complex, H5T__conv_dcomplex_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "dcomplex_lcomplex", native_double_complex,
                                native_ldouble_complex, H5T__conv_dcomplex_lcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_fcomplex", native_ldouble_complex,
                                native_float_complex, H5T__conv_lcomplex_fcomplex);
    status |= H5T__register_int(H5T_PERS_HARD, "lcomplex_dcomplex", native_ldouble_complex,
                                native_double_complex, H5T__conv_lcomplex_dcomplex);
#endif

    
    status |= H5T__register_int(H5T_PERS_HARD, "no-op", native_int, native_int, H5T__conv_noop);

    
    status |= H5T__init_inf();

    if (status < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to register conversion function(s)");

    

    
    if (H5P_LST_DATATYPE_CREATE_ID_g == (-1)) {
        
        assert(H5P_CLS_DATATYPE_CREATE_g != NULL);

        
        if ((H5P_LST_DATATYPE_CREATE_ID_g = H5P_create_id(H5P_CLS_DATATYPE_CREATE_g, false)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "can't insert property into class");
    } 

    
    H5T_top_package_initialize_s = true;

done:
    
    if (compound != NULL)
        (void)H5T_close_real(compound);
    if (enum_type != NULL)
        (void)H5T_close_real(enum_type);
    if (vlen != NULL)
        (void)H5T_close_real(vlen);
    if (array != NULL)
        (void)H5T_close_real(array);
    if (cplx != NULL)
        (void)H5T_close_real(cplx);

    
    if (ret_value < 0) {
        if (dt) {
            
            if (copied_dtype)
                (void)H5T_close_real(dt);
            else {
                if (dt->shared->owned_vol_obj && H5VL_free_object(dt->shared->owned_vol_obj) < 0)
                    HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "unable to close owned VOL object");
                dt->shared = H5FL_FREE(H5T_shared_t, dt->shared);
                dt         = H5FL_FREE(H5T_t, dt);
            } 
        }     
    }         

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5T__unlock_cb(void *_dt, hid_t H5_ATTR_UNUSED id, void *_udata)
{
    H5T_t *dt = (H5T_t *)_dt;
    int   *n  = (int *)_udata;

    FUNC_ENTER_PACKAGE_NOERR

    assert(dt);

    if (dt->shared && (H5T_STATE_IMMUTABLE == dt->shared->state)) {
        dt->shared->state = H5T_STATE_RDONLY;
        (*n)++;
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

int
H5T_top_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5T_top_package_initialize_s) {
        
        if (H5T_g.path) {
            H5T_conv_ctx_t conv_ctx = {0};

            conv_ctx.u.free.src_type_id = H5I_INVALID_HID;
            conv_ctx.u.free.dst_type_id = H5I_INVALID_HID;

            for (int i = 0; i < H5T_g.npaths; i++) {
                H5T_path_t *path = H5T_g.path[i];

                (void)H5T__path_free(path, &conv_ctx);

                H5T_g.path[i] = NULL;
            } 

            
            H5T_g.path   = (H5T_path_t **)H5MM_xfree(H5T_g.path);
            H5T_g.npaths = 0;
            H5T_g.apaths = 0;
            H5T_g.soft   = (H5T_soft_t *)H5MM_xfree(H5T_g.soft);
            H5T_g.nsoft  = 0;
            H5T_g.asoft  = 0;

            n++;
        } 

        
        
        
        H5I_iterate(H5I_DATATYPE, H5T__unlock_cb, &n, false);

        
        if (H5I_nmembers(H5I_DATATYPE) > 0) {
            (void)H5I_clear_type(H5I_DATATYPE, false, false);
            n++; 
        }        

        
        if (H5T_IEEE_F32BE_g > 0) {
            H5T_IEEE_F16BE_g = H5I_INVALID_HID;
            H5T_IEEE_F16LE_g = H5I_INVALID_HID;
            H5T_IEEE_F32BE_g = H5I_INVALID_HID;
            H5T_IEEE_F32LE_g = H5I_INVALID_HID;
            H5T_IEEE_F64BE_g = H5I_INVALID_HID;
            H5T_IEEE_F64LE_g = H5I_INVALID_HID;

            H5T_FLOAT_BFLOAT16BE_g = H5I_INVALID_HID;
            H5T_FLOAT_BFLOAT16LE_g = H5I_INVALID_HID;
            H5T_FLOAT_F8E4M3_g     = H5I_INVALID_HID;
            H5T_FLOAT_F8E5M2_g     = H5I_INVALID_HID;

            H5T_COMPLEX_IEEE_F16BE_g = H5I_INVALID_HID;
            H5T_COMPLEX_IEEE_F16LE_g = H5I_INVALID_HID;
            H5T_COMPLEX_IEEE_F32BE_g = H5I_INVALID_HID;
            H5T_COMPLEX_IEEE_F32LE_g = H5I_INVALID_HID;
            H5T_COMPLEX_IEEE_F64BE_g = H5I_INVALID_HID;
            H5T_COMPLEX_IEEE_F64LE_g = H5I_INVALID_HID;

            H5T_STD_I8BE_g        = H5I_INVALID_HID;
            H5T_STD_I8LE_g        = H5I_INVALID_HID;
            H5T_STD_I16BE_g       = H5I_INVALID_HID;
            H5T_STD_I16LE_g       = H5I_INVALID_HID;
            H5T_STD_I32BE_g       = H5I_INVALID_HID;
            H5T_STD_I32LE_g       = H5I_INVALID_HID;
            H5T_STD_I64BE_g       = H5I_INVALID_HID;
            H5T_STD_I64LE_g       = H5I_INVALID_HID;
            H5T_STD_U8BE_g        = H5I_INVALID_HID;
            H5T_STD_U8LE_g        = H5I_INVALID_HID;
            H5T_STD_U16BE_g       = H5I_INVALID_HID;
            H5T_STD_U16LE_g       = H5I_INVALID_HID;
            H5T_STD_U32BE_g       = H5I_INVALID_HID;
            H5T_STD_U32LE_g       = H5I_INVALID_HID;
            H5T_STD_U64BE_g       = H5I_INVALID_HID;
            H5T_STD_U64LE_g       = H5I_INVALID_HID;
            H5T_STD_B8BE_g        = H5I_INVALID_HID;
            H5T_STD_B8LE_g        = H5I_INVALID_HID;
            H5T_STD_B16BE_g       = H5I_INVALID_HID;
            H5T_STD_B16LE_g       = H5I_INVALID_HID;
            H5T_STD_B32BE_g       = H5I_INVALID_HID;
            H5T_STD_B32LE_g       = H5I_INVALID_HID;
            H5T_STD_B64BE_g       = H5I_INVALID_HID;
            H5T_STD_B64LE_g       = H5I_INVALID_HID;
            H5T_STD_REF_OBJ_g     = H5I_INVALID_HID;
            H5T_STD_REF_DSETREG_g = H5I_INVALID_HID;
            H5T_STD_REF_g         = H5I_INVALID_HID;

            H5T_UNIX_D32BE_g = H5I_INVALID_HID;
            H5T_UNIX_D32LE_g = H5I_INVALID_HID;
            H5T_UNIX_D64BE_g = H5I_INVALID_HID;
            H5T_UNIX_D64LE_g = H5I_INVALID_HID;

            H5T_C_S1_g = H5I_INVALID_HID;

            H5T_FORTRAN_S1_g = H5I_INVALID_HID;

            H5T_NATIVE_SCHAR_g           = H5I_INVALID_HID;
            H5T_NATIVE_UCHAR_g           = H5I_INVALID_HID;
            H5T_NATIVE_SHORT_g           = H5I_INVALID_HID;
            H5T_NATIVE_USHORT_g          = H5I_INVALID_HID;
            H5T_NATIVE_INT_g             = H5I_INVALID_HID;
            H5T_NATIVE_UINT_g            = H5I_INVALID_HID;
            H5T_NATIVE_LONG_g            = H5I_INVALID_HID;
            H5T_NATIVE_ULONG_g           = H5I_INVALID_HID;
            H5T_NATIVE_LLONG_g           = H5I_INVALID_HID;
            H5T_NATIVE_ULLONG_g          = H5I_INVALID_HID;
            H5T_NATIVE_FLOAT16_g         = H5I_INVALID_HID;
            H5T_NATIVE_FLOAT_g           = H5I_INVALID_HID;
            H5T_NATIVE_DOUBLE_g          = H5I_INVALID_HID;
            H5T_NATIVE_LDOUBLE_g         = H5I_INVALID_HID;
            H5T_NATIVE_FLOAT_COMPLEX_g   = H5I_INVALID_HID;
            H5T_NATIVE_DOUBLE_COMPLEX_g  = H5I_INVALID_HID;
            H5T_NATIVE_LDOUBLE_COMPLEX_g = H5I_INVALID_HID;
            H5T_NATIVE_B8_g              = H5I_INVALID_HID;
            H5T_NATIVE_B16_g             = H5I_INVALID_HID;
            H5T_NATIVE_B32_g             = H5I_INVALID_HID;
            H5T_NATIVE_B64_g             = H5I_INVALID_HID;
            H5T_NATIVE_OPAQUE_g          = H5I_INVALID_HID;
            H5T_NATIVE_HADDR_g           = H5I_INVALID_HID;
            H5T_NATIVE_HSIZE_g           = H5I_INVALID_HID;
            H5T_NATIVE_HSSIZE_g          = H5I_INVALID_HID;
            H5T_NATIVE_HERR_g            = H5I_INVALID_HID;
            H5T_NATIVE_HBOOL_g           = H5I_INVALID_HID;

            H5T_NATIVE_INT8_g        = H5I_INVALID_HID;
            H5T_NATIVE_UINT8_g       = H5I_INVALID_HID;
            H5T_NATIVE_INT_LEAST8_g  = H5I_INVALID_HID;
            H5T_NATIVE_UINT_LEAST8_g = H5I_INVALID_HID;
            H5T_NATIVE_INT_FAST8_g   = H5I_INVALID_HID;
            H5T_NATIVE_UINT_FAST8_g  = H5I_INVALID_HID;

            H5T_NATIVE_INT16_g        = H5I_INVALID_HID;
            H5T_NATIVE_UINT16_g       = H5I_INVALID_HID;
            H5T_NATIVE_INT_LEAST16_g  = H5I_INVALID_HID;
            H5T_NATIVE_UINT_LEAST16_g = H5I_INVALID_HID;
            H5T_NATIVE_INT_FAST16_g   = H5I_INVALID_HID;
            H5T_NATIVE_UINT_FAST16_g  = H5I_INVALID_HID;

            H5T_NATIVE_INT32_g        = H5I_INVALID_HID;
            H5T_NATIVE_UINT32_g       = H5I_INVALID_HID;
            H5T_NATIVE_INT_LEAST32_g  = H5I_INVALID_HID;
            H5T_NATIVE_UINT_LEAST32_g = H5I_INVALID_HID;
            H5T_NATIVE_INT_FAST32_g   = H5I_INVALID_HID;
            H5T_NATIVE_UINT_FAST32_g  = H5I_INVALID_HID;

            H5T_NATIVE_INT64_g        = H5I_INVALID_HID;
            H5T_NATIVE_UINT64_g       = H5I_INVALID_HID;
            H5T_NATIVE_INT_LEAST64_g  = H5I_INVALID_HID;
            H5T_NATIVE_UINT_LEAST64_g = H5I_INVALID_HID;
            H5T_NATIVE_INT_FAST64_g   = H5I_INVALID_HID;
            H5T_NATIVE_UINT_FAST64_g  = H5I_INVALID_HID;

            n++;
        } 

        
        if (0 == n)
            H5T_top_package_initialize_s = false;
    } 

    FUNC_LEAVE_NOAPI(n)
} 

int
H5T_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        
        assert(0 == H5I_nmembers(H5I_DATATYPE));
        assert(false == H5T_top_package_initialize_s);

        
        n += (H5I_dec_type_ref(H5I_DATATYPE) > 0);

        
        if (0 == n)
            H5_PKG_INIT_VAR = false;
    } 

    FUNC_LEAVE_NOAPI(n)
} 

static herr_t
H5T__close_cb(void *_dt, void **request)
{
    H5T_t *dt = (H5T_t *)_dt;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dt);

    
    if (NULL != dt->vol_obj) {
        
        if (H5VL_datatype_close(dt->vol_obj, H5P_DATASET_XFER_DEFAULT, request) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CLOSEERROR, FAIL, "unable to close datatype");

        
        if (H5VL_free_object(dt->vol_obj) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTDEC, FAIL, "unable to free VOL object");
        dt->vol_obj = NULL;
    } 

    
    if (H5T_close(dt) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CLOSEERROR, FAIL, "unable to close datatype");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Tcreate(H5T_class_t type, size_t size)
{
    H5T_t *dt = NULL; 
    hid_t  ret_value; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (size <= 0 && size != H5T_VARIABLE)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "size must be positive");

    
    if (NULL == (dt = H5T__create(type, size)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, H5I_INVALID_HID, "unable to create type");

    
    if ((ret_value = H5I_register(H5I_DATATYPE, dt, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register datatype ID");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Tcopy(hid_t obj_id)
{
    H5T_t *dt        = NULL;            
    H5T_t *new_dt    = NULL;            
    hid_t  dset_tid  = H5I_INVALID_HID; 
    hid_t  ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    switch (H5I_get_type(obj_id)) {
        case H5I_DATATYPE:
            
            if (NULL == (dt = (H5T_t *)H5I_object(obj_id)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "obj_id is not a datatype ID");
            break;

        case H5I_DATASET: {
            H5VL_object_t          *vol_obj;     
            H5VL_dataset_get_args_t vol_cb_args; 

            
            if (NULL == (vol_obj = (H5VL_object_t *)H5I_object_verify(obj_id, H5I_DATASET)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "type_id is not a dataset ID");

            
            vol_cb_args.op_type               = H5VL_DATASET_GET_TYPE;
            vol_cb_args.args.get_type.type_id = H5I_INVALID_HID;

            
            if (H5VL_dataset_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, H5I_INVALID_HID,
                            "unable to get datatype from the dataset");
            dset_tid = vol_cb_args.args.get_type.type_id;

            
            if (NULL == (dt = (H5T_t *)H5I_object(dset_tid)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, H5I_INVALID_HID,
                            "received invalid datatype from the dataset");
        } break;

        case H5I_UNINIT:
        case H5I_BADID:
        case H5I_FILE:
        case H5I_GROUP:
        case H5I_DATASPACE:
        case H5I_ATTR:
        case H5I_MAP:
        case H5I_VFL:
        case H5I_VOL:
        case H5I_GENPROP_CLS:
        case H5I_GENPROP_LST:
        case H5I_ERROR_CLASS:
        case H5I_ERROR_MSG:
        case H5I_ERROR_STACK:
        case H5I_SPACE_SEL_ITER:
        case H5I_EVENTSET:
        case H5I_NTYPES:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a datatype or dataset");
    } 

    
    if (NULL == (new_dt = H5T_copy(dt, H5T_COPY_TRANSIENT)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, H5I_INVALID_HID, "unable to copy");

    
    if ((ret_value = H5I_register(H5I_DATATYPE, new_dt, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register datatype atom");

done:

    
    if (dset_tid != H5I_INVALID_HID)
        if (H5I_dec_app_ref(dset_tid) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_BADID, FAIL, "problem freeing temporary dataset type ID");

    
    if (H5I_INVALID_HID == ret_value)
        if (new_dt && H5T_close_real(new_dt) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, H5I_INVALID_HID, "unable to release datatype info");

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tclose(hid_t type_id)
{
    H5T_t *dt;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_STATE_IMMUTABLE == dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "immutable datatype");

    
    if (H5I_dec_app_ref(type_id) < 0)
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "problem freeing id");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tclose_async(const char *app_file, const char *app_func, unsigned app_line, hid_t type_id, hid_t es_id)
{
    H5T_t            *dt;                          
    void             *token     = NULL;            
    void            **token_ptr = H5_REQUEST_NULL; 
    H5VL_object_t    *vol_obj   = NULL;            
    H5VL_connector_t *connector = NULL;            
    herr_t            ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_STATE_IMMUTABLE == dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "immutable datatype");

    
    if (NULL == (vol_obj = H5VL_vol_object(type_id)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "can't get VOL object for dataset");

    
    if (H5ES_NONE != es_id) {
        
        connector = H5VL_OBJ_CONNECTOR(vol_obj);
        H5VL_conn_inc_rc(connector);

        
        token_ptr = &token;
    } 

    
    if (H5I_dec_app_ref_async(type_id, token_ptr) < 0)
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "problem freeing id");

    
    if (NULL != token)
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE5(__func__, "*s*sIuii", app_file, app_func, app_line, type_id, es_id)) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    if (connector && H5VL_conn_dec_rc(connector) < 0)
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement ref count on connector");

    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5Tequal(hid_t type1_id, hid_t type2_id)
{
    const H5T_t *dt1;       
    const H5T_t *dt2;       
    htri_t       ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt1 = (H5T_t *)H5I_object_verify(type1_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (NULL == (dt2 = (H5T_t *)H5I_object_verify(type2_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");

    ret_value = (0 == H5T_cmp(dt1, dt2, false)) ? true : false;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tlock(hid_t type_id)
{
    H5T_t *dt;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_STATE_NAMED == dt->shared->state || H5T_STATE_OPEN == dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to lock named datatype");

    if (H5T_lock(dt, true) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to lock transient datatype");

done:
    FUNC_LEAVE_API(ret_value)
} 

H5T_class_t
H5Tget_class(hid_t type_id)
{
    H5T_t      *dt;        
    H5T_class_t ret_value; 

    FUNC_ENTER_API(H5T_NO_CLASS)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5T_NO_CLASS, "not a datatype");

    
    ret_value = H5T_get_class(dt, false);

done:
    FUNC_LEAVE_API(ret_value)
} 

H5T_class_t
H5T_get_class(const H5T_t *dt, htri_t internal)
{
    H5T_class_t ret_value = H5T_NO_CLASS; 

    FUNC_ENTER_NOAPI(H5T_NO_CLASS)

    assert(dt);

    
    if (internal) {
        ret_value = dt->shared->type;
    }
    else {
        if (H5T_IS_VL_STRING(dt->shared))
            ret_value = H5T_STRING;
        else
            ret_value = dt->shared->type;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5Tdetect_class(hid_t type, H5T_class_t cls)
{
    H5T_t *dt;        
    htri_t ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5T_NO_CLASS, "not a datatype");
    if (!(cls > H5T_NO_CLASS && cls < H5T_NCLASSES))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5T_NO_CLASS, "not a datatype class");

    
    if ((ret_value = H5T_detect_class(dt, cls, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, H5T_NO_CLASS, "can't get datatype class");

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5T_detect_class(const H5T_t *dt, H5T_class_t cls, bool from_api)
{
    unsigned i;
    htri_t   ret_value = false; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(dt);
    assert(cls > H5T_NO_CLASS && cls < H5T_NCLASSES);

    
    
    if (from_api && H5T_IS_VL_STRING(dt->shared))
        HGOTO_DONE(H5T_STRING == cls);

    
    if (dt->shared->type == cls)
        HGOTO_DONE(true);

    
    switch (dt->shared->type) {
        case H5T_COMPOUND:
            for (i = 0; i < dt->shared->u.compnd.nmembs; i++) {
                htri_t nested_ret; 

                
                if (dt->shared->u.compnd.memb[i].type->shared->type == cls)
                    HGOTO_DONE(true);

                
                if (H5T_IS_COMPOSITE(dt->shared->u.compnd.memb[i].type->shared->type))
                    if ((nested_ret = H5T_detect_class(dt->shared->u.compnd.memb[i].type, cls, from_api)) !=
                        false)
                        HGOTO_DONE(nested_ret);
            } 
            break;

        case H5T_ARRAY:
        case H5T_VLEN:
        case H5T_ENUM:
            HGOTO_DONE(H5T_detect_class(dt->shared->parent, cls, from_api));
            break;
        case H5T_COMPLEX:
            HGOTO_DONE(H5T_detect_class(dt->shared->parent, cls, from_api));
            break;
        case H5T_NO_CLASS:
        case H5T_INTEGER:
        case H5T_FLOAT:
        case H5T_TIME:
        case H5T_STRING:
        case H5T_BITFIELD:
        case H5T_OPAQUE:
        case H5T_REFERENCE:
        case H5T_NCLASSES:
        default:
            break;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5Tis_variable_str(hid_t dtype_id)
{
    H5T_t *dt;        
    htri_t ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(dtype_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");

    
    if ((ret_value = H5T_is_variable_str(dt)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "can't determine if datatype is VL-string");

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5T_is_variable_str(const H5T_t *dt)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    FUNC_LEAVE_NOAPI(H5T_IS_VL_STRING(dt->shared))
} 

size_t
H5Tget_size(hid_t type_id)
{
    H5T_t *dt;        
    size_t ret_value; 

    FUNC_ENTER_API(0)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, 0, "not a datatype");

    
    ret_value = H5T_GET_SIZE(dt);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tset_size(hid_t type_id, size_t size)
{
    H5T_t *dt;                  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_STATE_TRANSIENT != dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTINIT, FAIL, "datatype is read-only");
    if (size <= 0 && size != H5T_VARIABLE)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "size must be positive");
    if (size == H5T_VARIABLE && !H5T_IS_STRING(dt->shared))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "only strings may be variable length");
    if (H5T_ENUM == dt->shared->type && dt->shared->u.enumer.nmembs > 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "operation not allowed after members are defined");
    if (H5T_ARRAY == dt->shared->type || H5T_REFERENCE == dt->shared->type || H5T_COMPLEX == dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "operation not defined for this datatype");
    if (H5T_VLEN == dt->shared->type && H5T_VLEN_STRING != dt->shared->u.vlen.type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "operation not defined for this datatype");

    
    if (H5T__set_size(dt, size) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to set size for datatype");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Tget_super(hid_t type)
{
    H5T_t *dt;                          
    H5T_t *super     = NULL;            
    hid_t  ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    if (NULL == (dt = (H5T_t *)H5I_object_verify(type, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a datatype");
    if (NULL == (super = H5T_get_super(dt)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, H5I_INVALID_HID, "not a datatype");
    if ((ret_value = H5I_register(H5I_DATATYPE, super, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register parent datatype");

done:
    if (H5I_INVALID_HID == ret_value)
        if (super && H5T_close_real(super) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, H5I_INVALID_HID,
                        "unable to release super datatype info")

    FUNC_LEAVE_API(ret_value)
} 

H5T_t *
H5T_get_super(const H5T_t *dt)
{
    H5T_t *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    assert(dt);

    if (!dt->shared->parent)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "not a derived data type");
    if (NULL == (ret_value = H5T_copy(dt->shared->parent, H5T_COPY_ALL)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to copy parent data type");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__register_int(H5T_pers_t pers, const char *name, H5T_t *src, H5T_t *dst, H5T_lib_conv_t func)
{
    H5T_conv_func_t conv_func;           
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(H5T_PERS_HARD == pers || H5T_PERS_SOFT == pers);
    assert(name && *name);
    assert(src);
    assert(dst);
    assert(func);

    
    conv_func.is_app     = false;
    conv_func.u.lib_func = func;

    
    if (H5T__register(pers, name, src, dst, &conv_func) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL,
                    "unable to register internal datatype conversion routine");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__register(H5T_pers_t pers, const char *name, H5T_t *src, H5T_t *dst, H5T_conv_func_t *conv)
{
    H5T_path_t *old_path = NULL;             
    H5T_path_t *new_path = NULL;             
    H5T_cdata_t cdata;                       
    H5T_t      *tmp_stype = NULL;            
    H5T_t      *tmp_dtype = NULL;            
    hid_t       tmp_sid   = H5I_INVALID_HID; 
    hid_t       tmp_did   = H5I_INVALID_HID; 
    int         i;                           
    herr_t      ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(src);
    assert(dst);
    assert(conv);
    assert(H5T_PERS_HARD == pers || H5T_PERS_SOFT == pers);
    assert(name && *name);

    if (H5T_PERS_HARD == pers) {
        
        if (H5T_cmp(src, dst, false)) {
            
            if (NULL == (new_path = H5T__path_find_real(src, dst, name, conv)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to locate/allocate conversion path");

            
            for (i = 0; i < H5T_g.npaths; i++)
                if (new_path != H5T_g.path[i])
                    H5T_g.path[i]->cdata.recalc = true;
        } 
    }     
    else {
        H5T_conv_ctx_t conv_ctx = {0};

        
        if (!conv->is_app && H5CX_pushed() && (H5CX_get_dt_conv_cb(&conv_ctx.u.init.cb_struct) < 0))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "unable to get conversion exception callback");

        
        if ((size_t)H5T_g.nsoft >= H5T_g.asoft) {
            size_t      na = MAX(32, 2 * H5T_g.asoft);
            H5T_soft_t *x;

            if (NULL == (x = (H5T_soft_t *)H5MM_realloc(H5T_g.soft, na * sizeof(H5T_soft_t))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
            H5T_g.asoft = na;
            H5T_g.soft  = x;
        } 
        strncpy(H5T_g.soft[H5T_g.nsoft].name, name, (size_t)H5T_NAMELEN - 1);
        H5T_g.soft[H5T_g.nsoft].name[H5T_NAMELEN - 1] = '\0';
        H5T_g.soft[H5T_g.nsoft].src                   = src->shared->type;
        H5T_g.soft[H5T_g.nsoft].dst                   = dst->shared->type;
        H5T_g.soft[H5T_g.nsoft].conv                  = *conv;
        H5T_g.nsoft++;

        
        for (i = 1; i < H5T_g.npaths; i++) {
            old_path = H5T_g.path[i];
            assert(old_path);

            
            if (old_path->is_hard || old_path->src->shared->type != src->shared->type ||
                old_path->dst->shared->type != dst->shared->type)
                continue;

            memset(&cdata, 0, sizeof cdata);
            cdata.command = H5T_CONV_INIT;
            if (conv->is_app) {
                if (NULL == (tmp_stype = H5T_copy(old_path->src, H5T_COPY_ALL)))
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "unable to copy src datatype");
                if (NULL == (tmp_dtype = H5T_copy(old_path->dst, H5T_COPY_ALL)))
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "unable to copy dst datatype");

                if ((tmp_sid = H5I_register(H5I_DATATYPE, tmp_stype, false)) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL,
                                "unable to register ID for source datatype");
                if ((tmp_did = H5I_register(H5I_DATATYPE, tmp_dtype, false)) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL,
                                "unable to register ID for destination datatype");

                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        ret_value = (conv->u.app_func)(tmp_sid, tmp_did, &cdata, 0, 0, 0, NULL, NULL,
                                                       H5CX_get_dxpl());
                    }
                H5_AFTER_USER_CB(FAIL)
                if (ret_value < 0) {
                    if (H5I_dec_ref(tmp_sid) < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL,
                                    "unable to decrement reference count on temporary ID");
                    if (H5I_dec_ref(tmp_did) < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL,
                                    "unable to decrement reference count on temporary ID");
                    tmp_sid = tmp_did = H5I_INVALID_HID;
                    tmp_stype = tmp_dtype = NULL;
                    if (H5E_clear_stack() < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRESET, FAIL, "unable to clear current error stack");
                    continue;
                } 
            }     
            else if ((conv->u.lib_func)(old_path->src, old_path->dst, &cdata, &conv_ctx, 0, 0, 0, NULL,
                                        NULL) < 0) {
                if (H5E_clear_stack() < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRESET, FAIL, "unable to clear current error stack");
                continue;
            } 

            
            if (NULL == (new_path = H5FL_CALLOC(H5T_path_t)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
            strncpy(new_path->name, name, (size_t)H5T_NAMELEN - 1);
            new_path->name[H5T_NAMELEN - 1] = '\0';
            if (NULL == (new_path->src = H5T_copy(old_path->src, H5T_COPY_ALL)) ||
                NULL == (new_path->dst = H5T_copy(old_path->dst, H5T_COPY_ALL)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to copy data types");
            new_path->conv    = *conv;
            new_path->is_hard = false;
            new_path->cdata   = cdata;

            
            H5T_g.path[i] = new_path;
            new_path      = NULL; 

            
            conv_ctx.u.free.src_type_id = tmp_sid;
            conv_ctx.u.free.dst_type_id = tmp_did;
            if (H5T__path_free(old_path, &conv_ctx) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "unable to free datatype conversion path");

            
            if (tmp_sid >= 0) {
                if (H5I_dec_ref(tmp_sid) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement reference on temporary ID");
                tmp_sid   = H5I_INVALID_HID;
                tmp_stype = NULL;
            }
            if (tmp_did >= 0) {
                if (H5I_dec_ref(tmp_did) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement reference on temporary ID");
                tmp_did   = H5I_INVALID_HID;
                tmp_dtype = NULL;
            }

            
            if (H5E_clear_stack() < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRESET, FAIL, "unable to clear current error stack");
        } 
    }     

done:
    if (ret_value < 0) {
        if (new_path) {
            if (new_path->src && (H5T_close_real(new_path->src) < 0))
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "unable to close datatype");
            if (new_path->dst && (H5T_close_real(new_path->dst) < 0))
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "unable to close datatype");
            new_path = H5FL_FREE(H5T_path_t, new_path);
        } 
    }     

    if (tmp_sid >= 0) {
        if (H5I_dec_ref(tmp_sid) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement reference on temporary ID");
    }
    else if (tmp_stype) {
        if (H5T_close(tmp_stype) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary datatype");
    }
    if (tmp_did >= 0) {
        if (H5I_dec_ref(tmp_did) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement reference on temporary ID");
    }
    else if (tmp_dtype) {
        if (H5T_close(tmp_dtype) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary datatype");
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Tregister(H5T_pers_t pers, const char *name, hid_t src_id, hid_t dst_id, H5T_conv_t func)
{
    H5T_t          *src;                 
    H5T_t          *dst;                 
    H5T_conv_func_t conv_func;           
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5T_PERS_HARD != pers && H5T_PERS_SOFT != pers)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid function persistence");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "conversion must have a name for debugging");
    if (NULL == (src = (H5T_t *)H5I_object_verify(src_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data type");
    if (NULL == (dst = (H5T_t *)H5I_object_verify(dst_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data type");
    if (!func)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no conversion function specified");

    
    conv_func.is_app     = true;
    conv_func.u.app_func = func;

    
    if (H5T__register(pers, name, src, dst, &conv_func) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "can't register conversion function");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5T_unregister(H5T_pers_t pers, const char *name, H5T_t *src, H5T_t *dst, H5VL_object_t *owned_vol_obj,
               H5T_conv_t func)
{
    H5T_conv_ctx_t conv_ctx  = {0};  
    H5T_path_t    *path      = NULL; 
    H5T_soft_t    *soft      = NULL; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    conv_ctx.u.free.src_type_id = H5I_INVALID_HID;
    conv_ctx.u.free.dst_type_id = H5I_INVALID_HID;

    
    if ((H5T_PERS_DONTCARE == pers || H5T_PERS_SOFT == pers) && !owned_vol_obj) {
        for (int i = H5T_g.nsoft - 1; i >= 0; --i) {
            soft = H5T_g.soft + i;
            assert(soft);
            if (name && *name && strcmp(name, soft->name) != 0)
                continue;
            if (src && src->shared->type != soft->src)
                continue;
            if (dst && dst->shared->type != soft->dst)
                continue;
            if (func && func != soft->conv.u.app_func)
                continue;

            memmove(H5T_g.soft + i, H5T_g.soft + i + 1, (size_t)(H5T_g.nsoft - (i + 1)) * sizeof(H5T_soft_t));
            --H5T_g.nsoft;
        }
    }

    
    for (int i = H5T_g.npaths - 1; i > 0; --i) {
        bool nomatch;

        path = H5T_g.path[i];
        assert(path);

        nomatch = !H5T_path_match(path, pers, name, src, dst, owned_vol_obj, func);

        
        if (nomatch) {
            
            path->cdata.recalc = true;
        }
        else {
            
            memmove(H5T_g.path + i, H5T_g.path + i + 1,
                    (size_t)(H5T_g.npaths - (i + 1)) * sizeof(H5T_path_t *));
            --H5T_g.npaths;

            if (H5T__path_free(path, &conv_ctx) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "unable to free datatype conversion path");
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Tunregister(H5T_pers_t pers, const char *name, hid_t src_id, hid_t dst_id, H5T_conv_t func)
{
    H5T_t *src = NULL, *dst = NULL; 
    herr_t ret_value = SUCCEED;     

    FUNC_ENTER_API(FAIL)

    
    if (src_id > 0 && (NULL == (src = (H5T_t *)H5I_object_verify(src_id, H5I_DATATYPE))))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "src is not a data type");
    if (dst_id > 0 && (NULL == (dst = (H5T_t *)H5I_object_verify(dst_id, H5I_DATATYPE))))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "dst is not a data type");

    if (H5T_unregister(pers, name, src, dst, NULL, func) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDELETE, FAIL, "internal unregister function failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

H5T_conv_t
H5Tfind(hid_t src_id, hid_t dst_id, H5T_cdata_t **pcdata )
{
    H5T_t      *src, *dst;
    H5T_path_t *path;
    H5T_conv_t  ret_value; 

    FUNC_ENTER_API(NULL)

    
    if (NULL == (src = (H5T_t *)H5I_object_verify(src_id, H5I_DATATYPE)) ||
        NULL == (dst = (H5T_t *)H5I_object_verify(dst_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");
    if (!pcdata)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "no address to receive cdata pointer");

    
    if (NULL == (path = H5T_path_find(src, dst)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_NOTFOUND, NULL, "conversion function not found");

    if (pcdata)
        *pcdata = &(path->cdata);

    
    ret_value = path->conv.u.app_func;

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5Tcompiler_conv(hid_t src_id, hid_t dst_id)
{
    H5T_t *src, *dst;
    htri_t ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (src = (H5T_t *)H5I_object_verify(src_id, H5I_DATATYPE)) ||
        NULL == (dst = (H5T_t *)H5I_object_verify(dst_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data type");

    
    if ((ret_value = H5T__compiler_conv(src, dst)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_NOTFOUND, FAIL, "conversion function not found");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tconvert(hid_t src_id, hid_t dst_id, size_t nelmts, void *buf, void *background, hid_t dxpl_id)
{
    H5T_path_t *tpath;               
    H5T_t      *src, *dst;           
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (src = (H5T_t *)H5I_object_verify(src_id, H5I_DATATYPE)) ||
        NULL == (dst = (H5T_t *)H5I_object_verify(dst_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data type");
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not dataset transfer property list");

    
    H5CX_set_dxpl(dxpl_id);

    
    if (NULL == (tpath = H5T_path_find(src, dst)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to convert between src and dst data types");

    if (H5T_convert(tpath, src, dst, nelmts, 0, 0, buf, background) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "data type conversion failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Treclaim(hid_t type_id, hid_t space_id, hid_t dxpl_id, void *buf)
{
    const H5T_t *type;
    H5S_t       *space;     
    herr_t       ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (buf == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "'buf' pointer is NULL");
    if (NULL == (type = (const H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid datatype");
    if (NULL == (space = (H5S_t *)H5I_object_verify(space_id, H5I_DATASPACE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid dataspace");
    if (!(H5S_has_extent(space)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "dataspace does not have extent set");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not xfer parms");

    
    H5CX_set_dxpl(dxpl_id);

    
    ret_value = H5T_reclaim(type, space, buf);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Tencode(hid_t obj_id, void *buf, size_t *nalloc)
{
    H5T_t *dtype;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dtype = (H5T_t *)H5I_object_verify(obj_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (nalloc == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL pointer for buffer size");

    
    if (H5T_encode(dtype, (unsigned char *)buf, nalloc) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL, "can't encode datatype");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Tdecode2(const void *buf, size_t buf_size)
{
    H5T_t *dt;
    hid_t  ret_value; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (buf == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "empty buffer");

    
    if (NULL == (dt = H5T_decode(buf_size, buf)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, H5I_INVALID_HID, "can't decode object");

    
    if ((ret_value = H5I_register(H5I_DATATYPE, dt, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register data type");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5T_encode(H5T_t *obj, unsigned char *buf, size_t *nalloc)
{
    size_t buf_size;         
    H5F_t *f         = NULL; 
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NOINIT

    
    if (NULL == (f = H5F_fake_alloc((uint8_t)0)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "can't allocate fake file struct");

    
    if ((buf_size = H5O_msg_raw_size(f, H5O_DTYPE_ID, true, obj)) == 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADSIZE, FAIL, "can't find datatype size");

    
    if (!buf || *nalloc < (buf_size + 1 + 1))
        *nalloc = buf_size + 1 + 1;
    else {
        
        *buf++ = H5O_DTYPE_ID;

        
        *buf++ = H5T_ENCODE_VERSION;

        
        if (H5O_msg_encode(f, H5O_DTYPE_ID, true, buf, obj) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL, "can't encode object");
    } 

done:
    
    if (f && H5F_fake_free(f) < 0)
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, FAIL, "unable to release fake file struct");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5T_t *
H5T_decode(size_t buf_size, const unsigned char *buf)
{
    H5F_t *f         = NULL; 
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOINIT

    
    if (NULL == (f = H5F_fake_alloc((uint8_t)0)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, NULL, "can't allocate fake file struct");

    if (buf_size != SIZE_MAX && H5_IS_BUFFER_OVERFLOW(buf, 1, buf + buf_size - 1))
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADMESG, NULL, "buffer too small to be datatype message");

    
    if (*buf++ != H5O_DTYPE_ID)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADMESG, NULL, "not an encoded datatype");

    if (buf_size != SIZE_MAX && H5_IS_BUFFER_OVERFLOW(buf, 1, buf + buf_size - 1))
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADMESG, NULL, "buffer too small to be datatype message");

    
    if (*buf++ != H5T_ENCODE_VERSION)
        HGOTO_ERROR(H5E_DATATYPE, H5E_VERSION, NULL, "unknown version of encoded datatype");

    
    if (NULL == (ret_value = (H5T_t *)H5O_msg_decode(f, NULL, H5O_DTYPE_ID, buf_size, buf)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, NULL, "can't decode object");

    
    if (H5T_set_loc(ret_value, NULL, H5T_LOC_MEMORY) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "invalid datatype location");

    
    ret_value->vol_obj = NULL;
done:
    
    if (f && H5F_fake_free(f) < 0)
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, NULL, "unable to release fake file struct");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5T_t *
H5T__create(H5T_class_t type, size_t size)
{
    H5T_t *dt        = NULL;
    H5T_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    switch (type) {
        case H5T_INTEGER:
        case H5T_FLOAT:
        case H5T_TIME:
        case H5T_STRING: {
            H5T_t *origin_dt = NULL;

            if (NULL == (origin_dt = (H5T_t *)H5I_object(H5T_C_S1)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, NULL, "can't get structure for string type");

            
            if (NULL == (dt = H5T_copy(origin_dt, H5T_COPY_TRANSIENT)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to copy");
            dt->shared->type = type;

            
            if (H5T__set_size(dt, size) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to set size for string type");
        } break;

        case H5T_BITFIELD:
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, NULL, "type class is not appropriate - use H5Tcopy()");

        case H5T_OPAQUE:
        case H5T_COMPOUND:
            if (NULL == (dt = H5T__alloc()))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
            dt->shared->type = type;

            if (type == H5T_COMPOUND) {
                dt->shared->u.compnd.packed    = false; 
                dt->shared->u.compnd.memb_size = 0;
            } 
            else if (type == H5T_OPAQUE)
                
                dt->shared->u.opaque.tag = H5MM_strdup("");
            break;

        case H5T_ENUM: {
            hid_t  subtype;
            H5T_t *sub_t_obj;

            if (sizeof(char) == size)
                subtype = H5T_NATIVE_SCHAR_g;
            else if (sizeof(short) == size)
                subtype = H5T_NATIVE_SHORT_g;
            else if (sizeof(int) == size)
                subtype = H5T_NATIVE_INT_g;
            else if (sizeof(long) == size)
                subtype = H5T_NATIVE_LONG_g;
#if H5_SIZEOF_LONG != H5_SIZEOF_LONG_LONG
            else if (sizeof(long long) == size)
                subtype = H5T_NATIVE_LLONG_g;
#endif 
            else
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "no applicable native integer type");
            if (NULL == (dt = H5T__alloc()))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
            dt->shared->type = type;
            if (NULL == (sub_t_obj = (H5T_t *)H5I_object(subtype)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, NULL, "unable to get datatype object");
            if (NULL == (dt->shared->parent = H5T_copy(sub_t_obj, H5T_COPY_ALL)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "unable to copy base datatype");
        } break;

        case H5T_VLEN: 
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, NULL, "base type required - use H5Tvlen_create()");

        case H5T_ARRAY: 
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, NULL, "base type required - use H5Tarray_create2()");

        case H5T_COMPLEX: 
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, NULL, "base type required - use H5Tcomplex_create()");

        case H5T_NO_CLASS:
        case H5T_REFERENCE:
        case H5T_NCLASSES:
        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, NULL, "unknown data type class");
    } 

    
    if (H5T_STRING != type || H5T_VARIABLE != size)
        dt->shared->size = size;

    
    dt->vol_obj = NULL;

    
    ret_value = dt;

done:
    if (NULL == ret_value) {
        if (dt) {
            if (dt->shared->owned_vol_obj && H5VL_free_object(dt->shared->owned_vol_obj) < 0)
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, NULL, "unable to close owned VOL object");
            dt->shared = H5FL_FREE(H5T_shared_t, dt->shared);
            dt         = H5FL_FREE(H5T_t, dt);
        }
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5T_t *
H5T__initiate_copy(const H5T_t *old_dt)
{
    H5T_t *new_dt    = NULL; 
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (new_dt = H5FL_MALLOC(H5T_t)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, NULL, "H5T_t memory allocation failed");
    if (NULL == (new_dt->shared = H5FL_MALLOC(H5T_shared_t)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, NULL, "H5T_shared_t memory allocation failed");

    
    *(new_dt->shared) = *(old_dt->shared);

    
    if (new_dt->shared->owned_vol_obj)
        (void)H5VL_object_inc_rc(new_dt->shared->owned_vol_obj);

    
    new_dt->vol_obj = NULL;

    
    ret_value = new_dt;

done:
    if (ret_value == NULL)
        if (new_dt) {
            if (new_dt->shared) {
                if (new_dt->shared->owned_vol_obj && H5VL_free_object(new_dt->shared->owned_vol_obj) < 0)
                    HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, NULL, "unable to close owned VOL object");
                new_dt->shared = H5FL_FREE(H5T_shared_t, new_dt->shared);
            } 
            new_dt = H5FL_FREE(H5T_t, new_dt);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5T_t *
H5T__copy_transient(H5T_t *old_dt)
{
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (ret_value = H5T_copy(old_dt, H5T_COPY_TRANSIENT)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "can't make 'transient' copy of datatype");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5T_t *
H5T__copy_all(H5T_t *old_dt)
{
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (ret_value = H5T_copy(old_dt, H5T_COPY_ALL)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "can't make 'all' copy of datatype");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__complete_copy(H5T_t *new_dt, const H5T_t *old_dt, H5T_shared_t *reopened_fo, bool set_memory_type,
                   H5T_copy_func_t copyfn)
{
    H5T_t   *tmp = NULL;          
    char    *s;                   
    unsigned i;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (!reopened_fo) {
        
        if (old_dt->shared->parent)
            if (NULL == (new_dt->shared->parent = (*copyfn)(old_dt->shared->parent)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "can't copy datatype's parent type");

        switch (new_dt->shared->type) {
            case H5T_COMPOUND: {
                ssize_t accum_change = 0; 

                
                
                if (new_dt->shared->u.compnd.nalloc > 0) {
                    if (NULL == (new_dt->shared->u.compnd.memb =
                                     H5MM_malloc(new_dt->shared->u.compnd.nalloc * sizeof(H5T_cmemb_t))))
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "memory allocation failed");

                    H5MM_memcpy(new_dt->shared->u.compnd.memb, old_dt->shared->u.compnd.memb,
                                new_dt->shared->u.compnd.nmembs * sizeof(H5T_cmemb_t));
                } 

                for (i = 0; i < new_dt->shared->u.compnd.nmembs; i++) {
                    unsigned j;
                    int      old_match;

                    if (NULL == (s = H5MM_xstrdup(new_dt->shared->u.compnd.memb[i].name)))
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL,
                                    "can't copy string for compound field's name");
                    new_dt->shared->u.compnd.memb[i].name = s;
                    if (NULL == (tmp = (*copyfn)(old_dt->shared->u.compnd.memb[i].type)))
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "can't copy compound field's datatype");
                    new_dt->shared->u.compnd.memb[i].type = tmp;
                    assert(tmp != NULL);

                    
                    if ((accum_change < 0) &&
                        ((ssize_t)new_dt->shared->u.compnd.memb[i].offset < accum_change))
                        HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "invalid field size in datatype");

                    
                    new_dt->shared->u.compnd.memb[i].offset += (size_t)accum_change;

                    if (old_dt->shared->u.compnd.sorted != H5T_SORT_VALUE) {
                        for (old_match = -1, j = 0; j < old_dt->shared->u.compnd.nmembs; j++) {
                            if (!strcmp(new_dt->shared->u.compnd.memb[i].name,
                                        old_dt->shared->u.compnd.memb[j].name)) {
                                old_match = (int)j;
                                break;
                            } 
                        }     

                        
                        if (old_match < 0)
                            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "fields in datatype corrupted");
                    } 
                    else
                        old_match = (int)i;

                    
                    if (new_dt->shared->u.compnd.memb[i].type->shared->size !=
                        old_dt->shared->u.compnd.memb[old_match].type->shared->size) {
                        
                        new_dt->shared->u.compnd.memb[i].size =
                            (old_dt->shared->u.compnd.memb[old_match].size * tmp->shared->size) /
                            old_dt->shared->u.compnd.memb[old_match].type->shared->size;

                        accum_change +=
                            (ssize_t)(new_dt->shared->u.compnd.memb[i].type->shared->size -
                                      old_dt->shared->u.compnd.memb[old_match].type->shared->size);
                    } 
                }     

                
                if ((accum_change < 0) && ((ssize_t)new_dt->shared->size < accum_change))
                    HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "invalid field size in datatype");

                
                new_dt->shared->size += (size_t)accum_change;
            } break;

            case H5T_ENUM:
                
                if (old_dt->shared->u.enumer.nalloc > 0) {
                    if (NULL == (new_dt->shared->u.enumer.name =
                                     H5MM_malloc(new_dt->shared->u.enumer.nalloc * sizeof(char *))))
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL,
                                    "enam name array memory allocation failed");
                    if (NULL == (new_dt->shared->u.enumer.value =
                                     H5MM_malloc(new_dt->shared->u.enumer.nalloc * new_dt->shared->size)))
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL,
                                    "enam value array memory allocation failed");
                    H5MM_memcpy(new_dt->shared->u.enumer.value, old_dt->shared->u.enumer.value,
                                new_dt->shared->u.enumer.nmembs * new_dt->shared->size);
                    for (i = 0; i < new_dt->shared->u.enumer.nmembs; i++) {
                        if (NULL == (s = H5MM_xstrdup(old_dt->shared->u.enumer.name[i])))
                            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL,
                                        "can't copy string for enum value's name");
                        new_dt->shared->u.enumer.name[i] = s;
                    }
                }
                else {
                    
                    memset(&new_dt->shared->u.enumer, 0, sizeof(H5T_enum_t));
                }
                break;

            case H5T_VLEN:
            case H5T_REFERENCE:
                if (set_memory_type)
                    
                    if (H5T_set_loc(new_dt, NULL, H5T_LOC_MEMORY) < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "invalid datatype location");
                break;

            case H5T_OPAQUE:
                
                new_dt->shared->u.opaque.tag = H5MM_xstrdup(new_dt->shared->u.opaque.tag);
                break;

            case H5T_ARRAY:
                
                new_dt->shared->size = new_dt->shared->u.array.nelem * new_dt->shared->parent->shared->size;
                break;

            case H5T_NO_CLASS:
            case H5T_INTEGER:
            case H5T_FLOAT:
            case H5T_TIME:
            case H5T_STRING:
            case H5T_BITFIELD:
            case H5T_COMPLEX:
            case H5T_NCLASSES:
            default:
                break;
        } 
    }     

    
    if (H5O_loc_reset(&new_dt->oloc) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRESET, FAIL, "unable to initialize location");
    if (H5G_name_reset(&new_dt->path) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, FAIL, "unable to reset path");

    if (new_dt->shared->state == H5T_STATE_NAMED || new_dt->shared->state == H5T_STATE_OPEN) {
        if (H5O_loc_copy_deep(&(new_dt->oloc), &(old_dt->oloc)) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "can't copy object location");
        if (H5G_name_copy(&(new_dt->path), &(old_dt->path), H5_COPY_DEEP) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, FAIL, "unable to copy path");
    } 

    
    if ((old_dt->sh_loc.type == H5O_SHARE_TYPE_SOHM || old_dt->sh_loc.type == H5O_SHARE_TYPE_HERE) ||
        new_dt->shared->state == H5T_STATE_NAMED || new_dt->shared->state == H5T_STATE_OPEN) {
        if (H5O_set_shared(&(new_dt->sh_loc), &(old_dt->sh_loc)) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "can't copy shared information");
    } 
    else
        
        H5O_msg_reset_share(H5O_DTYPE_ID, new_dt);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5T_t *
H5T_copy(const H5T_t *old_dt, H5T_copy_t method)
{
    H5T_t          *new_dt = NULL;    
    H5T_copy_func_t copyfn;           
    H5T_t          *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(old_dt);

    
    if (NULL == (new_dt = H5T__initiate_copy(old_dt)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "can't copy core datatype info");

    
    switch (method) {
        case H5T_COPY_TRANSIENT:
            
            new_dt->shared->state = H5T_STATE_TRANSIENT;
            copyfn                = H5T__copy_transient;
            break;

        case H5T_COPY_ALL:
            
            if (H5T_STATE_OPEN == old_dt->shared->state)
                new_dt->shared->state = H5T_STATE_NAMED;
            else if (H5T_STATE_IMMUTABLE == old_dt->shared->state)
                new_dt->shared->state = H5T_STATE_RDONLY;
            copyfn = H5T__copy_all;
            break;

        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, NULL, "invalid copy method type");
    } 

    
    if (H5T__complete_copy(new_dt, old_dt, NULL, (method == H5T_COPY_TRANSIENT), copyfn) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "can't complete datatype initialization");

    
    ret_value = new_dt;

done:
    if (ret_value == NULL)
        if (new_dt) {
            assert(new_dt->shared);
            if (new_dt->shared->owned_vol_obj && H5VL_free_object(new_dt->shared->owned_vol_obj) < 0)
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, NULL, "unable to close owned VOL object");
            new_dt->shared = H5FL_FREE(H5T_shared_t, new_dt->shared);
            new_dt         = H5FL_FREE(H5T_t, new_dt);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5T_t *
H5T_copy_reopen(H5T_t *old_dt)
{
    H5T_t        *new_dt      = NULL; 
    H5T_shared_t *reopened_fo = NULL; 
    H5T_t        *ret_value   = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(old_dt);

    
    if (NULL == (new_dt = H5T__initiate_copy(old_dt)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "can't copy core datatype info");

    
    if (old_dt->sh_loc.type == H5O_SHARE_TYPE_COMMITTED) {
        
        if (NULL ==
            (reopened_fo = (H5T_shared_t *)H5FO_opened(old_dt->sh_loc.file, old_dt->sh_loc.u.loc.oh_addr))) {
            
            H5E_clear_stack();

            
            if (H5O_open(&old_dt->oloc) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, NULL, "unable to reopen named data type");

            
            if (H5FO_insert(old_dt->sh_loc.file, old_dt->sh_loc.u.loc.oh_addr, new_dt->shared, false) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINSERT, NULL,
                            "can't insert datatype into list of open objects");

            
            if (H5FO_top_incr(old_dt->sh_loc.file, old_dt->sh_loc.u.loc.oh_addr) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINC, NULL, "can't increment object count");

            new_dt->shared->fo_count = 1;
        } 
        else {
            
            if (new_dt->shared->owned_vol_obj && H5VL_free_object(new_dt->shared->owned_vol_obj) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, NULL, "unable to close owned VOL object");
            new_dt->shared = H5FL_FREE(H5T_shared_t, new_dt->shared);
            new_dt->shared = reopened_fo;

            reopened_fo->fo_count++;

            
            if (H5FO_top_count(old_dt->sh_loc.file, old_dt->sh_loc.u.loc.oh_addr) == 0) {
                
                if (H5O_open(&old_dt->oloc) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, NULL, "unable to open object header");
            } 

            
            if (H5FO_top_incr(old_dt->sh_loc.file, old_dt->sh_loc.u.loc.oh_addr) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINC, NULL, "can't increment object count");
        } 

        
        new_dt->shared->state = H5T_STATE_OPEN;
    } 
    else
        
        if (H5T_STATE_IMMUTABLE == old_dt->shared->state)
            new_dt->shared->state = H5T_STATE_RDONLY;

    
    if (H5T__complete_copy(new_dt, old_dt, reopened_fo, true, H5T_copy_reopen) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "can't complete datatype initialization");

    
    ret_value = new_dt;

done:
    if (ret_value == NULL)
        if (new_dt) {
            assert(new_dt->shared);
            if (new_dt->shared->owned_vol_obj && H5VL_free_object(new_dt->shared->owned_vol_obj) < 0)
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, NULL, "unable to close owned VOL object");
            new_dt->shared = H5FL_FREE(H5T_shared_t, new_dt->shared);
            new_dt         = H5FL_FREE(H5T_t, new_dt);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T_lock(H5T_t *dt, bool immutable)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(dt);

    switch (dt->shared->state) {
        case H5T_STATE_TRANSIENT:
            dt->shared->state = immutable ? H5T_STATE_IMMUTABLE : H5T_STATE_RDONLY;
            break;
        case H5T_STATE_RDONLY:
            if (immutable)
                dt->shared->state = H5T_STATE_IMMUTABLE;
            break;
        case H5T_STATE_IMMUTABLE:
        case H5T_STATE_NAMED:
        case H5T_STATE_OPEN:
            
            break;
        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "invalid datatype state");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

H5T_t *
H5T__alloc(void)
{
    H5T_t *dt        = NULL; 
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (dt = H5FL_CALLOC(H5T_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    H5O_loc_reset(&(dt->oloc));
    H5G_name_reset(&(dt->path));
    H5O_msg_reset_share(H5O_DTYPE_ID, dt);

    
    if (NULL == (dt->shared = H5FL_CALLOC(H5T_shared_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    dt->shared->version = H5O_DTYPE_VERSION_1;

    
    dt->vol_obj = NULL;

    
    ret_value = dt;

done:
    if (ret_value == NULL)
        if (dt) {
            if (dt->shared) {
                assert(!dt->shared->owned_vol_obj);
                dt->shared = H5FL_FREE(H5T_shared_t, dt->shared);
            } 
            dt = H5FL_FREE(H5T_t, dt);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__free(H5T_t *dt)
{
    unsigned i;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(dt && dt->shared);

    
    H5G_name_free(&(dt->path));

    
    if (H5T_STATE_IMMUTABLE == dt->shared->state)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "unable to close immutable datatype");

    
    switch (dt->shared->type) {
        case H5T_COMPOUND:
            for (i = 0; i < dt->shared->u.compnd.nmembs; i++) {
                dt->shared->u.compnd.memb[i].name = (char *)H5MM_xfree(dt->shared->u.compnd.memb[i].name);
                if (H5T_close_real(dt->shared->u.compnd.memb[i].type) < 0)
                    
                    HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL,
                                "unable to close datatype for compound member");
            }
            dt->shared->u.compnd.memb   = (H5T_cmemb_t *)H5MM_xfree(dt->shared->u.compnd.memb);
            dt->shared->u.compnd.nmembs = 0;
            break;

        case H5T_ENUM:
            for (i = 0; i < dt->shared->u.enumer.nmembs; i++)
                dt->shared->u.enumer.name[i] = (char *)H5MM_xfree(dt->shared->u.enumer.name[i]);
            dt->shared->u.enumer.name   = (char **)H5MM_xfree(dt->shared->u.enumer.name);
            dt->shared->u.enumer.value  = (uint8_t *)H5MM_xfree(dt->shared->u.enumer.value);
            dt->shared->u.enumer.nmembs = 0;
            break;

        case H5T_OPAQUE:
            dt->shared->u.opaque.tag = (char *)H5MM_xfree(dt->shared->u.opaque.tag);
            break;

        case H5T_NO_CLASS:
        case H5T_INTEGER:
        case H5T_FLOAT:
        case H5T_TIME:
        case H5T_STRING:
        case H5T_BITFIELD:
        case H5T_REFERENCE:
        case H5T_VLEN:
        case H5T_ARRAY:
        case H5T_COMPLEX:
        case H5T_NCLASSES:
        default:
            break;
    } 
    dt->shared->type = H5T_NO_CLASS;

    
    assert(dt->shared->parent != dt);
    if (dt->shared->parent && H5T_close_real(dt->shared->parent) < 0)
        
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "unable to close parent data type");
    dt->shared->parent = NULL;

    
    if (dt->shared->owned_vol_obj && H5VL_free_object(dt->shared->owned_vol_obj) < 0)
        
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "unable to close owned VOL object");
    dt->shared->owned_vol_obj = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T_close_real(H5T_t *dt)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dt);

    
    if (dt->shared && (dt->shared->state != H5T_STATE_OPEN)) {
        if (H5T__free(dt) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "unable to free datatype");

        assert(!dt->shared->owned_vol_obj);
        dt->shared = H5FL_FREE(H5T_shared_t, dt->shared);
    } 
    else
        
        H5G_name_free(&(dt->path));

    
    dt = H5FL_FREE(H5T_t, dt);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T_close(H5T_t *dt)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dt);

    
    if (dt->shared && (dt->shared->state == H5T_STATE_OPEN)) {
        
        dt->shared->fo_count--;

        
        assert(dt->sh_loc.type == H5O_SHARE_TYPE_COMMITTED);
        assert(H5_addr_defined(dt->sh_loc.u.loc.oh_addr));
        assert(H5_addr_defined(dt->oloc.addr));

        

        
        if (H5FO_top_decr(dt->sh_loc.file, dt->sh_loc.u.loc.oh_addr) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, FAIL, "can't decrement count for object");

        
        if (0 == dt->shared->fo_count) {
            bool corked; 

            
            if (H5AC_cork(dt->oloc.file, dt->oloc.addr, H5AC__GET_CORKED, &corked) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "unable to retrieve an object's cork status");
            if (corked)
                if (H5AC_cork(dt->oloc.file, dt->oloc.addr, H5AC__UNCORK, NULL) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTUNCORK, FAIL, "unable to uncork an object");

            
            if (H5FO_delete(dt->sh_loc.file, dt->sh_loc.u.loc.oh_addr) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, FAIL,
                            "can't remove datatype from list of open objects");
            if (H5O_close(&dt->oloc, NULL) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to close data type object header");

            
            dt->shared->state = H5T_STATE_NAMED;
        } 
        else {
            
            if (H5FO_top_count(dt->sh_loc.file, dt->sh_loc.u.loc.oh_addr) == 0) {
                
                if (H5O_close(&dt->oloc, NULL) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to close");
            } 
            else
                
                if (H5O_loc_free(&(dt->oloc)) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, FAIL, "problem attempting to free location");
        } 
    }     

    
    if (H5T_close_real(dt) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, FAIL, "unable to free datatype");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__set_size(H5T_t *dt, size_t size)
{
    size_t prec, offset;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dt);
    assert(dt->shared);
    assert(size != 0);
    assert(H5T_ARRAY != dt->shared->type);
    assert(H5T_REFERENCE != dt->shared->type);
    assert(H5T_COMPLEX != dt->shared->type);
    assert(H5T_VLEN != dt->shared->type || H5T_VLEN_STRING == dt->shared->u.vlen.type);
    assert(!(H5T_ENUM == dt->shared->type && 0 == dt->shared->u.enumer.nmembs));

    if (dt->shared->parent) {
        if (H5T__set_size(dt->shared->parent, size) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to set size for parent data type");
    }
    else {
        if (H5T_IS_ATOMIC(dt->shared)) {
            offset = dt->shared->u.atomic.offset;
            prec   = dt->shared->u.atomic.prec;

            
            if (prec > 8 * size)
                offset = 0;
            else if (offset + prec > 8 * size)
                offset = 8 * size - prec;
            if (prec > 8 * size)
                prec = 8 * size;
        }
        else
            prec = offset = 0;

        switch (dt->shared->type) {
            case H5T_INTEGER:
            case H5T_TIME:
            case H5T_BITFIELD:
            case H5T_OPAQUE:
                
                break;

            case H5T_COMPOUND:
                
                if (size < dt->shared->size) {
                    int      num_membs = 0;
                    unsigned i, max_index            = 0;
                    size_t   memb_offset, max_offset = 0;
                    size_t   max_size;

                    if ((num_membs = H5T_get_nmembers(dt)) < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to get number of members");

                    if (num_membs) {
                        for (i = 0; i < (unsigned)num_membs; i++) {
                            memb_offset = H5T_get_member_offset(dt, i);
                            if (memb_offset > max_offset) {
                                max_offset = memb_offset;
                                max_index  = i;
                            } 
                        }     

                        max_size = H5T__get_member_size(dt, max_index);

                        if (size < (max_offset + max_size))
                            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                                        "size shrinking will cut off last member ");
                    } 

                    
                    
                    assert(!dt->shared->u.compnd.packed);
                } 

                break;

            case H5T_STRING:
                
                if (size == H5T_VARIABLE) {
                    H5T_t     *base = NULL; 
                    H5T_cset_t tmp_cset;    
                    H5T_str_t  tmp_strpad;  

                    
                    if (NULL == (base = (H5T_t *)H5I_object(H5T_NATIVE_UCHAR)))
                        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid base datatype");
                    if (NULL == (dt->shared->parent = H5T_copy(base, H5T_COPY_ALL)))
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "unable to copy base datatype");

                    
                    dt->shared->type = H5T_VLEN;

                    
                    dt->shared->force_conv = true;

                    
                    tmp_cset   = dt->shared->u.atomic.u.s.cset;
                    tmp_strpad = dt->shared->u.atomic.u.s.pad;

                    
                    dt->shared->u.vlen.type = H5T_VLEN_STRING;

                    
                    dt->shared->u.vlen.cset = tmp_cset;
                    dt->shared->u.vlen.pad  = tmp_strpad;

                    
                    if (H5T_set_loc(dt, NULL, H5T_LOC_MEMORY) < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "invalid datatype location");
                } 
                else {
                    prec   = 8 * size;
                    offset = 0;
                } 
                break;

            case H5T_FLOAT:
                
                if (dt->shared->u.atomic.u.f.sign >= prec + offset ||
                    dt->shared->u.atomic.u.f.epos + dt->shared->u.atomic.u.f.esize > prec + offset ||
                    dt->shared->u.atomic.u.f.mpos + dt->shared->u.atomic.u.f.msize > prec + offset) {
                    HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                                "adjust sign, mantissa, and exponent fields first");
                }
                break;

            case H5T_ENUM:
            case H5T_VLEN:
            case H5T_ARRAY:
            case H5T_REFERENCE:
            case H5T_COMPLEX:
                assert("can't happen" && 0);
                break;

            case H5T_NO_CLASS:
            case H5T_NCLASSES:
                assert("invalid type" && 0);
                break;

            default:
                assert("not implemented yet" && 0);
                break;
        } 

        
        if (dt->shared->type != H5T_VLEN) {
            dt->shared->size = size;
            if (H5T_IS_ATOMIC(dt->shared)) {
                dt->shared->u.atomic.offset = offset;
                dt->shared->u.atomic.prec   = prec;
            }
        } 

        
        if (dt->shared->type == H5T_COMPOUND)
            H5T__update_packed(dt);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t
H5T_get_size(const H5T_t *dt)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(dt);
    assert(dt->shared);

    FUNC_LEAVE_NOAPI(dt->shared->size)
} 

int
H5T_cmp(const H5T_t *dt1, const H5T_t *dt2, bool superset)
{
    unsigned *idx1 = NULL, *idx2 = NULL;
    size_t    base_size;
    bool      swapped;
    unsigned  u;
    int       tmp;
    int       ret_value = 0;

    FUNC_ENTER_NOAPI(0)

    
    assert(dt1);
    assert(dt2);

    
    if (dt1 == dt2)
        HGOTO_DONE(0);

    assert(dt1->shared);
    assert(dt2->shared);

    
    if (dt1->shared->type < dt2->shared->type)
        HGOTO_DONE(-1);
    if (dt1->shared->type > dt2->shared->type)
        HGOTO_DONE(1);

    if (dt1->shared->size < dt2->shared->size)
        HGOTO_DONE(-1);
    if (dt1->shared->size > dt2->shared->size)
        HGOTO_DONE(1);

    if (dt1->shared->parent && !dt2->shared->parent)
        HGOTO_DONE(-1);
    if (!dt1->shared->parent && dt2->shared->parent)
        HGOTO_DONE(1);
    if (dt1->shared->parent) {
        tmp = H5T_cmp(dt1->shared->parent, dt2->shared->parent, superset);
        if (tmp < 0)
            HGOTO_DONE(-1);
        if (tmp > 0)
            HGOTO_DONE(1);
    } 

    switch (dt1->shared->type) {
        case H5T_COMPOUND:
            
            if (dt1->shared->u.compnd.nmembs == 0 && dt2->shared->u.compnd.nmembs == 0)
                HGOTO_DONE(0);
            if (dt1->shared->u.compnd.nmembs < dt2->shared->u.compnd.nmembs)
                HGOTO_DONE(-1);
            if (dt1->shared->u.compnd.nmembs > dt2->shared->u.compnd.nmembs)
                HGOTO_DONE(1);

            
            if (NULL == (idx1 = (unsigned *)H5MM_malloc(dt1->shared->u.compnd.nmembs * sizeof(unsigned))) ||
                NULL == (idx2 = (unsigned *)H5MM_malloc(dt2->shared->u.compnd.nmembs * sizeof(unsigned))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, 0, "memory allocation failed");
            for (u = 0; u < dt1->shared->u.compnd.nmembs; u++)
                idx1[u] = idx2[u] = u;
            if (dt1->shared->u.enumer.nmembs > 1) {
                int i;

                for (i = (int)dt1->shared->u.compnd.nmembs - 1, swapped = true; swapped && i >= 0; --i) {
                    int j;

                    for (j = 0, swapped = false; j < i; j++)
                        if (strcmp(dt1->shared->u.compnd.memb[idx1[j]].name,
                                   dt1->shared->u.compnd.memb[idx1[j + 1]].name) > 0) {
                            unsigned tmp_idx = idx1[j];
                            idx1[j]          = idx1[j + 1];
                            idx1[j + 1]      = tmp_idx;
                            swapped          = true;
                        }
                }
                for (i = (int)dt2->shared->u.compnd.nmembs - 1, swapped = true; swapped && i >= 0; --i) {
                    int j;

                    for (j = 0, swapped = false; j < i; j++)
                        if (strcmp(dt2->shared->u.compnd.memb[idx2[j]].name,
                                   dt2->shared->u.compnd.memb[idx2[j + 1]].name) > 0) {
                            unsigned tmp_idx = idx2[j];
                            idx2[j]          = idx2[j + 1];
                            idx2[j + 1]      = tmp_idx;
                            swapped          = true;
                        }
                }
            } 

#ifdef H5T_DEBUG
            
            if (dt1->shared->u.compnd.nmembs > 0) {
                for (u = 0; u < dt1->shared->u.compnd.nmembs - 1; u++) {
                    assert(strcmp(dt1->shared->u.compnd.memb[idx1[u]].name,
                                  dt1->shared->u.compnd.memb[idx1[u + 1]].name));
                    assert(strcmp(dt2->shared->u.compnd.memb[idx2[u]].name,
                                  dt2->shared->u.compnd.memb[idx2[u + 1]].name));
                }
            }
#endif

            
            for (u = 0; u < dt1->shared->u.compnd.nmembs; u++) {
                tmp = strcmp(dt1->shared->u.compnd.memb[idx1[u]].name,
                             dt2->shared->u.compnd.memb[idx2[u]].name);
                if (tmp < 0)
                    HGOTO_DONE(-1);
                if (tmp > 0)
                    HGOTO_DONE(1);

                if (dt1->shared->u.compnd.memb[idx1[u]].offset < dt2->shared->u.compnd.memb[idx2[u]].offset)
                    HGOTO_DONE(-1);
                if (dt1->shared->u.compnd.memb[idx1[u]].offset > dt2->shared->u.compnd.memb[idx2[u]].offset)
                    HGOTO_DONE(1);

                if (dt1->shared->u.compnd.memb[idx1[u]].size < dt2->shared->u.compnd.memb[idx2[u]].size)
                    HGOTO_DONE(-1);
                if (dt1->shared->u.compnd.memb[idx1[u]].size > dt2->shared->u.compnd.memb[idx2[u]].size)
                    HGOTO_DONE(1);

                tmp = H5T_cmp(dt1->shared->u.compnd.memb[idx1[u]].type,
                              dt2->shared->u.compnd.memb[idx2[u]].type, superset);
                if (tmp < 0)
                    HGOTO_DONE(-1);
                if (tmp > 0)
                    HGOTO_DONE(1);
            }
            break;

        case H5T_ENUM:
            
            if (dt1->shared->u.enumer.nmembs == 0 && dt2->shared->u.enumer.nmembs == 0)
                HGOTO_DONE(0);

            
            if (superset) {
                if (dt1->shared->u.enumer.nmembs > dt2->shared->u.enumer.nmembs)
                    HGOTO_DONE(1);
            } 
            else {
                if (dt1->shared->u.enumer.nmembs < dt2->shared->u.enumer.nmembs)
                    HGOTO_DONE(-1);
                if (dt1->shared->u.enumer.nmembs > dt2->shared->u.enumer.nmembs)
                    HGOTO_DONE(1);
            } 

            
            if (NULL == (idx1 = (unsigned *)H5MM_malloc(dt1->shared->u.enumer.nmembs * sizeof(unsigned))) ||
                NULL == (idx2 = (unsigned *)H5MM_malloc(dt2->shared->u.enumer.nmembs * sizeof(unsigned))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, 0, "memory allocation failed");
            for (u = 0; u < dt1->shared->u.enumer.nmembs; u++)
                idx1[u] = u;
            if (dt1->shared->u.enumer.nmembs > 1) {
                int i;
                for (i = (int)dt1->shared->u.enumer.nmembs - 1, swapped = true; swapped && i >= 0; --i) {
                    int j;

                    for (j = 0, swapped = false; j < i; j++)
                        if (strcmp(dt1->shared->u.enumer.name[idx1[j]],
                                   dt1->shared->u.enumer.name[idx1[j + 1]]) > 0) {
                            unsigned tmp_idx = idx1[j];
                            idx1[j]          = idx1[j + 1];
                            idx1[j + 1]      = tmp_idx;
                            swapped          = true;
                        }
                }
            }
            for (u = 0; u < dt2->shared->u.enumer.nmembs; u++)
                idx2[u] = u;
            if (dt2->shared->u.enumer.nmembs > 1) {
                int i;

                for (i = (int)dt2->shared->u.enumer.nmembs - 1, swapped = true; swapped && i >= 0; --i) {
                    int j;

                    for (j = 0, swapped = false; j < i; j++)
                        if (strcmp(dt2->shared->u.enumer.name[idx2[j]],
                                   dt2->shared->u.enumer.name[idx2[j + 1]]) > 0) {
                            unsigned tmp_idx = idx2[j];
                            idx2[j]          = idx2[j + 1];
                            idx2[j + 1]      = tmp_idx;
                            swapped          = true;
                        }
                }
            }

#ifdef H5T_DEBUG
            
            if (dt1->shared->u.enumer.nmembs > 0) {
                for (u = 0; u < dt1->shared->u.enumer.nmembs - 1; u++) {
                    assert(
                        strcmp(dt1->shared->u.enumer.name[idx1[u]], dt1->shared->u.enumer.name[idx1[u + 1]]));
                    assert(
                        strcmp(dt2->shared->u.enumer.name[idx2[u]], dt2->shared->u.enumer.name[idx2[u + 1]]));
                }
            }
#endif

            
            base_size = dt1->shared->parent->shared->size;
            for (u = 0; u < dt1->shared->u.enumer.nmembs; u++) {
                unsigned idx = 0;

                if (superset) {
                    unsigned lt  = 0, rt; 
                    int      cmp = 1;     

                    
                    rt = dt2->shared->u.enumer.nmembs;

                    while (lt < rt && cmp) {
                        idx = (lt + rt) / 2;

                        
                        if ((cmp = strcmp(dt1->shared->u.enumer.name[idx1[u]],
                                          dt2->shared->u.enumer.name[idx2[idx]])) < 0)
                            rt = idx;
                        else
                            lt = idx + 1;
                    }
                    
                    if (cmp)
                        HGOTO_DONE(-1);
                } 
                else {
                    
                    tmp = strcmp(dt1->shared->u.enumer.name[idx1[u]], dt2->shared->u.enumer.name[idx2[u]]);
                    if (tmp < 0)
                        HGOTO_DONE(-1);
                    if (tmp > 0)
                        HGOTO_DONE(1);

                    
                    idx = u;
                } 

                tmp = memcmp((uint8_t *)dt1->shared->u.enumer.value + idx1[u] * base_size,
                             (uint8_t *)dt2->shared->u.enumer.value + idx2[idx] * base_size, base_size);
                if (tmp < 0)
                    HGOTO_DONE(-1);
                if (tmp > 0)
                    HGOTO_DONE(1);
            }
            break;

        case H5T_VLEN:
            assert(dt1->shared->u.vlen.type > H5T_VLEN_BADTYPE &&
                   dt1->shared->u.vlen.type < H5T_VLEN_MAXTYPE);
            assert(dt2->shared->u.vlen.type > H5T_VLEN_BADTYPE &&
                   dt2->shared->u.vlen.type < H5T_VLEN_MAXTYPE);
            assert(dt1->shared->u.vlen.loc >= H5T_LOC_BADLOC && dt1->shared->u.vlen.loc < H5T_LOC_MAXLOC);
            assert(dt2->shared->u.vlen.loc >= H5T_LOC_BADLOC && dt2->shared->u.vlen.loc < H5T_LOC_MAXLOC);

            
            if (dt1->shared->u.vlen.type == H5T_VLEN_SEQUENCE &&
                dt2->shared->u.vlen.type == H5T_VLEN_STRING) {
                HGOTO_DONE(-1);
            }
            else if (dt1->shared->u.vlen.type == H5T_VLEN_STRING &&
                     dt2->shared->u.vlen.type == H5T_VLEN_SEQUENCE) {
                HGOTO_DONE(1);
            }
            
            if (dt1->shared->u.vlen.loc == H5T_LOC_MEMORY && dt2->shared->u.vlen.loc == H5T_LOC_DISK) {
                HGOTO_DONE(-1);
            }
            else if (dt1->shared->u.vlen.loc == H5T_LOC_DISK && dt2->shared->u.vlen.loc == H5T_LOC_MEMORY) {
                HGOTO_DONE(1);
            }
            else if (dt1->shared->u.vlen.loc == H5T_LOC_BADLOC && dt2->shared->u.vlen.loc != H5T_LOC_BADLOC) {
                HGOTO_DONE(1);
            }

            
            if (dt1->shared->u.vlen.file < dt2->shared->u.vlen.file)
                HGOTO_DONE(-1);
            if (dt1->shared->u.vlen.file > dt2->shared->u.vlen.file)
                HGOTO_DONE(1);
            break;

        case H5T_OPAQUE:
            if (dt1->shared->u.opaque.tag && dt2->shared->u.opaque.tag)
                HGOTO_DONE(strcmp(dt1->shared->u.opaque.tag, dt2->shared->u.opaque.tag));
            break;

        case H5T_ARRAY:
            if (dt1->shared->u.array.ndims < dt2->shared->u.array.ndims)
                HGOTO_DONE(-1);
            if (dt1->shared->u.array.ndims > dt2->shared->u.array.ndims)
                HGOTO_DONE(1);

            for (u = 0; u < dt1->shared->u.array.ndims; u++) {
                if (dt1->shared->u.array.dim[u] < dt2->shared->u.array.dim[u])
                    HGOTO_DONE(-1);
                if (dt1->shared->u.array.dim[u] > dt2->shared->u.array.dim[u])
                    HGOTO_DONE(1);
            }

            tmp = H5T_cmp(dt1->shared->parent, dt2->shared->parent, superset);
            if (tmp < 0)
                HGOTO_DONE(-1);
            if (tmp > 0)
                HGOTO_DONE(1);
            break;

        case H5T_COMPLEX:
            
            tmp = (dt1->shared->u.cplx.form > dt2->shared->u.cplx.form) -
                  (dt1->shared->u.cplx.form < dt2->shared->u.cplx.form);
            if (tmp < 0)
                HGOTO_DONE(-1);
            if (tmp > 0)
                HGOTO_DONE(1);

            tmp = H5T_cmp(dt1->shared->parent, dt2->shared->parent, superset);
            if (tmp < 0)
                HGOTO_DONE(-1);
            if (tmp > 0)
                HGOTO_DONE(1);

            break;

        case H5T_NO_CLASS:
        case H5T_INTEGER:
        case H5T_FLOAT:
        case H5T_TIME:
        case H5T_STRING:
        case H5T_BITFIELD:
        case H5T_REFERENCE:
        case H5T_NCLASSES:
        default:
            
            if (dt1->shared->u.atomic.order < dt2->shared->u.atomic.order)
                HGOTO_DONE(-1);
            if (dt1->shared->u.atomic.order > dt2->shared->u.atomic.order)
                HGOTO_DONE(1);

            if (dt1->shared->u.atomic.prec < dt2->shared->u.atomic.prec)
                HGOTO_DONE(-1);
            if (dt1->shared->u.atomic.prec > dt2->shared->u.atomic.prec)
                HGOTO_DONE(1);

            if (dt1->shared->u.atomic.offset < dt2->shared->u.atomic.offset)
                HGOTO_DONE(-1);
            if (dt1->shared->u.atomic.offset > dt2->shared->u.atomic.offset)
                HGOTO_DONE(1);

            if (dt1->shared->u.atomic.lsb_pad < dt2->shared->u.atomic.lsb_pad)
                HGOTO_DONE(-1);
            if (dt1->shared->u.atomic.lsb_pad > dt2->shared->u.atomic.lsb_pad)
                HGOTO_DONE(1);

            if (dt1->shared->u.atomic.msb_pad < dt2->shared->u.atomic.msb_pad)
                HGOTO_DONE(-1);
            if (dt1->shared->u.atomic.msb_pad > dt2->shared->u.atomic.msb_pad)
                HGOTO_DONE(1);

            switch (dt1->shared->type) {
                case H5T_INTEGER:
                    if (dt1->shared->u.atomic.u.i.sign < dt2->shared->u.atomic.u.i.sign)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.i.sign > dt2->shared->u.atomic.u.i.sign)
                        HGOTO_DONE(1);
                    break;

                case H5T_FLOAT:
                    if (dt1->shared->u.atomic.u.f.sign < dt2->shared->u.atomic.u.f.sign)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.f.sign > dt2->shared->u.atomic.u.f.sign)
                        HGOTO_DONE(1);

                    if (dt1->shared->u.atomic.u.f.epos < dt2->shared->u.atomic.u.f.epos)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.f.epos > dt2->shared->u.atomic.u.f.epos)
                        HGOTO_DONE(1);

                    if (dt1->shared->u.atomic.u.f.esize < dt2->shared->u.atomic.u.f.esize)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.f.esize > dt2->shared->u.atomic.u.f.esize)
                        HGOTO_DONE(1);

                    if (dt1->shared->u.atomic.u.f.ebias < dt2->shared->u.atomic.u.f.ebias)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.f.ebias > dt2->shared->u.atomic.u.f.ebias)
                        HGOTO_DONE(1);

                    if (dt1->shared->u.atomic.u.f.mpos < dt2->shared->u.atomic.u.f.mpos)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.f.mpos > dt2->shared->u.atomic.u.f.mpos)
                        HGOTO_DONE(1);

                    if (dt1->shared->u.atomic.u.f.msize < dt2->shared->u.atomic.u.f.msize)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.f.msize > dt2->shared->u.atomic.u.f.msize)
                        HGOTO_DONE(1);

                    if (dt1->shared->u.atomic.u.f.norm < dt2->shared->u.atomic.u.f.norm)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.f.norm > dt2->shared->u.atomic.u.f.norm)
                        HGOTO_DONE(1);

                    if (dt1->shared->u.atomic.u.f.pad < dt2->shared->u.atomic.u.f.pad)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.f.pad > dt2->shared->u.atomic.u.f.pad)
                        HGOTO_DONE(1);

                    break;

                case H5T_TIME: 
                    
                    break;

                case H5T_STRING:
                    if (dt1->shared->u.atomic.u.s.cset < dt2->shared->u.atomic.u.s.cset)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.s.cset > dt2->shared->u.atomic.u.s.cset)
                        HGOTO_DONE(1);

                    if (dt1->shared->u.atomic.u.s.pad < dt2->shared->u.atomic.u.s.pad)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.s.pad > dt2->shared->u.atomic.u.s.pad)
                        HGOTO_DONE(1);

                    break;

                case H5T_BITFIELD:
                    
                    break;

                case H5T_REFERENCE:
                    if (dt1->shared->u.atomic.u.r.rtype < dt2->shared->u.atomic.u.r.rtype)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.r.rtype > dt2->shared->u.atomic.u.r.rtype)
                        HGOTO_DONE(1);
                    if (dt1->shared->u.atomic.u.r.loc < dt2->shared->u.atomic.u.r.loc)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.r.loc > dt2->shared->u.atomic.u.r.loc)
                        HGOTO_DONE(1);
                    if (dt1->shared->u.atomic.u.r.file < dt2->shared->u.atomic.u.r.file)
                        HGOTO_DONE(-1);
                    if (dt1->shared->u.atomic.u.r.file > dt2->shared->u.atomic.u.r.file)
                        HGOTO_DONE(1);
                    break;

                case H5T_NO_CLASS:
                case H5T_OPAQUE:
                case H5T_COMPOUND:
                case H5T_ENUM:
                case H5T_VLEN:
                case H5T_ARRAY:
                case H5T_COMPLEX:
                case H5T_NCLASSES:
                default:
                    assert("not implemented yet" && 0);
                    break;
            }
            break;
    } 

done:
    if (NULL != idx1)
        H5MM_xfree(idx1);
    if (NULL != idx2)
        H5MM_xfree(idx2);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static bool
H5T__path_table_search(const H5T_t *src, const H5T_t *dst, int *idx, int *last_cmp)
{
    int  lt, rt, md;        
    int  cmp;               
    bool ret_value = false; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(0 != H5T_g.npaths);
    assert(src);
    assert(src->shared);
    assert(dst);
    assert(dst->shared);
    assert(idx);

    
    
    lt = md = 1;
    rt      = H5T_g.npaths;
    cmp     = -1;

    while (cmp && lt < rt) {
        md = (lt + rt) / 2;
        assert(H5T_g.path[md]);
        cmp = H5T_cmp(src, H5T_g.path[md]->src, false);
        if (0 == cmp)
            cmp = H5T_cmp(dst, H5T_g.path[md]->dst, false);
        if (cmp < 0)
            rt = md;
        else if (cmp > 0)
            lt = md + 1;
        else
            
            ret_value = true;
    }

    
    *idx = md;
    if (last_cmp)
        *last_cmp = cmp;

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5T_path_t *
H5T_path_find(const H5T_t *src, const H5T_t *dst)
{
    H5T_conv_func_t conv_func;        
    H5T_path_t     *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(src);
    assert(src->shared);
    assert(dst);
    assert(dst->shared);

    
    conv_func.is_app     = false;
    conv_func.u.lib_func = NULL;

    
    if (NULL == (ret_value = H5T__path_find_real(src, dst, NULL, &conv_func)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, NULL, "can't find datatype conversion path");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5T_path_t *
H5T__path_find_real(const H5T_t *src, const H5T_t *dst, const char *name, H5T_conv_func_t *conv)
{
    H5T_conv_ctx_t tmp_ctx      = {0};   
    H5T_path_t    *matched_path = NULL;  
    H5T_path_t    *path         = NULL;  
    bool           noop_conv    = false; 
    bool           new_path     = false; 
    bool new_api_func = false; 
    bool new_lib_func = false; 
    int  old_npaths;           
    int  last_cmp         = 0; 
    int  path_idx         = 0; 
    H5T_path_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(src);
    assert(src->shared);
    assert(dst);
    assert(dst->shared);

    
    if (H5CX_pushed() && (H5CX_get_dt_conv_cb(&tmp_ctx.u.init.cb_struct) < 0))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, NULL, "unable to get conversion exception callback");

    
    if ((0 == H5T_g.npaths) && (H5T__init_path_table() < 0))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to initialize type conversion path table");

    
    noop_conv =
        src->shared->force_conv == false && dst->shared->force_conv == false && 0 == H5T_cmp(src, dst, true);
    if (noop_conv)
        matched_path = H5T_g.path[0];
    else {
        
        if (H5T__path_table_search(src, dst, &path_idx, &last_cmp))
            matched_path = H5T_g.path[path_idx];
    }

    
    old_npaths = H5T_g.npaths;

    
    new_api_func = (matched_path && !noop_conv && conv->is_app && conv->u.app_func);
    new_lib_func = (matched_path && !noop_conv && !conv->is_app && conv->u.lib_func);

    
    new_path = !matched_path || new_api_func || (new_lib_func && !matched_path->is_hard);

    if (new_path) {
        if (NULL == (path = H5FL_CALLOC(H5T_path_t)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for type conversion path");
        if (name && *name) {
            strncpy(path->name, name, (size_t)H5T_NAMELEN - 1);
            path->name[H5T_NAMELEN - 1] = '\0';
        } 
        else
            snprintf(path->name, sizeof(path->name), "NONAME");
        if (NULL == (path->src = H5T_copy(src, H5T_COPY_ALL)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to copy datatype for conversion path");
        if (NULL == (path->dst = H5T_copy(dst, H5T_COPY_ALL)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to copy datatype for conversion path");
    } 
    else
        path = matched_path;

    
    if (new_path && H5T__path_find_init_new_path(path, src, dst, conv, &tmp_ctx) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to initialize new conversion path");

    
    if (!path->conv.u.app_func)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "no appropriate function for conversion path");

    
    if (old_npaths != H5T_g.npaths)
        if (H5T__path_table_search(src, dst, &path_idx, &last_cmp))
            matched_path = H5T_g.path[path_idx];

    
    if (matched_path && new_path) {
        assert(matched_path == H5T_g.path[path_idx]);

        tmp_ctx.u.free.src_type_id = H5I_INVALID_HID;
        tmp_ctx.u.free.dst_type_id = H5I_INVALID_HID;
        if (H5T__path_free(matched_path, &tmp_ctx) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFREE, NULL, "unable to free datatype conversion path");

        H5T_g.path[path_idx] = path;
    }
    else if (new_path) {
        
        if ((size_t)H5T_g.npaths >= H5T_g.apaths) {
            size_t       na = MAX(H5T_DEF_CONV_TABLE_SLOTS, 2 * H5T_g.apaths);
            H5T_path_t **x;

            if (NULL == (x = H5MM_realloc(H5T_g.path, na * sizeof(H5T_path_t *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
            H5T_g.apaths = na;
            H5T_g.path   = x;
        }

        
        assert(last_cmp != 0);
        if (last_cmp > 0)
            path_idx++;
        memmove(H5T_g.path + path_idx + 1, H5T_g.path + path_idx,
                (size_t)(H5T_g.npaths - path_idx) * sizeof(H5T_path_t *));

        H5T_g.npaths++;
        H5T_g.path[path_idx] = path;
    }

    ret_value = path;

done:
    if (!ret_value && path && new_path) {
        if (path->src && (H5T_close_real(path->src) < 0))
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, NULL, "unable to close datatype");
        if (path->dst && (H5T_close_real(path->dst) < 0))
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, NULL, "unable to close datatype");
        path = H5FL_FREE(H5T_path_t, path);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__init_path_table(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    if (0 != H5T_g.npaths)
        HGOTO_ERROR(H5E_DATATYPE, H5E_ALREADYINIT, FAIL,
                    "datatype conversion path table is already initialized");

    if (NULL == (H5T_g.path = H5MM_calloc(H5T_DEF_CONV_TABLE_SLOTS * sizeof(H5T_path_t *))))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL,
                    "memory allocation failed for type conversion path table");
    H5T_g.apaths = H5T_DEF_CONV_TABLE_SLOTS;

    
    if (NULL == (H5T_g.path[0] = H5FL_CALLOC(H5T_path_t)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "memory allocation failed for no-op conversion path");

    
    snprintf(H5T_g.path[0]->name, sizeof(H5T_g.path[0]->name), "no-op");
    H5T_g.path[0]->conv.is_app     = false;
    H5T_g.path[0]->conv.u.lib_func = H5T__conv_noop;
    H5T_g.path[0]->cdata.command   = H5T_CONV_INIT;

    if (H5T__conv_noop(NULL, NULL, &(H5T_g.path[0]->cdata), NULL, 0, 0, 0, NULL, NULL) < 0) {
#ifdef H5T_DEBUG
        if (H5DEBUG(T))
            Rfprintf(H5DEBUG(T), "H5T: unable to initialize no-op conversion function (ignored)\n");
#endif
        
        if (H5E_clear_stack() < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRESET, FAIL, "unable to clear current error stack");
    } 

    H5T_g.path[0]->is_noop = true;

    
    H5T_g.npaths = 1;

done:
    if (ret_value < 0) {
        if (H5T_g.path)
            H5FL_FREE(H5T_path_t, H5T_g.path[0]);
        H5MM_free(H5T_g.path);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__path_find_init_new_path(H5T_path_t *path, const H5T_t *src, const H5T_t *dst, H5T_conv_func_t *conv,
                             H5T_conv_ctx_t *conv_ctx)
{
    H5T_t *tmp_stype = NULL;            
    H5T_t *tmp_dtype = NULL;            
    hid_t  src_id    = H5I_INVALID_HID; 
    hid_t  dst_id    = H5I_INVALID_HID; 
    herr_t status    = SUCCEED;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(path);
    assert(conv);
    assert(conv_ctx);
    assert(NULL == path->conv.u.app_func);

    
    if (conv->u.app_func) {
        path->cdata.command = H5T_CONV_INIT;
        if (conv->is_app) {
            
            if (path->src && (NULL == (tmp_stype = H5T_copy(path->src, H5T_COPY_ALL))))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "unable to copy source datatype");
            if (path->dst && (NULL == (tmp_dtype = H5T_copy(path->dst, H5T_COPY_ALL))))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "unable to copy destination datatype");

            if (tmp_stype && ((src_id = H5I_register(H5I_DATATYPE, tmp_stype, false)) < 0))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL,
                            "unable to register ID for source datatype");
            if (tmp_dtype && ((dst_id = H5I_register(H5I_DATATYPE, tmp_dtype, false)) < 0))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL,
                            "unable to register ID for destination datatype");

            
            H5_BEFORE_USER_CB(FAIL)
                {
                    status = (conv->u.app_func)(src_id, dst_id, &(path->cdata), 0, 0, 0, NULL, NULL,
                                                H5CX_get_dxpl());
                }
            H5_AFTER_USER_CB(FAIL)
        }
        else
            status = (conv->u.lib_func)(path->src, path->dst, &(path->cdata), conv_ctx, 0, 0, 0, NULL, NULL);

        if (status < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to initialize conversion function");

        if (src_id >= 0) {
            if (H5I_dec_ref(src_id) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement reference on temporary ID");
            src_id    = H5I_INVALID_HID;
            tmp_stype = NULL;
        }
        if (dst_id >= 0) {
            if (H5I_dec_ref(dst_id) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement reference on temporary ID");
            dst_id    = H5I_INVALID_HID;
            tmp_dtype = NULL;
        }

        path->conv    = *conv;
        path->is_hard = true;
    }

    
    assert(path->conv.u.app_func || (src && dst));
    for (int i = H5T_g.nsoft - 1; i >= 0 && !path->conv.u.app_func; --i) {
        bool path_init_error = false;

        if (src->shared->type != H5T_g.soft[i].src || dst->shared->type != H5T_g.soft[i].dst)
            continue;

        path->cdata.command = H5T_CONV_INIT;
        if (H5T_g.soft[i].conv.is_app) {
            
            if (NULL == (tmp_stype = H5T_copy(path->src, H5T_COPY_ALL)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "unable to copy source datatype");
            if (NULL == (tmp_dtype = H5T_copy(path->dst, H5T_COPY_ALL)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "unable to copy destination datatype");

            if ((src_id = H5I_register(H5I_DATATYPE, tmp_stype, false)) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL,
                            "unable to register ID for source datatype");
            if ((dst_id = H5I_register(H5I_DATATYPE, tmp_dtype, false)) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL,
                            "unable to register ID for destination datatype");

            
            H5_BEFORE_USER_CB(FAIL)
                {
                    status = (H5T_g.soft[i].conv.u.app_func)(src_id, dst_id, &(path->cdata), 0, 0, 0, NULL,
                                                             NULL, H5CX_get_dxpl());
                }
            H5_AFTER_USER_CB(FAIL)
        }
        else
            status = (H5T_g.soft[i].conv.u.lib_func)(path->src, path->dst, &(path->cdata), conv_ctx, 0, 0, 0,
                                                     NULL, NULL);

        if (status < 0) {
            memset(&(path->cdata), 0, sizeof(H5T_cdata_t));
            
            if (H5E_clear_stack() < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRESET, FAIL, "unable to clear current error stack");
            path_init_error = true;
        }

        
        if (!path_init_error) {
            strncpy(path->name, H5T_g.soft[i].name, (size_t)H5T_NAMELEN);
            path->name[H5T_NAMELEN - 1] = '\0';
            path->conv                  = H5T_g.soft[i].conv;
            path->is_hard               = false;
        }

        if (src_id >= 0) {
            if (H5I_dec_ref(src_id) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement reference on temporary ID");
            src_id    = H5I_INVALID_HID;
            tmp_stype = NULL;
        }
        if (dst_id >= 0) {
            if (H5I_dec_ref(dst_id) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement reference on temporary ID");
            dst_id    = H5I_INVALID_HID;
            tmp_dtype = NULL;
        }
    }

done:
    if (src_id >= 0) {
        if (H5I_dec_ref(src_id) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement reference on temporary ID");
    }
    else if (tmp_stype) {
        if (H5T_close(tmp_stype) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary datatype");
    }
    if (dst_id >= 0) {
        if (H5I_dec_ref(dst_id) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTDEC, FAIL, "can't decrement reference on temporary ID");
    }
    else if (tmp_dtype) {
        if (H5T_close(tmp_dtype) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary datatype");
    }

    FUNC_LEAVE_NOAPI(ret_value)
}

static herr_t
H5T__path_free(H5T_path_t *path, H5T_conv_ctx_t *conv_ctx)
{
    herr_t status    = SUCCEED;
    int    nprint    = 0;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(path);
    assert(conv_ctx);

    if (path->conv.u.app_func) {
        H5T__print_path_stats(path, &nprint);

        path->cdata.command = H5T_CONV_FREE;

        if (path->conv.is_app) {
            
            H5_BEFORE_USER_CB_NOERR(FAIL)
                {
                    status =
                        (path->conv.u.app_func)(conv_ctx->u.free.src_type_id, conv_ctx->u.free.dst_type_id,
                                                &(path->cdata), 0, 0, 0, NULL, NULL, H5CX_get_dxpl());
                }
            H5_AFTER_USER_CB_NOERR(FAIL)
        }
        else
            status =
                (path->conv.u.lib_func)(path->src, path->dst, &(path->cdata), conv_ctx, 0, 0, 0, NULL, NULL);

        if (status < 0) {
            
            if (H5E_clear_stack() < 0)
                
                HDONE_ERROR(H5E_DATATYPE, H5E_CANTRESET, FAIL, "unable to clear current error stack");

#ifdef H5T_DEBUG
            if (H5DEBUG(T)) {
                Rfprintf(H5DEBUG(T), "H5T: conversion function 0x%016zx free failed for %s (ignored)\n",
                        path->conv.is_app ? (size_t)path->conv.u.app_func : (size_t)path->conv.u.lib_func,
                        path->name);
            }
#endif
        }
    }

    if (path->src && (H5T_close_real(path->src) < 0))
        
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "unable to close source datatype");
    if (path->dst && (H5T_close_real(path->dst) < 0))
        
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "unable to close destination datatype");

    path = H5FL_FREE(H5T_path_t, path);

    FUNC_LEAVE_NOAPI(ret_value)
}

static bool
H5T_path_match(H5T_path_t *path, H5T_pers_t pers, const char *name, H5T_t *src, H5T_t *dst,
               H5VL_object_t *owned_vol_obj, H5T_conv_t func)
{
    bool ret_value = true;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(path);

    if (
        
        ((H5T_PERS_SOFT == pers && path->is_hard) || (H5T_PERS_HARD == pers && !path->is_hard)) ||

        
        (name && *name && strcmp(name, path->name) != 0) ||

        
        (src && H5T_cmp(src, path->src, false)) || (dst && H5T_cmp(dst, path->dst, false)) ||

        
        (owned_vol_obj && (H5T_path_match_find_type_with_volobj(path->src, owned_vol_obj) == false) &&
         (H5T_path_match_find_type_with_volobj(path->dst, owned_vol_obj) == false)) ||

        
        (func && func != path->conv.u.app_func))
        ret_value = false;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static bool
H5T_path_match_find_type_with_volobj(const H5T_t *datatype, const H5VL_object_t *owned_vol_obj)
{
    bool ret_value = false;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(datatype);
    assert(owned_vol_obj);

    ret_value = (datatype->shared->owned_vol_obj == owned_vol_obj);
    if (!ret_value) {
        switch (datatype->shared->type) {
            case H5T_COMPOUND:
                for (unsigned i = 0; i < datatype->shared->u.compnd.nmembs; i++) {
                    if (ret_value)
                        break;
                    ret_value = H5T_path_match_find_type_with_volobj(datatype->shared->u.compnd.memb[i].type,
                                                                     owned_vol_obj);
                }
                break;

            case H5T_VLEN:
                
                if (datatype->shared->parent)
                    ret_value = H5T_path_match_find_type_with_volobj(datatype->shared->parent, owned_vol_obj);
                break;

            case H5T_ARRAY:
                
                if (datatype->shared->parent)
                    ret_value = H5T_path_match_find_type_with_volobj(datatype->shared->parent, owned_vol_obj);
                break;

            case H5T_INTEGER:
            case H5T_FLOAT:
            case H5T_TIME:
            case H5T_STRING:
            case H5T_BITFIELD:
            case H5T_OPAQUE:
            case H5T_REFERENCE: 
            case H5T_ENUM:
            case H5T_COMPLEX:
            case H5T_NO_CLASS: 
            case H5T_NCLASSES: 
            default:
                ret_value = false;
                break;
        }
    }

    FUNC_LEAVE_NOAPI(ret_value)
}

bool
H5T_path_noop(const H5T_path_t *p)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(p);

    FUNC_LEAVE_NOAPI(p->is_noop || (p->is_hard && 0 == H5T_cmp(p->src, p->dst, false)))
} 

bool
H5T_noop_conv(const H5T_t *src, const H5T_t *dst)
{
    bool ret_value = false; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(src);
    assert(src->shared);
    assert(dst);
    assert(dst->shared);

    
    if (src->shared->force_conv == false && dst->shared->force_conv == false &&
        0 == H5T_cmp(src, dst, true)) {
        ret_value = true;
    } 
    else {
        int idx = 0; 

        
        if (H5T__path_table_search(src, dst, &idx, NULL))
            ret_value = H5T_path_noop(H5T_g.path[idx]);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5T_subset_info_t *
H5T_path_compound_subset(const H5T_path_t *p)
{
    H5T_subset_info_t *ret_value = NULL;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(p);

    
    if (!p->conv.is_app && (p->conv.u.lib_func == H5T__conv_struct))
        ret_value = H5T__conv_struct_subset(&(p->cdata));

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5T_bkg_t
H5T_path_bkg(const H5T_path_t *p)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(p);

    FUNC_LEAVE_NOAPI(p->cdata.need_bkg)
} 

static htri_t
H5T__compiler_conv(H5T_t *src, H5T_t *dst)
{
    H5T_path_t *path;
    htri_t      ret_value = FAIL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (path = H5T_path_find(src, dst)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_NOTFOUND, FAIL, "conversion function not found");

    ret_value = (htri_t)path->is_hard;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T_convert(H5T_path_t *tpath, const H5T_t *src_type, const H5T_t *dst_type, size_t nelmts, size_t buf_stride,
            size_t bkg_stride, void *buf, void *bkg)
{
    H5T_conv_ctx_t conv_ctx = {0};
#ifdef H5T_DEBUG
    H5_timer_t timer = {0}; 
#endif
    hid_t  src_type_id = H5I_INVALID_HID;
    hid_t  dst_type_id = H5I_INVALID_HID;
    herr_t ret_value   = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

#ifdef H5T_DEBUG
    if (H5DEBUG(T)) {
        
        H5_timer_init(&timer);
        H5_timer_start(&timer);
    } 
#endif

    
    if (H5CX_get_dt_conv_cb(&conv_ctx.u.conv.cb_struct) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "unable to get conversion exception callback");

    
    if (tpath->conv.is_app || conv_ctx.u.conv.cb_struct.func) {
        if ((src_type_id = H5I_register(H5I_DATATYPE, src_type, false)) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL, "unable to register ID for source datatype");
        if ((dst_type_id = H5I_register(H5I_DATATYPE, dst_type, false)) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL,
                        "unable to register ID for destination datatype");

        if (tpath->conv.is_app)
            conv_ctx.u.conv.dxpl_id = H5CX_get_dxpl();
    }
    conv_ctx.u.conv.src_type_id = src_type_id;
    conv_ctx.u.conv.dst_type_id = dst_type_id;

    if (H5T_convert_with_ctx(tpath, src_type, dst_type, &conv_ctx, nelmts, buf_stride, bkg_stride, buf, bkg) <
        0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "datatype conversion failed");

done:
    
    if ((src_type_id >= 0) && (NULL == H5I_remove(src_type_id)))
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "can't decrement temporary datatype ID");
    if ((dst_type_id >= 0) && (NULL == H5I_remove(dst_type_id)))
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTFREE, FAIL, "can't decrement temporary datatype ID");

#ifdef H5T_DEBUG
    if (H5DEBUG(T)) {
        
        H5_timer_stop(&timer);

        
        H5_timer_get_times(timer, &tpath->stats.times);

        
        tpath->stats.ncalls++;
        tpath->stats.nelmts += nelmts;
    } 
#endif

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T_convert_with_ctx(H5T_path_t *tpath, const H5T_t *src_type, const H5T_t *dst_type,
                     const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride, size_t bkg_stride,
                     void *buf, void *bkg)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    tpath->cdata.command = H5T_CONV_CONV;
    if (tpath->conv.is_app) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (tpath->conv.u.app_func)(
                    conv_ctx->u.conv.src_type_id, conv_ctx->u.conv.dst_type_id, &(tpath->cdata), nelmts,
                    buf_stride, bkg_stride, buf, bkg, conv_ctx->u.conv.dxpl_id);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "datatype conversion failed");
    } 
    else if ((tpath->conv.u.lib_func)(src_type, dst_type, &(tpath->cdata), conv_ctx, nelmts, buf_stride,
                                      bkg_stride, buf, bkg) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "datatype conversion failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5O_loc_t *
H5T_oloc(H5T_t *dt)
{
    H5O_loc_t *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    assert(dt);

    switch (dt->shared->state) {
        case H5T_STATE_TRANSIENT:
        case H5T_STATE_RDONLY:
        case H5T_STATE_IMMUTABLE:
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "not a named datatype");
        case H5T_STATE_NAMED:
        case H5T_STATE_OPEN:
            assert(dt->sh_loc.type == H5O_SHARE_TYPE_COMMITTED);
            ret_value = &dt->oloc;
            break;
        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, NULL, "invalid datatype state");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5G_name_t *
H5T_nameof(H5T_t *dt)
{
    H5G_name_t *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    assert(dt);

    switch (dt->shared->state) {
        case H5T_STATE_TRANSIENT:
        case H5T_STATE_RDONLY:
        case H5T_STATE_IMMUTABLE:
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "not a named datatype");
        case H5T_STATE_NAMED:
        case H5T_STATE_OPEN:
            ret_value = &(dt->path);
            break;
        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, NULL, "invalid datatype state");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5T_is_immutable(const H5T_t *dt)
{
    htri_t ret_value = false;

    FUNC_ENTER_NOAPI(FAIL)

    assert(dt);

    if (dt->shared->state == H5T_STATE_IMMUTABLE)
        ret_value = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

htri_t
H5T_is_named(const H5T_t *dt)
{
    htri_t ret_value = false;

    FUNC_ENTER_NOAPI(FAIL)

    assert(dt);

    if (dt->vol_obj)
        ret_value = true;
    else
        ret_value = (H5T_STATE_OPEN == dt->shared->state || H5T_STATE_NAMED == dt->shared->state);

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

herr_t
H5T_convert_committed_datatype(H5T_t *dt, H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(dt);
    assert(f);

    if (H5T_is_named(dt) && (dt->sh_loc.file != f)) {
        assert(dt->sh_loc.type == H5O_SHARE_TYPE_COMMITTED);

        H5O_msg_reset_share(H5O_DTYPE_ID, dt);
        if (H5O_loc_free(&dt->oloc) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRESET, FAIL, "unable to initialize location");
        if (H5G_name_free(&dt->path) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, FAIL, "unable to reset path");

        
        if (NULL != dt->vol_obj) {
            H5VL_object_t *vol_obj = dt->vol_obj;

            
            if (H5VL_datatype_close(vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CLOSEERROR, FAIL, "unable to close datatype");

            
            if (H5VL_free_object(vol_obj) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTDEC, FAIL, "unable to free VOL object");
            dt->vol_obj = NULL;
        } 

        dt->shared->state = H5T_STATE_TRANSIENT;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5R_type_t
H5T_get_ref_type(const H5T_t *dt)
{
    H5R_type_t ret_value = H5R_BADTYPE;

    FUNC_ENTER_NOAPI(H5R_BADTYPE)

    assert(dt);

    if (dt->shared->type == H5T_REFERENCE)
        ret_value = dt->shared->u.atomic.u.r.rtype;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5T_is_sensible(const H5T_t *dt)
{
    htri_t ret_value = FAIL; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(dt);

    switch (dt->shared->type) {
        case H5T_COMPOUND:
            
            if (dt->shared->u.compnd.nmembs > 0)
                ret_value = true;
            else
                ret_value = false;
            break;

        case H5T_ENUM:
            
            if (dt->shared->u.enumer.nmembs > 0)
                ret_value = true;
            else
                ret_value = false;
            break;

        case H5T_NO_CLASS:
        case H5T_INTEGER:
        case H5T_FLOAT:
        case H5T_TIME:
        case H5T_STRING:
        case H5T_BITFIELD:
        case H5T_OPAQUE:
        case H5T_REFERENCE:
        case H5T_VLEN:
        case H5T_ARRAY:
        case H5T_COMPLEX:
        case H5T_NCLASSES:
        default:
            
            ret_value = true;
            break;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

htri_t
H5T_set_loc(H5T_t *dt, H5VL_object_t *file, H5T_loc_t loc)
{
    htri_t   changed;       
    htri_t   ret_value = 0; 
    unsigned i;             
    size_t   old_size;      

    FUNC_ENTER_NOAPI(FAIL)

    assert(dt);
    assert(loc >= H5T_LOC_BADLOC && loc < H5T_LOC_MAXLOC);

    
    if (dt->shared->force_conv) {
        
        switch (dt->shared->type) {
            case H5T_ARRAY: 
                
                
                if (dt->shared->parent->shared->force_conv &&
                    H5T_IS_COMPOSITE(dt->shared->parent->shared->type)) {
                    
                    old_size = dt->shared->parent->shared->size;

                    
                    if ((changed = H5T_set_loc(dt->shared->parent, file, loc)) < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "Unable to set VL location");
                    if (changed > 0)
                        ret_value = changed;

                    
                    if (old_size != dt->shared->parent->shared->size) {
                        
                        dt->shared->size = dt->shared->u.array.nelem * dt->shared->parent->shared->size;
                    } 
                }     
                break;

            case H5T_COMPOUND: 
            {
                ssize_t accum_change = 0; 

                
                H5T__sort_value(dt, NULL);

                for (i = 0; i < dt->shared->u.compnd.nmembs; i++) {
                    H5T_t *memb_type; 

                    
                    if ((accum_change < 0) && ((ssize_t)dt->shared->u.compnd.memb[i].offset < accum_change))
                        HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "invalid field size in datatype");

                    
                    dt->shared->u.compnd.memb[i].offset += (size_t)accum_change;

                    
                    memb_type = dt->shared->u.compnd.memb[i].type;

                    
                    
                    if (memb_type->shared->force_conv && H5T_IS_COMPOSITE(memb_type->shared->type)) {
                        
                        old_size = memb_type->shared->size;

                        
                        if ((changed = H5T_set_loc(memb_type, file, loc)) < 0)
                            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "Unable to set VL location");
                        if (changed > 0)
                            ret_value = changed;

                        
                        if (old_size != memb_type->shared->size) {

                            
                            if (0 == old_size)
                                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL,
                                            "old_size of zero would cause division by zero");

                            
                            dt->shared->u.compnd.memb[i].size =
                                (dt->shared->u.compnd.memb[i].size * memb_type->shared->size) / old_size;

                            
                            accum_change += (ssize_t)(memb_type->shared->size - old_size);
                        } 
                    }     
                }         

                
                if ((accum_change < 0) && ((ssize_t)dt->shared->size < accum_change))
                    HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "invalid field size in datatype");

                
                dt->shared->size += (size_t)accum_change;
            } break;

            case H5T_VLEN: 
                
                
                if (dt->shared->parent->shared->force_conv &&
                    H5T_IS_COMPOSITE(dt->shared->parent->shared->type) &&
                    (dt->shared->parent->shared->type != H5T_REFERENCE)) {
                    if ((changed = H5T_set_loc(dt->shared->parent, file, loc)) < 0)
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "Unable to set VL location");
                    if (changed > 0)
                        ret_value = changed;
                } 

                
                if ((changed = H5T__vlen_set_loc(dt, file, loc)) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "Unable to set VL location");
                if (changed > 0)
                    ret_value = changed;
                break;

            case H5T_REFERENCE:
                
                if ((ret_value = H5T__ref_set_loc(dt, file, loc)) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "Unable to set reference location");
                break;

            case H5T_NO_CLASS:
            case H5T_INTEGER:
            case H5T_FLOAT:
            case H5T_TIME:
            case H5T_STRING:
            case H5T_BITFIELD:
            case H5T_OPAQUE:
            case H5T_ENUM:
            case H5T_COMPLEX:
            case H5T_NCLASSES:
            default:
                break;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5T_is_relocatable(const H5T_t *dt)
{
    htri_t ret_value = false;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dt);

    
    if (H5T_detect_class(dt, H5T_VLEN, false) || H5T_detect_class(dt, H5T_REFERENCE, false))
        ret_value = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static bool
H5T__detect_vlen_ref(const H5T_t *dt)
{
    unsigned u;                 
    bool     ret_value = false; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dt);

    
    
    if (H5T_REFERENCE == dt->shared->type && !dt->shared->u.atomic.u.r.opaque)
        HGOTO_DONE(true);

    
    switch (dt->shared->type) {
        case H5T_COMPOUND:
            
            for (u = 0; u < dt->shared->u.compnd.nmembs; u++)
                
                if (H5T__detect_vlen_ref(dt->shared->u.compnd.memb[u].type))
                    HGOTO_DONE(true);
            break;

        case H5T_ARRAY:
        case H5T_VLEN:
        case H5T_ENUM:
            HGOTO_DONE(H5T__detect_vlen_ref(dt->shared->parent));
            break;

        case H5T_NO_CLASS:
        case H5T_INTEGER:
        case H5T_FLOAT:
        case H5T_TIME:
        case H5T_STRING:
        case H5T_BITFIELD:
        case H5T_OPAQUE:
        case H5T_REFERENCE:
        case H5T_COMPLEX:
        case H5T_NCLASSES:
        default:
            break;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5T_is_vl_storage(const H5T_t *dt)
{
    htri_t ret_value = false;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dt);

    
    if (H5T_detect_class(dt, H5T_VLEN, false))
        ret_value = true;
    else if (H5T_detect_class(dt, H5T_REFERENCE, false))
        ret_value = H5T__detect_vlen_ref(dt);
    else
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__upgrade_version_cb(H5T_t *dt, void *op_value)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dt);
    assert(op_value);

    
    switch (dt->shared->type) {
        case H5T_COMPOUND:
        case H5T_ARRAY:
        case H5T_ENUM:
            
            if (*(unsigned *)op_value > dt->shared->version)
                dt->shared->version = *(unsigned *)op_value;
            break;

        case H5T_VLEN:
            if (dt->shared->parent->shared->version > dt->shared->version)
                dt->shared->version = dt->shared->parent->shared->version;
            break;

        case H5T_COMPLEX:
            if (dt->shared->parent->shared->version > dt->shared->version)
                dt->shared->version = dt->shared->parent->shared->version;
            break;

        case H5T_NO_CLASS:
        case H5T_INTEGER:
        case H5T_FLOAT:
        case H5T_TIME:
        case H5T_STRING:
        case H5T_BITFIELD:
        case H5T_OPAQUE:
        case H5T_REFERENCE:
        case H5T_NCLASSES:
        default:
            break;
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5T__upgrade_version(H5T_t *dt, unsigned new_version)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dt);

    
    if (H5T__visit(dt, (H5T_VISIT_SIMPLE | H5T_VISIT_COMPOSITE_LAST), H5T__upgrade_version_cb, &new_version) <
        0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADITER, FAIL, "iteration to upgrade datatype encoding version failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T_set_version(H5F_t *f, H5T_t *dt)
{
    unsigned vers;                
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(dt);

    vers = H5O_dtype_ver_bounds[H5F_LOW_BOUND(f)];
    if (vers > dt->shared->version) {
        
        if (H5T__upgrade_version(dt, vers) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "can't upgrade datatype encoding");
    }

    
    if (dt->shared->version > H5O_dtype_ver_bounds[H5F_HIGH_BOUND(f)])
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADRANGE, FAIL, "Datatype version out of bounds");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T_patch_file(H5T_t *dt, H5F_t *f)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dt);
    assert(f);

    if (H5T_STATE_OPEN == dt->shared->state || H5T_STATE_NAMED == dt->shared->state) {
        dt->oloc.file   = f;
        dt->sh_loc.file = f;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T_patch_vlen_file(H5T_t *dt, H5VL_object_t *file)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(dt);
    assert(dt->shared);
    assert(file);

    if ((dt->shared->type == H5T_VLEN) && dt->shared->u.vlen.file != file)
        dt->shared->u.vlen.file = file;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5T_own_vol_obj(H5T_t *dt, H5VL_object_t *vol_obj)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dt);
    assert(dt->shared);
    assert(vol_obj);

    
    if (dt->shared->owned_vol_obj && H5VL_free_object(dt->shared->owned_vol_obj) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "unable to close owned VOL object");

    
    dt->shared->owned_vol_obj = vol_obj;
    (void)H5VL_object_inc_rc(vol_obj);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5T__get_path_table_npaths(void)
{
    int ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    ret_value = H5T_g.npaths;

    FUNC_LEAVE_NOAPI(ret_value)
}

bool
H5T_is_numeric_with_unusual_unused_bits(const H5T_t *dt)
{
    bool ret_value = false;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(dt);
    assert(dt->shared);

    
    if (H5T_INTEGER == dt->shared->type || H5T_FLOAT == dt->shared->type ||
        H5T_BITFIELD == dt->shared->type) {

        
        if (dt->shared->size > 1 && dt->shared->u.atomic.prec < (dt->shared->size * 8))
            
            ret_value =
                (dt->shared->size * 8) > (2 * (dt->shared->u.atomic.prec + dt->shared->u.atomic.offset));
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 
