/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Mmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Mpkg.h"      
#include "H5Eprivate.h"  
#include "H5ESprivate.h" 
#include "H5Iprivate.h"  
#include "H5VLprivate.h" 

static herr_t H5M__close_cb(void *map_vol_obj, void **request);

#ifdef H5_HAVE_MAP_API
static hid_t  H5M__create_api_common(hid_t loc_id, const char *name, hid_t key_type_id, hid_t val_type_id,
                                     hid_t lcpl_id, hid_t mcpl_id, hid_t mapl_id, void **token_ptr,
                                     H5VL_object_t **_vol_obj_ptr);
static hid_t  H5M__open_api_common(hid_t loc_id, const char *name, hid_t mapl_id, void **token_ptr,
                                   H5VL_object_t **_vol_obj_ptr);
static herr_t H5M__put_api_common(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id,
                                  const void *value, hid_t dxpl_id, void **token_ptr,
                                  H5VL_object_t **_vol_obj_ptr);
static herr_t H5M__get_api_common(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id,
                                  void *value, hid_t dxpl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr);
#endif 

bool H5_PKG_INIT_VAR = false;

static const H5I_class_t H5I_MAP_CLS[1] = {{
    H5I_MAP,                  
    0,                        
    0,                        
    (H5I_free_t)H5M__close_cb 
}};

static bool H5M_top_package_initialize_s = false;

herr_t
H5M_init(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)
    

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5M__init_package(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5I_register_type(H5I_MAP_CLS) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTINIT, FAIL, "unable to initialize interface");

    
    H5M_top_package_initialize_s = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5M_top_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5M_top_package_initialize_s) {
        if (H5I_nmembers(H5I_MAP) > 0) {
            (void)H5I_clear_type(H5I_MAP, false, false);
            n++;
        }

        
        if (0 == n)
            H5M_top_package_initialize_s = false;
    } 

    FUNC_LEAVE_NOAPI(n)
} 

int
H5M_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        
        assert(0 == H5I_nmembers(H5I_MAP));
        assert(false == H5M_top_package_initialize_s);

        
        n += (H5I_dec_type_ref(H5I_MAP) > 0);

        
        if (0 == n)
            H5_PKG_INIT_VAR = false;
    } 

    FUNC_LEAVE_NOAPI(n)
} 

static herr_t
H5M__close_cb(void *_map_vol_obj, void **request)
{
    H5VL_object_t *map_vol_obj = (H5VL_object_t *)_map_vol_obj;
    H5VL_optional_args_t vol_cb_args;         
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(map_vol_obj);

    
    vol_cb_args.op_type = H5VL_MAP_CLOSE;
    vol_cb_args.args    = NULL;

    
    if (H5VL_optional(map_vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, request) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CLOSEERROR, FAIL, "unable to close map");

    
    if (H5VL_free_object(map_vol_obj) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTDEC, FAIL, "unable to free VOL object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5_HAVE_MAP_API

static hid_t
H5M__create_api_common(hid_t loc_id, const char *name, hid_t key_type_id, hid_t val_type_id, hid_t lcpl_id,
                       hid_t mcpl_id, hid_t mapl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    void           *map         = NULL; 
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_optional_args_t vol_cb_args;                 
    H5VL_map_args_t      map_args;                    
    hid_t                ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_PACKAGE

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be an empty string");

    
    if (H5P_DEFAULT == lcpl_id)
        lcpl_id = H5P_LINK_CREATE_DEFAULT;
    else if (true != H5P_isa_class(lcpl_id, H5P_LINK_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "lcpl_id is not a link creation property list");

    
    if (H5P_DEFAULT == mcpl_id)
        mcpl_id = H5P_MAP_CREATE_DEFAULT;
    else if (true != H5P_isa_class(mcpl_id, H5P_MAP_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "mcpl_id is not a map create property list ID");

    
    if (H5VL_setup_acc_args(loc_id, H5P_CLS_MACC, true, &mapl_id, vol_obj_ptr, &map_args.create.loc_params) <
        0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTSET, H5I_INVALID_HID, "can't set object access arguments");
    map_args.create.name        = name;
    map_args.create.lcpl_id     = lcpl_id;
    map_args.create.key_type_id = key_type_id;
    map_args.create.val_type_id = val_type_id;
    map_args.create.mcpl_id     = mcpl_id;
    map_args.create.mapl_id     = mapl_id;
    map_args.create.map         = NULL;
    vol_cb_args.op_type         = H5VL_MAP_CREATE;
    vol_cb_args.args            = &map_args;

    
    if (H5VL_optional(*vol_obj_ptr, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, token_ptr) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTINIT, H5I_INVALID_HID, "unable to create map");
    map = map_args.create.map;

    
    if ((ret_value = H5VL_register(H5I_MAP, map, H5VL_OBJ_CONNECTOR(*vol_obj_ptr), true)) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register map handle");

done:
    
    if (H5I_INVALID_HID == ret_value) {
        
        vol_cb_args.op_type = H5VL_MAP_CLOSE;
        vol_cb_args.args    = NULL;

        if (map && H5VL_optional(*vol_obj_ptr, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release map");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Mcreate(hid_t loc_id, const char *name, hid_t key_type_id, hid_t val_type_id, hid_t lcpl_id, hid_t mcpl_id,
          hid_t mapl_id)
{
    hid_t ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if ((ret_value = H5M__create_api_common(loc_id, name, key_type_id, val_type_id, lcpl_id, mcpl_id, mapl_id,
                                            NULL, NULL)) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTCREATE, H5I_INVALID_HID, "unable to create map synchronously");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Mcreate_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id, const char *name,
                hid_t key_type_id, hid_t val_type_id, hid_t lcpl_id, hid_t mcpl_id, hid_t mapl_id,
                hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token;

    
    if ((ret_value = H5M__create_api_common(loc_id, name, key_type_id, val_type_id, lcpl_id, mcpl_id, mapl_id,
                                            token_ptr, &vol_obj)) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTCREATE, H5I_INVALID_HID, "unable to create map asynchronously");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE11(__func__, "*s*sIui*siiiiii", app_file, app_func, app_line, loc_id, name, key_type_id, val_type_id, lcpl_id, mcpl_id, mapl_id, es_id)) < 0) {
            
            if (H5I_dec_app_ref_always_close(ret_value) < 0)
                HDONE_ERROR(H5E_MAP, H5E_CANTDEC, H5I_INVALID_HID, "can't decrement count on map ID");
            HGOTO_ERROR(H5E_MAP, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");
        } 

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Mcreate_anon(hid_t loc_id, hid_t key_type_id, hid_t val_type_id, hid_t mcpl_id, hid_t mapl_id)
{
    void                *map     = NULL;              
    H5VL_object_t       *vol_obj = NULL;              
    H5VL_optional_args_t vol_cb_args;                 
    H5VL_map_args_t      map_args;                    
    hid_t                ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5P_DEFAULT == mcpl_id)
        mcpl_id = H5P_MAP_CREATE_DEFAULT;
    else if (true != H5P_isa_class(mcpl_id, H5P_MAP_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not map create property list ID");

    
    if (H5CX_set_apl(&mapl_id, H5P_CLS_MACC, loc_id, true) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTSET, H5I_INVALID_HID, "can't set access property list info");

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    

    
    map_args.create.loc_params.type     = H5VL_OBJECT_BY_SELF;
    map_args.create.loc_params.obj_type = H5I_get_type(loc_id);
    map_args.create.name                = NULL;
    map_args.create.lcpl_id             = H5P_LINK_CREATE_DEFAULT;
    map_args.create.key_type_id         = key_type_id;
    map_args.create.val_type_id         = val_type_id;
    map_args.create.mcpl_id             = mcpl_id;
    map_args.create.mapl_id             = mapl_id;
    map_args.create.map                 = NULL;
    vol_cb_args.op_type                 = H5VL_MAP_CREATE;
    vol_cb_args.args                    = &map_args;

    
    if (H5VL_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTINIT, H5I_INVALID_HID, "unable to create map");
    map = map_args.create.map;

    
    if ((ret_value = H5VL_register(H5I_MAP, map, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register map");

done:
    
    if (H5I_INVALID_HID == ret_value) {
        
        vol_cb_args.op_type = H5VL_MAP_CLOSE;
        vol_cb_args.args    = NULL;

        if (map && H5VL_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release map");
    } 

    FUNC_LEAVE_API(ret_value)
} 

static hid_t
H5M__open_api_common(hid_t loc_id, const char *name, hid_t mapl_id, void **token_ptr,
                     H5VL_object_t **_vol_obj_ptr)
{
    void           *map         = NULL; 
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_optional_args_t vol_cb_args;                 
    H5VL_map_args_t      map_args;                    
    hid_t                ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_PACKAGE

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be an empty string");

    
    if (H5VL_setup_acc_args(loc_id, H5P_CLS_MACC, false, &mapl_id, vol_obj_ptr, &map_args.open.loc_params) <
        0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTSET, H5I_INVALID_HID, "can't set object access arguments");
    map_args.open.name    = name;
    map_args.open.mapl_id = mapl_id;
    map_args.open.map     = NULL;
    vol_cb_args.op_type   = H5VL_MAP_OPEN;
    vol_cb_args.args      = &map_args;

    
    if (H5VL_optional(*vol_obj_ptr, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, token_ptr) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open map");
    map = map_args.open.map;

    
    if ((ret_value = H5VL_register(H5I_MAP, map, H5VL_OBJ_CONNECTOR(*vol_obj_ptr), true)) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTREGISTER, H5I_INVALID_HID, "can't register map ID");

done:
    
    if (H5I_INVALID_HID == ret_value) {
        
        vol_cb_args.op_type = H5VL_MAP_CLOSE;
        vol_cb_args.args    = NULL;

        if (map && H5VL_optional(*vol_obj_ptr, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release map");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Mopen(hid_t loc_id, const char *name, hid_t mapl_id)
{
    hid_t ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if ((ret_value = H5M__open_api_common(loc_id, name, mapl_id, NULL, NULL)) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTCREATE, H5I_INVALID_HID, "unable to open map synchronously");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Mopen_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id, const char *name,
              hid_t mapl_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token;

    
    if ((ret_value = H5M__open_api_common(loc_id, name, mapl_id, token_ptr, &vol_obj)) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTCREATE, H5I_INVALID_HID, "unable to open map asynchronously");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE7(__func__, "*s*sIui*sii", app_file, app_func, app_line, loc_id, name, mapl_id, es_id)) < 0) {
            
            if (H5I_dec_app_ref_always_close(ret_value) < 0)
                HDONE_ERROR(H5E_MAP, H5E_CANTDEC, H5I_INVALID_HID, "can't decrement count on map ID");
            HGOTO_ERROR(H5E_MAP, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");
        } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Mclose(hid_t map_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_MAP != H5I_get_type(map_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a map ID");

    
    if (H5I_dec_app_ref_always_close(map_id) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTDEC, FAIL, "can't decrement count on map ID");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Mclose_async(const char *app_file, const char *app_func, unsigned app_line, hid_t map_id, hid_t es_id)
{
    void             *token     = NULL;            
    void            **token_ptr = H5_REQUEST_NULL; 
    H5VL_object_t    *vol_obj   = NULL;            
    H5VL_connector_t *connector = NULL;            
    herr_t            ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5I_MAP != H5I_get_type(map_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a map ID");

    
    if (NULL == (vol_obj = H5VL_vol_object(map_id)))
        HGOTO_ERROR(H5E_MAP, H5E_CANTGET, FAIL, "can't get VOL object for map");

    
    if (H5ES_NONE != es_id) {
        
        connector = H5VL_OBJ_CONNECTOR(vol_obj);
        H5VL_conn_inc_rc(connector);

        
        token_ptr = &token;
    } 

    
    if (H5I_dec_app_ref_always_close_async(map_id, token_ptr) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTDEC, FAIL, "can't decrement count on dataset ID");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE5(__func__, "*s*sIuii", app_file, app_func, app_line, map_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_MAP, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    if (connector && H5VL_conn_dec_rc(connector) < 0)
        HDONE_ERROR(H5E_MAP, H5E_CANTDEC, FAIL, "can't decrement ref count on connector");

    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Mget_key_type(hid_t map_id)
{
    H5VL_object_t       *vol_obj;                     
    H5VL_optional_args_t vol_cb_args;                 
    H5VL_map_args_t      map_args;                    
    hid_t                ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(map_id, H5I_MAP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid map identifier");

    
    map_args.get.get_type                  = H5VL_MAP_GET_KEY_TYPE;
    map_args.get.args.get_key_type.type_id = H5I_INVALID_HID;
    vol_cb_args.op_type                    = H5VL_MAP_GET;
    vol_cb_args.args                       = &map_args;

    
    if (H5VL_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTGET, H5I_INVALID_HID, "unable to get key datatype");

    
    ret_value = map_args.get.args.get_key_type.type_id;

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Mget_val_type(hid_t map_id)
{
    H5VL_object_t       *vol_obj;                     
    H5VL_optional_args_t vol_cb_args;                 
    H5VL_map_args_t      map_args;                    
    hid_t                ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(map_id, H5I_MAP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid map identifier");

    
    map_args.get.get_type                  = H5VL_MAP_GET_VAL_TYPE;
    map_args.get.args.get_val_type.type_id = H5I_INVALID_HID;
    vol_cb_args.op_type                    = H5VL_MAP_GET;
    vol_cb_args.args                       = &map_args;

    
    if (H5VL_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTGET, H5I_INVALID_HID, "unable to get value datatype");

    
    ret_value = map_args.get.args.get_val_type.type_id;

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Mget_create_plist(hid_t map_id)
{
    H5VL_object_t       *vol_obj;                     
    H5VL_optional_args_t vol_cb_args;                 
    H5VL_map_args_t      map_args;                    
    hid_t                ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(map_id, H5I_MAP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid map identifier");

    
    map_args.get.get_type              = H5VL_MAP_GET_MCPL;
    map_args.get.args.get_mcpl.mcpl_id = H5I_INVALID_HID;
    vol_cb_args.op_type                = H5VL_MAP_GET;
    vol_cb_args.args                   = &map_args;

    
    if (H5VL_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTGET, H5I_INVALID_HID, "unable to get map creation properties");

    
    ret_value = map_args.get.args.get_mcpl.mcpl_id;

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Mget_access_plist(hid_t map_id)
{
    H5VL_object_t       *vol_obj;                     
    H5VL_optional_args_t vol_cb_args;                 
    H5VL_map_args_t      map_args;                    
    hid_t                ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(map_id, H5I_MAP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid map identifier");

    
    map_args.get.get_type              = H5VL_MAP_GET_MAPL;
    map_args.get.args.get_mapl.mapl_id = H5I_INVALID_HID;
    vol_cb_args.op_type                = H5VL_MAP_GET;
    vol_cb_args.args                   = &map_args;

    
    if (H5VL_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTGET, H5I_INVALID_HID, "unable to get map access properties");

    
    ret_value = map_args.get.args.get_mapl.mapl_id;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Mget_count(hid_t map_id, hsize_t *count , hid_t dxpl_id)
{
    H5VL_object_t       *vol_obj;             
    H5VL_optional_args_t vol_cb_args;         
    H5VL_map_args_t      map_args;            
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(map_id, H5I_MAP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid map identifier");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not xfer parms");

    
    map_args.get.get_type             = H5VL_MAP_GET_COUNT;
    map_args.get.args.get_count.count = 0;
    vol_cb_args.op_type               = H5VL_MAP_GET;
    vol_cb_args.args                  = &map_args;

    
    if (H5VL_optional(vol_obj, &vol_cb_args, dxpl_id, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTGET, H5I_INVALID_HID, "unable to get KV pair count for map");

    
    if (count)
        *count = map_args.get.args.get_count.count;

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5M__put_api_common(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id,
                    const void *value, hid_t dxpl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_optional_args_t vol_cb_args;                 
    H5VL_map_args_t      map_args;                    
    herr_t               ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    if (key_mem_type_id < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid key memory datatype ID");
    if (val_mem_type_id < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid value memory datatype ID");

    
    if (NULL == (*vol_obj_ptr = H5VL_vol_object_verify(map_id, H5I_MAP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "map_id is not a map ID");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not xfer parms");

    
    map_args.put.key_mem_type_id   = key_mem_type_id;
    map_args.put.key               = key;
    map_args.put.value_mem_type_id = val_mem_type_id;
    map_args.put.value             = value;
    vol_cb_args.op_type            = H5VL_MAP_PUT;
    vol_cb_args.args               = &map_args;

    
    if (H5VL_optional(*vol_obj_ptr, &vol_cb_args, dxpl_id, token_ptr) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTSET, FAIL, "unable to put key/value pair");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Mput(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id, const void *value,
       hid_t dxpl_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if ((ret_value = H5M__put_api_common(map_id, key_mem_type_id, key, val_mem_type_id, value, dxpl_id, NULL,
                                         NULL)) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTPUT, FAIL, "unable to put value to map synchronously");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Mput_async(const char *app_file, const char *app_func, unsigned app_line, hid_t map_id,
             hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id, const void *value, hid_t dxpl_id,
             hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token;

    
    if ((ret_value = H5M__put_api_common(map_id, key_mem_type_id, key, val_mem_type_id, value, dxpl_id,
                                         token_ptr, &vol_obj)) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTPUT, FAIL, "unable to put value to map asynchronously");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE10(__func__, "*s*sIuii*xi*xii", app_file, app_func, app_line, map_id, key_mem_type_id, key, val_mem_type_id, value, dxpl_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_MAP, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5M__get_api_common(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id, void *value,
                    hid_t dxpl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_optional_args_t vol_cb_args;                 
    H5VL_map_args_t      map_args;                    
    herr_t               ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    if (key_mem_type_id < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid key memory datatype ID");
    if (val_mem_type_id < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid value memory datatype ID");

    
    if (NULL == (*vol_obj_ptr = H5VL_vol_object_verify(map_id, H5I_MAP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "map_id is not a map ID");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not xfer parms");

    
    map_args.get_val.key_mem_type_id   = key_mem_type_id;
    map_args.get_val.key               = key;
    map_args.get_val.value_mem_type_id = val_mem_type_id;
    map_args.get_val.value             = value;
    vol_cb_args.op_type                = H5VL_MAP_GET_VAL;
    vol_cb_args.args                   = &map_args;

    
    if (H5VL_optional(*vol_obj_ptr, &vol_cb_args, dxpl_id, token_ptr) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTGET, FAIL, "unable to get value from map");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Mget(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id, void *value,
       hid_t dxpl_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if ((ret_value = H5M__get_api_common(map_id, key_mem_type_id, key, val_mem_type_id, value, dxpl_id, NULL,
                                         NULL)) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTGET, FAIL, "unable to get value from map synchronously");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Mget_async(const char *app_file, const char *app_func, unsigned app_line, hid_t map_id,
             hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id, void *value, hid_t dxpl_id,
             hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token;

    
    if ((ret_value = H5M__get_api_common(map_id, key_mem_type_id, key, val_mem_type_id, value, dxpl_id,
                                         token_ptr, &vol_obj)) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTGET, FAIL, "unable to get value from map asynchronously");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE10(__func__, "*s*sIuii*xi*xii", app_file, app_func, app_line, map_id, key_mem_type_id, key, val_mem_type_id, value, dxpl_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_MAP, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Mexists(hid_t map_id, hid_t key_mem_type_id, const void *key, bool *exists, hid_t dxpl_id)
{
    H5VL_object_t       *vol_obj = NULL;
    H5VL_optional_args_t vol_cb_args;         
    H5VL_map_args_t      map_args;            
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (key_mem_type_id < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid key memory datatype ID");

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(map_id, H5I_MAP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "map_id is not a map ID");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not xfer parms");

    
    map_args.exists.key_mem_type_id = key_mem_type_id;
    map_args.exists.key             = key;
    map_args.exists.exists          = false;
    vol_cb_args.op_type             = H5VL_MAP_EXISTS;
    vol_cb_args.args                = &map_args;

    
    if (H5VL_optional(vol_obj, &vol_cb_args, dxpl_id, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTGET, ret_value, "unable to check if key exists");

    
    if (exists)
        *exists = map_args.exists.exists;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Miterate(hid_t map_id, hsize_t *idx, hid_t key_mem_type_id, H5M_iterate_t op, void *op_data, hid_t dxpl_id)
{
    H5VL_object_t       *vol_obj = NULL;
    H5VL_optional_args_t vol_cb_args;         
    H5VL_map_args_t      map_args;            
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (key_mem_type_id < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid key memory datatype ID");
    if (!op)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no operator specified");

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(map_id, H5I_MAP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "map_id is not a map ID");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not xfer parms");

    
    map_args.specific.specific_type                    = H5VL_MAP_ITER;
    map_args.specific.args.iterate.loc_params.type     = H5VL_OBJECT_BY_SELF;
    map_args.specific.args.iterate.loc_params.obj_type = H5I_get_type(map_id);
    map_args.specific.args.iterate.idx                 = (idx ? *idx : 0);
    map_args.specific.args.iterate.key_mem_type_id     = key_mem_type_id;
    map_args.specific.args.iterate.op                  = op;
    map_args.specific.args.iterate.op_data             = op_data;
    vol_cb_args.op_type                                = H5VL_MAP_SPECIFIC;
    vol_cb_args.args                                   = &map_args;

    
    if ((ret_value = H5VL_optional(vol_obj, &vol_cb_args, dxpl_id, H5_REQUEST_NULL)) < 0)
        HERROR(H5E_MAP, H5E_BADITER, "unable to iterate over keys");

    
    if (idx)
        *idx = map_args.specific.args.iterate.idx;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Miterate_by_name(hid_t loc_id, const char *map_name, hsize_t *idx, hid_t key_mem_type_id, H5M_iterate_t op,
                   void *op_data, hid_t dxpl_id, hid_t lapl_id)
{
    H5VL_object_t       *vol_obj = NULL;
    H5VL_optional_args_t vol_cb_args;         
    H5VL_map_args_t      map_args;            
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!map_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "map_name parameter cannot be NULL");
    if (!*map_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "map_name parameter cannot be an empty string");
    if (key_mem_type_id < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid key memory datatype ID");
    if (!op)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no operator specified");

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not xfer parms");

    
    map_args.specific.specific_type                                        = H5VL_MAP_ITER;
    map_args.specific.args.iterate.loc_params.type                         = H5VL_OBJECT_BY_NAME;
    map_args.specific.args.iterate.loc_params.obj_type                     = H5I_get_type(loc_id);
    map_args.specific.args.iterate.loc_params.loc_data.loc_by_name.name    = map_name;
    map_args.specific.args.iterate.loc_params.loc_data.loc_by_name.lapl_id = lapl_id;
    map_args.specific.args.iterate.idx                                     = (idx ? *idx : 0);
    map_args.specific.args.iterate.key_mem_type_id                         = key_mem_type_id;
    map_args.specific.args.iterate.op                                      = op;
    map_args.specific.args.iterate.op_data                                 = op_data;
    vol_cb_args.op_type                                                    = H5VL_MAP_SPECIFIC;
    vol_cb_args.args                                                       = &map_args;

    
    if ((ret_value = H5VL_optional(vol_obj, &vol_cb_args, dxpl_id, H5_REQUEST_NULL)) < 0)
        HERROR(H5E_MAP, H5E_BADITER, "unable to ierate over keys");

    
    if (idx)
        *idx = map_args.specific.args.iterate.idx;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Mdelete(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t dxpl_id)
{
    H5VL_object_t       *vol_obj = NULL;
    H5VL_optional_args_t vol_cb_args;         
    H5VL_map_args_t      map_args;            
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (key_mem_type_id < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid key memory datatype ID");

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(map_id, H5I_MAP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "map_id is not a map ID");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not xfer parms");

    
    map_args.specific.specific_type                = H5VL_MAP_DELETE;
    map_args.specific.args.del.loc_params.type     = H5VL_OBJECT_BY_SELF;
    map_args.specific.args.del.loc_params.obj_type = H5I_get_type(map_id);
    map_args.specific.args.del.key_mem_type_id     = key_mem_type_id;
    map_args.specific.args.del.key                 = key;
    vol_cb_args.op_type                            = H5VL_MAP_SPECIFIC;
    vol_cb_args.args                               = &map_args;

    
    if (H5VL_optional(vol_obj, &vol_cb_args, dxpl_id, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_MAP, H5E_CANTSET, FAIL, "unable to delete key/value pair");

done:
    FUNC_LEAVE_API(ret_value)
} 

#endif 
