#' @title Function to calculate the true value of the mean frequency function.
#' @description Computes the theoretical mean frequency function for the composite endpoint under the assumed data-generating mechanisim.
#' The calculation uses the parameters stored in the simulated data.
#'
#' @param data A data frame generated by \code{Onesample.generate.sequential()} (optionally after applying \code{Apply.calendar.censoring()})
#' containing composite endpoint data in long format. Each subject may contribute multiple rows corresponding to recurrent events,
#' terminal event, or censoring.
#' @param t A numeric vector specifying event times (time since enrollment) at which the true value of the mean frequency is evaluated.
#'
#' @returns A numeric vector giving the true mean frequency evaluated at \code{t}.
#' @export
#' @references Mao L, Lin DY. Semiparametric regression for the weighted composite endpoint of recurrent and terminal events. \emph{Biostatistics}. 2016 Apr; \strong{17(2)} :390-403.
#'
#'
#' @examples
#' df <- Onesample.generate.sequential(size = 200, recruitment = 3, calendar = 5,
#' random.censor.rate = 0.05, seed = 1123)
#' True.mu(data = df, t = c(1.5, 2.9, 4.6))
True.mu <- function(data, t){

  lambda_0 <- unique(data$lambda_0)
  lambda_star <- unique(data$lambda_star)
  gamma_shape <- unique(data$gamma_shape)
  gamma_scale <- unique(data$gamma_scale)
  mu0 <- (lambda_0/lambda_star)*(1 - (lambda_star*t*gamma_scale + 1)^(-gamma_shape))
  return(mu0)
}
