% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eba.R
\name{run_eba}
\alias{run_eba}
\title{Extreme-Bounds Analysis (EBA) over Control-Variable Combinations}
\usage{
run_eba(DT, spec = "C", k_bma_table = NULL, seed = 123)
}
\arguments{
\item{DT}{A \code{data.table} or \code{data.frame} with the data passed
to \code{fit_one()}.}

\item{spec}{Character scalar; model specification (e.g.\ \code{"A"},
\code{"B"}, \code{"C"}, \code{"D"}) passed to \code{fit_one()}.}

\item{k_bma_table}{Optional object (typically a named list or
\code{list}-like structure) indexed by control-combination tags that
indicates for which combinations a BMA selection table already exists.
If \code{k_bma_table[[tag]]} is \code{NULL} or \code{bma_weights_*}
CSV is missing, the function falls back to a default fit with
\code{k = 2} and default horseshoe hyperparameters.}

\item{seed}{Integer; base random seed for the fits. For different
control combinations, the seed is jittered to avoid identical
pseudo-random sequences.}
}
\value{
A \code{data.frame} with the columns:
\itemize{
\item \code{name}: name of the covariate (design-matrix column).
\item \code{mean}: posterior mean of the corresponding coefficient.
\item \code{block}: block identifier (\code{"mu_I"}, \code{"pi_I"},
\code{"mu_C"}, \code{"pi_C"}).
\item \code{combo}: control-combination tag used for that fit.
}
}
\description{
Runs an Extreme-Bounds Analysis (EBA) over a predefined set of control
variable combinations, fitting (or re-fitting) the bivariate hurdle
model for each combination and extracting posterior mean coefficients
for all regression blocks (\code{mu_I}, \code{pi_I}, \code{mu_C},
\code{pi_C}).
}
\details{
The function assumes the existence of:
\itemize{
\item \code{control_combos}: a named object whose names are control
tags (e.g.\ \code{"None"}, \code{"X1+X2"}, \code{"X1+X3+X4"}),
defining which control sets to explore.
\item \code{dir_csv}: a character scalar with the directory where
CSV files will be read/written.
\item \code{fit_one()}: a function that fits a single bivariate
hurdle model and returns at least \code{$fit} (CmdStanR fit) and
\code{$des} (design matrices).
}

For each control-combination tag \code{tag}:
\itemize{
\item If a BMA weights file
\code{"bma_weights_spec<spec>_ctrl<tag>.csv"} exists in
\code{dir_csv} and \code{k_bma_table[[tag]]} is not \code{NULL},
the top-weighted row (highest \code{weight}) is used to select
\code{k} and horseshoe hyperparameters (\code{hs_tau0},
\code{hs_slab_scale}, \code{hs_slab_df}) for the fit.
\item Otherwise, the model is fit with \code{k = 2} and default
horseshoe hyperparameters.
\item Posterior means of the regression coefficients with prefixes
\code{"b_mu_I"}, \code{"b_pi_I"}, \code{"b_mu_C"},
\code{"b_pi_C"} are extracted and mapped back to the corresponding
column names of the design matrices.
}

All coefficient summaries are stacked into a single table and written
to \code{"eba_coefficients.csv"} in \code{dir_csv}.
}
\examples{
\donttest{
library(data.table)

# 1. Create a COMPLETE dummy dataset
# This satisfies ALL requirements of build_design() and fit_one()
DT <- data.table(
  year = 2000:2020,
  # Dependent variables (Raw)
  I = rpois(21, lambda = 4),
  C = rpois(21, lambda = 3),
  # Dependent variables (Standardized/Transformed - required by build_design)
  zI = rnorm(21),
  zC = rnorm(21),
  # Trend variables (required if include_trend=TRUE)
  t_norm = seq(-1, 1, length.out = 21),
  t_poly2 = seq(-1, 1, length.out = 21)^2,
  # Regime (required if include_regimes=TRUE)
  Regime = factor(sample(c("A", "B"), 21, replace = TRUE)),
  # Transition dummies (required if include_transitions=TRUE)
  # Specifically: trans_PS, trans_SF, trans_FC
  trans_PS = sample(0:1, 21, replace = TRUE),
  trans_SF = sample(0:1, 21, replace = TRUE),
  trans_FC = sample(0:1, 21, replace = TRUE),
  # Exposure offset (required by fit_one)
  log_exposure50 = rep(0, 21),
  # Control variables (used in this specific example)
  X1 = rnorm(21),
  X2 = rnorm(21),
  X3 = rnorm(21)
)

# 2. Define global objects required by run_eba
control_combos <- list(
  None      = character(0),
  "X1+X2"   = c("X1", "X2"),
  "X1+X2+X3"= c("X1", "X2", "X3")
)

# 3. Define global paths using tempdir()
tmp_dir <- tempdir()
dir_csv <- file.path(tmp_dir, "csv")
if (!dir.exists(dir_csv)) dir.create(dir_csv, recursive = TRUE)

# 4. Run the function
# Note: This will attempt to run Stan. If CmdStan is not configured,
# it might fail later, but the DATA error is fixed.
try({
  eba_tab <- run_eba(DT, spec = "C", k_bma_table = list(), seed = 123)
  print(head(eba_tab))
})
}

}
