#' Fit VARX model with diagnostics for I and C
#'
#' Estimates a bivariate VAR model for \code{I} and \code{C} with
#' exogenous covariates (VARX), and computes a set of standard
#' diagnostics (stability, serial correlation, normality, ARCH). The
#' fitted model and diagnostics are saved to disk and also returned.
#'
#' @param DT A \code{data.table} (or \code{data.frame}) containing at
#'   least the following columns:
#'   \itemize{
#'     \item \code{I}, \code{C}: endogenous variables for the VAR.
#'     \item \code{EconCycle}, \code{PopDensity}, \code{Epidemics},
#'           \code{Climate}, \code{War}, \code{t_norm}: exogenous
#'           regressors included in the VARX.
#'   }
#' @param p Integer; lag order of the VAR part (number of lags for
#'   \code{I} and \code{C}).
#'
#' @details
#' The endogenous vector is \eqn{y_t = (I_t, C_t)'} and the exogenous
#' regressors are:
#' \code{EconCycle}, \code{PopDensity}, \code{Epidemics},
#' \code{Climate}, \code{War}, \code{t_norm}. The model is fit using
#' \code{vars::VAR()} with \code{type = "const"} and the exogenous
#' matrix passed via \code{exogen}.
#'
#' After estimation, the following diagnostics from \pkg{vars} are
#' (attempted to be) computed:
#' \itemize{
#'   \item \code{vars::stability(fit, type = "OLS-CUSUM")} for stability.
#'   \item \code{vars::serial.test(fit, lags.pt = 10, type = "PT.asymptotic")}
#'         for serial correlation.
#'   \item \code{vars::normality.test(fit)} for residual normality.
#'   \item \code{vars::arch.test(fit, lags.multi = 5)} for ARCH effects.
#' }
#' Each diagnostic call is wrapped in \code{try()}, so if a diagnostic
#' fails, the corresponding element in the output will contain a
#' \code{"try-error"} instead of stopping the function.
#'
#' The result is saved as an RDS file named \code{"varx_fit.rds"} in
#' the directory specified by a global object \code{dir_out}
#' (character scalar).
#'
#' @return A list with components:
#' \itemize{
#'   \item \code{fit}: the estimated VAR model (\code{vars} object).
#'   \item \code{stability}: result of \code{vars::stability()} (or
#'         \code{"try-error"} on failure).
#'   \item \code{serial}: result of \code{vars::serial.test()} (or
#'         \code{"try-error"} on failure).
#'   \item \code{normal}: result of \code{vars::normality.test()} (or
#'         \code{"try-error"} on failure).
#'   \item \code{arch}: result of \code{vars::arch.test()} (or
#'         \code{"try-error"} on failure).
#' }
#'
#' @examples
#' \donttest{
#' library(data.table)
#'
#' # 1. Create dummy data with ALL required columns for VARX
#' # The function explicitly requires these specific exogenous variables
#' DT <- data.table(
#'   year = 2000:2049, # 50 obs to ensure diagnostics (lags.pt=10) don't fail
#'   I = rpois(50, lambda = 10),
#'   C = rpois(50, lambda = 8),
#'   # Exogenous regressors required by the function
#'   EconCycle = rnorm(50),
#'   PopDensity = rnorm(50),
#'   Epidemics = rnorm(50),
#'   Climate = rnorm(50),
#'   War = rnorm(50),
#'   t_norm = seq(-1, 1, length.out = 50)
#' )
#'
#' # 2. Define global output directory using tempdir() (Fixes CRAN policy)
#' # run_varx looks for 'dir_out' in the global environment
#' tmp_dir <- tempdir()
#' dir_out <- file.path(tmp_dir, "varx")
#' if (!dir.exists(dir_out)) dir.create(dir_out, recursive = TRUE)
#'
#' # 3. Run the function
#' # We use p=1 to keep it fast and stable for the example check
#' res_varx <- run_varx(DT, p = 1)
#'
#' # Inspect the fitted VAR object if it didn't fail
#' if (!inherits(res_varx$fit, "try-error")) {
#'   print(res_varx$fit)
#' }
#' }
#'
#' @export

run_varx <- function(DT, p=2) {
  y_cols <- c("I","C")
  x_cols <- c("EconCycle","PopDensity","Epidemics","Climate","War","t_norm")
  
  y <- as.matrix(DT[, ..y_cols])
  exog <- as.matrix(DT[, ..x_cols])
  
  colnames(exog) <- c("EconCycle","PopDensity","Epidemics","Climate","War","t_norm")
  fit <- vars::VAR(y, p=p, type="const", exogen=exog)
  st <- try(vars::stability(fit, type="OLS-CUSUM"), silent=TRUE)
  serial <- try(vars::serial.test(fit, lags.pt=10, type="PT.asymptotic"), silent=TRUE)
  normal <- try(vars::normality.test(fit), silent=TRUE)
  arch <- try(vars::arch.test(fit, lags.multi=5), silent=TRUE)
  saveRDS(list(fit=fit, stability=st, serial=serial, normal=normal, arch=arch), file.path(dir_out, "varx_fit.rds"))
  list(fit=fit, stability=st, serial=serial, normal=normal, arch=arch)
}
