% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check-response.R
\name{brs_check}
\alias{brs_check}
\title{Transform and validate a scale-derived response variable}
\usage{
brs_check(y, ncuts = 100L, lim = 0.5, delta = NULL)
}
\arguments{
\item{y}{Numeric vector: the raw response. Can be either
integer scores on the scale \eqn{\{0, 1, \ldots, K\}} or
continuous values already in \eqn{(0, 1)}.}

\item{ncuts}{Integer: number of scale categories \eqn{K}
(default 100). Must be \eqn{\geq \max(y)}.}

\item{lim}{Numeric: half-width \eqn{h} of the uncertainty
region (default 0.5). Controls the width of the interval
around each scale point.}

\item{delta}{Integer vector or \code{NULL}. If \code{NULL}
  (default), censoring types are inferred automatically from the
  boundary rules described above.

  If provided, must have the same length as \code{y}, with every
  element in \code{\{0, 1, 2, 3\}}. The supplied values override
  the automatic classification on a per-observation basis, and the
  endpoint formulas adapt to non-boundary observations as
  described in the table above.

  This parameter is used internally by the simulation functions
  when the analyst forces a specific censoring type
  (e.g., \code{brs_sim(..., delta = 2)}).}
}
\value{
A numeric matrix with \eqn{n} rows and 5 columns:
\describe{
  \item{\code{left}}{Lower endpoint \eqn{l_i} on \eqn{(0, 1)},
    clamped to \eqn{[\epsilon, 1 - \epsilon]}.}
  \item{\code{right}}{Upper endpoint \eqn{u_i} on \eqn{(0, 1)},
    clamped to \eqn{[\epsilon, 1 - \epsilon]}.}
  \item{\code{yt}}{Midpoint approximation \eqn{y_t} for
    starting-value computation (does not enter the likelihood).}
  \item{\code{y}}{Original response value (preserved unchanged).}
  \item{\code{delta}}{Censoring indicator: 0 = exact (density),
    1 = left-censored \eqn{F(u)}, 2 = right-censored
    \eqn{1 - F(l)}, 3 = interval-censored \eqn{F(u) - F(l)}.}
}
}
\description{
Takes a discrete (or continuous) response on the scale
\eqn{0, 1, \ldots, K} (where \eqn{K =} \code{ncuts}) and converts
it to a pair of interval endpoints on the open unit interval
\eqn{(0, 1)}.  Each observation is classified into one of four
censoring types following the complete likelihood used in this package:

\describe{
  \item{\eqn{\delta = 0}}{Uncensored (exact): the observation is a
    continuous value already in \eqn{(0, 1)}.  The likelihood
    contribution is the density \eqn{f(y_i | \theta)}.
    Endpoints: \eqn{l_i = u_i = y_i} (or \eqn{y_i / K} when on
    the scale).}
  \item{\eqn{\delta = 1}}{Left-censored: the latent value is below
    some upper bound \eqn{u_i}.  The contribution is
    \eqn{F(u_i | \theta)}.
    When the observation is at the scale minimum (\eqn{y = 0}),
    the upper bound is \eqn{u_i = \mathrm{lim} / K}.
    When the user forces \eqn{\delta = 1} on a non-boundary
    observation (\eqn{y \neq 0}), the upper bound is
    \eqn{u_i = (y + \mathrm{lim}) / K}, preserving observation-
    specific variation.  In both cases \eqn{l_i = \epsilon}.}
  \item{\eqn{\delta = 2}}{Right-censored: the latent value is above
    some lower bound \eqn{l_i}.  The contribution is
    \eqn{1 - F(l_i | \theta)}.
    When the observation is at the scale maximum (\eqn{y = K}),
    the lower bound is \eqn{l_i = (K - \mathrm{lim}) / K}.
    When the user forces \eqn{\delta = 2} on a non-boundary
    observation (\eqn{y \neq K}), the lower bound is
    \eqn{l_i = (y - \mathrm{lim}) / K}, preserving observation-
    specific variation.  In both cases \eqn{u_i = 1 - \epsilon}.}
  \item{\eqn{\delta = 3}}{Interval-censored: the standard case for
    scale data.  The contribution is
    \eqn{F(u_i | \theta) - F(l_i | \theta)} with midpoint interval
    endpoints \eqn{[(y - \mathrm{lim})/K,\; (y + \mathrm{lim})/K]}.}
}
}
\details{
\strong{Automatic classification} (\code{delta = NULL}):

If the entire input vector is already in \eqn{(0, 1)} (i.e., all
values satisfy \eqn{0 < y < 1}), all observations are treated as
uncensored (\eqn{\delta = 0}).

Otherwise, for scale (integer) data:
\itemize{
  \item \eqn{y = 0}: left-censored (\eqn{\delta = 1}).
  \item \eqn{y = K}: right-censored (\eqn{\delta = 2}).
  \item \eqn{0 < y < K}: interval-censored (\eqn{\delta = 3}).
}

\strong{User-supplied delta} (\code{delta} vector):

When the \code{delta} argument is provided, the user-supplied
censoring indicators override the automatic boundary-based rules
on a per-observation basis.  This is the mechanism used by
\code{\link{brs_sim}} when the analyst forces a
specific censoring type in Monte Carlo studies.

The endpoint formulas for each delta value are:

\tabular{llll}{
  \eqn{\delta} \tab Condition \tab \eqn{l_i} (left) \tab \eqn{u_i} (right) \cr
  0 \tab \eqn{y \in (0, 1)} \tab \eqn{y}
    \tab \eqn{y} \cr
  0 \tab \eqn{y} on scale \tab \eqn{y / K}
    \tab \eqn{y / K} \cr
  1 \tab \eqn{y = 0} (boundary) \tab \eqn{\epsilon}
    \tab \eqn{\mathrm{lim} / K} \cr
  1 \tab \eqn{y \neq 0} (forced) \tab \eqn{\epsilon}
    \tab \eqn{(y + \mathrm{lim}) / K} \cr
  2 \tab \eqn{y = K} (boundary) \tab \eqn{(K - \mathrm{lim}) / K}
    \tab \eqn{1 - \epsilon} \cr
  2 \tab \eqn{y \neq K} (forced) \tab \eqn{(y - \mathrm{lim}) / K}
    \tab \eqn{1 - \epsilon} \cr
  3 \tab midpoint interval \tab \eqn{(y - \mathrm{lim}) / K}
    \tab \eqn{(y + \mathrm{lim}) / K} \cr
}

All endpoints are clamped to \eqn{[\epsilon, 1 - \epsilon]} with
\eqn{\epsilon = 10^{-5}} to avoid boundary issues in the beta
likelihood.

The midpoint approximation \code{yt} is computed as:
\itemize{
  \item \eqn{y_t = y} when \eqn{y \in (0, 1)} (continuous data).
  \item \eqn{y_t = y / K} when \eqn{y} is on the integer scale.
}
This value is used exclusively as an initialization aid for
starting-value computation and does not enter the likelihood.

\strong{Interaction with the fitting pipeline}:

This function is called internally by \code{.extract_response()}
  when the data does \emph{not} carry the \code{"is_prepared"}
  attribute.  If data has already been processed by
  \code{\link{brs_prep}} or by simulation with forced delta
(\code{\link{brs_sim}} with \code{delta != NULL}),
the pre-computed columns are used directly and
\code{brs_check()} is skipped.
}
\examples{
# Scale data with boundary observations
y <- c(0, 3, 5, 7, 9, 10)
brs_check(y, ncuts = 10)

# Force all observations to be exact (delta = 0)
brs_check(y, ncuts = 10, delta = rep(0L, length(y)))

# Force delta = 1 on non-boundary observations:
# endpoints use actual y values, preserving variation
y2 <- c(30, 60)
brs_check(y2, ncuts = 100, delta = c(1L, 1L))
#  left = (eps, eps), right = (30.5/100, 60.5/100)
}
\references{
Lopes, J. E. (2023). \emph{Modelos de regressao beta para dados de escala}.
Master's dissertation, Universidade Federal do Parana, Curitiba.
URI: \url{https://hdl.handle.net/1884/86624}.

Hawker, G. A., Mian, S., Kendzerska, T., and French, M. (2011).
Measures of adult pain: Visual Analog Scale for Pain (VAS Pain),
Numeric Rating Scale for Pain (NRS Pain), McGill Pain Questionnaire (MPQ),
Short-Form McGill Pain Questionnaire (SF-MPQ), Chronic Pain Grade Scale
(CPGS), Short Form-36 Bodily Pain Scale (SF-36 BPS), and Measure of
Intermittent and Constant Osteoarthritis Pain (ICOAP).
Arthritis Care and Research, 63(S11), S240-S252.
\doi{10.1002/acr.20543}

Hjermstad, M. J., Fayers, P. M., Haugen, D. F., et al. (2011).
Studies comparing Numerical Rating Scales, Verbal Rating Scales, and
Visual Analogue Scales for assessment of pain intensity in adults:
a systematic literature review.
Journal of Pain and Symptom Management, 41(6), 1073-1093.
\doi{10.1016/j.jpainsymman.2010.08.016}
}
\seealso{
\code{\link{brs_prep}} for the analyst-facing
  pre-processing function; \code{\link{brs_sim}}
  for simulation with forced delta.
}
