# Stan runner for appraise

#' Run the appraise Stan model
#'
#' Executes the posterior mixture model described in Kabali (2025)
#'
#' @param bias_spec Output from build_bias_specification()
#' @param yhat Reported point estimate
#' @param stdev Reported standard error
#' @param threshold_value Threshold for significance
#' @param iter_sampling Number of sampling iterations
#' @param iter_warmup Number of warmup iterations
#' @param chains Number of MCMC chains
#' @param seed Random seed
#'
#' @return A list containing the CmdStan fit object and posterior draws
#'
#' @references
#' Kabali C (2025). AppRaise: Software for quantifying evidence uncertainty
#' in systematic reviews using a posterior mixture model. *Journal of
#' Evaluation in Clinical Practice*, 31, 1-12. https://doi.org/10.1111/jep.70272.
#'
#' @seealso
#' \itemize{
#'   \item \code{\link{build_bias_specification}}
#'   \item \code{\link{posterior_probability}}
#'   \item \code{\link{posterior_summary}}
#'   \item \code{vignette("appraise-introduction")}
#' }
#'
#' @examples
#' # Define a simple bias specification with one bias
#' bias_spec <- build_bias_specification(
#'   num_biases = 1,
#'   b_types = "Confounding",
#'   ab_params = list(
#'     Confounding = c(2, 5)
#'   )
#' )
#'
#' bias_spec
#'
#' @examples
#' \donttest{
#' # Run the AppRaise model for a single study. Requires cmdstanr and a CmdStan
#' # installation.
#' fit <- run_appraise_model(
#'   bias_spec = bias_spec,
#'   yhat = -0.6,
#'   stdev = 0.12,
#'   threshold_value = -0.4,
#'   iter_sampling = 500,
#'   iter_warmup = 250,
#'   chains = 2,
#'   seed = 123
#' )
#'
#' # Posterior draws of the causal effect
#' head(fit$theta)
#'
#' # Posterior probability of exceeding the threshold
#' posterior_probability(fit$mid)
#'
#' # Posterior summary
#' posterior_summary(fit$theta)
#' }

#' @export
run_appraise_model <- function(
    bias_spec,
    yhat,
    stdev,
    threshold_value,
    iter_sampling = 5000,
    iter_warmup = 1000,
    chains = 4,
    seed = 12345
) {

  if (!requireNamespace("cmdstanr", quietly = TRUE)) {
    stop("cmdstanr is required but not installed", call. = FALSE)
  }

  stan_file <- system.file(
    "stan",
    "appraise_model.stan",
    package = "appraise"
  )

  if (stan_file == "") {
    stop("Stan model file not found in inst/stan", call. = FALSE)
  }

  # Build Stan data (matches Stan data block exactly)
  model_data <- c(
    list(
      yhat = yhat,
      yhat2 = yhat,
      stdev = stdev,
      threshold_value = threshold_value
    ),
    bias_spec
  )

  mod <- cmdstanr::cmdstan_model(stan_file)

  fit <- mod$sample(
    data = model_data,
    iter_sampling = iter_sampling,
    iter_warmup = iter_warmup,
    chains = chains,
    parallel_chains = chains,
    seed = seed
  )

  # Extract posterior draws (post-warmup)
  theta <- fit$draws(variables = "theta", format = "draws_matrix")
  mid   <- fit$draws(variables = "mid",   format = "draws_matrix")

  list(
    fit = fit,
    theta = as.vector(theta),
    mid = as.vector(mid)
  )
}
