#' catch_advice
#'
#' @description
#' Calculates the advised catch using the SlotLim framework and (optionally) returns
#' a plot of the percentage change relative to \code{Cy} across a grid of
#' (\code{TBA}, \code{SAM}) values, with the output overlaid.
#'
#' @param Cy Numeric (length 1) > 0. Most recent annual catch, or multi-year average.
#'   If landing size restrictions have changed, use \code{catch_adjust} to adjust
#'   the starting catch value accordingly.
#' @param TBA Numeric (length 1) > 0. Targeted Biomass Adjustment (see \code{TBA()}).
#' @param SAM Numeric (length 1) > 0. Size Adherence Multiplier (see \code{SAM()}).
#' @param T1 Optional numeric (length 1) in (0,1). Maximum allowed proportional
#'   \emph{decrease}. If \code{NULL}, no lower cap.
#' @param T2 Optional numeric (length 1) in (0,1). Maximum allowed proportional
#'   \emph{increase}. If \code{NULL}, no upper cap.
#' @param plot Logical. If \code{TRUE}, return a \pkg{ggplot2} heatmap (default \code{FALSE}).
#'
#' @return
#' \itemize{
#'   \item \code{Ay}: Catch advice (same units as \code{Cy}).
#'   \item \code{Ay_percent}: Percent change of advice relative to \code{Cy}.
#'   \item \code{plot}: (only when \code{plot = TRUE}) a \pkg{ggplot2} object
#'     visualizing percent change across \eqn{TBA \times SAM}.
#' }
#'
#' @examples
#' Cy <- 1000; TBA <- 1.1; SAM <- 0.9
#' catch_advice(Cy, TBA, SAM)  # compute only
#'
#' \donttest{
#' catch_advice(Cy, TBA, SAM, plot = TRUE)
#' catch_advice(Cy, TBA, SAM, T1 = 0.2, T2 = 0.2, plot = TRUE)
#' }
#'
#' @seealso \code{\link{TBA}}, \code{\link{SAM}}
#' @export
catch_advice <- function(Cy = NULL, TBA = NULL, SAM = NULL, T1 = NULL, T2 = NULL, plot = FALSE) {
  # ---- validation ----
  args <- list(Cy = Cy, TBA = TBA, SAM = SAM)
  bad <- vapply(args, function(x) !is.numeric(x) || length(x) != 1L || !is.finite(x) || x <= 0, logical(1))
  if (any(bad)) stop("`Cy`, `TBA`, and `SAM` must be finite, positive numeric scalars.", call. = FALSE)

  for (nm in c("T1","T2")) {
    val <- get(nm, inherits = FALSE)
    if (!is.null(val)) {
      if (!is.numeric(val) || length(val) != 1L || !is.finite(val) || val <= 0 || val >= 1)
        stop(sprintf("`%s` must be a single numeric in (0, 1), or NULL.", nm), call. = FALSE)
    }
  }

  if (!is.logical(plot) || length(plot) != 1L || is.na(plot))
    stop("`plot` must be a single logical (TRUE/FALSE).", call. = FALSE)

  # ---- compute Ay with asymmetric caps ----
  raw <- Cy * TBA * SAM
  lower_bound <- if (is.null(T1)) -Inf else (1 - T1) * Cy
  upper_bound <- if (is.null(T2))  Inf else (1 + T2) * Cy
  Ay <- min(upper_bound, max(lower_bound, raw))
  Ay_percent <- (Ay / Cy - 1) * 100

  out <- list(Ay = Ay, Ay_percent = Ay_percent)

  # ---- optional plot ----
  if (isTRUE(plot)) {
    if (!requireNamespace("ggplot2", quietly = TRUE))
      stop("Package 'ggplot2' is required for plotting.", call. = FALSE)

    step <- 0.01
    TBA_vals <- seq(1 - 0.99, 1 + 0.99, by = step)
    SAM_vals <- seq(1 - 0.99, 1 + 0.99, by = step)
    grid <- expand.grid(TBA_vals = TBA_vals, SAM_vals = SAM_vals, KEEP.OUT.ATTRS = FALSE)

    grid_raw <- Cy * grid$TBA_vals * grid$SAM_vals
    grid_capped <- pmin(upper_bound, pmax(lower_bound, grid_raw))
    grid$Deviation <- (grid_capped / Cy - 1) * 100

    # color limits (work with or without caps)
    if (is.null(T1) && is.null(T2)) {
      color_limits <- c(-100, 100)
    } else {
      lo <- if (is.null(T1)) min(-100, floor(min(grid$Deviation))) else -T1 * 100
      hi <- if (is.null(T2)) max( 100, ceiling(max(grid$Deviation))) else  T2 * 100
      color_limits <- c(lo, hi)
    }
    # clamp to limits for a stable legend
    grid$Deviation <- pmax(pmin(grid$Deviation, max(color_limits)), min(color_limits))

    # axis ranges: include the inputs and some breathing room
    xlim <- range(c(TBA_vals, TBA)); ylim <- range(c(SAM_vals, SAM))

    # build plot (fully qualified calls; no rlang needed)
    p <- ggplot2::ggplot(grid, ggplot2::aes(x = TBA_vals, y = SAM_vals, fill = Deviation)) +
      ggplot2::geom_tile(alpha = 0.85) +
      ggplot2::scale_fill_gradient2(
        name = "",
        low = "darkred", mid = "lightyellow", high = "darkgreen",
        midpoint = 0, limits = color_limits
      ) +
      ggplot2::guides(
        fill = ggplot2::guide_colorbar(
          title = NULL,
          direction = "horizontal",
          barwidth = grid::unit(0.3, "npc"),
          barheight = grid::unit(10, "pt"),
          label.position = "bottom",
          ticks = FALSE
        )
      ) +
      ggplot2::annotate("point", x = TBA, y = SAM, size = 3, color = "black") +
      ggplot2::labs(
        title = "Catch advice %",
        x = "TBA",
        y = "SAM"
      ) +
      ggplot2::coord_cartesian(xlim = xlim, ylim = ylim, expand = FALSE) +
      ggplot2::theme_bw() +
      ggplot2::theme(
        plot.title = ggplot2::element_text(hjust = 0.5),
        plot.margin = grid::unit(c(0.25, 0.25, 0.25, 0.25), "cm"),
        text = ggplot2::element_text(size = 15),
        legend.position = "top",
        legend.direction = "horizontal",
        legend.background = ggplot2::element_blank(),
        legend.key = ggplot2::element_blank()
      )

    out$plot <- p
  }

  out
}
